/*-----------------------------------------------------------------------
QMBTagger - Qt Based MusicBrainz MP3/OGG/FLAC Tagger
Copyright (C) 2003,2004 Ewen Cheslack-Postava

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-----------------------------------------------------------------------*/

/****************************************************************************
 * madlld.c -- A simple program decoding an mpeg audio stream to 16-bit     *
 * PCM from stdin to stdout. This program is just a simple sample           *
 * demonstrating how the low-level libmad API can be used.                  *
 *--------------------------------------------------------------------------*
 * (c) 2001, 2002 Bertrand Petit                                            *
 *                                                                          *
 * Redistribution and use in source and binary forms, with or without       *
 * modification, are permitted provided that the following conditions       *
 * are met:                                                                 *
 *                                                                          *
 * 1. Redistributions of source code must retain the above copyright        *
 *    notice, this list of conditions and the following disclaimer.         *
 *                                                                          *
 * 2. Redistributions in binary form must reproduce the above               *
 *    copyright notice, this list of conditions and the following           *
 *    disclaimer in the documentation and/or other materials provided       *
 *    with the distribution.                                                *
 *                                                                          *
 * 3. Neither the name of the author nor the names of its contributors      *
 *    may be used to endorse or promote products derived from this          *
 *    software without specific prior written permission.                   *
 *                                                                          *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''       *
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED        *
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A          *
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR       *
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,             *
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT         *
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF         *
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND      *
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,       *
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT       *
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF       *
 * SUCH DAMAGE.                                                             *
 *                                                                          *
 ****************************************************************************/

#include "qmbmad.h"

#if HAVE_LIBMUSICBRAINZ
#if HAVE_LIBMAD

#include <string.h>

/*
 * The following utility routine performs simple rounding, clipping, and
 * scaling of MAD's high-resolution samples down to 16 bits. It does not
 * perform any dithering or noise shaping, which would be recommended to
 * obtain any exceptional audio quality. It is therefore not recommended to
 * use this routine if high-quality output is desired.
 */

signed int QMBmad::scale(mad_fixed_t sample)
{
	/* round */
	sample += (1L << (MAD_F_FRACBITS - 16));

	/* clip */
	if (sample >= MAD_F_ONE)
		sample = MAD_F_ONE - 1;
	else if (sample < -MAD_F_ONE)
		sample = -MAD_F_ONE;

	/* quantize */
	return sample >> (MAD_F_FRACBITS + 1 - 16);
}


#define INPUT_BUFFER_SIZE   (5*8192)
#define OUTPUT_BUFFER_SIZE  8192    /* Must be an integer multiple of 4. */
//returns:
//0 on success
//-1 on decode error
int QMBmad::ReadMPEG(FILE* file, char* buffer, int bufferlen)
{
    struct mad_stream Stream;
    struct mad_frame Frame;
    struct mad_synth Synth;
    mad_timer_t Timer;
    unsigned char InputBuffer[INPUT_BUFFER_SIZE],
    OutputBuffer[OUTPUT_BUFFER_SIZE], *OutputPtr = OutputBuffer;
    const unsigned char *OutputBufferEnd = OutputBuffer + OUTPUT_BUFFER_SIZE;
    int Status = 0, i, filledbuffer = 0;
    unsigned long FrameCount = 0;
	int bufferpos = 0;

    /* First the structures used by libmad must be initialized. */
    mad_stream_init (&Stream);
    mad_frame_init (&Frame);
    mad_synth_init (&Synth);
    mad_timer_reset (&Timer);

    /* Decoding options can here be set in the options field of the
     * Stream structure.
     */

    /* This is the decoding loop. */
    do
    {
        /* The input bucket must be filled if it becomes empty or if
         * it's the first execution of the loop.
         */
        if (Stream.buffer == NULL || Stream.error == MAD_ERROR_BUFLEN)
        {
            size_t ReadSize, Remaining;
            unsigned char *ReadStart;

            /* {1} libmad may not consume all bytes of the input
             * buffer. If the last frame in the buffer is not wholly
             * contained by it, then that frame's start is pointed by
             * the next_frame member of the Stream structure. This
             * common situation occurs when mad_frame_decode() fails,
             * sets the stream error code to MAD_ERROR_BUFLEN, and
             * sets the next_frame pointer to a non NULL value. (See
             * also the comment marked {2} bellow.)
             *
             * When this occurs, the remaining unused bytes must be
             * put back at the beginning of the buffer and taken in
             * account before refilling the buffer. This means that
             * the input buffer must be large enough to hold a whole
             * frame at the highest observable bit-rate (currently 448
             * kb/s). XXX=XXX Is 2016 bytes the size of the largest
             * frame? (448000*(1152/32000))/8
             */
            if (Stream.next_frame != NULL)
            {
                Remaining = Stream.bufend - Stream.next_frame;
                memmove (InputBuffer, Stream.next_frame, Remaining);
                ReadStart = InputBuffer + Remaining;
                ReadSize = INPUT_BUFFER_SIZE - Remaining;
            }
            else
                ReadSize = INPUT_BUFFER_SIZE,
            ReadStart = InputBuffer, Remaining = 0;

	    /* Fill-in the buffer. If an error occurs print a message
             * and leave the decoding loop. If the end of stream is
             * reached we also leave the loop but the return status is
             * left untouched.
             */
            ReadSize = fread (ReadStart, 1, ReadSize, file);
            if (ReadSize <= 0)
            {
                if (ferror (file))
                {
                    Status = -1;
                }
                break;
            }

            /* Pipe the new buffer content to libmad's stream decoder
             * facility.
             */
            mad_stream_buffer (&Stream, InputBuffer, ReadSize + Remaining);
            Stream.error = MAD_ERROR_NONE;
        }

        /* Decode the next mpeg frame. The streams is read from the
         * buffer, its constituents are break down and stored the the
         * Frame structure, ready for examination/alteration or PCM
         * synthesis. Decoding options are carried in the Frame
         * structure from the Stream structure.
         *
         * Error handling: mad_frame_decode() returns a non zero value
         * when an error occurs. The error condition can be checked in
         * the error member of the Stream structure. A mad error is
         * recoverable or fatal, the error status is checked with the
         * MAD_RECOVERABLE macro.
         *
         * {2} When a fatal error is encountered all decoding
         * activities shall be stopped, except when a MAD_ERROR_BUFLEN
         * is signaled. This condition means that the
         * mad_frame_decode() function needs more input to achieve
         * it's work. One shall refill the buffer and repeat the
         * mad_frame_decode() call. Some bytes may be left unused at
         * the end of the buffer if those bytes forms an incomplete
         * frame. Before refilling, the remainign bytes must be moved
         * to the begining of the buffer and used for input for the
         * next mad_frame_decode() invocation. (See the comments marked
         * {1} earlier for more details.)
         *
         * Recoverable errors are caused by malformed bit-streams, in
         * this case one can call again mad_frame_decode() in order to
         * skip the faulty part and re-sync to the next frame.
         */
        if (mad_frame_decode (&Frame, &Stream))
        {
            if (MAD_RECOVERABLE (Stream.error))
            {
                continue;
            }
            else if (Stream.error == MAD_ERROR_BUFLEN)
                continue;
            else
            {
                Status = -1;
                break;
            }
        }
        /* The characteristics of the stream's first frame is printed
         * on stderr. The first frame is representative of the entire
         * stream.
         */


	/* Accounting. The computed frame duration is in the frame
         * header structure. It is expressed as a fixed point number
         * whole data type is mad_timer_t. It is different from the
         * samples fixed point format and unlike it, it can't directly
         * be added or substracted. The timer module provides several
         * functions to operate on such numbers. Be careful there, as
         * some functions of mad's timer module receive some of their
         * mad_timer_t arguments by value!
         */
        FrameCount++;
        mad_timer_add (&Timer, Frame.header.duration);

        /* Once decoded the frame is synthesized to PCM samples. No errors
         * are reported by mad_synth_frame();
         */
        mad_synth_frame (&Synth, &Frame);

        /* Synthesized samples must be converted from mad's fixed
         * point number to the consumer format. Here we use unsigned
         * 16 bit big endian integers on two channels. Integer samples
         * are temporarily stored in a buffer that is flushed when
         * full.
         */
        for (i = 0; i < Synth.pcm.length; i++)
        {
            signed int Sample;

            /* Left channel */
            Sample = QMBmad::scale(Synth.pcm.samples[0][i]);

            *(OutputPtr++) = (Sample >> 0) & 0xff;
            *(OutputPtr++) = (Sample >> 8) & 0xff;

            /* Right channel. If the decoded stream is monophonic then
             * the right output channel is the same as the left one.
             */
            if (MAD_NCHANNELS (&Frame.header) == 2)
                Sample = scale(Synth.pcm.samples[1][i]);

            *(OutputPtr++) = (Sample >> 0) & 0xff;
            *(OutputPtr++) = (Sample >> 8) & 0xff;

            /* Flush the buffer if it is full. */
            if (OutputPtr == OutputBufferEnd)
            {
                OutputPtr = OutputBuffer;

                //copy the data into the provided buffer
		if (bufferlen - bufferpos >= OUTPUT_BUFFER_SIZE) {
			memcpy(buffer + bufferpos, OutputBuffer, OUTPUT_BUFFER_SIZE);
			bufferpos += OUTPUT_BUFFER_SIZE;
		} else {
			memcpy(buffer + bufferpos, OutputBuffer, bufferlen - bufferpos);
			filledbuffer = 1;
			break;
		}
            }
        }
    }
    while (!filledbuffer);

    /* Mad is no longer used, the structures that were initialized must
     * now be cleared.
     */
    mad_synth_finish (&Synth);
    mad_frame_finish (&Frame);
    mad_stream_finish (&Stream);

    return (Status);
}

#endif /* HAVE_LIBMAD */
#endif /*HAVE_LIBMUSICBRAINZ*/
