/***************************************************************************
                          database_mysql.cpp  -  description
                             -------------------
    begin                : Thu May 3 2001
    copyright            : (C) 2001 by Holger Sattel
    email                : hsattel@rumms.uni-mannheim.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "database_mysql.h"
#include "gui.h"
#include "pixmapcache.h"

#include <stdlib.h>
#include <stdio.h>
#include <cassert>
#include <sstream>
#include <iostream>

#include <qmessagebox.h>
#include <qfile.h>
#include <qcstring.h>
#include <qprocess.h>

#ifdef EMBEDDED 
  #include <mysql.h>
#endif

using namespace std;


// *** constructor ***
DataBase_MySQL::DataBase_MySQL() {
#ifdef EMBEDDED
	db = qts::QSqlDatabase::addDatabase("QMYSQL3");
#else
	db = QSqlDatabase::addDatabase("QMYSQL3");
#endif   // ifdef EMBEDDED
	isConnected = false;
	error = 0;

    connect(this, SIGNAL(connected(bool)), gui, SLOT(slot_setConnectionState(bool)));
    connect(this, SIGNAL(connected(bool)), gui->getSelector(), SLOT(slot_setConnectionState(bool)));
    connect(this, SIGNAL(connected(bool)), gui->getSourcer(), SLOT(slot_setConnectionState(bool)));
    connect(this, SIGNAL(connected(bool)), gui->getTagListing(), SLOT(slot_setConnectionState(bool)));
    connect(this, SIGNAL(connected(bool)), gui->getPlaylisting(), SLOT(slot_setConnectionState(bool)));
    connect(this, SIGNAL(connected(bool)), gui->getListing(), SLOT(slot_setConnectionState(bool)));
    connect(this, SIGNAL(connected(bool)), gui->getListing()->getSQL(), SLOT(slot_setConnectionState(bool)));
    connect(this, SIGNAL(connected(bool)), gui->getListing()->getSearcher(), SLOT(slot_setConnectionState(bool)));
#ifdef HAVE_MIXXX
    connect(this, SIGNAL(connected(bool)), gui->getListing()->getMixxxQueue(), SLOT(slot_setConnectionState(bool)));
#endif   // ifdef HAVE_MIXXX
  
    emit connected(false);
}

// ##############################################
// # establishes connection to database
// ##############################################
void DataBase_MySQL::establishConnection(QString host, QString dbname, QString user, QString pass)
{
#ifdef EMBEDDED
    if(!qts::QSqlDatabase::contains()) {
#else
    if(!QSqlDatabase::contains()) {
#endif   // ifdef EMBEDDED
	error = 1;
	isConnected = false;
	cout << _("Qt3 MYSQL plugin not available") << endl;
	return; 
    }

    error = 0;
    if(db->isOpen()) db->close();
    db->setDatabaseName(dbname);
    db->setUserName(user);
    db->setPassword(pass);
    db->setHostName(host);
    if(db->open()) {
	    isConnected = true;
#ifdef EMBEDDED
    	qts::QSqlQuery query("SHOW tables;", db);
#else
    	QSqlQuery query("SHOW tables;", db);
#endif   // ifdef EMBEDDED
        if(query.isActive()) {
            bool artists_exists = false;
            bool mediums_exists = false;
            bool tracks_exists = false;
            bool playlists_exists = false;
            bool playlist_tracks_exists = false;
            bool version_history_exists = false;
            bool mextras_exists = false;
            bool albums_exists = false;
            bool custom_queries_exists = false;
            bool q1 = true, q2 = true, q3 = true;
            bool q4 = true, q5 = true, q6 = true;
            bool q7 = true, q8 = true;
            while(query.next()) {
            	if(query.value(0).toString() == "artists") artists_exists = true;
            	else if(query.value(0).toString() == "mediums") mediums_exists = true;
            	else if(query.value(0).toString() == "tracks") tracks_exists = true;
            	else if(query.value(0).toString() == "playlists") playlists_exists = true;
            	else if(query.value(0).toString() == "playlist_tracks") playlist_tracks_exists = true;
            	else if(query.value(0).toString() == "version_history") version_history_exists = true;
            	else if(query.value(0).toString() == "mextras") mextras_exists = true;
                else if(query.value(0).toString() == "albums") albums_exists = true;
                else if(query.value(0).toString() == "custom_queries") custom_queries_exists = true;
            }
            
            if(!custom_queries_exists) {
        	    q8 = query.exec("CREATE TABLE custom_queries (name varchar(255) NOT NULL, where_clause text NOT NULL, PRIMARY KEY (name)) TYPE=MyISAM;");
                if(!query.isActive()) setError(1, query);
        	}
            
            if(!version_history_exists) {  // table created before 0.9.2
        	    q6 = query.exec("CREATE TABLE version_history (version varchar(255) NOT NULL, date datetime NOT NULL, encoding varchar(255) NOT NULL) TYPE=MyISAM;");
                if(!query.isActive()) setError(1, query);
        	}
        
            if(!artists_exists) {
            	q1 = query.exec("CREATE TABLE artists (id int(11) AUTO_INCREMENT NOT NULL, name varchar(255) BINARY NOT NULL, total int(11) default NULL, local int(11) default NULL, isFavourite tinyint(4) default NULL, image_id int(11) default NULL, years_active VARCHAR(255) default NULL, biography_id int(11) default NULL, PRIMARY KEY (id), UNIQUE name_unique (name), INDEX isFavourite_index (isFavourite)) TYPE=MyISAM;");
                if(!query.isActive()) setError(1, query);
            } else {
                if( !checkExtra("artists", "id", "auto_increment")) {
                   query.exec("ALTER TABLE artists CHANGE id id INT(11) NOT NULL AUTO_INCREMENT;");
                   if(!query.isActive()) setError(1, query);
                }
                if( !checkExtra("artists", "name", "varchar(255) binary")) {
                   query.exec("ALTER TABLE artists CHANGE name name VARCHAR(255) BINARY NOT NULL;");
                   if(!query.isActive()) setError(1, query);
                }
                if ( !checkIndex( "artists", "name_unique" ) )  {
                   // creates index if missing  
        	        query.exec("ALTER TABLE artists ADD UNIQUE name_unique (name);");
                    if(!query.isActive()) setError(1, query);
                }
                if ( !checkIndex( "artists", "isFavourite_index" ) )  {
                   // creates index if missing  
                  query.exec("ALTER TABLE artists ADD INDEX isFavourite_index (isFavourite);");
                    if(!query.isActive()) setError(1, query);
                }
            }
    
            if(!albums_exists) {
                q7 = query.exec("CREATE TABLE albums (id int(11) AUTO_INCREMENT NOT NULL, "
                                                     "name varchar(255) NOT NULL, "
                                                     "artist varchar(255) NOT NULL, "
                                                     "tracks_available int(11) NOT NULL default 1, "
                                                     "cover_id int(11) default NULL, "
                                                     "album_review_id int(11) default NULL, "
                                                     "album_tracks_id int(11) default NULL, "
                                                     "year int(4) default NULL, "
                                                     "PRIMARY KEY (id), "
                                                     "INDEX name_index (name), "
                                                     "INDEX artist_index (artist)"
                                                    ") TYPE=MyISAM;");
                if(!query.isActive()) {
                    setError(1, query);
                } else if(tracks_exists) {
                    if(checkFieldExists("tracks", "cover_id") && checkFieldExists("tracks", "album_review_id")) {
                        fillTableAlbumsFromTableTracks(true);
                    } else {
                        fillTableAlbumsFromTableTracks(false);
                    }
                }
            } else {
                if(!checkFieldExists("albums", "album_tracks_id")) {
                    query.exec("ALTER TABLE albums ADD album_tracks_id int(11) default NULL;");
                    if(!query.isActive()) setError(1, query);
                }
                if(!checkVersion()) {
    	            // fix 0.9.3beta1 bug
                    query.exec("DELETE FROM albums WHERE tracks_available <= 0;");
                    if(!query.isActive()) {
                        setError(1, query);
                    }
                }
            }
            
            if(!mediums_exists) {
    		    q2 = query.exec("CREATE TABLE mediums (id int(11) NOT NULL default 0, type int(11) default NULL, label varchar(255) default NULL, path varchar(255) default NULL, checksum int(11) default NULL, PRIMARY KEY (id)) TYPE=MyISAM;");
                if(!query.isActive()) setError(1, query);
    	    }
    	    if(!tracks_exists) {
    		    q3 = query.exec("CREATE TABLE tracks (id int(11) NOT NULL default 0,"
                                                     "path varchar(255) default NULL, "
                                                     "filename varchar(255) default NULL, "
                                                     "medium int(11) default NULL, "
                                                     "lastModified datetime default NULL, "
                                                     "hasChanged tinyint(4) default NULL, "
                                                     "mimetype int(11) default NULL, "
                                                     "version int(11) default NULL, "
                                                     "layer int(11) default NULL, "
                                                     "mode int(11) default NULL, "
                                                     "bitrate int(11) default NULL, "
                                                     "samplerate int(11) default NULL, "
                                                     "length int(11) default NULL, "
                                                     "size int(11) default NULL, "
                                                     "artist varchar(255) default NULL, "
                                                     "title varchar(255) default NULL, "
                                                     "lyrics_id int(11) default NULL, "
                                                     "album varchar(255) default NULL, "
                                                     "tracknumber int(11) default NULL, "
                                                     "year varchar(255) default NULL, "
                                                     "genre int(11) default NULL, "
                                                     "comment varchar(255) default NULL, "
                                                     "notes varchar(255) default NULL, "
                                                     "rating int(11) default 3, "
                                                     "PRIMARY KEY (id), "
                                                     "INDEX artist_index (artist), "
                                                     "INDEX album_index (album), "
                                                     "INDEX medium_index (medium), "
                                                     "INDEX comment_index (comment)"
                                                    ") TYPE=MyISAM;");
                if(!query.isActive()) setError(1, query);
    		} else {
                if ( !checkIndex( "tracks", "artist_index" ) ) {  // creates index if missing  
    		      query.exec("ALTER TABLE tracks ADD INDEX artist_index (artist);");
                  if(!query.isActive()) setError(1, query);
                }
                if ( !checkIndex( "tracks", "album_index" ) ) {  // creates index if missing  
                  query.exec("ALTER TABLE tracks ADD INDEX album_index (album);");
                  if(!query.isActive()) setError(1, query);
                }
                if ( !checkIndex( "tracks", "medium_index" ) ) {  // creates index if missing  
                  query.exec("ALTER TABLE tracks ADD INDEX medium_index (medium);");
                  if(!query.isActive()) setError(1, query);
                }
                if ( !checkIndex( "tracks", "comment_index" ) ) {  // creates index if missing  
                  query.exec("ALTER TABLE tracks ADD INDEX comment_index (comment);");
                  if(!query.isActive()) setError(1, query);
                }
                if(!checkVersion()) {
                  // fix 0.9.3beta2 bug
                  query.exec("UPDATE tracks SET comment = '" DRAMA "' WHERE comment = 'Drama';");
                  if(!query.isActive()) {
                    setError(1, query);
                  }
                }
            }
    	    if(!playlists_exists) {
    		    q4 = query.exec("CREATE TABLE playlists (id int(11) NOT NULL default 0, name varchar(255) default NULL, PRIMARY KEY (id)) TYPE=MyISAM;");
                if(!query.isActive()) setError(1, query);
    	    }
    	    if(!playlist_tracks_exists) {
        		q5 = query.exec("CREATE TABLE playlist_tracks (id int(11) NOT NULL auto_increment, track_id int(11) default NULL, playlist_id int(11) default NULL, position_in_playlist int(11) default NULL, PRIMARY KEY(id)) TYPE=MyISAM;");
                if(!query.isActive()) setError(1, query);
    	    }
    	    if(!mextras_exists) {
    		    query.exec("CREATE TABLE mextras (id int(11) NOT NULL auto_increment, data longblob default NULL, PRIMARY KEY(id)) TYPE=MyISAM;");
                if(!query.isActive()) setError(1, query);
    	    }
            
    	    if (!checkVersion()) {  // tag new prokyon3 version 
                QString sqlquery;
                sqlquery.sprintf("INSERT INTO version_history (version, date) VALUES ('%s', NOW());", VERSION);
                query.exec(sqlquery);
                if(!query.isActive()) setError(1, query);
    	    }
    
    	    if(!checkFieldExists("version_history", "encoding")) {   // table created with <=0.9.2
    	        query.exec("ALTER TABLE version_history ADD encoding VARCHAR(32) default 'Latin1/ISO-8859-1';");
                if(!query.isActive()) setError(1, query);
    	    }

	    if ( !config->db_encoding.isNull() ) {   // New database created from wizard
	      stringstream sqlquery;
	      sqlquery << "UPDATE version_history SET encoding = '" 
		       << config->db_encoding.ascii() << "' WHERE version = '"
		       << VERSION << "';";
	      if(!query.exec(sqlquery.str().c_str())) {
		setError(1, query);
		return;
	      }
	    } else {
	      stringstream sqlquery;
	      sqlquery << "SELECT encoding FROM version_history WHERE version = '"
		       << VERSION << "';";
	      query.exec(sqlquery.str().c_str());
	      if ( query.isActive() && query.next() ) {
		config->db_encoding = query.value(0).toString();
	      } else {
		setError(1, query);
		return;
	      }
	    }

    	    if(!checkExtra("playlist_tracks", "id", "auto_increment")) {
    	        query.exec("ALTER TABLE playlist_tracks CHANGE id id INT(11) NOT NULL AUTO_INCREMENT;");
                if(!query.isActive()) setError(1, query);
    	    }
    
    	    if(!checkFieldExists("artists", "image_id")) {
    	        query.exec("ALTER TABLE artists ADD image_id INT(11) default NULL;");
                if(!query.isActive()) setError(1, query);
    	    }
    
            if(!checkFieldExists("artists", "biography_id")) {
                query.exec("ALTER TABLE artists ADD biography_id INT(11) default NULL;");
                if(!query.isActive()) setError(1, query);
            }
    
            if(!checkFieldExists("artists", "years_active")) {
                query.exec("ALTER TABLE artists ADD years_active VARCHAR(255) default NULL;");
                if(!query.isActive()) setError(1, query);
            }
    
    	    if(!checkFieldExists("tracks", "lyrics_id")) {
    	        query.exec("ALTER TABLE tracks ADD lyrics_id INT(11) default NULL;");
                if(!query.isActive()) setError(1, query);
    	    }
    
            if(!checkFieldExists("tracks", "notes")) {
                query.exec("ALTER TABLE tracks ADD notes VARCHAR(255) default NULL;");
                if(!query.isActive()) setError(1, query);
            }
    
            if(!checkFieldExists("tracks", "rating")) {
                query.exec("ALTER TABLE tracks ADD rating TINYINT(2) default 3;");
                if(!query.isActive()) setError(1, query);
            }
            
    	    if(q1 && q2 && q3 && q4 && q5 && q6 && q7) {
         		maxID_artists   = getMaxIDFromTable("artists");
        		maxID_tracks    = getMaxIDFromTable("tracks");
        		maxID_mediums   = getMaxIDFromTable("mediums");
        		maxID_playlists = getMaxIDFromTable("playlists");
        		maxID_albums    = getMaxIDFromTable("albums");
        		deltamap = new QMap<int, QMap<int, DELTAINFO*> >;
    	    }
        } else {
            setError(1, query);
        }
	if (verbose) qWarning( "MySQL encoding: %s", config->db_encoding.ascii() );
    } else { // if(db->open())
       error = 1;
    }
    
    if(error) {
        isConnected = false;
        cout << _("ERROR CONNECTION") << endl;
    } else {
       loadCaches();
       emit connected(true);
    }
    
}

// ##############################################
// # load caches
// ##############################################
void DataBase_MySQL::loadCaches() {
   if(isConnected) {
  
#ifdef EMBEDDED
        qts::QSqlQuery query("SELECT id, name, total FROM artists;", db);
#else
        QSqlQuery query("SELECT id, name, total FROM artists;", db);
#endif // ifdef EMBEDDED

        if(query.isActive()) {
            while(query.next()) {
                baselistArtists[query.value(0).toInt()] = query.value(2).toInt();
                artistToID[toLocaleString( query.value(1) )] = query.value(0).toInt();
            }
        } else {
            setError(1, query);
        }
        
        if(query.exec("SELECT id, artist, name, tracks_available, cover_id FROM albums;") && query.isActive()) {
            qApp->lock(); // avoid async during pixmap creation
            while(query.next()) {
                baselistArtists[query.value(0).toInt()] = query.value(3).toInt();
                albumToID[toLocaleString( query.value(1) ) + toLocaleString( query.value(2) )] = query.value(0).toInt();
#ifdef HAVE_MEXTRAS
                if(query.value(4).toInt() > 0) {
                    QImage image(getByteArrayMextras(query.value(4).toInt()));
                    albumImageThumbnails[toLocaleString( query.value(1) ) + toLocaleString( query.value(2) )] 
		                = new QPixmap(image.scale(32,32));
                }
#endif /* HAVE_MEXTRAS */
            }
            qApp->unlock();
        } else {
            setError(1, query);
        }
        
#ifdef HAVE_MEXTRAS
        if(query.exec("SELECT DISTINCT comment, album, cover_id FROM tracks, albums WHERE comment IN ('" SAMPLER "','" SOUNDTRACK "','" MUSICAL "','" COMEDY "','" DRAMA "') AND tracks.album=albums.name AND tracks.artist=albums.artist;") && query.isActive()) {
            qApp->lock(); // avoid async during pixmap creation
            while(query.next()) {
                if(query.value(2).toInt() > 0) {
                    QImage image(getByteArrayMextras(query.value(2).toInt()));
                    albumImageThumbnails[_(toLocaleString( query.value(0) ).local8Bit().data()) + toLocaleString( query.value(1) )] 
                        = new QPixmap(image.scale(32,32));
                }
            }
            qApp->unlock();
        } else {
            setError(1, query);
        }
#endif /* HAVE_MEXTRAS */
        
        qApp->lock();
#ifdef HAVE_MEXTRAS
        defaultAlbumThumbnail = new QPixmap(pixmapcache->getFaded("lvi_album.png"));
#else        
        defaultAlbumThumbnail = new QPixmap(pixmapcache->get("/lvi_album16x16.png"));
#endif /* HAVE_MEXTRAS */
        qApp->unlock();
   }
}

QPixmap* DataBase_MySQL::getAlbumThumbnail(const QString artist, const QString album)
{
    if(albumImageThumbnails.contains(artist + album)) {
        return albumImageThumbnails[artist + album];
    } else {
        return defaultAlbumThumbnail;
    }
}

// ##############################################
// # creates new database
// ##############################################
QString DataBase_MySQL::createNewDatabase(QString host, QString dbname, QString user, QString pass, QString su, QString su_pass, QString clienthost)
{

#ifdef EMBEDDED
    MYSQL *db = mysql_init(NULL);
  if (!db) return("mysql_init failed!!!!");

  if (!mysql_real_connect(db, NULL, NULL, NULL, NULL, 0, NULL, 0))
    return(QString("EMBEDDED SERVER: mysql_real_connect failed: ") + mysql_error(db));

  if (mysql_query(db, "CREATE DATABASE prokyon"))  
    return(QString("EMBEDDED SERVER: db_do_query failed: ") + mysql_error(db));  

  mysql_close(db);
  return "";
#else
  QSqlDatabase *dbase = QSqlDatabase::addDatabase("QMYSQL3", "newdb");
  dbase->setHostName(host);
  dbase->setDatabaseName("mysql");
  dbase->setUserName(su);
  dbase->setPassword(su_pass);
  if(dbase->open())
    {
      if("localhost" != clienthost) {
          // TODO: replace localhost by real hostname or IP if database
          // is located on another host.
          MessageEvent* e = new MessageEvent(_("Warning: Settings for connections to other hosts than localhost isn't safely implemented yet!\nYou can't rely on reported SUCCESS, but it should work :-("));
          QApplication::postEvent( gui, e );
      }
      QSqlQuery query("CREATE DATABASE " + dbname + ";", dbase);
      query.exec("GRANT USAGE ON * TO " + user + "@" + clienthost + " IDENTIFIED BY '" + pass + "' ;");
      query.exec("GRANT ALTER, CREATE, DELETE, INDEX, INSERT, SELECT, UPDATE ON " + dbname + ".* TO " + user + "@" + clienthost + ";");
      query.exec("FLUSH PRIVILEGES;");
      dbase->close();
      QSqlDatabase::removeDatabase("newdb");
      if(query.lastError().type() == QSqlError::None) return "";
      else return query.lastError().databaseText();
    }
  else
    {
      if(dbase->lastError().type() != QSqlError::None) {
	MessageEvent* e = new MessageEvent(dbase->lastError().driverText() + "\n" + dbase->lastError().databaseText() );
	QApplication::postEvent( gui, e );
      }    
      QSqlDatabase::removeDatabase("newdb");
      return _("Cannot connect to server (maybe wrong user/pass ?!)");
    }
#endif   // ifdef EMBEDDED 
}


// ##############################################
// # Get the base sampler list
// ##############################################
QList<ARTISTALBUM> *DataBase_MySQL::getAlbumBasis()
{
    QString sqlquery;
    sqlquery.sprintf("SELECT comment,album,count(*),count(NULLIF(medium=0,0)),0,0,medium FROM tracks WHERE comment IN ('%s','%s','%s','%s','%s') GROUP BY comment, BINARY album;", SAMPLER ,SOUNDTRACK, MUSICAL, COMEDY, DRAMA);
    return getArtistAlbumBasisByQuery(sqlquery);
}

// ##############################################
// # Get the base favourites list
// ##############################################
QList<ARTISTALBUM> *DataBase_MySQL::getFavouritesBasis()
{
    return getArtistAlbumBasisByQuery("SELECT artist,album,count(*),count(NULLIF(medium=0,0)),isFavourite,artists.id,medium FROM tracks,artists WHERE isFavourite=1 AND BINARY artist=name GROUP BY BINARY artist, BINARY album;");
}

// ##############################################
// # Get the base artist and album list
// ##############################################
QList<ARTISTALBUM> *DataBase_MySQL::getArtistAlbumBasis()
{
    return getArtistAlbumBasisByQuery("SELECT artist,album,count(*),count(NULLIF(medium=0,0)),isFavourite,artists.id,medium FROM tracks,artists WHERE BINARY artist=name GROUP by BINARY artist, BINARY album;");
}

QList<ARTISTALBUM> *DataBase_MySQL::getArtistAlbumBasisByQuery(QString sql)
{
    QList<ARTISTALBUM>  *list;

#ifdef EMBEDDED
    qts::QSqlQuery query(sql, db);
#else
    QSqlQuery query(sql, db);
#endif   // ifdef EMBEDDED

    if(query.isActive())
    {
        list = new QList<ARTISTALBUM>;
        while(query.next())
        {
            ARTISTALBUM *item = new ARTISTALBUM();
            item->artist      = toLocaleString( query.value(0) );
            item->album       = toLocaleString( query.value(1) );
            item->total       = query.value(2).toInt();
            item->local       = query.value(3).toInt();
            item->isFavourite = query.value(4).toBool();
            item->artistId    = query.value(5).toInt();
            item->medium      = query.value(6).toInt();
            list->append(item);
        }
        return list;
    }

    setError(1, query);
    return 0;
}

// ##############################################
// # Get the base medium list
// ##############################################
QList<MEDIUM> *DataBase_MySQL::getMediumBasis()
{
  QList<MEDIUM>	*list;
  
#ifdef EMBEDDED
  qts::QSqlQuery query("SELECT id, type, label, path, checksum FROM mediums;", db);
#else
  QSqlQuery query("SELECT id, type, label, path, checksum FROM mediums;", db);
#endif   // ifdef EMBEDDED
  
  if(query.isActive()) {
    list = new QList<MEDIUM>;
    MEDIUM *item = new MEDIUM();
    item->id = 0;
    item->type = MEDIUM_HARDDISK;
    item->label = "Harddisk";
    item->path = "";
    item->checksum = 0;
    list->append(item);
    while(query.next()) {
      MEDIUM *item      = new MEDIUM();
      item->id          = query.value(0).toInt();
      item->type        = query.value(1).toInt();
      item->label       = toLocaleString( query.value(2) );
      item->path        = toLocaleString( query.value(3) );
      item->checksum    = query.value(4).toInt();
      list->append(item);
    }
    return list;
  }
  
  setError(1, query);
  return 0;
}

// ##############################################
// # Get the base playlist list
// ##############################################
QList<PLAYLIST> *DataBase_MySQL::getPlaylistBasis()
{
  QList<PLAYLIST> *list;
  
#ifdef EMBEDDED
  qts::QSqlQuery query("SELECT id, name FROM playlists ORDER BY name;", db);
#else
  QSqlQuery query("SELECT id, name FROM playlists ORDER BY name;", db);
#endif   // ifdef EMBEDDED
  
  if(query.isActive()) {
    list = new QList<PLAYLIST>;
    while(query.next()) {
      PLAYLIST *item = new PLAYLIST();
      item->id       = query.value(0).toInt();
      item->name     = toLocaleString( query.value(1) );
      list->append(item);
    }
    return list;
  }
  
  setError(1, query);
  return 0;
}

// ##############################################
// # Get the base playlist tracks list
// ##############################################
QList<PLAYLIST_TRACK> *DataBase_MySQL::getPlaylistTracksBasis(int playlistID)
{
  QList<PLAYLIST_TRACK> *list = new QList<PLAYLIST_TRACK>;
  
  if(0 == playlistID) return list;

#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  stringstream sqlquery;
  sqlquery << "SELECT t1.id, t1.track_id, t2.medium, t1.playlist_id, t1.position_in_playlist, t2.artist, t2.title, t2.path, t2.filename, t2.size, t2.length"
	  	   << " FROM playlist_tracks AS t1, tracks AS t2"
		   << " WHERE t1.playlist_id = " << playlistID << " AND t1.track_id = t2.id"
		   << " ORDER BY t1.position_in_playlist;";
  query.exec(sqlquery.str().c_str());
  if(query.isActive()) {
    list = new QList<PLAYLIST_TRACK>;
    while(query.next()) {
      PLAYLIST_TRACK *item = new PLAYLIST_TRACK();
      item->id                   = query.value(0).toInt();
      item->track_id             = query.value(1).toInt();
      item->medium_id		     = query.value(2).toInt();
      item->playlist_id          = query.value(3).toInt();
      item->position_in_playlist = query.value(4).toInt();
      item->artist               = toLocaleString( query.value(5) );
      item->title                = toLocaleString( query.value(6) );
      item->path                 = toLocaleString( query.value(7) );
      item->filename             = toLocaleString( query.value(8) );
      item->size                 = query.value(9).toInt();
      item->length               = query.value(10).toInt();
      list->append(item);
    }
    return list;
  }
  
  setError(1, query);
  return list;
}


// ##############################################
// # Generate a playlist
// ##############################################
void DataBase_MySQL::generatePlaylist(int playlistID, int numberTracks, int minTrackLength, int maxTrackLength, bool onlyFavouriteArtist, bool onlyHarddisk, int minRating)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  stringstream sqlquery;
  sqlquery << "INSERT INTO playlist_tracks (track_id,playlist_id,position_in_playlist) "
           << "SELECT DISTINCT tracks.id," << playlistID << ",0 "
           << "FROM tracks "
           << ((onlyFavouriteArtist) ? "LEFT JOIN artists ON BINARY artist=name " : "")
           << "WHERE length BETWEEN " << minTrackLength << " AND " << maxTrackLength << " "
           << ((onlyHarddisk) ? "AND medium=0 " : "")
           << ((onlyFavouriteArtist) ? "AND isFavourite=1 " : "");
  if(minRating > 0) sqlquery << "AND rating>" << minRating << " ";
  sqlquery << "ORDER BY RAND() "
           << "LIMIT 0," << numberTracks << ";";
  query.exec(sqlquery.str().c_str());
  if(!query.isActive()) {
    setError(1, query);
  }
}

// ##############################################
// # Test presence of an index
// ##############################################
bool DataBase_MySQL::checkIndex(const QString& table, const QString& indexname)
{
  QString sqlquery;
  sqlquery.sprintf("SHOW INDEX FROM %s", table.ascii() );
#ifdef EMBEDDED
  qts::QSqlQuery query(sqlquery, db);
#else
  QSqlQuery query(sqlquery, db);
#endif   // ifdef EMBEDDED

  if(query.isActive()) {
    while(query.next()) {
      if (query.value(2).toString() == indexname) return true;
    }
    return false;
  } else {
    setError(1, query);
    return false;
  }
}

// ##############################################
// # Test presence of an additional feature of a column of a table (do not relate to Musicextras....)
// ##############################################
bool DataBase_MySQL::checkExtra(const QString& table, const QString& column, const QString& extra)
{
  QString sqlquery;
  sqlquery.sprintf("DESC %s", table.ascii());
#ifdef EMBEDDED
  qts::QSqlQuery query(sqlquery, db);
#else
  QSqlQuery query(sqlquery, db);
#endif   // ifdef EMBEDDED

  if(query.isActive()) {
    while(query.next()) {
      if (query.value(0).toString() == column &&
          query.value(5).toString() == extra) return true;
    }
    return false;
  } else {
    setError(1, query);
    return false;
  }
}

// ##############################################
// # Test presence of an additional feature of a column of a table (do not relate to Musicextras....)
// ##############################################
bool DataBase_MySQL::checkFieldExists(const QString& table, const QString& column)
{
  QString sqlquery;
  sqlquery.sprintf("DESC %s", table.ascii());
#ifdef EMBEDDED
  qts::QSqlQuery query(sqlquery, db);
#else
  QSqlQuery query(sqlquery, db);
#endif   // ifdef EMBEDDED

  if(query.isActive()) {
    while(query.next()) {
      if (query.value(0).toString() == column) return true;
    }
    return false;
  } else {
    setError(1, query);
    return false;
  }
}

// ##############################################
// # Test column type
// ##############################################
bool DataBase_MySQL::checkType(const QString& table, const QString& column, const QString& type)
{
  QString sqlquery;
  sqlquery.sprintf("DESC %s", table.ascii());
#ifdef EMBEDDED
  qts::QSqlQuery query(sqlquery, db);
#else
  QSqlQuery query(sqlquery, db);
#endif   // ifdef EMBEDDED

  if(query.isActive()) {
    while(query.next()) {
      if (query.value(0).toString() == column &&
          query.value(1).toString() == type) return true;
    }
    return false;
  } else {
    setError(1, query);
    return false;
  }
}

// ##############################################
// # Check if database layout fits current version
// ##############################################
bool DataBase_MySQL::checkVersion()
{
  QString sqlquery;
  sqlquery.sprintf("SELECT date FROM version_history WHERE version='%s'", VERSION);
#ifdef EMBEDDED
  qts::QSqlQuery query(sqlquery, db);
#else
  QSqlQuery query(sqlquery, db);
#endif   // ifdef EMBEDDED

  if(query.isActive()) {
    if(query.next()) {
      return true;
    }
    return false;
  } else {
    setError(1, query);
    return false;
  }
}

// ##############################################
// # Returns the deltaMap and creates a new one
// ##############################################
QMap<int, QMap<int, DELTAINFO*> > *DataBase_MySQL::getArtistDelta()
{
  QMap<int, QMap<int, DELTAINFO*> > *t = deltamap;	
  deltamap = new QMap<int, QMap<int, DELTAINFO*> >;
  return t;
}

// ##############################################
// # Returns the deltaMap for non local media
// ##############################################
QMap<int, QMap<int, int> > *DataBase_MySQL::getLocalDeltaBasis()
{
  QMap<int, QMap<int, int> > *deltalocal = new QMap<int, QMap<int, int> >;
 
#ifdef EMBEDDED
  qts::QSqlQuery query("SELECT id FROM mediums WHERE type = 1 OR type = 2 OR type = 4;", db); // CDROM oder network
#else
  QSqlQuery query("SELECT id FROM mediums WHERE type = 1 OR type = 2 OR type = 4;", db); // CDROM oder network
#endif   // ifdef EMBEDDED
  
  if(query.isActive()) {
    while(query.next()) {
      int medium = query.value(0).toInt();
#ifdef EMBEDDED
      qts::QSqlQuery subquery("SELECT artist FROM tracks WHERE medium = " + toLocaleString( query.value(0) ) + ";", db);
#else
      QSqlQuery subquery("SELECT artist FROM tracks WHERE medium = " + toLocaleString( query.value(0) ) + ";", db);
#endif   // ifdef EMBEDDED
      if(subquery.isActive()) {
		while(subquery.next()) {
	  	  if((*deltalocal)[medium].contains(artistToID[toLocaleString( subquery.value(0) )])) (*deltalocal)[medium][artistToID[toLocaleString( subquery.value(0) )]]++;
	  	  else (*deltalocal)[medium][artistToID[toLocaleString( subquery.value(0) )]] = 1;
		}
      } else {
		setError(1, subquery);
		return 0;
      }
    }
  } else {
    setError(1, query);
    return 0;
  }
  return deltalocal;
}

QMap<int, QMap<QString, int> > *DataBase_MySQL::getLocalAlbumDeltaBasis()
{
  QMap<int, QMap<QString, int> > *deltalocal = new QMap<int, QMap<QString, int> >;
 
#ifdef EMBEDDED
  qts::QSqlQuery query("SELECT id FROM mediums WHERE type = 1 OR type = 2 OR type = 4;", db); // CDROM oder network
#else
  QSqlQuery query("SELECT id FROM mediums WHERE type = 1 OR type = 2 OR type = 4;", db); // CDROM oder network
#endif   // ifdef EMBEDDED
  
  if(query.isActive()) {
    while(query.next()) {
      int medium = query.value(0).toInt();
#ifdef EMBEDDED
      qts::QSqlQuery subquery("SELECT artist,album FROM tracks WHERE medium = " + QString::number( medium ) + ";", db);
#else
      QSqlQuery subquery("SELECT artist,album FROM tracks WHERE medium = " + toLocaleString( query.value(0) ) + ";", db);
#endif   // ifdef EMBEDDED
      if(subquery.isActive()) {
		while(subquery.next()) {
	  	  if((*deltalocal)[medium].contains(toLocaleString( subquery.value(0) )+'\0'+ toLocaleString( subquery.value(1) ))) 
		    (*deltalocal)[medium][toLocaleString( subquery.value(0) )+'\0'+ toLocaleString( subquery.value(1) )]++;
	  	  else (*deltalocal)[medium][toLocaleString( subquery.value(0) )+'\0'+ toLocaleString( subquery.value(1) )] = 1;
	  	  if((*deltalocal)[medium].contains(toLocaleString( subquery.value(0) ))) 
		    (*deltalocal)[medium][toLocaleString( subquery.value(0) )]++;
	  	  else (*deltalocal)[medium][toLocaleString( subquery.value(0) )] = 1;
		}
      } else {
		setError(1, subquery);
		return 0;
      }
    }
  } else {
    setError(1, query);
    return 0;
  }
  return deltalocal;
}

// ##############################################
// # Returns the deltaMap for one non-local media
// ##############################################
QMap<int, int> DataBase_MySQL::getLocalDelta(int id)
{
  QMap<int, int> delta;

  QString sqlquery;
  sqlquery.sprintf("SELECT artist FROM tracks WHERE medium = %d;", id);
#ifdef EMBEDDED
  qts::QSqlQuery query(sqlquery, db);
#else
  QSqlQuery query(sqlquery, db);
#endif   // ifdef EMBEDDED

  if(query.isActive()) {
    while(query.next()) {
      if(delta.contains(artistToID[toLocaleString( query.value(0) )])) delta[artistToID[toLocaleString( query.value(0) )]]++;
      else delta[artistToID[toLocaleString( query.value(0) )]] = 1;
    }
  } else setError(1, query);
  return delta;
}

QMap<QString, int> DataBase_MySQL::getLocalAlbumDelta(int id)
{
  QMap<QString, int> delta;

  QString sqlquery;
  sqlquery.sprintf("SELECT artist,album FROM tracks WHERE medium = %d;", id);
#ifdef EMBEDDED
  qts::QSqlQuery query(sqlquery, db);
#else
  QSqlQuery query(sqlquery, db);
#endif   // ifdef EMBEDDED

  if(query.isActive()) {
    while(query.next()) {
      if(delta.contains(toLocaleString( query.value(0) )+'\0'+ toLocaleString( query.value(1) ))) 
	delta[toLocaleString( query.value(0) )+'\0'+ toLocaleString( query.value(1) )]++;
      else delta[toLocaleString( query.value(0) )+'\0'+ toLocaleString( query.value(1) )] = 1;
      if(delta.contains( toLocaleString( query.value(0) ))) 
	delta[toLocaleString( query.value(0) )]++;
      else delta[toLocaleString( query.value(0) )] = 1;
    }
  } else setError(1, query);
  return delta;
}

// ##############################################
// # append tracks to database
// ##############################################
int DataBase_MySQL::appendTracks(QList<TRACK> *tracklist, int mediumtype, QString mediumlabel, QString mediumpath, unsigned mediumchecksum, bool createNewMedium, int mediumID)
{
  int incrLocal, medium, track;
  
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  
  switch(mediumtype) {
  case MEDIUM_HARDDISK:
    incrLocal = 1;
    medium    = 0;
    break;
  case MEDIUM_CDROM:
  case MEDIUM_SMB:
  case MEDIUM_NFS:
  case MEDIUM_EXTERN:
    incrLocal = 0;
    if(createNewMedium) {
      medium    = ++maxID_mediums;
      stringstream sqlquery;
      sqlquery << "INSERT INTO mediums (id, type, label, path, checksum) VALUES ("
	       << medium << ", "
	       << mediumtype << ", '"
	       << adjustString(mediumlabel) << "', '"
	       << adjustString(mediumpath) << "', "
	       << mediumchecksum << ");";
	  query.exec(sqlquery.str().c_str());
      if(!query.isActive()) {
    	setError(1, query);
    	return 0;
      }
    } else {
      medium = mediumID;
    }
  }
  
  
  TRACK *curr = tracklist->first();
  for(; curr != 0; curr = tracklist->next()) {
    track = ++maxID_tracks;
    stringstream sqlquery;
    sqlquery << "INSERT INTO tracks (id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment) VALUES ("
	     << track << ", '"
	     << adjustString(curr->path) << "', '"
	     << adjustString(curr->filename) << "', "
	     << medium << ", '"
	     << dateTimeToString(curr->lastModified) << "', "
	     << (curr->hasChanged ? 1 : 0) << ", "
	     << curr->mimetype << ", "
	     << curr->version << ", "
	     << curr->layer << ", "
	     << curr->mode << ", "
	     << curr->bitrate << ", "
	     << curr->samplerate << ", "
	     << curr->length << ", "
	     << curr->size << ", '"	
	     << adjustString(curr->artist) << "', '"
	     << adjustString(curr->title) << "', '"
	     << adjustString(curr->album) << "', "
	     << curr->tracknumber << ", '"
	     << adjustString(curr->year) << "', "
	     << curr->genre << ", '"
	     << adjustString(curr->comment) << "');";
	query.exec(sqlquery.str().c_str());
    if(!query.isActive()) {
      setError(1, query);
      return 0;
    }
    
    int artist;
    stringstream sqlquery2;
    if(!artistToID.contains(curr->artist)) {
      artist = ++maxID_artists;
      artistToID[curr->artist] = artist;
      baselistArtists[artist] = 1;
      sqlquery2 << "INSERT INTO artists (id, name, total, local, isFavourite) VALUES ("
		<< artist << ", '" << adjustString(curr->artist) << "', 1, " << incrLocal << ", 0);";
    } else {
      artist = artistToID[curr->artist];
      ++baselistArtists[artist];
      sqlquery2 << "UPDATE artists SET total=total+1, local=local+" << incrLocal << " WHERE id = " << artist << ";";
    }
    query.exec(sqlquery2.str().c_str());
    if(!query.isActive()) {
      setError(1, query);
      return 0;
    }

    int album;
    stringstream sqlquery3;
    if(!albumToID.contains(curr->artist + curr->album)) {
      album = ++maxID_albums;
      albumToID[curr->artist + curr->album] = album;
      baselistAlbums[album] = 1;
      sqlquery3 << "INSERT INTO albums (id, name, artist, tracks_available) VALUES ("
       << album << ", '" << adjustString(curr->album) << "', '" << adjustString(curr->artist) << "', 1);";
    } else {
      album = albumToID[curr->artist + curr->album];
      ++baselistAlbums[album];
      sqlquery3 << "UPDATE albums SET tracks_available=tracks_available+1 WHERE id = " << album << ";";
    }
    query.exec(sqlquery3.str().c_str());
    if(!query.isActive()) {
      setError(1, query);
      return 0;
    }

    if(!deltamap->contains(artist) || !(*deltamap)[artist].contains(medium)) (*deltamap)[artist][medium] = new DELTAINFO(1, incrLocal, curr->artist);
    else {
      (*deltamap)[artist][medium]->total++;
      if(incrLocal != 0) (*deltamap)[artist][medium]->local++;
    }

    QMap< QString, ALBUMINFO > *deltaalbum = (*deltamap)[artist][medium]->deltaalbum;
    if ( !deltaalbum->contains( curr->album ) ) (*deltaalbum)[curr->album] = ALBUMINFO( +1, +incrLocal, curr->album );
    else {
      (*deltaalbum)[curr->album].total++;
      if (incrLocal != 0) (*deltaalbum)[curr->album].local++;
    };
  }
  if ( verbose == 5 )  {
     qWarning( "DataBase_MySQL::appendTracks" );  
     dumpdeltamap( deltamap );  
  }
  return medium;
}


// ##############################################
// # update tracks in database
// ##############################################
void DataBase_MySQL::updateTracks(QList<TRACK> *tracklist)
{
    // *********************************************************************
    // *********************************************************************
    // *********************************************************************
    // This method really needs a transaction context!
    // Considering to offer it, if InnoDB is available
    // *********************************************************************
    // *********************************************************************
    // *********************************************************************
  
    QString origArtist("");
    QString origAlbum("");
    QString origTitle("");
    QString origYearsActive("");
    QString origComment("");
    int origImageId = 0;
    int origBiographyId = 0;
    int origLyricsId = 0;
    int origCoverId = 0;
    int origAlbumReviewId = 0;
    int origAlbumTracksId = 0;
    
#ifdef EMBEDDED
    qts::QSqlQuery query;
#else
    QSqlQuery query;
#endif   // ifdef EMBEDDED
  
    TRACK *curr = tracklist->first();
    for(; curr != 0; curr = tracklist->next()) {
        stringstream sqlquery;
        sqlquery << "SELECT tracks.artist, tracks.album, title, image_id, years_active, biography_id, lyrics_id, cover_id, album_review_id, album_tracks_id, comment"
            << " FROM tracks, artists, albums WHERE tracks.id = " << curr->id
            << " AND BINARY tracks.artist = artists.name"
            << " AND BINARY tracks.artist = albums.artist"
            << " AND BINARY tracks.album  = albums.name;";
        if(query.exec(sqlquery.str().c_str())) {
            if(query.next()) {
                origArtist        = toLocaleString( query.value(0) );
                origAlbum         = toLocaleString( query.value(1) );
                origTitle         = toLocaleString( query.value(2) );
                origImageId       = query.value(3).toInt();
                origYearsActive   = toLocaleString( query.value(4) );
                origBiographyId   = query.value(5).toInt();
                origLyricsId      = query.value(6).toInt();
                origCoverId       = query.value(7).toInt();
                origAlbumReviewId = query.value(8).toInt();
                origAlbumTracksId = query.value(9).toInt();
                origComment       = toLocaleString( query.value(10) );
            }
        } else {
            setError(1, query);
            return;
        }

        stringstream sqlquery2;
        sqlquery2 << "UPDATE tracks SET" 
	      << " path = '" << adjustString(curr->path) << "',"
	      << " filename = '" << adjustString(curr->filename) << "',"
	      << " medium = " << curr->medium << ","
	      << " lastModified = '" << dateTimeToString(curr->lastModified) << "',"
	      << " hasChanged = " << (curr->hasChanged ? 1 : 0) << ","
	      << " mimetype = " << curr->mimetype << ","
	      << " version = " << curr->version << ","
	      << " layer = " << curr->layer << ","
	      << " mode = " << curr->mode << ","
	      << " bitrate = " << curr->bitrate << ","
	      << " samplerate = " << curr->samplerate << ","
	      << " length = " << curr->length << ","
	      << " size = " << curr->size << ","
	      << " artist = '" << adjustString(curr->artist) << "',"
	      << " title = '" << adjustString(curr->title) << "',"
	      << " album = '" << adjustString(curr->album) << "',"
	      << " tracknumber = " << curr->tracknumber << ", "
	      << " year = '" << adjustString(curr->year) << "',"
	      << " genre = " << curr->genre << ","
	      << " comment = '" << adjustString(curr->comment) << "',"
          << " notes = '" << adjustString(curr->notes) << "',"
	      << " rating = " << curr->rating;
        if(curr->title != origTitle && origLyricsId) {
            qApp->lock();
            int keepExtraData = QMessageBox::question(NULL, 
                                        _("Question"),
                                        QString(_("You changed the tracks's title from '<b>%1</b>' to '<b>%2</b>'.\n"
                                                  "There are lyrics stored for '<b>%3</b>' in database.\n"
                                                  "Would you associate these lyrics to '<b>%4</b>'.")).arg(origTitle).arg(curr->title).arg(origTitle).arg(curr->title),
                                        QMessageBox::Yes, QMessageBox::No);
            qApp->unlock();
            if(keepExtraData == QMessageBox::No) {
                sqlquery2 << ", lyrics_id = NULL";
            }
        }
        sqlquery2 << " WHERE id = " << curr->id << ";";
        if(!query.exec(sqlquery2.str().c_str())) {
            setError(1, query);
            return;
        }
    
        if(curr->album != origAlbum || curr->artist != origArtist) {
            int album = albumToID[origArtist + origAlbum];
            stringstream sqlquery;
            bool removeCoverThumbnail = false;
            if(baselistAlbums[album] == 1) {
                // delete album from albums table if no more tracks with original album name remaining
                baselistAlbums.remove(album);
                albumToID.remove(origArtist + origAlbum);
                removeCoverThumbnail = true;
                sqlquery << "DELETE FROM albums WHERE id = " << album << ";";
            } else {
                // adjust number of tracks of album with original name
                --baselistAlbums[album];
                sqlquery << "UPDATE albums SET tracks_available=tracks_available-1 WHERE id = " << album << ";";
            }
            if(!query.exec(sqlquery.str().c_str())) {
               setError(1, query);
                return;
            }
            
            stringstream sqlquery2;
            if(!albumToID.contains(curr->artist + curr->album)) {
                // It's the first album having this name -> create an entry at album table
                int keepExtraData = QMessageBox::No;
                if(origCoverId || origAlbumReviewId) {
                    qApp->lock();
                    keepExtraData = QMessageBox::question(NULL, 
                                        _("Question"),
                                        QString(_("You changed the albums's title from '<b>%1</b>' to '<b>%2</b>'.\n"
                                                  "There's a cover or review associated to '<b>%3</b>' in database.\n"
                                                  "Would you keep this association for '<b>%4</b>'.")).arg(origAlbum).arg(curr->album).arg(origAlbum).arg(curr->album),
                                        QMessageBox::Yes, QMessageBox::No);
                    qApp->unlock();
                }
                album = ++maxID_albums;
                albumToID[curr->artist + curr->album] = album;
                baselistAlbums[album] = 1;
                if(keepExtraData == QMessageBox::Yes) {
                    albumImageThumbnails[curr->artist + curr->album] = albumImageThumbnails[origArtist + origAlbum];
                    NewCoverImageEvent* e = new NewCoverImageEvent(curr->artist, curr->album);
                    QApplication::postEvent( gui->getSelector(), e );
                    sqlquery2 << "INSERT INTO albums (id, name, artist, tracks_available, cover_id, album_review_id, album_tracks_id) VALUES ("
                        << album << ", '" << adjustString(curr->album) << "', '" << adjustString(curr->artist) << "', 1, " << origCoverId << ", " << origAlbumReviewId << ", " << origAlbumTracksId << ");";
                } else {
                    sqlquery2 << "INSERT INTO albums (id, name, artist, tracks_available) VALUES ("
                        << album << ", '" << adjustString(curr->album) << "', '" << adjustString(curr->artist) << "', 1);";
                }
            } else {
               // The album already exists in albums table -> adjust number of tracks
                album = albumToID[curr->artist + curr->album];
                ++baselistArtists[album];
                sqlquery2 << "UPDATE albums SET tracks_available=tracks_available+1 WHERE id = " << album << ";";
            }
            
            if(removeCoverThumbnail) albumImageThumbnails.remove(origArtist + origAlbum);
        
            if(!query.exec(sqlquery2.str().c_str())) {
             setError(1, query);
                return;
            }
        }
        
        if(   (curr->comment != origComment || curr->album != origAlbum || curr->artist != origArtist)
           && !isNormalAlbum(curr->comment)
           && albumImageThumbnails.contains(curr->artist + curr->album)) {
            // We just add another thumbnail to the map without taking care of maybe still remaining but
            // unused entries for the album tab. The waste of memory will not be so big, if we assume
            // that changes to the album type by setting it to something different than NORMAL will
            // not happen very often.
            albumImageThumbnails[albumTypeTranslation(curr->comment) + curr->album] = albumImageThumbnails[curr->artist + curr->album];
            NewCoverImageEvent* e = new NewCoverImageEvent(albumTypeTranslation(curr->comment), curr->album);
            QApplication::postEvent( gui->getSelector(), e );
        }
        
        int origArtistID, artistID;
        int incrLocal;
        if(curr->medium == MEDIUM_HARDDISK) incrLocal = 1; else incrLocal = 0;
    
        if(curr->artist != origArtist) {
            int artist = artistToID[origArtist];
            stringstream sqlquery;
            if(baselistArtists[artist] == 1) {
                // delete artist from artists table if no more tracks with original artist name remaining
                baselistArtists.remove(artist);
                artistToID.remove(origArtist);
                sqlquery << "DELETE FROM artists WHERE id = " << artist << ";";
            } else {
                // adjust number of tracks of artists with original name
                --baselistArtists[artist];
                sqlquery << "UPDATE artists SET total=total-1, local=local-" << incrLocal << " WHERE id = " << artist << ";";
            }
            if(!query.exec(sqlquery.str().c_str())) {
    	        setError(1, query);
    	        return;
            }
    
            origArtistID = artist;
    
            if(!deltamap->contains(artist) || !(*deltamap)[artist].contains(curr->medium)) (*deltamap)[artist][curr->medium] = new DELTAINFO(-1, -incrLocal, origArtist);
            else {
    	        (*deltamap)[artist][curr->medium]->total--;
    	        if(incrLocal != 0) (*deltamap)[artist][curr->medium]->local--;
            }
     
            stringstream sqlquery2;
            if(!artistToID.contains(curr->artist)) {
                // It's the first artist having this name -> create an entry at artist table
                int keepExtraData = QMessageBox::No;
                if(origImageId || origBiographyId || !origYearsActive.isEmpty()) {
                    qApp->lock();
                    keepExtraData = QMessageBox::question(NULL, 
                                        _("Question"),
                                        QString(_("You changed the artist's name from '<b>%1</b>' to '<b>%2</b>'.\n"
                                                  "There's some extra data like artist image, biography or active years\n"
                                                  "associated to '<b>%3</b>'. Would you keep this association for '<b>%4</b>?")).arg(origArtist).arg(curr->artist).arg(origArtist).arg(curr->artist),
                                        QMessageBox::Yes, QMessageBox::No);
                    qApp->unlock();
                }
                artist = ++maxID_artists;
                artistToID[curr->artist] = artist;
                baselistArtists[artist] = 1;
                if(keepExtraData == QMessageBox::Yes) {
                    sqlquery2 << "INSERT INTO artists (id, name, total, local, isFavourite, image_id, years_active, biography_id) VALUES ("
                        << artist << ", '" << adjustString(curr->artist) << "', 1, " << incrLocal << ", 0, " << origImageId << ", '" << adjustString(origYearsActive) << "', " << origBiographyId << ");";
                } else {
                    sqlquery2 << "INSERT INTO artists (id, name, total, local, isFavourite) VALUES ("
    		            << artist << ", '" << adjustString(curr->artist) << "', 1, " << incrLocal << ", 0);";
                }
            } else {
    	        // The artist already exists in artist table -> adjust number of tracks
                artist = artistToID[curr->artist];
                ++baselistArtists[artist];
                sqlquery2 << "UPDATE artists SET total=total+1, local=local+" << incrLocal << " WHERE id = " << artist << ";";
            }
        
            if(!query.exec(sqlquery2.str().c_str())) {
    	        setError(1, query);
    	        return;
            }
          
            artistID = artist;
    
            if(!deltamap->contains(artist) || !(*deltamap)[artist].contains(curr->medium)) {
                (*deltamap)[artist][curr->medium] = new DELTAINFO(1, incrLocal, curr->artist);
            } else {
    	        (*deltamap)[artist][curr->medium]->total++;
    	        if(incrLocal != 0) (*deltamap)[artist][curr->medium]->local++;
            }
        } else {
            origArtistID = artistToID[curr->artist];
            artistID = origArtistID;
        }
    
        if(  ( artistID != origArtistID) ) {
            QMap< QString, ALBUMINFO > *deltaalbum = (*deltamap)[origArtistID][curr->medium]->deltaalbum;
            if ( !deltaalbum->contains( origAlbum ) ) {
                (*deltaalbum)[origAlbum] = ALBUMINFO( -1, -incrLocal, origAlbum );
            } else {
                (*deltaalbum)[origAlbum].total--;
                if (incrLocal != 0) (*deltaalbum)[origAlbum].local--;
            }
    
            deltaalbum = (*deltamap)[artistID][curr->medium]->deltaalbum;
            if ( !deltaalbum->contains( curr->album ) ) {
                (*deltaalbum)[curr->album] = ALBUMINFO( +1, +incrLocal, curr->album );
            } else {
                (*deltaalbum)[curr->album].total++;
                if (incrLocal != 0) (*deltaalbum)[curr->album].local++;
            }
        }
    
        if ( ( origAlbum != curr->album ) && ( artistID == origArtistID ) ) {
            if(!deltamap->contains(artistID) || !(*deltamap)[artistID].contains(curr->medium)) {
                (*deltamap)[artistID][curr->medium] = new DELTAINFO(0, 0, curr->artist);
            }
            QMap< QString, ALBUMINFO > *deltaalbum = (*deltamap)[artistID][curr->medium]->deltaalbum;
            if ( !deltaalbum->contains( origAlbum ) ) {
                (*deltaalbum)[origAlbum] = ALBUMINFO( -1, -incrLocal, origAlbum );
            } else {
                (*deltaalbum)[origAlbum].total--;
                if (incrLocal != 0) (*deltaalbum)[origAlbum].local--;
            }
    
            if ( !deltaalbum->contains( curr->album ) ) {
                (*deltaalbum)[curr->album] = ALBUMINFO( +1, +incrLocal, curr->album );
            } else {
                (*deltaalbum)[curr->album].total++;
                if (incrLocal != 0) (*deltaalbum)[curr->album].local++;
            }
        }
    }
    
    if ( verbose == 5 )  {
        qWarning( "DataBase_MySQL::updateTracks" );  
        dumpdeltamap( deltamap );  
    }
}

// ##############################################
// # delete tracks out of database
// ##############################################
void DataBase_MySQL::deleteTracks(QList<TRACK> *tracklist)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  TRACK *curr = tracklist->first();
  for(; curr != 0; curr = tracklist->next()) {
    stringstream sqlquery;
    sqlquery << "DELETE FROM tracks WHERE id = " << curr->id << ";";
    if(!query.exec(sqlquery.str().c_str())) {
      setError(1, query);
      return;
    }
    
    int incrLocal;
    if(curr->medium == MEDIUM_HARDDISK) incrLocal = 1; else incrLocal = 0;

    int artist = artistToID[curr->artist];
    stringstream sqlquery2;
    if(baselistArtists[artist] == 1) {
      baselistArtists.remove(artist);
      artistToID.remove(curr->artist);
      sqlquery2 << "DELETE FROM artists WHERE id = " << artist << ";";
    } else {
      baselistArtists[artist]--;
      sqlquery2 << "UPDATE artists SET total=total-1, local=local-" << incrLocal << " WHERE id = " << artist << ";";
    }
    if(!query.exec(sqlquery2.str().c_str())) {
      setError(1, query);
      return;
    }
    
    if(!deltamap->contains(artist) || !(*deltamap)[artist].contains(curr->medium)) (*deltamap)[artist][curr->medium] = new DELTAINFO(-1, -incrLocal, curr->artist);
    else {
      (*deltamap)[artist][curr->medium]->total--;
      if(incrLocal != 0) (*deltamap)[artist][curr->medium]->local--;
    }

    QMap< QString, ALBUMINFO > *deltaalbum = (*deltamap)[artist][curr->medium]->deltaalbum;
    if ( !deltaalbum->contains( curr->album ) ) (*deltaalbum)[curr->album] = ALBUMINFO( -1, -incrLocal, curr->album );
    else {
      (*deltaalbum)[curr->album].total--;
      if (incrLocal != 0) (*deltaalbum)[curr->album].local--;
    };
  }
  if ( verbose == 5 )  {
     qWarning( "DataBase_MySQL::deleteTracks" );  
     dumpdeltamap( deltamap );  
  }
  deletePlaylistTracks(tracklist); // also delete tracks from playlist
}

int DataBase_MySQL::getPlaylistIdByName(QString listname)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  stringstream sqlquery;
  sqlquery << "SELECT id FROM playlists WHERE name = '" << adjustString(listname) << "';";
  if(!query.exec(sqlquery.str().c_str())) {
    setError(1, query);
    return -1;
  }

  if(query.size() == 0) {
    return 0;
  }
  
  if(query.size() == 1 && query.next()) {
    return query.value(0).toInt();
  }
  
  // should only have one playlist named <listname>
  error = 1;
  return -1;
}

// ##############################################
// # append new playlist to database
// ##############################################
int DataBase_MySQL::appendPlaylist(QString listname)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  if(getPlaylistIdByName(listname) > 0) {
    // should only have one playlist named <listname>
    error = 1;
    return -1;
  }

  stringstream sqlquery2;
  int playlist = ++maxID_playlists;
  sqlquery2 << "INSERT INTO playlists (id, name) VALUES ("
	  		<< playlist << ", '" << adjustString(listname) << "');";
  query.exec(sqlquery2.str().c_str());
  if(!query.isActive()) {
    setError(1, query);
    return -1;
  }

  return playlist;
}

// ##############################################
// # rename playlist
// ##############################################
void DataBase_MySQL::renamePlaylist(int id, QString newname)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  stringstream sqlquery;

  stringstream sqlquery2;
  sqlquery2 << "UPDATE playlists SET"
	  		<< " name = '" << adjustString(newname) << "'"
			<< " WHERE id = " << id << ";";
  if(!query.exec(sqlquery2.str().c_str())) {
    setError(1, query);
    return;
  }
}

// ##############################################
// # delete playlist from database
// ##############################################
void DataBase_MySQL::deletePlaylist(int id)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  stringstream sqlquery;

  sqlquery << "DELETE FROM playlists WHERE id = " << id << ";";
  if(!query.exec(sqlquery.str().c_str())) {
    setError(1, query);
    return;
  }

  sqlquery.str("");
  sqlquery << "DELETE FROM playlist_tracks WHERE playlist_id = "
	  	   << id << ";";
  if(!query.exec(sqlquery.str().c_str())) {
    setError(1, query);
    return;
  }
}

// ##############################################
// # append track(s) to a playlist
// ##############################################
int DataBase_MySQL::appendPlaylistTracks(QList<TRACK> *tracklist, int playlistID, int *position)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

//  int oldID = maxID_playlist_tracks;
  int oldID = getMaxIDFromTable("playlist_tracks");

//  int pl_track,cur_position;
  int cur_position;
  if(position != 0)
	cur_position = *position;
  else
	cur_position = -1;

  // if position is NULL, then  used the next one available  
if(cur_position < 0) {
    stringstream sqlquery;
    sqlquery << "SELECT position_in_playlist FROM playlist_tracks"
			 << " WHERE playlist_id = " << playlistID
			 << " ORDER BY position_in_playlist DESC LIMIT 1;";
    if(!query.exec(sqlquery.str().c_str())) {
      setError(1, query);
      return -1;
    }
    if(query.next())
      cur_position = query.value(0).toInt()+1;
    else
      cur_position = 1;
    if(position != 0) *position = cur_position;
  }
  
  for(TRACK *curr = tracklist->first(); curr != 0; curr = tracklist->next()) {
    stringstream sqlquery;
    sqlquery << "SELECT id FROM playlist_tracks"
			 << " WHERE playlist_id = " << playlistID << ""
			 << "   AND track_id = " << curr->id << ";";
    if(!query.exec(sqlquery.str().c_str())) {
      setError(1, query);
      return -1;
    }
    if(query.next()) continue;   // don't allow duplicates in one playlist

    stringstream sqlquery2;
    sqlquery2 << "INSERT INTO playlist_tracks (track_id, playlist_id, position_in_playlist) VALUES (" 
			  << curr->id << ", "
			  << playlistID << ", "
			  << cur_position++ << ");";
    query.exec(sqlquery2.str().c_str());
    if(!query.isActive()) {
      setError(1, query);
      return -1;
    }
  }
  return oldID;
}

// ##############################################
// # delete track(s) from a playlist
// ##############################################
void DataBase_MySQL::deletePlaylistTracks(QList<TRACK> *tracklist)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  TRACK *curr = tracklist->first();
  for(; curr != 0; curr = tracklist->next()) {
    stringstream sqlquery;
    sqlquery << "DELETE FROM playlist_tracks"
			 << " WHERE track_id = " << curr->id << ";";

    if(!query.exec(sqlquery.str().c_str())) {
      setError(1, query);
      return;
    }
  }

  stringstream sqlquery;
  sqlquery << "SELECT id FROM playlists;";
  query.exec(sqlquery.str().c_str());
  if(query.isActive()) {
    while(query.next()) {
      int playlistID = query.value(0).toInt();
#ifdef EMBEDDED
      qts::QSqlQuery query2;
#else
      QSqlQuery query2;
#endif   // ifdef EMBEDDED
      stringstream sqlquery2;
      sqlquery2 << "SELECT track_id FROM playlist_tracks"
				<< " WHERE playlist_id = " << playlistID
				<< " ORDER BY position_in_playlist;";
      query2.exec(sqlquery2.str().c_str());
      if(query2.isActive()) {
	int temppos = 1;
	while(query2.next()) {
	  int trackID = query2.value(0).toInt();
	  updatePlaylistTrack(playlistID, trackID, temppos++);
	}
      } else {
	setError(1, query2);
	return;
      }
    }
  } else {
    setError(1, query);
    return;
  }
}

void DataBase_MySQL::deletePlaylistTracks(QList<PLAYLIST_TRACK> *playlist_tracklist)
{
  if(!playlist_tracklist) return;
  int playlistID = playlist_tracklist->first()->playlist_id;
    
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  for(PLAYLIST_TRACK *curr = playlist_tracklist->first();
      curr != 0; curr = playlist_tracklist->next()) {
    stringstream sqlquery;
    sqlquery << "DELETE FROM playlist_tracks"
   			 << " WHERE playlist_id = " << playlistID
			 << "   AND track_id = " << curr->track_id << ";";
    if(!query.exec(sqlquery.str().c_str())) {
      setError(1, query);
      return;
    }
  }

  stringstream sqlquery;
  sqlquery << "SELECT track_id FROM playlist_tracks"
		   << " WHERE playlist_id = " << playlistID
		   << " ORDER BY position_in_playlist;";
  query.exec(sqlquery.str().c_str());
  if (query.isActive()) {
    int temppos = 1;
    while(query.next()) {
      int trackID = query.value(0).toInt();
      updatePlaylistTrack(playlistID, trackID, temppos++);
    }
  } else {
    setError(1, query);
    return;
  }
}

// ##############################################
// # update tracks(s) in playlist
// # (position change)
// ##############################################
void DataBase_MySQL::updatePlaylistTrack(int playlistID, int trackID, int new_pos)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  stringstream sqlquery;
  sqlquery << "UPDATE playlist_tracks SET"
		   << " position_in_playlist = " << new_pos
		   << " WHERE playlist_id = " << playlistID
		   << "   AND track_id = " << trackID << ";";
  if(!query.exec(sqlquery.str().c_str())) {
    setError(1, query);
    return;
  }
}

// ##############################################
// # delete entire medium from database
// ##############################################
void DataBase_MySQL::deleteMedium(int id)
{
  deleteTracks(getTracksByMedium(id));

#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  stringstream sqlquery;
  sqlquery << "DELETE FROM mediums WHERE id = " << id << ";";
  if(!query.exec(sqlquery.str().c_str())) {
    setError(1, query);
    return;
  }
}

// ##################
// # rename a medium 
// ##################
void DataBase_MySQL::renameMedium(MEDIUM *medium)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  stringstream sqlquery;
  sqlquery << "UPDATE mediums SET label = '" << adjustString(medium->label) << "' WHERE id = " << medium->id << ";";
  if(!query.exec(sqlquery.str().c_str())) {
    setError(1, query);
    return;
  }
}

// ##############################################
// # toggles favourite flag of artist
// ##############################################
void DataBase_MySQL::modifyFavouriteArtist(int id, bool state)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  stringstream sqlquery;
  sqlquery << "UPDATE artists SET isFavourite=" << ((state) ? "1" : "0") << " WHERE id = " << id << ";";
  if(!query.exec(sqlquery.str().c_str())) setError(1, query);
}

// ##############################################
// # get list of tracks by medium
// ##############################################
QList<TRACK> *DataBase_MySQL::getTracksByMedium(int id)
{
  stringstream sqlquery;
  sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment, notes, rating FROM tracks WHERE medium = " << id << ";";
  return applySelectTracksQuery(sqlquery.str());
}

// ##############################################
// # query list of tracks by medium
// ##############################################
QList<TRACK> *DataBase_MySQL::queryTracksByMedium(int id)
{
  stringstream sqlquery;
  sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment, notes, rating FROM tracks WHERE medium = " << id << ";";
  lastquery = sqlquery.str();
  return applySelectTracksQuery(sqlquery.str());
}

// ##############################################
// # query list of tracks by artist
// ##############################################
QList<TRACK> *DataBase_MySQL::queryTracksByArtist(int id)
{
  stringstream sqlquery;
  QString artist;
  for(QMap<QString, int>::Iterator it = artistToID.begin(); it != artistToID.end(); ++it) if(it.data() == id) artist = it.key();
  sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment, notes, rating FROM tracks WHERE artist = BINARY '" << adjustString(artist) << "';";
  lastquery = sqlquery.str();
  return applySelectTracksQuery(sqlquery.str());
}

// ##############################################
// # query list of tracks by artist and album
// ##############################################
  QList<TRACK> *DataBase_MySQL::queryTracksByArtistAlbum( QString artist, QString album )
  {
    stringstream sqlquery;
    sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment, notes, rating FROM tracks WHERE artist = BINARY '" << adjustString(artist) << "'";
    if (!album.isNull() )
      sqlquery << " AND album = BINARY '" << adjustString(album) << "'";
    sqlquery << ";";
    lastquery = sqlquery.str();
    return applySelectTracksQuery(sqlquery.str());
  }

// ##############################################
// # query list of tracks by album
// ##############################################
  QList<TRACK> *DataBase_MySQL::queryTracksByAlbum( QString comment, QString album )
  {
    stringstream sqlquery;
    sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment, notes, rating FROM tracks WHERE comment = '" << adjustString(comment) << "'";
    if (!album.isNull() )
      sqlquery << " AND album = BINARY '" << adjustString(album) << "'";
    sqlquery << ";";
    lastquery = sqlquery.str();
    return applySelectTracksQuery(sqlquery.str());
  }


// ##############################################
// # query list of tracks by playlist
// ##############################################
QList<TRACK> *DataBase_MySQL::queryTracksByPlaylist(int id)
{
  stringstream sqlquery;
  sqlquery << "SELECT t2.id, t2.path, t2.filename, t2.medium, t2.lastModified, t2.hasChanged, t2.mimetype, t2.version, t2.layer, t2.mode, t2.bitrate, t2.samplerate, t2.length, t2.size, t2.artist, t2.title, t2.album, t2.tracknumber, t2.year, t2.genre, t2.comment, t2.notes, t2.rating"
    	   << " FROM playlist_tracks AS t1, tracks AS t2"
		   << " WHERE t1.playlist_id = " << id << " AND t1.track_id = t2.id"
		   << " ORDER BY t1.position_in_playlist;";
  lastquery = sqlquery.str();
  return applySelectTracksQuery(sqlquery.str());
}

// ##############################################
// # query list of tracks by phrase
// ##############################################
QList<TRACK> *DataBase_MySQL::queryTracksByPhrase(QString phrase, bool byFilename, bool byArtist, bool byTitle, bool byAlbum, bool byComment, bool byNotes)
{
  phrase = adjustString(phrase);

  if(phrase.isEmpty() || (!byFilename && !byArtist && !byTitle && !byAlbum && !byComment && !byNotes)) return 0;

  stringstream sqlquery;
  sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment, notes, rating FROM tracks WHERE";
  bool isFirst = true;
  if(byFilename) {
    if(!isFirst) sqlquery << " OR";
    sqlquery << " UCASE(filename) REGEXP UCASE('" << adjustString(phrase) << "') OR UCASE(path) REGEXP UCASE('" << adjustString(phrase) << "')";
    isFirst = false;
  }
  if(byArtist) {
    if(!isFirst) sqlquery << " OR";
    sqlquery << " UCASE(artist) REGEXP UCASE('" << adjustString(phrase) << "')";
    isFirst = false;
  }
  if(byTitle) {
    if(!isFirst) sqlquery << " OR";
    sqlquery << " UCASE(title) REGEXP UCASE('" << adjustString(phrase) << "')";
    isFirst = false;
  }
  if(byAlbum) {
    if(!isFirst) sqlquery << " OR";
    sqlquery << " UCASE(album) REGEXP UCASE('" << adjustString(phrase) << "')";
    isFirst = false;
  }
  if(byComment) {
    if(!isFirst) sqlquery << " OR";
    sqlquery << " UCASE(comment) REGEXP UCASE('" << adjustString(phrase) << "')";
    isFirst = false;
  }
  if(byNotes) {
    if(!isFirst) sqlquery << " OR";
    sqlquery << " UCASE(notes) REGEXP UCASE('" << adjustString(phrase) << "')";
    isFirst = false;
  }
  
  sqlquery << ";";
  lastquery = sqlquery.str();
  return applySelectTracksQuery(sqlquery.str());
}

QList<TRACK> *DataBase_MySQL::queryTracksByWhere(QString phrase)
{
  if(phrase.isEmpty()) return 0;

  stringstream sqlquery;
  sqlquery << "SELECT id, path, filename, medium, lastModified, hasChanged, mimetype, version, layer, mode, bitrate, samplerate, length, size, artist, title, album, tracknumber, year, genre, comment, notes, rating FROM tracks WHERE "
           << toLocaleString(phrase).local8Bit().data() << ";";
  lastquery = sqlquery.str();
  return applySelectTracksQuery(sqlquery.str());
}

// ##############################################
// # reget last query
// ##############################################
QList<TRACK> *DataBase_MySQL::getLastQuery() {
  if(lastquery == "") return 0; else return applySelectTracksQuery(lastquery);
}

// ##############################################
// # disconnect from database
// ##############################################
void DataBase_MySQL::disconnect() {
  if(isConnected) {
    for(QMap<int, QMap<int, DELTAINFO*> >::Iterator it = deltamap->begin(); it != deltamap->end(); ++it)
      for(QMap<int, DELTAINFO*>::Iterator it2 = it.data().begin(); it2 != it.data().end(); ++it2) delete it2.data();
    delete deltamap;
    baselistArtists.clear();
    artistToID.clear();
    albumToID.clear();
    albumImageThumbnails.clear();
    if(db->isOpen()) db->close();
    isConnected = false;
    error = 0;
    lastquery = "";
    if (config) config->db_encoding = QString::null;
    emit connected(false);
  }
}

// *** destructor ***
DataBase_MySQL::~DataBase_MySQL() {
  disconnect();
  // not called automatically apparently.... 
  //  if (db) delete db;  
}

// *** get max ID from a table ***
int DataBase_MySQL::getMaxIDFromTable(QString table)
{
  error = 0;
  stringstream sqlquery;
  sqlquery << "SELECT MAX(id) FROM " << table.ascii() <<";";
#ifdef EMBEDDED
  qts::QSqlQuery query(sqlquery.str().c_str(), db);
#else
  QSqlQuery query(sqlquery.str().c_str(), db);
#endif   // ifdef EMBEDDED
  
  if(query.isActive())
    {
      if(query.next()) return query.value(0).toInt();
    }
  else setError(1, query);

  return 0;
}

// *** get number of records from a table ***
int DataBase_MySQL::getCountFromTable(QString table)
{
  error = 0;
  stringstream sqlquery;
  sqlquery << "SELECT COUNT(id) FROM " << table.ascii() <<";";
#ifdef EMBEDDED
  qts::QSqlQuery query(sqlquery.str().c_str(), db);
#else
  QSqlQuery query(sqlquery.str().c_str(), db);
#endif   // ifdef EMBEDDED

  if(query.isActive())
    {
      if(query.next()) return query.value(0).toInt();
    }
  else setError(1, query);

  return 0;
}

// *** get number of records from a table ***
int DataBase_MySQL::getCountColumnDistinctFromTable(QString column, QString table)
{
  error = 0;
  stringstream sqlquery;
  sqlquery << "SELECT COUNT(DISTINCT " << column.ascii() << ") FROM " << table.ascii() <<";";
#ifdef EMBEDDED
  qts::QSqlQuery query(sqlquery.str().c_str(), db);
#else
  QSqlQuery query(sqlquery.str().c_str(), db);
#endif   // ifdef EMBEDDED

  if(query.isActive())
    {
      if(query.next()) return query.value(0).toInt();
    }
  else setError(1, query);

  return 0;
}

// *** apply query and convert to list ***
QList<TRACK> *DataBase_MySQL::applySelectTracksQuery(string sqlquery)
{
  error = 0;
#ifdef EMBEDDED
  qts::QSqlQuery query(sqlquery.c_str());
#else
  QSqlQuery query(sqlquery.c_str());
#endif   // ifdef EMBEDDED
  
  if(query.isActive()) {
    QList<TRACK> *list = new QList<TRACK>;
    while(query.next()) {
      TRACK *track = new TRACK;
      track->id           = query.value(0).toInt();
      track->path         = toLocaleString( query.value(1) );
      track->filename     = toLocaleString( query.value(2) );
      track->medium       = query.value(3).toInt();
      track->lastModified = query.value(4).toDateTime();
      track->hasChanged   = (query.value(5).toInt() != 0);
      track->mimetype     = query.value(6).toInt();
      track->version      = query.value(7).toInt();
      track->layer        = query.value(8).toInt();
      track->mode         = query.value(9).toInt();
      track->bitrate      = query.value(10).toInt();
      track->samplerate   = query.value(11).toInt();
      track->length       = query.value(12).toInt();
      track->size         = query.value(13).toInt();
      track->artist       = toLocaleString( query.value(14) );
      track->title        = toLocaleString( query.value(15) );
      track->album        = toLocaleString( query.value(16) );
      track->tracknumber  = query.value(17).toInt();
      track->year         = toLocaleString( query.value(18) );
      track->genre        = query.value(19).toInt();
      track->comment      = toLocaleString( query.value(20) );
      track->notes        = toLocaleString( query.value(21) );
      track->rating       = query.value(22).toInt();
      list->append(track);
    }
    return list;
  } else {
#ifndef EMBEDDED
    if(query.lastError().type() == QSqlError::Statement) {
      setError(2, query);
    } else  setError(1, query);
#else
    setError(2, query);  // do not disconnect
#endif   // ifndef EMBEDDED
    return 0;
  }
}

// *** convert mysql string to datetime ***
 QDateTime DataBase_MySQL::stringToDatetime(QString s)
 {
   int y, m, d, h, mi, sec;
   sscanf(s.ascii(), "%d-%d-%d %d:%d:%d", &y, &m, &d, &h, &mi, &sec);
   return QDateTime(QDate(y,m,d), QTime(h,mi,sec));
 }

// *** convert datetime to mysql string ***
QCString DataBase_MySQL::dateTimeToString(QDateTime d)
{
  QString s;
  s.sprintf("%d-%02d-%02d %d:%d:%d", d.date().year(), d.date().month(), d.date().day(), d.time().hour(), d.time().minute(), d.time().second());
  return s.ascii();
}

#ifdef HAVE_MEXTRAS
// ##############################################
// # check musicextras data in prokyon3 cache
// ##############################################

EXTRA_STATUS DataBase_MySQL::checkMextras(TRACK *track) 
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  EXTRA_STATUS status;

  stringstream sqlquery;
  sqlquery << "SELECT image_id,biography_id,years_active FROM artists  WHERE BINARY name = '" << adjustString(track->artist) << "' LIMIT 1;";
  query.exec(sqlquery.str().c_str());

  int image_id;
  int biography_id;
  QString years_active;
  if(query.isActive() && query.next()) {
    image_id = query.value( 0 ).toInt();
    biography_id = query.value( 1 ).toInt();
    years_active = toLocaleString( query.value( 2 ) );
    status.artist = image_id;
    status.biography = biography_id;
    status.years_active = years_active.isEmpty(); 
  } else {
    setError(1, query);
    return EXTRA_STATUS();
  }

  stringstream sqlquery2;
  sqlquery2 << "SELECT lyrics_id FROM tracks" 
      << " WHERE artist = '" << adjustString(track->artist)
      << "' AND title  = '" << adjustString(track->title)
      << "' LIMIT 1;";
  query.exec(sqlquery2.str().c_str());

  int lyrics_id;
  if(query.isActive() && query.next()) {
    lyrics_id = query.value( 0 ).toInt();
    status.lyrics = lyrics_id;
  } else {
    setError(1, query);
    return EXTRA_STATUS();
  }

  stringstream sqlquery3;
  sqlquery3 << "SELECT cover_id, album_review_id, album_tracks_id FROM albums" 
	    << " WHERE artist = '" << adjustString(track->artist)
	    << "' AND name  = '" << adjustString(track->album)
	    << "';";
  query.exec(sqlquery3.str().c_str());

  int cover_id, album_review_id, album_tracks_id;
  if(query.isActive() && query.next()) {
    cover_id = query.value( 0 ).toInt();
    album_review_id = query.value( 1 ).toInt();
    album_tracks_id = query.value( 2 ).toInt();
    status.cover = cover_id;
    status.album_review = album_review_id;
    status.album_tracks = album_tracks_id;
  } else {
    setError(1, query);
    return EXTRA_STATUS();
  }
  return status;
}

// ##############################################
// # update musicextras data in database
// ##############################################
void DataBase_MySQL::updateMextras(EXTRADATA *data, TRACK *track)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  if (verbose == 9)  qWarning(" >>>>>updateMextras track->id:%d",track->id);

  if ( !data->artist_image.isNull() || !biography_isEmpty(data->biography) || !years_active_isEmpty(data->years_active)) {

    stringstream sqlquery1;
    sqlquery1 << "SELECT id, image_id, biography_id, years_active FROM artists WHERE BINARY name = '" << adjustString(track->artist) << "' LIMIT 1;";
    query.exec(sqlquery1.str().c_str());
    
    int artists_id, image_id, biography_id;
    QString years_active;

    if(query.isActive() && query.next()) {
        artists_id   = query.value( 0 ).toInt();
        image_id     = query.value( 1 ).toInt();
        biography_id = query.value( 2 ).toInt();
        years_active = toLocaleString( query.value( 3 ) );
    } else {
        setError(1, query);
        return;
    }

    if ( !data->artist_image.isNull() ) {
        if (verbose == 9) {
            qWarning("image_id:%d",image_id);
            qWarning("        >>>artists_id:%d",artists_id);
        }

        if (image_id) {
            // image already in mextras. Do nothing;
            // may be we should check if data is still valid?
            // that will make dropping mextras table an easy way to erase all mextras data...
        } else {
            alterImageArtistsMextras(addByteArrayMextras(data->artist_image), artists_id);
        }
    }

    if ( !biography_isEmpty(  data->biography ) ) {
        if (verbose == 9) {
            qWarning("biography_id:%d",biography_id);
            qWarning("        >>>artists_id:%d",artists_id);
        }

        if (biography_id) {
            // biography already in mextras. Do nothing;
            // may be we should check if data is still valid?
            // that will make dropping mextras table an easy way to erase all mextras data...
        } else {
            alterBiographyArtistsMextras(addStringMextras(data->biography), artists_id);
        }
    }

    if ( !years_active_isEmpty( data->years_active ) ) {
        if ( !years_active.isEmpty() ) 
            alterYearsActiveArtistsMextras(data->years_active, artists_id);
    }
  }

  if ( !lyrics_isEmpty(data->lyrics) ) {
     stringstream sqlquery2;
    sqlquery2 << "SELECT lyrics_id FROM tracks"
              << " WHERE artist = '" << adjustString(track->artist)
              << "' AND title  = '" << adjustString(track->title)
	          << "' LIMIT 1;";
    query.exec(sqlquery2.str().c_str());

    int lyrics_id;
    if(query.isActive() && query.next()) {
      lyrics_id = query.value( 0 ).toInt();
    } else {
      setError(1, query);
      return;
    }

    if (verbose == 9) qWarning("lyrics_id:%d",lyrics_id);

    if (lyrics_id) {
      // image already in mextras. Do nothing;
      // may be we should check if data is still valid?
      // that will make dropping mextras table an easy way to erase all mextras data...
    } else {
      alterLyricsTracksMextras(addStringMextras(data->lyrics), track->id);
    }
  }

  if (!data->cover_image.isNull() || !album_review_isEmpty(data->album_review) || !album_tracks_isEmpty(data->album_tracks) ) {
      stringstream sqlquery3;
      sqlquery3 << "SELECT id, cover_id, album_review_id, album_tracks_id FROM albums" 
                << " WHERE artist = '" << adjustString(track->artist)
                << "' AND name  = '" << adjustString(track->album)
                << "' LIMIT 1;";
      query.exec(sqlquery3.str().c_str());

    int album_id=0;
    int cover_id=0;
    int album_review_id = 0;
    int album_tracks_id = 0;
    
    if(query.isActive() && query.next()) {
        album_id        = query.value( 0 ).toInt();
        cover_id        = query.value( 1 ).toInt();
        album_review_id = query.value( 2 ).toInt();
        album_tracks_id = query.value( 3 ).toInt();
    } else {
      setError(1, query);
      return;
    }
  
    if (!data->cover_image.isNull()) {
        if (verbose == 9) qWarning("cover_id:%d", cover_id);
    
        if(cover_id) {
            // image already in mextras. Do nothing;
            // may be we should check if data is still valid?
            // that will make dropping mextras table an easy way to erase all mextras data...
        } else {
            alterCoverTracksMextras(addByteArrayMextras(data->cover_image), album_id, track->artist, track->album); // create a new one
        }
    }

    if ( !album_review_isEmpty( data->album_review ) ) {
        if (verbose == 9) qWarning("album_review_id:%d",album_review_id);

        if (album_review_id) {
            // album review already in mextras. Do nothing;
            // may be we should check if data is still valid?
            // that will make dropping mextras table an easy way to erase all mextras data...
        } else {
            alterAlbumReviewTracksMextras(addStringMextras(data->album_review), album_id);
        }
    }
  
    if ( !album_tracks_isEmpty( data->album_tracks ) ) {
        if (verbose == 9) qWarning("album_tracks_id:%d",album_review_id);

        if (album_tracks_id) {
            // album tracks already in mextras. Do nothing;
            // may be we should check if data is still valid?
            // that will make dropping mextras table an easy way to erase all mextras data...
        } else {
            alterAlbumTracksTracksMextras(addStringMextras(data->album_tracks), album_id);
        }
    }
  }
}


// ##############################################
// # change image_id in artists database
// ##############################################
void DataBase_MySQL::alterImageArtistsMextras(int image_id, int artists_id)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  stringstream sqlquery;
  sqlquery << "UPDATE artists SET"
	   << " image_id = " << image_id
	   << " WHERE id = " << artists_id << ";";
  if(!query.exec(sqlquery.str().c_str())) {
    setError(1, query);
    return;
  }
}

// ##############################################
// # change biography_id in artists database
// ##############################################
void DataBase_MySQL::alterBiographyArtistsMextras(int biography_id, int artists_id)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  stringstream sqlquery;
  sqlquery << "UPDATE artists SET"
      << " biography_id = " << biography_id
      << " WHERE id = " << artists_id << ";";
  if(!query.exec(sqlquery.str().c_str())) {
    setError(1, query);
    return;
  }
}

// ##############################################
// # change years_active in artists database
// ##############################################
void DataBase_MySQL::alterYearsActiveArtistsMextras(QString& years_active, int artists_id)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  stringstream sqlquery;
  sqlquery << "UPDATE artists SET"
      << " years_active = '" << adjustString(years_active) << "'"
      << " WHERE id = " << artists_id << ";";
  if(!query.exec(sqlquery.str().c_str())) {
    setError(1, query);
    return;
  }
}

// ##############################################
// # change cover_id in albums database
// ##############################################
void DataBase_MySQL::alterCoverTracksMextras(int cover_id, int album_id, const QString artist, const QString album)
{
#ifdef EMBEDDED
    qts::QSqlQuery query;
#else
    QSqlQuery query;
#endif   // ifdef EMBEDDED

    stringstream sqlquery;
    sqlquery << "UPDATE albums SET cover_id = " << cover_id
             << " WHERE id = " << album_id << ";";
    if(query.exec(sqlquery.str().c_str())) {
        qApp->lock(); // avoid async during pixmap creation
        QImage image(getByteArrayMextras(cover_id));
        if(albumImageThumbnails.contains(artist + album)) {
            albumImageThumbnails[artist + album]->detach();
        }
        albumImageThumbnails[artist + album] = new QPixmap(image.scale(32,32));
        qApp->unlock();
        NewCoverImageEvent* e = new NewCoverImageEvent(artist, album);
        QApplication::postEvent( gui->getSelector(), e );
        return;
    }
    
    setError(1, query);
    return;
}

// ##############################################
// # change lyrics_id in tracks database
// ##############################################
void DataBase_MySQL::alterLyricsTracksMextras(int lyrics_id, int tracks_id)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  stringstream sqlquery;
  sqlquery << "UPDATE tracks SET"
	   << " lyrics_id = " << lyrics_id
	   << " WHERE id = " << tracks_id << ";";
  if(!query.exec(sqlquery.str().c_str())) {
    setError(1, query);
    return;
  }
}

// ##############################################
// # change album_review_id in tracks database
// ##############################################
void DataBase_MySQL::alterAlbumReviewTracksMextras(int album_review_id, int album_id)
{
#ifdef EMBEDDED
    qts::QSqlQuery query;
#else
    QSqlQuery query;
#endif   // ifdef EMBEDDED

    stringstream sqlquery;
    sqlquery << "UPDATE albums SET album_review_id = " << album_review_id
             << " WHERE id = " << album_id << ";";
    if(query.exec(sqlquery.str().c_str())) {
        return;
    }
    
    setError(1, query);
    return;
}

// ##############################################
// # change album_tracks_id in tracks database
// ##############################################
void DataBase_MySQL::alterAlbumTracksTracksMextras(int album_tracks_id, int album_id)
{
#ifdef EMBEDDED
    qts::QSqlQuery query;
#else
    QSqlQuery query;
#endif   // ifdef EMBEDDED

    stringstream sqlquery;
    sqlquery << "UPDATE albums SET album_tracks_id = " << album_tracks_id
             << " WHERE id = " << album_id << ";";
    if(query.exec(sqlquery.str().c_str())) {
        return;
    }
    
    setError(1, query);
    return;
}


// ##############################################
// # add QByteArray BLOB in database
// ##############################################
int DataBase_MySQL::addByteArrayMextras(QByteArray& pix)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  QString st;
  for ( int i = 0; i<pix.size();i++)  st.append( pix[i] );

  // We insert escaped binary data into  a BLOB column
  // as described if MySQL info file, node: string syntax 

   if ( !query.exec( "INSERT INTO mextras ( data ) VALUES (\'"
 		    + QString(adjustQString( st )) + "\');" ) ) {
     setError(1, query);
     return 0;
   }

  if ( !query.exec("SELECT LAST_INSERT_ID();") ) {
    setError(1, query);
    return 0;
  }

  if(query.isActive() && query.next()) {
    return query.value( 0 ).toInt();
  } else {
    setError(1, query);
    return 0;
  }
}

// ##############################################
// # get QByteArray BLOB in database
// ##############################################
const QByteArray DataBase_MySQL::getByteArrayMextras(int ID)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  // read the BLOB back from the database
  stringstream sqlquery;
  sqlquery << "SELECT data FROM mextras WHERE id = " << ID << ";";

  if ( !query.exec(sqlquery.str().c_str()) ) {
    if (verbose == 9) qWarning( "Unable to execute query - 3 exiting" );
    setError(1, query);
    return 0;
  }
  if(query.isActive() && query.next()) {
    return query.value( 0 ).toByteArray();
  }
  setError(1, query);
  return 0;
}

// ##############################################
// # add QString as a BLOB to database
// ##############################################
int DataBase_MySQL::addStringMextras(QString& st)
{
  QByteArray temp;
  QDataStream str(temp, IO_WriteOnly);
  str << st;
  return addByteArrayMextras( temp );
}

// ##############################################
// # Get QString as a BLOB from database
// ##############################################
const QString DataBase_MySQL::getStringMextras(int ID)
{
  QDataStream str(getByteArrayMextras( ID), IO_ReadOnly);
  QString ms;
  str >> ms;
  return ms;
}

// ##############################################
// # delete  BLOB in database if ID exists 
// #     >>>No error if !exists
// ##############################################
void DataBase_MySQL::deleteMextras(int ID)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  stringstream sqlquery;
  sqlquery << "DELETE FROM mextras WHERE id = " << ID << ";";

  if ( !query.exec(sqlquery.str().c_str()) ) {
    setError(1, query);
    return;
  }
  return;
}

// ##############################################
// # Get the extra data
// ##############################################
EXTRADATA_GUI *DataBase_MySQL::getExtraData(TRACK *track)
{

  error = 0;

  EXTRADATA_GUI *data = new EXTRADATA_GUI();
  
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  stringstream sqlquery1;
  sqlquery1 << "SELECT image_id,years_active,biography_id FROM artists WHERE BINARY name = '" 
	    << adjustString(track->artist)
	    << "' LIMIT 1;";
  query.exec(sqlquery1.str().c_str());

  if(query.isActive() && query.next()) {
    int image_id = query.value( 0 ).toInt();
    data->artist_pixmap = NULL;
    if (image_id) {
      QIDPixmap *pixmap = new QIDPixmap(image_id, QPixmap(getByteArrayMextras( image_id )));
      if (pixmap->image.isNull()) {
         // TRANSLATORS: %1 = artist name
         MessageEvent* e = new MessageEvent(QString(_("The artist image of '%1' stored in database is corrupt!")).arg(track->artist));
         QApplication::postEvent( gui, e );
         return new EXTRADATA_GUI();
      } else {
         data->artist_pixmap = pixmap;
      }
    }
    data->years_active = toLocaleString( query.value( 1 ) );
    int biography_id = query.value( 2 ).toInt();
    if (biography_id) 
      data->biography = getStringMextras( biography_id );
    else data->biography = "";
  } else {
    setError(1, query);
    return new EXTRADATA_GUI();
  }

  stringstream sqlquery2;
  sqlquery2 << "SELECT lyrics_id, cover_id, album_review_id, album_tracks_id"
      << " FROM tracks LEFT JOIN albums ON (BINARY tracks.album = albums.name AND BINARY tracks.artist = albums.artist)" 
	  << " WHERE tracks.id = " << track->id << ";";
  query.exec(sqlquery2.str().c_str());

  if(query.isActive() && query.next()) {
    int lyrics_id = query.value( 0 ).toInt();
    int cover_id = query.value( 1 ).toInt();
    int album_review_id = query.value( 2 ).toInt();
    int album_tracks_id = query.value( 3 ).toInt();
    data->cover_pixmap = NULL;
    if (cover_id) {
      QIDPixmap *pixmap = new QIDPixmap( cover_id, QPixmap(getByteArrayMextras( cover_id )));
      if (pixmap->image.isNull()) {
         // TRANSLATORS: %1 = album name
         MessageEvent* e = new MessageEvent(QString(_("The cover image of '%1' stored in database is corrupt!")).arg(track->album));
         QApplication::postEvent( gui, e );
         return new EXTRADATA_GUI();
      } else {
         data->cover_pixmap = pixmap;
      }
    }
    if (lyrics_id)
      data->lyrics = getStringMextras( lyrics_id );
    else data->lyrics = "";
    if (album_review_id)
      data->album_review = getStringMextras( album_review_id );
    else data->album_review = "";
    if (album_tracks_id)
      data->album_tracks = getStringMextras( album_tracks_id );
    else data->album_tracks = "";
    return data;
  } else {
    setError(1, query);
    return new EXTRADATA_GUI();
  }
}

// ##############################################
// # zap all extra datas
// ##############################################
void DataBase_MySQL::zapExtraData() 
{

  error =0;

#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  qApp->lock();
  int YES =  QMessageBox::warning(NULL, 
				  "Musicextras",
				  _("OK to erase all cover and lyrics data cached into prokyon3 database?")
				  ,QMessageBox::Yes, QMessageBox::No);

  if ( YES == QMessageBox::Yes ) {

    stringstream sqlquery;
    sqlquery << "DELETE FROM mextras;";

    if ( !query.exec(sqlquery.str().c_str()) ) {
      setError(1, query);
      return;
    }

    stringstream sqlquery2;
    sqlquery2 << "UPDATE tracks SET lyrics_id = NULL;";

    if ( !query.exec(sqlquery2.str().c_str()) ) {
      setError(1, query);
      return;
    }

    stringstream sqlquery3;
    sqlquery3 << "UPDATE artists SET image_id = NULL, years_active = NULL, biography_id = NULL;";

    if ( !query.exec(sqlquery3.str().c_str()) ) {
      setError(1, query);
      return;
    }
    
    stringstream sqlquery4;
    sqlquery4 << "UPDATE albums SET cover_id = NULL, album_review_id = NULL, album_tracks_id = NULL;";

    if ( !query.exec(sqlquery4.str().c_str()) ) {
      setError(1, query);
      return;
    }
  }
  
  albumImageThumbnails.clear();
  NewCoverImagesEvent* e = new NewCoverImagesEvent();
  QApplication::postEvent( gui->getSelector(), e );
  

  int YESCACHE =  QMessageBox::warning(NULL, 
				       "musicextras",
				       _("OK to also erase all Musicextras internal cache?\n"
					 "This includes local copies of html pages containing images/lyrics files.\n\n"
					 "WARNING: This could take a long time to retrieve from www later on....")
				       ,QMessageBox::Yes, QMessageBox::No);
  qApp->unlock();

  if (YESCACHE == QMessageBox::Yes) {
    QProcess *proc;
    proc = new QProcess( QString("musicextras") );
    proc->addArgument( "--clear-cache" );
    bool  procres = proc->start();

    if ( !procres ) {
      MessageEvent* e = new MessageEvent(_("Could not find  \"musicextras\".\n"
					   "Please install from http://kapheine.hypa.net/musicextras.\n"
					   "And make sure it is in your $PATH" ));
      QApplication::postEvent( gui, e );
      return;
    }
    delete proc;

    //zap log file
    QFile::remove( QDir::homeDirPath()+"/.musicextras/debug.log" );
    QFile file( QDir::homeDirPath()+"/.musicextras/debug.log" );
    file.open( IO_ReadWrite );
    file.close();

    proc = new QProcess( QString("musicextras") );
    proc->addArgument( "--clear-greylist" );
    proc->start();
    delete proc;

    proc = new QProcess( QString("musicextras") );
    proc->addArgument( "--clear-webcache" );
    proc->start();
    delete proc;

  }
  return;
}
#endif /* HAVE_MEXTRAS */

#ifdef EMBEDDED
void DataBase_MySQL::setError(int number, qts::QSqlQuery& query)
#else
void DataBase_MySQL::setError(int number, QSqlQuery& query)
#endif   // ifdef EMBEDDED
{
  error = number;
#ifdef EMBEDDED
  if(query.lastError().type() == qts::QSqlError::Statement) {
#else
  if(query.lastError().type() == QSqlError::Statement) {
#endif   // ifdef EMBEDDED
    QString driver_text = query.lastError().driverText();
    for(int i = 100; i < driver_text.length(); i += 100) driver_text = driver_text.insert(i, "\n");
    QString database_text = query.lastError().databaseText();
    for(int i = 100; i < database_text.length(); i += 100) database_text = database_text.insert(i, "\n");
    QString last_query = query.lastQuery();
    if("ALTER TABLE artists ADD UNIQUE name_unique (name);" == last_query) {
        qApp->lock();
        int repair = QMessageBox::warning(NULL, 
                         _("Warning"),
                         _("It seems that the artists table of your database contains duplicate entries.\n"
                           "This is caused by a bug in previous versions of prokyon3.\n"
                           "This version of prokyon3 will be able to repair your data.\n"
                           "But we recommend that you create a backup of your database first.\n\n"
                           "Note: This message may appear twice. In this case you have to repair again.\n\n"
                           "Do you want prokyon3 to repair the artists table now?"),
                         QMessageBox::Yes, QMessageBox::No);
        qApp->unlock();
        if(repair == QMessageBox::Yes) repairTableArtists();
    } else {
        for(int i = 100; i < last_query.length(); i += 100) last_query = last_query.insert(i, "\n");
        MessageEvent* e = new MessageEvent( driver_text + "\n" + database_text + "\n\n" + last_query );
        QApplication::postEvent( gui, e );
    }
  }
}

// ##############################################
// # zap extra datas from a track
// ##############################################
#ifdef HAVE_MEXTRAS
void DataBase_MySQL::zapExtraData(TRACK *track) 
{

  error =0;

#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  // zap cover_id for album
  stringstream sqlquery1;
  sqlquery1 << "UPDATE albums SET cover_id = NULL, album_review_id = NULL, album_tracks_id = NULL" 
	    << " WHERE BINARY artist = '" << adjustString(track->artist)
	    << "' AND BINARY name  = '" << adjustString(track->album)
	    << "';";

  if ( !query.exec(sqlquery1.str().c_str()) ) {
    setError(1, query);
    return;
  }

  albumImageThumbnails.remove(track->artist + track->album);
  NewCoverImageEvent* e = new NewCoverImageEvent(track->artist, track->album);
  QApplication::postEvent( gui->getSelector(), e );
  
  // zap other id  for specific track
  stringstream sqlquery2;
  sqlquery2 << "UPDATE tracks SET lyrics_id = NULL" 
	    << " WHERE id = " << track->id << ";";

  if ( !query.exec(sqlquery2.str().c_str()) ) {
    setError(1, query);
    return;
  }

// zap all id for specific artist
  stringstream sqlquery3;
  sqlquery3 << "UPDATE artists SET image_id = NULL, years_active = NULL, biography_id = NULL WHERE BINARY name = '"
	    << adjustString(track->artist) 
	    << "'";

  if ( !query.exec(sqlquery3.str().c_str()) ) {
    setError(1, query);
    return;
  }
  return;
}
#endif /* HAVE_MEXTRAS */

// ##############################################
// # repairs artists table created before 0.9.3
// ##############################################
void DataBase_MySQL::repairTableArtists()
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  if(query.exec("CREATE TABLE artists_repair (name varchar(255) BINARY NOT NULL, isFavourite tinyint(4) default NULL, image_id int(11) default NULL, years_active VARCHAR(255) default NULL, biography_id int(11) default NULL);")
     && query.isActive())
    if(query.exec("INSERT INTO artists_repair (name, isFavourite, image_id, years_active, biography_id) SELECT name, BIT_OR(isFavourite), MIN(image_id), MAX(years_active), MIN(biography_id) FROM artists GROUP BY name;")
       && query.isActive())
      if(query.exec("DELETE FROM artists;")
         && query.isActive())
        if(query.exec("INSERT INTO artists (name, total, local, isFavourite, image_id, years_active, biography_id) SELECT artist, count(*), SUM(IF(medium > 0,0,1)), isFavourite, image_id, years_active, biography_id FROM tracks LEFT JOIN artists_repair ON (BINARY(name)=BINARY(artist)) GROUP BY BINARY(artist), isFavourite, image_id, years_active, biography_id;")
         && query.isActive()) {
          MessageEvent* e = new MessageEvent(_("The artists table has now been repaired.\n"
                                               "You can save a little disk space by login\n"
                                               "to your prokyon3 database as root and run\n"
                                               "'DROP TABLE artists_repair;'\n"));
          QApplication::postEvent( gui, e );
         } else setError(1, query);
      else setError(1, query);
    else setError(1, query);
  else setError(1, query);
}

void DataBase_MySQL::fillTableAlbumsFromTableTracks(bool includingMusicExtras)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED

  if(includingMusicExtras) {
    if(query.exec("INSERT INTO albums (name, artist, tracks_available, cover_id, album_review_id) "
                  "SELECT album, artist, COUNT(*), MIN(cover_id), MIN(album_review_id) FROM tracks GROUP BY BINARY(artist), BINARY(album);")
       && query.isActive()) {
      if(query.exec("ALTER TABLE tracks DROP COLUMN cover_id;") && query.isActive()) {
        if(!query.exec("ALTER TABLE tracks DROP COLUMN album_review_id;") || !query.isActive()) {
          setError(1, query);
        }
      } else {
        setError(1, query);
      }
    } else {
        setError(1, query);
    }
  } else {
    if(!query.exec("INSERT INTO albums (name, artist, tracks_available) "
                   "SELECT album, artist, COUNT(*) FROM tracks GROUP BY BINARY(artist), BINARY(album);")
       || !query.isActive()) {
      setError(1, query);
    }
  }
}

// ##############################################
// # Get custom queries
// ##############################################
QMap<QString,QString> DataBase_MySQL::getCustomQueries()
{
  QMap<QString,QString> map;
  
#ifdef EMBEDDED
  qts::QSqlQuery query("SELECT name, where_clause FROM custom_queries ORDER BY name;", db);
#else
  QSqlQuery query("SELECT name, where_clause FROM custom_queries ORDER BY name;", db);
#endif   // ifdef EMBEDDED
  
  if(query.isActive()) {
    while(query.next()) {
      map[toLocaleString(query.value(0))] = query.value(1).toString();
    }
    return map;
  }
  
  setError(1, query);
  return map;
}

void DataBase_MySQL::storeCustomQuery(QString name, QString whereClause)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  
  stringstream sqlquery;
  sqlquery << "INSERT INTO custom_queries VALUES ('"
	       << adjustString(name)  << "', '"
	       << adjustString(whereClause) << "');";
  
  query.exec(sqlquery.str().c_str());
  if(!query.isActive()) {
    setError(1, query);
  }
}

void DataBase_MySQL::removeCustomQuery(QString name)
{
#ifdef EMBEDDED
  qts::QSqlQuery query;
#else
  QSqlQuery query;
#endif   // ifdef EMBEDDED
  
  stringstream sqlquery;
  sqlquery << "DELETE FROM custom_queries WHERE name='"
	       << adjustString(name)  << "';";
  
  query.exec(sqlquery.str().c_str());
  if(!query.isActive()) {
    setError(1, query);
  }
}

