/* Copyright (C) 1999-2000 Bernhard Trummer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * $Log: itemizenode.cc,v $
 * Revision 1.3  2001/04/27 20:02:50  slash
 * Make use of xmlChildrenNode and xmlRootNode.
 *
 * Revision 1.2  2001/04/20 15:40:36  slash
 * Make use of the function xmlGetProp().
 *
 * Revision 1.1.1.1  2001/01/30 18:36:28  slash
 * Initial release.
 *
 */

#include "main.h"
#include "itemnode.h"
#include "itemizenode.h"
#include "enumeratenode.h"


// The following macros are used to provide an easy handling,
// if the apropriate property isn't given in the XML-file.
#define TYPE_  (type_ != UNDEFINED ? type_ : (level_ == 0 ? DISC : CIRCLE))
#define STEP_  (step_ != -1 ? step_ : 0)
#define CLOSING_P_  (closing_p_ != -1 ? closing_p_ : 1)


//---------------------------------------------------------------------------
ItemizeNode::ItemizeNode(xmlNodePtr node,
                         unsigned int level,
                         int with_left_margin)
  : ListNode(level, with_left_margin)
{
    type_ = UNDEFINED;
    step_ = -1;
    closing_p_ = -1;

    // Extract the properties of the itemize-tag.
    char *type = (char*)xmlGetProp(node, (xmlChar*)"type");
    if (type) {
        if      (!strcmp(type, "none"))   type_ = NONE;
        else if (!strcmp(type, "disc"))   type_ = DISC;
        else if (!strcmp(type, "circle")) type_ = CIRCLE;
        else if (!strcmp(type, "square")) type_ = SQUARE;
    }
    char *step = (char*)xmlGetProp(node, (xmlChar*)"step");
    if (step) {
        if      (!strcmp(step, "yes")) step_ = 1;
        else if (!strcmp(step, "no"))  step_ = 0;
    }
    char *closing_p = (char*)xmlGetProp(node, (xmlChar*)"closing_p");
    if (closing_p) {
        if      (!strcmp(closing_p, "yes")) closing_p_ = 1;
        else if (!strcmp(closing_p, "no"))  closing_p_ = 0;
    }

    // Extract the sub-nodes.
    for (node = node->xmlChildrenNode; node; node = node->next) {
        if (!strcmp((char*)node->name,"item")) {
            child_nodes_.push_back(new ItemNode(node, CLOSING_P_));
            is_sub_list_.push_back(0);
            number_of_html_steps_++;

        } else if (!strcmp((char*)node->name,"itemize")) {
            ListNode *sub_list = new ItemizeNode(node, level_+1);
            child_nodes_.push_back(sub_list);
            is_sub_list_.push_back(1);
            if (sub_list->isHTMLStepped()) {
                number_of_html_steps_ += sub_list->getNumberOfHTMLSteps();
            } else {
                number_of_html_steps_++;
            }

        } else if (!strcmp((char*)node->name,"enumerate")) {
            ListNode *sub_list = new EnumerateNode(node, level_+1);
            child_nodes_.push_back(sub_list);
            is_sub_list_.push_back(1);
            if (sub_list->isHTMLStepped()) {
                number_of_html_steps_ += sub_list->getNumberOfHTMLSteps();
            } else {
                number_of_html_steps_++;
            }

        } else {
            Number_Of_Errors_++;
        }
    }
}

//---------------------------------------------------------------------------
ItemizeNode::~ItemizeNode()
{
}


//---------------------------------------------------------------------------
void ItemizeNode::writeLaTeX() const
{
    // Set the enumeration-type.
    char *level;
    switch (level_) {
      // LaTeX supports only 4 levels.
      case 0:  level = "i"; break;
      case 1:  level = "ii"; break;
      case 2:  level = "iii"; break;
      case 3:  level = "iv"; break;
      default: level = ""; break;
    }
    Output_ << "\\renewcommand{\\labelitem" << level << "}{";
    switch (TYPE_) {
      case NONE:   Output_ << ""; break;
      case DISC:   Output_ << "$\\bullet$"; break;
      case CIRCLE: Output_ << "$\\circ$"; break;
      case SQUARE: Output_ << "$\\Box$"; break;
      case UNDEFINED:   break;  // To prevent a compiler-warning.
    }
    Output_ << "}" << endl;

    Output_ << "\\begin{itemize}" << endl;
    for (unsigned int i=0; i<child_nodes_.size(); i++) {
        child_nodes_[i]->writeLaTeX();
    }
    Output_ << "\\end{itemize}" << endl;
}

//---------------------------------------------------------------------------
void ItemizeNode::writeXML() const
{
    Output_ << "<itemize";
    switch (type_) {
      case UNDEFINED: break;
      case NONE:   Output_ << " type=\"none\""; break;
      case DISC:   Output_ << " type=\"disc\""; break;
      case CIRCLE: Output_ << " type=\"circle\""; break;
      case SQUARE: Output_ << " type=\"square\""; break;
    }
    if (step_ != -1) {
        Output_ << " step=\"" << (step_ == 1 ? "yes" : "no") << "\"";
    }
    if (closing_p_ != -1) {
        Output_ << " closing_p=\"" << (closing_p_ == 1 ? "yes" : "no") << "\"";
    }
    Output_ << ">" << endl;

    for (unsigned int i=0; i<child_nodes_.size(); i++) {
        child_nodes_[i]->writeXML();
    }

    Output_ << "</itemize>" << endl;
}


//---------------------------------------------------------------------------
int ItemizeNode::isHTMLStepped() const
{
    return STEP_;
}

//---------------------------------------------------------------------------
unsigned int ItemizeNode::getNumberOfHTMLSteps() const
{
    return number_of_html_steps_;
}


//---------------------------------------------------------------------------
void ItemizeNode::writeHTMLHeader() const
{
    if (with_left_margin_ && (level_ == 0)) {
        Output_ << "<DIV class=\"";
        switch (getLength() / 16) {
          case 0:  Output_ << "listtiny"; break;
          case 1:  Output_ << "listmedium"; break;
          default: Output_ << "listlarge"; break;
        }
        Output_ << "\">" << endl;
    }

    Output_ << "<UL style=\"list-style-type:";
    switch (TYPE_) {
      case NONE:   Output_ << "none"; break;
      case DISC:   Output_ << "disc"; break;
      case CIRCLE: Output_ << "circle"; break;
      case SQUARE: Output_ << "square"; break;
      case UNDEFINED: break;  // To prevent a compiler-warning.
    }
    Output_ << "\">" << endl;
}

//---------------------------------------------------------------------------
void ItemizeNode::writeHTMLFooter() const
{
    Output_ << "</UL>" << endl;
    if (with_left_margin_ && (level_ == 0)) {
        Output_ << "</DIV>" << endl;
    }
}
