/*****
*
* Copyright (C) 2000, 2002 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/time.h>

#include "config.h"
#include "packet.h"
#include "hostdb.h"
#include "packet-decode.h"

#include <libprelude/prelude-log.h>
#include <libprelude/extract.h>
#include <libprelude/prelude-log.h>
#include <libprelude/plugin-common.h>
#include <libprelude/plugin-common-prv.h>


#define HASH_SIZE 1024


#define hkey(saddr, daddr, prot) \
               (( (saddr) ^ (daddr) ^ (prot)) & (HASH_SIZE - 1))




static hostdb_t *host_hash[HASH_SIZE];



inline static unsigned int host_key(const iphdr_t *ip) 
{
        return hkey(align_uint32(&ip->ip_src.s_addr), align_uint32(&ip->ip_dst.s_addr), ip->ip_p);
}




/*
 * Compare two IP address using the hash criteria.
 */
inline static int ip_match(hostdb_t *entry,
                           const struct in_addr *src, const struct in_addr *dst, uint8_t proto) 
{        
        return ( entry->saddr == align_uint32(&src->s_addr) &&
                 entry->daddr == align_uint32(&dst->s_addr) &&
                 entry->proto == proto );
}



/*
 * Loop all hostdb entry from bucket 'bucket',
 * and call ip_match() to see if the wanted IP is part of this entry.
 */
static hostdb_t *search(hostdb_t *bucket, const iphdr_t *ip) 
{        
        while ( bucket != NULL ) {
                
                if ( ip_match(bucket, &ip->ip_src, &ip->ip_dst, ip->ip_p) ) 
                        break;
            
                bucket = bucket->next;
        }
        
        return bucket;
}



/*
 * Add an hostdb entry to the hash table.
 */
static void host_add(hostdb_t **bucket, hostdb_t *h) 
{
        h->next = *bucket;
        if ( h->next )
                h->next->prev = h;
        *bucket = h;
}



/*
 * Delete an hostdb entry from the hash table. 
 */
static void host_free(hostdb_t **bucket, hostdb_t *h) 
{        
        if ( h->next )
                h->next->prev = h->prev;
        if ( h->prev )
                h->prev->next = h->next;
        else *bucket = h->next;

        free(h->pdata);
        free(h);
}




/**
 * hostdb_new:
 * @ip: the #ip header used to match this entry.
 *
 * Create a new hostdb entry and allocate space for plugin
 * to associate data with this entry.
 *
 * Returns: A new #hostdb structure or %NULL on error.
 */
hostdb_t *hostdb_new(packet_container_t *pkt, const iphdr_t *ip) 
{
        hostdb_t *new;
        int key = host_key(ip);
        
        new = malloc(sizeof(hostdb_t));
        if (! new ) {
                log(LOG_ERR,"malloc(%d).\n", sizeof(hostdb_t));
                return NULL;
        }

        new->pdata = calloc(1, plugin_get_highest_id() * sizeof(void *));
        if ( ! new->pdata ) {
                log(LOG_ERR,"malloc(%d).\n",  plugin_get_highest_id() * sizeof(void *));
                return NULL;
        }
        
        new->prev = NULL;
        new->refcount = 0;
        new->proto = ip->ip_p;
        new->saddr = align_uint32(&ip->ip_src.s_addr);
        new->daddr = align_uint32(&ip->ip_dst.s_addr);
              
        host_add(&host_hash[key], new);
        
        return new;
}



/**
 * hostdb_del:
 * @h: The #hostdb entry to delete.
 * @pid: The id for the calling plugin.
 *
 * Delete a #hostdb entry if the decreased refcount for this
 * entry is now 0.
 */
void hostdb_del(hostdb_t *h, const unsigned int pid)
{        
        h->pdata[pid] = 0;

        if ( --h->refcount == 0 ) 
                host_free(&host_hash[hkey(h->saddr, h->daddr, h->proto)], h);
}



/**
 * hostdb_search:
 * @ip: #ip header to match.
 *
 * Search for a host in the database.
 *
 * Returns: a pointer on the #hostdb entry if host was found or %NULL.
 */
hostdb_t *hostdb_search(const iphdr_t *ip) 
{
        return search(host_hash[host_key(ip)], ip);
}




void hostdb_init(void) 
{
        memset(host_hash, 0, sizeof(host_hash));
}














