/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *              Copyright (C) 2005 SUSE Linux Products GmbH                *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef POWERSAVE_MAIN_LOOP_H
#define POWERSAVE_MAIN_LOOP_H

#include <list>
#include "dbus_server.h"

class PM_interface;

/** @brief class representing a callback function inside the main loop
 *
 * Every callback has a unique id, an interval on which it is executed and
 * the callback function
*/
class Callback {
public:
	/** @brief contructor */
	Callback();

	/** @brief constructing a new callback object from the given values 
	 * 
	 * @param interval the interval on which the callback is executed
	 * @param function the callback function which will be called
	 */
	Callback(int interval, gboolean (*function)(gpointer data));

	/** @brief interval in which the callback function is executed */
	int interval;

	/** @brief the callback function */
	gboolean (*function)(gpointer data);

	/** @brief the id of the event source assigned by glib */
	int id;
};


/** @brief class implementing a main loop with glib
 *
 * Cares about callbacks and executes appropriate functions from
 * PM_Interface
 */
class MainLoop {
public:
	/** @brief constructor initializing the MainLoop object
	 *
	 * Constructor setting up the dbus connection and the gmain loop.
	 * For running the main loop, you have to call run().
	 */
	MainLoop();

	/** @brief runs the main loop
	 *
	 * Runs the main loop and blocks until quit() is called.
	 */
	void run();

	/** @brief quits the main loop */
	void quit();

 	/** @brief updates all callbacks within the main loop
	 *
	 * Updates all callbacks within the main loop. If a callback
	 * already exists, do nothing. This function should be called
	 * whenever there is a config/scheme/hardware change. It needs
	 * less ressources, so it is no problem to call it as often as
	 * needed.
	 */
	void updateCallbacks();

	/** @brief adds a callback to the main loop
	 *
	 * adds a callback function with the given interval to the gmain loop
	 *
	 * @param interval interval on which the function should be called
	 * @param function the function pointer serving as callback
	 */
	int addCallback(int interval, gboolean (*function)(gpointer data));

protected:

private:
	/** @brief callback for events on acpi/apm event file
	 *
	 * Function is called whenever there is an event on the hw event
	 * filedescriptor. Defined static to be used as gmain loop
	 * callback
	 *
	 * @param io_channel the io channel used by glib
	 * @param io_condition conditions on what we listen
	 * @param data user data
	 *
	 * @return true on success, false otherwise. If false, the
	 * callback is not called again until readded
	 */
	static gboolean hwEvent_callback(GIOChannel *io_channel,
					 GIOCondition io_condition,
					 gpointer data);

	/** @brief callback for reconnection to the hwEvent channel
	 *
	 * Function is called successively when hw event channel broke
	 * away until it is back again.. Defined static to be used as
	 * gmain loop callback
	 *
	 * @param data user data
	 *
	 * @return true on success, false otherwise. If false, the
	 * callback is never called again
	 */
	static gboolean hwEvent_reconnect(gpointer data);

	/** @brief connects to the hw event file (apm/acpi)
	 *
	 * Function is called successively when hw event channel broke
	 * away. Defined static to be used inside static callback
	 * functions.
	 *
	 * @return true on success, false otherwise.
	 */
	static gboolean hwEvent_connect();

	/** @brief callback for updating the cpu state inside PM_interface
	 *
	 * calls pm->updateCPUState();
	 *
	 * @return true on success, false otherwise. If false, the
	 * callback is never called again.
	 */
	static gboolean checkThrottling_callback(gpointer data);

	/** @brief callback for checking pending powersave events
	 *
	 * calls pm->checkEventTimeouts();
	 *
	 * @param data user data
	 *
	 * @return true on success, false otherwise. If false, the
	 * callback is never called again.
	 */
	static gboolean checkEventTimeouts_callback(gpointer data);

	/** @brief adds callbacks to the main loop
	 *
	 * Adds the default callbacks to the main loop.
	 */
	void setupCallbacks();

	/** @brief gmain loop object */
	GMainLoop *_gmain;

	/** @brief list storing all added callbacks */
	std::list<Callback> _callback_functions;

	DBus_Server _dbus_server;
};

#endif // POWERSAVE_MAIN_LOOP_H
