/*
 * Potamus: an audio player
 * Copyright (C) 2004, 2005, 2006, 2007 Adam Sampson <ats@offog.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <ao/ao.h>
#include <audiofile.h>
#include "buffer.h"
#include "format.h"
#include "input.h"
#include "input-audiofile.h"

typedef struct {
	AFfilehandle fh;
	double rate;
	AFframecount pos, length;
	int frame_size;
} aoaf;

static int audiofile_open(input *p, const char *fn) {
	aoaf *af = malloc(sizeof *af);
	if (af == NULL)
		g_error("out of memory");
	p->data = af;

	af->fh = afOpenFile(fn, "r", NULL);
	if (af->fh == AF_NULL_FILEHANDLE)
		return -1;

	af->pos = 0;
	af->length = afGetFrameCount(af->fh, AF_DEFAULT_TRACK);

	int format, width, channels;
	afGetVirtualSampleFormat(af->fh, AF_DEFAULT_TRACK, &format, &width);
	channels = afGetVirtualChannels(af->fh, AF_DEFAULT_TRACK);
	af->rate = afGetRate(af->fh, AF_DEFAULT_TRACK);

	// Get audiofile to convert floating-point formats to 24-bit integers,
	// and unsigned integers to signed integers, and everything to LE.
	switch (format) {
	case AF_SAMPFMT_FLOAT:
	case AF_SAMPFMT_DOUBLE:
		width = 24;
	default:
		break;
	}
	if (afSetVirtualSampleFormat(af->fh, AF_DEFAULT_TRACK,
	                             AF_SAMPFMT_TWOSCOMP, width) < 0)
		return -1;
	if (afSetVirtualByteOrder(af->fh, AF_DEFAULT_TRACK,
	                          AF_BYTEORDER_LITTLEENDIAN) < 0)
		return -1;

	af->frame_size = afGetVirtualFrameSize(af->fh, AF_DEFAULT_TRACK, 1);
	if (af->frame_size < 0)
		return -1;

	p->fmt.bits = width;
	p->fmt.rate = af->rate;
	p->fmt.channels = channels;
	p->fmt.byte_format = END_LITTLE;

	return 0;
}

static int audiofile_get_audio(input *p, buffer *buf) {
	// Use a small chunk size so that position updates aren't too
	// infrequent.
	const int chunk_size = 4096;
	aoaf *af = (aoaf *) p->data;

	unsigned char *out = buffer_reserve(buf, chunk_size);
	if (out == NULL)
		g_error("out of memory");

	int n = afReadFrames(af->fh, AF_DEFAULT_TRACK, out,
	                     chunk_size / af->frame_size);
	if (n < 0)
		return -1;
	af->pos += n;
	n *= af->frame_size;
	buf->used += n;

	return n;
}

static int audiofile_get_pos(input *p, double *pos) {
	aoaf *af = (aoaf *) p->data;

	*pos = af->pos / p->fmt.rate;

	return 0;
}

static int audiofile_get_len(input *p, double *len) {
	aoaf *af = (aoaf *) p->data;

	*len = af->length / p->fmt.rate;

	return 0;
}

static int audiofile_get_seekable(input *p) {
	return 1;
}

static int audiofile_set_pos(input *p, double pos) {
	aoaf *af = (aoaf *) p->data;

	AFframecount new_pos = afSeekFrame(af->fh, AF_DEFAULT_TRACK,
	                                   pos * p->fmt.rate);
	if (new_pos < 0)
		return -1;

	af->pos = new_pos;
	return 0;
}

static int audiofile_close(input *p) {
	aoaf *af = (aoaf *) p->data;

	if (af->fh != AF_NULL_FILEHANDLE)
		afCloseFile(af->fh);

	free(af);
	free(p);

	return 0;
}

input *input_new_audiofile(void) {
	input *p = input_alloc();

	p->open = audiofile_open;
	p->get_audio = audiofile_get_audio;
	p->get_pos = audiofile_get_pos;
	p->get_len = audiofile_get_len;
	p->get_seekable = audiofile_get_seekable;
	p->set_pos = audiofile_set_pos;
	p->close = audiofile_close;

	return p;
}
