/*
 * Potamus: an audio player
 * Copyright (C) 2004, 2005, 2006, 2007 Adam Sampson <ats@offog.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <math.h>
#include <samplerate.h>
#include <glib.h>
#include "buffer.h"
#include "format.h"
#include "input.h"
#include "process.h"

// The sample format conversion code is based on that from libsamplerate (but
// rewritten to avoid walking memory backwards).

int bytes_per_sample(int bits) {
	if (bits == 24) {
		return 4;
	} else {
		return (bits + 7) / 8;
	}
}

inline void reverse(uint8_t *p, int bytes) {
	for (int i = 0; i < (bytes / 2); i++) {
		uint8_t c = p[i];
		p[i] = p[bytes - i];
		p[bytes - i] = c;
	}
}

void copy_format(sample_format *out, const sample_format *in) {
	memcpy(out, in, sizeof *out);
}

int format_same(const sample_format *a, const sample_format *b) {
	return (a->rate == b->rate
	        && a->channels == b->channels
	        && a->bits == b->bits
	        && a->byte_format == b->byte_format);
}

static enum endianness get_native_endianness(void) {
	const uint32_t x = 1;
	if (*((uint8_t *) &x) == 1)
		return END_LITTLE;
	else
		return END_BIG;
}

void print_format(sample_format *fmt) {
	printf("%dHz %dch %dbit %d", fmt->rate, fmt->channels, fmt->bits, fmt->byte_format);
}

int convert_format(buffer *in, sample_format *in_fmt, buffer *out, sample_format *out_fmt, int processing, const char **error) {
	const int channels = in_fmt->channels;

	static SRC_STATE *src = NULL;
	static sample_format old_in_fmt, old_out_fmt;
	if (src == NULL
	    || !format_same(in_fmt, &old_in_fmt)
	    || !format_same(out_fmt, &old_out_fmt)) {
		// We only reset the src when the format changes so that
		// multipart files don't have discontinuities.

		if (src != NULL)
			src_delete(src);

		int error;
		// FIXME: The quality probably ought to be configurable.
		src = src_new(SRC_SINC_BEST_QUALITY, channels, &error);
		if (src == NULL)
			g_error("error creating src: %d", error);

		copy_format(&old_in_fmt, in_fmt);
		copy_format(&old_out_fmt, out_fmt);
	}

	static buffer *in_f = NULL, *out_f = NULL;
	if (in_f == NULL) {
		in_f = buffer_new();
		if (in_f == NULL)
			g_error("out of memory (in_f)");
		out_f = buffer_new();
		if (out_f == NULL)
			g_error("out of memory (out_f)");
	}

	const int nsamples_in = in->used / bytes_per_sample(in_fmt->bits);
	const double ratio = (1.0L * out_fmt->rate) / in_fmt->rate;
	const int nsamples_out = (nsamples_in * ratio) + channels;

	in_f->used = 0;
	out_f->used = 0;

	uint8_t *ip = in->data;
	float *data_in = (float *) buffer_reserve(in_f, nsamples_in * sizeof(float));
	if (data_in == NULL)
		g_error("out of memory (data_in)");

	float *d = data_in;
	int n = nsamples_in;
	enum endianness native = get_native_endianness();
	int is_le = (native == END_LITTLE);
	int is_native = (in_fmt->byte_format == END_NATIVE || in_fmt->byte_format == native);
	int bytes = bytes_per_sample(in_fmt->bits);
	while (n--) {
		int32_t val = 0;
		uint8_t *vp = (uint8_t *) &val;
		memcpy(vp + (is_le ? (sizeof val) - bytes : 0), ip, bytes);
		ip += bytes;
		if (!is_native)
			reverse(vp, bytes);
		*d++ = val / (8.0 * 0x10000000);
	}

	float *data_out = (float *) buffer_reserve(out_f, nsamples_out * sizeof(float));
	if (data_out == NULL)
		g_error("out of memory (data_out)");

	SRC_DATA data;
	data.data_in = data_in;
	data.data_out = data_out;
	data.input_frames = nsamples_in / channels;
	data.output_frames = nsamples_out / channels;
	data.src_ratio = ratio;
	data.end_of_input = 0;

	if (in_fmt->rate == out_fmt->rate) {
		// The rates are the same; simulate src_process rather than
		// actually doing it, since it isn't smart enough to avoid
		// resampling the data when there's no rate change.
		data.input_frames_used = data.input_frames;
		data.output_frames_gen = data.input_frames;
		memcpy(data_out, data_in, nsamples_in * sizeof(float));
	} else {
		if (src_process(src, &data) != 0) {
			*error = "Sample rate conversion failed";
			return -1;
		}
	}

	const size_t ip_size = data.input_frames_used * bytes_per_sample(in_fmt->bits) * channels;
	buffer_remove(in, ip_size);

	const int out_channels = out_fmt->channels;
	if (channels == 0 || out_channels == 0) {
		*error = "No input channels, or no output channels";
		return -1;
	}
	int channel_map[channels][out_channels];
	int channel_map_used[channels];
	for (int i = 0; i < channels; i++)
		channel_map_used[i] = 0;
	int max_channels = (channels > out_channels) ? channels : out_channels;
	for (int i = 0; i < max_channels; i++) {
		int p = channel_map_used[i % channels]++;
		channel_map[i % channels][p] = i % out_channels;
	}

	const size_t op_size = data.output_frames_gen * bytes_per_sample(out_fmt->bits) * out_fmt->channels;
	uint8_t *op = buffer_reserve(out, op_size + 1);
	if (op == NULL)
		g_error("out of memory (op)");
	d = data_out;
	is_native = (out_fmt->byte_format == END_NATIVE || out_fmt->byte_format == native);
	bytes = bytes_per_sample(out_fmt->bits);
	for (int frame = 0; frame < data.output_frames_gen; frame++) {
		float channel_sum[out_channels];
		int channel_count[out_channels];
		for (int och = 0; och < out_channels; och++) {
			channel_sum[och] = 0.0;
			channel_count[och] = 0;
		}

		for (int ch = 0; ch < channels; ch++) {
			float f = *d++;
			for (int i = 0; i < channel_map_used[ch]; i++) {
				int och = channel_map[ch][i];
				channel_sum[och] += f;
				channel_count[och]++;
			}
		}
		for (int ch = 0; ch < out_channels; ch++) {
			channel_sum[ch] /= channel_count[ch];
		}

		if (processing != 0) {
			process_audio(channel_sum, out_channels, processing);
		}

		for (int och = 0; och < out_channels; och++) {
			float scaled_value = channel_sum[och] * (8.0 * 0x10000000);
			int32_t val;
			if (scaled_value >= (1.0 * 0x7FFFFFFF)) {
				val = 0x7FFFFFFF;
			} else if (scaled_value <= (-8.0 * 0x10000000)) {
				val = -0x80000000;
			} else {
				val = lrintf(scaled_value);
			}

			val >>= 8 * ((sizeof val) - bytes);
			uint8_t *vp = (uint8_t *) &val;
			if (!is_le)
				vp += (sizeof val) - bytes;
			if (!is_native)
				reverse(vp, bytes);
			memcpy(op, vp, bytes);
			op += bytes;
		}
	}
	out->used += op_size;

	return 0;
}
