/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxdef-globals.h"

#include "polyxdef-ui-monomers.h"
#include "polyxdef-ui-polchemdef.h"



gboolean 
polyxdef_ui_polchemdef_wnd_setup_monomers_notebook_page (PxmDefCtxt *defctxt,
							 GladeXML *xml)
{
  GtkWidget *widget = NULL;


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->polchemdef != NULL);
  
 
  /* The monomers' notebook page must be correctly set up:
   */
  widget = glade_xml_get_widget (xml, "monomers_available_vbox");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "monomers_available_vbox", widget);

  if (FALSE == 
      polyxdef_ui_polchemdef_wnd_setup_monomers_treeview (defctxt))
    {
      g_critical (_("%s@%d: failed to set up the monomers' treeview\n"),
	     __FILE__, __LINE__);
      
      return FALSE;
    }
  
  /* Each polymer chemistry definition knows how may authorized
   * letters it can handle to describe a monomer code.
   */
  widget = glade_xml_get_widget (xml, "monomers_codelen_spinbutton");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "monomers_codelen_spinbutton", widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "value_changed",
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_monomer_codelen_value_changed),
     defctxt);

  gtk_spin_button_set_value (GTK_SPIN_BUTTON (widget),
			     (gdouble) defctxt->polchemdef->codelen);
  
  widget = glade_xml_get_widget (xml, "polyxdef_monomer_add_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polyxdef_monomer_add_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_monomer_add_button),
		    defctxt);

  widget = glade_xml_get_widget (xml, "polyxdef_monomer_remove_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polyxdef_monomer_remove_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_monomer_remove_button),
		    defctxt);

  return TRUE;
}



gboolean
polyxdef_ui_polchemdef_wnd_setup_monomers_treeview (PxmDefCtxt *defctxt)
{
  GtkWidget *vbox = NULL;
  
  GPtrArray *monomerGPA = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkWidget *sw = NULL;
  

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  g_assert (defctxt->polchemdef != NULL);

  monomerGPA = defctxt->polchemdef->monomerGPA;
  g_assert (monomerGPA != NULL);
  
  vbox = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			   "monomers_available_vbox");
  g_assert (vbox != NULL);
  


  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  
  

  /* Create the treeview model.
   */
  model = 
    polyxdef_ui_polchemdef_wnd_create_monomers_treeview_model (monomerGPA);

  /* Set to the model a datum with a pointer to the window!
  */
  g_object_set_data (G_OBJECT (model), "window", defctxt->definition_wnd);

  /* And now set to the window a datum with a pointer to the model, so
     that later the model is accessible (add/remove button handlers).
  */
  g_object_set_data (G_OBJECT (defctxt->definition_wnd), 
		     "monomer_treeview_model", model);
  
  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);

  g_object_unref (G_OBJECT (model));

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
     is accessible later (remove item handler).
  */
  g_object_set_data (G_OBJECT (defctxt->definition_wnd), 
		     "monomer_treeview", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);
  

  /* Monomer Name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Name"), renderer,
     "text", COLUMN_MNM_NAME,
     "editable", COLUMN_MNM_EDITABLE,
     NULL);
  
  g_signal_connect 
    (G_OBJECT (renderer), "edited",
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_monomers_treeview_cell_edited), defctxt);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MNM_NAME);


  /* Monomer Code column.
   */
  renderer = gtk_cell_renderer_text_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Code"), renderer,
     "text", COLUMN_MNM_CODE,
     "editable", COLUMN_MNM_EDITABLE,
     NULL);
  
  g_signal_connect 
    (G_OBJECT (renderer), "edited",
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_monomers_treeview_cell_edited), defctxt);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MNM_CODE);


  /* Monomer Formula column.
   */
  renderer = gtk_cell_renderer_text_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Formula"), renderer,
     "text", COLUMN_MNM_FORMULA,
     "editable", COLUMN_MNM_EDITABLE,
     NULL);
  
  g_signal_connect 
    (G_OBJECT (renderer), "edited",
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_monomers_treeview_cell_edited), defctxt);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MNM_FORMULA);


  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);
  
  return TRUE;
}


GtkTreeModel *
polyxdef_ui_polchemdef_wnd_create_monomers_treeview_model (GPtrArray *monomerGPA)
{
  gint iter = 0;

  GtkListStore *model;
  GtkTreeIter tree_iter;

  PxmMonomer *monomer;
  
  
    
  /* Create the list store.
   */
  model = gtk_list_store_new (COLUMN_MNM_COL_COUNT, /*Numb. of columns*/
			      G_TYPE_STRING, /* monomer name */
			      G_TYPE_STRING, /* monomer code */
			      G_TYPE_STRING /* monomer formula */,
			      G_TYPE_BOOLEAN /* editability */);

  /* Add the items if GPA is non-null and there are items in it.
   */
  if (monomerGPA != NULL)
    {
      for (iter = 0 ; iter < monomerGPA->len ; iter++)
	{
	  /* Acquire an iterator.
	   */
	  gtk_list_store_append (model, &tree_iter);
	  
	  monomer = g_ptr_array_index (monomerGPA, iter);
	  g_assert (monomer != NULL);
	  
	  gtk_list_store_set (model, &tree_iter,
			      
			      COLUMN_MNM_NAME, 
			      monomer->name,
			      
			      COLUMN_MNM_CODE, 
			      monomer->code,
			      
			      COLUMN_MNM_FORMULA, 
			      monomer->formula,

			      COLUMN_MNM_EDITABLE, TRUE,
			      -1);
	}
    }
  
  return GTK_TREE_MODEL (model);
}


void
polyxdef_ui_polchemdef_wnd_monomers_treeview_cell_edited (GtkCellRendererText *cell,
							  const gchar *path_string,
							  const gchar *new_text,
							  gpointer data)
{
  PxmDefCtxt *defctxt = data;
  
  GtkTreeModel *model = NULL;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
  GtkTreeIter tree_iter;
  
  PxmMonomer *monomer = NULL;
  
  GPtrArray *monomerGPA = NULL;

  gint *column;
  gint idx;
  
  gsize size = 0;
  
  gint result = 0;
  gint codelen = -1;
  
  gchar *old_text = NULL;
  gchar *help = NULL;
  

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  monomerGPA = defctxt->polchemdef->monomerGPA;
  g_assert (monomerGPA != NULL);

  g_assert (defctxt->polchemdef->atomGPA != NULL);
  




  model = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			     "monomer_treeview_model");
  g_assert (model != NULL);

  column = g_object_get_data (G_OBJECT (cell), "column");

  gtk_tree_model_get_iter (model, &tree_iter, path);

  switch (GPOINTER_TO_INT (column))
    {
    case COLUMN_MNM_NAME:
      {
	/* Before proceeding and accepting the new text, we should 
	 * make sure that no other monomer already exists by the same
	 * name.
	 */
	result = pxmchem_monomer_get_index_by_name ((gchar *) new_text,
						    monomerGPA);

	if (-1 != result)
	  {
	    help = g_strdup_printf (_("A monomer by the same name"
				      " exists already: '%s'"),
				    new_text);
	    
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
					      help, 
					      POLYXMASS_MEDI_MSG_TIMEOUT);
	    g_free (help);
	    
	    return ;
	  }

	/* At this point, all is OK, we can make the replacement.
	 */
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the item that is edited.
	 */
	idx = gtk_tree_path_get_indices (path) [0];

	monomer = g_ptr_array_index (monomerGPA, idx);
	g_assert (monomer != NULL);
	
	pxmchem_monomer_set_name (monomer, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
	
	gtk_list_store_set (GTK_LIST_STORE (model), &tree_iter, 
			    column,
			    (gchar *) new_text,
			    -1);
      }
      break;

    case COLUMN_MNM_CODE:
      {
	/* Before proceeding and accepting the new text, we should
	 * make sure that no other monomer already exists by the same
	 * code. But first yet, verify that the length of the new code
	 * is not greater than authorized for _this_ polymer chemistry
	 * definition.
	 */
	size = strlen ((gchar *) new_text);
	g_assert (size < G_MAXINT) ;
	codelen = (gint) size ;

	if (codelen > defctxt->polchemdef->codelen)
	  {
	    gdk_beep ();

	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd,
					      _("Code too long"), 
					      POLYXMASS_MEDI_MSG_TIMEOUT);
	    return ;
	  }
	
	result = pxmchem_monomer_get_index_by_code ((gchar* ) new_text, 
						    monomerGPA);
	
	if (-1 != result)
	  {
	    help = g_strdup_printf (_("A monomer by the same code"
				      " exists already: '%s'"),
				    new_text);
	        
	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd, 
					      help, 
					      POLYXMASS_MEDI_MSG_TIMEOUT);
	    g_free (help);
	    
	    return ;
	  }

	/* At this point, all is OK, we can make the replacement.
	 */
        gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the item that is edited.
	 */
	idx = gtk_tree_path_get_indices (path) [0];

	monomer = g_ptr_array_index (monomerGPA, idx);
	g_assert (monomer != NULL);
	
	pxmchem_monomer_set_code (monomer, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
	
	gtk_list_store_set (GTK_LIST_STORE (model), &tree_iter, 
			    column,
			    (gchar *) new_text,
			    -1);

	/* Now we should update the string in the polchemdef that
	   represents all the codes available in the polchemdef. The
	   delimiter between each monomer code in the string is a
	   global variable.
	*/
	g_assert (TRUE == 
		  pxmchem_polchemdef_update_delim_codes 
		  (defctxt->polchemdef, 
		   libpolyxmass_globals_delim));
	
      }
      break;

    case COLUMN_MNM_FORMULA:
      {
	/* Before proceeding and accepting the new text, we should
	 * check that the formula in it is valid.
	 */
	help = g_strdup (new_text);
  	result = pxmchem_formula_check (help, defctxt->polchemdef->atomGPA);
	g_free (help);
	
	if (FALSE == (gboolean) result)
	  {
	    help = g_strdup_printf (_("The new formula"
				      " is invalid: '%s'"),
				    new_text);

	    polyxmass_timeoutmsg_message_set 
	      ((GtkWindow *) defctxt->definition_wnd, 
					      help, 
					      POLYXMASS_MEDI_MSG_TIMEOUT);
	    g_free (help);
	    
	    return ;
	  }

	/* At this point, all is OK, we can make the replacement.
	 */
	gtk_tree_model_get (model, &tree_iter, column, &old_text, -1);
	g_free (old_text);

	/* Get the index of the item that is edited.
	 */
	idx = gtk_tree_path_get_indices (path) [0];

	monomer = g_ptr_array_index (monomerGPA, idx);
	g_assert (monomer != NULL);
	
	pxmchem_monomer_set_formula (monomer, (gchar *) new_text);

	defctxt->is_modified = TRUE;
	polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
	
	gtk_list_store_set (GTK_LIST_STORE (model), &tree_iter, 
			    column,
			    (gchar *) new_text,
			    -1);
      }
      break;
    }

  gtk_tree_path_free (path);
}


void
polyxdef_ui_polchemdef_wnd_monomer_codelen_value_changed  (GtkSpinButton *
							   spinbutton,
							   gpointer data)
{
  PxmDefCtxt *defctxt = data;
  
  gchar *help = NULL;
  
  int codelen_old = -1;
  int codelen_new = -1;
  

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  g_assert (defctxt->polchemdef != NULL);



  codelen_old = defctxt->polchemdef->codelen;
  codelen_new = gtk_spin_button_get_value_as_int (spinbutton);
  
  if (codelen_new <= 0)
    {
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 _("The code length must be > 0"), 
	 POLYXMASS_MEDI_MSG_TIMEOUT); 
      return ;
    }
  
  if (codelen_new != codelen_old)
    {
      pxmchem_polchemdef_set_codelen (defctxt->polchemdef, codelen_new);

      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

      help = g_strdup_printf (_("New code length: '%d'"), 
			      defctxt->polchemdef->codelen);
      
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd,  
	 help, 
	 POLYXMASS_MEDI_MSG_TIMEOUT); 

      g_free (help);
    }
  
  return;
}


void
polyxdef_ui_polchemdef_wnd_monomer_add_button (GtkWidget *widget, 
					       gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;

  GtkTreeIter tree_iter;
  GtkTreeIter tree_iter_sel;

  GtkTreePath *path = NULL;
  GtkTreeSelection* tree_selection = NULL;

  gboolean result = FALSE;

  gint idx_mnm = -1;
  
  PxmMonomer *monomer = NULL;
  PxmMonomer *monomer_test = NULL;
  
  GPtrArray *monomerGPA = NULL;


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  g_assert (defctxt->polchemdef != NULL);

  monomerGPA = defctxt->polchemdef->monomerGPA;
  g_assert (monomerGPA != NULL);
  


  /* Get the monomer treeview model from the window pointer passed
   * a data parameter.
   */
  model = (GtkTreeModel *) 
    g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
		       "monomer_treeview_model");
  g_assert (model != NULL);

  treeview = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
				"monomer_treeview");
  g_assert (treeview != NULL);

  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  result = gtk_tree_selection_get_selected (tree_selection,
					    &model,
					    &tree_iter_sel);

  if (result == FALSE)
    {
      /* If there is no current selection, that means that the user
       * wants to add a monomer at then end of the currently displayed 
       * items. Just an append operation.
       */
      gtk_list_store_append (GTK_LIST_STORE (model), &tree_iter);      
    }
  else
    {
      /* There is a selection, so we understand that the user
       * wants to actually insert a new monomer below the one
       * that is currently selected. Get index of currently
       * selected item.
       */
      path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				      &tree_iter_sel);
      g_assert (path != NULL);

      gtk_list_store_insert_after (GTK_LIST_STORE (model),
				   &tree_iter,
				   &tree_iter_sel);
  
      gtk_tree_path_free (path);
    }

  /* After one or the other call to append or store, tree_iter
   * contains the location where the placement is going to happen.
   * Get a path for this iter and and index from  it in turn.
   */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);
      
  idx_mnm = gtk_tree_path_get_indices (path) [0];
  //printf ("idx_mnm is %d\n", idx_mnm);
  
  gtk_tree_path_free (path);

  /* Allocate a new monomer so that we can later fill its
   * member data.
   */
  monomer = pxmchem_monomer_new ();
  
  monomer->name = g_strdup (_("enter name here"));
  monomer->code = g_strdup (_("enter code here"));
  monomer->formula = g_strdup (_("enter formula here"));

  if (idx_mnm + 1 >= monomerGPA->len)
    {
      g_ptr_array_add (monomerGPA, monomer);

      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
    }
  else if (idx_mnm + 1 < monomerGPA->len)
    {
      monomer_test = g_ptr_array_insert_val (monomerGPA, idx_mnm, monomer);
      g_assert (monomer == monomer_test);

      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
    }
  else
    {
      g_error (_("%s@%d: failed to insert/add monomer item to array\n"),
	     __FILE__, __LINE__);
    }
  
  gtk_list_store_set (GTK_LIST_STORE (model), &tree_iter,
		      
		      COLUMN_MNM_NAME, 
		      monomer->name,

		      COLUMN_MNM_CODE, 
		      monomer->code,
		      
		      COLUMN_MNM_FORMULA, 
		      monomer->formula,
		      
		      COLUMN_MNM_EDITABLE, TRUE,
		      -1);

  /* Now we should update the string in the polchemdef that represents all
     the codes available in the polchemdef. The delimiter between each 
     monomer code in the string is a global variable.
  */
  g_assert (TRUE == pxmchem_polchemdef_update_delim_codes 
	    (defctxt->polchemdef, 
	     libpolyxmass_globals_delim));

  /* Manage to have the new item added visible, so scroll to where
   * it is and select it.
   */
  path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				  &tree_iter);
  g_assert (path != NULL);

  gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (treeview),
				path,
				NULL,
				TRUE,
				0.5,
				0.5);

  tree_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_assert (tree_selection != NULL);
  
  gtk_tree_selection_select_iter (tree_selection,
				  &tree_iter);
  gtk_tree_path_free (path);
  
  polyxmass_timeoutmsg_message_set ((GtkWindow *) defctxt->definition_wnd,
				    _("Added new monomer; set its data!"), 
				    POLYXMASS_LONG_MSG_TIMEOUT); 

  return;
}



void
polyxdef_ui_polchemdef_wnd_monomer_remove_button (GtkWidget *widget,
						  gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkTreeModel *model = NULL;
  GtkTreeIter tree_iter ;
  GtkTreeView *treeview = NULL;
  GtkTreeSelection *selection = NULL;
  GtkTreePath *path = NULL;
  
  PxmMonomer *monomer = NULL;
  
  GPtrArray *monomerGPA = NULL;

  gint idx = -1;

  gchar *help = NULL;
  

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  g_assert (defctxt->polchemdef != NULL);

  monomerGPA = defctxt->polchemdef->monomerGPA;
  g_assert (monomerGPA != NULL);
  



  model = (GtkTreeModel *) 
    g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
		       "monomer_treeview_model");
  g_assert (model != NULL);

  /* Get the monomer treeview from the window pointer passed
   * a data parameter.
   */
  treeview = 
    (GtkTreeView *) g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
				       "monomer_treeview");
  
  selection = gtk_tree_view_get_selection (treeview);

  if (gtk_tree_selection_get_selected (selection, NULL, &tree_iter))
    {
      path = gtk_tree_model_get_path (model, &tree_iter);

      idx = gtk_tree_path_get_indices (path) [0];

      gtk_list_store_remove (GTK_LIST_STORE (model), &tree_iter);

      monomer = g_ptr_array_remove_index (monomerGPA, idx);
      g_assert (monomer != NULL);
      
      help = g_strdup_printf (_("Removed monomer: '%s'"), monomer->name);
      
      pxmchem_monomer_free (monomer);
      
      gtk_tree_path_free (path);

      polyxmass_timeoutmsg_message_set ((GtkWindow *) defctxt->definition_wnd,
					help, 
					POLYXMASS_LONG_MSG_TIMEOUT); 

      g_free (help);

      defctxt->is_modified = TRUE;
      polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
    }

  /* Now we should update the string in the polchemdef that represents all
     the codes available in the polchemdef. The delimiter between each 
     monomer code in the string is a global variable.
  */
  g_assert (TRUE == pxmchem_polchemdef_update_delim_codes 
	    (defctxt->polchemdef, 
	     libpolyxmass_globals_delim));

  return;
}


