/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxmass-main.h"
#include "polyxmass-config.h"
#include "polyxmass-ui-main.h"
#include "polyxmass-ui-console.h"
#include "polyxmass-polchemdefctxt.h"
#include "polyxmass-timeoutmsg.h"
#include "polyxmass-window-mngmt.h"

#include "polyxdef-init.h"
#include "polyxcalc-init.h"
#include "polyxedit-init.h"
#include "polyxedit-main.h"


static gdouble details = FALSE;
static gdouble license = FALSE;
static gdouble version = FALSE;
static gdouble polchemdefversion = FALSE;
static gdouble polymersequenceversion  = FALSE;

static GOptionEntry options_entries[] = 
{
  { "polchemdef-version", 'p', 0, G_OPTION_ARG_NONE, &polchemdefversion, 
    "Version of the polymer chemistry definition", NULL },

  { "polymersequence-version", 's', 0, G_OPTION_ARG_NONE, &polymersequenceversion, 
    "Version of the polymer sequence", NULL },

  { "details", 'd', 0, G_OPTION_ARG_NONE, &details, 
    "Program Details", NULL },

  { "license", 'l', 0, G_OPTION_ARG_NONE, &license, 
    "Program License", NULL },

  { "version", 'v', 0, G_OPTION_ARG_NONE, &version, 
    "Program version", NULL },

  { NULL }
};


gint
main (int argc, char *argv[])
{
  gint iter = 0;

  const gchar *name = NULL;
  
  GtkWidget *textview = NULL;

  setlocale (LC_ALL, "");
  setlocale (LC_NUMERIC, "POSIX");
  setlocale (LC_MESSAGES, "");
  setlocale (LC_CTYPE, "POSIX");
  setlocale (LC_COLLATE, "POSIX");
  setlocale (LC_MONETARY, "POSIX");
  setlocale (LC_TIME, "POSIX");

  bindtextdomain (PACKAGE, LOCALEDIR);

#ifdef ENABLE_NLS
  bind_textdomain_codeset(PACKAGE, "UTF-8");
#endif

  textdomain (PACKAGE);
  
  name = g_get_user_name ();
  g_assert (name != NULL);

  if (0 == strcmp ("root", name))
    {
      g_error (_("%s@%d: for security reasons, running as root"
	       " is not possible\n"),
	     __FILE__, __LINE__);
    }
      
  /*
    debug_printf ((_("This is an internationalization test\n")));
  */

  /* This is called in all GTK applications. Arguments are parsed from
     the command line and are returned to the application. Make sure
     Gtk+ does not override our settings above even if they differ
     from the user's environment.
  */
  gtk_disable_setlocale();
  gtk_init (&argc, &argv);

  /* Now deal with the options passed as parameters to the program
   */
  /* First the '-' or '--' options.
   */
  polyxmass_main_parse_options (argc, argv);
  
  /* Initialize the program, so that later we can open files if names
     are leftover the params treated by the parsing function above.
  */
  if (FALSE == polyxmass_main_init_program ())
    {
      g_error (_("%s@%d: failed to initialize the application\n"),
	     __FILE__, __LINE__);
    }

  /* Construct the graphical interface of the program -the main window-.
   */
  polyxmass_main_wnd = polyxmass_ui_main_setup_wnd ();
  
  if (polyxmass_main_wnd == NULL)
    {
      g_error (_("%s@%d: failed to create the main program window\n"),
	     __FILE__, __LINE__);
    }

  
  /* Construct the graphical interface of the console window.
   */
  polyxmass_console_wnd = (GtkWindow*) polyxmass_console_setup_wnd ();
  
  if (polyxmass_console_wnd == NULL)
    {
      g_error (_("%s@%d: failed to create the console window\n"),
	     __FILE__, __LINE__);
    }

  /* Now that we have the pointer to the window where the messages are
     going to be logged, we can set up the log handlers with a pointer
     to the textview where the messages are to be printed.
  */
  textview = g_object_get_data (G_OBJECT (polyxmass_console_wnd),
				"console_textview");
  g_assert (textview != NULL);
  
  g_log_set_handler (G_LOG_DOMAIN,
		     G_LOG_LEVEL_WARNING,
		     polyxmass_globals_log_warning_msg,
		     (gpointer) textview);

  g_log_set_handler ("libpolyxmass",
		     G_LOG_LEVEL_WARNING,
		     polyxmass_globals_log_libpolyxmass_warning_msg,
		     (gpointer) textview);

  g_log_set_handler (G_LOG_DOMAIN,
		     G_LOG_LEVEL_CRITICAL,
		     polyxmass_globals_log_critical_msg,
		     (gpointer) textview);

  g_log_set_handler ("libpolyxmass",
		     G_LOG_LEVEL_CRITICAL,
		     polyxmass_globals_log_libpolyxmass_critical_msg,
		     (gpointer) textview);

  g_log_set_handler (G_LOG_DOMAIN,
		     G_LOG_LEVEL_MESSAGE,
		     polyxmass_globals_log_message_msg,
		     (gpointer) textview);

  g_log_set_handler ("libpolyxmass",
		     G_LOG_LEVEL_MESSAGE,
		     polyxmass_globals_log_libpolyxmass_message_msg,
		     (gpointer) textview);


  
  /*
    And now let's try to look at all the remaining stuff.
    We just take for granted that the remaining params are
    filenames of polymer sequences.
  */

  /* Remember that argv [0] is the name of the program !
   */
  iter = 1;
  
  while (argv [iter] != NULL)
    {
      if (NULL == polyxedit_main_prepare_new_context_open_file (argv [iter]))
	g_critical (_("%s@%d: failed to create a polymer sequence editor "
		      "context for file: '%s'.\n"),
		    __FILE__, __LINE__, argv [iter]);
      
      printf ("index %d is %s\n", iter, argv [iter]);
      iter++;
    }
  
  /* Finally start the Gtk+ loop... 
   */
  gtk_main ();
  


  /* Note that the exiting of this program is performed by clicking
     the Quit button, so look at the handler to this menu item. That
     handler ensures that the app is un-init'ed.
  */

  return 0;
}


gboolean
polyxmass_main_parse_options (int argc, char *argv[])
{
  GError *error = NULL;

  GOptionContext *context = g_option_context_new ("GNU polyxmass options");

  g_option_context_add_main_entries (context, options_entries, /* GETTEXT_PACKAGE */ "polyxmass");
  g_option_context_add_group (context, gtk_get_option_group (TRUE));

  g_option_context_set_ignore_unknown_options (context, FALSE);
  
  g_option_context_parse (context, &argc, &argv, &error);

 if (polchemdefversion)
    {
      polyxmass_globals_print_polchemdef_xml_file_version ();

      exit (0);
    }
  
 if (polymersequenceversion)
    {
      polyxmass_globals_print_polymer_sequence_xml_file_version ();

      exit (0);
    }
  
 if (details)
    {
      polyxmass_globals_print_greeting ();

      exit (0);
    }
  
  if (license)
    {
      fprintf (stdout, 
	       "%s is published under the GNU"
	       " General Public License Version 2\n"
	       "(check the COPYING file)\n\n",
	       PACKAGE);
      
      exit (0);
    }
  
  if (version)
    {
      fprintf (stdout, "%s version %s\n\n",
	       PACKAGE, VERSION);
      
      exit (0);
    }
  
  return TRUE;
}


gboolean
polyxmass_main_init_program (void)
{
  /* Initialize the libpolyxmass library's pxmchem part.
   */
  if (FALSE == libpolyxmass_init_library ())
    {
      g_critical (_("%s@%d: failed to initialize the libpolyxmass library.\n"),
	     __FILE__, __LINE__);
      
      polyxmass_main_uninit_program ();
      
      return FALSE;
    }
  
  userspec = libpolyxmass_userspec_new ();
  
  if (FALSE == polyxmass_config_userspec (userspec))
    {
      g_critical (_("%s@%d: failed to initialize the user specifications.\n"),
	     __FILE__, __LINE__);
      
      polyxmass_main_uninit_program ();
      
      return FALSE;
    }

  /* Get the number formats from the configuration file.
   */
  polyxmass_main_set_num_formats ();
  
  /* Allocate all the main global arrays that we'll need.
   */
  if (FALSE == polyxmass_main_allocate_global_arrays ())
    {
      g_error (_("%s@%d: failed to allocate the main global arrays\n"),
	     __FILE__, __LINE__);

      polyxmass_main_uninit_program ();
      
      exit (1);
    }

  /* Initialize the arrays that will contain catalogues and dictionaries
     so that GNU polyxmass knows at each instant where to look for
     atom definition and polymer chemistry data...
  */
  polyxmass_main_initialize_polchemdef_atomdef_arrays ();
  
  /* Initialize the array that will contain all the PxmWinMngmt instances
     describing all the windows opened in the program space.
  */
  polyxmass_winmngmtGPA_GPA = g_ptr_array_new ();
  

  if (FALSE == polyxdef_init ())
    {
      polyxdef_uninit ();
      return FALSE;
    }

  if (FALSE == polyxcalc_init ())
    {
      polyxcalc_uninit ();
      return FALSE;
    }
  
  if (FALSE == polyxedit_init ())
    {
      polyxedit_uninit ();
      return FALSE;
    }

  return TRUE;
}


gboolean
polyxmass_main_uninit_program (void)
{
  /* First make sure that all parts of the program (polyxdef,
     polyxcald, polyxedit) are clean.
   */
  if (FALSE == polyxdef_uninit ())
    return FALSE;
  
  if (FALSE == polyxcalc_uninit ())
    return FALSE;
  
  if (FALSE == polyxedit_uninit ())
    return FALSE;


  libpolyxmass_userspec_free (userspec);

  /* We have to free all the numerical format strings that we had allocated.
   */
  g_assert (libpolyxmass_globals_atom_num_format != NULL);
  g_free (libpolyxmass_globals_atom_num_format);
  
  g_assert (libpolyxmass_globals_monomer_num_format != NULL);
  g_free (libpolyxmass_globals_monomer_num_format);
  
  g_assert (libpolyxmass_globals_oligomer_num_format != NULL);
  g_free (libpolyxmass_globals_oligomer_num_format);
  
  g_assert (libpolyxmass_globals_polymer_num_format != NULL);
  g_free (libpolyxmass_globals_polymer_num_format);
  
  
  polyxmass_polchemdefctxt_GPA_free (polyxmass_polchemdefctxtGPA);
  
  polyxmass_main_uninitialize_polchemdef_atomdef_arrays ();
  
  /*
    Free the array that holds all the allocated PxmTimeoutmsg objects.
  */
  polyxmass_timeoutmsg_GPA_free (polyxmass_timeoutmsgGPA);
  
  /* Free the array that hold all the allocated winmngmt objects
     (themselves in different arrays).
   */
  polyxmass_winmngmt_GPA_of_GPA_free (polyxmass_winmngmtGPA_GPA);
  
  libpolyxmass_uninit_library ();
  
  /* Will take care to close the console window and the main program
     window.
  */
  gtk_main_quit ();

  return TRUE;
}

gboolean
polyxmass_main_initialize_polchemdef_atomdef_arrays (void)
{
  gint count = 0;
  
  
  /* We have to get to know where all the data available on the system
     are located. This is true for the atom definition files as is it for 
     the polymer chemistry definition files. 
     
     We'll store all these data in arrays (GPtrArray).
  */
  polyxmass_atomdefsGPA = g_ptr_array_new ();
  
  count = libpolyxmass_atomspec_parse_atom_defs_cat_files 
    (polyxmass_atomdefsGPA,
     CONFIG_APPEND,
     CONFIG_BOTH);

  g_assert (count != -1);

  polyxmass_polchemdefsGPA = g_ptr_array_new ();

  count = libpolyxmass_polchemdefspec_parse_polchem_defs_cat_files 
    (polyxmass_polchemdefsGPA,
     CONFIG_APPEND,
     CONFIG_BOTH);
  
  g_assert (count != -1);

  polyxmass_polchematomdicGPA = g_ptr_array_new ();
  
  count = libpolyxmass_atomspec_parse_polchem_defs_atom_defs_dic_files 
    (polyxmass_polchematomdicGPA, 
     CONFIG_APPEND, 
     CONFIG_BOTH);

  g_assert (count != -1);
  
#if 0
  {
    gint iter = 0;
    
    PxmPolchemdefSpec *ps = NULL;
    PxmAtomSpec *as = NULL;
    
    for (iter = 0; iter < polyxmass_atomdefsGPA->len ; iter++)
      {
	as = g_ptr_array_index (polyxmass_atomdefsGPA, iter);
	g_assert (as != NULL);
	
	/*
	  debug_printf (("as in polyxmass_atomdefsGPA: %s - %s - %s\n",
	  as->poltype, as->atomdef, as->file));
	*/
      }
    
    for (iter = 0; iter < polyxmass_polchemdefsGPA->len ; iter++)
      {
	ps = g_ptr_array_index (polyxmass_polchemdefsGPA, iter);
	g_assert (as != NULL);
	
	/*
	  debug_printf (("as in polyxmass_polchemdefsGPA: %s - %s - %s\n",
	  ps->type, ps->file, ps->dir));
	*/
      }
    
    for (iter = 0; iter < polyxmass_polchematomdicGPA->len ; iter++)
      {
	as = g_ptr_array_index (polyxmass_polchematomdicGPA, iter);
	g_assert (as != NULL);
	
	/*
	  debug_printf (("as in polyxmass_polchematomdicGPA: %s - %s - %s\n",
	  as->poltype, as->atomdef, as->file));
	*/
      }
  }
#endif   

  return TRUE;
}


gboolean
polyxmass_main_uninitialize_polchemdef_atomdef_arrays (void)
{
  if (polyxmass_atomdefsGPA != NULL)
    libpolyxmass_atomspec_GPA_free (polyxmass_atomdefsGPA);
  
  if (polyxmass_polchemdefsGPA != NULL)
    libpolyxmass_polchemdefspec_GPA_free (polyxmass_polchemdefsGPA);
  
  if (polyxmass_polchematomdicGPA != NULL)
    libpolyxmass_atomspec_GPA_free (polyxmass_polchematomdicGPA);
  
  return TRUE;
}


gboolean
polyxmass_main_set_num_formats (void)
{
  
  /****** atom number format *******
   */
  libpolyxmass_globals_atom_num_format = g_strdup ("%.10f") ;
 
  /****** monomer number format *******
   */
  libpolyxmass_globals_monomer_num_format = g_strdup ("%.5f") ;
    
  /****** oligomer number format *******
   */
  libpolyxmass_globals_oligomer_num_format = g_strdup ("%.4f") ;
     
  /****** polymer number format *******
   */
  libpolyxmass_globals_polymer_num_format = g_strdup ("%.3f") ;
  
  /*
    printf ("atom = %s; monomer = %s; oligomer = %s; polymer = %s\n",
    libpolyxmass_globals_atom_num_format,
    libpolyxmass_globals_monomer_num_format,
    libpolyxmass_globals_oligomer_num_format,
    libpolyxmass_globals_polymer_num_format);
  */
  
 return TRUE;
}


gboolean
polyxmass_main_allocate_global_arrays (void)
{
  polyxmass_polchemdefctxtGPA = g_ptr_array_new ();

  /* 
     We need a global array to store all the timed-out messages
     that are sent to widgets... This array is to be stuffed with objects
     of type PxmTimeoutmsg.
  */
  polyxmass_timeoutmsgGPA = g_ptr_array_new ();
  
  return TRUE;
}

