// ------------------------------------------------------------------------- //
// $Id: primitive.cpp,v 1.45 2003/05/20 21:15:13 pandr Exp $
// ------------------------------------------------------------------------- //
 
/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

// ------------------------------------------------------------------------- //

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if HAVE_GL_GLU_H
# ifndef __GNUC__
#  ifdef _WIN32
#   include <windows.h>
#  endif
# endif
# include <GL/glu.h>
#endif

#if HAVE_OPENGL_GLU_H   // MACOS
# include <OpenGL/glu.h>
#endif

#if HAVE_ALGORITHM
# include <algorithm>
#endif

#include "primitive.h"
#include "image.h"
#include "mainwindow.h"

// ------------------------------------------------------------------------- //

int Primitive::_num_prims_drawn;

void Primitive::init()
{
	_num_vertices   = 0;
	_vertices       = 0;
	_tex_coords     = 0;
	_colors         = 0;
	_blended_colors = 0;

	_use_tex_coords = false;
	_use_colors     = false;
}

Primitive::Primitive() : _tile(0)
{
	init();
}

Primitive::Primitive(Ref<Tile> tile) : _tile(0)
{
	_tile = tile;
	init();

	set_prim_type(GL_QUADS);
	set_num_vertices(4);

	v3* vertices = new v3[4];
	vertices[0].set(        0.0f,          0.0f, 0.0f);
	vertices[1].set(tile->width(),          0.0f, 0.0f);
	vertices[2].set(tile->width(), tile->height(), 0.0f);
	vertices[3].set(        0.0f, tile->height(), 0.0f);

	set_vertices(vertices);

	set_use_tex_coords(true);

	set_texture(tile->get_texture());

	v2* texcoords = new v2[4];
	v2  tex_dim(tile->get_texture()->get_width(), tile->get_texture()->get_height());
	texcoords[0].set(tile->left(),  tile->bottom());
	texcoords[0] /= tex_dim;
	texcoords[1].set(tile->right(), tile->bottom());
	texcoords[1] /= tex_dim;
	texcoords[2].set(tile->right(), tile->top());
	texcoords[2] /= tex_dim;
	texcoords[3].set(tile->left(),  tile->top());
	texcoords[3] /= tex_dim;
	set_tex_coords(texcoords);
}

Primitive::Primitive(Texture* tex, uint width, uint height) : _tile(0)
{
	init();

	set_prim_type(GL_QUADS);
	set_num_vertices(4);

	if (width == 0 && height == 0) {
		width  = tex->get_width();
		height = tex->get_width();
	}

	v3* vertices = new v3[4];
	vertices[0].set( 0.0f,   0.0f, 0.0f);
	vertices[1].set(width,   0.0f, 0.0f);
	vertices[2].set(width, height, 0.0f);
	vertices[3].set( 0.0f, height, 0.0f);

	set_vertices(vertices);

/*	set_use_colors(true);

	rgba* colors = new rgba[4];
	colors[0].set(255, 255, 255, 255);
	colors[1].set(255, 255, 255, 255);
	colors[2].set(255, 255, 255, 255);
	colors[3].set(255, 255, 255, 255);
	set_colors(colors);
*/

	set_use_tex_coords(true);

	set_texture(tex);

	v2* texcoords = new v2[4];
	float texwidth  = (float)width  / tex->get_width();
	float texheight = (float)height / tex->get_height();
	texcoords[0].set(0.0f, 0.0f);
	texcoords[1].set(texwidth, 0.0f);
	texcoords[2].set(texwidth, texheight);
	texcoords[3].set(0.0f, texheight);
	set_tex_coords(texcoords);
}

Primitive::~Primitive()
{
	if (_vertices)   delete [] _vertices;
	if (_tex_coords) delete [] _tex_coords;
	if (_colors)     delete [] _colors;
	if (_blended_colors) delete [] _blended_colors;
}

void Primitive::line_prim(const v3& end_pos)
{
	if (!_vertices) {
		set_prim_type(GL_LINES);
		_num_vertices = 2;
		_vertices = new v3[2];
		_vertices[0].set(0.0f, 0.0f, 0.0f);
		_use_tex_coords = false;
		_use_colors = false;
	}
	_vertices[1] = end_pos;
}

void Primitive::box_prim()
{
	set_prim_type(GL_LINES);
	_num_vertices = 24;
	_vertices = new v3[24];

	_vertices[0].set(-5.0f, -5.0f, -5.0f);
	_vertices[1].set( 5.0f, -5.0f, -5.0f);

	_vertices[2].set( 5.0f, -5.0f, -5.0f);
	_vertices[3].set( 5.0f, -5.0f,  5.0f);

	_vertices[4].set( 5.0f, -5.0f,  5.0f);
	_vertices[5].set(-5.0f, -5.0f,  5.0f);

	_vertices[6].set(-5.0f, -5.0f,  5.0f);
	_vertices[7].set(-5.0f, -5.0f, -5.0f);


	_vertices[8].set(-5.0f,  5.0f, -5.0f);
	_vertices[9].set( 5.0f,  5.0f, -5.0f);

	_vertices[10].set( 5.0f,  5.0f, -5.0f);
	_vertices[11].set( 5.0f,  5.0f,  5.0f);

	_vertices[12].set( 5.0f,  5.0f,  5.0f);
	_vertices[13].set(-5.0f,  5.0f,  5.0f);

	_vertices[14].set(-5.0f,  5.0f,  5.0f);
	_vertices[15].set(-5.0f,  5.0f, -5.0f);


	_vertices[16].set(-5.0f,  5.0f, -5.0f);
	_vertices[17].set(-5.0f, -5.0f, -5.0f);

	_vertices[18].set( 5.0f,  5.0f, -5.0f);
	_vertices[19].set( 5.0f, -5.0f, -5.0f);

	_vertices[20].set( 5.0f,  5.0f,  5.0f);
	_vertices[21].set( 5.0f, -5.0f,  5.0f);

	_vertices[22].set(-5.0f,  5.0f,  5.0f);
	_vertices[23].set(-5.0f, -5.0f,  5.0f);

	_use_tex_coords = false;
	_use_colors = false;

}

void Primitive::default_prim()
{
	set_prim_type(GL_QUADS);

	_num_vertices = 4;

	_vertices = new v3[4];
	_vertices[0].set(10.0f, 10.0f, 0.0f);
	_vertices[1].set(100.0f, 10.0f, 0.0f);
	_vertices[2].set(100.0f, 100.0f, 0.0f);
	_vertices[3].set(10.0f, 100.0f, 0.0f);

	_use_tex_coords = false;

	_use_colors = true;
	_colors    = new rgba[4];
	_colors[0].set(255,   0,   0, 255);
	_colors[1].set(  0, 255,   0, 255);
	_colors[2].set(  0,   0, 255, 255);
	_colors[3].set(255, 255, 255,   0);

}

void Primitive::blend_colors(rgba& color)
{
	memcpy(_blended_colors, _colors, _num_vertices*4);
	GLubyte *p = _blended_colors;
	for (int i = 0; i < _num_vertices; i++) {
		*p++ = (*p * color.r() + 255) >> 8;
		*p++ = (*p * color.g() + 255) >> 8;
		*p++ = (*p * color.b() + 255) >> 8;
		*p++ = (*p * color.a() + 255) >> 8;
	}
}

void Primitive::draw(rgba& color)
{
	_num_prims_drawn++;
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, 0, _vertices);
	if (_use_colors) {
		assert(_colors);
		assert(_blended_colors);

		GLubyte *colors;

		if (color.r() == 255 && color.g() == 255 &&
		    color.b() == 255 && color.a() == 255) {
			// Just use original colors
			colors = &_colors[0].r();
		}
		else {
			// Blend given color with vertex colors
			blend_colors(color);
			colors = _blended_colors;
		}
		glEnableClientState(GL_COLOR_ARRAY);
		glColorPointer(4, GL_UNSIGNED_BYTE, 0, colors);
	} else {
		glColor4ub(color.r(), color.g(), color.b(), color.a());
	}
	if (!g_renderer->get_wireframe() && _use_tex_coords) {
		assert(_tex_coords);
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(2, GL_FLOAT, 0, _tex_coords);
		glEnable(GL_TEXTURE_2D);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
		glBindTexture(GL_TEXTURE_2D, _texture->get_handle());
	}

	glDrawArrays(_GL_prim_type, 0, _num_vertices);

	if (!g_renderer->get_wireframe() && _use_tex_coords) {
		glDisableClientState(GL_TEXTURE_COORD_ARRAY);
		glDisable(GL_TEXTURE_2D);
	}
	if (_use_colors)
		glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);

#if 0
	if (1 || !_use_tex_coords) {
	glColor3f(0.0f, 1.0f, 0.0f);
	glBegin(GL_LINES);
	glVertex3f(0.0f, 0.0f, 0.0f);
	glVertex3f(10.0f, 0.0f, 0.0f);
	glEnd();
	glPointSize(4.0f);
	glBegin(GL_POINTS);
	glVertex3f(0.0f, 0.0f, 0.0f);
	glEnd();
	}
#endif

}

void Primitive::set_colors(rgba* colors)
{
	_colors = colors;
	assert(!_blended_colors);
	_blended_colors = new GLubyte[_num_vertices*4];
}

void Primitive::offset(float x, float y, float z)
{
	for (int i=0; i< _num_vertices; i++)
		_vertices[i] += v3(x, y, z);
}

// ------------------------------------------------------------------------- //

Texture::Texture(uint w, uint h)
{
	//std::cout << "Made texture at " << this << std::endl;
	assert(is_pow_two(w) && is_pow_two(h));
	_image = new Image(w,h,4);
	gl_bind();
}

/*
Texture::Texture(const Image& img)
{
	int w = img.get_width();
	int h = img.get_height();
	w = next_pow_two(w);
	h = next_pow_two(h);
	_width  = w;
	_height = h;
	gl_bind();
	load_from_subimage(img, 0, 0, img.get_width(), img.get_height());
}
*/

void Texture::gl_bind()
{
	glGenTextures(1, &_tex_name);
}

Texture::~Texture()
{ 
	log_debug("Destroyed texture (this: %x)\n", this);
	glDeleteTextures(1, &_tex_name);
	delete _image;
}

void Texture::load_from_subimage(const Image& img, uint x, uint y, 
		uint w, uint h)
{
	assert(x+w<=img.get_width());
	assert(y+h<=img.get_height());
	assert(w <= get_width() && h <= get_height());

	glBindTexture(GL_TEXTURE_2D, _tex_name);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glPixelStorei(GL_UNPACK_SKIP_PIXELS, x);
	glPixelStorei(GL_UNPACK_SKIP_ROWS, y);
	glPixelStorei(GL_UNPACK_ROW_LENGTH, img.get_width());
	if (w == get_width() && h == get_height()) {
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, get_width(), get_height(),
			0, GL_RGBA, GL_UNSIGNED_BYTE, img.get_buffer());
	}
	else {
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, get_width(), get_height(),
			0, GL_RGBA, GL_UNSIGNED_BYTE, 0);
		glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, w, h, 
				GL_RGBA, GL_UNSIGNED_BYTE, img.get_buffer());
	}

	PrintGLError;
}

void Texture::load()
{
	glBindTexture(GL_TEXTURE_2D, _tex_name);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glPixelStorei(GL_UNPACK_SKIP_PIXELS, 0);
	glPixelStorei(GL_UNPACK_SKIP_ROWS, 0);
	glPixelStorei(GL_UNPACK_ROW_LENGTH, get_width());
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, get_width(), get_height(),
			0, GL_RGBA, GL_UNSIGNED_BYTE, get_image()->get_buffer());
}

// ------------------------------------------------------------------------- //

Tile::~Tile()
{
	TileBank::instance()->free_tile(this, _texture);
}

void Tile::copy_from_subimage(Image *img, uint xoffset, uint yoffset)
{
	_texture->get_image()->copy_from(img, iv2(left(), bottom()), 
			Rect(xoffset, yoffset, xoffset+width(), yoffset+height()));
}

Tile::handle TiledTexture::get_tile(iv2 size)
{
	if (get_width() < size.x() || get_height() < size.y() + _baseline) {
		// This will never fit -- just too big
		return 0;
	}
	if (get_width() - _current_line_used.x() < size.x())
	{
		// Will not fit on this line -- too wide
		if (get_height() < size.y() + _baseline + _current_line_used.y() ) {
			// Will not fit on next line either -- too high
			return 0;
		}
		// Newline
		_baseline += _current_line_used.y();
		_current_line_used = iv2(0,0);
	}

	iv2 lower_left(_current_line_used.x(), _baseline);
	Tile::handle tile = new Tile(Rect(lower_left, lower_left + size), this);

	int h_increase = max(0, _baseline + size.y() - _current_line_used.y());
	_current_line_used += iv2(size.x(), h_increase);

	_tiles_used++;

	return tile;
}

bool TiledTexture::free_tile(Tile *tile)
{
	assert(tile->get_texture() == this);
	_tiles_used--;
	return _tiles_used == 0;
}

// ------------------------------------------------------------------------- //

TileBank TileBank::_instance;

TileBank* TileBank::instance()
{
	return &_instance;
}

TileBank::TileBank() : _texture_size(256, 256)
{
}

TileBank::~TileBank()
{
	for (_texture_iterator i = _textures.begin();
	     i != _textures.end(); i++)
	{
		delete (*i);
	}
}

bool TileBank::new_texture()
{
	log_debug("New texture, no %i", _textures.size());
	if (_textures.size() > 100) { // FIXME MAGIC NUMBER
		log_warning("No more room for textures in texture-bank");
		return false;
	}
	TiledTexture *t = new TiledTexture(_texture_size.x(), _texture_size.y());
	_textures.push_back(t);
	log_debug("Created texture (now %d in tilebank)", _textures.size());
	return true;
}

Tile::handle TileBank::get_tile(iv2 size)
{
	Tile::handle h(0);
	for (_texture_iterator i = _textures.begin();
	     i != _textures.end(); i++)
	{
		TiledTexture* t = *i;
		h = t->get_tile(size);
		if (h) return h;
	}
	if (size.x() <= _texture_size.x() && size.y() <= _texture_size.y())
	{
		log_info("Creating new texture in texture-bank");
		if (new_texture()) {
			return _textures.back()->get_tile(size);
		}
		assert(0); // Catch me...
	}
	log_warning("No room for tile");
	return 0;
}

void TileBank::load_textures()
{
	uint mem_used = 0;
	for (_texture_iterator i = _textures.begin();
	     i != _textures.end(); i++)
	{
		TiledTexture* t = *i;
		t->load();
		mem_used += 4 * t->get_width() * t->get_height();
	}
	log_info("TileBank: %d textures loaded using %d bytes of texture ram", 
			_textures.size(), mem_used);
}

Texture* TileBank::debug_texture(uint i)
{
	return _textures[i];
}

void TileBank::free_tile(Tile* tile, TiledTexture* texture)
{
	if (texture->free_tile(tile)) {
		delete texture;
		_textures.erase(std::find(_textures.begin(), _textures.end(), texture));
		//std::cout << "Freed texture (" << _textures.size() << " left in tilebank)\n";
	}
}

// ------------------------------------------------------------------------- //

