/*
 * $Id: paragraph.c,v 1.43 2001/10/27 18:56:41 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "anchor.h"
#include "const.h"
#include "debug.h"
#include "document.h"
#include "documentdata.h"
#include "history.h"
#include "image.h"
#include "os.h"
#include "paragraph.h"
#include "prefsdata.h"
#include "search.h"
#include "uncompress.h"
#include "util.h"


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define NO_IMAGE    0


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
/* Used to keep track of multi-line anchors */
static Boolean multilineAnchor;

/* Used to keep track of multi-line italic */
static Boolean multilineItalic;

/* Used to keep track of multi-line underlining */
static Boolean multilineUnderline;

/* Used to keep track of multi-line strikethrough */
static Boolean multilineStrike;

/* Used to know if margins should be added to the
 * current line or only the following lines */
static Boolean addMarginToCurrent;

/* Used to keep track of new line functions */
static Boolean newLine;

/* Used to keep track of horizontal rules */
static Boolean horizontalRule;

/* Used to keep track of images */
static Boolean image;

/* Used to keep track of the search pattern */
static Int16 patternCount;
static Int16 findPatternPos;
static Int16 findPatternLen;

/* Used for full justification */
static Int16 spaceCount     = 0;    /* How many spaces in current line */
static Int16 bigSpace       = 0;    /* How many get 1 extra */
static Int16 littleSpace    = 0;    /* Extra pixels in each */



/* Check if current font is the fixed width font */
static Boolean FixedWidthFont(void)
{
    return ( FntGetFont() == narrowFixedFont );
}



/* Set position and length of find pattern */
void SetFindPatternData
    (
    const Int16 pos,    /* position of find pattern */
    const Int16 len     /* length of find pattern */
    )
{
    findPatternPos = pos;
    findPatternLen = len;
}



/* Clear position and length of find pattern */
void ClearFindPatternData( void )
{
    findPatternPos = -1;
    findPatternLen = -1;
}



/* Find offset to paragraph in text stream */
static Int16 GetParagraphOffset
    (
    Header*     record,     /* pointer to record */
    Paragraph*  paragraph   /* pointer to paragraph */
    )
{
    Paragraph*  p;
    Int16       pOffset;

    pOffset = 0;
    p       = GET_PARAGRAPH( record, 0 );
    while ( p < paragraph ) {
        pOffset += p->size;
        p++;
    }
    return pOffset;
}



/* Align text/image */
static void AlignText
    (
    TextContext*        tContext,   /* pointer to text context */
    ParagraphContext*   pContext    /* pointer to paragraph context */
    )
{
    Int16 diff;

    diff = pContext->maxPixels - pContext->linePixels;

    bigSpace = littleSpace = 0;

    if ( diff <= 0 )
        return;

    if ( pContext->type == ALIGNMENT_CENTER )
        tContext->cursorX += diff / 2;
    else if ( pContext->type == ALIGNMENT_RIGHT )
        tContext->cursorX += diff;
    else if ( pContext->type == ALIGNMENT_JUSTIFY && ! multilineAnchor && 
              0 < spaceCount && diff < pContext->maxPixels / 4 ) {
        littleSpace = diff / spaceCount;    /* each space gets pixels */
        bigSpace    = diff % spaceCount;    /* this many get 1 extra */
    }
}



/* Check if given character is a non-visible character */
static Boolean CharIsSpace
    (
    Char c
    )
{
    return ( ( 0 <= c ) && ( c <= ' ' ) );
}



/* Set new font style. */
static void SetStyle
    (
    ParagraphContext*   pContext,   /* pointer to paragraph context */
    Int16               style       /* new font style */
    )
{
    if ( MINSTYLES <= style && style < MAXSTYLES ) {
        FntSetFont( GetStyle( style ) );
        pContext->fontHeight = FntLineHeight();
    }
}



/* Initialize a new block of italic text */
static void StartItalic
    (
    TextContext* tContext   /* tContext - pointer to text context */
    )
{
    tContext->italic.topLeft.x = tContext->cursorX;
}



/* Handle several lines with italic text */
static void ContinueItalic
    (
    TextContext* tContext   /* pointer to text context */
    )
{
  StartItalic( tContext );
}



/* Mark the end of a block of italic text */
static void StopItalic
    (
    TextContext*    tContext,   /* pointer to text context */
    const Int16     height      /* height of line */
    )
{
    if ( tContext->writeMode ) {
        tContext->italic.topLeft.y  = tContext->cursorY - height;
        tContext->italic.extent.x   = tContext->cursorX - 
                                      tContext->italic.topLeft.x + 1;
        tContext->italic.extent.y   = height / 2 + 1;

        /* Move upper half of text 1 pixel to the right -> italic font */
        WinScrollRectangle( &tContext->italic, winRight, 1, &tContext->italic );
        WinEraseRectangle( &tContext->italic, 0 );
    }
}



/* Initialize a new block of underlined text */
static void StartUnderline( void )
{
    if ( Prefs()->underlineMode )
        WinSetUnderlineMode( solidUnderline );
}



/* Handle several lines with underlined text */
static void ContinueUnderline( void )
{
    StartUnderline();
}



/* Mark the end of a block of underlined text */
static void StopUnderline()
{
    if ( Prefs()->underlineMode )
        WinSetUnderlineMode( noUnderline );
}



/* Initialize a new block of strikethrough text */
static void StartStrike
    (
    TextContext* tContext   /* pointer to text context */
    )
{
    tContext->strike.topLeft.x = tContext->cursorX - 1;
}



/* Handle several lines with strikethrough text */
static void ContinueStrike
    (
    TextContext* tContext   /* pointer to text context */
    )
{
    StartStrike( tContext );
}



/* Mark the end of a block of strikethrough text */
static void StopStrike
    (
    TextContext*    tContext,   /* pointer to text context */
    const Int16     height      /* height of line */
    )
{
    if ( tContext->writeMode ) {
        tContext->strike.topLeft.y  = tContext->cursorY - height;
        tContext->strike.extent.x   = tContext->cursorX - 
                                      tContext->strike.topLeft.x - 1;
        tContext->strike.extent.y   = height - 1;

        StrikeThrough( &tContext->strike );
    }
}



/* Draw a horizontal rule */
static void DrawHorizontalRule
    (
    TextContext*    tContext,   /* pointer to text context */
    Int16           height,     /* height of horizontal rule */
    Int16           width       /* width of horizontal rule */
    )
{
    if ( tContext->writeMode ) {
        RectangleType hr;

        hr.topLeft.x    = tContext->cursorX;
        hr.topLeft.y    = tContext->cursorY - height;
        hr.extent.x     = width;
        hr.extent.y     = height;

        WinDrawRectangle( &hr, 0 );
    }
}



/* Initialize paragraph context with data from a paragraph */
static void InitializeParagraphContext
    (
    ParagraphContext*   pContext,   /* pointer to paragraph context */
    Paragraph*          paragraph,  /* pointer to paragraph */
    Header*             record      /* pointer to record */
    )
{
    if ( record->type == DATATYPE_PHTML_COMPRESSED )
        pContext->position = (Char*) MemHandleLock( GetUncompressTextHandle() );
    else
        pContext->position = (Char*) GET_DATA( record );

    pContext->position     += GetParagraphOffset( record, paragraph );
    pContext->last          = pContext->position + paragraph->size;
    pContext->function      = NULL;
    pContext->left          = 0;
    pContext->right         = 0;
    pContext->maxPixels     = ExtentX();
    pContext->linePixels    = 0;
    pContext->type          = ALIGNMENT_LEFT;

    SetStyle( pContext, DEFAULTSTYLE );

    if ( record->type == DATATYPE_PHTML_COMPRESSED )
        MemHandleUnlock( GetUncompressTextHandle() );
}



/* Handle anchor data */
static void HandleAnchor
    (
    ParagraphContext*   pContext,   /* pointer to paragraph context */
    TextContext*        tContext    /* pointer to text context */
    )
{
    ParagraphFunction   function;
    UInt8*              functionArgs;
    Int16               reference;
    Int16               paragraphOffset;
    Int16               imageref;

    function        = (ParagraphFunction) *pContext->function;
    functionArgs    = pContext->function + 1;
    reference       = *functionArgs * 256 + *( functionArgs + 1 );
    paragraphOffset = NO_OFFSET;
    imageref        = NO_IMAGE;

    switch ( function ) {
        case ANCHOR_START:
        case MULTI_IMAGE:
            break;

        case NAMED_ANCHOR_START:
            paragraphOffset = *( functionArgs + 2 ) * 256 + *( functionArgs + 3 );
            break;

        default:
            return;
    }
    StartAnchor( tContext, reference, paragraphOffset, imageref );
}



/* Handle a function */
static void HandleFunction
    (
    ParagraphContext*   pContext,       /* pointer to paragraph context */
    TextContext*        tContext,       /* pointer to text context */
    Int16*              functionWidth   /* upon return, set to the width of a
                                           handled inline image or the size of a
                                           horizontal rule ( will be 0 for other
                                           function codes ) */
    )
{
    ParagraphFunction   function;
    UInt8*              functionArgs;
    Int16               reference;
    Int16               width;
    Int16               widthPercentage;

    function        = (ParagraphFunction) *pContext->function;
    functionArgs    = pContext->function + 1;
    *functionWidth  = 0;

    switch ( function ) {
        case ANCHOR_START:
        case NAMED_ANCHOR_START:
            if ( tContext == NULL )
            break;

            tContext->activeAnchor  = true;
            multilineAnchor         = true;
            HandleAnchor( pContext, tContext );
            break;

        case ANCHOR_END:
            if ( tContext == NULL )
                break;

            tContext->activeAnchor  = false;
            multilineAnchor         = false;
            StopAnchor( tContext, pContext->fontHeight );
            break;

        case SET_STYLE:
            SetStyle( pContext, (Int16) *functionArgs );
            break;

        case MULTI_IMAGE:
            image       = true;
            reference   = *( functionArgs + 2 ) * 256 + *( functionArgs + 3 );

            if ( tContext == NULL ) {
                GetImageMetrics( reference, functionWidth, &pContext->fontHeight );
            }
            else if ( tContext->activeAnchor ) {
                AddImageAnchor( tContext, pContext->fontHeight, 
                    ( *functionArgs ) * 256 + *( functionArgs + 1 ) );
                DrawInlineImage( reference, tContext, functionWidth );
                StopImageAnchor( tContext, pContext->fontHeight, *functionWidth );
            }
            else {
                HandleAnchor( pContext, tContext );
                DrawInlineImage( reference, tContext, functionWidth );
                StopImageAnchor( tContext, pContext->fontHeight, *functionWidth );
                StopAnchor( tContext, pContext->fontHeight );
            }
            break;

        case INLINE_IMAGE:
            image       = true;
            reference   = *functionArgs * 256 + *( functionArgs + 1 );

            if ( tContext == NULL ) {
                GetImageMetrics( reference, functionWidth, &pContext->fontHeight );
            }
            else if ( tContext->activeAnchor ) {
                RestartAnchor( tContext, pContext->fontHeight );
                DrawInlineImage( reference, tContext, functionWidth );
                StopImageAnchor( tContext, pContext->fontHeight, *functionWidth );
            }
            else
                DrawInlineImage( reference, tContext, functionWidth );
            break;

        case SET_MARGIN:
            if ( addMarginToCurrent ) {
                *functionWidth = ( *functionArgs - pContext->left );
                if ( tContext == NULL )
                    *functionWidth += *( functionArgs + 1 ) - pContext->right;
            }
            pContext->left  = *functionArgs;
            pContext->right = *( functionArgs + 1 );
            break;

        case ALIGNMENT:
            if ( tContext == NULL )
                pContext->type = (AlignmentType) *functionArgs;
            break;

        case NEWLINE:
            if ( tContext == NULL )
                newLine = true;
            break;

        case HRULE:
            pContext->fontHeight    = *functionArgs;
            width                   = *( functionArgs + 1 );
            widthPercentage         = *( functionArgs + 2 );

            /* Check values and assign default value if necessary  */
            if ( pContext->fontHeight == 0 )
                pContext->fontHeight = DEFAULT_HRULE_SIZE;

            if ( width != 0 )
                *functionWidth = width;
            else if ( widthPercentage != 0 )
                *functionWidth = pContext->maxPixels * widthPercentage / 100;
            else
                *functionWidth = pContext->maxPixels;

            if ( tContext == NULL ) {
                horizontalRule = true;
                break;
            }
            DrawHorizontalRule( tContext, pContext->fontHeight, *functionWidth );

            /* Restore the current font height */
            pContext->fontHeight = FntLineHeight();

            break;

        case ITALIC_START:
            if ( tContext == NULL )
                break;

            multilineItalic = true;
            StartItalic( tContext );
            break;

        case ITALIC_END:
            if ( tContext == NULL )
                break;

            multilineItalic = false;
            StopItalic( tContext, pContext->fontHeight );
            break;

        case UNDERLINE_START:
            if ( tContext == NULL )
                break;

            multilineUnderline = true;
            StartUnderline();
            break;

        case UNDERLINE_END:
            if ( tContext == NULL )
                break;

            multilineUnderline = false;
            StopUnderline();
            break;

        case STRIKE_START:
            if ( tContext == NULL )
                break;

            multilineStrike = true;
            StartStrike( tContext );
            break;

        case STRIKE_END:
            if ( tContext == NULL )
                break;

            multilineStrike = false;
            StopStrike( tContext, pContext->fontHeight );
            break;
    }
}



/* Get next token from paragraph context */
static TokenType GetNextToken
    (
    ParagraphContext*   pContext,   /* pointer to paragraph context */
    Char*               nextToken   /* token value */
    )
{
    Char    nextChar;
    Int16   offset;

    if ( pContext->last <= pContext->position )
        return TOKEN_PARAGRAPH_END;

    nextChar = *( pContext->position++ );

    if ( nextChar != '\0' ) {
        *nextToken = nextChar;
        return TOKEN_CHARACTER;
    }
    pContext->function  = (UInt8*) pContext->position;
    offset              = 1 + ( *pContext->position & 0x07 );
    pContext->position += offset;
    patternCount       += offset;

    return TOKEN_FUNCTION;
}



/* Return the number of characters that will fit in a line */
static void GetLineMetrics
    (
    ParagraphContext*   pContext,           /* pointer to paragraph context */
    Boolean             skipLeadingSpace,   /* true if initial non-visible 
                                               characters should be skipped */
    Int16*              length,             /* upon return, set to number of 
                                               characters that will fit in the 
                                               line */
    Int16*              height              /* upon return, set to the height 
                                               of the line */
    )
{
    Char*   startPosition;
    Boolean lastSpaceIsVisible;
    FontID  initialFontStyle;
    Int16   initialLeftMargin;
    Int16   initialRightMargin;
    Int16   tokenCount;
    Int16   lastSpace;
    Int16   linePixels;
    Int16   lastSpacePixels;
    Int16   lastSpaceHeight;
    Int16   charWidth;

    startPosition       = pContext->position;
    initialFontStyle    = FntGetFont();
    initialLeftMargin   = pContext->left;
    initialRightMargin  = pContext->right;

    addMarginToCurrent = true;

    *height     = 0;
    tokenCount  = 0;

    spaceCount          = 0;
    lastSpace           = 0;
    linePixels          = 0;
    lastSpacePixels     = 0;
    lastSpaceHeight     = 0;
    lastSpaceIsVisible  = false;

    for ( ;; ) {
        Char        nextToken;
        TokenType   nextTokenType;

        nextTokenType = GetNextToken( pContext, &nextToken );

        if ( nextTokenType == TOKEN_PARAGRAPH_END )
            break;
        else if ( nextTokenType == TOKEN_FUNCTION ) {
            HandleFunction( pContext, NULL, &charWidth );

            tokenCount++;
            if ( newLine ) {
                newLine = false;
                break;
            }
            if ( horizontalRule ) {
                horizontalRule  = false;
                linePixels     += charWidth;
                *height         = pContext->fontHeight;
                break;
            }
            if ( image ) {
                image               = false;
                skipLeadingSpace    = false;
                if ( ( ( linePixels + charWidth ) <= pContext->maxPixels ) || 
                     linePixels == 0 ) {
                    if ( *height < pContext->fontHeight )
                        *height = pContext->fontHeight;
                    lastSpaceIsVisible  = false;
                    lastSpacePixels     = linePixels;
                    lastSpaceHeight     = *height;
                    lastSpace           = tokenCount;
                }
                else {
                    tokenCount--;
                    break;
                }
            }
            linePixels += charWidth;
            if ( pContext->maxPixels < linePixels )
                break;

            continue;
        }
        addMarginToCurrent = false;

        if ( skipLeadingSpace && CharIsSpace( nextToken ) && ! FixedWidthFont() )
            continue;

        skipLeadingSpace = false;

        /* Count the spaces, leading ones are skipped above */
        if ( nextToken == ' ' ) {
            spaceCount++;
            lastSpaceIsVisible  = true;
            lastSpacePixels     = linePixels;
            lastSpaceHeight     = *height;
            lastSpace           = tokenCount;
        }
        charWidth = FntCharWidth( nextToken );

        if ( pContext->maxPixels < ( charWidth + linePixels ) ) {
            if ( ( pContext->maxPixels / 2 ) < lastSpacePixels ) {
                if ( lastSpaceIsVisible )
                    spaceCount--;
                tokenCount  = lastSpace;
                linePixels  = lastSpacePixels;
                *height     = lastSpaceHeight;
            }
            break;
        }
        linePixels += charWidth;

        if ( *height < FntLineHeight() )
            *height = pContext->fontHeight = FntLineHeight();

        tokenCount++;
    }
    if ( pContext->fontHeight != *height )
        pContext->fontHeight = *height;

    pContext->position      = startPosition;
    pContext->linePixels    = linePixels;
    *length                 = tokenCount;
    pContext->maxPixels    += ( initialLeftMargin - pContext->left ) + 
                              ( initialRightMargin - pContext->right );

    /* Restore font style and margins */
    FntSetFont( initialFontStyle );
    pContext->left  = initialLeftMargin;
    pContext->right = initialRightMargin;
}



/* Draw characters from a paragraph onto the display */
static void WriteLine
    (
    ParagraphContext*   pContext,           /* pointer to paragraph context */
    TextContext*        tContext,           /* pointer to text context */
    Int16               characters,         /* number of characters to draw */
    Boolean             skipLeadingSpace    /* whether initial non-visible 
                                               characters should be skipped */
    )
{
    const Int16 findPatternStart  = findPatternPos;
    const Int16 findPatternStop   = findPatternStart + findPatternLen - 1;

    Boolean   invertPattern;
    Char      chars[ characters + 1 ];
    Int16     count;
    Int16     storeChar;

    invertPattern = false;
    count         = 0;
    storeChar     = 0;

    addMarginToCurrent = true;

    AlignText( tContext, pContext );

    while ( count < characters ) {
        TokenType nextTokenType;
        Char      nextChar;
        Int16     imageWidth;

        nextTokenType = GetNextToken( pContext, &nextChar );

        if ( nextTokenType == TOKEN_PARAGRAPH_END )
            break;
        else if ( nextTokenType == TOKEN_FUNCTION ) {
            if ( 0 < storeChar ) {
                if ( tContext->writeMode ) {
                    if ( invertPattern )
                        WinDrawInvertedChars( chars, storeChar, 
                            tContext->cursorX, 
                            tContext->cursorY - FntCharHeight() );
                    else
                        WinDrawChars( chars, storeChar, tContext->cursorX, 
                            tContext->cursorY - FntCharHeight() );
                }
                tContext->cursorX  += FntCharsWidth( chars, storeChar );
                storeChar           = 0;
            }
            HandleFunction( pContext, tContext, &imageWidth );
            tContext->cursorX += imageWidth;

            count++;
            patternCount++;
            if ( image ) {
                image               = false;
                skipLeadingSpace    = false;
            }
            continue;
        }
        addMarginToCurrent = false;

        patternCount++;

        if ( skipLeadingSpace && CharIsSpace( nextChar ) && ! FixedWidthFont() )
            continue;

        skipLeadingSpace = false;

        if ( patternCount == findPatternStart ) {
            if ( 0 < storeChar ) {
                if ( tContext->writeMode ) {
                    WinDrawChars( chars, storeChar, tContext->cursorX, 
                        tContext->cursorY - FntCharHeight() );
                }
                tContext->cursorX  += FntCharsWidth( chars, storeChar );
                storeChar           = 0;
            }
            invertPattern           = true;
            tContext->patternOffset = tContext->cursorY - FntCharHeight() - 
                                      TopLeftY();
        }

        chars[ storeChar++ ] = nextChar;
        count++;

        if ( patternCount == findPatternStop ) {
            if ( tContext->writeMode ) {
                WinDrawInvertedChars( chars, storeChar, tContext->cursorX, 
                    tContext->cursorY - FntCharHeight() );
            }
            invertPattern       = false;
            tContext->cursorX  += FntCharsWidth( chars, storeChar );
            storeChar           = 0;
        }

        if ( pContext->type == ALIGNMENT_JUSTIFY && nextChar == ' ' ) {
            if ( 0 < storeChar ) {
                if ( tContext->writeMode ) {
                    WinDrawChars( chars, storeChar, tContext->cursorX, 
                        tContext->cursorY - FntCharHeight() );
                }
                tContext->cursorX  += FntCharsWidth( chars, storeChar );
                storeChar           = 0;
            }
            if ( bigSpace ) {
                bigSpace--;
                tContext->cursorX++;
            }
            tContext->cursorX += littleSpace;
        }
    }
    if ( 0 < storeChar ) {
        if ( tContext->writeMode ) {
            if ( invertPattern )
                WinDrawInvertedChars( chars, storeChar, tContext->cursorX, 
                    tContext->cursorY - FntCharHeight() );
            else
                WinDrawChars( chars, storeChar, tContext->cursorX, 
                    tContext->cursorY - FntCharHeight() );
        }
        tContext->cursorX += FntCharsWidth( chars, storeChar );
    }
}



/* Draw a paragraph using the given text context */
void DrawParagraph
    (
    TextContext*    tContext,   /* pointer to text context */
    Paragraph*      paragraph,  /* pointer to paragraph */
    Header*         record      /* pointer to record */
    )
{
    ParagraphContext    pContext;
    Int16               length;
    Int16               height;
    Int16               storeCount;

    InitializeParagraphContext( &pContext, paragraph, record );
    tContext->cursorX = TopLeftX();

    multilineAnchor     = false;
    multilineItalic     = false;
    multilineUnderline  = false;
    multilineStrike     = false;
    newLine             = false;
    horizontalRule      = false;
    image               = false;
    patternCount        = GetParagraphOffset( record, paragraph ) - 1;

    /* Check for extra paragraph spacing */
    tContext->cursorY += ( paragraph->attributes & 0x07 ) * DEFAULT_PARAGRAPH_SPACING;

    if ( record->type == DATATYPE_PHTML_COMPRESSED )
        MemHandleLock( GetUncompressTextHandle() );

    for ( ;; ) {
        storeCount = patternCount;
        GetLineMetrics( &pContext, true, &length, &height );
        patternCount = storeCount;

        if ( length <= 0 )
            break;

        tContext->cursorY += height;

        /* Continue any anchor started on the line before */
        if ( multilineAnchor ) {
            Int16 tempX;

            tempX = tContext->cursorX;

            AlignText( tContext, &pContext );
            ContinueAnchor( tContext );

            tContext->cursorX = tempX;
        }
        /* Continue any italic text started on the line before */
        if ( multilineItalic )
            ContinueItalic( tContext );

        /* Continue any underlined text started on the line before */
        if ( multilineUnderline )
            ContinueUnderline();

        /* Continue any strikethrough text started on the line before */
        if ( multilineStrike )
            ContinueStrike( tContext );

        /* Draw the characters and move the cursor...  */
        WriteLine( &pContext, tContext, length, true );

        /* End any anchor which was going on the last line; continue it on the
           next line. */
        if ( multilineAnchor )
            StopAnchor( tContext, height );

        /* End any italic text which was going on the last line; continue it on the
           next line. */
        if ( multilineItalic )
            StopItalic( tContext, height );

        /* End any underlined text which was going on the last line; continue it on the
           next line. */
        if ( multilineUnderline )
            StopUnderline();

        /* End any strikethrough text which was going on the last line; continue it on the
           next line. */
        if ( multilineStrike )
            StopStrike( tContext, height );

        tContext->cursorX   = TopLeftX() + pContext.left;
        pContext.fontHeight = FntLineHeight();
    }
    if ( record->type == DATATYPE_PHTML_COMPRESSED )
        MemHandleUnlock( GetUncompressTextHandle() );
}
