/*
    Bear Engine - Model editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/model_view_ctrl.cpp
 * \brief Implementation of the bf::model_view_ctrl class.
 * \author Julien Jorge
 */
#include "bf/model_view_ctrl.hpp"

#include "bf/model_view.hpp"

#include <wx/sizer.h>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param parent The parent window.
 * \param m The model to display.
 */
bf::model_view_ctrl::model_view_ctrl
( wxWindow& parent, wxWindowID id, gui_model& m )
  : wxPanel(&parent, id)
{
  create_controls(m);
  create_sizers();

  adjust_scrollbars();
} // model_view_ctrl::model_view_ctrl()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell the view to reload the action.
 */
void bf::model_view_ctrl::reload_action()
{
  m_model_view->reload_action();
} // model_view_ctrl::reload_action()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the action to display.
 * \param d The date at which the action is displayed.
 */
void bf::model_view_ctrl::set_date( double d )
{
  m_model_view->set_date(d);
} // model_view_ctrl::set_date()

/*----------------------------------------------------------------------------*/
/**
 * \brief Go to the next frame or snapshot.
 * \brief Return the new date.
 */
double bf::model_view_ctrl::next()
{
  return m_model_view->next();
} // model_view_ctrl::next()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if we are at the end of the action.
 */
bool bf::model_view_ctrl::is_finished() const
{
  return m_model_view->is_finished();
} // model_view_ctrl::is_finished()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the duration until the next change.
 */
double bf::model_view_ctrl::get_duration_until_next() const
{
  return m_model_view->get_duration_until_next();
} // model_view_ctrl::get_duration_until_next()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the zoom ratio.
 * \param z The zoom ratio.
 */
void bf::model_view_ctrl::set_zoom( unsigned int z )
{
  m_model_view->set_zoom(z);
} // model_view_ctrl::set_zoom()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the zoom ratio.
 */
unsigned int bf::model_view_ctrl::get_zoom() const
{
  return m_model_view->get_zoom();
} // model_view_ctrl::get_zoom()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compute the zoom ratio such that the image fits the control.
 */
void bf::model_view_ctrl::auto_zoom()
{
  m_model_view->auto_zoom();
} // model_view_ctrl::auto_zoom()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the controls.
 * \param m The model to display.
 */
void bf::model_view_ctrl::create_controls( gui_model& m )
{
  m_model_view = new model_view(*this, ID_MODEL_VIEW, m);

  m_h_scrollbar = new wxScrollBar(this, wxID_ANY);
  m_v_scrollbar = new wxScrollBar
    ( this, wxID_ANY, wxDefaultPosition, wxDefaultSize, wxSB_VERTICAL );
} // model_view_ctrl::create_controls()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sizers.
 */
void bf::model_view_ctrl::create_sizers()
{
  wxBoxSizer* h_sizer = new wxBoxSizer( wxHORIZONTAL );
  wxBoxSizer* v_sizer = new wxBoxSizer( wxVERTICAL );

  v_sizer->Add(m_v_scrollbar, 1, wxEXPAND);

  h_sizer->Add(m_model_view, 1, wxEXPAND);
  h_sizer->Add(v_sizer, 0, wxEXPAND);

  v_sizer = new wxBoxSizer(wxVERTICAL);
  v_sizer->Add(h_sizer, 1, wxEXPAND);

  h_sizer = new wxBoxSizer(wxHORIZONTAL);
  h_sizer->Add(m_h_scrollbar, 1, wxALIGN_BOTTOM);
  h_sizer->Add
    ( new wxWindow
      ( this, wxID_ANY, wxDefaultPosition,
        wxSize( m_v_scrollbar->GetSize().x, m_h_scrollbar->GetSize().y ) ),
      0
      );

  v_sizer->Add(h_sizer, 0, wxEXPAND);

  SetSizer( v_sizer );
} // model_view_ctrl::create_sizers()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the bounds and the position of the toolbars.
 */
void bf::model_view_ctrl::adjust_scrollbars()
{
  m_h_scrollbar->SetScrollbar
    ( m_model_view->get_view_delta().x, m_model_view->GetSize().x,
      m_model_view->get_view_size().x,
      (3 * m_model_view->GetSize().x) / 4, true );
  m_v_scrollbar->SetScrollbar
    ( m_model_view->get_view_delta().y, m_model_view->GetSize().y,
      m_model_view->get_view_size().y,
      (3 * m_model_view->GetSize().y) / 4, true );

  m_model_view->set_view_delta
    ( m_h_scrollbar->GetThumbPosition(), m_v_scrollbar->GetThumbPosition() );
} // model_view_ctrl::adjust_scrollbars()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the position of the view.
 * \param event The scroll event.
 */
void bf::model_view_ctrl::on_scroll(wxScrollEvent& event)
{
  m_model_view->set_view_delta
    ( m_h_scrollbar->GetThumbPosition(), m_v_scrollbar->GetThumbPosition() );
} // model_view_ctrl::on_scroll()

/*----------------------------------------------------------------------------*/
/**
 * \brief Event sent to a resized window.
 * \param event The event.
 */
void bf::model_view_ctrl::on_size(wxSizeEvent& event)
{
  adjust_scrollbars();
  event.Skip();
} // model_view_ctrl::on_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief The selected mark is modified.
 * \param event This event occured.
 */
void bf::model_view_ctrl::on_mark_change(mark_event& event)
{
  mark_event e
     ( event.get_value(), mark_event::selection_change_event_type, GetId() );
  e.SetEventObject(this);
  ProcessEvent(e);
} // model_view_ctrl::on_mark_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The selected mark move.
 * \param event This event occured.
 */
void bf::model_view_ctrl::on_placement_change(placement_event& event)
{
  placement_event e
     ( event.get_label(), event.get_left(), event.get_bottom(), 
       placement_event::move_event_type, GetId() );
  e.SetEventObject(this);
  ProcessEvent(e);
} // model_view_ctrl::on_placement_change()

/*----------------------------------------------------------------------------*/
/**
 * \brief The bounding box change.
 * \param event This event occured.
 */
void bf::model_view_ctrl::on_bounding_box_change(bounding_box_event& event)
{
  bounding_box_event e
     ( event.get_left(), event.get_bottom(), 
       event.get_width(), event.get_height(), 
       event.get_mark_move_x(), event.get_mark_move_y(),
       bounding_box_event::change_event_type, GetId() );
  e.SetEventObject(this);
  ProcessEvent(e);
} // model_view_ctrl::on_bounding_box_change()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::model_view_ctrl, wxPanel)
  EVT_SCROLL( bf::model_view_ctrl::on_scroll )
  EVT_SIZE( bf::model_view_ctrl::on_size )
  EVT_MARK_SELECTION_CHANGE( bf::model_view_ctrl::ID_MODEL_VIEW,
			     bf::model_view_ctrl::on_mark_change )
  EVT_PLACEMENT_MOVE( bf::model_view_ctrl::ID_MODEL_VIEW,
		      bf::model_view_ctrl::on_placement_change )
  EVT_BOUNDING_BOX_CHANGE( bf::model_view_ctrl::ID_MODEL_VIEW,
			   bf::model_view_ctrl::on_bounding_box_change )
END_EVENT_TABLE()
