/*
    Bear Engine - Level editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/layer.hpp
 * \brief A layer from our level.
 * \author Julien Jorge
 */
#ifndef __BF_LAYER_HPP__
#define __BF_LAYER_HPP__

#include "bf/item_instance.hpp"

#include <claw/functional.hpp>
#include <claw/iterator.hpp>

#include <set>
#include <list>
#include <map>

namespace bf
{
  /**
   * \brief A layer from our level.
   * \author Julien Jorge
   */
  class layer
  {
  public:
    typedef std::set<item_instance*> item_set_type;

    typedef claw::wrapped_iterator
    < const item_instance,
      item_set_type::const_iterator,
      claw::const_dereference<item_instance>
    >::iterator_type const_item_iterator;

    typedef claw::wrapped_iterator
    < item_instance,
      item_set_type::iterator,
      claw::dereference<item_instance>
    >::iterator_type item_iterator;

  public:
    layer
    ( unsigned int width, unsigned int height, const std::string& layer_type );
    layer( const layer& that );
    ~layer();

    void resize( unsigned int width, unsigned int height );
    void set_class_name( const std::string& class_name );

    void add_item( item_instance* item );
    void remove_item( item_instance* item );

    const std::string& get_class_name() const;
    unsigned int get_width() const;
    unsigned int get_height() const;
    std::size_t get_items_count() const;

    bool empty() const;

    std::list<item_instance*> check
    ( std::string& error_msg,
      const std::map<std::string,item_instance*>& map_id);
    void compile( compiled_file& f ) const;

    item_iterator item_begin();
    item_iterator item_end();

    const_item_iterator item_begin() const;
    const_item_iterator item_end() const;

    layer& operator=( const layer& that );

  private:
    void clear();
    void assign( const layer& that );
    void sort_and_identify
    ( std::list<item_instance*>& referenced,
      std::map<std::string, unsigned int>& id_to_int ) const;
    void sort_by_dependency
    ( item_instance* p, std::map<std::string, item_instance*>& items,
      std::list<item_instance*>& referenced ) const;

  private:
    /** \brief The width of the layer. */
    unsigned int m_width;

    /** \brief The height of the layer. */
    unsigned int m_height;

    /** \brief The type of the layer. */
    std::string m_layer_type;

    /** \brief The items in this layer. */
    item_set_type m_item;

  }; // class layer
} // namespace bf

#endif // __BF_LAYER_HPP__
