/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file expression_creator.hpp
 * \brief Template class to create items that can be use as a binary expression.
 * \author Julien Jorge
 */
#ifndef __BEAR_EXPRESSION_CREATOR_HPP__
#define __BEAR_EXPRESSION_CREATOR_HPP__

#include "engine/base_item.hpp"
#include "engine/expr/binary_boolean_expression.hpp"

#include "engine/export.hpp"

namespace bear
{
  /**
   * \brief Template class to create items that can be use as a binary
   *        expression.
   *
   * This item is a parameter for other items. He kills himself at the end of
   * his build() method.
   *
   * The fields of this item are
   *  - \a left_operand: (item) \c the left operand in the expression. Must
   *    inherit from Expression::super,
   *  - \a right_operand: (item) \c the right operand in the expression. Must
   *    inherit from Expression::super.
   *
   * \author Julien Jorge
   */
  template<typename Expression>
  class binary_expression_creator:
    public engine::base_item,
    public Expression
  {
  public:
    /** \brief The type of the parent class. */
    typedef engine::base_item super;

  public:
    bool set_item_field( const std::string& name, engine::base_item* value );

    void build();

  }; // class binary_expression_creator

  /**
   * \brief This class is a boolean expression of type "boolean_equality".
   * \author Julien Jorge
   */
  class boolean_equality_creator:
    public binary_expression_creator<engine::boolean_equality>
  {
    DECLARE_BASE_ITEM(boolean_equality_creator, bear);
  }; // class boolean_equality_creator

  /**
   * \brief This class is a boolean expression of type "boolean_disequality".
   * \author Julien Jorge
   */
  class boolean_disequality_creator:
    public binary_expression_creator<engine::boolean_disequality>
  {
    DECLARE_BASE_ITEM(boolean_disequality_creator, bear);
  }; // class boolean_disequality_creator

  /**
   * \brief This class is a boolean expression of type "logical_and".
   * \author Julien Jorge
   */
  class logical_and_creator:
    public binary_expression_creator<engine::logical_and>
  {
    DECLARE_BASE_ITEM(logical_and_creator, bear);
  }; // class logical_and_creator

  /**
   * \brief This class is a boolean expression of type "logical_or".
   * \author Julien Jorge
   */
  class logical_or_creator:
    public binary_expression_creator<engine::logical_or>
  {
    DECLARE_BASE_ITEM(logical_or_creator, bear);
  }; // class logical_or_creator

} // namespace bear

#include "generic_items/expr/impl/expression_creator.tpp"

#endif // __BEAR_EXPRESSION_CREATOR_HPP__
