/***************************************************************************
 *            qof-datebook.c
 *
 *  Thu Oct 21 15:38:39 2004
 *  Copyright  2004-2005  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
/** @addtogroup DB
	@{ */
/** @file  qof-datebook.c
	@brief QOF datebook definitions for pilot-link
	@author Copyright (c) 2004,2006 Neil Williams <linux@codehelp.co.uk>
*/

#define _GNU_SOURCE
#include "config.h"
#include <glib.h>
#include <glib/gprintf.h>
#include <qof.h>
#include <stdlib.h>
#include "pi-datebook.h"
#include "qof-main.h"
#include "pilot-qof.h"
#include "qof-datebook.h"

static QofLogModule log_module = PQ_MOD_PILOT;

#define DATEBOOK_VERSION datebook_v1
#define DATEBOOK_EVENT        "untimed_event"
#define DATEBOOK_BEGIN        "start_time"
#define DATEBOOK_END          "end_time"
#define DATEBOOK_ALARM        "use_alarm"
#define DATEBOOK_ADVANCE      "alarm_advance"
#define DATEBOOK_ADV_UNIT     "advance_unit"
#define DATEBOOK_REPEAT_TYPE  "repeat_type"
/** \brief Does the repeat have an end?

If set, any value in repeat-end is ignored.
*/
#define DATEBOOK_REPEAT_FOREVER    "repeat_forever"
#define DATEBOOK_REPEAT_END        "repeat_end"
#define DATEBOOK_REPEAT_FREQUENCY  "repeat_frequency"
#define DATEBOOK_REPEAT_DAY        "repeat_day"
#define DATEBOOK_REPEAT_WEEK_START "repeat_week_start"
#define DATEBOOK_EXCEPTIONS        "exception_count"
#define DATEBOOK_EXCEPTION         "exception_list"
#define DATEBOOK_NOTE              "note"
#define DATEBOOK_CATEGORY          "category"
#define DATEBOOK_KVP_PATH          "datebook/exceptions"

/** Wraps QOF around the libpisock object. */
typedef struct
{
	QofInstance inst;
	Appointment_t wrap;			//0.12
	gchar *category;
	gboolean repeater;	 /**< \brief Repeat appointment marker
	
	Set to TRUE if this is a transient instance created
	from the repeat data of another instance. This is done to
	make it easier to query repeat appointments. Sufficient
	transient repeaters are created to fill the time between
	start_time and repeat_end, up to a maximum of a year and a
	day - exceptions in the original appointment are not created.
	Unlike qof_instance_gemini, the repeater is created in the
	SAME book as the original. Repeater entities are ignored when
	packing a book into a Palm database. */
} QofDateBook;

/** \brief see QOF or alioth for information on this macro.

To document enums generated using the macro, look at the
doxygen.cfg file in this package, under "PREPROCESSOR".

Note the use of the NON_TYPEDEF versions of the macro here. enum values
cannot be defined twice so if the original definition in libpisock is
NOT a typedef, use the NON_TYPEDEF version of the macro.

http://qof.sourceforge.net/doxy/group__Utilities.html

http://alioth.debian.org/snippet/detail.php?type=snippet&id=13
*/

#define ENUM_REPEAT_LIST(_) \
	_(repeatNone,) \
	_(repeatDaily,) \
	_(repeatWeekly,) \
	_(repeatMonthlyByDay,) \
	_(repeatMonthlyByDate,) \
	_(repeatYearly,)

AS_STRING_DEC_NON_TYPEDEF (repeatTypes, ENUM_REPEAT_LIST)
FROM_STRING_DEC_NON_TYPEDEF (repeatTypes, ENUM_REPEAT_LIST)
/** \brief Units to measure the digit entered in alarm_advance. */
#define ENUM_ALARM_UNIT(_) \
	_(advMinutes,) _(advHours,) _(advDays,)
	AS_STRING_DEC_NON_TYPEDEF (alarmTypes, ENUM_ALARM_UNIT)
FROM_STRING_DEC_NON_TYPEDEF (alarmTypes, ENUM_ALARM_UNIT)

/** Possible settings for repeats on certain days of each month.
dom == day of month in each case. */
#define ENUM_MONTH_DAYQ(_) \
	_(dom1stSun,) _(dom1stMon,) _(dom1stTue,) _(dom1stWen,) _(dom1stThu,) \
	_(dom1stFri,) _(dom1stSat,) _(dom2ndSun,) _(dom2ndMon,) _(dom2ndTue,) \
	_(dom2ndWen,) _(dom2ndThu,) _(dom2ndFri,) _(dom2ndSat,) _(dom3rdSun,) \
	_(dom3rdMon,) _(dom3rdTue,) _(dom3rdWen,) _(dom3rdThu,) _(dom3rdFri,) \
	_(dom3rdSat,) _(dom4thSun,) _(dom4thMon,) _(dom4thTue,) _(dom4thWen,) \
	_(dom4thThu,) _(dom4thFri,) _(dom4thSat,) \
	_(domLastSun,) _(domLastMon,) _(domLastTue,) _(domLastWen,) \
	_(domLastThu,) _(domLastFri,) _(domLastSat,)
AS_STRING_DEC_NON_TYPEDEF (DayOfMonthType, ENUM_MONTH_DAYQ)
FROM_STRING_DEC_NON_TYPEDEF (DayOfMonthType, ENUM_MONTH_DAYQ)
AS_STRING_FUNC_NON_TYPEDEF (repeatTypes, ENUM_REPEAT_LIST)
FROM_STRING_FUNC_NON_TYPEDEF (repeatTypes, ENUM_REPEAT_LIST)
AS_STRING_FUNC_NON_TYPEDEF (DayOfMonthType, ENUM_MONTH_DAYQ)
FROM_STRING_FUNC_NON_TYPEDEF (DayOfMonthType, ENUM_MONTH_DAYQ)
AS_STRING_FUNC_NON_TYPEDEF (alarmTypes, ENUM_ALARM_UNIT)
FROM_STRING_FUNC_NON_TYPEDEF (alarmTypes, ENUM_ALARM_UNIT)
/** \brief Create a datebook object in QOF

Sets a default time for start and end as the current time.
Creates a KVP frame ready for any exceptions which is
an array of struct tm. The length of array = datebook_getExceptions()
*/
static QofDateBook *
datebook_create (QofBook * book)
{
	Appointment_t *qd;
	QofDateBook *obj;
	QofCollection *coll;
	GList *all;

	obj = g_new0 (QofDateBook, 1);
	qof_instance_init (&obj->inst, PILOT_LINK_QOF_DATEBOOK, book);
	coll = qof_book_get_collection (book, PILOT_LINK_QOF_DATEBOOK);
	all = qof_collection_get_data (coll);
	all = g_list_prepend (all, obj);
	qof_collection_set_data (coll, all);
	qd = &obj->wrap;
	obj->repeater = FALSE;
	/** let QofTime remain NULL */
	qd->exception = NULL;
	return obj;
}

static gboolean
datebook_getEvent (QofDateBook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, -1);
	qd = &d->wrap;
	return (qd->event) ? TRUE : FALSE;
}

static QofTime *
datebook_getBegin (QofDateBook * d)
{
	Appointment_t *qd;
	QofTime *qt;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	qt = qof_time_from_tm (&qd->begin, 0);
	return qt;
}

static QofTime *
datebook_getEnd (QofDateBook * d)
{
	Appointment_t *qd;
	QofTime *qt;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	qt = qof_time_from_tm (&qd->end, 0);
	return qt;
}

static gboolean
datebook_getAlarm (QofDateBook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, FALSE);
	qd = &d->wrap;
	return (qd->alarm == 1) ? TRUE : FALSE;
}

static gint
datebook_getAdvance (QofDateBook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, -1);
	qd = &d->wrap;
	return (datebook_getAlarm (d)) ? qd->advance : 0;
}

/** \brief Uses ENUM_ALARM_UNIT to convert to a string

The enumerator values are converted to text for XML.
*/
static const gchar *
datebook_getAdvanceUnit (QofDateBook * d)
{
	Appointment_t *qd;
	gchar *unit;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	if (datebook_getAlarm (d) == FALSE)
		return NULL;
	unit = g_strdup (alarmTypesasString (qd->advanceUnits));
	return unit;
}

static const gchar *
datebook_getRepeatType (QofDateBook * d)
{
	Appointment_t *qd;
	gchar *type_label;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	type_label = g_strdup (repeatTypesasString (qd->repeatType));
	return type_label;
}

static gboolean
datebook_getRepeatForever (QofDateBook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, -1);
	qd = &d->wrap;
	return (qd->repeatForever == 1) ? TRUE : FALSE;
}

static QofTime *
datebook_getRepeatEnd (QofDateBook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	if (qd->repeatType == repeatNone)
		qd->repeatEnd = qd->end;
	return qof_time_from_tm (&qd->repeatEnd, 0);
}

/** \brief How often to repeat

Actual frequency depends on the value of repeatType.
1 == every time that repeat type comes around.
2 == every other repeat.
*/
static gint
datebook_getRepeatFrequency (QofDateBook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, -1);
	qd = &d->wrap;
	return qd->repeatFrequency;
}

static const gchar *
datebook_getRepeatDay (QofDateBook * d)
{
	Appointment_t *qd;
	enum DayOfMonthType type;
	gchar *g;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	if (qd->repeatType == repeatNone)
		return NULL;
	type = qd->repeatDay;
	g = g_strdup (DayOfMonthTypeasString (type));
	return g;
}

/** \brief Get the locale name of the day to start the week.

Calculate the current day of the week, compare with the integer
set in the preferences and deduce the full weekday name
according to the current locale. 

\note \b Current. QOF deals with the timezone issues but if
	you take your Palm to another country, be prepared for this
	value to be translated!

This value is not used to set the preference in the Palm, so
it does not need to be translated back. Setting the value in
\ref DB only affects any exported XML.

\verbatim
repeatWeekstart contains 0=sunday, 1=monday etc.
QofDate->qd_wday also starts at 0=sunday.
Say it's Tues 26th and weekstart is sunday
qd_wday = 2, qd_mday = 26, repeatWeekstart = 0;

	if (qdate->qd_mday < qdate->qd_wday)
		qdate->qd_mday += 7;
	qdate->qd_mday -= qdate->qd_wday + repeatWeekstart;

	qd_mday = 26 - 2 + 0 = 24. Sunday 24th.

in case it's near the start of the month, go to the next week. 
If it's Thursday 1st and weekstart is Monday:
qd_wday = 4, qd_mday = 1.
qd_mday = qd_mday + 7 = 8. (Thurs 8th)
qd_mday = 8 - 4 + 1 = 5. (Mon 5th.)

\endverbatim
*/
static QofDateFormat df = 0;

static gchar *
datebook_getRepeatWeekStart (QofDateBook * d)
{
	Appointment_t *qd;
	QofTime *qt;
	QofDate *qdate;
	gboolean result;
	gchar *retval;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	if (qd->repeatType == repeatNone)
		return NULL;
	qt = qof_time_get_current ();
	if (df == 0)
	{
		result = qof_date_format_add ("%A", &df);
		if (!result)
		{
			PERR (" failed to fetch the locale format");
			return NULL;
		}
	}
	qdate = qof_date_from_qtime (qt);
	if (qdate->qd_mday < qdate->qd_wday)
		qdate->qd_mday += 7;
	qdate->qd_mday -= (qdate->qd_wday + qd->repeatWeekstart);
	qof_date_valid (qdate);
	qof_time_free (qt);
	retval = qof_date_print (qdate, df);
	qof_date_free (qdate);
	return retval;
}

static gint
datebook_getExceptions (QofDateBook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, -1);
	qd = &d->wrap;
	return qd->exceptions;
}

static gchar *
datebook_getDescription (QofDateBook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	return qd->description;
}

static gchar *
datebook_getNote (QofDateBook * d)
{
	Appointment_t *qd;

	g_return_val_if_fail (d != NULL, NULL);
	qd = &d->wrap;
	return qd->note;
}

static gchar *
datebook_getCategory (QofDateBook * d)
{
	g_return_val_if_fail (d != NULL, NULL);
	return d->category;
}

static double
datebook_getDuration (QofDateBook * d)
{
	QofTime *qt_diff, *qt_start, *qt_end;
	gdouble retval;

	g_return_val_if_fail (d, 0);
	qt_start = datebook_getBegin (d);
	qt_end = datebook_getEnd (d);
	qt_diff = qof_time_diff (qt_end, qt_start);
	retval = qof_time_get_secs (qt_diff) / 3600;
	qof_time_free (qt_diff);
	qof_time_free (qt_start);
	qof_time_free (qt_end);
	return retval;
}

static gboolean
datebook_check_repeater (QofDateBook * d)
{
	g_return_val_if_fail (d, FALSE);
	return d->repeater;
}

static void
datebook_set_repeater (QofDateBook * d, gboolean e)
{
	g_return_if_fail (d != NULL);
	d->repeater = e;
}

static void
datebook_setEvent (QofDateBook * d, gboolean e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	qd->event = 0;
	if (e == TRUE)
		qd->event = 1;
}

static void
datebook_setBegin (QofDateBook * d, QofTime *qt)
{
	Appointment_t *qd;
	gboolean result;
	QofDate *qdate;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	qdate = qof_date_from_qtime (qt);
	result = qof_date_to_struct_tm (qdate, &qd->begin, 0);
	if(!result)
		PERR (" Date too large for begin.");
	qof_date_free (qdate);
}

static void
datebook_setEnd (QofDateBook * d, QofTime *qt)
{
	Appointment_t *qd;
	gboolean result;
	QofDate *qdate;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	qdate = qof_date_from_qtime (qt);
	result = qof_date_to_struct_tm (qdate, &qd->end, 0);
	if(!result)
		PERR (" Date too large for end.");
	qof_date_free (qdate);
}

static void
datebook_setAlarm (QofDateBook * d, gboolean e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	if (e)
		qd->alarm = 1;
}

static void
datebook_setAdvance (QofDateBook * d, gint e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	qd->advance = e;
}

static void
datebook_setAdvanceUnit (QofDateBook * d, const gchar * e)
{
	Appointment_t *qd;
	enum alarmTypes type;

	g_return_if_fail (d != NULL);
	if (e == NULL)
		return;
	qd = &d->wrap;
	alarmTypesfromString (e, &type);
	qd->advanceUnits = type;
}

/** \brief Uses ENUM_REPEAT_LIST to convert to an enum value.

The XML string is converted back to an enumerator value.
*/
static void
datebook_setRepeatType (QofDateBook * d, const gchar * type_label)
{
	Appointment_t *qd;
	enum repeatTypes type;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	repeatTypesfromString (type_label, &type);
	qd->repeatType = type;
}

static void
datebook_setRepeatForever (QofDateBook * d, gboolean e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	if (e)
		qd->repeatForever = 1;
}

static void
datebook_setRepeatEnd (QofDateBook * d, QofTime *qt)
{
	Appointment_t *qd;
	gboolean result;
	QofDate *qdate;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	qdate = qof_date_from_qtime (qt);
	result = qof_date_to_struct_tm (qdate, &qd->repeatEnd, 0);
	if(!result)
		PERR (" Date too large for repeatEnd.");
	qof_date_free (qdate);
}

static void
datebook_setRepeatFrequency (QofDateBook * d, gint e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	qd->repeatFrequency = e;
}

static void
datebook_setRepeatDay (QofDateBook * d, const gchar * e)
{
	Appointment_t *qd;
	enum DayOfMonthType type;

	g_return_if_fail (d != NULL);
	if (e == NULL)
		return;
	qd = &d->wrap;
	DayOfMonthTypefromString (e, &type);
	qd->repeatDay = type;
}

/** \brief Set the day to start the week from the locale name.

If the value is not recognised, QOF sets a default of zero -
which would start the week on Sunday.

However, this value is not synced to the Palm - it is XML only.
*/
static void
datebook_setRepeatWeekStart (QofDateBook * d, gchar * e)
{
	gchar day[MAX_DATE_LENGTH];
	Appointment_t *qd;
	time_t local;
	struct tm *local_tm;
	gint i, diff;
	gboolean found;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	diff = 0;
	found = FALSE;
	local = time (NULL);
	local_tm = localtime (&local);
	if (local_tm->tm_wday <= 7)
		diff = 1;
	else
		diff = -1;
	for (i = 0; i < 7; i++)
	{
		strftime (day, MAX_DATE_LENGTH, "%A", local_tm);
		if (0 == safe_strcmp (e, day))
		{
			found = TRUE;
			break;
		}
		local_tm->tm_mday += diff;
	}
	if (!found)
		i = 0;
	qd->repeatWeekstart = i;
}

static void
datebook_setExceptions (QofDateBook * d, gint e)
{
	Appointment_t *qd;

	g_return_if_fail (d != NULL);
	qd = &d->wrap;
	qd->exceptions = e;
}

static void
datebook_setDescription (QofDateBook * d, gchar * h)
{
	Appointment_t *qd;

	g_return_if_fail (d);
	qd = &d->wrap;
	qd->description = g_strdup (qof_main_make_utf8 (h));
}

static void
datebook_setNote (QofDateBook * d, gchar * h)
{
	Appointment_t *qd;

	g_return_if_fail (d);
	qd = &d->wrap;
	if (!h)
		return;
	qd->note = g_strdup (qof_main_make_utf8 (h));
}

static void
datebook_setCategory (QofDateBook * d, gchar * n)
{
	g_return_if_fail (d != NULL);
	d->category = g_strdup (qof_main_make_utf8 (n));
}

/** unpack the Palm datebook record into the entity. 

\todo Problems with queries not finding repeat events.
*/

static gint
datebook_unpack (QofEntity * ent, gpointer user_data)
{
	pi_buffer_t *pi_buf;
	Appointment_t *qa;
	QofDateBook *obj, *clone;
	QofInstance *inst;
	QofBook *book;
	KvpFrame *inst_frame;
	QofTime *qt, *qt_increment, *qt_end, *qt_repeat;
	PQContext *context;
	gint size, i, day_interval, month_interval;

	context = (PQContext *) user_data;
	day_interval = 0;
	month_interval = 0;
	g_return_val_if_fail (context != NULL, -1);
	g_return_val_if_fail (ent != NULL, -1);
	obj = (QofDateBook *) ent;
	inst = (QofInstance *) ent;
	inst_frame = qof_instance_get_slots (inst);
	qa = &obj->wrap;
	pi_buf = (pi_buffer_t *) context->pi_buf;
	size = 0;
	size = unpack_Appointment (qa, pi_buf, DATEBOOK_VERSION);	// 0.12
	datebook_setCategory (obj, context->names[context->ent_category]);
	/* Use <= and handle zero if < omits last entry (&623) */
	for (i = 0; i < qa->exceptions; i++)
	{
		gchar *extend;

		extend = NULL;
		DEBUG (" exceptions=%d", qa->exceptions);
		qt = qof_time_from_tm (&qa->exception[i], 0);
		extend = g_strdup_printf ("%s/%d", DATEBOOK_KVP_PATH, i + 1);
		kvp_frame_set_time (inst_frame, extend, qt);
		inst->kvp_data = inst_frame;
		g_free (extend);
	}
	if (qa->repeatType == repeatNone)
		qa->repeatEnd = qa->end;
	/* set a marker for the interval. Do the iteration once, outside the switch. */
	switch (qa->repeatType)
	{
	case repeatNone:
		{
			day_interval = 0;
			month_interval = 0;
			break;
		}
	case repeatDaily:
		{
			day_interval = 1;
			month_interval = 0;
			break;
		}
	case repeatWeekly:
		{
			day_interval = 7;
			month_interval = 0;
			break;
		}
	case repeatMonthlyByDay:
	case repeatMonthlyByDate:
		{
			day_interval = 0;
			month_interval = 1;
			break;
		}
	case repeatYearly:
		{
			day_interval = 0;
			month_interval = 12;
			break;
		}
	default:
		{
			PERR (" unsupported repeatType=%d", qa->repeatType);
		}
	}
	if (day_interval == 0 && month_interval == 0)
		return size;
	/* Now create a repeater in the SAME book. */
	qt = datebook_getBegin (obj);
	qt_end = datebook_getEnd (obj);
	if (qa->repeatForever == 0)
	{
		qt_repeat = datebook_getRepeatEnd (obj);
	}
	else
	{
		QofDate *qd;
		/*  if qa->repeatForever == 1 (true), calculate year and a day from
		   start_time. qof_date_add_months(12)
		   qof_date_add_days(1). Store for use as repeatEnd
		 */
		DEBUG (" qa->repeatForever == 1");
		qt_repeat = qt;
		qd = qof_date_from_qtime (qt_repeat);
		qof_date_addmonths (qd, 12, FALSE);
		qof_date_adddays (qd, 1);
		qof_date_free (qd);
	}
	qt_increment = qt;
	/*  qa->exception is an array of struct tm* qa->exceptions long. */
	/* while datebook_getBegin is less (earlier) than repeat_end */
	while (qof_time_cmp (qt_increment, qt_repeat) < 0)
	{
		gboolean skip;

		skip = FALSE;
		if (day_interval)
		{
			QofDate *qd;
			qd = qof_date_from_qtime (qt_increment);
			qof_date_adddays (qd, day_interval);
			qt_increment = qof_date_to_qtime (qd);
			qof_date_free (qd);
			qd = qof_date_from_qtime (qt_end);
			qof_date_adddays (qd, day_interval);
			qt_end = qof_date_to_qtime (qd);
			qof_date_free (qd);
		}
		if (month_interval)
		{
			QofDate *qd;
			qd = qof_date_from_qtime (qt_increment);
			qof_date_addmonths (qd, month_interval, FALSE);
			qt_increment = qof_date_to_qtime (qd);
			qof_date_free (qd);
			qd = qof_date_from_qtime (qt_end);
			qof_date_addmonths (qd, month_interval, FALSE);
			qt_end = qof_date_to_qtime (qd);
			qof_date_free (qd);
		}
		for (i = 0; i < qa->exceptions; i++)
		{
			QofDate *qd;

			qd = qof_date_from_qtime(qt_increment);
			if ((qd->qd_year == qa->exception[i].tm_year) &&
				(qd->qd_mday == qa->exception[i].tm_mday) &&
				(qd->qd_mon == qa->exception[i].tm_mon))
			{
				/* exclude */
				skip = TRUE;
			}
		}
		/* create the repeater */
		if (!skip)
		{
			book = inst->book;
			clone = datebook_create (book);
			clone->repeater = TRUE;
			/* clone the main parts of the event */
			datebook_setEvent (clone, datebook_getEvent (obj));
			datebook_setAlarm (clone, datebook_getAlarm (obj));
			datebook_setAdvance (clone, datebook_getAdvance (obj));
			datebook_setAdvanceUnit (clone, datebook_getAdvanceUnit (obj));
			datebook_setDescription (clone, datebook_getDescription (obj));
			datebook_setNote (clone, datebook_getNote (obj));
			datebook_setCategory (clone, datebook_getCategory (obj));
			/* the clone does not repeat */
			datebook_setExceptions (clone, 0);
			datebook_setRepeatEnd (clone, qt_end);
			datebook_setRepeatForever (clone, FALSE);
			datebook_setRepeatType (clone,
				repeatTypesasString (repeatNone));
			datebook_setBegin (clone, qt_increment);
			datebook_setEnd (clone, qt_end);
		}
	}
	return size;
}

/** \brief pack the entity into a Palm datebook record */
static gint
datebook_pack (QofEntity * ent, gpointer user_data)
{
	PQContext *context;
	Appointment_t *qa;
	QofDateBook *obj;
	QofTime *qt;
	KvpFrame *frame;
	gint size, i;
	gchar *path;

	size = 0;
	i = 1;
	context = (PQContext *) user_data;
	g_return_val_if_fail ((context || ent), -1);
	ENTER (" ");
	obj = (QofDateBook *) ent;
	if (obj->repeater == TRUE)
		return 0;
	qa = &obj->wrap;
	size = pack_Appointment (qa, context->pi_buf, DATEBOOK_VERSION);	// 0.12
	/* pack slots into exceptions */
	frame = qof_instance_get_slots ((QofInstance *) ent);
	if (frame)
	{
		path = g_strdup_printf ("%s/%d", DATEBOOK_KVP_PATH, i);
		while (kvp_frame_get_value (frame, path))
		{
			qt = kvp_frame_get_time (frame, path);
			if (qt)
			{
				QofDate *qd;
				gboolean result;
				qd = qof_date_from_qtime (qt);
				result = qof_date_to_struct_tm (qd,
					&qa->exception[i], 0);
				if(!result)
					PERR (" failed to set exception time: "
					"out of range");
			}
			g_free (path);
			i++;
			path = g_strdup_printf ("%s/%d", DATEBOOK_KVP_PATH, i);
		}
		g_free (path);
	}
	LEAVE (" ");
	return size;
}

/** \brief unpack the application information.

The AppInfo contains the category list (each record only holds
the index value of this list as the category value) and other
preferences.
*/
static gint
datebook_appinfo_unpack (QofEntity * ent, gpointer user_data)
{
	AppointmentAppInfo_t app_db;
	PQContext *context;
	gint name_count;

	context = (PQContext *) user_data;
	g_return_val_if_fail (context != NULL, -1);
	ENTER (" ");
	unpack_AppointmentAppInfo (&app_db, context->app_buf->data,
		PQ_DEF_BUFSZ);
	for (name_count = 0; name_count < 16; name_count++)
	{
		g_sprintf (context->names[name_count], "%s",
			app_db.category.name[name_count]);
	}
	context->pi_cat = &app_db.category;
	LEAVE (" ");
	return 0;
}

/** free all memory related to the datebook record. */
static gint
qof_datebook_free (QofEntity * ent, gpointer user_data)
{
	Appointment_t *qd;
	QofDateBook *obj;

	g_return_val_if_fail (ent != NULL, -1);
	ENTER (" ");
	obj = (QofDateBook *) ent;
	qd = &obj->wrap;
	free_Appointment (qd);
	LEAVE (" ");
	return 0;
}

static const gchar *
datebookPrintable (gpointer instance)
{
	return datebook_getDescription ((QofDateBook *) instance);
}

static QofObject datebook_object_def = {
  interface_version:QOF_OBJECT_VERSION,
  e_type:PILOT_LINK_QOF_DATEBOOK,
  type_label:QOF_DATEBOOK_DESC,
  create:(gpointer) datebook_create,
  book_begin:NULL,
  book_end:NULL,
  is_dirty:qof_collection_is_dirty,
  mark_clean:qof_collection_mark_clean,
  foreach:qof_collection_foreach,
  printable:datebookPrintable,
  version_cmp:(gint (*)(gpointer, gpointer)) qof_instance_version_cmp,
};

static PQPack datebook_pack_def = {
  e_type:PILOT_LINK_QOF_DATEBOOK,
  pack_func:datebook_pack,
  unpack_func:datebook_unpack,
  free_pack_func:qof_datebook_free,
  palm_db_name:"DatebookDB",
  app_info_unpack:datebook_appinfo_unpack,
};

gboolean
DateBookRegister (void)
{
	static QofParam params[] = {
		{DATEBOOK_EVENT, QOF_TYPE_BOOLEAN,
				(QofAccessFunc) datebook_getEvent,
			(QofSetterFunc) datebook_setEvent},
		{DATEBOOK_BEGIN, QOF_TYPE_TIME, (QofAccessFunc) datebook_getBegin,
			(QofSetterFunc) datebook_setBegin},
		{DATEBOOK_END, QOF_TYPE_TIME, (QofAccessFunc) datebook_getEnd,
			(QofSetterFunc) datebook_setEnd},
		{DATEBOOK_ALARM, QOF_TYPE_BOOLEAN,
				(QofAccessFunc) datebook_getAlarm,
			(QofSetterFunc) datebook_setAlarm},
		{DATEBOOK_ADVANCE, QOF_TYPE_INT32,
				(QofAccessFunc) datebook_getAdvance,
			(QofSetterFunc) datebook_setAdvance},
		{DATEBOOK_ADV_UNIT, QOF_TYPE_STRING,
				(QofAccessFunc) datebook_getAdvanceUnit,
			(QofSetterFunc) datebook_setAdvanceUnit},
		{DATEBOOK_REPEAT_TYPE, QOF_TYPE_STRING,
				(QofAccessFunc) datebook_getRepeatType,
			(QofSetterFunc) datebook_setRepeatType},
		{DATEBOOK_REPEAT_FOREVER, QOF_TYPE_BOOLEAN,
				(QofAccessFunc) datebook_getRepeatForever,
			(QofSetterFunc) datebook_setRepeatForever},
		{DATEBOOK_REPEAT_END, QOF_TYPE_TIME,
				(QofAccessFunc) datebook_getRepeatEnd,
			(QofSetterFunc) datebook_setRepeatEnd},
		{DATEBOOK_REPEAT_FREQUENCY, QOF_TYPE_INT32,
				(QofAccessFunc) datebook_getRepeatFrequency,
			(QofSetterFunc) datebook_setRepeatFrequency},
		{DATEBOOK_REPEAT_DAY, QOF_TYPE_STRING,
				(QofAccessFunc) datebook_getRepeatDay,
			(QofSetterFunc) datebook_setRepeatDay},
		{DATEBOOK_REPEAT_WEEK_START, QOF_TYPE_STRING,
				(QofAccessFunc) datebook_getRepeatWeekStart,
			(QofSetterFunc) datebook_setRepeatWeekStart},
		{DATEBOOK_EXCEPTIONS, QOF_TYPE_INT32,
				(QofAccessFunc) datebook_getExceptions,
			(QofSetterFunc) datebook_setExceptions},
		{DATEBOOK_DESCRIPTION, QOF_TYPE_STRING,
				(QofAccessFunc) datebook_getDescription,
			(QofSetterFunc) datebook_setDescription},
		{DATEBOOK_CATEGORY, QOF_TYPE_STRING,
				(QofAccessFunc) datebook_getCategory,
			(QofSetterFunc) datebook_setCategory},
		{DATEBOOK_REPEATER, QOF_TYPE_BOOLEAN,
				(QofAccessFunc) datebook_check_repeater,
			(QofSetterFunc) datebook_set_repeater},
		{DATEBOOK_NOTE, QOF_TYPE_STRING, (QofAccessFunc) datebook_getNote,
			(QofSetterFunc) datebook_setNote},
		{DATEBOOK_DURATION, QOF_TYPE_DOUBLE,
				(QofAccessFunc) datebook_getDuration, NULL},
		{DATEBOOK_EXCEPTION, QOF_TYPE_KVP,
				(QofAccessFunc) qof_instance_get_slots, NULL},
		{QOF_PARAM_BOOK, QOF_ID_BOOK,
				(QofAccessFunc) qof_instance_get_book, NULL},
		{QOF_PARAM_GUID, QOF_TYPE_GUID,
				(QofAccessFunc) qof_instance_get_guid, NULL},
		{NULL},
	};

	qof_class_register (PILOT_LINK_QOF_DATEBOOK, NULL, params);

	pilot_qof_pack_register (&datebook_pack_def);

	return qof_object_register (&datebook_object_def);
}

/** @} */
