/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * GTK+ widget for embedding a Pigment viewport
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/**
 * SECTION:pgmgtk
 * @short_description: A GTK+ widget embedding viewports.
 *
 * #PgmGtk is a GTK+ widget allowing to embed a #PgmViewport inside a
 * GTK+ application.
 *
 * Last reviewed on 2007-11-09 (0.3.2)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <gtk/gtkwidget.h>
#include "pgmgtk.h"

G_DEFINE_TYPE (PgmGtk, pgm_gtk, GTK_TYPE_SOCKET);

/* GtkObject methods */

static void
pgm_gtk_size_request (GtkWidget *widget,
                      GtkRequisition *requisition)
{
  requisition->width = 1;
  requisition->height = 1;
}

static void
pgm_gtk_size_allocate (GtkWidget *widget,
                       GtkAllocation *allocation)
{
  PgmGtk *gtk = PGM_GTK (widget);

  if (gtk->viewport)
    pgm_viewport_set_size (gtk->viewport, allocation->width, allocation->height);

  GTK_WIDGET_CLASS (pgm_gtk_parent_class)->size_allocate (widget, allocation);
}

static void
pgm_gtk_map (GtkWidget *widget)
{
  PgmGtk *gtk = PGM_GTK (widget);

  if (gtk->viewport)
    {
      gtk_socket_add_id (GTK_SOCKET (widget), gtk->embedding_id);
      pgm_viewport_show (gtk->viewport);
    }

  GTK_WIDGET_CLASS (pgm_gtk_parent_class)->map (widget);
}

static void
pgm_gtk_destroy (GtkObject *object)
{
  PgmGtk *gtk = PGM_GTK (object);

  if (gtk->viewport)
    {
      g_object_unref (gtk->viewport);
      gtk->viewport = NULL;
      gtk->embedding_id = 0;
    }

  GTK_OBJECT_CLASS (pgm_gtk_parent_class)->destroy (object);
}

/* GObject stuff */

static void
pgm_gtk_class_init (PgmGtkClass *klass)
{
  GtkObjectClass *gtkobject_class = GTK_OBJECT_CLASS (klass);
  GtkWidgetClass *gtkwidget_class = GTK_WIDGET_CLASS (klass);

  gtkobject_class->destroy = pgm_gtk_destroy;
  gtkwidget_class->size_request = pgm_gtk_size_request;
  gtkwidget_class->size_allocate = pgm_gtk_size_allocate;
  gtkwidget_class->map = pgm_gtk_map;
}

static void
pgm_gtk_init (PgmGtk *gtk)
{
  gtk->viewport = NULL;
  gtk->embedding_id = 0;

  /* Viewport handles double-buffering on its own */
  gtk_widget_set_double_buffered (GTK_WIDGET (gtk), FALSE);
}

/* Public methods */

/**
 * pgm_gtk_new:
 *
 * Creates a new #PgmGtk object.
 *
 * Returns: a new #PgmGtk instance.
 */
GtkWidget*
pgm_gtk_new (void)
{
  return g_object_new (PGM_TYPE_GTK, NULL);
}

/**
 * pgm_gtk_set_viewport:
 * @gtk: A #PgmGtk object.
 * @viewport: A #PgmViewport object.
 *
 * Embeds @viewport in the GTK+ widget @gtk. @viewport must support application
 * embedding for the function to success.
 *
 * Returns: %TRUE if @viewport has been embedded successfully, %FALSE otherwise.
 */
gboolean
pgm_gtk_set_viewport (PgmGtk *gtk,
                      PgmViewport *viewport)
{
  gulong caps_mask = 0;

  g_return_val_if_fail (PGM_IS_GTK (gtk), FALSE);

  /* Clean up if there's already a viewport */
  if (gtk->viewport)
    {
      g_object_unref (gtk->viewport);
      gtk->viewport = NULL;
      gtk->embedding_id = 0;
    }

  if (!PGM_IS_VIEWPORT (viewport))
    return FALSE;

  gtk->viewport = g_object_ref (viewport);

  /* Check if it supports embedding */
  pgm_viewport_get_caps_mask (viewport, &caps_mask);
  if (!(caps_mask & PGM_VIEWPORT_APPLICATION_EMBEDDING))
    {
      g_object_unref (viewport);
      return FALSE;
    }

  /* And finally retrieve the ID */
  pgm_viewport_get_embedding_id (gtk->viewport, &gtk->embedding_id);

  return TRUE;
}

/**
 * pgm_gtk_get_viewport:
 * @gtk: a #PgmViewport object.
 * @viewport: a pointer to a #PgmViewport pointer where the @viewport is going
 * to be stored. Unref after usage.
 *
 * Retrieves the @viewport of @gtk.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_gtk_get_viewport (PgmGtk *gtk,
                      PgmViewport **viewport)

{
  g_return_val_if_fail (PGM_IS_GTK (gtk), PGM_ERROR_X);
  g_return_val_if_fail (viewport != NULL, PGM_ERROR_X);

  if (gtk->viewport)
    *viewport = gst_object_ref (GST_OBJECT (gtk->viewport));
  else
    *viewport = NULL;

  return PGM_ERROR_OK;
}
