/* handle error/warning reporting */
/* stdout will be flushed before any write to stderr
 * stderr will be flushed after any write to it
 */

#include	"includes.h"

bool
	displayErrors,								/* used to determine if errors should be displayed */
	displayWarnings,							/* used to determine if warnings should be displayed */
	displayMessages,							/* used to determine if messages should be displayed */
	displayDiagnostics;							/* used to determine if diagnostics should be displayed */
ULONG
	numErrors,									/* keep track of the number of calls to each of these */
	numWarnings,
	numMessages,
	numDiagnostics;
ULONG
	numDisplayedErrors,							/* keep track of the number of calls to each of these that printed */
	numDisplayedWarnings,
	numDisplayedMessages,
	numDisplayedDiagnostics;

char
	errorFileName[MAXERRORFILENAME+1];			/* name of file that error occured in (if NULL, then no file) */
ULONG
	errorLineNum;								/* line number that error occured on (if no file name, this is meaningless) */

void SetErrorFile(char *theString)
/* set the name of the file, so that if ReportError, or ReportWarning, the
 * file name can be displayed
 * if theString is of length 0, then no file, or line number will be reported
 */
{
	strncpy(&(errorFileName[0]),theString,MAXERRORFILENAME);	/* copy the passed file name */
	errorFileName[MAXERRORFILENAME]='\0';						/* terminate the name */
}

void GetErrorFile(char *theString,int theStringSize)
/* copy the current error file string into theString
 */
{
	if(theStringSize)
		{
		strncpy(theString,&(errorFileName[0]),theStringSize-1);	/* leave room for null terminator */
		theString[theStringSize-1]='\0';		/* terminate if strncpy could not */
		}
}

void SetErrorLine(ULONG theLine)
/* set the line number so that if ReportError, or ReportWarning are called, the
 * line number will be correct
 */
{
	errorLineNum=theLine;
}

ULONG GetErrorLine()
/* return the line number that will be reported if an error occurs
 */
{
	return(errorLineNum);
}

void SetNumErrors(ULONG theNumber)
/* set the count of the number of errors
 */
{
	numErrors=theNumber;
}

ULONG GetNumErrors()
/* get the count of the number of errors
 */
{
	return(numErrors);
}

void SetNumWarnings(ULONG theNumber)
/* set the count of the number of warnings
 */
{
	numWarnings=theNumber;
}

ULONG GetNumWarnings()
/* get the count of the number of warnings
 */
{
	return(numWarnings);
}

void SetNumMessages(ULONG theNumber)
/* set the count of the number of messages
 */
{
	numMessages=theNumber;
}

ULONG GetNumMessages()
/* get the count of the number of messages
 */
{
	return(numMessages);
}

void SetNumDiagnostics(ULONG theNumber)
/* set the count of the number of diagnostics
 */
{
	numDiagnostics=theNumber;
}

ULONG GetNumDiagnostics()
/* get the count of the number of diagnostics
 */
{
	return(numDiagnostics);
}

void SetNumDisplayedErrors(ULONG theNumber)
/* set the count of the number of displayed errors
 */
{
	numDisplayedErrors=theNumber;
}

ULONG GetNumDisplayedErrors()
/* get the count of the number of displayed errors
 */
{
	return(numDisplayedErrors);
}

void SetNumDisplayedWarnings(ULONG theNumber)
/* set the count of the number of displayed warnings
 */
{
	numDisplayedWarnings=theNumber;
}

ULONG GetNumDisplayedWarnings()
/* get the count of the number of displayed warnings
 */
{
	return(numDisplayedWarnings);
}

void SetNumDisplayedMessages(ULONG theNumber)
/* set the count of the number of displayed messages
 */
{
	numDisplayedMessages=theNumber;
}

ULONG GetNumDisplayedMessages()
/* get the count of the number of displayed messages
 */
{
	return(numDisplayedMessages);
}

void SetNumDisplayedDiagnostics(ULONG theNumber)
/* set the count of the number of displayed diagnostics
 */
{
	numDisplayedDiagnostics=theNumber;
}

ULONG GetNumDisplayedDiagnostics()
/* get the count of the number of displayed diagnostics
 */
{
	return(numDisplayedDiagnostics);
}

void SetErrorDisplay(bool doDisplay)
/* set the status of error display
 */
{
	displayErrors=doDisplay;
}

bool GetErrorDisplay()
/* get the status of error display
 */
{
	return(displayErrors);
}

void SetWarningDisplay(bool doDisplay)
/* set the status of warning display
 */
{
	displayWarnings=doDisplay;
}

bool GetWarningDisplay()
/* get the status of warning display
 */
{
	return(displayWarnings);
}

void SetMessageDisplay(bool doDisplay)
/* set the status of message display
 */
{
	displayMessages=doDisplay;
}

bool GetMessageDisplay()
/* get the status of message display
 */
{
	return(displayMessages);
}

void SetDiagnosticDisplay(bool doDisplay)
/* set the status of diagnostic display
 */
{
	displayDiagnostics=doDisplay;
}

bool GetDiagnosticDisplay()
/* get the status of diagnostic display
 */
{
	return(displayDiagnostics);
}

void vReportError(char *format,va_list args)
/* if an error occurs, report it with this function
 */
{
	if(displayErrors)
	{
		fflush(stdout);							/* accomodate machines that may be buffering */
		fprintf(stderr,"\nerror: ");
		vfprintf(stderr,format,args);			/* call dump the string */
		if(strlen(&(errorFileName[0])))
		{
			fprintf(stderr,"File '%s'; Line %d\n",&(errorFileName[0]),errorLineNum);
		}
		fflush(stderr);							/* accomodate machines that may be buffering */
		numDisplayedErrors++;
	}
	numErrors++;
}

void vReportWarning(char *format,va_list args)
/* if a warning occurs, report it with this function
 */
{
	if(displayWarnings)
	{
		fflush(stdout);							/* accomodate machines that may be buffering */
		fprintf(stderr,"\nWarning --------\n");
		vfprintf(stderr,format,args);			/* call dump the string */
		if(strlen(&(errorFileName[0])))
		{
			fprintf(stderr,"File '%s'; Line %d\n",&(errorFileName[0]),errorLineNum);
		}
		fflush(stderr);							/* accomodate machines that may be buffering */
		numDisplayedWarnings++;
	}
	numWarnings++;
}

void vReportMessage(char *format,va_list args)
/* report a message to the user
 */
{
	if(displayMessages)
		{
		vfprintf(stdout,format,args);				/* call dump the string */
		numDisplayedMessages++;
		}
	numMessages++;
}

void vReportDiagnostic(char *format,va_list args)
/* report current diagnostic conditions with this
 */
{
	if(displayDiagnostics)
		{
		fflush(stdout);							/* accomodate machines that may be buffering */
		vfprintf(stderr,format,args);			/* call dump the string */
		fflush(stderr);							/* accomodate machines that may be buffering */
		numDisplayedDiagnostics++;
		}
	numDiagnostics++;
}

void ReportError(char *format,...)
/* if an error occurs, report it with this function
 */
{
va_list
	args;

	va_start(args,format);
	vReportError(format,args);				/* do actual reporting */
	va_end(args);
}

void ReportWarning(char *format,...)
/* if an warning occurs, report it with this function
 */
{
va_list
	args;

	va_start(args,format);
	vReportWarning(format,args);			/* do actual reporting */
	va_end(args);
}

void ReportMessage(char *format,...)
/* report a message to the user
 */
{
va_list
	args;

	va_start(args,format);
	vReportMessage(format,args);			/* do actual reporting */
	va_end(args);
}

void ReportDiagnostic(char *format,...)
/* report current diagnostic conditions with this
 */
{
va_list
	args;

	va_start(args,format);
	vReportDiagnostic(format,args);			/* do actual reporting */
	va_end(args);
}

bool InitReportErrors()
/* initialize error handling structures
 * if there is a problem, return false
 */
{
	SetErrorFile("");
	SetErrorLine((ULONG)0);
	SetNumErrors((ULONG)0);
	SetNumWarnings((ULONG)0);
	SetNumMessages((ULONG)0);
	SetNumDiagnostics((ULONG)0);
	SetNumDisplayedErrors((ULONG)0);
	SetNumDisplayedWarnings((ULONG)0);
	SetNumDisplayedMessages((ULONG)0);
	SetNumDisplayedDiagnostics((ULONG)0);
	SetErrorDisplay(true);
	SetWarningDisplay(true);
	SetMessageDisplay(true);
	SetDiagnosticDisplay(false);
	return(true);
}

void UnInitReportErrors()
/* uninitialize error handling structures
 */
{
}