/*
 * profileselector.c - provides a custom widget for selecting ICC profiles
 * given a SearchPath pointing to the profile locations.
 *
 * Copyright (c) 2006 by Alastair M. Robinson
 * Distributed under the terms of the GNU General Public License -
 * see the file named "COPYING" for more details.
 *
 */


/* FIXME:  A destructor is needed for this class, to free up the option list.
   As it stands, a certain amount of memory will be lost when the widget is
   destroyed.
*/
#include <iostream>

#include <string.h>

#include <gtk/gtkentry.h>
#include <gtk/gtklist.h>
#include <gtk/gtkfilesel.h>
#include <gtk/gtkfilechooser.h>
#include <gtk/gtkmenuitem.h>

#include "../support/generaldialogs.h"

#include "profileselector.h"

static char *PS_ESCAPESTRING="Other...";

using namespace std;

enum {
	CHANGED_SIGNAL,
	LAST_SIGNAL
};

static guint profileselector_signals[LAST_SIGNAL] = { 0 };

static void profileselector_class_init (ProfileSelectorClass *klass);
static void profileselector_init (ProfileSelector *sel);

static int mycmp(const void *s1,const void *s2)
{
	return(strcmp((const char *)s1,(const char *)s2));
}


static bool verifyprofile(ProfileSelector *c,ProfileInfo *pi)
{
	try
	{
		if(c->colourspace || (!c->allowdevicelink))
		{
			if(c->colourspace)
			{
				if(pi->GetColourSpace()!=c->colourspace)
					throw "Ignoring profile - wrong colourspace";
			}
			if(!c->allowdevicelink)
			{
				if(pi->IsDeviceLink())
					throw "Ignoring profile - can't use devicelinks";
			}
		}
		return(true);
	}
	catch(const char *err)
	{
//		cerr << "Profile Selector: " << err << endl;
	}
	return(false);
}


static void rebuild_menu(ProfileSelector *c)
{
	if(c->menu)
		gtk_option_menu_remove_menu(GTK_OPTION_MENU(c->optionmenu));
	c->menu=gtk_menu_new();

	// Build popup menu from optionlist...
	GList *iter=c->optionlist;
	while(iter)
	{
		gchar *text=(gchar *)iter->data;
		GtkWidget *menu_item = gtk_menu_item_new_with_label (text);
		gtk_menu_shell_append (GTK_MENU_SHELL (c->menu), menu_item);
		gtk_widget_show (menu_item);
		iter=g_list_next(iter);
	}	
	gtk_option_menu_set_menu(GTK_OPTION_MENU(c->optionmenu),c->menu);
}


static void profileselector_build_options(ProfileSelector *c)
{
	cerr << "Building option list" << endl;
	if(c->optionlist)
	{
		GList *element;
		element=c->optionlist;
		while(element)
		{
			g_free(element->data);
			element=g_list_next(element);
		}
		g_list_free(c->optionlist);
		c->optionlist=NULL;
	}

	ProfileInfo *pi=c->pm->GetFirstProfileInfo();
	while(pi)
	{
		const char *filename=pi->GetFilename();
		if(g_list_find_custom(c->optionlist,filename,mycmp))
			cerr << "Profile already in list - ignoring" << endl;
		else
		{
			if(verifyprofile(c,pi))
			{
				c->optionlist=g_list_append(c->optionlist,g_strdup(filename));
			}
		}
		pi=pi->Next();
	}

	c->optionlist=g_list_sort(c->optionlist,mycmp);

	// "Other..." option, pops up a file selector.
	c->optionlist=g_list_append(c->optionlist,g_strdup(PS_ESCAPESTRING));

	rebuild_menu(c);
}


static void	profileselector_entry_changed(GtkEntry *entry,gpointer user_data)
{
	ProfileSelector *c=PROFILESELECTOR(user_data);
	GtkOptionMenu *om=GTK_OPTION_MENU(c->optionmenu);

	gint index=gtk_option_menu_get_history(om);
	const char *val=NULL;
	if(c->optionlist)
		val=(const char *)g_list_nth_data(c->optionlist,index);

	if(val && strlen(val))
	{
		if(strcmp(val,PS_ESCAPESTRING)==0)
		{
			char *fn=File_Dialog("Choose ICC Profile...",NULL,NULL);
			if(fn)
			{
				profileselector_set_filename(c,fn);
				val=NULL;
			}
			else
				val=c->filename;
		}
		if(val)
			profileselector_set_filename(c,val);

		g_signal_emit(G_OBJECT (c),
			profileselector_signals[CHANGED_SIGNAL], 0);
	}
}


GtkWidget*
profileselector_new (ProfileManager *pm,IS_TYPE colourspace,bool allowdevicelink)
{
	ProfileSelector *c=PROFILESELECTOR(g_object_new (profileselector_get_type (), NULL));

	c->pm=pm;
	c->colourspace=colourspace;
	c->allowdevicelink=allowdevicelink;

	c->optionmenu=gtk_option_menu_new();
	c->menu=NULL;  // Built on demand...

	gtk_box_pack_start(GTK_BOX(c),GTK_WIDGET(c->optionmenu),TRUE,TRUE,0);

	GtkWidget *frame=gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(frame),GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(c),GTK_WIDGET(frame),TRUE,FALSE,0);

	c->description=gtk_label_new("");
	gtk_container_add(GTK_CONTAINER(frame),GTK_WIDGET(c->description));
	
	gtk_widget_show(c->description);
	gtk_widget_show(frame);
	gtk_widget_show(c->optionmenu);

	profileselector_build_options(c);
	g_signal_connect(c->optionmenu,"changed",G_CALLBACK(profileselector_entry_changed),c);
	
	return(GTK_WIDGET(c));
}


GType
profileselector_get_type (void)
{
	static GType stpuic_type = 0;

	if (!stpuic_type)
	{
		static const GTypeInfo profileselector_info =
		{
			sizeof (ProfileSelectorClass),
			NULL, /* base_init */
			NULL, /* base_finalize */
			(GClassInitFunc) profileselector_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
			sizeof (ProfileSelector),
			0,
			(GInstanceInitFunc) profileselector_init,
		};
		stpuic_type = g_type_register_static (GTK_TYPE_VBOX, "ProfileSelector", &profileselector_info, GTypeFlags(0));
	}
	return stpuic_type;
}


static void
profileselector_class_init (ProfileSelectorClass *klass)
{
	profileselector_signals[CHANGED_SIGNAL] =
	g_signal_new ("changed",
		G_TYPE_FROM_CLASS (klass),
		GSignalFlags(G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION),
		G_STRUCT_OFFSET (ProfileSelectorClass, changed),
		NULL, NULL,
		g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}


static void
profileselector_init (ProfileSelector *c)
{
	c->optionlist=NULL;
	c->filename=NULL;
}


gboolean profileselector_refresh(ProfileSelector *c)
{
	g_signal_handlers_block_matched (G_OBJECT (c->optionmenu), 
                                         G_SIGNAL_MATCH_DATA,
                                         0, 0, NULL, NULL, c);
	profileselector_build_options(c);
	g_signal_handlers_unblock_matched (G_OBJECT (c->optionmenu), 
                                         G_SIGNAL_MATCH_DATA,
                                         0, 0, NULL, NULL, c);
	return(true);
}


void profileselector_set_filename(ProfileSelector *c,const char *filename)
{
	char *fn=c->pm->MakeRelative(filename);
//	if(verifyprofile(c,fn))
//	{
		if(c->filename)
			free(c->filename);
		c->filename=fn; // Don't free fn, c->filename has "adopted" it.
		
		// If the filename is not present, add it.

		GList *node;
		if((node=g_list_find_custom(c->optionlist,c->filename,mycmp)))
		{
			gint i=g_list_position(c->optionlist,node);
			gtk_option_menu_set_history(GTK_OPTION_MENU(c->optionmenu),i);			
		}
		else
		{
			g_signal_handlers_block_matched (G_OBJECT (c->optionmenu), 
		                                         G_SIGNAL_MATCH_DATA,
		                                         0, 0, NULL, NULL, c);

			c->optionlist=g_list_append(c->optionlist,g_strdup(c->filename));
			rebuild_menu(c);
			gtk_option_menu_set_history(GTK_OPTION_MENU(c->optionmenu),g_list_length(c->optionlist)-1);

			g_signal_handlers_unblock_matched (G_OBJECT (c->optionmenu), 
		                                         G_SIGNAL_MATCH_DATA,
		                                         0, 0, NULL, NULL, c);
		}

		try
		{
			char *fn=c->pm->Search(c->filename);
			if(fn && strlen(fn))
			{
				CMSProfile profile(fn);
				free(fn);
				gsize in,out;
				gchar *entry=g_locale_to_utf8(profile.GetDescription(),-1,&in,&out,NULL);
				gtk_label_set_text(GTK_LABEL(c->description),entry);
			}
		}
		catch(const char *err)
		{
			cerr << "Profile Selector: " << err << endl;
		}
//	}
}


const char *profileselector_get_filename(ProfileSelector *p)
{
	return(p->filename);
}
