/*
 * PhotoPrint
 * Copyright (c) 2004-2006 by Alastair M. Robinson
 * Distributed under the terms of the GNU General Public License -
 * see the file named "COPYING" for more details.
 *
 */

#include <iostream>

#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>

#include "support/configdb.h"
#include "photoprint_state.h"

#include "pp_mainwindow.h"
#include "support/progressbar.h"
#include "dialogs.h"
#include "support/generaldialogs.h"
#include "splashscreen/splashscreen.h"
#include "util.h"

#include "config.h"

using namespace std;

bool ParseOptions(int argc,char *argv[],char **presetname)
{
	int batchmode=false;
	static struct option long_options[] =
	{
		{"help",no_argument,NULL,'h'},
		{"version",no_argument,NULL,'v'},
		{"preset",required_argument,NULL,'p'},
		{"batch",no_argument,NULL,'b'},
		{0, 0, 0, 0}
	};

	while(1)
	{
		int c;
		c = getopt_long(argc,argv,"hvp:b",long_options,NULL);
		if(c==-1)
			break;
		switch (c)
		{
			case 'h':
				printf("Usage: %s [options] image1 [image2] ... \n",argv[0]);
				printf("\t -h --help\t\tdisplay this message\n");
				printf("\t -v --version\t\tdisplay version\n");
				printf("\t -p --preset\t\tread a specific preset file\n");
				printf("\t -b --batch\t\trun without user interface\n");
				throw 0;
				break;
			case 'v':
				printf("%s\n",PACKAGE_STRING);
				throw 0;
				break;
			case 'p':
				*presetname=optarg;
				break;
			case 'b':
				batchmode=true;
				break;
		}
	}
	return(batchmode);
}


static void destroy( GtkWidget *widget,
                     gpointer   data )
{
    gtk_main_quit ();
}


int main(int argc,char **argv)
{
	try
	{
		cerr << "Photoprint starting..." << endl;
		gboolean have_gtk=false;
		char *presetname=NULL;

		bool batchmode=ParseOptions(argc,argv,&presetname);
		if(!batchmode)
			have_gtk=gtk_init_check (&argc, &argv);

		SplashScreen *splash=NULL;
		if(have_gtk)
		{
			splash=new SplashScreen;
			splash->SetMessage("Initializing...");
		}

		cerr << "SplashScreen displayed..." << endl;

		PhotoPrint_State state(batchmode);

		cerr << "State created..." << endl;

		if(presetname)
			state.SetFilename(presetname);
		cerr << "Set initial filename" << endl;

		if(have_gtk)
			splash->SetMessage("Checking .photoprint directory...");

		CheckSettingsDir(".photoprint");

		if(have_gtk)
			splash->SetMessage("Loading preset...");
		state.ParseFile();

		if(have_gtk)
		{
			splash->SetMessage("Creating layout...");
			delete splash;
		}

		state.NewLayout();

		if(batchmode)
		{
			try
			{
				cerr << "Running in batch mode" << endl;
				if(argc>optind)
				{
					for(int i=optind;i<argc;++i)
					{
						cerr << "Adding file: " << argv[i] << endl;
						state.layout->AddImage(argv[i]);
					}
					Progress p;
					state.layout->Print(&p);
				}
			}
			catch(const char *err)
			{
				cerr << "Error: " << err << endl;
			}
		}
		else
		{
			try
			{
				GtkWidget *mainwindow;
				mainwindow = pp_mainwindow_new(&state);
				g_signal_connect (G_OBJECT (mainwindow), "destroy",
					    G_CALLBACK (destroy), NULL);
				gtk_widget_show (mainwindow);
		
				if(argc>optind)
				{
					ProgressBar p("Loading images...",true,mainwindow);
					int lastpage=0;
					for(int i=optind;i<argc;++i)
					{
						if(!p.DoProgress(i-optind,argc-optind))
							break;
						lastpage=state.layout->AddImage(argv[i]);
					}
					state.layout->SetCurrentPage(lastpage);
				}
	
				pp_mainwindow_refresh(PP_MAINWINDOW(mainwindow));
	
				gtk_main ();
			}
			catch (const char *err)
			{
				ErrorMessage_Dialog(err);
			}
		}
	}
	catch(const char *err)
	{
		cerr << "Error: " << err << endl;
	}
	catch(int retcode)
	{
		return(retcode);
	}
	return(0);
}
