#==========================================================
# Functions --
#
#   procedures for manipulating PostgreSQL Functions
#
#==========================================================
#
namespace eval Functions {
    variable Win
    variable name
    variable nametodrop
    variable parameterstodrop
    variable returns
    variable returnstodrop
    variable language
}


#----------------------------------------------------------
#----------------------------------------------------------
#
proc ::Functions::new {} {

    global PgAcVar

    Window show .pgaw:Function

    set PgAcVar(function,name) {}
    set PgAcVar(function,nametodrop) {}
    set PgAcVar(function,parameters) {}
    set PgAcVar(function,parameterstodrop) {}
    set PgAcVar(function,returns) {}
    set PgAcVar(function,returnstodrop) {}
    set PgAcVar(function,language) {}

    .pgaw:Function.fs.text1 delete 1.0 end
    focus .pgaw:Function.fp.e1
    wm transient .pgaw:Function .pgaw:Main

}; # end proc ::Functions::new


#----------------------------------------------------------
#----------------------------------------------------------
#
proc ::Functions::design {functionname_} {

    global PgAcVar CurrentDB

    Window show .pgaw:Function
    .pgaw:Function.fs.text1 delete 1.0 end

    set sql "SELECT *
               FROM [::Database::qualifySysTable pg_proc]
              WHERE
                (SELECT SUBSTRING('$functionname_'
                   FROM 1
                    FOR POSITION('(' IN '$functionname_')-1)=proname)
                AND
                (SELECT SUBSTRING('$functionname_'
                   FROM POSITION('(' IN '$functionname_')+1
                        FOR LENGTH('$functionname_')-POSITION('(' IN '$functionname_')-1)=OIDVECTORTYPES(proargtypes))"

    wpg_select $CurrentDB $sql rec {
        set PgAcVar(function,name) $rec(proname)
        set temppar $rec(proargtypes)
        set PgAcVar(function,returns) [::Database::getPgType $rec(prorettype)]
        if {$PgAcVar(function,returns) == "unknown"} {set PgAcVar(function,returns) "opaque"}
        set PgAcVar(function,returnstodrop) $PgAcVar(function,returns)
        set funcnrp $rec(pronargs)
        set prolanguage $rec(prolang)
        .pgaw:Function.fs.text1 insert end $rec(prosrc)
    }

    set sql "SELECT lanname 
	      FROM [::Database::qualifySysTable pg_language] 
	     WHERE oid=$prolanguage"

    wpg_select $CurrentDB "$sql" rec {
        set PgAcVar(function,language) $rec(lanname)
    }
    if { $PgAcVar(function,language)=="C" || $PgAcVar(function,language)=="c" } {

	set sql "SELECT probin 
		   FROM [::Database::qualifySysTable pg_proc]
		  WHERE proname='$functionname_'"

        wpg_select $CurrentDB "$sql" rec {
        .pgaw:Function.fs.text1 delete 1.0 end
        .pgaw:Function.fs.text1 insert end $rec(probin)
        }
    }
    set PgAcVar(function,parameters) {}
    for {set i 0} {$i<$funcnrp} {incr i} {
        lappend PgAcVar(function,parameters) [Database::getPgType [lindex $temppar $i]]
    }
    set PgAcVar(function,parameters) [join $PgAcVar(function,parameters) ,]
    set PgAcVar(function,nametodrop) $PgAcVar(function,name)
    set PgAcVar(function,parameterstodrop) $PgAcVar(function,parameters)

    Syntax::highlight .pgaw:Function.fs.text1 $PgAcVar(function,language)

}; # end proc ::Functions::design


#----------------------------------------------------------
#----------------------------------------------------------
#
proc ::Functions::save {} {

    global PgAcVar

    if {$PgAcVar(function,name)==""} {
        focus .pgaw:Function.fp.e1
        showError [intlmsg "You must supply a name for this function!"]
    } elseif {$PgAcVar(function,returns)==""} {
        focus .pgaw:Function.fp.e3
        showError [intlmsg "You must supply a return type!"]
    } elseif {$PgAcVar(function,language)==""} {
        focus .pgaw:Function.fp.e4
        showError [intlmsg "You must supply the function language!"]
    } else {
        set funcbody [.pgaw:Function.fs.text1 get 1.0 end]
        # regsub -all "\n" $funcbody " " funcbody
        regsub -all {'} $funcbody {''} funcbody
        regsub -all {\\} $funcbody {\\\\} funcbody

        set OK "no"

        if {$PgAcVar(function,nametodrop)==$PgAcVar(function,name) &&
                $PgAcVar(function,returnstodrop)==$PgAcVar(function,returns) &&
                $PgAcVar(function,parameterstodrop)==$PgAcVar(function,parameters)} {
            set sql "CREATE OR REPLACE
                   FUNCTION $PgAcVar(function,name)
                            ($PgAcVar(function,parameters))
                    RETURNS $PgAcVar(function,returns) AS '$funcbody'
                   LANGUAGE '$PgAcVar(function,language)'"
            if {[sql_exec noquiet $sql]} {
                set OK "yes"
            }

        } else {

            set change [tk_messageBox -default no -type yesno -title [intlmsg "Continue ?"] -parent .pgaw:Main -message [intlmsg "You are to change name, return type or parameters type of function.\nSaving will change function OID."]]

            if {[string match $change "yes"]} {

                sql_exec noquiet "BEGIN TRANSACTION"

                if {$PgAcVar(function,nametodrop) != ""} {
                    set sql "DROP
                         FUNCTION $PgAcVar(function,nametodrop)
                                  ($PgAcVar(function,parameterstodrop))"
                    if {! [sql_exec noquiet $sql]} {
                        # return
                    }
                }

                set sql "CREATE
                       FUNCTION $PgAcVar(function,name)
                                ($PgAcVar(function,parameters))
                        RETURNS $PgAcVar(function,returns) AS '$funcbody'
                       LANGUAGE  '$PgAcVar(function,language)'"
                if {[sql_exec noquiet $sql]} {
                    sql_exec noquiet "COMMIT TRANSACTION"
                    set OK "yes"
                    set PgAcVar(function,returnstodrop) $PgAcVar(function,returns)
                    set PgAcVar(function,parameterstodrop) $PgAcVar(function,parameters)
                    set PgAcVar(function,nametodrop) $PgAcVar(function,name)
                } else {
                    sql_exec noquiet "ROLLBACK TRANSACTION"
                }
            }
        }

        if {[string match $OK "yes"]} {
#            Window destroy .pgaw:Function
#            tk_messageBox -title PostgreSQL -parent .pgaw:Main -message [intlmsg "Function saved!"]
            Mainlib::tab_click Functions
        }

        Syntax::highlight .pgaw:Function.fs.text1 $PgAcVar(function,language)
    }

}; # ::Functions::save


proc ::Functions::save_as {} {

    global PgAcVar

    if {$PgAcVar(function,name)==""} {
        focus .pgaw:Function.fp.e1
        showError [intlmsg "You must supply a name for this function!"]
    } elseif {$PgAcVar(function,returns)==""} {
        focus .pgaw:Function.fp.e3
        showError [intlmsg "You must supply a return type!"]
    } elseif {$PgAcVar(function,language)==""} {
        focus .pgaw:Function.fp.e4
        showError [intlmsg "You must supply the function language!"]
    } else {
        set funcbody [.pgaw:Function.fs.text1 get 1.0 end]
        # regsub -all "\n" $funcbody " " funcbody
        regsub -all {'} $funcbody {''} funcbody
        regsub -all {\\} $funcbody {\\\\} funcbody

        set sql "CREATE
               FUNCTION $PgAcVar(function,name)
                        ($PgAcVar(function,parameters))
                RETURNS $PgAcVar(function,returns) AS '$funcbody'
               LANGUAGE '$PgAcVar(function,language)'"
        if {[sql_exec noquiet $sql]} {
#            Window destroy .pgaw:Function
#            tk_messageBox -title PostgreSQL -parent .pgaw:Main -message [intlmsg "Function saved!"]
            Mainlib::tab_click Functions
            Syntax::highlight .pgaw:Function.fs.text1 $PgAcVar(function,language)
        }
    }

}; # ::Functions::save_as


#----------------------------------------------------------
# ::Functions::introspect --
#
#   Given a functionname, returns the SQL needed to recreate it
#
# Arguments:
#   functionname_   name of a function to introspect
#   dbh_            an optional database handle
#
# Returns:
#   insql      the CREATE statement to make this function
#----------------------------------------------------------
#
proc ::Functions::introspect {functionname_ {dbh_ ""}} {

    set insql [::Functions::clone $functionname_ $functionname_ $dbh_]

    return $insql

}; # end proc ::Functions::introspect


#----------------------------------------------------------
# ::Functions::clone --
#
#   Like introspect, only changes the functionname
#
# Arguments:
#   srcfunction_    the original function
#   destfunction_   the clone function
#   dbh_            an optional database handle
#
# Returns:
#   insql       the CREATE statement to clone this function
#----------------------------------------------------------
#
proc ::Functions::clone {srcfunction_ destfunction_ {dbh_ ""}} {

    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    set insql ""

    set sql "SELECT *
               FROM pg_proc
              WHERE
                (SELECT SUBSTRING('$srcfunction_'
                   FROM 1
                    FOR POSITION('(' IN '$srcfunction_')-1)=proname)
                AND
                (SELECT SUBSTRING('$srcfunction_'
                   FROM POSITION('(' IN '$srcfunction_')+1
                        FOR LENGTH('$srcfunction_')-POSITION('(' IN '$srcfunction_')-1)=OIDVECTORTYPES(proargtypes))"

    wpg_select $dbh_ $sql rec {
        set insql "CREATE
                 FUNCTION [lindex [split $destfunction_ (] 0]
                          ([::Functions::getParameters $rec(proargtypes) $rec(pronargs) $dbh_])
                  RETURNS [::Functions::getPgType $rec(prorettype) $dbh_]
                       AS '[::Database::quoteSQL [::Functions::getBody $rec(proname) $rec(prolang) $rec(prosrc) $dbh_]]'
                 LANGUAGE '[::Functions::getLanguage $rec(prolang) $dbh_]'"
    }

    return $insql

}; # end proc ::Functions::clone


#----------------------------------------------------------
#----------------------------------------------------------
#
proc ::Functions::getPgType {oid_ {dbh_ ""}} {

    set ret [::Database::getPgType $oid_ $dbh_]

    if {[string match $ret "unknown"]} {
        set ret "opaque"
    }

    return $ret

}; # end proc ::Functions::getPgType


#----------------------------------------------------------
#----------------------------------------------------------
#
proc ::Functions::getLanguage {oid_ {dbh_ ""}} {

    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    set lang ""

    set sql "SELECT lanname
               FROM pg_language
              WHERE oid=$oid_"

    wpg_select $dbh_ $sql rec {
        set lang $rec(lanname)
    }

    return $lang

}; # end proc ::Functions::getLanguage


#----------------------------------------------------------
#----------------------------------------------------------
#
proc ::Functions::getBody {func_ lang_ src_ {dbh_ ""}} {

    global CurrentDB

    if {[string match "" $dbh_]} {
        set dbh_ $CurrentDB
    }

    set body ""

    if {$lang_=="C" || $lang_=="c"} {
        set sql "SELECT probin
                   FROM [::Database::qualifySysTable pg_proc]
                  WHERE proname='$func_'"
        wpg_select $dbh_ $sql rec {
            set body $rec(probin)
        }
    } else {
        set body $src_
    }

    return $body

}; # end proc ::Functions::getBody


#----------------------------------------------------------
#----------------------------------------------------------
#
proc ::Functions::getParameters {argtypes_ argcount_ {dbh_}} {

    set params {}

    for {set i 0} {$i<$argcount_} {incr i} {
        lappend params [::Database::getPgType [lindex $argtypes_ $i]]
    }

    set params [join $params ,]

    return $params

}; # end proc ::Functions::getParameters



#==========================================================
#==========================================================



proc vTclWindow.pgaw:Function {base} {

    global PgAcVar

    if {$base == ""} {
        set base .pgaw:Function
    }

    if {[winfo exists $base]} {
        wm deiconify $base; return
    }

    toplevel $base -class Toplevel

    wm focusmodel $base passive
    wm geometry $base 780x630+98+212
    wm overrideredirect $base 0
    wm resizable $base 1 1
    wm deiconify $base
    wm title $base [intlmsg "Function"]

    # some helpful key bindings
    bind $base <Control-Key-w> [subst {destroy $base}]

    bind $base <Key-F1> "Help::load functions"
    bind $base <Escape> {Window destroy .pgaw:Function}

    #
    # main window
    #
    frame $base.fp \
        -height 88 \
        -relief groove \
        -width 125

    label $base.fp.l1 \
        -borderwidth 0 \
        -relief raised \
        -text [intlmsg Name]

    entry $base.fp.e1 \
        -background #fefefe \
        -borderwidth 1 \
        -textvariable PgAcVar(function,name) \
        -width 64
    bind $base.fp.e1 <Key-Return> {
        focus .pgaw:Function.fp.e2
    }

    label $base.fp.l2 \
        -borderwidth 0 \
        -relief raised \
        -text [intlmsg Parameters]

    entry $base.fp.e2 \
        -background #fefefe \
        -borderwidth 1 \
        -textvariable PgAcVar(function,parameters) \
        -width 64
    bind $base.fp.e2 <Key-Return> {
        focus .pgaw:Function.fp.e3
    }

    label $base.fp.l3 \
        -borderwidth 0 \
        -relief raised \
        -text [intlmsg Returns]

    entry $base.fp.e3 \
        -background #fefefe \
        -borderwidth 1 \
        -textvariable PgAcVar(function,returns) \
        -width 15
    bind $base.fp.e3 <Key-Return> {
        focus .pgaw:Function.fp.e4
    }

    label $base.fp.l4 \
        -borderwidth 0 \
        -relief raised \
        -text [intlmsg Language]

    entry $base.fp.e4 \
        -background #fefefe \
        -borderwidth 1 \
        -textvariable PgAcVar(function,language) \
        -width 15
    bind $base.fp.e4 <Key-Return> {
        focus .pgaw:Function.fs.text1
    }

    label $base.fp.lspace \
        -borderwidth 0 \
        -relief raised \
        -text {    }

    #
    # new frame
    #
    frame $base.fs \
        -borderwidth 2 \
        -height 75 \
        -relief groove \
        -width 125

    text $base.fs.text1 \
        -background #fefefe \
        -foreground #000000 \
        -borderwidth 1 \
        -font $PgAcVar(pref,font_fix) \
        -height 16 \
        -tabs {20 40 60 80 100 120} \
        -width 43 \
        -wrap none \
        -yscrollcommand {.pgaw:Function.fs.vsb set} \
        -xscrollcommand {.pgaw:Function.fs.hsb set}
#    bind $base.fs.text1 <KeyRelease> {
#        Syntax::highlight .pgaw:Function.fs.text1 $PgAcVar(function,language)
#    }
    bind $base.fs.text1 <Control-Key-s> {
        Functions::save
    }

    # add Ctrl-x|c|v for cut, copy, paste
    bind $base.fs.text1 <Control-Key-x> {
        set PgAcVar(shared,curseltext) [%W get sel.first sel.last]
        %W delete sel.first sel.last
    }
    bind $base.fs.text1 <Control-Key-c> {
        set PgAcVar(shared,curseltext) [%W get sel.first sel.last]
    }
    bind $base.fs.text1 <Control-Key-v> {
        if {[info exists PgAcVar(shared,curseltext)]} {
            catch {%W delete sel.first sel.last}
            %W insert insert $PgAcVar(shared,curseltext)
            %W see current
        }
    }

    scrollbar $base.fs.vsb \
        -borderwidth 1 \
        -command {.pgaw:Function.fs.text1 yview} \
        -orient vert

    scrollbar $base.fs.hsb \
        -borderwidth 1 \
        -command {.pgaw:Function.fs.text1 xview} \
        -orient horiz

    #
    # button frames
    #
    frame $base.fb \
        -borderwidth 2 \
        -height 75 \
        -width 125
    frame $base.fb.fbc \
        -borderwidth 2 \
        -height 75 \
        -width 125

    button $base.fb.fbc.btnsave \
        -command {Functions::save} \
        -borderwidth 1 \
        -padx 9 \
        -pady 3 \
        -text [intlmsg Save]
    button $base.fb.fbc.btnsave_as \
        -command {Functions::save_as} \
        -borderwidth 1 \
        -padx 9 \
        -pady 3 \
        -text [intlmsg "Save as"]
    button $base.fb.fbc.btnhelp \
        -command {Help::load functions} \
        -borderwidth 1 \
        -padx 9 \
        -pady 3 \
        -text [intlmsg Help]
    button $base.fb.fbc.btnclose \
        -borderwidth 1 \
        -command {Window destroy .pgaw:Function} \
        -padx 9 \
        -pady 3 \
        -text [intlmsg Close]

    pack $base.fp \
        -in .pgaw:Function -anchor center -expand 0 -fill x -side top 

    grid $base.fp.l1 \
        -in .pgaw:Function.fp -column 0 -row 0 -columnspan 1 -rowspan 1 -sticky w 
    grid $base.fp.e1 \
        -in .pgaw:Function.fp -column 1 -row 0 -columnspan 1 -rowspan 1 
    grid $base.fp.l2 \
        -in .pgaw:Function.fp -column 0 -row 4 -columnspan 1 -rowspan 1 -sticky w 
    grid $base.fp.e2 \
        -in .pgaw:Function.fp -column 1 -row 4 -columnspan 1 -rowspan 1 -pady 2
    grid $base.fp.l3 \
        -in .pgaw:Function.fp -column 3 -row 0 -columnspan 1 -rowspan 1 -sticky w 
    grid $base.fp.e3 \
        -in .pgaw:Function.fp -column 4 -row 0 -columnspan 1 -rowspan 1 
    grid $base.fp.l4 \
        -in .pgaw:Function.fp -column 3 -row 4 -columnspan 1 -rowspan 1 -sticky w 
    grid $base.fp.e4 \
        -in .pgaw:Function.fp -column 4 -row 4 -columnspan 1 -rowspan 1 -pady 2 
    grid $base.fp.lspace \
        -in .pgaw:Function.fp -column 2 -row 4 -columnspan 1 -rowspan 1 

    pack $base.fs \
        -in .pgaw:Function -anchor center -expand 1 -fill both -side top 
    pack $base.fs.hsb \
        -in .pgaw:Function.fs -anchor center -expand 0 -fill x -side bottom 
    pack $base.fs.text1 \
        -in .pgaw:Function.fs -anchor center -expand 1 -fill both -side left 
    pack $base.fs.vsb \
        -in .pgaw:Function.fs -anchor center -expand 0 -fill y -side right 
    pack $base.fb \
        -in .pgaw:Function -anchor center -expand 0 -fill x -side bottom 
    pack $base.fb.fbc \
        -in .pgaw:Function.fb -anchor center -expand 0 -fill none -side top 
    pack $base.fb.fbc.btnsave \
        -in .pgaw:Function.fb.fbc -anchor center -expand 0 -fill none -side left 
    pack $base.fb.fbc.btnsave_as \
        -in .pgaw:Function.fb.fbc -anchor center -expand 0 -fill none -side left 
    pack $base.fb.fbc.btnhelp \
        -in .pgaw:Function.fb.fbc -anchor center -expand 0 -fill none -side left 
    pack $base.fb.fbc.btnclose \
        -in .pgaw:Function.fb.fbc -anchor center -expand 0 -fill none -side right 

}


