/** 
 *  Implements dictionary and wordlists.
 *
 *  Copyright (C) Tektronix, Inc. 1998 - 2001. All rights reserved.
 *
 *  @see     GNU LGPL
 *  @author  Tektronix CTE              @(#) %derived_by: guidod %
 *  @version %version: bln_mpt1!5.33 %
 *    (%date_modified: Tue Oct 02 15:46:38 2001 %)
 */
/*@{*/
#if defined(__version_control__) && defined(__GNUC__)
static char* id __attribute__((unused)) = 
"@(#) $Id: %full_filespec:  dict-sub.c~bln_mpt1!5.33:csrc:bln_12xx!1 % $";
#endif

#define _P4_SOURCE 1

#include <pfe/pfe-base.h>
#include <pfe/def-xtra.h>

#include <string.h>
#include <ctype.h>
#include <stddef.h> /*offsetof*/
#include <stdlib.h>

#include <pfe/def-comp.h>
#include <pfe/term-sub.h>
#include <pfe/lined.h>
#include <pfe/_missing.h>
#include <pfe/logging.h>

/*
 * A vocabulary is organized as a mixture between hash-table and
 * linked list. (This is a practice you can observe in several
 * systems.) It works like this: Given a name, first a hash-code is
 * generated. This hash-code selects one of several linked lists
 * called threads. The hooks to these threads are stored in a table.
 *
 * The body of a WORDLIST is essentially such a table of pointers to
 * threads, while in FIG-Forth it was just a pointer to the one and
 * only linked list a VOCABULARY consists of in FIG-Forth.
 */
_export int
p4_wl_hash (const char *s, int l)
/* s string, l length of string, returns hash-code for that name */
{
#  if 0 /*GD* original by DUZ */
    int n = *s++ - '@';

    while (--l > 0)
        n = n * 37 + *s++ - '@';        /* a maybe-stupid hash function :-) */
    return n & (THREADS - 1);   /* i.e. modulo threads */

#  else /*GD* more simpler, avoiding multiply */
    register char c = *s;

    while(--l > 0)
    { c += *s++; c ^= l; }
    return c & (THREADS - 1);
#  endif
}

/*
 * If we want to traverse a WORDLIST in it's entirety, we must follow
 * all threads simultaneously. The following definition eases this by
 * locating the thread with the hook pointing to the highest memory
 * location, assuming that this thread contains the latest definition
 * entered in the given WORDLIST. For usage refer to the definition of
 * WORDS.
 *
 * When following a wordlist using topmost, a copy of the word list
 * must be made. Everytime the topmost item was processed it must be
 * replaced by its successor in the linked list.
 */

/* find the thread with the latest word in the given word list */
_export p4char **
p4_topmost (p4_Wordl *w)

{
    int n = THREADS;
    p4char **p, **s = w->thread;

    for (p = s++; --n; s++)
        if (*s > *p)
            p = s;
    return p;
}

/* return the NFA of the latest definition in the CURRENT WORDLIST */
_export p4char * 
p4_latest (void) 
{
    return *p4_topmost (CURRENT);
}

/* --------------------------------
 * word list and forget 
 */

/** 
 * create a word list in the dictionary 
 */
_export p4_Wordl *
p4_make_wordlist (p4char* nfa)
{
    p4_Wordl *w = (Wordl *) DP; /* allocate word list in HERE */
    P4_INC (DP, Wordl);
    
    ZERO (w->thread);           /* initialize all threads to empty */
    w->nfa = nfa;               /* set name for the wordlist (if any) */
    w->flag = WORDL_FLAG;       /* init flags from global flags */
    w->prev = VOC_LINK;         /* chain word list in VOC-LINK */
    VOC_LINK = w;
    w->id = w->prev ? (w->prev->id << 1) : 1;
    if (w->flag & WORDL_CURRENT)
        w->also = CURRENT;      /* if WORDL_CURRENT, search also this */
    else
        w->also = 0;
    return w;
}

_export p4_Wordl *
p4_find_wordlist (const char* nm, int nmlen)
{
    p4_Wordl* wl;
    char* nfa;

    /* a special, since FORTH has no vocabulary_RT anymore */
    if (nmlen == 5 && ! memcmp (nm, "FORTH", 5))
        return PFE.forth_wl;
    if (nmlen == 11 && ! memcmp (nm, "ENVIRONMENT", 11))
        return PFE.environ_wl;

    for (wl = VOC_LINK; wl ; wl = wl->prev)
    {
        if (! wl->nfa) continue;
        nfa = wl->nfa;
        if (NFACNT(*nfa) == nmlen && !p4_strncmpi (nfa+1, nm, nmlen))
            return wl;
    }
    return 0;
}

/** ((FORGET)) 
 * remove words from dictionary, free dictionary space, this is the
 * runtime helper of => (FORGET)
 */
FCode (p4_forget_dp)
{
    register p4_Wordl *wl;
    register p4char* new_dp = PFE.forget_dp;

    /* unchain words in all threads of all word lists: */
    for (wl = VOC_LINK; wl; wl = wl->prev)
    {
        p4char **p = wl->thread;
        int i;
        
        for (i = THREADS; --i >= 0; p++)
        {  /* unchain words in thread: */
            while (*p >= new_dp) 
            {
                if (PFE_IS_DESTROYER(*p))
                {
                    P4_info2 (" destroy: \"%.*s\"", NFACNT(**p), *p+1);
                    p4_call (p4_name_from (*p));
                    new_dp = PFE.forget_dp; /* forget_dp is volatile */
                    /* and may have changed through recursive forget */
                }
                *p = *p4_name_to_link (*p);
            }
        }
    }

    /* unchain word lists: */
    while (VOC_LINK && VOC_LINK >= (p4_Wordl *) new_dp) 
    {   
        {   /* delete from search-order */   
            int i;
            for (i=0; i < ORDER_LEN; i++) 
            {
                if (CONTEXT[i] == VOC_LINK) 
                {
                    CONTEXT[i] = NULL;
                    if (! PFE.atexit_running)
                    {
                        const p4char* nfa = VOC_LINK->nfa ? VOC_LINK->nfa 
                            : (const p4char*) "\1?";
                        P4_note3 ("deleted '%.*s' "
                                  "from context search-order [%i]", 
                                  NFACNT(*nfa), nfa+1, i);
                    }
                }
            
                if (PFE.dforder[i] == VOC_LINK) 
                {
                    PFE.dforder[i] = NULL;
                    if (! PFE.atexit_running)
                    {
                        const p4char* nfa = VOC_LINK->nfa ? VOC_LINK->nfa 
                            : (const p4char*) "\1?";
                        P4_note3 ("deleted '%.*s' "
                                  "from default search-order [%i]", 
                                  NFACNT(*nfa), nfa+1, i);
                    }
                }
            }
        }
        
        VOC_LINK = VOC_LINK->prev;
    }
    
    /* compact search-order */
    { register int i, j;
      for (i=0, j=0; i < ORDER_LEN; i++)
      {
        if (CONTEXT[i]) CONTEXT[j++] = CONTEXT[i];
      }
      while (j < ORDER_LEN) CONTEXT[j++] = NULL;

      for (i=0, j=0; i < ORDER_LEN; i++)
      {
        if (PFE.dforder[i]) PFE.dforder[j++] = PFE.dforder[i];
      }
      while (j < ORDER_LEN) PFE.dforder[j++] = NULL;
    }
    
    /* free dictionary space: */
    DP = (p4char *) new_dp; 
    LAST = NULL;
    PFE.forget_dp = 0;

    if (CURRENT >= (p4_Wordl *) new_dp) 
    {
        if (CONTEXT[0]) CURRENT = PFE.forth_wl; /* initial CURRENT */
        if (! PFE.atexit_running)
            p4_throw (P4_ON_CURRENT_DELETED);  /* and still throw */
    }
}

/** (FORGET)
 * forget anything above address
 */
_export void
p4_forget (char* above)
{
    if ((p4char*) above < FENCE)
        p4_throw (P4_ON_INVALID_FORGET);

    if (PFE.forget_dp) /* some p4_forget_dp already started */
    {
        /* P4_info1 ("recursive forget %p", above); */
        if (PFE.forget_dp > above) 
        {
            PFE.forget_dp = above; /* update p4_forget_dp argument */
        }
    }else{ 
        /* P4_info1 ("forget start %p", above); */
        PFE.forget_dp = above; /* put new argument for p4_forget_dp */
        FX (p4_forget_dp);     /* forget execution start */
    }
}

FCode (p4_destroyer_RT)
{
    /* put the argument for the destroyer-call in PFA[0] and
     * the call-address in PFA[1]. And here we call it.
     */
    p4_call ((p4xt) &(WP_PFA [1]));
}

/**
 * create a destroyer word. Upon =>'FORGET' the code will will be
 * run with the given argument. The structure is code/CFA and what/PFA.
 */
_export char*
p4_forget_word (const char *name, p4cell id, p4code ccode, p4cell what)
{
    char* nfa;
    char nm[255];

    sprintf (nm, name, id);

#  if defined PFE_WITH_FFA  
    nfa = p4_make_header (ccode, 0, nm, strlen(nm), PFE.atexit_wl);
    *_FFA(nfa) |= (P4xIMMEDIATE|P4xONxDESTROY);
    FX_VCOMMA (what);  /*pfa*/
#  else
    p4_make_header (PFX(p4_destroyer_RT, 0, 
                        nm, strlen(nm), PFE.atexit_wl)
    *_FFA(nfa) |= P4xIMMEDIATE;
    FX_VCOMMA (what); /*pfa*/
    FX_RCOMMA (ccode); /*pfa+1*/
#endif    
    
    return nfa;
}

/* ------------------------------ 
 * search a header 
 */

static char *
search_thread (const char *nm, int l, char *t, p4cell wl_flag )
{
    if (l > NFACNTMAX)
        return NULL;

# if P4_LOG /* additional sanity check */
    if (p4_LogMask & P4_LOG_DEBUG) /* if any debug level */
        if (t && !((char*)PFE.dict <= t && t <= (char*)PFE.dictlimit)) 
        { 
            P4_fail3 ("hashlink pointer invalid %p in search for '%.*s'", 
              t, l, nm);
        }
# endif

    if( LOWER_CASE && (wl_flag & WORDL_NOCASE) )
    {
        /* this thread does only contain upper-case defs 
           AND lower-case input shall match definitions */
        while (t)
        {
            if ( !(*_FFA(t) & P4xSMUDGED) 
              && NFACNT(*t) == l && !p4_strncmpi (nm, t+1, l))
                break;
            t = *p4_name_to_link (t);
        }
    }else{
        /* input is case-sensitive OR vocabulary contains mixed-case defs */
        while (t)
        {
            if ( !(*_FFA(t) & P4xSMUDGED) 
              && NFACNT(*t) == l && !strncmp (nm, t+1, l))
                break;
            t = *p4_name_to_link (t);
        }
    }

    return t;
}

_export char *
p4_search_wordlist (const char *nm, int l, const p4_Wordl *w)
{
    if( w->flag & WORDL_NOHASH )
    { return search_thread (nm, l, w->thread[0], w->flag ); }
    else
    { return search_thread (nm, l, w->thread[p4_wl_hash (nm, l)], w->flag ); }
}

/* search all word lists in the search order for name, return NFA 
 * (we use the id speedup here - the first WLs have each a unique bitmask
 *  in the wl->id. Especially the FORTH wordlist can be present multiple
 *  time - even in being just search via wl->also. With w->id each is just
 *  searched once - atleast for each of the WLs that have gotten an id-bit
 *  which on a 32bit system are 32 WLs - enough for many system setups.
 *  It might be possible to use the old code even here (that walked the
 *  ORDER to see if the next WL is present in an earlier slot) but in a
 *  system with so many vocs it is quite improbable to find duplicates
 *  other than the basic vocs like FORTH in there anyway - so we use this
 *  one that might search a WL twice theoretically. Tell me of occasions
 *  where that is really a problem - in my setups it happens that the ORDER
 *  overflows much before getting duplicates other than the basic wordlists.
 */
_export char *
p4_find (const char *nm, int l)
{
    register Wordl **p;
    register Wordl *wordl;
    register char *w = NULL;
    register int n = p4_wl_hash (nm, l);
    register p4ucell searched = 0;
    
    for (p = CONTEXT; p <= &ONLY; p++)
    {
        for (wordl = *p; wordl ; wordl=wordl->also)
	{
	    if (searched&wordl->id)
		continue;
	    searched |= wordl->id;

            if( wordl->flag & WORDL_NOHASH )
                w = search_thread (nm, l, wordl->thread[0], wordl->flag );
            else
                w = search_thread (nm, l, wordl->thread[n], wordl->flag );

	    if (w) return w;
        }
    }
    return w; /*0*/
}

/**
 * tick next word,  and
 * return count byte pointer of name field (to detect immediacy)
 */
_export char *
p4_tick_nfa (void) 
{
    register char *p;

    p4_word_parseword (' '); *DP=0; /* PARSE-WORD-NOHERE */
    p = p4_find (PFE.word.ptr, PFE.word.len);
    if (! p)
        p4_throw (P4_ON_UNDEFINED);
    return p;
}

/**
 * tick next word,  and return xt
 */
_export p4xt
p4_tick_cfa (void)
{
    return p4_name_from (p4_tick_nfa ());
}

#if 0
/**
 * tick next word, store p4xt in xt, and
 * return count byte pointer of name field (to detect immediacy)
 */
___export char *
p4_tick (p4xt *xt)
{
    register char *p = p4_tick_nfa ();
    *xt = p4_name_from (p);
    return p;
}
#endif

/* ---------------------------
 * create a header 
 */

/* writes counted string into dictionary, returns address */
_export char *
p4_string_comma (const char *s, int len)
{
    char *p = (char *) DP;
    
    if (len >= (1 << CHAR_BIT))
        p4_throw (P4_ON_ARG_TYPE);
    *DP++ = len;                /* store count byte */
    while (--len >= 0)          /* store string */
        *DP++ = (p4char) *s++;
    FX (p4_align);
    return p;
}

#if 0
_export char*
/*use FX(p4_parse_comma_quote)*/   p4_parse_comma(char del)
{
    p4_word_parse (del); /* PARSE-WORD-NOTHROW */
    return p4_string_comma (PFE.word.ptr, (int) PFE.word.len);
}
#endif

static void                     /* written to cfa following make_header() */
illegal_xt (void)               /* to give an error msg when calling */
{                               /* a word without execution semantics */
    p4_throw (P4_ON_INVALID_NAME);
}

/* ----------------------
 * words with wildcards 
*/

/*
 * Show words in word list matching pattern, and of one of the
 * categories in string `categories'. NULL pointer or zero length
 * string means all kinds of words.
 */
_export void
p4_wild_words (const p4_Wordl *wl, const char *pattern, const char *categories)
{
    Wordl wcopy = *wl;          /* clobbered while following it */
    p4char **t;

# ifndef WILD_TAB
# define WILD_TAB 26 /* traditional would be 20 (26*4=80), now 26*3=78 */
# endif

    FX (p4_cr);
    FX (p4_start_Q_cr);
    if (categories && *categories == '\0')
        categories = NULL;
    for (t = p4_topmost (&wcopy); *t; t = p4_topmost (&wcopy))
    {
        char wbuf[NFACNTMAX+1];
        p4char *w = *t;
        p4char **s = p4_name_to_link (w);
        int l = NFACNT(*w++);
        
        p4_store_c_string (w, l, wbuf, sizeof wbuf);
        if (p4_match (pattern, wbuf, wl->flag & WORDL_NOCASE))
        {
            char c = p4_category (*P4_LINK_FROM (s));
            if (! categories || strchr (categories, c))
            {
                if (p4_OUT+WILD_TAB - p4_OUT%WILD_TAB + 2 + l > p4_COLS ||
                    p4_OUT+WILD_TAB - p4_OUT%WILD_TAB + WILD_TAB*2/3 > p4_COLS)
                {
                    if (p4_Q_cr ())
                        break;
                }else{
                    if (p4_OUT)
                        p4_tab (WILD_TAB);
                }
                p4_outf ("%c %.*s ", c, l, w);
            }
        }
        *t = *s;
    }
}

/* completion of word against dictionary */
static p4char *
search_incomplete (const char *name, int len, Wordl *w)
/*
 * traverses the entire given wordlist to find a matching word
 * caution: clobbers *w. This is needed to be able to continue the search.
 */
{
    p4char **t, *s;
    
    for (t = p4_topmost (w); *t; t = p4_topmost (w))
    {
        s = *t;
        *t = *p4_name_to_link (*t);
        if (NFACNT(*s) >= len 
          && ( (w->flag & WORDL_NOCASE) 
            ? p4_strncmpi (s + 1, name, len) 
            : strncmp  (s + 1, name, len) 
               ) == 0)
            return s;
    }
    return NULL;
}

/*
 * Try to complete string in/len from dictionary.
 * Store completion in out (asciiz), return number of possible completions.
 * If display is true, display alternatives.
 * (if (display && !len) { don't print 200 words, just the number })
 */
static int
p4_complete_word (const char *in, int len, char *out, int display)
{
    Wordl w, *wl, **p;
    char *s = NULL, *t = NULL;  
    int n = 0, m = 0, cnt = 0, searched_n = 0;
    Wordl* searched[32] = {0}; 
    
    for (p = CONTEXT; p <= &ONLY; p++)
    {
        for (wl = *p; wl; wl = wl->also ) 
        {
            for (n=0; n < searched_n; n++)
                if (wl == searched[n])
                    break;       /* continue at second outer for */
            if (wl == searched[n]) 
                continue;        /* must expressed like that in C*/
        
            if (searched_n < 32) 
                searched[searched_n++] = wl;
             
            for (w = *wl; (t = search_incomplete (in, len, &w)) != NULL; cnt++)
            {
                if (display && len) 
                {
                    FX (p4_space);
                    p4_type_on_line (t + 1, NFACNT(*t));
                }
                if (cnt == 0) 
                {
                    s = t + 1;
                    m = NFACNT(*t);
                }else{
                    ++t;
                    for (n = 0; n < m; n++)
                        if (s[n] != t[n])
                            break;
                    m = n;
                }
            }
        }
    }
    if (cnt)
        p4_store_c_string (s, m, out, NFACNTMAX+1);
    if (display && !len)
    { p4_outf (" %i words ", cnt); }
    return cnt;
}

_export int
p4_complete_dictionary (char *in, char *out, int display)
{
    char *lw, buf[NFACNTMAX+1];
    int n;
    
    lw = strrchr (in, ' ');
    if (lw)
        lw++;
    else
        lw = in;
    memcpy (out, in, lw - in);
    n = p4_complete_word (lw, strlen (lw), buf, display);
    strcpy (&out[lw - in], buf);
    return n;
}

/* ---------------------------------------------------------------------- *
 * initial dictionary setup                                             
 */

FCode (p4_forget_wordset_RT)
{
    /* do nothing so far, forget_wordset_RT_ is just a type-marker */
}

extern int p4_slot_use (int*); /* FIXME: move to header file ? */
extern int p4_slot_unuse (int*); /* FIXME: move to header file ? */

static FCode (p4_forget_slot_RT)
{
    int* slot = (int*)(WP_PFA[0]);
    P4_info1 ("unuse load-slot '%i'", *slot);
   
    if (slot && *slot && PFE.p[*slot]) 
    {
        p4_xfree (PFE.p[*slot]); PFE.p[*slot] = 0;
    }
   
    p4_slot_unuse (slot);
}

static void
p4_load_slot_open (int* slot)
{
    int e;
    if (!slot) return;
                
    if ((e=p4_slot_use (slot))) 
    {
        P4_fail2 ("load-slot %i failed : %s", *slot, strerror(-e));
        return; 
    }
}

static void
p4_load_slot_init (int* slot, p4ucell size)
{
    if (!slot || !*slot || size < 4)
        return;

    if (!(PFE.p)[*slot]) 
    {
        (PFE.p)[*slot] = p4_calloc (1, size);
        P4_info3 ("load-slot %i size %lu alloc (%p)", 
                  *slot, (unsigned long)size, (PFE.p)[*slot]);
    }else{ 
        P4_warn2 ("load-slot %i already allocated (%p)", 
                  *slot, (PFE.p)[*slot]);
    }
    
    p4_forget_word ("(load-slot: %i)", *slot, 
                    p4_forget_slot_RT_, (p4cell) slot);
}

static void
p4_load_into (const char* vocname)
{
    Wordl* voc;
    if (! vocname) return;

    voc = p4_find_wordlist (vocname, strlen(vocname));
    if (voc) 
    {
        {    
            register int i;
            for (i=ORDER_LEN; --i > 0; )
                if (CONTEXT[i] == voc) 
                {
                    P4_info1 ("search also '%s' : already there", 
                              vocname);
                    return;
                }
        }
        FX (p4_also);    /* the top-of-order (CONTEXT) isn't changed */
        CONTEXT [1] = voc; /* instead we place it under-the-top */
        P4_info1 ("search also '%s' : done", vocname);
    }else{
        P4_warn2 ("search also failed: no '%s' vocabulary (%lu)", 
                  vocname, (unsigned long) strlen(vocname));
    }
} 

static void p4_exception_string (const char* name, p4cell id)
{
    /* FIXME: instead of compiling to the forth-dict we should better
       create a way to let functions search the loaded wordset tables
    */
    p4_Exception* expt = (void*) DP; DP += sizeof(*expt);
    if (id < PFE.next_exception) PFE.next_exception = id - 1;
    expt->next = PFE.exception_link; PFE.exception_link = expt;
    expt->name = name; expt->id = id;
}

_export void
p4_load_words (const p4Words* ws, p4_Wordl* wid, int unused)
{
    Wordl* save_current = CURRENT;
    int k = ws->n;
    const p4Word* w = ws->w;
    char dictname[NFACNTMAX+1]; char* dn;
    int* slot = 0;

    if (!wid) wid = CURRENT;
    
    if (ws->name) 
    {  
        P4_info1 ("load '%s'", (ws->name));
        strncpy (dictname, ws->name, NFACNTMAX);
        dictname[NFACNTMAX] = '\0';
        if ((dn= strchr (dictname, ' '))
            ||  (dn= strchr (dictname, '(')))
            *dn = '\0';
    }else{
        sprintf (dictname, "%p", DP);
    }
    
    p4_forget_word ("wordset:%s", (p4cell) dictname,
                    p4_forget_wordset_RT_, 
                    (p4cell) (ws));
    
    for ( ; --k >= 0; w++)
    {
        wid = CURRENT;
        if (w)
        {
            /* the C-name is really type-byte + count-byte away */
            char type = *w->name;
            const char* name = w->name+2;
            int len = strlen (w->name+2);
            void* ptr = w->ptr;
            Wordl* wid = CURRENT;

            /* and paste over make_word inherited from pre 0.30.28 times */
            p4xt  cfa;
            char *nfa;

            /* part 1: specials... */
  
            switch (type)
            {
            case p4_LOAD:
                if (ptr)
                    p4_load_words ((p4Words*) ptr, 0, 0); /* RECURSION !! */
                continue;
            case p4_INTO:
            {
                register void* p;
                p = p4_find_wordlist (name, strlen (name));
                if (p) 
                {   
                    P4_debug1 (13, "load into old '%s'", name);
                    CURRENT = p;
                }else{
                    Wordl* current = 0;
                    if (ptr) {
                        current = p4_find_wordlist (ptr, strlen(ptr));
                        if (! current) 
                            P4_warn1 ("could not find also-voc %s", 
				      (char*)(ptr));
                    }
                    if (! current) current = CURRENT;
                    P4_info1 ("load into new '%s'", name);
                    CURRENT = p4_make_wordlist (p4_make_header (
                        PFX(p4_vocabulary_RT), P4xIMMEDIATE,
                        name, strlen(name), current) );
                }

                if (ptr) 
                {
                    if (! CURRENT->also)
                        CURRENT->also = p4_find_wordlist (ptr, strlen(ptr));

                    p4_load_into (name); /* search-also */
                }
            } continue;
            case p4_SLOT:
                slot = (int*) ptr;
                p4_load_slot_open (slot);
                continue;
            case p4_SSIZ:
                p4_load_slot_init (slot, (p4ucell)(ptr));
                continue;
            case p4_EXPT:
                p4_exception_string(name, (p4cell)(ptr));
                continue;
            case p4_XXCO: /* constructors are registered in => LOADED */
                wid = PFE.atexit_wl;
                break;
            } /*switch*/

            /* part 2: general... CREATE a name and setup its CFA field */

            nfa = p4_make_header (illegal_xt, 0, name, len, wid);
            if ('A' <= type && type <= 'Z')
                *_FFA(nfa) |= P4xIMMEDIATE;
            cfa = P4_BODY_FROM(DP);
            switch (type)
            {
            case p4_SXCO:
#             ifndef HOST_WIN32
                *cfa = ((p4_Semant *) ptr) ->comp;
                if (! ((p4_Semant *)ptr) ->name)
                    (char*) ((p4_Semant *)ptr) ->name = name-1; 
                /* discard const */
                /* BEWARE: the arg' name must come from a wordset entry to
                   be both static and have byte in front that could be 
                   a maxlen
                */
#             else
		/* on WIN32, the ptr is a function that returns a SemantP */
		*cfa = ((p4_Semant*(*)())ptr) () -> comp;
#             endif
                continue;
            case p4_RTCO:
                *cfa = ((p4_Runtime2 *) ptr) ->comp;
		/* and start registering the runtimes centrally FIXME:
		   FX_COMMA(PFE.runtime); PFE.runtime = p4_HERE;
		   FX_COMMA(ptr);
		   but that sys-link should be honoured in p4_forget too
		*/
                continue;
            case p4_IXCO:
            case p4_FXCO:
                *cfa = (p4code) ptr;
                continue;
            case p4_XXCO:
                *cfa = (p4code) ptr;
                ((p4code)ptr) ();     /* runs *now* !! no checks !! */
                continue;
            case p4_IVOC:
            case p4_OVOC:
                /* creating a VO before IN will make sure that the */
                /* other words will go in there. Nice stuff, eh ;-) */
                *cfa = p4_vocabulary_RT_ ;
                /* (((WList*) ptr)->wid = p4_make_wordlist (nfa)); */
                continue;
            case p4_DVAR:
                *cfa = p4_dictvar_RT_ ;
                break;
            case p4_DCON:
                *cfa = p4_dictget_RT_ ;
                break;
            case p4_DSET:
                *cfa = p4_dictset_RT_ ;
                break;
            case p4_OVAR:
            case p4_IVAR:
                *cfa = p4_var_RT_ ;
                break;
            case p4_OVAL:
            case p4_IVAL:
                *cfa = p4_value_RT_ ;
                break;
            case p4_OFFS:
                *cfa = p4_offset_RT_ ;
                break;
	    case p4_iOLD:
	    case p4_xOLD:
		*cfa = p4_obsoleted_RT_;
		if (p4_LogMask && p4_LogMask^P4_LOG_FATAL) goto synonym;
            case p4_SNYM:
            case p4_FNYM:
                *cfa = p4_synonym_RT_ ;
	    synonym:
		ptr = p4_find (ptr, strlen(ptr));
		if (ptr) ptr = p4_name_from (ptr);
		else P4_fail3 ("could not resolve SYNONYM %.*s %s",
			       NFACNT(*nfa), nfa+1, (char*)w->ptr);
		break;
            default:
                P4_fail3 ("unknown typecode for loadlist entry: "
                          "0x%x -> \"%.*s\"", 
                          type, len, name);
            case p4_OCON:
            case p4_ICON:
                *cfa = p4_constant_RT_ ;
                break;
            }
            FX_VCOMMA (ptr);
            continue;
        } /* if(w) */
    } /* for w in ws->w */

    CURRENT = save_current; /* should save_current moved to the caller? */
}

/* ------------------------------------------------------------------- */

/** ((ONLY)) ( -- )
 * the only-vocabulary is special. Calling it will erase
 * the search => ORDER of vocabularies and only allows
 * to name some very basic vocabularies. Even => ALSO
 * is not available.
 example:
   ONLY FORTH ALSO EXTENSIONS ALSO DEFINITIONS
 */
FCode (p4_only_RT)
{
    ZERO (CONTEXT);
    CONTEXT[0] = CURRENT = ONLY;
}

/** FORTH ( -- )
 : FORTH FORTH-WORDLIST CONTEXT ! ;
 */
FCode (p4_forth_RT)
{
    CONTEXT[0] = PFE.forth_wl;
}

_export void
p4_preload_only (void)
{
    Wordl only;                   /* scratch ONLY word list */
    char* nfa;
    
    DP = (p4char *) &PFE.dict[1];
  
    /* Load the ONLY word list to the scratch ONLY: */
    memset (&only, 0, sizeof only);
    /* # only.flag |= WORDL_NOHASH; */
    nfa = p4_make_header (p4_only_RT_, 0, "ONLY", 4, &only );
    ONLY = p4_make_wordlist (nfa);
    /* # ONLY->flag |= WORDL_NOHASH; */
    COPY (ONLY->thread, only.thread);   /* Copy scratch ONLY to real ONLY */
    CURRENT = ONLY;

    /* FORTH -> [ANS] -> ONLY */
    nfa = p4_make_header (p4_forth_RT_ , 0, "FORTH", 5, ONLY);
    PFE.forth_wl = p4_make_wordlist (nfa); 
    nfa = p4_make_header (p4_vocabulary_RT_, P4xIMMEDIATE, "[ANS]", 5, ONLY);
    PFE.forth_wl->also = p4_make_wordlist (nfa);
    PFE.forth_wl->also->also = ONLY;

    /* destroyers :: LOADED */
    nfa = p4_make_header (p4_vocabulary_RT_ , 0, "LOADED", 6, ONLY);
    PFE.atexit_wl = p4_make_wordlist (nfa); 
    PFE.atexit_wl->flag |= WORDL_NOHASH; /* see environment_dump in core.c */

    /* ENVIRONMENT -> LOADED */
    nfa = p4_make_header (p4_vocabulary_RT_ , P4xIMMEDIATE, 
                          "ENVIRONMENT", 11, ONLY);
    PFE.environ_wl = p4_make_wordlist (nfa);
    PFE.environ_wl->also = PFE.atexit_wl;
    PFE.environ_wl->flag |= WORDL_NOHASH;          /* for option-ext */
    PFE.environ_wl->thread[0] = PFE.set->opt.link; /* that goes here */
}

/*@}*/

/*
 * Local variables:
 * c-file-style: "stroustrup"
 * End:
 */











