/* Copyright (c) 1997-1999 Miller Puckette.
* For information on usage and redistribution, and for a DISCLAIMER OF ALL
* WARRANTIES, see the file, "LICENSE.txt," in this distribution.  */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "m_pd.h"
#include "g_canvas.h"

/*
This file contains text objects you would put in a canvas to define a
template.  Templates describe objects of type "array" (g_array.c) and
"scalar" (g_scalar.c).
*/


/* -- templates, the active ingredient in gtemplates defined below. ------- */

t_template *template_new(int argc, t_atom *argv)
{
    t_template *x = (t_template *)t_getbytes(sizeof(*x));
    x->t_n = 0;
    x->t_vec = (t_dataslot *)t_getbytes(0);
    while (argc > 0)
    {
    	int newtype, oldn, newn;
	t_symbol *newname, *newarraytemplate = &s_, *newtypesym;
    	if (argc < 2 || argv[0].a_type != A_SYMBOL ||
	    argv[1].a_type != A_SYMBOL)
	    	goto bad;
    	newtypesym = argv[0].a_w.w_symbol;
	newname = argv[1].a_w.w_symbol;
	if (newtypesym == &s_float)
	    newtype = DT_FLOAT;
	else if (newtypesym == &s_symbol)
	    newtype = DT_SYMBOL;
	else if (newtypesym == &s_list)
    	    newtype = DT_LIST;
	else if (newtypesym == gensym("array"))
	{
	    if (argc < 3 || argv[2].a_type != A_SYMBOL)
	    {
	    	pd_error(x, "array lacks element template or name");
		goto bad;
	    }
	    newarraytemplate = canvas_makebindsym(argv[2].a_w.w_symbol);
	    newtype = DT_ARRAY;
	    argc--;
	    argv++;
	}
	else
	{
    	    pd_error(x, "%s: no such type", newtypesym->s_name);
    	    return (0);
	}
	newn = (oldn = x->t_n) + 1;
	x->t_vec = (t_dataslot *)t_resizebytes(x->t_vec,
	    oldn * sizeof(*x->t_vec), newn * sizeof(*x->t_vec));
	x->t_n = newn;
	x->t_vec[oldn].ds_type = newtype;
	x->t_vec[oldn].ds_name = newname;
	x->t_vec[oldn].ds_arraytemplate = newarraytemplate;
    bad: 
    	argc -= 2; argv += 2;
    }
    return (x);
}

    /* stringent check to see if a "saved" template matches the current one */
int template_match(t_template *x1, t_template *x2)
{
    int i;
    if (x1->t_n != x2->t_n)
    	return (0);
    for (i = 0; i < x1->t_n; i++)
    {
    	if (x1->t_vec[i].ds_name != x2->t_vec[i].ds_name ||
	    x1->t_vec[i].ds_type != x2->t_vec[i].ds_type ||
	    	(x1->t_vec[i].ds_type == DT_ARRAY &&
		    x1->t_vec[i].ds_arraytemplate !=
		    	x2->t_vec[i].ds_arraytemplate))
		    	    return (0);
    }
    return (1);
}

void template_free(t_template *x)
{
    	/* LATER figure out how to get rid of data having this template??? */
    t_freebytes(x->t_vec, x->t_n * sizeof(*x->t_vec));
    t_freebytes(x,  sizeof(*x));
}

/* ---------------- gtemplates.  One per canvas. ----------- */

t_class *gtemplate_class;

struct _gtemplate
{
    t_object x_obj;
    t_template *x_template;
    t_canvas *x_owner;
};

static void *gtemplate_new(t_symbol *s, int argc, t_atom *argv)
{
    t_gtemplate *x = (t_gtemplate *)pd_new(gtemplate_class);
    	/* LATER deal with typing in a "template" box recreating it */
    x->x_owner = canvas_getcurrent();
    x->x_template = template_new(argc, argv);
    return (x);
}

t_template *gtemplate_get(t_gtemplate *x)
{
    return (x->x_template);
}

static void gtemplate_free(t_gtemplate *x)
{
    template_free(x->x_template);
}

static void gtemplate_setup(void)
{
    gtemplate_class = class_new(gensym("template"),
    	(t_newmethod)gtemplate_new, (t_method)gtemplate_free,
    	sizeof(t_gtemplate), CLASS_NOINLET, A_GIMME, 0);
}

/* ---------------  FIELD DESCRIPTORS ---------------------- */

/* a field descriptor can hold a constant or a variable; if a variable,
it's the name of a field in the template we belong to.  LATER, we might
want to cache the offset of the field so we don't have to search for it
every single time we draw the object.
*/

typedef struct _fielddesc
{
    char fd_type;   	/* LATER consider removing this? */
    char fd_var;
    union
    {
    	t_float fd_float;   	/* the field is a constant float */
    	t_symbol *fd_symbol;	/* the field is a constant symbol */
    	t_symbol *fd_varsym;	/* the field is variable and this is the name */
    } fd_un;
} t_fielddesc;

#define FIELDDESC_SETFLOAT(x, f) \
    ((x)->fd_type = A_FLOAT, (x)->fd_var = 0, (x)->fd_un.fd_float = (f))
#define FIELDDESC_SETSYMBOL(x, s) \
    ((x)->fd_type = A_SYMBOL, (x)->fd_var = 0, (x)->fd_un.fd_symbol = (s))
#define FIELDDESC_SETVAR(x, s, type) \
    ((x)->fd_type = type, (x)->fd_var = 1, (x)->fd_un.fd_varsym = (s))

#define CLOSED 1
#define BEZ 2
#define A_ARRAY 55  	/* LATER decide whether to enshrine this in m_pd.h */

static void fielddesc_setfloatarg(t_fielddesc *fd, int argc, t_atom *argv)
{
    	if (argc <= 0) FIELDDESC_SETFLOAT(fd, 0);
    	else if (argv->a_type == A_SYMBOL)
    	    FIELDDESC_SETVAR(fd, argv->a_w.w_symbol, A_FLOAT);
    	else FIELDDESC_SETFLOAT(fd, argv->a_w.w_float);
}

static void fielddesc_setarrayarg(t_fielddesc *fd, int argc, t_atom *argv)
{
    	if (argc <= 0) FIELDDESC_SETFLOAT(fd, 0);
    	else if (argv->a_type == A_SYMBOL)
    	    FIELDDESC_SETVAR(fd, argv->a_w.w_symbol, A_ARRAY);
    	else FIELDDESC_SETFLOAT(fd, argv->a_w.w_float);
}

static t_float fielddesc_getfloat(t_fielddesc *f, t_canvas *template,
    t_word *wp, int loud)
{
    if (f->fd_type == A_FLOAT)
    {
    	if (f->fd_var)
   	    return (canvas_getfloat(template, f->fd_un.fd_varsym, wp, loud));
    	else return (f->fd_un.fd_float);
    }
    else
    {
    	if (loud)
	    error("symbolic data field used as number");
    	return (0);
    }
}

static t_symbol *fielddesc_getsymbol(t_fielddesc *f, t_canvas *template,
    t_word *wp, int loud)
{
    if (f->fd_type == A_SYMBOL)
    {
    	if (f->fd_var)
   	    return(canvas_getsymbol(template, f->fd_un.fd_varsym, wp, loud));
    	else return (f->fd_un.fd_symbol);
    }
    else
    {
    	if (loud)
	    error("numeric data field used as symbol");
    	return (&s_);
    }
}

/* ---------------- curves and polygons (joined segments) ---------------- */

/*
curves belong to templates and describe how the data in the template are to
be drawn.  The coordinates of the curve (and other display features) can
be attached to fields in the template.
*/

t_class *curve_class;

typedef struct _curve
{
    t_object x_obj;
    int x_flags;     	    /* CLOSED and/or BEZ */
    t_fielddesc x_fillcolor;
    t_fielddesc x_outlinecolor;
    t_fielddesc x_width;
    int x_npoints;
    t_fielddesc *x_vec;
} t_curve;

static void *curve_new(t_symbol *classsym, t_int argc, t_atom *argv)
{
    t_curve *x = (t_curve *)pd_new(curve_class);
    char *classname = classsym->s_name;
    int flags = 0;
    int nxy, i;
    t_fielddesc *fd;
    if (classname[0] == 'f')
    {
    	classname += 6;
    	flags |= CLOSED;
    	if (argc) fielddesc_setfloatarg(&x->x_fillcolor, argc--, argv++);
    	else FIELDDESC_SETFLOAT(&x->x_outlinecolor, 0); 
    }
    else classname += 4;
    if (classname[0] == 'c') flags |= BEZ;
    x->x_flags = flags;
    if (argc) fielddesc_setfloatarg(&x->x_outlinecolor, argc--, argv++);
    else FIELDDESC_SETFLOAT(&x->x_outlinecolor, 0);
    if (argc) fielddesc_setfloatarg(&x->x_width, argc--, argv++);
    else FIELDDESC_SETFLOAT(&x->x_width, 1);
    if (argc < 0) argc = 0;
    nxy =  (argc + (argc & 1));
    x->x_npoints = (nxy>>1);
    x->x_vec = (t_fielddesc *)t_getbytes(nxy * sizeof(t_fielddesc));
    for (i = 0, fd = x->x_vec; i < argc; i++, fd++, argv++)
    	fielddesc_setfloatarg(fd, 1, argv);
    if (argc & 1) FIELDDESC_SETFLOAT(fd, 0);

    return (x);
}

/* -------------------- widget behavior for curve ------------ */

static void curve_getrect(t_gobj *z, t_glist *glist,
    t_word *data, t_canvas *template, float basex, float basey,
    int *xp1, int *yp1, int *xp2, int *yp2)
{
    t_curve *x = (t_curve *)z;
    int i, n = x->x_npoints;
    t_fielddesc *f = x->x_vec;
    int x1 = 0x7fffffff, x2 = -0x7fffffff, y1 = 0x7fffffff, y2 = -0x7fffffff;
    for (i = 0, f = x->x_vec; i < n; i++, f += 2)
    {
    	int xloc = glist_xtopixels(glist,
    	    basex + fielddesc_getfloat(f, template, data, 0));
    	int yloc = glist_ytopixels(glist,
    	    basey + fielddesc_getfloat(f+1, template, data, 0));
    	if (xloc < x1) x1 = xloc;
    	if (xloc > x2) x2 = xloc;
    	if (yloc < y1) y1 = yloc;
    	if (yloc > y2) y2 = yloc;
    }
    *xp1 = x1;
    *yp1 = y1;
    *xp2 = x2;
    *yp2 = y2; 
}

static void curve_displace(t_gobj *z, t_glist *glist,
    t_word *data, t_canvas *template, float basex, float basey,
    int dx, int dy)
{
    /* refuse */
}

static void curve_select(t_gobj *z, t_glist *glist,
    t_word *data, t_canvas *template, float basex, float basey,
    int state)
{
    /* fill in later */
}

static void curve_activate(t_gobj *z, t_glist *glist,
    t_word *data, t_canvas *template, float basex, float basey,
    int state)
{
    /* fill in later */
}

static int rangecolor(int n)	/* 0 to 9 in 5 steps */
{
    int n2 = n/2;   	    	/* 0 to 4 */
    int ret = (n << 6);     	/* 0 to 256 in 5 steps */
    if (ret > 255) ret = 255;
    return (ret);
}

static void numbertocolor(int n, char *s)
{
    int red, blue, green;
    if (n < 0) n = 0;
    red = n / 100;
    blue = ((n / 10) % 10);
    green = n % 10;
    sprintf(s, "#%2.2x%2.2x%2.2x", rangecolor(red), rangecolor(blue),
    	rangecolor(green));
}

static void curve_vis(t_gobj *z, t_glist *glist, 
    t_word *data, t_canvas *template, float basex, float basey,
    int vis)
{
    t_curve *x = (t_curve *)z;
    int i, n = x->x_npoints;
    t_fielddesc *f = x->x_vec;
    
    if (vis)
    {
    	if (n > 1)
    	{
    	    int flags = x->x_flags, closed = (flags & CLOSED);
    	    float width = fielddesc_getfloat(&x->x_width, template, data, 1);
    	    char outline[20], fill[20];
    	    if (width < 1) width = 1;
    	    numbertocolor(
    	    	fielddesc_getfloat(&x->x_outlinecolor, template, data, 1),
    	    	outline);
    	    if (flags & CLOSED)
    	    {
    	    	numbertocolor(
    	    	    fielddesc_getfloat(&x->x_fillcolor, template, data, 1),
    	    	    fill);
    	    	sys_vgui(".x%x.c create polygon\\\n",
    	    	    glist_getcanvas(glist));
    	    }
    	    else sys_vgui(".x%x.c create line\\\n",
    	    	    glist_getcanvas(glist));
    	    for (i = 0, f = x->x_vec; i < n; i++, f += 2)
    	    {
    		float xloc = glist_xtopixels(glist,
    	    	    basex + fielddesc_getfloat(f, template, data, 1));
    		float yloc = glist_ytopixels(glist,
    	    	    basey + fielddesc_getfloat(f+1, template, data, 1));
    		sys_vgui("%d %d\\\n", (int)xloc, (int)yloc);
    	    }
    	    sys_vgui("-width %f\\\n",
    	    	fielddesc_getfloat(&x->x_width, template, data, 1));
    	    if (flags & CLOSED) sys_vgui("-fill %s -outline %s\\\n",
    	    	fill, outline);
    	    else sys_vgui("-fill %s\\\n", outline);
    	    if (flags & BEZ) sys_vgui("-smooth 1\\\n");
    	    sys_vgui("-tags curve%x\n", data);
    	}
    	else post("warning: curves need at least two points to be graphed");
    }
    else
    {
    	if (n > 1) sys_vgui(".x%x.c delete curve%x\n",
    	    glist_getcanvas(glist), data);    	
    }
}

static int curve_motion_field;
static float curve_motion_xcumulative;
static float curve_motion_xbase;
static float curve_motion_xper;
static float curve_motion_ycumulative;
static float curve_motion_ybase;
static float curve_motion_yper;
static t_glist *curve_motion_glist;
static t_gobj *curve_motion_gobj;
static t_word *curve_motion_wp;
static t_canvas *curve_motion_template;

    /* LATER protect against the template changing or the scalar disappearing
    probably by attaching a gpointer here ... */

static void curve_motion(void *z, t_floatarg dx, t_floatarg dy)
{
    t_curve *x = (t_curve *)z;
    t_fielddesc *f = x->x_vec + curve_motion_field;
    curve_motion_xcumulative += dx;
    curve_motion_ycumulative += dy;
    if (f->fd_var)
    {
	canvas_setfloat(curve_motion_template,
	    f->fd_un.fd_varsym,
	    curve_motion_wp, 
    	    curve_motion_xbase + curve_motion_xcumulative * curve_motion_xper,
	    	1);
    }
    if ((f+1)->fd_var)
    {
	canvas_setfloat(curve_motion_template,
	    (f+1)->fd_un.fd_varsym,
	    curve_motion_wp, 
    	    curve_motion_ybase + curve_motion_ycumulative * curve_motion_yper,
	    	1);
    }
    glist_redrawitem(curve_motion_glist, curve_motion_gobj);
}

static int curve_click(t_gobj *z, t_glist *glist, 
    t_scalar *sc, t_canvas *template, float basex, float basey,
    int xpix, int ypix, int shift, int alt, int dbl, int doit)
{
    t_curve *x = (t_curve *)z;
    int i, n = x->x_npoints;
    int bestn = -1;
    int besterror = 0x7fffffff;
    t_fielddesc *f = x->x_vec;
    t_word *data = sc->x_vec;
    for (i = 0, f = x->x_vec; i < n; i++, f += 2)
    {
    	int xloc = glist_xtopixels(glist,
    	    basex + fielddesc_getfloat(f, template, data, 0));
    	int yloc = glist_ytopixels(glist,
    	    basey + fielddesc_getfloat(f+1, template, data, 0));
    	int xerr = xloc - xpix, yerr = yloc - ypix;
	if (!f->fd_var && !(f+1)->fd_var)
	    continue;
	if (xerr < 0)
	    xerr = -xerr;
	if (yerr < 0)
	    yerr = -yerr;
	if (yerr > xerr)
	    xerr = yerr;
	if (xerr < besterror)
	{
	    besterror = xerr;
	    bestn = i;
	    curve_motion_xbase = fielddesc_getfloat(f, template, data, 0);
	    curve_motion_ybase = fielddesc_getfloat(f+1, template, data, 0);
	}
    }
    if (besterror > 10)
    	return (0);
    if (doit)
    {
    	curve_motion_xper = glist_pixelstox(glist, 1)
	    - glist_pixelstox(glist, 0);
    	curve_motion_yper = glist_pixelstoy(glist, 1)
	    - glist_pixelstoy(glist, 0);
	curve_motion_xcumulative = curve_motion_ycumulative = 0;
	curve_motion_glist = glist;
	curve_motion_gobj = &sc->x_gobj;
	curve_motion_wp = data;
	curve_motion_field = 2*bestn;
	curve_motion_template = template;
    	glist_grab(glist, z, curve_motion, 0, xpix, ypix);
    }
    return (1);
}

t_parentwidgetbehavior curve_widgetbehavior =
{
    curve_getrect,
    curve_displace,
    curve_select,
    curve_activate,
    curve_vis,
    curve_click,
};

static void curve_free(t_curve *x)
{
    t_freebytes(x->x_vec, 2 * x->x_npoints * sizeof(*x->x_vec));
}

static void curve_setup(void)
{
    curve_class = class_new(gensym("drawpolygon"), (t_newmethod)curve_new,
    	(t_method)curve_free, sizeof(t_curve), CLASS_NOINLET, A_GIMME, 0);
    class_setdrawcommand(curve_class);
    class_addcreator((t_newmethod)curve_new, gensym("drawcurve"),
    	A_GIMME, 0);
    class_addcreator((t_newmethod)curve_new, gensym("filledpolygon"),
    	A_GIMME, 0);
    class_addcreator((t_newmethod)curve_new, gensym("filledcurve"),
    	A_GIMME, 0);
    class_setparentwidget(curve_class, &curve_widgetbehavior);
}

/* --------- plots for showing array elements --------------- */

t_class *plot_class;

typedef struct _plot
{
    t_object x_obj;
    int x_flags;
    t_fielddesc x_outlinecolor;
    t_fielddesc x_width;
    t_fielddesc x_xloc;
    t_fielddesc x_yloc;
    t_fielddesc x_xinc;
    t_fielddesc x_data;
} t_plot;

static void *plot_new(t_symbol *classsym, t_int argc, t_atom *argv)
{
    t_plot *x = (t_plot *)pd_new(plot_class);
    int flags = 0;
    int nxy, i;
    t_fielddesc *fd;
    t_symbol *firstarg = atom_getsymbolarg(0, argc, argv);
    if (!strcmp(firstarg->s_name, "curve"))
    {
    	flags |= BEZ;
    	argc--, argv++;
    }
    if (argc) fielddesc_setarrayarg(&x->x_data, argc--, argv++);
    else FIELDDESC_SETFLOAT(&x->x_data, 1);
    if (argc) fielddesc_setfloatarg(&x->x_outlinecolor, argc--, argv++);
    else FIELDDESC_SETFLOAT(&x->x_outlinecolor, 0);
    if (argc) fielddesc_setfloatarg(&x->x_width, argc--, argv++);
    else FIELDDESC_SETFLOAT(&x->x_width, 1);
    if (argc) fielddesc_setfloatarg(&x->x_xloc, argc--, argv++);
    else FIELDDESC_SETFLOAT(&x->x_xloc, 1);
    if (argc) fielddesc_setfloatarg(&x->x_yloc, argc--, argv++);
    else FIELDDESC_SETFLOAT(&x->x_yloc, 1);
    if (argc) fielddesc_setfloatarg(&x->x_xinc, argc--, argv++);
    else FIELDDESC_SETFLOAT(&x->x_xinc, 1);
    x->x_flags = flags;
    return (x);
}

/* -------------------- widget behavior for plot ------------ */

    /* aux routine to get screen coordinates of a point */
static void plot_getcoordinate(t_plot *p, t_glist *glist,
    char *elem, int xonset, int yonset, int wonset, int indx,
    float basex, float basey, float xinc,
    float *xp, float *yp, float *wp)
{
    float xval, yval, ypix, wpix;
    if (xonset >= 0)
    	xval = *(float *)(elem + xonset);
    else xval = indx * xinc;
    yval = *(float *)(elem + yonset);
    ypix = glist_ytopixels(glist, basey + yval);
    if (wonset >= 0)
    {
    	    /* found "w" field which controls linewidth. */
    	float wval = *(float *)(elem + wonset);
	wpix = glist_ytopixels(glist, basey + yval + wval) - ypix;
	if (wpix < 0)
	    wpix = -wpix;
    }
    else wpix = 1;
    *xp = glist_xtopixels(glist, basex + xval);
    *yp = ypix;
    *wp = wpix;
}

    /* get everything you could possibly need about a plot */
static int plot_getfields(t_plot *x, t_glist *glist, 
    t_word *data, t_canvas *templatecanvas, 
    t_canvas **elemtemplatecanvasp, int *elemsizep, int *xonsetp, int *yonsetp,
    int *wonsetp, t_array **arrayp, int *nelemp, char **elemp,
    float *linewidthp, float *xlocp, float *xincp, float *ylocp)
{
    int arrayonset, elemsize, yonset, wonset, xonset, type;
    t_template *ownertemplate, *elemtemplate;
    t_symbol *elemtemplatesym, *dummy;
    t_canvas *elemtemplatecanvas;
    t_array *array;
    int nelem;
    char *elem;
    float linewidth, xloc, xinc, yloc;

    	/* find the data and verify it's an array */
    if (x->x_data.fd_type != A_ARRAY || !x->x_data.fd_var)
    {
    	error("plot: needs an array field");
    	return (-1);
    }
    ownertemplate = canvas_gettemplate(templatecanvas);
    if (!template_find_field(ownertemplate, x->x_data.fd_un.fd_varsym,
    	&arrayonset, &type, &elemtemplatesym))
    {
    	error("plot: %s: no such field", x->x_data.fd_un.fd_varsym->s_name);
    	return (-1);
    }
    if (type != DT_ARRAY)
    {
    	error("plot: %s: not an array", x->x_data.fd_un.fd_varsym->s_name);
    	return (-1);
    }
    if (!(elemtemplatecanvas =
    	(t_canvas *)pd_findbyclass(elemtemplatesym, canvas_class))
	    || !(elemtemplate =  canvas_gettemplatebyname(elemtemplatesym)))
    {
    	error("plot: %s: no such template", elemtemplatesym->s_name);
    	return (-1);
    }
    elemsize = elemtemplate->t_n * sizeof(t_word);
    if (!template_find_field(elemtemplate, gensym("y"), &yonset, &type, &dummy)
    	|| type != DT_FLOAT)	
    {
    	error("%s: needs floating-point 'y' field", elemtemplatesym->s_name);
    	return (-1);
    }
    if (!template_find_field(elemtemplate, gensym("x"), &xonset, &type, &dummy)
    	|| type != DT_FLOAT) 
	    xonset = -1;
    if (!template_find_field(elemtemplate, gensym("w"), &wonset, &type, &dummy)
    	|| type != DT_FLOAT) 
	    wonset = -1;

    array = *(t_array **)(((char *)data) + arrayonset);
    nelem = array->a_n;
    elem = ((char *)(array->a_vec));

    linewidth = fielddesc_getfloat(&x->x_width, templatecanvas, data, 1);
    xloc = fielddesc_getfloat(&x->x_xloc, templatecanvas, data, 1);
    xinc = fielddesc_getfloat(&x->x_xinc, templatecanvas, data, 1);
    yloc = fielddesc_getfloat(&x->x_yloc, templatecanvas, data, 1);

    	/* fill in slots for return values */
    *elemtemplatecanvasp = elemtemplatecanvas;
    *elemsizep = elemsize;
    *xonsetp = xonset;
    *yonsetp = yonset;
    *wonsetp = wonset;
    *arrayp = array;
    *nelemp = nelem;
    *elemp = elem;
    *linewidthp = linewidth;
    *xlocp = xloc;
    *xincp = xinc;
    *ylocp = yloc;
    return (0);
}

static void plot_getrect(t_gobj *z, t_glist *glist,
    t_word *data, t_canvas *templatecanvas, float basex, float basey,
    int *xp1, int *yp1, int *xp2, int *yp2)
{
    t_plot *x = (t_plot *)z;
    int elemsize, yonset, wonset, xonset;
    t_canvas *elemtemplatecanvas;
    int nelem;
    char *elem;
    float linewidth, xloc, xinc, yloc;
    t_array *array;
    float x1 = 0x7fffffff, y1 = 0x7fffffff, x2 = -0x7fffffff, y2 = -0x7fffffff;
    int i;
    float xpix, ypix, wpix;

    if (!plot_getfields(x, glist, data, templatecanvas, 
    	&elemtemplatecanvas, &elemsize, &xonset, &yonset,
    	&wonset, &array, &nelem, &elem, &linewidth, &xloc, &xinc, &yloc))
    {
	for (i = 0; i < nelem; i++)
	{
    	    plot_getcoordinate(x, glist, elem + i * elemsize,
    		xonset, yonset, wonset, i, basex + xloc, basey + yloc, xinc,
    		&xpix, &ypix, &wpix);
	    if (xpix < x1)
		x1 = xpix;
	    if (xpix > x2)
		x2 = xpix;
	    if (ypix - wpix < y1)
		y1 = ypix - wpix;
	    if (ypix + wpix > y2)
		y2 = ypix + wpix;
	}
    }

    *xp1 = x1;
    *yp1 = y1;
    *xp2 = x2;
    *yp2 = y2;
}

static void plot_displace(t_gobj *z, t_glist *glist,
    t_word *data, t_canvas *template, float basex, float basey,
    int dx, int dy)
{
    	/* not yet */
}

static void plot_select(t_gobj *z, t_glist *glist,
    t_word *data, t_canvas *template, float basex, float basey,
    int state)
{
    /* not yet */
}

static void plot_activate(t_gobj *z, t_glist *glist,
    t_word *data, t_canvas *template, float basex, float basey,
    int state)
{
    	/* not yet */
}

static void plot_vis(t_gobj *z, t_glist *glist, 
    t_word *data, t_canvas *templatecanvas, float basex, float basey,
    int vis)
{
    t_plot *x = (t_plot *)z;
    int elemsize, yonset, wonset, xonset;
    t_canvas *elemtemplatecanvas;
    int nelem;
    char *elem;
    float linewidth, xloc, xinc, yloc;
    t_array *array;

    if (plot_getfields(x, glist, data, templatecanvas, 
    	&elemtemplatecanvas, &elemsize, &xonset, &yonset,
    	&wonset, &array, &nelem, &elem, &linewidth, &xloc, &xinc, &yloc))
    	    return;

    if (vis)
    {    	
    	char outline[20];
    	int lastpixel = -1, ndrawn = 0;
    	float xsum, yval = 0, wval, xpix;
    	int ixpix = 0, i;

    	    /* draw the trace */
    	numbertocolor(
    	    	fielddesc_getfloat(&x->x_outlinecolor, templatecanvas, data, 1),
    	    	outline);
    	if (wonset >= 0)
    	{
    	    	/* found "w" field which controls linewidth.  The trace is
    	    	a filled polygon with 2n points. */
    	    sys_vgui(".x%x.c create polygon \\\n",
    	    	glist_getcanvas(glist));
    	    
    	    for (i = 0, xsum = xloc; i < nelem; i++)
    	    {
    	    	float usexloc;
    	    	if (xonset >= 0)
    	    	    usexloc = xloc + *(float *)((elem + elemsize * i) + xonset);
    	    	else usexloc = xsum, xsum += xinc;
    		yval = *(float *)((elem + elemsize * i) + yonset);
    		wval = *(float *)((elem + elemsize * i) + wonset);
    		xpix = glist_xtopixels(glist, basex + usexloc);
    		ixpix = xpix + 0.5;
    		if (xonset >= 0 || ixpix != lastpixel)
    		{
    	    	    sys_vgui("%d %f \\\n", ixpix,
    	    		glist_ytopixels(glist,
			    basey + yloc + yval - wval));
    	    	    ndrawn++;
    		}
    		lastpixel = ixpix;
    		if (ndrawn >= 1000) goto ouch;
    	    }
	    lastpixel = -1;
    	    for (i = nelem-1; i >= 0; i--)
    	    {
    	    	float usexloc;
    	    	if (xonset >= 0)
    	    	    usexloc = xloc + *(float *)((elem + elemsize * i) + xonset);
    	    	else xsum -= xinc, usexloc = xsum;
    		yval = *(float *)((elem + elemsize * i) + yonset);
    		wval = *(float *)((elem + elemsize * i) + wonset);
    		xpix = glist_xtopixels(glist, basex + usexloc);
    		ixpix = xpix + 0.5;
    		if (xonset >= 0 || ixpix != lastpixel)
    		{
    	    	    sys_vgui("%d %f \\\n", ixpix, glist_ytopixels(glist,
			    basey + yloc + yval + wval));
    	    	    ndrawn++;
    		}
    		lastpixel = ixpix;
    		if (ndrawn >= 1000) goto ouch;
     	    }
    		/* TK will complain if there aren't at least 2 points... */
    	    if (ndrawn == 0) sys_vgui("0 0 0 0 \\\n");
    	    else if (ndrawn == 1) sys_vgui("%d %f \\\n", ixpix + 10,
    	    		glist_ytopixels(glist, basey + yloc + yval));
    	ouch:
    	    sys_vgui(" -width 0 -fill %s\\\n", outline);
    	    if (x->x_flags & BEZ) sys_vgui("-smooth 1\\\n");

    	    sys_vgui("-tags plot%x\n", data);
    	}
    	else if (linewidth > 0)
    	{
    	    	/* no "w" field.  If the linewidth is positive, draw a
    	    	segmented line with the requested width; otherwise don't
    	    	draw the trace at all. */
    	    sys_vgui(".x%x.c create line \\\n", glist_getcanvas(glist));

    	    for (xsum = xloc, i = 0; i < nelem; i++)
    	    {
    	    	float usexloc;
    	    	if (xonset >= 0)
    	    	    usexloc = xloc + *(float *)((elem + elemsize * i) + xonset);
    	    	else usexloc = xsum, xsum += xinc;
    		yval = *(float *)((elem + elemsize * i) + yonset);
    		xpix = glist_xtopixels(glist, basex + usexloc);
    		ixpix = xpix + 0.5;
    		if (xonset >= 0 || ixpix != lastpixel)
    		{
    	    	    sys_vgui("%d %f \\\n", ixpix,
    	    		glist_ytopixels(glist, basey + yloc + yval));
    	    	    ndrawn++;
    		}
    		lastpixel = ixpix;
    		if (ndrawn >= 1000) break;
    	    }
    		/* TK will complain if there aren't at least 2 points... */
    	    if (ndrawn == 0) sys_vgui("0 0 0 0 \\\n");
    	    else if (ndrawn == 1) sys_vgui("%d %f \\\n", ixpix + 10,
    	    		glist_ytopixels(glist, basey + yloc + yval));

    	    sys_vgui("-width %f\\\n", linewidth);
    	    sys_vgui("-fill %s\\\n", outline);
    	    if (x->x_flags & BEZ) sys_vgui("-smooth 1\\\n");

    	    sys_vgui("-tags plot%x\n", data);
    	}
    	    /* We're done with the outline; now draw all the points.
    	    This code is inefficient since the template has to be
    	    searched for drawing instructions for every last point. */
    	
    	for (xsum = xloc, i = 0; i < nelem; i++)
    	{
    	    float usexloc, useyloc;
    	    t_gobj *y;
    	    if (xonset >= 0)
    	    	usexloc = basex + xloc +
    	    	    *(float *)((elem + elemsize * i) + xonset);
    	    else usexloc = basex + xsum, xsum += xinc;
    	    yval = *(float *)((elem + elemsize * i) + yonset);
    	    useyloc = basey + yloc + yval;
    	    for (y = ((t_glist *)elemtemplatecanvas)->gl_list; y; y = y->g_next)
	    {
		t_parentwidgetbehavior *wb = pd_getparentwidget(&y->g_pd);
		if (!wb) continue;
		(*wb->w_parentvisfn)(y, glist,
		    (t_word *)(elem + elemsize * i),
		    	(t_canvas *)elemtemplatecanvas,
		    	usexloc, useyloc, vis);
	    }
    	}
    }
    else
    {
    	    /* un-draw the individual points */
    	int i;
    	for (i = 0; i < nelem; i++)
    	{
    	    t_gobj *y;
    	    for (y = ((t_glist *)elemtemplatecanvas)->gl_list; y; y = y->g_next)
	    {
		t_parentwidgetbehavior *wb = pd_getparentwidget(&y->g_pd);
		if (!wb) continue;
		(*wb->w_parentvisfn)(y, glist,
		    (t_word *)(elem + elemsize * i), elemtemplatecanvas,
		    	0, 0, 0);
	    }
    	}
    	    /* and then the trace */
    	sys_vgui(".x%x.c delete plot%x\n",
    	    glist_getcanvas(glist), data);    	
    }
}

static float plot_motion_xcumulative;
static float plot_motion_ycumulative;
static t_symbol *plot_motion_xfield;
static t_symbol *plot_motion_yfield;
static t_glist *plot_motion_glist;
static t_gobj *plot_motion_gobj;
static t_word *plot_motion_wp;
static t_canvas *plot_motion_templatecanvas;
static int plot_motion_fattendirection;
static int plot_motion_npoints;
static int plot_motion_elemsize;
static int plot_motion_altkey;
static float plot_motion_initx;
static float plot_motion_xperpix;
static int plot_motion_lastx;

    /* LATER protect against the template changing or the scalar disappearing
    probably by attaching a gpointer here ... */

static void plot_motion(void *z, t_floatarg dx, t_floatarg dy)
{
    t_plot *x = (t_plot *)z;
    plot_motion_xcumulative += dx;
    if (plot_motion_fattendirection == -1)
    	plot_motion_ycumulative += dy;
    else plot_motion_ycumulative -= dy;
    if (plot_motion_fattendirection)
    {
    	float newy;
	if (plot_motion_fattendirection && plot_motion_ycumulative <0)
    	    newy = 0;
	else newy = plot_motion_ycumulative;
	canvas_setfloat(plot_motion_templatecanvas,
    	    plot_motion_yfield, plot_motion_wp, newy, 1);
    }
    else if (*plot_motion_xfield->s_name)
    {
    	    /* it's an x, y plot; can drag many points at once */
    	int i;
	char *charword = (char *)plot_motion_wp;
	for (i = 0; i < plot_motion_npoints; i++)
	{
	    t_word *thisword = (t_word *)(charword + i * plot_motion_elemsize);
	    if (*plot_motion_xfield->s_name)
	    {
	    	float xwas = canvas_getfloat(plot_motion_templatecanvas,
    		    plot_motion_xfield, thisword, 1);
		canvas_setfloat(plot_motion_templatecanvas,
    		    plot_motion_xfield, thisword, xwas + dx, 1);
	    }
	    if (*plot_motion_yfield->s_name)
	    {
	    	float ywas = canvas_getfloat(plot_motion_templatecanvas,
    		    plot_motion_yfield, thisword, 1);
		canvas_setfloat(plot_motion_templatecanvas,
    		    plot_motion_yfield, thisword, ywas - dy, 1);
	    }
	}
    }
    else
    {
    	    /* a y-only plot. */
    	int thisx = plot_motion_initx +
	    plot_motion_xcumulative * plot_motion_xperpix, x2;
	int increment, i, nchange;
	char *charword = (char *)plot_motion_wp;
	float newy = plot_motion_ycumulative, oldy = canvas_getfloat(
	    plot_motion_templatecanvas, plot_motion_yfield,
    	    (t_word *)(charword + plot_motion_elemsize * plot_motion_lastx), 1);
	if (thisx < 0) thisx = 0;
	else if (thisx >= plot_motion_npoints)
	    thisx = plot_motion_npoints - 1;
	increment = (thisx > plot_motion_lastx ? -1 : 1);
	nchange = 1 + increment * (plot_motion_lastx - thisx);

    	for (i = 0, x2 = thisx; i < nchange; i++, x2 += increment)
	{
	    canvas_setfloat(plot_motion_templatecanvas,
    		plot_motion_yfield,
	    	    (t_word *)(charword + plot_motion_elemsize * x2),
	    		newy, 1);
	    if (nchange > 1) newy += (oldy - newy) * (1./(nchange - 1));
    	 }
	 plot_motion_lastx = thisx;
    }
    glist_redrawitem(plot_motion_glist, plot_motion_gobj);
}

static int plot_click(t_gobj *z, t_glist *glist, 
    t_scalar *sc, t_canvas *templatecanvas, float basex, float basey,
    int xpix, int ypix, int shift, int alt, int dbl, int doit)
{
    t_plot *x = (t_plot *)z;
    int elemsize, yonset, wonset, xonset;
    t_canvas *elemtemplatecanvas;
    int nelem, i;
    char *elem;
    float linewidth, xloc, xinc, yloc;
    t_array *array;
    t_word *data = sc->x_vec;

    if (!plot_getfields(x, glist, data, templatecanvas, 
    	&elemtemplatecanvas, &elemsize, &xonset, &yonset,
    	&wonset, &array, &nelem, &elem, &linewidth, &xloc, &xinc, &yloc))
    {
    	int best = 100;
	for (i = 0; i < nelem; i++)
    	{
	    float pxpix, pypix, pwpix, dx, dy;
    	    plot_getcoordinate(x, glist, elem + i * elemsize,
    		xonset, yonset, wonset, i, basex + xloc, basey + yloc, xinc,
    		&pxpix, &pypix, &pwpix);
	    if (pwpix < 4)
	    	pwpix = 4;
	    dx = pxpix - xpix;
	    if (dx < 0) dx = -dx;
	    if (dx > 8)
	    	continue;   
	    dy = pypix - ypix;
	    if (dy < 0) dy = -dy;
	    if (dx + dy < best)
	    	best = dx + dy;
	    if (!alt && wonset >= 0)
	    {
		dy = (pypix + pwpix) - ypix;
		if (dy < 0) dy = -dy;
		if (dx + dy < best)
	    	    best = dx + dy;
		dy = (pypix - pwpix) - ypix;
		if (dy < 0) dy = -dy;
		if (dx + dy < best)
	    	    best = dx + dy;
	    }
	}
	if (best > 8)
	    return (0);
	for (i = 0; i < nelem; i++)
    	{
	    float pxpix, pypix, pwpix, dx, dy, dy2, dy3;
	    int fattendirection;
    	    plot_getcoordinate(x, glist, elem + i * elemsize,
    		xonset, yonset, wonset, i, basex + xloc, basey + yloc, xinc,
    		&pxpix, &pypix, &pwpix);
	    if (pwpix < 4)
	    	pwpix = 4;
	    dx = pxpix - xpix;
	    if (dx < 0) dx = -dx;
	    dy = pypix - ypix;
	    if (dy < 0) dy = -dy;
	    if (!alt && wonset >= 0)
	    {
		dy2 = (pypix + pwpix) - ypix;
		if (dy2 < 0) dy2 = -dy2;
		dy3 = (pypix - pwpix) - ypix;
		if (dy3 < 0) dy3 = -dy3;
	    }
	    else dy2 = dy3 = 100;
	    if (dx + dy <= best || dx + dy2 <= best || dx + dy3 <= best )
	    {
	    	if (dy < dy2 && dy < dy3)
		    plot_motion_fattendirection = 0;
    		else if (dy2 < dy3)
		    plot_motion_fattendirection = -1;
		else plot_motion_fattendirection = 1;
		if (doit)
		{
    	    	    plot_motion_elemsize = elemsize;
    	    	    plot_motion_npoints = 1;
		    plot_motion_glist = glist;
		    plot_motion_gobj = &sc->x_gobj;
		    plot_motion_wp = (t_word *)(elem + i * elemsize);
		    plot_motion_templatecanvas = elemtemplatecanvas;
		    if (plot_motion_fattendirection)
		    {
		    	plot_motion_xfield = &s_;
			plot_motion_xcumulative = 0;
		    	plot_motion_yfield = gensym("w");
			plot_motion_ycumulative = 
			    *(float *)((elem + elemsize * i) + wonset);
		    }
		    else if (alt && xpix < pxpix) /* delete a point */
		    {
		    	if (nelem <= 1)
			    return (0);
		    	memmove(elem + elemsize * i, 
			    elem + elemsize * (i+1),
			    	(nelem - 1 - i) * elemsize);
			array_resize(array,
			    canvas_gettemplate(elemtemplatecanvas), nelem - 1);
			glist_redrawitem(plot_motion_glist, plot_motion_gobj);
		    }
		    else
		    {
		    	if (alt)
			{
			    /* add a point (after the clicked-on one) */
			    array_resize(array,
			    	canvas_gettemplate(elemtemplatecanvas),
				    nelem + 1);
		    	    elem = (char *)array->a_vec;
			    memmove(elem + elemsize * (i+1), 
			    	elem + elemsize * i,
			    	    (nelem - i) * elemsize);
			    i++;
			    nelem++;
			    plot_motion_wp = (t_word *)(elem + i * elemsize);
			}
			if (xonset >= 0)
			{
		    	    plot_motion_xfield = gensym("x");
			    plot_motion_xcumulative = 
				*(float *)((elem + elemsize * i) + xonset);
			    plot_motion_npoints = nelem - i;
			}
			else
			{
		    	    plot_motion_xfield = &s_;
			    plot_motion_xcumulative = 0;
			    plot_motion_initx = i;
			    plot_motion_lastx = i;
			    plot_motion_xperpix = (xinc == 0 ? 1 : 1./xinc);
			    plot_motion_wp = (t_word *)elem;
			    plot_motion_npoints = nelem;	    
			}
			if (yonset >= 0)
			{
		    	    plot_motion_yfield = gensym("y");
			    plot_motion_ycumulative = 
				*(float *)((elem + elemsize * i) + yonset);
			}
			else
			{
		    	    plot_motion_yfield = &s_;
			    plot_motion_ycumulative = 0;
			}
    		    }
		    glist_grab(glist, z, plot_motion, 0, xpix, ypix);
		}
		if (alt)
		{
		    if (xpix < pxpix)
		    	return (CURSOR_EDITMODE_DISCONNECT);
		    else return (CURSOR_RUNMODE_ADDPOINT);
		}
    		else return (plot_motion_fattendirection ?
		    CURSOR_RUNMODE_THICKEN : CURSOR_RUNMODE_CLICKME);
    	    }
    	}   
    }
    return (0);
}

t_parentwidgetbehavior plot_widgetbehavior =
{
    plot_getrect,
    plot_displace,
    plot_select,
    plot_activate,
    plot_vis,
    plot_click,
};

static void plot_setup(void)
{
    plot_class = class_new(gensym("plot"), (t_newmethod)plot_new, 0,
    	sizeof(t_plot), CLASS_NOINLET, A_GIMME, 0);
    class_setdrawcommand(plot_class);
    class_setparentwidget(plot_class, &plot_widgetbehavior);
}

/* ---------------- drawnumber: draw a number ---------------- */

/*
    drawnumbers draw numeric fields at controllable locations, with
    controllable color and label  .
    	invocation: (drawnumber|drawsymbol) variable x y color label
*/

t_class *drawnumber_class;

#define DRAW_SYMBOL 1

typedef struct _drawnumber
{
    t_object x_obj;
    t_fielddesc x_value;
    t_fielddesc x_xloc;
    t_fielddesc x_yloc;
    t_fielddesc x_color;
    t_symbol *x_label;
    int x_flags;
} t_drawnumber;

static void *drawnumber_new(t_symbol *classsym, t_int argc, t_atom *argv)
{
    t_drawnumber *x = (t_drawnumber *)pd_new(drawnumber_class);
    char *classname = classsym->s_name;
    int flags = 0;
    if (classname[4] == 's')
    	flags |= DRAW_SYMBOL;
    x->x_flags = flags;
    if (argc) fielddesc_setfloatarg(&x->x_value, argc--, argv++);
    else FIELDDESC_SETFLOAT(&x->x_value, 0);
    if (argc) fielddesc_setfloatarg(&x->x_xloc, argc--, argv++);
    else FIELDDESC_SETFLOAT(&x->x_xloc, 0);
    if (argc) fielddesc_setfloatarg(&x->x_yloc, argc--, argv++);
    else FIELDDESC_SETFLOAT(&x->x_yloc, 0);
    if (argc) fielddesc_setfloatarg(&x->x_color, argc--, argv++);
    else FIELDDESC_SETFLOAT(&x->x_color, 1);
    if (argc)
    	x->x_label = atom_getsymbolarg(0, argc, argv);
    else x->x_label = &s_;

    return (x);
}

/* -------------------- widget behavior for drawnumber ------------ */

#define DRAWNUMBER_BUFSIZE 80
static void drawnumber_sprintf(t_drawnumber *x, char *buf, t_atom *ap)
{
    int nchars;
    strncpy(buf, x->x_label->s_name, DRAWNUMBER_BUFSIZE);
    buf[DRAWNUMBER_BUFSIZE - 1] = 0;
    nchars = strlen(buf);
    atom_string(ap, buf + nchars, DRAWNUMBER_BUFSIZE - nchars);
}


static void drawnumber_getrect(t_gobj *z, t_glist *glist,
    t_word *data, t_canvas *template, float basex, float basey,
    int *xp1, int *yp1, int *xp2, int *yp2)
{
    t_drawnumber *x = (t_drawnumber *)z;
    t_atom at;
    int xloc = glist_xtopixels(glist,
    	basex + fielddesc_getfloat(&x->x_xloc, template, data, 0));
    int yloc = glist_ytopixels(glist,
    	basey + fielddesc_getfloat(&x->x_yloc, template, data, 0));
    int font = glist_getfont(glist);
    int fontwidth = sys_fontwidth(font), fontheight = sys_fontheight(font);
    char buf[DRAWNUMBER_BUFSIZE];
    if (x->x_flags & DRAW_SYMBOL)
    	SETSYMBOL(&at, fielddesc_getsymbol(&x->x_value, template, data, 0));
    else SETFLOAT(&at, fielddesc_getfloat(&x->x_value, template, data, 0));
    drawnumber_sprintf(x, buf, &at);
    *xp1 = xloc;
    *yp1 = yloc;
    *xp2 = xloc + fontwidth * strlen(buf);
    *yp2 = yloc + fontheight;
}

static void drawnumber_displace(t_gobj *z, t_glist *glist,
    t_word *data, t_canvas *template, float basex, float basey,
    int dx, int dy)
{
    /* refuse */
}

static void drawnumber_select(t_gobj *z, t_glist *glist,
    t_word *data, t_canvas *template, float basex, float basey,
    int state)
{
    post("drawnumber_select %d", state);
    /* fill in later */
}

static void drawnumber_activate(t_gobj *z, t_glist *glist,
    t_word *data, t_canvas *template, float basex, float basey,
    int state)
{
    post("drawnumber_activate %d", state);
}

static void drawnumber_vis(t_gobj *z, t_glist *glist, 
    t_word *data, t_canvas *template, float basex, float basey,
    int vis)
{
    t_drawnumber *x = (t_drawnumber *)z;
    
    if (vis)
    {
    	t_atom at;
	int xloc = glist_xtopixels(glist,
    	    basex + fielddesc_getfloat(&x->x_xloc, template, data, 0));
	int yloc = glist_ytopixels(glist,
    	    basey + fielddesc_getfloat(&x->x_yloc, template, data, 0));
	char colorstring[20], buf[DRAWNUMBER_BUFSIZE];
    	numbertocolor(fielddesc_getfloat(&x->x_color, template, data, 1),
    	    colorstring);
	if (x->x_flags & DRAW_SYMBOL)
    	    SETSYMBOL(&at, fielddesc_getsymbol(&x->x_value, template, data, 0));
	else SETFLOAT(&at, fielddesc_getfloat(&x->x_value, template, data, 0));
	drawnumber_sprintf(x, buf, &at);
	sys_vgui(".x%x.c create text %d %d -anchor nw -fill %s -text {%s}",
    	    	glist_getcanvas(glist), xloc, yloc, colorstring, buf);
    	sys_vgui(" -tags drawnumber%x\n", data);
    }
    else sys_vgui(".x%x.c delete drawnumber%x\n", glist_getcanvas(glist), data);
}

static float drawnumber_motion_ycumulative;
static t_glist *drawnumber_motion_glist;
static t_gobj *drawnumber_motion_gobj;
static t_word *drawnumber_motion_wp;
static t_canvas *drawnumber_motion_template;

    /* LATER protect against the template changing or the scalar disappearing
    probably by attaching a gpointer here ... */

static void drawnumber_motion(void *z, t_floatarg dx, t_floatarg dy)
{
    t_drawnumber *x = (t_drawnumber *)z;
    t_fielddesc *f = &x->x_value;
    drawnumber_motion_ycumulative -= dy;
    canvas_setfloat(drawnumber_motion_template,
    	f->fd_un.fd_varsym,
	    drawnumber_motion_wp, 
    	    drawnumber_motion_ycumulative,
	    	1);
    glist_redrawitem(drawnumber_motion_glist, drawnumber_motion_gobj);
}

static int drawnumber_click(t_gobj *z, t_glist *glist, 
    t_scalar *sc, t_canvas *template, float basex, float basey,
    int xpix, int ypix, int shift, int alt, int dbl, int doit)
{
    t_drawnumber *x = (t_drawnumber *)z;
    int x1, y1, x2, y2;
    t_word *data = sc->x_vec;
    drawnumber_getrect(z, glist,
    	sc->x_vec, template, basex, basey,
    	&x1, &y1, &x2, &y2);
    if (xpix >= x1 && xpix <= x2 && ypix >= y1 && ypix <= y2
    	&& x->x_value.fd_var)
    {
    	if (doit)
	{
	    drawnumber_motion_glist = glist;
	    drawnumber_motion_gobj = &sc->x_gobj;
	    drawnumber_motion_wp = data;
	    drawnumber_motion_template = template;
	    drawnumber_motion_ycumulative =
	    	fielddesc_getfloat(&x->x_value, template, data, 0);
    	    glist_grab(glist, z, drawnumber_motion, 0, xpix, ypix);
	}
    	return (1);
    }
    else return (0);
}

t_parentwidgetbehavior drawnumber_widgetbehavior =
{
    drawnumber_getrect,
    drawnumber_displace,
    drawnumber_select,
    drawnumber_activate,
    drawnumber_vis,
    drawnumber_click,
};

static void drawnumber_free(t_drawnumber *x)
{
}

static void drawnumber_setup(void)
{
    drawnumber_class = class_new(gensym("drawnumber"),
    	(t_newmethod)drawnumber_new, (t_method)drawnumber_free,
	sizeof(t_drawnumber), CLASS_NOINLET, A_GIMME, 0);
    class_setdrawcommand(drawnumber_class);
    class_addcreator((t_newmethod)drawnumber_new, gensym("drawsymbol"),
    	A_GIMME, 0);
    class_setparentwidget(drawnumber_class, &drawnumber_widgetbehavior);
}

/* ---------------------- setup function ---------------------------- */

void g_template_setup(void)
{
    gtemplate_setup();
    curve_setup();
    plot_setup();
    drawnumber_setup();
}

