/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 2000 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <parted/parted.h>
#include <parted/endian.h>
#include <parted/disk_dos.h>
#include <parted/disk_loop.h>
#include <parted/disk_pc98.h>
#include <parted/disk_mac.h>
#include <parted/disk_gpt.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) dgettext (PACKAGE, String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#include <string.h>

#define HFS_SIGNATURE	0x4244

static int hfs_probe (const PedGeometry* geom);
static int hfs_clobber (PedGeometry* geom);
static int hfs_set_system (const PedFileSystem* fs, PedPartition* part,
			   const PedDiskType* disk_type);

static PedFileSystemOps hfs_ops = {
	probe:		hfs_probe,
	clobber:	hfs_clobber,
	open:		NULL,
	create:		NULL,
	close:		NULL,
	check:		NULL,
	copy:		NULL,
	resize:		NULL,
	get_resize_constraint:	NULL,
	set_system:	hfs_set_system
};

static PedFileSystemType hfs_type = {
	next:	NULL,
	ops:	&hfs_ops,
	name:	"hfs"
};

void
ped_file_system_hfs_init ()
{
	ped_file_system_type_register (&hfs_type);
}

void
ped_file_system_hfs_done ()
{
	ped_file_system_type_unregister (&hfs_type);
}

static int
hfs_probe (const PedGeometry* geom)
{
	char	buf[512];

	if (geom->length < 2)
		return 0;
	if (!ped_geometry_read (geom, buf, 2, 1))
		return 0;
	return PED_BE16_TO_CPU (*(__u16*) buf) == HFS_SIGNATURE;
}

static int
hfs_clobber (PedGeometry* geom)
{
	char	buf[512];

	memset (buf, 0, 512);
	return ped_geometry_write (geom, buf, 2, 1);
}

static int
hfs_set_system (const PedFileSystem* fs, PedPartition* part,
		const PedDiskType* disk_type)
{
	if (strcmp (disk_type->name, LOOP_NAME) == 0)
		return 1;

	if (strcmp (disk_type->name, DOS_NAME) == 0) {
		DosPartitionData*	dos_data = part->disk_specific;

		PED_ASSERT (dos_data != NULL, return 0);

		if (dos_data->raid) {
			dos_data->system = 0xfd;
			return 1;
		}
		if (dos_data->lvm) {
			dos_data->system = 0x8e;
			return 1;
		}

		if (dos_data->hidden) {
			ped_exception_throw (
				PED_EXCEPTION_ERROR,
			       	PED_EXCEPTION_CANCEL,
				_("HFS partitions can't be hidden on msdos "
				  "disk labels."));
			return 0;
		}

		dos_data->system = 0x83;
		return 1;
	}

	if (strcmp (disk_type->name, PC98_NAME) == 0) {
		PC98PartitionData*	pc98_data = part->disk_specific;

		PED_ASSERT (pc98_data != NULL, return 0);

		pc98_data->system = 0x20e2;
		return 1;
	}

	if (strcmp (disk_type->name, MAC_NAME) == 0) {
		MacPartitionData*	mac_data = part->disk_specific;

		PED_ASSERT (mac_data != NULL, return 0);

		if (mac_data->is_boot) {
			strcpy (mac_data->system_name, "Apple_Bootstrap");
			mac_data->status = 0x33;
		} else {
			strcpy (mac_data->system_name, "Apple_HFS");
			mac_data->status |= 0x7f;
		}
		return 1;
	}

	if (strcmp (disk_type->name, GPT_NAME) == 0) {
		GPTPartitionData *gpt_part_data = part->disk_specific;
		PED_ASSERT(gpt_part_data      != NULL, return 0);	
		PED_ASSERT(gpt_part_data->pte != NULL, return 0);
		gpt_part_data->pte->PartitionTypeGuid =
			PARTITION_BASIC_DATA_GUID;
		return 1;
	}

	return 0;
}

