#include "SpellStacker.h"
#include "Win32PluginAPI.cpp"

#define MAJOR "1"
#define MINOR "1"

static SpellStacker * spellStacker;

extern "C" G_MODULE_EXPORT void plugin_init(plugin_address_table_t * pat) {
	plugin_address_table_init(pat);
	spellStacker = new SpellStacker();
}

extern "C" G_MODULE_EXPORT void plugin_cleanup(void) {
  delete spellStacker;
}

extern "C" G_MODULE_EXPORT char * plugin_query_name() {
  return "SpellStacker";
}

extern "C" G_MODULE_EXPORT char * plugin_query_description() {
  return _("This plugin keeps a record of the number of spells (and selected other commands) you have stacked.");
}

extern "C" G_MODULE_EXPORT char * plugin_query_major() {
  return MAJOR;
}

extern "C" G_MODULE_EXPORT char * plugin_query_minor() {
  return MINOR;
}

SpellStacker::SpellStacker() {
  version = 0.1;
  name = strdup("SpellStacker");

  register_plugin(this, VERSION);
  plugin_handler_add_input_filter(get_plugin_handler(), this);
}

SpellStacker::~SpellStacker() {
  unregister_plugin(this);
  free(name);
}

static void set_tooltip(GtkWidget * w, const gchar * tip) {
  GtkTooltips * t = gtk_tooltips_new();
  gtk_tooltips_set_tip (t, w, tip, NULL);
}

struct command {
  char * cmd;
  char * min_abbrev;
};

struct command cmd_table[] = {

  { "cast", "c" },
  { "charge", "ch" },
  { "piledrive", "pil" },
  { "kick", "kic" },
  { "kata", "ka" },
  { "noogie", "noo" },
  { "strangle", "str" },

  { NULL, NULL }

};

void SpellStacker::input(Connection * c, char * in) {
  uint32_t len = 0;

  // Detect whether or not Turf Protocol is available for this connection.
  if (!turf_protocol_is_supported(c))
    return;

  // Is this one of the commands we're interested in?
  char * pc = strchr(in, ' ');
  if (pc)
    len = pc - in;
  else
    len = strlen(in);

  int i = 0;
  while (true) {
    if (!cmd_table[i].cmd)
      break;

    if (len >= strlen(cmd_table[i].min_abbrev)) {
      if (!strncasecmp(cmd_table[i].cmd, in, len)) {
	// This is a match.
	// Create a TurfProtocol callback.
	
	sendCommand(c, in);
	in[0] = '\0';
	return;
	  
      }
    }
    i++;
  }
}

G_MODULE_EXPORT void spellstacker_callback(Connection * c, char * buf, void * data) {
  spellStacker->callback(c, buf, data);
}

struct stacker_data * SpellStacker::find_entry(Connection * conn) {
  for (StackerDataList::iterator i = stackerData.begin(); i != stackerData.end(); i++) {
    if ((*i)->connection == conn)
      return (*i);
  }
  return NULL;
}

static int StackerDataCmp(struct stacker_data * s1, struct stacker_data * s2) {
  return (s1 < s2);
}

void SpellStacker::sendCommand(Connection * c, char * in) {

  struct stacker_data * data = find_entry(c);

  if (!data) {
    data = (struct stacker_data *)malloc(sizeof(struct stacker_data));
    data->count = 0;
    data->commands = NULL;
    data->connection = c;
    // Create the widget and add it.
    data->widget = gtk_button_new_with_label(_("Stacked: 0"));
    vt_add_to_tray(connection_get_vt(c), data->widget, &data->frame);
    

    StackerDataList::iterator i = std::lower_bound(stackerData.begin(),
						   stackerData.end(),
						   data,
						   StackerDataCmp);
    
    stackerData.insert(i, data);
  }

  struct stacker_command * cmd = (struct stacker_command *)malloc(sizeof(struct stacker_command));
  cmd->cmd = strdup(in);

  cmd->next = data->commands;
  data->commands = cmd;

  data->count++;
  
  updateDisplay(data);
  
  // Finally send the command.
  turf_protocol_add_command(c, in, (void *)spellstacker_callback, (gpointer)cmd);
}

void SpellStacker::callback(Connection * c, char * buf, void * data) {

  if (buf) { // Not finished getting callback data.

    if (buf[0] == '\x1f')
      return;

	vt_append(connection_get_vt(c), buf);
	vt_scroll(connection_get_vt(c));
    return;
  }

  struct stacker_command * cmd = (struct stacker_command *)data;

  struct stacker_data * stacker = find_entry(c);
  if (!stacker) {
    return;
  }

  stacker->count--;

  // Find the command in the list and remove it.

  if (stacker->commands == cmd) {
    stacker->commands = cmd->next;
  } else {
    for (struct stacker_command * tmp = stacker->commands; tmp; tmp = tmp->next) {
      if (tmp->next == cmd) {
	tmp->next = cmd->next;
	break;
      }
    }
  }

  updateDisplay(stacker);

  // Free the user data associated with this command.  We have to do this here as we
  // can't do it in TurfProtocol.cpp due to Win32 DLL hell.
  if (!buf && data)
	  free(data);
}

void SpellStacker::updateDisplay(struct stacker_data * stacker) {

  // Update the display area.
  char buf2[1024];
  snprintf(buf2, 1024, "Stacked: %d", stacker->count);
  GtkLabel * label = GTK_LABEL(GTK_BIN(stacker->widget)->child);
  gtk_label_set_text(GTK_LABEL(label), buf2);

  char buf3[32768];
  buf3[0] = '\0';

  int count = 1;
  if (stacker->count == 0) {
    snprintf(buf3, 32768, "You currently have no commands stacked.");
  } else {
    snprintf(buf3, 32768, "Stacked commands:\n");
    for (struct stacker_command * tmp = stacker->commands; tmp; tmp = tmp->next) {
      snprintf(buf2, 1024, "%d: %s\n", stacker->count - count, tmp->cmd);
      strcat(buf3, buf2);
      count++;
    }
  }

  set_tooltip(stacker->widget, buf3);
}

