/*
**  LocalFolder.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
#import <Pantomime/LocalFolder.h>

#import <Pantomime/Constants.h>
#import <Pantomime/Flags.h>
#import <Pantomime/LocalFolderCacheManager.h>
#import <Pantomime/LocalMessage.h>
#import <Pantomime/LocalStore.h>
#import <Pantomime/Parser.h>
#import <Pantomime/NSDataExtensions.h>

#import <fcntl.h>
#import <unistd.h>
#import <string.h>
#import <sys/file.h>

@implementation LocalFolder

- (id) initWithPathToFile: (NSString *) thePath
{
  LocalFolderCacheManager *aLocalFolderCacheManager;
  NSDictionary *attributes;
  NSString *pathToCache;
  
  self = [super initWithName: [thePath lastPathComponent]];

  // We verify if a <name>.tmp was present. If yes, we simply remove it.
  if ( [[NSFileManager defaultManager] fileExistsAtPath: [thePath stringByAppendingString: @".tmp"]] )
    {
      NSLog(@"Removed %@", [thePath stringByAppendingString: @".tmp"]);
      [[NSFileManager defaultManager] removeFileAtPath: [thePath stringByAppendingString: @".tmp"]
				      handler: nil];
    }

  [self setPath: thePath];
  
  NSLog(@"Opening %@...", [self path]);
  
  if (! [self _openAndLockFolder] )
    {
      AUTORELEASE(self);
      return nil;
    }
  
  // We load our cache
  pathToCache = [NSString stringWithFormat: @"%@/.%@.cache",
			  [[self path] substringToIndex: 
					 ([[self path] length] - [[[self path] lastPathComponent] length])],
			  [[self path] lastPathComponent] ];
  
  attributes = [[NSFileManager defaultManager] fileAttributesAtPath: [self path]
					       traverseLink: NO];

  // We set our initial file attributes for the mbox
  [self setFileAttributes: attributes];
  
  // We load our cache from the file, creating it if it doesn't exist
  aLocalFolderCacheManager = [LocalFolderCacheManager localFolderCacheFromDiskWithPath: pathToCache];  
  [self setLocalFolderCacheManager: aLocalFolderCacheManager];

  // We update the path to this folder for the cache manager
  [[self localFolderCacheManager] setPathToFolder: [self path]];
  
  NSLog(@"Folder (%@) opened...", [self path]);

  return self;
}

//
//
//
- (void) dealloc
{
  RELEASE(fileAttributes);
  RELEASE(path);
  
  TEST_RELEASE(localFolderCacheManager);

  [super dealloc];
}

//
// This method is used to parse the message headers (and only that)
// from the current folder.
//
- (void) parse
{
  LocalMessage *aLocalMessage;
  NSAutoreleasePool *pool;
  
  long begin, end, size;
  char aLine[1024];
  int index;
  
  //
  // We first verify if we need to parse the folder.
  // We invalidate our cache if our size OR or modification date have changed
  //
  if ( [[[self fileAttributes] objectForKey: NSFileModificationDate] isEqualToDate: [[self localFolderCacheManager] modificationDate]] ||
       [[[self fileAttributes] objectForKey: NSFileSize] intValue] == [[self localFolderCacheManager] fileSize] )
    {
      NSArray *array;
      int i;
      
      NSLog(@"Using cache for folder %@", [self name]);
      
      array = [[self localFolderCacheManager] messages];
      
      for (i = 0; i < [array count]; i++)
	{
	  [[array objectAtIndex: i] setFolder: self];
	}
      
      [self setMessages: array];

      return;
    }
  else
    {
      NSLog(@"Invalidating cache.");
      [[self localFolderCacheManager] invalidate];
    }
  
  NSLog(@"Rebuilding cache for folder %@", [self name]);
  NSLog(@"PLEASE, BE PATIENT!");

  // We create a temporary autorelease pool since parse can be
  // memory consuming on our default autorelease pool.
  pool = [[NSAutoreleasePool alloc] init];

  // We initialize our variables
  aLocalMessage = [[LocalMessage alloc] init];
  begin = ftell(stream);
  end = 0L;
  index = 0;
  
  while (fgets(aLine, 1024, stream) != NULL)
    {
      switch( tolower(aLine[0]) )
	{ 
	case 'b':
	  if (strncasecmp(aLine, "Bcc", 3) == 0)
	    {
	      [Parser parseDestination: [self unfoldLinesStartingWith: aLine]
	      	      forType: BCC
	      	      inMessage: aLocalMessage];
	    }
	  else
	    {
	      [Parser parseUnknownHeader: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  break;
	  
	case 'c':
	  if (strncasecmp(aLine, "Cc", 2) == 0)
	    {
	      [Parser parseDestination: [self unfoldLinesStartingWith: aLine]
	      	      forType: CC
	      	      inMessage: aLocalMessage];
	    }
	  else if (strncasecmp(aLine, "Content-Disposition", 19) == 0)
	    {
	      [Parser parseContentDisposition: [self unfoldLinesStartingWith: aLine]
	      	      inPart: aLocalMessage];
	    }
	  else if (strncasecmp(aLine, "Content-Transfer-Encoding", 25) == 0)
	    {
	      [Parser parseContentTransferEncoding: [self unfoldLinesStartingWith: aLine]
		      inPart: aLocalMessage];
	    }
	  else if (strncasecmp(aLine, "Content-Type", 12) == 0)
	    {
	      [Parser parseContentType: [self unfoldLinesStartingWith: aLine]
	      	      inPart: aLocalMessage];
	    }
	  else
	    {
	      [Parser parseUnknownHeader: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  break;
	  
	case 'd':
	  if (strncasecmp(aLine, "Date", 4) == 0)
	    {
	      [Parser parseDate: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  else
	    {
	      [Parser parseUnknownHeader: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  break;

	case 'f':
	  if (strncasecmp(aLine, "From ", 5) == 0)
	    {
	      // do nothing, it's our message separator
	    }
	  else if (strncasecmp(aLine, "From", 4) == 0)
	    {
	      [Parser parseFrom: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  else
	    {
	      [Parser parseUnknownHeader: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  break;

	case 'm':
	  if (strncasecmp(aLine, "Message-ID", 10) == 0)
	    {
	      [Parser parseMessageID: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  else if (strncasecmp(aLine, "MIME-Version", 12) == 0)
	    {
	      [Parser parseMimeVersion: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  else
	    {
	      [Parser parseUnknownHeader: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  break;
	  
	case 'o':
	  if (strncasecmp(aLine, "Organization", 12) == 0)
	    {
	      [Parser parseOrganization: [self unfoldLinesStartingWith: aLine]
		      inMessage: aLocalMessage];
	    }
	  else
	    {
	      [Parser parseUnknownHeader: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  break;
	 
	case 'r':
	  if (strncasecmp(aLine, "Reply-To", 8) == 0)
	    {
	      [Parser parseReplyTo: [self unfoldLinesStartingWith: aLine]
		      inMessage: aLocalMessage];
	    }
	  else
	    {
	      [Parser parseUnknownHeader: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  break;

	case 's':
	  if (strncasecmp(aLine, "Status", 6) == 0) 
	    {
	      [Parser parseStatus: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  else if (strncasecmp(aLine, "Subject", 7) == 0)
	    {
	      [Parser parseSubject: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  else
	    {
	      [Parser parseUnknownHeader: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  break;

	case 't':
	  if (strncasecmp(aLine, "To", 2) == 0)
	    {
	      [Parser parseDestination: [self unfoldLinesStartingWith: aLine]
	      	      forType: TO
	      	      inMessage: aLocalMessage];
	    }
	  else
	    {
	      [Parser parseUnknownHeader: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  break;

	case 'x':
	  if (strncasecmp(aLine, "X-Status", 8) == 0)
	    {
	      [Parser parseXStatus: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  else
	    {
	      [Parser parseUnknownHeader: [self unfoldLinesStartingWith: aLine]
	      	      inMessage: aLocalMessage];
	    }
	  break;

	case '\n':
	  
	  [aLocalMessage setFilePosition: begin];
	  [aLocalMessage setBodyFilePosition: ftell(stream)];

	  // We must set this in case the last message of our mbox is
	  // an "empty message", i.e, a message with all the headers but
	  // with an empty content.
	  end = ftell(stream);

	  while (fgets(aLine, 1024, stream) != NULL)
            {
              if (strncmp(aLine, "From ", 5) == 0) break;
              else end = ftell(stream);
            }

	  fseek(stream, end, SEEK_SET);
          size = end - begin;

	  // We increment our index
	  index = index + 1;

	  // We set the properties of our message object and we add it to our folder.
          [aLocalMessage setSize: size];
	  [aLocalMessage setMessageNumber: index];
	  [aLocalMessage setFolder: self];
	  [self appendMessage: aLocalMessage];

	  // We add to our cache
	  [[self localFolderCacheManager] addMessage: aLocalMessage];

	  RELEASE(aLocalMessage);

          begin = ftell(stream);
  
	  // We re-init our message and our mutable string for the next message we're gonna read
	  aLocalMessage = [[LocalMessage alloc] init];

	  break;
	  
	default:
	  [Parser parseUnknownHeader: [self unfoldLinesStartingWith: aLine]
	  	  inMessage: aLocalMessage];
	}
    }  

  // We sync our cache
  [[self localFolderCacheManager] synchronize];

  RELEASE(aLocalMessage);

  RELEASE(pool);
}

//
// This method is used to unfold the lines that have been folded
// by starting with the first line.
// 
- (NSData *) unfoldLinesStartingWith:(char *) firstLine
{
  NSMutableData *aMutableData;
  NSData *aData;
  char aLine[1024], buf[1024];
  long mark;
  
  // We initialize our buffers
  bzero(aLine, 1024);
  bzero(buf, 1024);
    
  mark = ftell(stream);
  fgets(aLine, 1024, stream);

  if (aLine == NULL)
    {
      return [NSData dataWithBytes: firstLine  length: strlen(firstLine)];
    }

  // We create our mutable data
  aMutableData = [[NSMutableData alloc] initWithCapacity: strlen(firstLine)];

  // We remove the trailing \n and we append our first line to our mutable data
  strncpy(buf, firstLine, strlen(firstLine) - 1);
  [aMutableData appendCFormat: @"%s ", buf];
  
  // We loop as long as we have a space or tab character as the first character 
  // of the line that we just read
  while ( aLine[0] == 9 || aLine[0] == 32 )
    {
      char *ptr;

      // We skip the first char
      ptr = aLine;
      ptr++;
      
      // We init our buffer and we copy the data into it by trimming the trailing \n
      bzero(buf,1024);
      strncpy(buf, ptr, strlen(ptr) - 1);
      [aMutableData appendCFormat: @"%s ", buf];

      // We set our mark and get the next folded line (if there's one)
      mark = ftell(stream);
      bzero(aLine, 1024);
      fgets(aLine, 1024, stream);
      
      if (aLine == NULL)
        {
	  RELEASE(aMutableData);
          return nil;
        }
    }

  // We reset our file pointer position and we free our C buffers.
  fseek(stream, mark, SEEK_SET);
  
  // We trim our last " " that we added to our data
  aData = [aMutableData subdataToIndex: [aMutableData length] - 1];
  
  RELEASE(aMutableData);

  return aData;
}


//
// This method is used to close the current folder.
// It creates a temporary file where the folder is written to and
// it replaces the current folder file by this one once everything is
// alright.
//
- (void) close
{
  LocalStore *aLocalStore = nil;

  // We first obtain a reference to our local store
  aLocalStore = (LocalStore *)[self store];

  // We close the current folder
  NSLog(@"Closing %@...", [self name]);
  fclose([self stream]);
  flock([self fd], LOCK_UN);
  close([self fd]);

  // We synchorize our cache one last time
  [[self localFolderCacheManager] synchronize];

  // We remove our current folder from the list of opened folders in the store
  [aLocalStore removeFolderFromOpenedFolders: self];
}


//
// This method permanently removes messages that have the flag 'DELETED'
// OR 'TRANSFERRED'
//
// This method returns all messages that have the flag 'DELETED'
// All the returned message ARE IN RAW SOURCE.
//
- (NSArray *) expunge: (BOOL) returnDeletedMessages 
{
  NSMutableArray *aMutableArray;
  NSArray *allMessagesInFolder;
  
  FILE *theOutputStream;
  LocalStore *aLocalStore;

  LocalMessage *aMessage;
  Flags *theFlags;

  BOOL writeWasSuccessful;
  NSString *pathToMailbox;

  int i, count, messageNumber;
  char aLine[1024];
  
  // We first obtain a reference to our local store
  aLocalStore = (LocalStore *)[self store];

  allMessagesInFolder = [self allMessages];
  count = [self count];
  
  pathToMailbox = [NSString stringWithFormat: @"%@/%@", [aLocalStore path],
			    [self name]];
  
  // The stream is used to store temporary our 'new local folder'
  theOutputStream = fopen([[NSString stringWithFormat:@"%@.tmp", pathToMailbox]
			    cString], "a");
  
  // We assume that our write operation was successful and we initialize our messageNumber to 1
  writeWasSuccessful = YES;
  messageNumber = 1;
  
  // We verify it the creation failed
  if ( ! theOutputStream )
    {
      NSLog(@"Unable to create the temporary folder in LocalFolder: -expunge.");
      return [NSArray array];
    }
  
  aMutableArray = [[NSMutableArray alloc] init];

  for (i = 0; i < count; i++)
    {
      aMessage = [[self allMessages] objectAtIndex: i];

      theFlags = [aMessage flags];

      if ( [theFlags contain: DELETED] )
	{
	  // We add our message to our array of deleted messages, if we need to.
	  if ( returnDeletedMessages )
	    {
	      [aMutableArray addObject: [aMessage rawSource]];
	    }
	  
	  [[self localFolderCacheManager] removeMessage: aMessage];
	}
      else if ( [theFlags contain: TRANSFERRED] )
	{
	  // We do nothing for a transferred message so it'll just get
	  // deleted.
	  [[self localFolderCacheManager] removeMessage: aMessage];
	}
      else
	{
	  int headers_length;
	  long position; 

	  // We get our position and headers_length
	  position = ftell(theOutputStream);
	  headers_length = ([aMessage bodyFilePosition] - [aMessage filePosition]); 
	  
	  // We seek
	  fseek([self stream], [aMessage filePosition], SEEK_SET);
	  
	  bzero(aLine, 1024);
	      
	  while( fgets(aLine, 1024, [self stream]) != NULL && 
		 (ftell([self stream]) < ([aMessage filePosition] + [aMessage size])) )
	    {
	      // We write our line to our new stream
	      if ( fputs(aLine, theOutputStream) < 0 )
		{
		  writeWasSuccessful = NO;
		}
	      
	      bzero(aLine, 1024);  
	    } // while (...)
	 
	  // We add our message separator
	  if ( fputs(aLine, theOutputStream) < 0 )
	    {
	      writeWasSuccessful = NO;
	    }

	  // We update our message's ivars (folder and size don't change)
	  [aMessage setFilePosition: position];
	  [aMessage setBodyFilePosition: (position + headers_length)];
	  [aMessage setMessageNumber: messageNumber];
	  
	  // We increment our messageNumber local variable
	  messageNumber++;
	}

    } // for (i = 0; i < count; i++)

  // We close our output stream
  if ( fclose(theOutputStream) != 0 )
    {
      writeWasSuccessful = NO;
    }
 
  // We verify if the last write was successful, if yes, we remove our original mailbox
  // and we replace it by our temporary mailbox.
  if ( writeWasSuccessful )
    {
      // We close the current folder
      fclose([self stream]);
      flock([self fd], LOCK_UN);
      close([self fd]);

      // Now that Everything is alright, replace <folder name> by <folder name>.tmp
      [[NSFileManager defaultManager] removeFileAtPath: pathToMailbox
				      handler: nil];
      [[NSFileManager defaultManager] movePath: [NSString stringWithFormat:@"%@.tmp", pathToMailbox]
				      toPath: pathToMailbox
				      handler: nil];
      
      // We sync our cache
      [[self localFolderCacheManager] synchronize];

      // Now we re-open our folder and update the 'allMessages' ivar in the Folder superclass
      if (! [self _openAndLockFolder] )
	{
	  NSLog(@"A fatal error occured in LocalFolder: -expunge.");
	}

      [self setMessages: [[self localFolderCacheManager] messages]];
    }
  
  // The last write failed, let's remove our temporary file and keep the original mbox which, might
  // contains non-updated status flags or messages that have been transferred/deleted.
  else
    {
      NSLog(@"Writing to %@ failed. We keep the original mailbox.", pathToMailbox);
      NSLog(@"This can be due to the fact that your partition containing this mailbox is full or that you don't have write permission in the directory where this mailbox is.");
      [[NSFileManager defaultManager] removeFileAtPath: [NSString stringWithFormat:@"%@.tmp", pathToMailbox]
				      handler: nil];
    }
  
  
  return AUTORELEASE(aMutableArray);
}


//
// access / mutation methods
//

//
// This method returns the file descriptor used by this local folder.
//
- (int) fd
{
  return fd;
}


//
// This method sets the file descriptor to be used by this local folder.
//
- (void) setFD: (int) theFD
{
  fd = theFD;
}

- (NSString *) path
{
  return path;
}

- (void) setPath: (NSString *) thePath
{
  RETAIN(thePath);
  RELEASE(path);
  path = thePath;
}

//
// This method returns the file stream used by this local folder.
//
- (FILE *) stream
{
  return stream;
}


//
// This method sets the file stream to be used by this local folder.
//
- (void) setStream: (FILE *) theStream
{
  stream = theStream;
}


- (LocalFolderCacheManager *) localFolderCacheManager
{
  return localFolderCacheManager;
}

- (void) setLocalFolderCacheManager: (LocalFolderCacheManager *) theLocalFolderCacheManager
{
  if ( theLocalFolderCacheManager )
    {
      RETAIN(theLocalFolderCacheManager);
      RELEASE(localFolderCacheManager);
      localFolderCacheManager = theLocalFolderCacheManager;
    }
  else
    {
      RELEASE(localFolderCacheManager);
      localFolderCacheManager = nil;
    }
}


- (NSDictionary *) fileAttributes
{
  return fileAttributes;
}

- (void) setFileAttributes: (NSDictionary *) theAttributes
{
  RETAIN(theAttributes);
  RELEASE(fileAttributes);
  fileAttributes = theAttributes;
}


//
// This method is used to append a message to this folder. The message
// must be specified in raw source. The message is appended to the 
// local file and is initialized after.
//
- (void) appendMessageFromRawSource: (NSData *) theData
{
  NSMutableData *aMutableData;
  NSAutoreleasePool *pool;
  LocalMessage *aMessage;
  NSRange aRange;
  
  long mark, filePosition, bodyFilePosition;

  pool = [[NSAutoreleasePool alloc] init];

  aMutableData = [[NSMutableData alloc] initWithData: theData];
  
  // We keep the position where we were in the file
  mark = ftell( [self stream] );
  
  //NSLog(@"|%s|", [aMutableData cString]);

  // If the message doesn't contain the "From ", we add it
  if (! [aMutableData hasCPrefix: "From "] )
    {
      // We must add our mbox delimiter
      [aMutableData insertCString: "From -\n"
		    atIndex: 0];
    }

  // We MUST replace every "\nFrom " in the message by "\n>From "
  aRange = [aMutableData rangeOfCString: "\nFrom "];
  
  while (aRange.location != NSNotFound)
    {
      [aMutableData replaceBytesInRange: aRange
		    withBytes: "\n>From "];
      
      aRange = [aMutableData rangeOfCString: "\nFrom "
			     options: 0
			     range: NSMakeRange(aRange.location + aRange.length,
						[aMutableData length] - aRange.location - aRange.length) ];
    }
  
  // We add our message separator to the end of the raw source of this message
  // It's a simple \n in the mbox format.
  [aMutableData appendCString: "\n"];

  // We go at the end of the file...
  if ( fseek([self stream], 0L, SEEK_END) < 0 )
    {
      NSLog(@"Error in seeking to the end of the local folder.");
      RELEASE(aMutableData);
      return;
    }
  
  // We get the position of our message in the file
  filePosition = ftell( [self stream] );

  // We get the body position of the body
  aRange = [aMutableData rangeOfCString: "\n\n"];
  bodyFilePosition = filePosition + aRange.location + 2;
  
  // We write the string to our local folder
  //if ( fputs([aMutableData cString], [self stream]) < 0)
  if ( fwrite([aMutableData bytes], 1, [aMutableData length], [self stream]) <= 0 )
    {
      NSLog(@"Error in appending the raw source of a message to the local folder.");
      RELEASE(aMutableData);
      return;
    }
  
  // We now create the message from the raw source by using only the headers.
  aMessage = [[LocalMessage alloc] initWithHeadersFromData:
				     [MimeUtility unfoldLinesFromData: [aMutableData subdataToIndex: aRange.location + 1]]];
    
  [aMessage setFilePosition: filePosition];
  [aMessage setBodyFilePosition: bodyFilePosition];
  [aMessage setSize: ( ftell( [self stream] ) - filePosition) ];
  [aMessage setMessageNumber: ([self count] + 1)];
  [aMessage setFolder: self];
  
  // We append it to our folder
  [self appendMessage: aMessage];
  
  // We also append it to our cache
  if ( localFolderCacheManager )
    {
      [localFolderCacheManager addMessage: aMessage];
    }
  
  RELEASE(aMessage);

  // We sync our cache
  [[self localFolderCacheManager] synchronize];

  // We finally reset our fp where the mark was set
  fseek([self stream], mark, SEEK_SET);

  RELEASE(aMutableData);
  RELEASE(pool);
}

@end


//
// Private methods
//
@implementation LocalFolder (Private)

- (BOOL) _openAndLockFolder
{
  fd = open([[self path] cString], O_RDWR);
  
  if (fd < 0)
    {
      NSLog(@"LocalFolder: Unable to get folder descriptor...");
      return NO;
    }
  
  if (flock(fd, LOCK_EX|LOCK_NB) < 0) 
    {
      NSLog(@"LocalFolder: Unable to obtain the lock on the folder descriptor...");
      return NO;
    }
  else 
    {
      flock(fd, LOCK_UN);
    }

  stream = fdopen(fd, "r+");

  if (stream == NULL)
    {
      NSLog(@"LocalFolder: Unable to open the specified mailbox...");
      return NO;
    }

  flock(fd, LOCK_EX|LOCK_NB);

  return YES;
}

@end
