/*
 * node-store.c
 *
 * This is where the circuit elements (pins/wires) are stored.
 * 
 * Author: 
 *  Richard Hult <rhult@hem.passagen.se>
 * 
 *  http://www.dtek.chalmers.se/~d4hult/oregano/ 
 * 
 * Copyright (C) 1999,2000  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <math.h>
#include <gtk/gtk.h>
#include "node-store.h"
#include "node.h"
#include "part.h"
#include "wire.h"
#include "wire-private.h"
#include "item-data.h"
#include "print.h"

/*
 * NODE_EPSILON is used to check for intersection.
 * HASH_EPSILON is used in the hash equality check function.
 */
#define NODE_EPSILON 1e-10
#define HASH_EPSILON 1e-3

/* Share an endpoint? */
#define SEP(p1x,p1y,p2x,p2y) (IS_EQ(p1x, p2x) && IS_EQ(p1y, p2y))

/* Equals? */
#define IS_EQ(a,b) (fabs (a - b) < NODE_EPSILON)

#define ON_THE_WIRE(p1,start,end) ( fabs((end.y-start.y)*(p1.x-start.x)-(end.x-start.x)*(p1.y-start.y))<NODE_EPSILON )

static void       node_store_class_init (NodeStoreClass *klass);
static void       node_store_init (NodeStore *store);
static Node      *node_store_get_or_create_node (NodeStore *store, SheetPos pos);
static guint      node_hash (gconstpointer key);
static int        node_equal (gconstpointer a, gconstpointer b);
static GSList    *wires_intersect (NodeStore *store, double x1, double y1, double x2, double y2);
static GSList    *wire_intersect_parts (NodeStore *store, Wire *wire);
static int        is_wire_at_pos (double x1, double y1, double x2, double y2, SheetPos pos);
static GSList    *wires_at_pos (NodeStore *store, SheetPos pos);
static int        do_wires_intersect (double Ax, double Ay, double Bx, double By, 
				      double Cx, double Cy, double Dx, double Dy, 
				      SheetPos *pos);

typedef struct {
	Wire *wire;
	SheetPos pos;
} IntersectionPoint;

enum {
	DOT_ADDED,
	DOT_REMOVED,
	LAST_SIGNAL
};

static guint node_store_signals [LAST_SIGNAL] = { 0 };
static GtkObject *parent_class = NULL;

guint
node_store_get_type (void)
{
	static guint node_store_type = 0;

	if (!node_store_type) {
		static const GtkTypeInfo node_store_info = {
			"NodeStore",
			sizeof (NodeStore),
			sizeof (NodeStoreClass),
			(GtkClassInitFunc) node_store_class_init,
			(GtkObjectInitFunc) node_store_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		
		node_store_type = gtk_type_unique (GTK_TYPE_OBJECT, &node_store_info);
	}

	return node_store_type;
}

static void
node_store_destroy (GtkObject *object)
{
	NodeStore *store;

	store = NODE_STORE (object);
       
	g_hash_table_destroy (store->nodes);

	g_list_free (store->wires);
	g_list_free (store->parts);
	g_list_free (store->items);

	GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
node_store_class_init (NodeStoreClass *klass)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *)klass;
	parent_class = gtk_type_class (GTK_TYPE_OBJECT);

	object_class->destroy = node_store_destroy;

	node_store_signals [DOT_ADDED] = 
		gtk_signal_new ("dot_added",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	node_store_signals [DOT_REMOVED] = 
		gtk_signal_new ("dot_removed",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, node_store_signals, LAST_SIGNAL);
}

static void
node_store_init (NodeStore *store)
{
	store->nodes = g_hash_table_new (node_hash, node_equal);
	store->wires = NULL;
	store->parts = NULL;
	store->items = NULL;
}

NodeStore *
node_store_new (void)
{
	NodeStore *store;

	store = gtk_type_new (node_store_get_type ());

	return store;
}

static void
dot_added_callback (Node *node, SheetPos *pos, NodeStore *store)
{
	g_return_if_fail (store != NULL);
	g_return_if_fail (IS_NODE_STORE (store));

	gtk_signal_emit (GTK_OBJECT (store), node_store_signals[DOT_ADDED], pos);
}

static void
dot_removed_callback (Node *node, SheetPos *pos, NodeStore *store)
{
	g_return_if_fail (store != NULL);
	g_return_if_fail (IS_NODE_STORE (store));

	gtk_signal_emit (GTK_OBJECT (store), node_store_signals[DOT_REMOVED], pos);
}

static Node *
node_store_get_or_create_node (NodeStore *store, SheetPos pos)
{
	Node *node;

	g_return_val_if_fail (store != NULL, NULL);
	g_return_val_if_fail (IS_NODE_STORE (store), NULL);
	
	node = g_hash_table_lookup (store->nodes, &pos);

	if (!node) {
		/*
		 * Create a node at (x, y) and return it.
		 */
		node = node_new (pos);

		gtk_signal_connect_while_alive (
			GTK_OBJECT (node),
			"dot_added",
			dot_added_callback,
			store, 
			GTK_OBJECT (store));

		gtk_signal_connect_while_alive (
			GTK_OBJECT (node),
			"dot_removed",
			dot_removed_callback,
			store,
			GTK_OBJECT (store));
		
		g_hash_table_insert (store->nodes, &node->key, node);
	}

	/* 
	 * If there was a previously stored node here, just
	 * return that node.
	 */
	return node;
}

int
node_store_add_part (NodeStore *store, Part *part)
{
	GSList *wire_list, *list;
	Node *node;
	SheetPos lookup_key;
	SheetPos part_pos;
	gdouble x, y;
	int i, num_pins;

	g_return_val_if_fail (store != NULL, FALSE);
	g_return_val_if_fail (IS_NODE_STORE (store), FALSE);
	g_return_val_if_fail (part != NULL, FALSE);
	g_return_val_if_fail (IS_PART (part), FALSE);

	num_pins = part_get_num_pins (part);

	item_data_get_pos (ITEM_DATA (part), &part_pos);

	for (i = 0; i < num_pins; i++) {
		Pin *pins;
		
		pins = part_get_pins (part);
		x = part_pos.x + pins[i].offset.x;
		y = part_pos.y + pins[i].offset.y;

		/*
		 * Use the position of the pin as hash key. 
		 */
		lookup_key.x = x;
		lookup_key.y = y;

		/* 
		 * Retrieve a node for this position.
		 */
		node = node_store_get_or_create_node (store, lookup_key);

		/*
		 * Add all the wires that intersect this pin to the node store.
		 */
		wire_list = wires_at_pos (store, lookup_key);

		for (list = wire_list; list; list = list->next) {
			Wire *wire = list->data;

/*			g_print ("Add pin to wire.\n");*/
			
			node_add_wire (node, wire);
			wire_add_node (wire, node);
		}
		
		g_slist_free (wire_list);

		node_add_pin (node, &pins[i]);
	}

	gtk_object_set (GTK_OBJECT (part), "store", store, NULL);
	store->parts = g_list_prepend (store->parts, part);		
	store->items = g_list_prepend (store->items, part);		

	return TRUE;
}

int
node_store_remove_part (NodeStore *store, Part *part)
{
	Node *node;
	SheetPos lookup_key;
	SheetPos pos;
	gdouble x, y;
	int i, num_pins;

	g_return_val_if_fail (store != NULL, FALSE);
	g_return_val_if_fail (IS_NODE_STORE (store), FALSE);
	g_return_val_if_fail (part != NULL, FALSE);
	g_return_val_if_fail (IS_PART (part), FALSE);

	store->parts = g_list_remove (store->parts, part);		
	store->items = g_list_remove (store->items, part);		

	num_pins = part_get_num_pins (part);
	item_data_get_pos (ITEM_DATA (part), &pos);

	for (i = 0; i < num_pins; i++) {
		Pin *pins;

		pins = part_get_pins (part);

		x = pos.x + pins[i].offset.x;
		y = pos.y + pins[i].offset.y;

		/*
		 * Use the position of the pin as lookup key. 
		 */
		lookup_key.x = x;
		lookup_key.y = y;
		
		node = g_hash_table_lookup (store->nodes, &lookup_key);
		if (node) {
			if (!node_remove_pin (node, &pins[i])) {
				g_warning ("Couldn't remove pin.");
				return FALSE;
			}

			/* 
			 * If the node is empty after removing the pin, 
			 * remove the node as well.
			 */
			if (node_is_empty (node)) {
				g_hash_table_remove (store->nodes, &lookup_key);
				gtk_object_destroy (GTK_OBJECT (node));
			}
			
		} else {
			/* FIXME: Fix this or just silently return if the part is non-existant? */
/*			g_warning ("No node to remove part from.");*/
			return FALSE;
		}
	}

	return TRUE;
}

int
node_store_add_wire (NodeStore *store, Wire *wire)
{
	gdouble x1, y1, x2, y2;
	GSList *ip_list, *list;
	IntersectionPoint *ipoint;
	Node *node;
	WirePriv *priv;
	SheetPos pos, length;

	g_return_val_if_fail (store != NULL, FALSE);
	g_return_val_if_fail (IS_NODE_STORE (store), FALSE);
	g_return_val_if_fail (wire != NULL, FALSE);
	g_return_val_if_fail (IS_WIRE (wire), FALSE);

/*	if (wire_get_store (wire) != NULL) {
		g_warning ("Trying to add already stored wire.");
		return FALSE;
	}
*/

/*	g_print ("ADD WIRE\n");*/

	priv = wire->priv;

	wire_get_pos_and_length (wire, &pos, &length);

	x1 = pos.x;
	y1 = pos.y;
	x2 = x1 + length.x;
	y2 = y1 + length.y;

	/*
	 * Check for intersection with other wires.
	 */ 
	ip_list = wires_intersect (store, x1, y1, x2, y2);

	for (list = ip_list; list; list = list->next) {
		ipoint = list->data;
		
/*
		g_print ("(%g %g) (%g %g), ip (%g %g)\n", x1, y1, x2, y2, ipoint->pos.x, ipoint->pos.y);


		if (IS_EQ (x1, x2) && ((ipoint->pos.y == y1) || (ipoint->pos.y == y2))) {
			g_print ("vertical match.\n");
		} else if (IS_EQ (y1, y2) && ((ipoint->pos.x == x1) || (ipoint->pos.x == x2))) {
			g_print ("horizontal match.\n");

		} else if (IS_EQ (x1, ipoint->pos.x) && ((x2 == x1) || (x2 == ipoint->pos.x))) {
			g_print ("vertical match 2.\n");
		} else if (IS_EQ (y1, y2) && ((ipoint->pos.x == x1) || (ipoint->pos.x == x2))) {
			g_print ("horizontal match 2.\n");

		} else {
			g_print ("No match.\n");

			g_free (ipoint);
			continue;
		}
*/
		node = node_store_get_or_create_node (store, ipoint->pos);

		/*
		 * Add the wire, and also the wire that is intersected.
		 */
		node_add_wire (node, wire);
		node_add_wire (node, ipoint->wire);

		wire_add_node (wire, node);
		wire_add_node (ipoint->wire, node);

/*		g_print ("Add wire to wire.\n");*/

		g_free (ipoint);
	}
	g_slist_free (ip_list);

	/*
	 * Check for intersection with parts (pins).
	 */
	ip_list = wire_intersect_parts (store, wire);

	for (list = ip_list; list; list = list->next) {
		node = list->data;

		/*
		 * Add the wire to the node (pin) that it intersected.
		 */
		node_add_wire (node, wire);
		wire_add_node (wire, node);

/*		g_print ("Add wire to pin.\n");*/
	}

	g_slist_free (ip_list);

	gtk_object_set (GTK_OBJECT (wire), "store", store, NULL);
	store->wires = g_list_prepend (store->wires, wire);
	store->items = g_list_prepend (store->items, wire);

	return TRUE;
}

int
node_store_remove_wire (NodeStore *store, Wire *wire)
{
	gdouble x1, y1, x2, y2;
	GSList *list;
	SheetPos lookup_key, pos, length;
	WirePriv *priv;

	g_return_val_if_fail (store != NULL, FALSE);
	g_return_val_if_fail (IS_NODE_STORE (store), FALSE);
	g_return_val_if_fail (wire != NULL, FALSE);
	g_return_val_if_fail (IS_WIRE (wire), FALSE);

	priv = wire->priv;

/*	if (item_data_get_store (ITEM_DATA (wire)) == NULL) {
		g_warning ("Trying to remove non-stored wire.");
		return FALSE;
	}
*/

	wire_get_pos_and_length (wire, &pos, &length);

	x1 = pos.x;
	y1 = pos.y;
	x2 = x1 + length.x;
	y2 = y1 + length.y;

	store->wires = g_list_remove (store->wires, wire);
	store->items = g_list_remove (store->items, wire);

	/*
	 * If the nodes that this wire passes through will be
	 * empty when the wire is removed, remove the node as well.
	 */

	/*
	 * We must work on a copy of the nodes list, since it
	 * changes as we remove nodes.
	 */
	list = g_slist_copy (wire_get_nodes (wire));

	for (; list; list = list->next) {
		Node *node = list->data;

		lookup_key = node->key;

		node_remove_wire (node, wire);

		wire_remove_node (wire, node);

		if (node_is_empty (node))
			g_hash_table_remove (store->nodes, &lookup_key);
	}

	g_slist_free (list);

	return TRUE;
}

static GSList *
wire_intersect_parts (NodeStore *store, Wire *wire)
{
	GList *list;
	GSList *ip_list;
	Node *node;
	SheetPos lookup_pos;
	SheetPos part_pos, wire_pos, wire_length;
	Part *part;
	double x, y, wire_x1, wire_y1, wire_x2, wire_y2;
	int i, num_pins;

	g_return_val_if_fail (store != NULL, FALSE);
	g_return_val_if_fail (IS_NODE_STORE (store), FALSE);
	g_return_val_if_fail (wire != NULL, FALSE);
	g_return_val_if_fail (IS_WIRE (wire), FALSE);
	
	ip_list = NULL;

	wire_get_pos_and_length (wire, &wire_pos, &wire_length);

	wire_x1 = wire_pos.x;
	wire_x2 = wire_pos.x + wire_length.x;
	wire_y1 = wire_pos.y;
	wire_y2 = wire_pos.y + wire_length.y;

	/*
	 * Go through all the parts and see which of their
	 * pins that intersect the wire.
	 */
	for (list = store->parts; list; list = list->next) {
		part = list->data;

		num_pins = part_get_num_pins (part);
		item_data_get_pos (ITEM_DATA (part), &part_pos);

		for (i = 0; i < num_pins; i++) {
			Pin *pins;
		
			pins = part_get_pins (part);
			x = part_pos.x + pins[i].offset.x;
			y = part_pos.y + pins[i].offset.y;

			lookup_pos.x = x;
			lookup_pos.y = y;
			
			/* 
			 * If there is a wire at this pin's position,
			 * add it to the return list.
			 */
			if (is_wire_at_pos (wire_x1, wire_y1, wire_x2, wire_y2, lookup_pos)) {
				node = node_store_get_node (store, lookup_pos);

				if (node == NULL)
					g_warning ("Bug: Found no node at pin at (%g %g).\n", x, y);
				else
					ip_list = g_slist_prepend (ip_list, node);
			}
		}
	}

	return ip_list;
}

static GSList *
wires_at_pos (NodeStore *store, SheetPos pos) 
{
	GList *list;
	GSList *wire_list;
	Wire *wire;
	SheetPos wire_pos, wire_length;
	double x1, y1, x2, y2;

	g_return_val_if_fail (store != NULL, FALSE);
	g_return_val_if_fail (IS_NODE_STORE (store), FALSE);

	wire_list = NULL;

	for (list = store->wires; list; list = list->next) {
		wire = list->data;

		wire_get_pos_and_length (wire, &wire_pos, &wire_length);
		x1 = wire_pos.x;
		y1 = wire_pos.y;
		x2 = x1 + wire_length.x;
		y2 = y1 + wire_length.y;
		
		if (is_wire_at_pos (x1, y1, x2, y2, pos))
			wire_list = g_slist_prepend (wire_list, wire);
	}
	
	return wire_list;
}

int
node_store_is_wire_at_pos (NodeStore *store, SheetPos pos) 
{
	GList *list;
	Wire *wire;
	SheetPos wire_pos, wire_length;
	double x1, y1, x2, y2;

	g_return_val_if_fail (store != NULL, FALSE);
	g_return_val_if_fail (IS_NODE_STORE (store), FALSE);

	for (list = store->wires; list; list = list->next) {
		wire = list->data;

		wire_get_pos_and_length (wire, &wire_pos, &wire_length);
		x1 = wire_pos.x;
		y1 = wire_pos.y;
		x2 = x1 + wire_length.x;
		y2 = y1 + wire_length.y;

		if (is_wire_at_pos (x1, y1, x2, y2, pos))
			return TRUE;
	}
	
	return FALSE;
}

static GSList *
wires_intersect (NodeStore *store, double x1, double y1, double x2, double y2)
{
	GList *list;
	GSList *ip_list;
	Wire *wire2;
	SheetPos pos, wire2_pos, wire2_length;
	SheetPos wire1_start_pos, wire1_end_pos;
	double wire2_x1, wire2_y1, wire2_x2, wire2_y2;

	g_return_val_if_fail (store != NULL, FALSE);
	g_return_val_if_fail (IS_NODE_STORE (store), FALSE);

	wire1_start_pos.x = x1;
	wire1_start_pos.y = y1;
	wire1_end_pos.x = x2;
	wire1_end_pos.y = y2;
	
	/*
	 * Search through all the wires. Is there a better way?
	 */
	ip_list = NULL;
	for (list = store->wires; list; list = list->next) {
		wire2 = list->data;

		wire_get_pos_and_length (wire2, &wire2_pos, &wire2_length);
		wire2_x1 = wire2_pos.x;
		wire2_y1 = wire2_pos.y;
		wire2_x2 = wire2_x1 + wire2_length.x;
		wire2_y2 = wire2_y1 + wire2_length.y;

		if (do_wires_intersect (x1, y1, x2, y2, wire2_x1, wire2_y1, wire2_x2, wire2_y2, &pos)) {
			IntersectionPoint *ip;

			ip = g_new0 (IntersectionPoint, 1);

			ip->wire = wire2;
			ip->pos = pos;
			ip_list = g_slist_prepend (ip_list, ip);
		}
	}

	return ip_list;
}

/*
 * node_store_get_node
 *
 * Find the node that has an element at a certain position.
 */
Node *
node_store_get_node (NodeStore *store, SheetPos pos) 
{
	Node *node;

	g_return_val_if_fail (store != NULL, NULL);
	g_return_val_if_fail (IS_NODE_STORE (store), NULL);
	
	node = g_hash_table_lookup (store->nodes, &pos);
	
/*	if (!node)
		g_print ("No node at (%g, %g)\n", pos.x, pos.y);
	else
		g_print ("Found node at (%g, %g)\n", pos.x, pos.y);
*/
	return node;
}

static guint
node_hash (gconstpointer key)
{
	SheetPos *sp = (SheetPos *) key;
	int x, y;

	/* hasha p varannan bit? */

	x = (int)rint (sp->x) % 256;
	y = (int)rint (sp->y) % 256;

	return (y << 8) | x;
}

static int
node_equal (gconstpointer a, gconstpointer b)
{
	SheetPos *spa, *spb;

	spa = (SheetPos *) a;
	spb = (SheetPos *) b;

	if (fabs (spa->y - spb->y) > HASH_EPSILON) {
/*		if (fabs (spa->y - spb->y) < 2.0)
			g_print ("y mellan EPSILON och 2.0!\n");
*/
		return 0;
	}

	if (fabs (spa->x - spb->x) > HASH_EPSILON) {
/*		if (fabs (spa->x - spb->x) < 5.0)
			g_print ("x mellan EPSILON och 2.0!\n");
*/
		return 0;
	}

	return 1;
}

void
node_store_node_foreach (NodeStore *store, GHFunc *func, gpointer user_data)
{
	g_return_if_fail (store != NULL);
	g_return_if_fail (IS_NODE_STORE (store));

	g_hash_table_foreach (store->nodes, (gpointer)func, user_data);
}

static int
is_wire_at_pos (double x1, double y1, double x2, double y2, SheetPos pos)
{
	double k, x0, y0;

	x0 = pos.x;
	y0 = pos.y;

	if (!IS_EQ (x1, x2) && !IS_EQ (y1, y2)) {
		k = ((y2 - y1)) / ((x2 - x1));
		if (IS_EQ (y0, (y1 + k * (x0 - x1)))) {
			return TRUE;
		}
	} else if (IS_EQ (x2, x1) && IS_EQ (x1, x0)) {
		if (y0 >= y1 && y0 <= y2) {
			return TRUE;
		}
	} else if (IS_EQ (y2, y1) && IS_EQ (y1, y0)) {
		if (x0 >= x1 && x0 <= x2) {
			return TRUE;
		}
	}

//	g_print ("no match: (%g %g) -> (%g %g), (%g %g)\n", x1, y1, x2, y2, pos.x, pos.y);
	
	return FALSE;
}

/*
 * Decides if two wires intersect. Note that wires that share an
 * endpoint are considered intersecting each other. Intersection point
 * is returned in pos.
 */
static int
do_wires_intersect (double Ax, double Ay, double Bx, double By, double Cx, double Cy, double Dx, double Dy, SheetPos *pos)
{
	double r, s, d;

	/*
	 * Wires don't intersect if they share an endpoint. NOTE: they do here...
	 */
	if (SEP (Ax, Ay, Cx, Cy)) { /* same starting point */
		pos->x = Ax;
		pos->y = Ay;
		return TRUE;
	} else if (SEP (Ax, Ay, Dx, Dy)) { /* 1st start == 2nd end */
		pos->x = Ax;
		pos->y = Ay;
		return TRUE;
	} else if (SEP (Bx, By, Cx, Cy)) { /* 1st end == 2nd start */
		pos->x = Bx;
		pos->y = By;
		return TRUE;
	} else if (SEP (Bx, By, Dx, Dy)) { /* 1st end == 2nd end */
		pos->x = Bx;
		pos->y = By;
		return TRUE;
	}

	/*
	 * Calculate the denominator.
	 */
	d = ((Bx - Ax) * (Dy - Cy) - (By - Ay) * (Dx - Cx));

	/*
	 * We have two parallell wires if d = 0.
	 */
	if (fabs (d) < NODE_EPSILON) {
		return FALSE;
	}

	r = ((Ay - Cy) * (Dx - Cx) - (Ax - Cx) * (Dy - Cy));

	/*
	 * Colinear wires, if r = 0. FIXME: check for intersection?
	 *            not needed since we already checked for starts and ends
	 */
	if (fabs (r) < NODE_EPSILON) {
		
	}

	r = r / d;
	s = ((Ay - Cy) * (Bx - Ax) - (Ax - Cx) * (By - Ay)) / d;

	/*
	 * Check for intersection, which we have for values of
	 * r and s in [0, 1].
	 */
	if (r >= 0 &&
	    (r - 1.0) < NODE_EPSILON &&
	    s >= 0 && 
	    (s - 1.0) < NODE_EPSILON) {

		/*
		 * Calculate the intersection point.
		 */
		pos->x = Ax + r * (Bx - Ax);
		pos->y = Ay + r * (By - Ay);
		
		/* 
		   to be accepted only if it coincides with the start or end 
		   of any of the wires
		*/
		if ( SEP(pos->x,pos->y,Ax,Ay) ||
		     SEP(pos->x,pos->y,Bx,By) ||
		     SEP(pos->x,pos->y,Cx,Cy) ||
		     SEP(pos->x,pos->y,Dx,Dy) 
		     )

		return TRUE;
		else
		   return FALSE;
	}

	return FALSE;
}

GList *
node_store_get_parts (NodeStore *store)
{
	g_return_val_if_fail (store != NULL, NULL);
	g_return_val_if_fail (IS_NODE_STORE (store), NULL);

	return store->parts;
}

GList *
node_store_get_wires (NodeStore *store)
{
	g_return_val_if_fail (store != NULL, NULL);
	g_return_val_if_fail (IS_NODE_STORE (store), NULL);

	return store->wires;
}

GList *
node_store_get_items (NodeStore *store)
{
	g_return_val_if_fail (store != NULL, NULL);
	g_return_val_if_fail (IS_NODE_STORE (store), NULL);

	return store->items;
}

/*
 * Debugging code.
 */

void
node_store_dump_wires (NodeStore *store)
{
	GList *wires;

	g_print ("\n------------------- Dump wires -------------------\n");

	for (wires = store->wires; wires; wires = wires->next) {
		Wire *wire;
		SheetPos start, length;

		wire = wires->data;
		wire_get_pos_and_length (wire, &start, &length);

		g_print ("(%g %g) -> (%g %g):   %d nodes.\n", 
			 start.x, start.y,
			 start.x + length.x, start.y + length.y,
			 g_slist_length (wire->priv->nodes));
	}

	g_print ("\n");
}

static void
add_node (gpointer key, Node *node, GList **list)
{
	*list = g_list_prepend (*list, node);
}

static void
add_node_position (gpointer key, Node *node, GList **list)
{
	if (node_needs_dot (node))
		*list = g_list_prepend (*list, key);
}

GList *
node_store_get_node_positions (NodeStore *store)
{
	GList *result;

	g_return_val_if_fail (store != NULL, NULL);
	g_return_val_if_fail (IS_NODE_STORE (store), NULL);

	result = NULL;
	g_hash_table_foreach (store->nodes, (GHFunc) add_node_position, &result);

	return result;
}

GList *
node_store_get_nodes (NodeStore *store)
{
	GList *result;

	g_return_val_if_fail (store != NULL, NULL);
	g_return_val_if_fail (IS_NODE_STORE (store), NULL);

	result = NULL;
	g_hash_table_foreach (store->nodes, (GHFunc) add_node, &result);

	return result;
}

void
node_store_get_bounds (NodeStore *store, ArtDRect *rect)
{
	GList *list;
	SheetPos p1, p2;

	g_return_if_fail (store != NULL);
	g_return_if_fail (IS_NODE_STORE (store));
	g_return_if_fail (rect != NULL);

	rect->x0 = G_MAXDOUBLE;
	rect->y0 = G_MAXDOUBLE;
	rect->x1 = -G_MAXDOUBLE;
	rect->y1 = -G_MAXDOUBLE;
	
	for (list = store->items; list; list = list->next) {
		item_data_get_absolute_bbox (ITEM_DATA (list->data), &p1, &p2);

		rect->x0 = MIN (rect->x0, p1.x);
		rect->y0 = MIN (rect->y0, p1.y);
		rect->x1 = MAX (rect->x1, p2.x);
		rect->y1 = MAX (rect->y1, p2.y);
	}
}

gint
node_store_count_items (NodeStore *store, ArtDRect *rect)
{
	GList *list;
	SheetPos p1, p2;
	ItemData *data;
	gint n;

	g_return_val_if_fail (store != NULL, 0);
	g_return_val_if_fail (IS_NODE_STORE (store), 0);

	if (rect == NULL)
		return g_list_length (store->items);

	for (list = store->items, n = 0; list; list = list->next) {
		data = ITEM_DATA (list->data);
		item_data_get_absolute_bbox (data, &p1, &p2);
		if (p1.x <= rect->x1 && p1.y <= rect->y1 && p2.x >= rect->x0 && p2.y >= rect->y0) {
			n++;
/*
		if (p1.x >= rect->x0 && p1.y >= rect->y0 && p2.x <= rect->x1 && p2.y <= rect->y1) {
*/
		}
	}

	return n;
}

void
node_store_print_items (NodeStore *store, OreganoPrintContext *opc, ArtDRect *rect)
{
	GList *list;
	SheetPos p1, p2;
	ItemData *data;

	g_return_if_fail (store != NULL);
	g_return_if_fail (IS_NODE_STORE (store));
	g_return_if_fail (rect != NULL);

	for (list = store->items; list; list = list->next) {
		data = ITEM_DATA (list->data);
		item_data_get_absolute_bbox (data, &p1, &p2);
//		if (p1.x >= rect->x0 && p1.y >= rect->y0 && p2.x <= rect->x1 && p2.y <= rect->y1) {
		if (p1.x <= rect->x1 && p1.y <= rect->y1 && p2.x >= rect->x0 && p2.y >= rect->y0) {
			item_data_print (data, opc);
		}
	}
}
