/*
 * pngxrjpg.c - libpng external I/O: JPEG reader stub.
 * Copyright (C) 2001-2008 Cosmin Truta.
 *
 * From the compression point of view, JPEG-to-PNG conversion is not
 * a worthwhile task.  Moreover, the complexity of JPEG decoding
 * would add a significant overhead to the application.  As such, we
 * provide a stub that recognizes the JPEG format, without actually
 * processing it.
 */

#define PNGX_INTERNAL
#include "pngx.h"
#include "pngxtern.h"
#include <stdio.h>
#include <string.h>


/*
 * These are the various JPEG format signatures.
 *
 * FF D8 FF E0 ........................... JFIF
 * FF D8 FF E1 ........................... EXIF
 * FF D8 FF F7 ........................... JPEG-LS
 * FF 4F FF 51 ........................... JPEG-2000 codestream
 * 00 00 00 0C 6A 50 20 20 0D 0A 87 0A ... JPEG-2000 .jp2
 * 8B 4A 4E 47 0D 0A 1A 0A ............... JNG
 * 00 00 00 10 4A 48 44 52 ............... JNG datastream
 */

#define JPEG_SIG_JP2_SIZE 12
#define JPEG_SIG_JPC_SIZE 4
#define JPEG_SIG_JNG_SIZE 8
#define JPEG_SIG_SIZE_MAX 12

static const png_byte jpeg_sig_jp2[JPEG_SIG_JP2_SIZE] =
{ 0x00, 0x00, 0x00, 0x0c, 0x6a, 0x50, 0x20, 0x20, 0x0d, 0x0a, 0x87, 0x0a };
static const png_byte jpeg_sig_jpc[JPEG_SIG_JPC_SIZE] =
{ 0xff, 0x4f, 0xff, 0x51 };
static const png_byte jpeg_sig_jng[JPEG_SIG_JNG_SIZE] =
{ 0x8b, 0x4a, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a };
static const png_byte jpeg_sig_jng_jhdr[JPEG_SIG_JNG_SIZE] =
{ 0x00, 0x00, 0x00, 0x1a, 0x4a, 0x48, 0x44, 0x52 };

int /* PRIVATE */
pngx_sig_is_jpeg(const png_bytep sig, png_size_t sig_size,
                 png_charp fmt_name_buf, png_size_t fmt_name_buf_size,
                 png_charp fmt_desc_buf, png_size_t fmt_desc_buf_size)
{
   const char *fmt;
   int result;

   if (sig_size < JPEG_SIG_SIZE_MAX)
      return -1;
   if (sig[0] == 0xff && sig[1] == 0xd8 && sig[2] == 0xff &&
       ((int)sig[3] >= 0xe0 && (int)sig[3] <= 0xfd))
   {
      fmt = "JPEG";
      result = 1;  /* JFIF, EXIF, JPEG-LS, etc. */
   }
   else if (memcmp(sig, jpeg_sig_jp2, JPEG_SIG_JP2_SIZE) == 0 ||
            memcmp(sig, jpeg_sig_jpc, JPEG_SIG_JPC_SIZE) == 0)
   {
      fmt = "JPEG-2000";
      result = 2;  /* .jp2 or JPEG-2000 codestream */
   }
   else if (memcmp(sig, jpeg_sig_jng, JPEG_SIG_JNG_SIZE) == 0 ||
            memcmp(sig, jpeg_sig_jng_jhdr, JPEG_SIG_JNG_SIZE) == 0)
   {
      fmt = "JNG";
      result = 3;  /* JNG, standalone or datastream */
   }
   else
      return 0;  /* not JPEG */

   /* Store the format name. */
   if (fmt_name_buf != NULL)
   {
      PNGX_ASSERT(fmt_name_buf_size > strlen(fmt));
      strcpy(fmt_name_buf, fmt);
   }
   /* TODO: make fmt_desc more informative. */
   if (fmt_desc_buf != NULL)
   {
      PNGX_ASSERT(fmt_desc_buf_size > strlen(fmt));
      strcpy(fmt_desc_buf, fmt);
   }
   return result;
}


int /* PRIVATE */
pngx_read_jpeg(png_structp png_ptr, png_infop info_ptr, FILE *stream)
{
   png_byte buf[JPEG_SIG_SIZE_MAX];
   int sig_code;

   if (fread(buf, JPEG_SIG_SIZE_MAX, 1, stream) != 1)
      return 0;  /* not a JPEG file */
   sig_code = pngx_sig_is_jpeg(buf, JPEG_SIG_SIZE_MAX, NULL, 0, NULL, 0);
   /* TODO: use the format names passed by pngx_sig_is_jpeg. */
   switch (sig_code)
   {
   case 1:
      png_error(png_ptr, "JPEG decoding is not supported");
      break;
   case 2:
      png_error(png_ptr, "JPEG-2000 decoding is not supported");
      break;
   case 3:
      png_error(png_ptr, "JNG (JPEG) decoding is not supported");
      break;
   }
   if (info_ptr == NULL)  /* dummy, keep compilers happy */
      return 0;
   return 0;  /* always fail */
}
