//                                               -*- C++ -*-
/**
 *  @file  RandomVectorImplementation.hxx
 *  @brief An interface for all implementation class of random vectors
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: RandomVectorImplementation.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_RANDOMVECTORIMPLEMENTATION_HXX
#define OPENTURNS_RANDOMVECTORIMPLEMENTATION_HXX

#include "PersistentObject.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "Indices.hxx"
#include "NumericalMathFunction.hxx"
#include "CovarianceMatrix.hxx"
#include "Distribution.hxx"
#include "ComparisonOperator.hxx"
#include "FunctionalChaosResult.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {



      /**
       * @class RandomVectorImplementation
       *
       * An interface for all implementation class of random vectors
       */
      class RandomVectorImplementation 
	: public Base::Common::PersistentObject
      {
	CLASSNAME;

      public:

	/** Some typedefs to ease reading */
	typedef Base::Common::Pointer<RandomVectorImplementation> Antecedent;
	typedef Antecedent                                        Implementation;
	typedef Base::Type::NumericalPoint                        NumericalPoint;
	typedef Base::Type::Description                           Description;
	typedef Base::Type::Indices                               Indices;
	typedef Base::Stat::NumericalSample                       NumericalSample;
	typedef Base::Stat::CovarianceMatrix                      CovarianceMatrix;
	typedef Base::Func::NumericalMathFunction                 NumericalMathFunction;
	typedef Base::Common::ComparisonOperator                  ComparisonOperator;
	typedef Algorithm::FunctionalChaosResult                  FunctionalChaosResult;
	typedef Base::Common::InvalidArgumentException            InvalidArgumentException;
	typedef Base::Common::StorageManager                      StorageManager;

	/** Default constructor */
	RandomVectorImplementation(const String & name = OT::DefaultName);

	/** Virtual constructor */
	virtual RandomVectorImplementation * clone() const;

	/** String converter */
	String __repr__() const;




	/* Here is the interface that all derived class must implement */

	/** Is the underlying random vector composite ? */
	virtual Bool isComposite() const;

	/** Dimension accessor */
	virtual UnsignedLong getDimension() const;

	/** Realization accessor */
	virtual NumericalPoint getRealization() const;

	/** Numerical sample accessor */
	virtual NumericalSample getNumericalSample(const UnsignedLong size) const;

	/** Mean accessor */
	virtual NumericalPoint getMean() const;

	/** Covariance accessor */
	virtual CovarianceMatrix getCovariance() const;

	/** Get the random vector corresponding to the i-th marginal component */
	virtual Implementation getMarginal(const UnsignedLong i) const /* throw(InvalidArgumentException) */;

	/** Get the marginal random vector corresponding to indices components */
	virtual Implementation getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */;

	/** Fake method due to pbs with dynamic_cast and Pointer */
	virtual Antecedent getAntecedent() const;

	/** Fake method due to pbs with dynamic_cast and Pointer */
	virtual NumericalMathFunction getFunction() const;

	/** Fake method due to pbs with dynamic_cast and Pointer */
	virtual Distribution getDistribution() const;

	/** Fake method due to pbs with dynamic_cast and Pointer */
	virtual ComparisonOperator getOperator() const;

	/** Fake method due to pbs with dynamic_cast and Pointer */
	virtual NumericalScalar getThreshold() const;

	/** Description accessor */
	virtual void setDescription(const Description & description);
	virtual Description getDescription() const;

	/** Method save() stores the object through the StorageManager */
	virtual void save(StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	virtual void load(StorageManager::Advocate & adv);

      protected:


      private:

	/** Description of each component */
	Description description_;
    
      }; /* class RandomVectorImplementation */


    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_RANDOMVECTORIMPLEMENTATION_HXX */
