//                                               -*- C++ -*-
/**
 *  @file  UserDefined.hxx
 *  @brief The UserDefined distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: UserDefined.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_USERDEFINED_HXX
#define OPENTURNS_USERDEFINED_HXX

#include "OTprivate.hxx"
#include "DiscreteDistribution.hxx"
#include "Exception.hxx"
#include "PersistentCollection.hxx"
#include "PersistentObject.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class UserDefined
       *
       * The UserDefined distribution.
       */
      class UserDefined
        : public Model::DiscreteDistribution
      {
        CLASSNAME;
      public:

        typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
        typedef Model::DiscreteDistribution                          DiscreteDistribution; // required by SWIG
        typedef DiscreteDistribution::NumericalPoint                 NumericalPoint;
        typedef DiscreteDistribution::NumericalSample                NumericalSample;
        typedef DiscreteDistribution::CovarianceMatrix               CovarianceMatrix;
        typedef DiscreteDistribution::NumericalPointWithDescriptionCollection       NumericalPointWithDescriptionCollection;
        typedef DiscreteDistribution::NotDefinedException            NotDefinedException;

#ifndef SWIG
      public:

        class Pair
          : public Base::Common::PersistentObject
        {
          CLASSNAME;
        public:
          Pair() : x_(), p_() {}
          Pair(NumericalPoint x, NumericalScalar p) : x_(x), p_(p) {}
          virtual Pair * clone() const { return new Pair(*this); }
          virtual ~Pair() {}

          void setX(const NumericalPoint & x) { x_ = x ; }
          void setP(const NumericalScalar & p) { p_ = p ; }
          const NumericalPoint & getX() const { return x_ ; }
          const NumericalScalar & getP() const { return p_ ; }

          /** Method save() stores the object through the StorageManager */
          void save(StorageManager::Advocate & adv) const
          {
            PersistentObject::save(adv);
            adv.saveAttribute( "x_", x_ );
            adv.saveAttribute( "p_", p_ );
          }

          /** Method load() reloads the object from the StorageManager */
          void load(StorageManager::Advocate & adv)
          {
            PersistentObject::load(adv);
            adv.loadAttribute( "x_", x_ );
            adv.loadAttribute( "p_", p_ );
          }

	  virtual String __repr__() const
          {
            OSS oss;
            oss << "(" << x_ << "," << p_ << ")";
            return oss;
          }

        private:

          NumericalPoint x_;
          NumericalScalar p_;

        }; /* end class Pair */

#else
        typedef OT::Uncertainty::Distribution::UserDefinedPair Pair;
#endif

      public:

        typedef Base::Type::PersistentCollection<Pair>            PairCollection;
        typedef Base::Type::PersistentCollection<NumericalScalar> NumericalScalarCollection;

        /** Default constructor */
        UserDefined();

        /** Constructor from PairCollection */
        explicit UserDefined(const PairCollection & collection)
          /* throw (InvalidArgumentException) */;

        /** Constructor from a sample */
        explicit UserDefined(const NumericalSample & sample)
          /* throw (InvalidArgumentException) */;

        /** Constructor from a sample and associated weights */
        explicit UserDefined(const NumericalSample & sample,
			     const NumericalPoint & weights)
          /* throw (InvalidArgumentException) */;


        /** Comparison operator */
        Bool operator ==(const UserDefined & other) const;

        /** String converter */
	String __repr__() const;



        /* Interface inherited from Distribution */

        /** Virtual constructor */
        virtual UserDefined * clone() const;

        /** Get one realization of the distribution */
        virtual NumericalPoint getRealization() const;

        /** Get the PDF of the distribution */
        using DiscreteDistribution::computePDF;
        virtual NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the distribution */
        using DiscreteDistribution::computeCDF;
        virtual NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the PDFGradient of the distribution */
        virtual NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the CDFGradient of the distribution */
        virtual NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

        /** Get the mean of the distribution */
        virtual NumericalPoint getMean() const /* throw(NotDefinedException) */;

        /** Get the covariance of the distribution */
        virtual CovarianceMatrix getCovariance() const /* throw(NotDefinedException) */;

        /** Parameters value and description accessor */
        virtual NumericalPointWithDescriptionCollection getParametersCollection() const;

        /* Interface specific to UserDefined */

        /** Pair collection accessor */
        void setPairCollection(const PairCollection & collection)
          /* throw (InvalidArgumentException) */;
        PairCollection getPairCollection() const;


        /** Method save() stores the object through the StorageManager */
        virtual void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        virtual void load(StorageManager::Advocate & adv);

      protected:


      private:

        /** Quantile computation for dimension=1 */
        NumericalScalar computeScalarQuantile(const NumericalScalar prob,
					      const Bool tail = false,
                                              const NumericalScalar precision = DefaultQuantileEpsilon) const;

      private:

        /** The collection of couple (xi,pi) of the UserDefined distribution */
        PairCollection collection_;

        /** The cumulative probabilities si = sum(pk, k=0..i) */
        NumericalScalarCollection cumulativeProbabilities_;

	/** Flag to accelerate computations in case of uniform weights */
	mutable Bool hasUniformWeights_;

      }; /* class UserDefined */


      typedef UserDefined::Pair           UserDefinedPair;
      typedef UserDefined::PairCollection UserDefinedPairPersistentCollection;

#ifndef SWIG
      // Comparison operator for Pair objects
      Bool operator == (const UserDefined::Pair & lhs, const UserDefined::Pair & rhs);
#endif

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_USERDEFINED_HXX */
