//                                               -*- C++ -*-
/**
 *  @file  Curve.cxx
 *  @brief Curve class for curve plots
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: Curve.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "Curve.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Graph {

      CLASSNAMEINIT(Curve);

      static Common::Factory<Curve> RegisteredFactory("Curve");

      /* Default constructor */
      Curve::Curve(const String & legend):
        DrawableImplementation(NumericalSample(2, 0), legend)
      {
        // Nothing to do
      }

      /* Default constructor */
      Curve::Curve(const NumericalSample & data,
                   const String & legend):
        DrawableImplementation(NumericalSample(2, 0), legend)
      {
        NumericalSample dataFull;
        // If data is unidimensional, assume that it means Y values with indices as X values
        if (data.getDimension() == 1)
          {
            const UnsignedLong size(data.getSize());
            dataFull = NumericalSample(size, 2);
            for (UnsignedLong i = 0; i < size; ++i)
              {
                dataFull[i][0] = i;
                dataFull[i][1] = data[i][0];
              }
          }
        else dataFull = data;
        // Check data validity
        setData(dataFull);
      }

      /* Contructor from 2 data sets */
      Curve::Curve(const NumericalSample & dataX,
                   const NumericalSample & dataY,
                   const String & legend):
        DrawableImplementation(NumericalSample(2, 0), legend)
      {
        const UnsignedLong size(dataX.getSize());
        if (dataY.getSize() != size) throw InvalidArgumentException(HERE) << "Error: cannot build a Curve based on two numerical samples with different size.";
        if ((dataX.getDimension() != 1) || (dataY.getDimension() != 1)) throw InvalidArgumentException(HERE) << "Error: cannot build a Curve based on two numerical samples of dimension greater than 1.";
        NumericalSample dataFull(size, 2);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            dataFull[i][0] = dataX[i][0];
            dataFull[i][1] = dataY[i][0];
          }
        // Check data validity
        setData(dataFull);
      }

      /* Constructor with parameters */
      Curve::Curve(const NumericalSample & data,
                   const String & color,
                   const String & lineStyle,
                   const UnsignedLong lineWidth,
                   const String & legend)
      /* throw(InvalidArgumentException) */
        : DrawableImplementation(NumericalSample(2, 0), legend)
      {
        NumericalSample dataFull;
        // If data is unidimensional, assume that it means Y values with indices as X values
        if (data.getDimension() == 1)
          {
            const UnsignedLong size(data.getSize());
            dataFull = NumericalSample(size, 2);
            for (UnsignedLong i = 0; i < size; ++i)
              {
                dataFull[i][0] = i;
                dataFull[i][1] = data[i][0];
              }
          }
        else dataFull = data;
        // Check data validity
        setData(dataFull);
        setLineStyle(lineStyle);
        setLineWidth(lineWidth);
        setColor(color);
      }

      /* String converter */
      String Curve::__repr__() const
      {
        OSS oss;
        oss << "class=" << Curve::GetClassName()
            << " name=" << getName()
            << " derived from " << DrawableImplementation::__repr__();
        return oss;
      }

      /* Draw method */
      String Curve::draw() const
      {
        OSS oss;
        // Stores the data in a temporary file
        oss << DrawableImplementation::draw() << "\n";
        // The specific R command for drawing
        oss << "lines(data[,1], data[,2]"
            << ", lty=\"" << lineStyle_
            << "\", col=\"" << color_
            << "\", lwd=" << lineWidth_
            << ")";

        return oss;
      }

      /* Clone method */
      Curve * Curve::clone() const
      {
        return new Curve(*this);
      }
      /* Check validity of data */
      Bool Curve::isValidData(const NumericalSample & data) const
      {
        return (data.getDimension() == 2);
      }

      /* Method save() stores the object through the StorageManager */
      void Curve::save(StorageManager::Advocate & adv) const
      {
        DrawableImplementation::save(adv);
      }

      /* Method load() reloads the object from the StorageManager */
      void Curve::load(StorageManager::Advocate & adv)
      {
        DrawableImplementation::load(adv);
      }

    } /* namespace Graph */

  }/* namespace Base */

}/* namespace OpenTURNS */
