//                                               -*- C++ -*-
/**
 *  @file  XMLStorageManager.cxx
 *  @brief XMLStorageManager provides an interface for different storage classes
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: XMLStorageManager.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <iomanip>
#include "XMLStorageManager.hxx"
#include "PersistentObject.hxx"
#include "InterfaceObject.hxx"
#include "Exception.hxx"
#include "Catalog.hxx"
#include "Study.hxx"
#include "OSS.hxx"
#include "Log.hxx"
#include "XMLToolbox.hxx"

#ifndef WIN32
#ifndef SWIG
#ifndef XML_SUPPORTED
#error "XML support is mandatory. Check configuration."
#endif
#endif
#endif

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

#if defined HAVE_LIBXML2

      using OpenTURNS::Base::Common::NotYetImplementedException;




      /************ Tags ************/

      struct root_tag
      {
	static inline String Get() { return "openturns-study"; }
      };

#define DEFINE_TAG( name ) struct name ## _tag { static inline String Get() { return #name ; } }

      DEFINE_TAG( bool             );
      DEFINE_TAG( unsignedlong     );
      DEFINE_TAG( numericalscalar  );
      DEFINE_TAG( numericalcomplex );
      DEFINE_TAG( real             );
      DEFINE_TAG( imag             );
      DEFINE_TAG( string           );
      DEFINE_TAG( object           );


      /************ Attributes ************/

#define DEFINE_ATTRIBUTE( name ) struct name ## _attribute { static inline String Get() { return #name ; } }

      DEFINE_ATTRIBUTE( StudyVisible );
      DEFINE_ATTRIBUTE( StudyLabel );
      DEFINE_ATTRIBUTE( version );
      DEFINE_ATTRIBUTE( class );
      DEFINE_ATTRIBUTE( id );
      DEFINE_ATTRIBUTE( name );
      DEFINE_ATTRIBUTE( index );
      DEFINE_ATTRIBUTE( member );






      struct XMLInternalObject : public StorageManager::InternalObject
      {
	XML::Node node_;
	XMLInternalObject() : node_(0) {}
	XMLInternalObject(XML::Node node) : node_(node) {}
	virtual ~XMLInternalObject() throw() {}
	virtual XMLInternalObject * clone() const { return new XMLInternalObject(*this); }
	virtual void first() { node_ = XML::GetFirstChild( node_ ); }
	virtual void next() { node_ = XML::GetNextNode( node_ ); }
	virtual String __repr__() const { return OSS() << "XMLInternalObject { node = <" << node_ << ">}"; }
	virtual String __str__()  const { return __repr__(); }
      };

      struct XMLStorageManagerState : public StorageManager::InternalObject
      {
	XML::Node root_;
	XML::Node current_;
	XMLStorageManagerState() : root_(0), current_(0) {}
	virtual ~XMLStorageManagerState() throw() {}
	virtual XMLStorageManagerState * clone() const { return new XMLStorageManagerState(*this); }
	virtual void first() { current_ = XML::GetFirstChild( current_ ); }
	virtual void next() { current_ = XML::GetNextNode( current_ );}
	virtual String __repr__() const { return OSS() << "XMLStorageManagerState { root = <" << root_
						       << ">, current_ = <" << current_ << ">}"; }
	virtual String __str__()  const { return __repr__(); }
      };



      CLASSNAMEINIT(XMLStorageManager);
      const XMLStorageManager::VersionList XMLStorageManager::SupportedVersions;

      /* Default constructor */
      XMLStorageManager::XMLStorageManager(const FileName & filename)
	: StorageManager(1),
	  fileName_(filename),
 	  p_state_(new XMLStorageManagerState),
	  p_document_()
      {
	// Nothing to do
      }
      


      /*
       * Virtual constructor
       */
      XMLStorageManager * XMLStorageManager::clone() const
      {
	return new XMLStorageManager(*this);
      }

      /* String converter */
      String XMLStorageManager::__repr__() const
      {
	return OSS() << "class=" << getClassName();
      }


      /* Filename accessors */
      String XMLStorageManager::getFileName() const
      {
	return fileName_;
      }

      void XMLStorageManager::setFileName(const String & fileName)
      {
	fileName_ = fileName;
      }


      /* Return the current state of the storage manager (for those having one) */
      const StorageManager::InternalObject & XMLStorageManager::getState() const
      {
	assert(p_state_);
	return *p_state_;
      }



      /* Query the manager if the version is correct */
      Bool XMLStorageManager::canManageVersion(UnsignedLong version) const
      {
	return XMLStorageManager::SupportedVersions.contains(version);
      }


      /* Do some administrative tasks before saving/reloading */
      void XMLStorageManager::initialize(const SaveAction caller)
      {
 	cleanSavedObjects();
 	OSS oss;
 	oss << getStudyVersion();
 	p_document_.reset(new XMLDoc);
	assert(p_document_);
	assert(p_state_);
 	p_state_->root_ = XML::NewNode( root_tag::Get() );
 	XML::SetAttribute(p_state_->root_, version_attribute::Get(), oss);
 	XML::SetRootNode(*p_document_, p_state_->root_);
      }

      /* Do some administrative tasks before saving/reloading */
      void XMLStorageManager::initialize(const LoadAction caller)
      {
	p_document_.reset(new XMLDoc( fileName_ ));
      }

      /* Do some administrative tasks after saving/reloading */
      void XMLStorageManager::finalize(const SaveAction caller)
      {
	// Nothing to do
      }

      /* Do some administrative tasks after saving/reloading */
      void XMLStorageManager::finalize(const LoadAction caller)
      {
	// Nothing to do
      }



      /* Read and create the internal representation */
      void XMLStorageManager::read()
      {
	assert(p_document_);
	assert(p_state_);
 	p_state_->root_ = XML::GetRootNode( *p_document_ );
 	if (! p_state_->root_) throw StudyFileParsingException(HERE) << "Study file has no root element (" << fileName_ << ")";
 	if (! XML::IsElement( p_state_->root_, root_tag::Get() ))
 	  throw StudyFileParsingException(HERE) << "Can NOT find root element '" << root_tag::Get() << "' in file '" << fileName_ << "'. Got '" << XML::GetNodeName( p_state_->root_ ) << "'";
 	UnsignedLong version = 0;
 	String stul = XML::GetAttributeByName( p_state_->root_, version_attribute::Get() );
 	std::istringstream iss (stul);
	iss >> version;
 	setStudyVersion(version);
      }

      /* Write the internal representation */
      void XMLStorageManager::write()
      {
	assert(p_document_);
	p_document_->save( fileName_ );
      }


      /*
       * This method saves the PersistentObject onto the medium
       */
      void XMLStorageManager::save(const PersistentObject & obj, const String & label, bool fromStudy)
      {
#ifdef SAVELOAD_CHECKING
	try {
#endif
	  if (! isSavedObject( obj.getId() )) {
	    obj.save(*this, label, fromStudy);
	    markObjectAsSaved( obj.getId() );
	  }
#ifdef SAVELOAD_CHECKING
	} catch ( const Exception & ex ) {
	  Log::Error( OSS() << "When saving " << obj.getClassName() << " : " << ex );
	}
#endif
      }

      /*
       * This method reloads the PersistentObject from the medium
       */
      void XMLStorageManager::load(Study & study)
      {
	setStudy( &study );
	assert(p_state_);
 	XML::Node node = XML::GetFirstChild( p_state_->root_ );
 	while( node ) {
 	  p_state_->current_ = node;
 	  ReadObject ro = readDOMElement();
 	  if (ro.p_obj_) {
 	    study.add( ro.label_, *(ro.p_obj_) );
 	  }
 	  node = XML::GetNextNode(node);
 	}
      }




      /* Methods to read DOM elements */
      XMLStorageManager::ReadObject XMLStorageManager::readDOMElement()
      {
	assert(p_state_);
	ReadObject ro;
 	if ( p_state_->current_ ) {
 	  if (XML::IsElement(p_state_->current_, object_tag::Get() ) ) {
 	    String className = XML::GetAttributeByName(p_state_->current_, class_attribute::Get());
 	    ro.label_      = XML::GetAttributeByName(p_state_->current_, StudyLabel_attribute::Get());
 	    ro.visibility_ = XML::GetAttributeByName(p_state_->current_, StudyVisible_attribute::Get());
 	    ro.p_obj_      = Catalog::GetInstance().get(className).build(*this);
 	    if (! ro.visibility_.empty())
	      ro.p_obj_->setVisibility(ro.visibility_ == "true");
 	  }
 	}

	return ro;
      }




	
      /* Create a new empty object that will gather all saved information */
      Pointer<StorageManager::InternalObject> XMLStorageManager::createObject(const String & tag) const
      {
	return new XMLInternalObject( XML::NewNode(tag) );
      }


      /* Append an internal object to the collection of saved ones */
      void XMLStorageManager::appendObject(Pointer<InternalObject> & p_obj)
      {
	assert(p_state_);
	assert(p_obj);
	XMLInternalObject & obj = dynamic_cast<XMLInternalObject&>(*p_obj);
	XML::Node node = obj.node_;
	assert(node);
	XML::AddChild( p_state_->root_, node );
      }


      void XMLStorageManager::setVisibility(Pointer<InternalObject> & p_obj, Bool visible)
      {
	addAttribute( p_obj, StudyVisible_attribute::Get(), visible);
      }

      void XMLStorageManager::setLabel(Pointer<InternalObject> & p_obj, const String & label)
      {
	addAttribute( p_obj, StudyLabel_attribute::Get(), label);
      }



      /************ Helper Functions ************/


      /* toStringConverter */

      static inline
      void toStringConverter(const String & value, String & st)
      {
	st = value;
      }

      template <typename _Tp>
      static inline
      void toStringConverter(const _Tp & value, String & st)
      {
	st = ( OSS() << value );
      }



      /* fromStringConverter */

      static inline
      void fromStringConverter(const String & st, Bool & value)
      {
	value = (st == "true" ) ? true : false;
      }

      static inline
      void fromStringConverter(const String & st, String & value)
      {
	value = st;
      }

      template <typename _Tp>
      static inline
      void fromStringConverter(const String & st, _Tp & value)
      {
	std::istringstream iss (st);
	iss >> value;
      }



      /* getValueToConvert */

      template <typename TAG>
      static inline
      String getValueToConvert(XML::Node node)
      {
	return XML::GetNodeValue( node );
      }

      template <>
      inline
      String getValueToConvert<object_tag>(XML::Node node)
      {
	return XML::GetAttributeByName( node, id_attribute::Get() );
      }



      /* fromNodeConverter */

      template <typename TAG, typename _Tp>
      static inline
      void fromNodeConverter(XML::Node node, _Tp & value)
      {
	fromStringConverter( getValueToConvert<TAG>( node ), value );
      }


      template <>
      inline
      void fromNodeConverter<numericalcomplex_tag,NumericalComplex>(XML::Node node,
								    NumericalComplex & value)
      {
	XML::Node node_real = XML::FindElementByName( node, real_tag::Get() );
	XML::Node node_imag = XML::FindElementByName( node, imag_tag::Get() );
	NumericalScalar real;
	NumericalScalar imag;
	fromStringConverter( getValueToConvert<real_tag>( node_real ), real );
	fromStringConverter( getValueToConvert<imag_tag>( node_imag ), imag );
	value = NumericalComplex( real,imag );
      }




      /************ Generic Functions ************/


      /* AttributeWriter */

      template <typename _Tp>
      static inline
      void AttributeWriter(Pointer<StorageManager::InternalObject> & p_obj,
			   const String & name,
			   _Tp value)
      {
	assert(p_obj);
	XMLInternalObject & obj = dynamic_cast<XMLInternalObject&>(*p_obj);
	XML::Node node = obj.node_;
	assert(node);
	String attrValue;
	toStringConverter( value, attrValue );
	XML::SetAttribute( node, name, attrValue );
      }


      template <>
      inline
      void AttributeWriter<NumericalComplex>(Pointer<StorageManager::InternalObject> & p_obj,
					     const String & name,
					     NumericalComplex value)
      {
	assert(p_obj);
	XMLInternalObject & obj = dynamic_cast<XMLInternalObject&>(*p_obj);
	XML::Node node = obj.node_;
	assert(node);
	XML::Node child = XML::NewNode( numericalcomplex_tag::Get() );
	assert(child);
	XML::SetAttribute( child, name_attribute::Get(), name );
	XML::Node real = XML::NewNode( real_tag::Get(), OSS() << value.real() );
	assert(real);
	XML::AddChild( child, real );
	XML::Node imag = XML::NewNode( imag_tag::Get(), OSS() << value.imag() );
	assert(imag);
	XML::AddChild( child, imag );
	XML::AddChild( node, child );
      }




      /* AttributeReader */

      template <typename TAG, typename _Tp>
      static inline
      void AttributeReader(TAG tag,
			   Pointer<XMLStorageManagerState> & p_state,
			   const String & name,
			   _Tp & value)
      {
	assert(p_state);
	String st = XML::GetAttributeByName( p_state->current_, name );
	fromStringConverter( st, value );
      }




      /* IndexedValueWriter */

      template <typename TAG, typename _Tp>
      static inline
      void IndexedValueWriter(TAG tag,
			      Pointer<StorageManager::InternalObject> & p_obj,
			      UnsignedLong index,
			      _Tp value)
      {
	assert(p_obj);
	XMLInternalObject & obj = dynamic_cast<XMLInternalObject&>(*p_obj);
	XML::Node node = obj.node_;
	assert(node);
	String attrValue;
	toStringConverter( value, attrValue );
	XML::Node child = XML::NewNode( tag.Get(), attrValue );
	assert(child);
	XML::SetAttribute( child, index_attribute::Get(), OSS() << index );
	XML::AddChild( node, child );
      }
      

      template <>
      inline
      void IndexedValueWriter<numericalcomplex_tag,NumericalComplex>(numericalcomplex_tag tag,
								     Pointer<StorageManager::InternalObject> & p_obj,
								     UnsignedLong index,
								     NumericalComplex value)
      {
	assert(p_obj);
	XMLInternalObject & obj = dynamic_cast<XMLInternalObject&>(*p_obj);
	XML::Node node = obj.node_;
	assert(node);
	XML::Node child = XML::NewNode( numericalcomplex_tag::Get() );
	assert(child);
	XML::SetAttribute( child, index_attribute::Get(), OSS() << index );
	XML::Node real = XML::NewNode( real_tag::Get(), OSS() << value.real() );
	assert(real);
	XML::AddChild( child, real );
	XML::Node imag = XML::NewNode( imag_tag::Get(), OSS() << value.imag() );
	assert(imag);
	XML::AddChild( child, imag );
	XML::AddChild( node, child );
      }
      




      /* IndexedValueReader */

      template <typename TAG, typename _Tp>
      static inline
      void IndexedValueReader(TAG tag,
			      Pointer<StorageManager::InternalObject> & p_obj,
			      UnsignedLong index,
			      _Tp & value)
      {
	assert(p_obj);
	XMLStorageManagerState & state = dynamic_cast<XMLStorageManagerState &>(*p_obj);

	XML::Node node;
	while ( node = XML::FindNextElementByName( state.current_, tag.Get() ) ) {
	  UnsignedLong idx;
	  fromStringConverter( XML::GetAttributeByName(node, index_attribute::Get()), idx );
	  state.next();
	  if (idx == index) {
	    fromNodeConverter<TAG,_Tp>( node, value );
	    return;
	  }
	}

	Log::Warn( OSS() << "Failed when reading indexed value. Expected tag '" << tag.Get()
			  << "'. Got '" << XML::GetNodeName(state.current_) << "'" );
      }



      /* NamedObjectWriter */

      template <typename TAG, typename _Tp>
      static inline
      void NamedObjectWriter(TAG tag,
			     Pointer<StorageManager::InternalObject> & p_obj,
			     const String & name,
			     _Tp & value)
      {
	assert(p_obj);
	XMLInternalObject & obj = dynamic_cast<XMLInternalObject&>(*p_obj);
	XML::Node node = obj.node_;
	assert(node);
	XML::Node child = XML::NewNode( tag.Get() );
	assert(child);
	XML::SetAttribute( child, member_attribute::Get(), name);
	XML::SetAttribute( child, id_attribute::Get(), OSS() << value.getId() );
	XML::AddChild( node, child );
      }



      /* NamedObjectReader */

      template <typename TAG, typename _Tp>
      static inline
      int NamedObjectReader(TAG tag,
			    Pointer<StorageManager::InternalObject> & p_obj,
			    const String & name,
			    _Tp & value)
      {
	assert(p_obj);
	XMLStorageManagerState state = dynamic_cast<XMLStorageManagerState &>(*p_obj);

	XML::Node node = XML::FindElementByName( state.current_, tag.Get() );
	while ( node ) {
	  String nm;
	  fromStringConverter( XML::GetAttributeByName(node, member_attribute::Get()), nm );
	  if (nm == name) {
	    fromNodeConverter<TAG,_Tp>( node, value );
	    return 1;
	  }
	  node = XML::FindNextElementByName( node, tag.Get() );
	}

	return 0;
      }





      /************ Type = Bool ************/

      /* Add an attribute to an internal object */
      void XMLStorageManager::addAttribute(Pointer<InternalObject> & p_obj,
					   const String & name,
					   Bool value)
      {
	AttributeWriter( p_obj, name, value );
      }

      /* Read an attribute */
      void XMLStorageManager::readAttribute(Pointer<InternalObject> & p_obj,
					    const String & name,
					    Bool & value)
      {
	AttributeReader( bool_tag(), p_state_, name, value );
      }

      /* Add an indexed value to an internal object */
      void XMLStorageManager::addIndexedValue(Pointer<InternalObject> & p_obj,
					      UnsignedLong index,
					      Bool value)
      {
	IndexedValueWriter( bool_tag(), p_obj, index, value );
      }
      
      /* Read an indexed value */
      void XMLStorageManager::readIndexedValue(Pointer<InternalObject> & p_obj,
					       UnsignedLong index,
					       Bool & value)
      {
	IndexedValueReader( bool_tag(), p_obj, index, value );
      }




      /************ Type = UnsignedLong ************/

      
      /* Add an attribute to an internal object */
      void XMLStorageManager::addAttribute(Pointer<InternalObject> & p_obj,
					   const String & name,
					   UnsignedLong value)
      {
	AttributeWriter( p_obj, name, value );
      }

      /* Read an attribute */
      void XMLStorageManager::readAttribute(Pointer<InternalObject> & p_obj,
					    const String & name,
					    UnsignedLong & value)
      {
	AttributeReader( unsignedlong_tag(), p_state_, name, value );
      }

      /* Add an indexed value to an internal object */
      void XMLStorageManager::addIndexedValue(Pointer<InternalObject> & p_obj,
					      UnsignedLong index,
					      UnsignedLong value)
      {
	IndexedValueWriter( unsignedlong_tag(), p_obj, index, value );
      }
      
      /* Read an indexed value */
      void XMLStorageManager::readIndexedValue(Pointer<InternalObject> & p_obj,
					       UnsignedLong index,
					       UnsignedLong & value)
      {
	IndexedValueReader( unsignedlong_tag(), p_obj, index, value );
      }




      /************ Type = NumericalScalar ************/

      
      /* Add an attribute to an internal object */
      void XMLStorageManager::addAttribute(Pointer<InternalObject> & p_obj,
					   const String & name,
					   NumericalScalar value)
      {
	AttributeWriter( p_obj, name, value );
      }

      /* Read an attribute */
      void XMLStorageManager::readAttribute(Pointer<InternalObject> & p_obj,
					    const String & name,
					    NumericalScalar & value)
      {
	AttributeReader( numericalscalar_tag(), p_state_, name, value );
      }

      /* Add an indexed value to an internal object */
      void XMLStorageManager::addIndexedValue(Pointer<InternalObject> & p_obj,
					      UnsignedLong index,
					      NumericalScalar value)
      {
	IndexedValueWriter( numericalscalar_tag(), p_obj, index, value );
      }
      
      /* Read an indexed value */
      void XMLStorageManager::readIndexedValue(Pointer<InternalObject> & p_obj,
					       UnsignedLong index,
					       NumericalScalar & value)
      {
	IndexedValueReader( numericalscalar_tag(), p_obj, index, value );
      }




      /************ Type = NumericalComplex ************/

      
      /* Add an attribute to an internal object */
      void XMLStorageManager::addAttribute(Pointer<InternalObject> & p_obj,
					   const String & name,
					   NumericalComplex value)
      {
	AttributeWriter( p_obj, name, value );
      }

      /* Read an attribute */
      void XMLStorageManager::readAttribute(Pointer<InternalObject> & p_obj,
					    const String & name,
					    NumericalComplex & value)
      {
	AttributeReader( numericalcomplex_tag(), p_state_, name, value );
      }

      /* Add an indexed value to an internal object */
      void XMLStorageManager::addIndexedValue(Pointer<InternalObject> & p_obj,
					      UnsignedLong index,
					      NumericalComplex value)
      {
	IndexedValueWriter( numericalcomplex_tag(), p_obj, index, value );
      }
      
      /* Read an indexed value */
      void XMLStorageManager::readIndexedValue(Pointer<InternalObject> & p_obj,
					       UnsignedLong index,
					       NumericalComplex & value)
      {
	IndexedValueReader( numericalcomplex_tag(), p_obj, index, value );
      }




      /************ Type = String ************/

      
      /* Add an attribute to an internal object */
      void XMLStorageManager::addAttribute(Pointer<InternalObject> & p_obj,
					   const String & name,
					   const String & value)
      {
	AttributeWriter( p_obj, name, value );
      }

      /* Read an attribute */
      void XMLStorageManager::readAttribute(Pointer<InternalObject> & p_obj,
					    const String & name,
					    String & value)
      {
	AttributeReader( string_tag(), p_state_, name, value );
      }

      /* Add an indexed value to an internal object */
      void XMLStorageManager::addIndexedValue(Pointer<InternalObject> & p_obj,
					      UnsignedLong index,
					      const String & value)
      {
	IndexedValueWriter( string_tag(), p_obj, index, value );
      }
      
      /* Read an indexed value */
      void XMLStorageManager::readIndexedValue(Pointer<InternalObject> & p_obj,
					       UnsignedLong index,
					       String & value)
      {
	IndexedValueReader( string_tag(), p_obj, index, value );
      }




      /************ Type = InterfaceObject ************/

      
      /* Add an attribute to an internal object */
      void XMLStorageManager::addAttribute(Pointer<InternalObject> & p_obj,
					   const String & name,
					   const InterfaceObject & value)
      {
	value.save( *this );
	NamedObjectWriter( object_tag(), p_obj, name, value );
      }

      /* Read an attribute */
      void XMLStorageManager::readAttribute(Pointer<InternalObject> & p_obj,
					    const String & name,
					    InterfaceObject & value)
      {
	Id shadowedId;
	if (NamedObjectReader( object_tag(), p_obj, name, shadowedId )) {
	  if (! getStudy()->hasObject(shadowedId))
	    throw StudyFileParsingException(HERE) << "Element of id = " << shadowedId
						  << " in file '" << getFileName()
						  << "' referenced before used";
	  value.setImplementationAsPersistentObject(getStudy()->getObject(shadowedId));
	}
      }

      /* Add an indexed value to an internal object */
      void XMLStorageManager::addIndexedValue(Pointer<InternalObject> & p_obj,
					      UnsignedLong index,
					      const InterfaceObject & value)
      {
	value.save( *this );

	assert(p_obj);
	XMLInternalObject & obj = dynamic_cast<XMLInternalObject&>(*p_obj);
	XML::Node node = obj.node_;
	assert(node);
	XML::Node child = XML::NewNode( object_tag::Get() );
	assert(child);
	XML::SetAttribute( child, index_attribute::Get(), OSS() << index );
	XML::SetAttribute( child, id_attribute::Get(), OSS() << value.getId());
	XML::AddChild( node, child );
      }
      
      /* Read an indexed value */
      void XMLStorageManager::readIndexedValue(Pointer<InternalObject> & p_obj,
					       UnsignedLong index,
					       InterfaceObject & value)
      {
	Id shadowedId;
	IndexedValueReader( object_tag(), p_obj, index, shadowedId );
	if (! getStudy()->hasObject(shadowedId))
	  throw StudyFileParsingException(HERE) << "Element of id = " << shadowedId
						<< " in file '" << getFileName()
						<< "' referenced before used";
	value.setImplementationAsPersistentObject(getStudy()->getObject(shadowedId));
      }




      /************ Type = PersistentObject ************/

      
      /* Add an attribute to an internal object */
      void XMLStorageManager::addAttribute(Pointer<InternalObject> & p_obj,
					   const String & name,
					   const PersistentObject & value)
      {
	value.save( *this );
	NamedObjectWriter( object_tag(), p_obj, name, value );
      }

      /* Read an attribute */
      void XMLStorageManager::readAttribute(Pointer<InternalObject> & p_obj,
					    const String & name,
					    PersistentObject & value)
      {
	Id shadowedId;
	if (NamedObjectReader( object_tag(), p_obj, name, shadowedId )) {
	  if (! getStudy()->hasObject(shadowedId))
	    throw StudyFileParsingException(HERE) << "Element of id = " << shadowedId
						  << " in file '" << getFileName()
						  << "' referenced before used";
	  Study * p_study = getStudy();
	  Pointer<PersistentObject> p_po = p_study->getObject(shadowedId);
	  Catalog::GetInstance().get(value.getClassName()).assign(value, *p_po);
	}
      }

      /* Add an indexed value to an internal object */
      void XMLStorageManager::addIndexedValue(Pointer<InternalObject> & p_obj,
					      UnsignedLong index,
					      const PersistentObject & value)
      {
	value.save( *this );

	assert(p_obj);
	XMLInternalObject & obj = dynamic_cast<XMLInternalObject&>(*p_obj);
	XML::Node node = obj.node_;
	assert(node);
	XML::Node child = XML::NewNode( object_tag::Get() );
	assert(child);
	XML::SetAttribute( child, index_attribute::Get(), OSS() << index );
	XML::SetAttribute( child, id_attribute::Get(), OSS() << value.getId());
	XML::AddChild( node, child );
      }

      /* Read an indexed value */
      void XMLStorageManager::readIndexedValue(Pointer<InternalObject> & p_obj,
					       UnsignedLong index,
					       PersistentObject & value)
      {
	Id shadowedId;
	IndexedValueReader( object_tag(), p_obj, index, shadowedId );
	if (! getStudy()->hasObject(shadowedId))
	  throw StudyFileParsingException(HERE) << "Element of id = " << shadowedId
						<< " in file '" << getFileName()
						<< "' referenced before used";
	Study * p_study = getStudy();
	Pointer<PersistentObject> p_po = p_study->getObject(shadowedId);
	Catalog::GetInstance().get(value.getClassName()).assign(value, *p_po);
      }

 


#endif /* HAVE_LIBXML2 */

    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */
