/*************************************************************************
 *
 *  $RCSfile: socket.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: vg $ $Date: 2003/04/15 17:44:32 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef __SYSTEM_H
#include "system.h"
#endif

#ifndef _OSL_TIME_H_
#include <osl/time.h>
#endif

#include <stdio.h>
#include <string.h>
#include <winsock.h>

#ifndef __SOCKET_HXX
#include "socket.hxx"
#endif

#ifndef __SCHEDULE_HXX
#include "schedule.hxx"
#endif

// SALSocketAddress

// #define ACCEPT_POLLING

#define	WOULD_BLOCK(error) ((error) == WSAEWOULDBLOCK || (error) == 0)
//////////////////////////////////////////////////////////////////////////////
//
//	SALSocketAddress
//
//////////////////////////////////////////////////////////////////////////////


//----------------------------------------------------------------------------
//	SALSocketAddress::SALSocketAddress
//----------------------------------------------------------------------------

SALSocketAddress::SALSocketAddress()
{
	memset(&maImplData, 0, sizeof(maImplData));
}


//----------------------------------------------------------------------------
//	SALSocketAddress::SALSocketAddress
//----------------------------------------------------------------------------

SALSocketAddress::SALSocketAddress(unsigned short nFamily)
{
	maImplData.maSockAddr.sa_family = nFamily;
	memset(maImplData.maSockAddr.sa_data, 0, sizeof(maImplData.maSockAddr.sa_data));
}


//////////////////////////////////////////////////////////////////////////////
//
//	SALSocketAddress
//
//////////////////////////////////////////////////////////////////////////////

//----------------------------------------------------------------------------
//	SALInetAddress::SALInetAddress
//----------------------------------------------------------------------------

SALInetAddress::SALInetAddress(short nFamily)
{
	maImplData.maInetAddr.sin_family = nFamily;
	maImplData.maInetAddr.sin_port = INPORT_ANY;
	maImplData.maInetAddr.sin_addr.s_addr = INADDR_ANY;
	memset(maImplData.maInetAddr.sin_zero, 0, sizeof(maImplData.maInetAddr.sin_zero));
}


//----------------------------------------------------------------------------
//	SALInetAddress::setPort
//----------------------------------------------------------------------------

unsigned short SALInetAddress::setPort(unsigned short nPort)
{
	if (maImplData.maInetAddr.sin_family == AF_INET)
    	return ntohs(maImplData.maInetAddr.sin_port = htons(nPort));
	else
		return INPORT_NONE;
}

//----------------------------------------------------------------------------
//	SALInetAddress::setHost
//----------------------------------------------------------------------------

unsigned long SALInetAddress::setHost(unsigned long nHost)
{
	if (maImplData.maInetAddr.sin_family == AF_INET)
		return ntohl(maImplData.maInetAddr.sin_addr.S_un.S_addr = htonl(nHost));
	else
		return INADDR_NONE;
}

//----------------------------------------------------------------------------
//	SALInetAddress::setHost "172.017.001.128"
//----------------------------------------------------------------------------

unsigned long SALInetAddress::setHost(const sal_Char *pszHost)
{
	if (maImplData.maInetAddr.sin_family == AF_INET)
		return ntohl(maImplData.maInetAddr.sin_addr.S_un.S_addr = inet_addr(pszHost));
	else
		return INADDR_NONE;
}

//----------------------------------------------------------------------------
//	SALInetAddress::getPort
//----------------------------------------------------------------------------

unsigned short SALInetAddress::getPort() const
{
	if (maImplData.maInetAddr.sin_family == AF_INET)
    	return ntohs(maImplData.maInetAddr.sin_port);
	else
		return INPORT_NONE;
}

//----------------------------------------------------------------------------
//	SALInetAddress::getHost
//----------------------------------------------------------------------------

unsigned long SALInetAddress::getHost() const
{
	if (maImplData.maInetAddr.sin_family == AF_INET)
    	return ntohl(maImplData.maInetAddr.sin_addr.S_un.S_addr);
	else
		return INADDR_NONE;
}



//----------------------------------------------------------------------------
//	SALInetAddress::operator const sal_Char *
//----------------------------------------------------------------------------

SALInetAddress::operator const sal_Char *()
{
    return inet_ntoa(maImplData.maInetAddr.sin_addr);
}



//////////////////////////////////////////////////////////////////////////////
//
//	SALHostAddress
//
//////////////////////////////////////////////////////////////////////////////


//----------------------------------------------------------------------------
//	SALHostAddress::SALHostAddress
//----------------------------------------------------------------------------

SALHostAddress::SALHostAddress() :
	mpszHostName(NULL)
{
}

//----------------------------------------------------------------------------
//	SALHostAddress::SALHostAddress
//----------------------------------------------------------------------------

SALHostAddress::SALHostAddress(const sal_Char *pszHostName, unsigned long nHostAddr)
{
	SALInetAddress aInetAddr(AF_INET);

	aInetAddr.setHost(nHostAddr);
	maHostAddr = aInetAddr;

	mpszHostName = new sal_Char[strlen(pszHostName) + 1];
}

//----------------------------------------------------------------------------
//	SALHostAddress::SALHostAddress
//----------------------------------------------------------------------------

SALHostAddress::SALHostAddress(const sal_Char *pszHostName, const SALSocketAddress &rAddr)
{
	maHostAddr = rAddr;
	mpszHostName = new sal_Char[strlen(pszHostName) + 1];
	if (mpszHostName)
		strcpy(mpszHostName, pszHostName);
}

//----------------------------------------------------------------------------
//	SALHostAddress::~SALHostAddress
//----------------------------------------------------------------------------

SALHostAddress::~SALHostAddress()
{
	if (mpszHostName)
		delete mpszHostName;
}

//----------------------------------------------------------------------------
//	ResolveHostName
//----------------------------------------------------------------------------

SALHostAddress *ResolveHostName(const sal_Char *pszHostName)
{
	SALHostAddress	*pHost = NULL;
	SALThread 		*pThread = SCHEDULER->activeThread();
   	sal_Char			cBuffer[MAXGETHOSTSTRUCT];
	HOSTENT			*pHostEntry = (HOSTENT *)cBuffer;
	HTASK			hTask;

	hTask = WSAAsyncGetHostByName(SALSocket::getWindowHandle(), WM_WSAASYNC, pszHostName, cBuffer, sizeof(cBuffer));
	pThread->setWSAError(WSAENOERROR);
	pThread->registerCallback(SALThread::WSATaskType, hTask);
	SCHEDULER->dispatch();

	if (pThread->getWSAError() == WSAENOERROR)
		pHost = new SALHostAddress(pszHostName, ntohl(*(unsigned long *)pHostEntry->h_addr));

	return pHost;
}

//----------------------------------------------------------------------------
//	ResolveHostAddr(unsigned long)
//----------------------------------------------------------------------------

SALHostAddress *ResolveHostAddress(unsigned long nHost)
{
	SALHostAddress	*pHost = NULL;
	SALThread 		*pThread = SCHEDULER->activeThread();
   	sal_Char			cBuffer[MAXGETHOSTSTRUCT];
	HOSTENT			*pHostEntry = (HOSTENT *)cBuffer;
	unsigned long			nHostNet = htonl(nHost);
	HTASK			hTask;

	hTask = WSAAsyncGetHostByAddr(SALSocket::getWindowHandle(), WM_WSAASYNC, (sal_Char *)&nHostNet, sizeof(nHostNet), PF_INET, cBuffer, sizeof(cBuffer));
	pThread->setWSAError(WSAENOERROR);
	pThread->registerCallback(SALThread::WSATaskType, hTask);
	SCHEDULER->dispatch();

	if (pThread->getWSAError() == WSAENOERROR)
		pHost = new SALHostAddress(pHostEntry->h_name, nHost);

	return pHost;
}

//----------------------------------------------------------------------------
//	ResolveHostAddress(const SALSocketAddress &)
//----------------------------------------------------------------------------

SALHostAddress *ResolveHostAddress(const SALInetAddress &rAddr)
{
	return ResolveHostAddress(rAddr.getHost());
}


//----------------------------------------------------------------------------
//	GetLocalHost
//----------------------------------------------------------------------------

SALHostAddress *GetLocalHost()
{
	sal_Char cBuffer[256];

	if (gethostname (cBuffer, sizeof(cBuffer)) == SOCKET_ERROR)
		return NULL;
	else
		return new SALHostAddress(cBuffer, INADDR_ANY);
}


//----------------------------------------------------------------------------
//	GetServicePort
//----------------------------------------------------------------------------

short GetServicePort(const sal_Char *pszServiceName, const sal_Char *pszProtocol)
{
	SALThread 		*pThread = SCHEDULER->activeThread();
   	sal_Char			cBuffer[MAXGETHOSTSTRUCT];
	SERVENT			*pServiceEntry = (SERVENT *)cBuffer;
	HTASK			hTask;

	hTask = WSAAsyncGetServByName(SALSocket::getWindowHandle(), WM_WSAASYNC, pszServiceName, pszProtocol, cBuffer, sizeof(cBuffer));

	pThread->setWSAError(WSAENOERROR);
	pThread->registerCallback(SALThread::WSATaskType, hTask);
	SCHEDULER->dispatch();

	if (pThread->getWSAError() == WSAENOERROR)
		return ntohs(pServiceEntry->s_port);
	else
		return INPORT_NONE;
}



// Static members and methods

SALSocketQueue  SALSocket::gaWSASocketQueue;


//----------------------------------------------------------------------------
//	SALSocket::handleWSASelect
//----------------------------------------------------------------------------

BOOL SALSocket::handleWSASelect(SOCKET nSocket, UINT nEvent, UINT nError)
{
	SALSocket *pSocket = findPendingSocket(nSocket);

	if (pSocket)
	{
   	switch (nEvent)
      {
      	case FD_READ:
         case FD_ACCEPT:
         case FD_OOB:
         	if (pSocket->mpReadCondition)
         		pSocket->mpReadCondition->set();
            break;
         case FD_WRITE:
         case FD_CONNECT:
         	if (pSocket->mpWriteCondition)
         		pSocket->mpWriteCondition->set();
            break;
         case FD_CLOSE:
         	if (pSocket->mpWriteCondition)
	         	pSocket->mpWriteCondition->set();
         	if (pSocket->mpReadCondition)
            	pSocket->mpReadCondition->set();
            break;
         default:
         	return FALSE;
      }
      pSocket->mnEvent = nEvent;
		pSocket->mnError = nError;
		return TRUE;
	}

	return FALSE;
}

//----------------------------------------------------------------------------
//	SALSocket::handleWSAAsync
//----------------------------------------------------------------------------

BOOL SALSocket::handleWSAAsync(HTASK hTask, UINT, UINT nError)
{
	SALThread *pThread = SCHEDULER->findCallbackThread(SALThread::WSATaskType, hTask);

	if (pThread)
	{
		pThread->setWSAError(nError);
		pThread->unregisterCallback();
		return TRUE;
	}
	else
		return FALSE;

}

HWND SALSocket::ghWindow = NULL;

HWND SALSocket::getWindowHandle()
{
	HINSTANCE hInstance = GetAppInstance();

	if (ghWindow == NULL)
	{
		WNDCLASS    wndClass;

		wndClass.style = 0;
		wndClass.lpfnWndProc = (WNDPROC)MakeProcInstance((FARPROC)WindowProc, hInstance);
		wndClass.cbClsExtra = 0;
		wndClass.cbWndExtra = 0;
		wndClass.hInstance = hInstance;
		wndClass.hIcon = NULL;
		wndClass.hCursor = NULL;
		wndClass.hbrBackground = NULL;
		wndClass.lpszMenuName = NULL;
		wndClass.lpszClassName = SAL_SOCKET_CLASSNAME;

		RegisterClass(&wndClass);

	    ghWindow = CreateWindow(
			SAL_SOCKET_CLASSNAME,
			SAL_SOCKET_WINDOWNAME,
			WS_POPUP,
			0,0,0,0,
			NULL,
			NULL,
			hInstance,
			NULL
		);
	}

	return ghWindow;
}

LRESULT CALLBACK __loadds SALSocket::WindowProc(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)
{
	switch (wMsg)
	{
		case WM_WSASELECT:
			return handleWSASelect((SOCKET)wParam, WSAGETSELECTEVENT(lParam), WSAGETSELECTERROR(lParam));
		case WM_WSAASYNC:
			return handleWSAAsync((HTASK)wParam, WSAGETASYNCBUFLEN(lParam), WSAGETASYNCERROR(lParam));
		case WM_USER + 123:
			SCHEDULER->dispatch();
			return TRUE;
		default:
			break;
	}
	return DefWindowProc(hWnd, wMsg, wParam, lParam);
}

SALSocket *SALSocket::findPendingSocket(SOCKET nSocket)
{
	SALSocket *pSocket = gaWSASocketQueue.first();

	while (pSocket && pSocket->mnSocket != nSocket)
		pSocket = pSocket->next();

	return pSocket;
}

//----------------------------------------------------------------------------
//	SALSocket::SALSocket
//----------------------------------------------------------------------------

SALSocket::SALSocket(int nFamily, int nType, int nProtocol) :
	mnSocket(socket(nFamily, nType, nProtocol)),
    mbBlocking(TRUE),
    mpReadCondition(new SALCondition),
    mpWriteCondition(new SALCondition),
    mnError(WSAENOERROR),
    mnEvent(0)
{
	mbValid = mpReadCondition && mpWriteCondition && mnSocket != INVALID_SOCKET;

    unsigned long  Param = 1; // Non-Blocking, reuse Addresses

    if (ioctlsocket(mnSocket, FIONBIO, &Param) == SOCKET_ERROR)
        mnError = WSAGetLastError();
	else if (setsockopt(mnSocket, SOL_SOCKET, SO_REUSEADDR, (const sal_Char *)&Param, sizeof(Param)) == SOCKET_ERROR)
        mnError = WSAGetLastError();

// Wir sind jetzt fuer alle Events offen
    if (WSAAsyncSelect(mnSocket, getWindowHandle(), WM_WSASELECT, FD_ALL) == SOCKET_ERROR)
        mnError = WSAGetLastError();
    SALSocket::gaWSASocketQueue.append(this);
}

//----------------------------------------------------------------------------
//	SALSocket::SALSocket
//----------------------------------------------------------------------------

SALSocket::SALSocket(SOCKET nSocket) :
    mnSocket(nSocket),
    mbBlocking(TRUE),
    mpReadCondition(new SALCondition),
    mpWriteCondition(new SALCondition),
    mnError(WSAENOERROR),
    mnEvent(0)
{
	unsigned long Param = 1;
	setsockopt(mnSocket, IPPROTO_TCP, SO_REUSEADDR, (const sal_Char *)&Param, sizeof(Param));

   // Wir sind jetzt fuer alle Events offen
    if (WSAAsyncSelect(mnSocket, getWindowHandle(), WM_WSASELECT, FD_ALL) == SOCKET_ERROR)
        mnError = WSAGetLastError();
    SALSocket::gaWSASocketQueue.append(this);
}

//----------------------------------------------------------------------------
//	SALSocket::SALSocket
//----------------------------------------------------------------------------

SALSocket::SALSocket(const SALSocketAddress &raAddr) :
    mnSocket(socket(PF_INET, SOCK_STREAM, 0)),
    mbBlocking(TRUE),
    mpReadCondition(new SALCondition),
    mpWriteCondition(new SALCondition),
    mnError(WSAENOERROR),
    mnEvent(0)
{
	unsigned long Param = 1;
	setsockopt(mnSocket, IPPROTO_TCP, SO_REUSEADDR, (const sal_Char *)&Param, sizeof(Param));

   // Wir sind jetzt fuer alle Events offen
    if (WSAAsyncSelect(mnSocket, getWindowHandle(), WM_WSASELECT, FD_ALL) == SOCKET_ERROR)
        mnError = WSAGetLastError();
    SALSocket::gaWSASocketQueue.append(this);

	bind(raAddr);
}


//----------------------------------------------------------------------------
//	SALSocket::~SALSocket
//----------------------------------------------------------------------------

SALSocket::~SALSocket()
{
    SALSocket::gaWSASocketQueue.remove(this);
    WSAAsyncSelect(mnSocket, getWindowHandle(), 0, 0);

	// Durch das Lschen der Condition wird der Thread der darazuf wartet aufgeweckt

	if (mpReadCondition)
	{
    	delete mpReadCondition;
    	mpReadCondition = NULL;
	}

	// Hier ist unbedingt ein Threadwechsel erforderlich, damit die asynchrone Methode,
	// die evtl. gerade betrieben wird und in einem anderen Thread luft, beendet werden
	// kann BEVOR der Socket zerstrt wird.
	// Ansonsten htten wir dort einen ungltigen this-Pointer. Siehe dazu die Methoden
	// accept, connect, send, sendTo, read, readFrom etc.

	SCHEDULER->dispatch();

   // Dasselbe nochmal fuer die Write-Condition

	if (mpWriteCondition)
	{
    	delete mpWriteCondition;
    	mpWriteCondition = NULL;
	}

	SCHEDULER->dispatch();

	// Jetzt den Socket-Handle schliessen

	if (mnSocket)
	{
    	closesocket(mnSocket);
		mnSocket = 0;
	}
}




//----------------------------------------------------------------------------
//	SALSocket::getType
//----------------------------------------------------------------------------

int SALSocket::getType()
{
	int nType = 0;
	int result = readOption(SOL_SOCKET, SO_TYPE, &nType, sizeof(nType));

	if (result == SOCKET_ERROR)
		return -1;
	else
		return nType;
}

//----------------------------------------------------------------------------
//	SALSocket::registerWriteCondition
//----------------------------------------------------------------------------

void SALSocket::registerWriteCondition()
{
    mpWriteCondition->reset();
    mnError = WSAENOERROR;
}

//----------------------------------------------------------------------------
//	SALSocket::registerWriteCondition
//----------------------------------------------------------------------------

void SALSocket::registerReadCondition()
{
    mpReadCondition->reset();
    mnError = WSAENOERROR;
}

//----------------------------------------------------------------------------
//	SALSocket::unregisterReadCondition
//----------------------------------------------------------------------------

void SALSocket::unregisterReadCondition()
{
	mpReadCondition->reset();
}

//----------------------------------------------------------------------------
//	SALSocket::unregisterWriteCondition
//----------------------------------------------------------------------------

void SALSocket::unregisterWriteCondition()
{
	mpWriteCondition->reset();
}

//----------------------------------------------------------------------------
//	SALSocket::receive
//----------------------------------------------------------------------------

int SALSocket::receive(SALSocketAddress &rSocketAddr, void *pBuf, int maxlen, int nFlags)
{
	if (getType() == SOCK_STREAM)
		return receive(pBuf, maxlen, nFlags);

	STATUS eStatus = STATUS_OK;
    mnError = WSAENOERROR;

   registerReadCondition();

	int nAddrSize = sizeof(SOCKADDR);
    int nResult = ::recvfrom(mnSocket, (sal_Char *)pBuf, maxlen, nFlags, (SOCKADDR *)&rSocketAddr, &nAddrSize);
    while (STATUS_OK == eStatus && SOCKET_ERROR == nResult && WOULD_BLOCK((mnError = WSAGetLastError())) && isBlocking())
    {
        eStatus = waitReadCondition(INFINITE_DELAY);

		if (eStatus == STATUS_OK)
    		nResult = ::recvfrom(mnSocket, (sal_Char *)pBuf, maxlen, nFlags, (SOCKADDR *)&rSocketAddr, &nAddrSize);
        else if (eStatus == STATUS_TIMEOUT)
            mnError = WSAETIMEDOUT;
		else
			mnError = WSAEINTR;
    }

  	unregisterReadCondition();

    return nResult;
}

int SALSocket::receive(void *pBuf, int maxlen, int nFlags)
{
	STATUS eStatus = STATUS_OK;
    mnError = WSAENOERROR;

    registerReadCondition();

    int nResult = ::recv(mnSocket, (sal_Char *)pBuf, maxlen, nFlags);
    while (STATUS_OK == eStatus && SOCKET_ERROR == nResult && WOULD_BLOCK((mnError = WSAGetLastError())) && isBlocking())
    {
        eStatus = waitReadCondition(INFINITE_DELAY);

		if (eStatus == STATUS_OK)
        	nResult = ::recv(mnSocket, (sal_Char *)pBuf, maxlen, nFlags);
        else if (eStatus == STATUS_TIMEOUT)
            mnError = WSAETIMEDOUT;
		else
			mnError = WSAEINTR;
  		unregisterReadCondition();
    }

  	unregisterReadCondition();

    return nResult;
}

//----------------------------------------------------------------------------
//	SALSocket::send
//----------------------------------------------------------------------------

int SALSocket::send(const SALSocket &, const void *pBuf, int maxlen, int nFlags)
{
	return send(pBuf, maxlen, nFlags);
}

int SALSocket::send(const void *pBuf, int maxlen, int nFlags)
{
	STATUS eStatus = STATUS_OK;
    mnError = WSAENOERROR;

    registerWriteCondition();

    int nResult = ::send(mnSocket, (sal_Char *)pBuf, maxlen, nFlags);
    while (STATUS_OK == eStatus && SOCKET_ERROR == nResult && WOULD_BLOCK((mnError = WSAGetLastError())) && isBlocking())
    {
        eStatus = waitWriteCondition(INFINITE_DELAY);

		if (eStatus == STATUS_OK)
        	nResult = ::send(mnSocket, (sal_Char *)pBuf, maxlen, 0);
        else if (eStatus == STATUS_TIMEOUT)
            mnError = WSAETIMEDOUT;
		else
			mnError = WSAEINTR;
	    unregisterWriteCondition();

    }

    unregisterWriteCondition();

    return nResult;
}


//----------------------------------------------------------------------------
//	SALSocket::getAddresses
//----------------------------------------------------------------------------

int SALSocket::getAddresses(SALSocketAddress *pLocal, SALSocketAddress *pRemote)
{
	int namelen = sizeof(SOCKADDR);

	if (pLocal)
		getsockname(mnSocket, (SOCKADDR *)pLocal, &namelen);

	if (pRemote)
		getpeername(mnSocket, (SOCKADDR *)pRemote, &namelen);

	return 0;
}


//----------------------------------------------------------------------------
//	SALSocket::connect
//----------------------------------------------------------------------------

int SALSocket::connect(const SALSocketAddress &rAddr, unsigned long milliseconds)
{
	STATUS eStatus = STATUS_OK;
    mnError = WSAENOERROR;

    registerWriteCondition();

  	int nResult = ::connect(mnSocket, rAddr.getSockAddr(), sizeof(SOCKADDR));
	if (nResult == SOCKET_ERROR)
	{
		mnError = WSAGetLastError();
		if (WOULD_BLOCK(mnError) && isBlocking())
		{
			eStatus = waitWriteCondition(milliseconds);
			if (eStatus == STATUS_TIMEOUT)
				mnError = WSAETIMEDOUT;
			else if (eStatus == STATUS_OK && mnError == WSAENOERROR)
				nResult = 0;
			else
				mnError = WSAEINTR;
		}
	}
	else
		mnError = WSAENOERROR;

	unregisterWriteCondition();

    return nResult;
}

//----------------------------------------------------------------------------
//	SALSocket::shutdown
//----------------------------------------------------------------------------

int SALSocket::shutdown(int nDir)
{
	mnError = WSAENOERROR;

	int result = ::shutdown(mnSocket, nDir);

	if (result == SOCKET_ERROR)
		mnError = WSAGetLastError();

	return result;
}


//----------------------------------------------------------------------------
//	SALSocket::bind
//----------------------------------------------------------------------------

int SALSocket::bind(const SALSocketAddress &raAddr)
{
    int nResult = ::bind(mnSocket, raAddr.getSockAddr(), sizeof(struct sockaddr));

	if (SOCKET_ERROR == nResult)
		mnError = WSAGetLastError();
	else
		mnError = WSAENOERROR;

	return nResult;
}


//----------------------------------------------------------------------------
//	SALSocket::listen
//----------------------------------------------------------------------------

int SALSocket::listen(int backlog)
{
	mnError = WSAENOERROR;

	int result = ::listen(mnSocket, backlog);

	if (result == SOCKET_ERROR)
		mnError = WSAGetLastError();

	if (mnError == WSAENOERROR)
		registerReadCondition();

	return result;
}

//----------------------------------------------------------------------------
//	SALSocket::accept
//----------------------------------------------------------------------------

SALSocket *SALSocket::accept(SALSocketAddress *pAddr)
{
	SOCKET	nSocket;
	int		nSize = sizeof(struct sockaddr);

#ifdef ACCEPT_POLLING
#error
	do
	{
		nSocket = ::accept(mnSocket, (SOCKADDR *)pAddr, pAddr ? &nSize : NULL);
		if (nSocket == INVALID_SOCKET)
			mnError = WSAGetLastError();
		else
			mnError = WSAENOERROR;
		_yieldthread();
	} while (nSocket == INVALID_SOCKET && WOULD_BLOCK(mnError) && isBlocking());
#else
    registerReadCondition();

	nSocket = ::accept(mnSocket, (struct sockaddr *)pAddr, pAddr ? &nSize : NULL);
	if (nSocket == INVALID_SOCKET && WOULD_BLOCK((mnError = WSAGetLastError())) && isBlocking())
	{
		do
		{
			STATUS eStatus = waitReadCondition(INFINITE_DELAY);
			if (eStatus == STATUS_OK)
			{
				nSocket = ::accept(mnSocket, (struct sockaddr *)pAddr, pAddr ? &nSize : NULL);
				if (nSocket == INVALID_SOCKET)
					mnError = WSAGetLastError();
				else
					mnError = WSAENOERROR;
			}
			else if (eStatus == STATUS_TIMEOUT)
				mnError = WSAETIMEDOUT;
			else
				mnError = WSAEINTR;
		} while (nSocket == INVALID_SOCKET && WOULD_BLOCK(mnError));
    	unregisterReadCondition();
	}
    unregisterReadCondition();

#endif

    SALSocket *pSocket = (nSocket == INVALID_SOCKET) ? NULL : new SALSocket(nSocket);

	if (pSocket)
		pSocket->mbBlocking = mbBlocking;

	return pSocket;
}

//----------------------------------------------------------------------------
//	SALSocket::readOption
//----------------------------------------------------------------------------

int SALSocket::readOption(int nLevel, int nOption, void *pBuffer, int nBufSize)
{
	if (::getsockopt(mnSocket, nLevel, nOption, (sal_Char *)pBuffer, &nBufSize) == SOCKET_ERROR)
		return SOCKET_ERROR;
	else
		return nBufSize;
}

//----------------------------------------------------------------------------
//	SALSocket::writeOption
//----------------------------------------------------------------------------

int SALSocket::writeOption(int nLevel, int nOption, const void *pBuffer, int nBufSize)
{
	return ::setsockopt(mnSocket, nLevel, nOption, (const sal_Char *)pBuffer, nBufSize);
}

//////////////////////////////////////////////////////////////////////////////
//
//	SALSocketAddress
//
//////////////////////////////////////////////////////////////////////////////


//----------------------------------------------------------------------------
//	SALSocketSet::SALSocketSet
//----------------------------------------------------------------------------

SALSocketSet::SALSocketSet()
{
	clear();
}


//----------------------------------------------------------------------------
//	SALSocketSet::add
//----------------------------------------------------------------------------

void SALSocketSet::add(const SALSocket &rSocket)
{
	FD_SET(rSocket.mnSocket, &m_Set);
}


//----------------------------------------------------------------------------
//	SALSocketSet::remove
//----------------------------------------------------------------------------

void SALSocketSet::remove(const SALSocket &rSocket)
{
	FD_CLR(rSocket.mnSocket, &m_Set);
}


//----------------------------------------------------------------------------
//	SALSocketSet::clear
//----------------------------------------------------------------------------

void SALSocketSet::clear()
{
	FD_ZERO(&m_Set);
}


//----------------------------------------------------------------------------
//	SALSocketSet::contains
//----------------------------------------------------------------------------

BOOL SALSocketSet::contains(const SALSocket &rSocket)
{
	return FD_ISSET(rSocket.mnSocket, &m_Set);
}



//============================================================================
//
//      OSL-Interface
//
//============================================================================

#include <osl/socket.h>

extern "C" {


/**@{ begin section oslSocketAddr
*/


/**     Creates a socket-address for the given family.
        If family == osl_af_inet the address is set to INADDR_ANY
        port 0.
        @return 0 if address could not be created.
*/
oslSocketAddr SAL_CALL osl_createEmptySocketAddr(oslAddrFamily Family)
{
	switch(Family)
	{
		case osl_Socket_FamilyInet:
			return (oslSocketAddr)new SALSocketAddress(AF_INET);
		default:
			return (oslSocketAddr)NULL;
	}
}


/**     Create a new SocketAddress and fill it from Addr.
*/
oslSocketAddr SAL_CALL osl_copySocketAddr(oslSocketAddr Addr)
{
	if (Addr)
		return (oslSocketAddr)new SALSocketAddress(*(SALSocketAddress *)Addr);
	else
		return (oslSocketAddr)NULL;
}


/** Uses the systems name-service interface to find an address for pszHostname.
        @param pszHostname The name for which you search for an address.
        @return The desired address if one could be found, otherwise 0.
        Don't forget to destroy the address if you don't need it any longer.
*/
oslSocketAddr SAL_CALL osl_resolveHostname(const sal_Char* pszHostName)
{
	return (oslSocketAddr)ResolveHostName(pszHostName);
}


/**     Create an internet-address, consisting of hostaddress and port.
        We interpret pszDottedAddr as a dotted-decimal inet-addr
        (e.g. "141.99.128.50").
        @param pszDottedAddr String with dotted address.
        @param Port portnumber in host byte order.
        @return 0 if address could not be created.
*/
oslSocketAddr SAL_CALL osl_createInetSocketAddr(const sal_Char* pszDottedAddr, sal_Int32 Port)
{
	SALInetAddress *pAddr = new SALInetAddress(AF_INET);

	if (pAddr)
	{
		unsigned long host = pAddr->setHost(pszDottedAddr);
		if (host != INADDR_NONE)
			pAddr->setPort(Port);
		else
		{
			delete pAddr;
			pAddr = NULL;
		}
	}

	return (oslSocketAddr)pAddr;
}

/** Create an IPX address.
*/
oslSocketAddr SAL_CALL osl_createIpxSocketAddr(	const sal_Char	/*NetNumber*/[4],
										const sal_Char	/*NodeNumber*/[6],
										sal_uInt32	/*SocketNumber*/)
{
/* IPX is currently not supported by WinSock */
	return NULL;
}

/**     Free all memory allocated by pAddress.
*/
void SAL_CALL osl_destroySocketAddr(oslSocketAddr Addr)
{
	SALSocketAddress *pAddr = (SALSocketAddress *)Addr;

	if (pAddr)
		delete pAddr;
}



/**     Looks up the port-number designated to the specified service/protocol-pair.
        (e.g. "ftp" "tcp").
        @return OSL_INVALID_PORT if no appropriate entry was found, otherwise the port-number.
*/

sal_Int32 SAL_CALL osl_getServicePort(const sal_Char* pszService, const sal_Char* pszProtocol)
{
	short result = GetServicePort(pszService, pszProtocol);

	return (result == (short)INPORT_NONE) ? (sal_Int32)OSL_INVALID_PORT : (sal_Int32)result;
}




/** Retrieves the address-family from the Addr.
        @return the family of the socket-address.
        In case of an unknown family you get osl_invalid_AddrFamily.
*/
oslAddrFamily SAL_CALL osl_getFamilyOfSocketAddr(oslSocketAddr Addr)
{
	SALSocketAddress *pAddr = (SALSocketAddress *)Addr;

	if (pAddr)
	{
		switch (pAddr->getFamily())
		{
			case AF_INET:
				return osl_Socket_FamilyInet;
			default:
				return osl_Socket_FamilyInvalid;
		}
	}
	else
		return osl_Socket_FamilyInvalid;
}


/** Retrieves the internet port-number of Addr.
        @return the port-number of the address in host-byte order. If Addr
        is not an address of type osl_af_inet, it returns OSL_INVALID_PORT
*/
sal_Int32 SAL_CALL osl_getInetPortOfSocketAddr(oslSocketAddr Addr)
{
	SALInetAddress *pAddr = (SALInetAddress *)Addr;

	if (pAddr && pAddr->getFamily() == AF_INET)
		return pAddr->getPort();
	else
		return (int)OSL_INVALID_PORT;
}


/** Sets the Port of Addr.
        @param Port is expected in host byte-order.
        @return False if Addr is not an inet-addr.
*/
sal_Bool SAL_CALL osl_setInetPortOfSocketAddr(oslSocketAddr Addr, sal_Int32 Port)
{
	SALInetAddress *pAddr = (SALInetAddress *)Addr;

	if (pAddr && pAddr->getFamily() == AF_INET)
	{
		pAddr->setPort(Port);
		return sal_True;
	}
	else
		return sal_False;
}


/** Gets the hostname represented by Addr.
        @return the hostname represented by the address. If
        there is no hostname to be found, it returns 0.
*/
oslSocketResult SAL_CALL osl_getHostnameOfSocketAddr(oslSocketAddr Addr, sal_Char *pBuffer, sal_uInt32 BufferSize)
{
	if (Addr == NULL || pBuffer == NULL)
		return osl_Socket_Error;

	SALHostAddress	*pHostAddr = ResolveHostAddress(*(SALInetAddress *)Addr);

	if (pHostAddr)
	{
		strncpy(pBuffer, pHostAddr->getName(), BufferSize);
		delete pHostAddr;
		return osl_Socket_Ok;
	}
	else
		return osl_Socket_Error;
}


/** Gets the address in dotted decimal format.
        @return the dotted decimal address (e.g. 141.99.20.34) represented
        by the address. If the address is invalid or not of type osl_af_inet,
        it returns 0.
*/
oslSocketResult SAL_CALL osl_getDottedInetAddrOfSocketAddr(oslSocketAddr Addr, sal_Char *pBuffer, sal_uInt32 BufferSize)
{
	if (Addr)
	{
		const sal_Char *pszDotted = *(SALInetAddress *)Addr;

		if (pszDotted && pBuffer)
		{
			strncpy(pBuffer, pszDotted, BufferSize);
			return osl_Socket_Ok;
		}
		else
			return osl_Socket_Error;
	}
	else
		return osl_Socket_Error;
}


/** Gets the IPX Net-Number of the address.
        @return the (4 bytes long) net-number or 0 if not an IPX address.
*/
oslSocketResult SAL_CALL osl_getIpxNetNumber(oslSocketAddr /*Addr*/, oslSocketIpxNetNumber /*NetNumber*/)
{
	return osl_Socket_Error;
}

/** Gets the IPX Node-Number of the address.
        @return the (6 bytes long) net-number or 0 if not an IPX address.
*/
oslSocketResult SAL_CALL osl_getIpxNodeNumber(oslSocketAddr /*Addr*/, oslSocketIpxNodeNumber /*NodeNumber*/)
{
	return osl_Socket_Error;
}

/** Gets the IPX Socket-Number of the address.
        @return the IPX socket number or OSL_INVALID_IPX_SOCKET_NO if not an IPX address.
*/
sal_Int32 SAL_CALL osl_getIpxSocketNumber(oslSocketAddr /*Addr*/)
{
	return (sal_Int32)OSL_INVALID_IPX_SOCKET_NO;
}

sal_Bool SAL_CALL osl_isEqualSocketAddr(oslSocketAddr Addr1, oslSocketAddr Addr2)
{
    sockaddr* pAddr1 = (sockaddr*) Addr1;
	sockaddr* pAddr2 = (sockaddr*) Addr2;

//	OSL_ASSERT(pAddr1);
//	OSL_ASSERT(pAddr2);

	if (pAddr1->sa_family == pAddr2->sa_family)
	{
		switch (pAddr1->sa_family)
		{
			case AF_INET:
			{
				sockaddr_in* pInetAddr1= (sockaddr_in *) pAddr1;
				sockaddr_in* pInetAddr2= (sockaddr_in *) pAddr2;

				if ((pInetAddr1->sin_family == pInetAddr2->sin_family) &&
					(pInetAddr1->sin_addr.s_addr == pInetAddr2->sin_addr.s_addr) &&
					(pInetAddr1->sin_port == pInetAddr2->sin_port))
					return (sal_True);
			}

			default:
			{
				return (memcmp(pAddr1, Addr2, sizeof(struct sockaddr)) == 0);
			}
		}
	}
	return (sal_False);
}

/**@} end section oslSocketAddr
*/

/**@{ begin section oslHostAddr
*/

/*
        Opaque datatype HostAddr.
*/
typedef void* oslHostAddr;


/** Create an oslHostAddr from given hostname and socket address.
    @param pszHostname The hostname to be stored.
        @param Addr The socket address to be stored.
        @return The created address or 0 upon failure.
*/
oslHostAddr SAL_CALL osl_createHostAddr (const sal_Char *pszHostName, const oslSocketAddr Addr)
{
	SALHostAddress *pHostAddr = new SALHostAddress(pszHostName, *(SALSocketAddress *)Addr);

	return (oslHostAddr)pHostAddr;
}


/** Create an oslHostAddr by resolving the given pszHostname.
    Successful name resolution should result in the fully qualified
        domain name (FQDN) and it's address as hostname and socket address
        members of the resulting oslHostAddr.
    @param The hostname to be resolved.
        @return The resulting address or 0 upon failure.
*/
oslHostAddr SAL_CALL osl_createHostAddrByName (const sal_Char *pszHostName)
{
	SALHostAddress *pHostAddr = ResolveHostName(pszHostName);

	return (oslHostAddr)pHostAddr;
}


/** Create an oslHostAddr by reverse resolution of the given Addr.
    Successful name resolution should result in the fully qualified
        domain name (FQDN) and it's address as hostname and socket address
        members of the resulting oslHostAddr.
    @param The socket address to be reverse resolved.
        @return The resulting address or 0 upon failure.
*/
oslHostAddr SAL_CALL osl_createHostAddrByAddr (const oslSocketAddr Addr)
{
	SALHostAddress *pHostAddr = ResolveHostAddress(*(SALInetAddress *)Addr);

	return (oslHostAddr)pHostAddr;
}


/**     Create a copy of the given Addr.
    @return The copied address or 0 upon failure.
*/
oslHostAddr SAL_CALL osl_copyHostAddr(const oslHostAddr Addr)
{
	SALHostAddress *pHostAddr = new SALHostAddress(*(SALHostAddress *)Addr);

	return (oslHostAddr)pHostAddr;
}


/** Free all memory allocated by Addr.
*/
void SAL_CALL osl_destroyHostAddr (oslHostAddr Addr)
{
	SALHostAddress *pHostAddr = (SALHostAddress *)Addr;

	if (pHostAddr)
		delete pHostAddr;
}


/** Get the hostname member of Addr.
    @return The hostname or 0 upon failure.
*/
const sal_Char* SAL_CALL osl_getHostnameOfHostAddr (const oslHostAddr Addr)
{
	SALHostAddress *pHostAddr = (SALHostAddress *)Addr;

	if (pHostAddr)
		return pHostAddr->getName();
	else
		return NULL;
}


/** Get the socket address member of Addr.
    @return The socket address or 0 upon failure.
*/
oslSocketAddr SAL_CALL osl_getSocketAddrOfHostAddr (const oslHostAddr Addr)
{
	SALHostAddress *pHostAddr = (SALHostAddress *)Addr;

	if (pHostAddr)
		return (oslSocketAddr)pHostAddr->getAddress();
	else
		return NULL;
}

/** Retrieve this machines hostname.
    May not always be a fully qualified domain name (FQDN).
        @param  sal_Char *pBuffer  The name will be copied to this buffer.
        @param  int   BufferSize   The size of the buffer.
        @return True upon success, False otherwise.
*/
oslSocketResult SAL_CALL osl_getLocalHostname (sal_Char *pBuffer, sal_uInt32 BufferSize)
{
	SALHostAddress	*pLocalHost = GetLocalHost();

	if (pLocalHost)
	{
		strncpy(pBuffer, pLocalHost->getName(), BufferSize);
		return osl_Socket_Ok;
	}
	else
		return osl_Socket_Error;
}


/**@} end section oslHostAddr
*/

/**@{ begin section oslSocket
*/


/*-***************************************************************************/
/* oslSocket */
/*-***************************************************************************/

typedef void* oslSocket;

/** Create a socket of the specified Family and Type. The semantic of
        the Protocol parameter depends on the given family and type.
        @return 0 if socket could not be created, otherwise you get a handle
        to the allocated socket-datastructure.
*/

static int FAMILY_TO_NATIVE(oslAddrFamily Family)
{
	switch (Family)
	{
		case osl_Socket_FamilyInet:
			return AF_INET;
		case osl_Socket_FamilyIpx:
			return AF_IPX;
		default:
			return AF_UNSPEC;
	}
}

static int TYPE_TO_NATIVE(oslSocketType Type)
{
	switch (Type)
	{
		case osl_Socket_TypeStream:
			return SOCK_STREAM;
		case osl_Socket_TypeDgram:
			return SOCK_DGRAM;
		case osl_Socket_TypeRaw:
			return SOCK_RAW;
		case osl_Socket_TypeSeqPacket:
			return SOCK_RDM;
		case osl_Socket_TypeInvalid:
			return SOCK_SEQPACKET;
		default:
			return -1;
	}
}

static int PROTOCOL_TO_NATIVE(oslProtocol Protocol)
{
	switch (Protocol)
	{
		case osl_Socket_ProtocolIp:			/* for all af_inet */
			return IPPROTO_IP;
		default:
			return -1;
	}
}

oslSocket SAL_CALL osl_createSocket(	oslAddrFamily	Family,
							oslSocketType	Type,
							oslProtocol		Protocol)
{
	int	nFamily = FAMILY_TO_NATIVE(Family);
	int	nType = TYPE_TO_NATIVE(Type);
	int	nProtocol = PROTOCOL_TO_NATIVE(Protocol);

	SALSocket *pSocket = new SALSocket(nFamily, nType, nProtocol);
	if (pSocket && !pSocket->isValid())
	{
		delete pSocket;
		pSocket = NULL;
	}

	return (oslSocket)pSocket;
}

/** Create a socket as a copy of another.
        @return 0 if socket could not be created, otherwise you get a handle
        to the allocated socket-datastructure.
*/
oslSocket SAL_CALL osl_copySocket(oslSocket Socket)
{
	SALSocket *pSocket = new SALSocket(*(SALSocket *)Socket);
	return (oslSocket)pSocket;
}

/** Closes the socket and frees the Socket data-structure.
        For a graceful termination of a connection, you should call
        osl_shutdownSocket() first.
        @param Socket The Socket to be closed and destroyed.
*/
void SAL_CALL osl_destroySocket(oslSocket Socket)
{
	SALSocket *pSocket = (SALSocket *)Socket;
	if (pSocket)
		delete pSocket;
}


/**     Retrieves the Address of the local end of the socket.
        Note that a socket must be bound or connected before
        a vaild address can be returned.
        @return 0 if socket-address could not be created, otherwise you get
        the created Socket-Address.
*/
oslSocketAddr SAL_CALL osl_getLocalAddrOfSocket(oslSocket Socket)
{
	SALSocket	*pSocket = (SALSocket *)Socket;
	if (pSocket)
	{
		SALSocketAddress	*pAddress = new SALSocketAddress;
		if (pAddress)
		{
			if (pSocket->getAddresses(pAddress, NULL) == SOCKET_ERROR)
			{
				delete pAddress;
				pAddress = NULL;
			}
		}
		return (oslSocketAddr)pAddress;
	}
	else
		return (oslSocketAddr)NULL;
}

/**     Retrieves the Address of the remote end of the socket.
        Note that a socket must be connected before
        a vaild address can be returned.
        @return 0 if socket-address could not be created, otherwise you get
        the created Socket-Address.
*/
oslSocketAddr SAL_CALL osl_getPeerAddrOfSocket(oslSocket Socket)
{
	SALSocket	*pSocket = (SALSocket *)Socket;
	if (pSocket)
	{
		SALSocketAddress	*pAddress = new SALSocketAddress;
		if (pAddress)
		{
			if (pSocket->getAddresses(NULL, pAddress) == SOCKET_ERROR)
			{
				delete pAddress;
				pAddress = NULL;
			}
		}
		return (oslSocketAddr)pAddress;
	}
	else
		return (oslSocketAddr)NULL;
}

/** Binds the given address to the socket.
        @param Socket
        @param Address
        @return False if the bind failed.
*/
sal_Bool SAL_CALL osl_bindAddrToSocket(oslSocket Socket, oslSocketAddr Addr)
{
	SALSocket			*pSocket = (SALSocket *)Socket;
	SALSocketAddress	*pAddress = (SALSocketAddress *)Addr;
	if (pSocket && pAddress)
	{
		if (pSocket->bind(*pAddress) == SOCKET_ERROR)
			return sal_False;
		else
			return sal_True;
	}
	else
		return sal_False;
}

/** Connects the socket to the given address.

    @param Socket a bound socket.
    @param Addr the peer address.
        @param pTimeout Timeout value or NULL for blocking.

        @return osl_sock_result_ok on successful connection,
                osl_sock_result_timeout if operation timed out,
            osl_sock_result_interrupted if operation was interrupted
            osl_sock_result_error if the connection failed.
*/
oslSocketResult SAL_CALL osl_connectSocketTo(oslSocket Socket,
									oslSocketAddr Addr,
									const TimeValue* pTimeout)
{
	SALSocket			*pSocket = (SALSocket *)Socket;
	SALSocketAddress	*pAddress = (SALSocketAddress *)Addr;
	if (pSocket && pAddress)
	{
		long nTime = MILLISECONDS(pTimeout);
		if (pSocket->connect(*pAddress, nTime) == SOCKET_ERROR)
		{
			switch (pSocket->getError())
			{
				case WSAETIMEDOUT:
					return osl_Socket_TimedOut;
				case WSAEINTR:
					return osl_Socket_Interrupted;
				default:
					return osl_Socket_Error;
			}
		}
		else
			return osl_Socket_Ok;
	}
	else
		return osl_Socket_Error;
}


/** Prepares the socket to act as an acceptor of incoming connections.
        You should call "listen" before you use "accept".
        @param MaxPendingConnections denotes the length of the queue of
        pending connections for this socket. If MaxPendingConnections is
        -1, the systems default value will be used (Usually 5).
        @return False if the listen failed.
*/
sal_Bool SAL_CALL osl_listenOnSocket(oslSocket Socket, sal_Int32 MaxPendingConnections)
{
	SALSocket	*pSocket = (SALSocket *)Socket;
	if (pSocket)
	{
		if (MaxPendingConnections == -1)
			MaxPendingConnections = SOMAXCONN;
		if (pSocket->listen(MaxPendingConnections) == SOCKET_ERROR)
			return sal_False;
		else
			return sal_True;
	}
	else
		return sal_False;
}


/** Waits for an ingoing connection on the socket.
        This call blocks if there is no incoming connection present.
        @param if pAddr is != 0, the peers address is returned.
        @return 0 if the accept-call failed, otherwise you get a socket
        representing the new connection.
*/
oslSocket SAL_CALL osl_acceptConnectionOnSocket(oslSocket Socket, oslSocketAddr* pAddr)
{
	SALSocket	*pAcceptor = (SALSocket *)Socket;
	if (pAcceptor)
	{
		SALSocketAddress *pAddress;

		if (pAddr != NULL)
		{
			pAddress = new SALSocketAddress;
			if (pAddress == NULL)
				return (oslSocket)NULL;
		}
		else
			pAddress = NULL;

		SALSocket *pSocket = pAcceptor->accept(pAddress);

		if (pSocket)
		{
			if (pAddr)
				*pAddr = (oslSocketAddr)pAddress;
			return (oslSocket)pSocket;
		}
		else
		{
			if (pAddress)
				delete pAddress;
			return (oslSocket)NULL;
		}
	}
	else
		return (oslSocket)NULL;
}

/** Tries to receive BytesToRead data from the connected socket,
        if no error occurs. Note that incomplete recvs due to
        packet boundaries may occur.

        @param Socket [in] A connected socket to be used to listen on.
    @param pBuffer [out] Points to a buffer that will be filled with the received
        data.
        @param BytesToRead [in] The number of bytes to read. pBuffer must have at least
        this size.
        @param Flag [in] Modifier for the call. Valid values are:
        <ul>
        <li> osl_msg_normal,
        <li> osl_msg_oob,
        <li> osl_msg_peek,
        <li> osl_msg_dontroute,
        <li> osl_msg_maxiovlen,
        </ul>

    @return the number of received bytes.
*/

static int FLAG_TO_NATIVE(oslSocketMsgFlag Flag)
{
	switch (Flag)
	{
		case osl_Socket_MsgNormal:
			return 0;
		case osl_Socket_MsgOOB:
			return MSG_OOB;
		case osl_Socket_MsgPeek:
			return MSG_PEEK;
		case osl_Socket_MsgDontRoute:
			return MSG_DONTROUTE;
		case osl_Socket_MsgMaxIOVLen:
			return MSG_MAXIOVLEN;
		default:
			return -1;
	}
}

sal_Int32 SAL_CALL osl_receiveSocket(oslSocket Socket, void* pBuffer, sal_uInt32 BytesToRead, oslSocketMsgFlag Flag)
{
	SALSocket	*pSocket = (SALSocket *)Socket;

   if (BytesToRead > MAX_TCP_BUFFER_SIZE)
   	BytesToRead = MAX_TCP_BUFFER_SIZE;

	if (pSocket)
		return pSocket->receive(pBuffer, BytesToRead, FLAG_TO_NATIVE(Flag));
	else
		return -1;
}

/** Tries to receives BufferSize data from the (usually unconnected)
        (datagram-)socket, if no error occurs.

        @param Socket [in] A bound socket to be used to listen for a datagram.
        @param SenderAddr [out] An initialized oslSocketAddress. It will be
        filled with the address of the datagrams sender. If SenderAddr is 0,
        it is ignored.
        @param pBuffer [out] Points to a buffer that will be filled with the received
        datagram.
        @param BufferSize [in] The size of pBuffer.
        @param Flag [in] Modifier for the call. Valid values are:
        <ul>
        <li> osl_msg_normal,
        <li> osl_msg_oob,
        <li> osl_msg_peek,
        <li> osl_msg_dontroute,
        <li> osl_msg_maxiovlen,
        </ul>

        @return the number of received bytes.
*/
sal_Int32 SAL_CALL osl_receiveFromSocket(	oslSocket Socket, oslSocketAddr SenderAddr,
							void* pBuffer, sal_uInt32 BufferSize, oslSocketMsgFlag Flag)
{
	SALSocket	*pSocket = (SALSocket *)Socket;

   if (BufferSize > MAX_TCP_BUFFER_SIZE)
   	BufferSize = MAX_TCP_BUFFER_SIZE;

	if (pSocket)
		return pSocket->receive(*(SALSocketAddress *)SenderAddr, pBuffer, BufferSize, FLAG_TO_NATIVE(Flag));
	else
		return -1;
}

/** Tries to send BytesToSend data from the connected socket,
        if no error occurs.

        @param Socket [in] A connected socket.
    @param pBuffer [in] Points to a buffer that contains the send-data.
        @param BytesToSend [in] The number of bytes to send. pBuffer must have at least
        this size.
        @param Flag [in] Modifier for the call. Valid values are:
        <ul>
        <li> osl_msg_normal,
        <li> osl_msg_oob,
        <li> osl_msg_peek,
        <li> osl_msg_dontroute,
        <li> osl_msg_maxiovlen,
        </ul>

        @return the number of transfered bytes.
*/
sal_Int32 SAL_CALL osl_sendSocket(oslSocket Socket, const void* pBuffer, sal_uInt32 BytesToSend, oslSocketMsgFlag Flag)
{
	SALSocket	*pSocket = (SALSocket *)Socket;

   if (BytesToSend > MAX_TCP_BUFFER_SIZE)
   	BytesToSend = MAX_TCP_BUFFER_SIZE;

	if (pSocket)
		return pSocket->send(pBuffer, BytesToSend, FLAG_TO_NATIVE(Flag));
	else
		return -1;
}

/** Tries to send one datagram with BytesToSend data to the given ReceiverAddr
        via the (implicitly unconnected) datagram-socket.
        Since we only send one packet, we don't need to concern ourselfes here with
        incomplete sends due to packet boundaries.

    @param Socket [in] A bound or unbound socket. Socket will be bound
        after a successful call.

        @param ReceiverAddr [in] An initialized oslSocketAddress that contains
        the destination address for this send.

    @param pBuffer [in] Points to a buffer that contains the send-data.
        @param BytesToSend [in] The number of bytes to send. pBuffer must have at least
        this size.
        @param Flag [in] Modifier for the call. Valid values are:
        <ul>
        <li> osl_msg_normal,
        <li> osl_msg_oob,
        <li> osl_msg_peek,
        <li> osl_msg_dontroute,
        <li> osl_msg_maxiovlen,
        </ul>

        @return the number of transfered bytes.
*/
sal_Int32 SAL_CALL osl_sendToSocket(	oslSocket Socket, oslSocketAddr ReceiverAddr,
						const void* pBuffer, sal_uInt32 BytesToSend, oslSocketMsgFlag Flag)
{
	SALSocket	*pSocket = (SALSocket *)Socket;

   if (BytesToSend > MAX_TCP_BUFFER_SIZE)
   	BytesToSend = MAX_TCP_BUFFER_SIZE;

	if (pSocket)
		return pSocket->send(*(SALSocketAddress *)ReceiverAddr, pBuffer, BytesToSend, FLAG_TO_NATIVE(Flag));
	else
		return -1;
}

/** Checks if read operations will block.
        You can specify a timeout-value in seconds/microseconds that denotes
        how long the operation will block if the Socket is not ready.
        @return True if read operations (recv, recvFrom, accept) on the Socket
        will NOT block; False if it would block or if an error occured.

        @param Socket the Socket to perfom the operation on.
        @param pTimeout if NULL, the operation will block without a timeout. Otherwise
        the time define by timeout value.
*/
sal_Bool SAL_CALL osl_isReceiveReady(oslSocket Socket, const TimeValue* pTimeout)
{
	SALSocket	*pSocket = (SALSocket *)Socket;

	if (pSocket)
	{
		return (sal_Bool)(pSocket->waitReadCondition(MILLISECONDS(pTimeout)) == STATUS_OK);
	}
	return sal_False;
}

/** Checks if send operations will block.
        You can specify a timeout-value in seconds/microseconds that denotes
        how long the operation will block if the Socket is not ready.
        @return True if send operations (send, sendTo) on the Socket
        will NOT block; False if it would block or if an error occured.

        @param Socket the Socket to perfom the operation on.
        @param pTimeout if NULL, the operation will block without a timeout. Otherwise
        the time define by timeout value.
*/
sal_Bool SAL_CALL osl_isSendReady(oslSocket Socket, const TimeValue* pTimeout)
{
	SALSocket	*pSocket = (SALSocket *)Socket;

	if (pSocket)
	{
		return (sal_Bool)(pSocket->waitWriteCondition(MILLISECONDS(pTimeout)) == STATUS_OK);
	}
	return sal_False;
}

/** Checks if a request for out-of-band data will block.
        You can specify a timeout-value in seconds/microseconds that denotes
        how long the operation will block if the Socket has no pending OOB data.
        @return True if OOB-request operations (recv with appropriate flags)
        on the Socket will NOT block; False if it would block or if an error occured.

        @param Socket the Socket to perfom the operation on.
        @param pTimeout if NULL, the operation will block without a timeout. Otherwise
        the time define by timeout value.
*/
sal_Bool SAL_CALL osl_isExceptionPending(oslSocket Socket, const TimeValue* pTimeout)
{
	SALSocket	*pSocket = (SALSocket *)Socket;

	if (pSocket)
	{
		return (sal_Bool)(pSocket->waitReadCondition(MILLISECONDS(pTimeout)) == STATUS_OK);
	}
	return sal_False;
}

/** Shuts down communication on a connected socket.
        @param Direction denotes which end of the socket
        should be closed:
        <ul>
        <li> osl_shut_read      closes read operations.
        <li> osl_shut_write closes write operations.
        <li> osl_shut_readwrite closes read and write operations.
        </ul>
        @return True if the socket could be closed down.
*/

static int DIRECTION_TO_NATIVE(oslSocketDirection Direction)
{
	switch (Direction)
	{
		case osl_Socket_DirRead:
			return 0;
		case osl_Socket_DirWrite:
			return 1;
		case osl_Socket_DirReadWrite:
			return 2;
		default:
			return -1;
	}
}

sal_Bool SAL_CALL osl_shutdownSocket(oslSocket Socket, oslSocketDirection Direction)
{
	SALSocket *pSocket = (SALSocket *)Socket;

	if (pSocket)
		return pSocket->shutdown(DIRECTION_TO_NATIVE(Direction));
	else
		return sal_False;
}

/** Retrieves attributes associated with the socket.
        @param Socket is the socket to query.

        @param Level selects the level for which an option should be queried.
        Valid values are:
        <ul>
        <li> osl_sol_socket:    Socket Level
        <li> osl_sol_tcp:               Level of Transmission Control Protocol
        </ul>

        @param Option denotes the option to query.
        Valid values (depending on the Level) are:
        <ul>
        <li> osl_so_debug,
        <li> osl_so_acceptconn,
        <li> osl_so_reuseaddr,
        <li> osl_so_keepalive,
        <li> osl_so_dontroute,
        <li> osl_so_broadcast,
        <li> osl_so_useloopback,
        <li> osl_so_linger,
        <li> osl_so_oobinline,
        <li> osl_so_sndbuf,
        <li> osl_so_rcvbuf,
        <li> osl_so_sndlowat,
        <li> osl_so_rcvlowat,
        <li> osl_so_sndtimeo,
        <li> osl_so_rcvtimeo,
        <li> osl_so_error,
        <li> osl_so_type,
        <li> osl_so_tcp_nodelay,        (sol_tcp)
        </ul>
        If not above mentioned otherwise, the options are only valid for
        level sol_socket.

        @param pBuffer Pointer to a Buffer with enough room to take the desired
        attribute-value.

        @param BufferSize contains the length of the Buffer.

        @return -1 if an error occured or else the size of the data copied into
        pBuffer.
*/

static unsigned int LEVEL_TO_NATIVE(oslSocketOptionLevel Level)
{
	switch (Level)
	{
		case osl_Socket_LevelSocket:
			return SOL_SOCKET;
		case osl_Socket_LevelTcp:
			return IPPROTO_TCP;
		default:
			return 0;
	}
}

static unsigned int OPTION_TO_NATIVE(oslSocketOption Option)
{
	switch (Option)
	{
		case osl_Socket_OptionDebug:
			return SO_DEBUG;
		case osl_Socket_OptionAcceptConn:
			return SO_ACCEPTCONN;
		case osl_Socket_OptionReuseAddr:
			return SO_REUSEADDR;
		case osl_Socket_OptionKeepAlive:
			return SO_KEEPALIVE;
		case osl_Socket_OptionDontRoute:
			return SO_DONTROUTE;
		case osl_Socket_OptionBroadcast:
			return SO_BROADCAST;
		case osl_Socket_OptionUseLoopback:
			return SO_USELOOPBACK;
		case osl_Socket_OptionLinger:
			return SO_LINGER;
		case osl_Socket_OptionOOBinLine:
			return SO_OOBINLINE;
		case osl_Socket_OptionSndBuf:
			return SO_SNDBUF;
		case osl_Socket_OptionRcvBuf:
			return SO_RCVBUF;
		case osl_Socket_OptionSndLowat:
			return SO_SNDLOWAT;
		case osl_Socket_OptionRcvLowat:
			return SO_RCVLOWAT;
		case osl_Socket_OptionSndTimeo:
			return SO_SNDTIMEO;
		case osl_Socket_OptionRcvTimeo:
			return SO_RCVTIMEO;
		case osl_Socket_OptionError:
			return SO_ERROR;
		case osl_Socket_OptionType:
			return SO_TYPE;
		case osl_Socket_OptionTcpNoDelay:
			return TCP_NODELAY;
		default:
			return 0;
	}
}

sal_Int32 SAL_CALL osl_getSocketOption(oslSocket Socket, oslSocketOptionLevel Level,
						oslSocketOption Option, void* pBuffer, sal_uInt32 BufferLen)
{
	SALSocket *pSocket = (SALSocket *)Socket;

	if (pSocket)
	{
		int result = pSocket->readOption(LEVEL_TO_NATIVE(Level), OPTION_TO_NATIVE(Option), pBuffer, BufferLen);

		return result == SOCKET_ERROR ? -1 : result;
	}

	return -1;
}


/** Sets the sockets attributes.

        @param Socket is the socket to modify.

        @param Level selects the level for which an option should be changed.
        Valid values are:
        <ul>
        <li> osl_sol_socket:    Socket Level
        <li> osl_sol_tcp:               Level of Transmission Control Protocol
        </ul>

        @param Option denotes the option to modify.
        Valid values (depending on the Level) are:
        <ul>
        <li> osl_so_debug,
        <li> osl_so_acceptconn,
        <li> osl_so_reuseaddr,
        <li> osl_so_keepalive,
        <li> osl_so_dontroute,
        <li> osl_so_broadcast,
        <li> osl_so_useloopback,
        <li> osl_so_linger,
        <li> osl_so_oobinline,
        <li> osl_so_sndbuf,
        <li> osl_so_rcvbuf,
        <li> osl_so_sndlowat,
        <li> osl_so_rcvlowat,
        <li> osl_so_sndtimeo,
        <li> osl_so_rcvtimeo,
        <li> osl_so_error,
        <li> osl_so_type,
        <li> osl_so_tcp_nodelay,        (sol_tcp)
        </ul>
        If not above mentioned otherwise, the options are only valid for
        level sol_socket.

        @param pBuffer Pointer to a Buffer which contains the attribute-value.

        @param BufferSize contains the length of the Buffer.

        @return True if the option could be changed.
*/
sal_Bool SAL_CALL osl_setSocketOption(oslSocket Socket, oslSocketOptionLevel Level,
							oslSocketOption Option, void* pBuffer, sal_uInt32 BufferLen)
{
	SALSocket *pSocket = (SALSocket *)Socket;

	if (pSocket)
	{
		int result = pSocket->writeOption(LEVEL_TO_NATIVE(Level), OPTION_TO_NATIVE(Option), pBuffer, BufferLen);

		return result == SOCKET_ERROR ? sal_False : sal_True;
	}

	return sal_False;
}

/** Enables/disables non-blocking-mode of the socket.
        @param Socket Change mode for this socket.
        @param On True: enable non-blocking mode, False: disable non-blocking mode.
        @return True if mode could be changed.
*/
sal_Bool SAL_CALL osl_enableNonBlockingMode(oslSocket Socket, sal_Bool On)
{
	SALSocket *pSocket = (SALSocket *)Socket;

	if (pSocket)
	{
		pSocket->setBlocking(!On);
		return sal_True;
	}
	else
		return sal_False;
}


/** Query state of non-blocking-mode of the socket.
        @param Socket Query mode for this socket.
        @return True if non-blocking-mode is enabled.
*/
sal_Bool SAL_CALL osl_isNonBlockingMode(oslSocket Socket)
{
	SALSocket *pSocket = (SALSocket *)Socket;

	if (pSocket)
		return !pSocket->isBlocking();
	else
		return sal_False;
}


/** Queries the socket for its type.
        @return one of:
        <ul>
        <li> osl_sock_stream,
        <li> osl_sock_dgram,
        <li> osl_sock_raw,
        <li> osl_sock_rdm,
        <li> osl_sock_seqpacket,
        <li> osl_invalid_SocketType, if an error occured
        </ul>

*/
oslSocketType SAL_CALL osl_getSocketType(oslSocket Socket)
{
	SALSocket *pSocket = (SALSocket *)Socket;

	if (pSocket)
	{
		switch(pSocket->getType())
		{
			case SOCK_STREAM:
				return osl_Socket_TypeStream;
			case SOCK_DGRAM:
				return osl_Socket_TypeDgram;
			case SOCK_RAW:
				return osl_Socket_TypeRaw;
			case SOCK_RDM:
				return osl_Socket_TypeRdm;
			case SOCK_SEQPACKET:
				return osl_Socket_TypeSeqPacket;
			default:
				return osl_Socket_TypeInvalid;
		}
	}
	else
		return osl_Socket_TypeInvalid;
}

/** Delivers a string which describes the last socket error.
        @param pBuffer The string-buffer how receives the error message.
        @param BufferSize The length of pBuffer. If the message is longer
        than the provided buffer, it will be cut short. Buffer sizes about 128 chars
        should be large enough.
*/
void SAL_CALL osl_getLastSocketErrorDescription(oslSocket Socket, sal_Char* pBuffer, sal_uInt32 BufferSize)
{
	int error;

	/* ensure pBuffer will be zero-terminated even when strncpy has to cut */
	pBuffer[BufferSize-1]= '\0';

	switch(error = WSAGetLastError())
	{
		case WSAENOTSOCK:
			strncpy(pBuffer, "WSAENOTSOCK, Socket operation on non-socket. A socket created in one process is used by another process.", BufferSize-1);
            return;

        case WSAEDESTADDRREQ:
            strncpy(pBuffer, "WSAEDESTADDRREQ, Destination Addr required", BufferSize-1);
            return;

        case WSAEMSGSIZE:
            strncpy(pBuffer, "WSAEMSGSIZE, Message too long", BufferSize-1);
            return;

        case WSAEPROTOTYPE:
            strncpy(pBuffer, "WSAEPROTOTYPE, Protocol wrong type for socket", BufferSize-1);
            return;

        case WSAENOPROTOOPT:
            strncpy(pBuffer, "WSAENOPROTOOPT, Protocol not available", BufferSize-1);
            return;

        case WSAEPROTONOSUPPORT:
            strncpy(pBuffer, "WSAEPROTONOSUPPORT, Protocol not supported", BufferSize-1);
            return;

        case WSAESOCKTNOSUPPORT:
            strncpy(pBuffer, "WSAESOCKTNOSUPPORT, Socket type not supported", BufferSize-1);
            return;

        case WSAEOPNOTSUPP:
            strncpy(pBuffer, "WSAEOPNOTSUPP, Operation not supported on socket", BufferSize-1);
            return;

        case WSAEPFNOSUPPORT:
            strncpy(pBuffer, "WSAEPFNOSUPPORT, Protocol family not supported", BufferSize-1);
            return;

        case WSAEAFNOSUPPORT:
            strncpy(pBuffer, "WSEAFNOSUPPORT, Addr family not supported by protocol family", BufferSize-1);
            return;

        case WSAEADDRINUSE:
            strncpy(pBuffer, "WSAEADDRINUSE, Triggered by bind() because a process went down without closing a socket.", BufferSize-1);
            return;

        case WSAEADDRNOTAVAIL:
            strncpy(pBuffer, "WSAEADDRNOTAVAIL, Can't assign requested Addr", BufferSize-1);
            return;

        case WSAENETDOWN:
            strncpy(pBuffer, "WSAENETDOWN, Network is down", BufferSize-1);
            return;

        case WSAENETUNREACH:
            strncpy(pBuffer, "WSAENETUNREACH, Network is unreachable", BufferSize-1);
            return;

        case WSAENETRESET:
            strncpy(pBuffer, "WSAENETRESET, Network dropped connection or reset", BufferSize-1);
            return;

        case WSAECONNABORTED:
            strncpy(pBuffer, "WSAECONNABORTED, Software caused connection abort", BufferSize-1);
            return;

        case WSAECONNRESET:
            strncpy(pBuffer, "WSAECONNRESET, Connection reset by peer", BufferSize-1);
            return;

        case WSAENOBUFS:
            strncpy(pBuffer, "WSAENOBUFS, No buffer space available.", BufferSize-1);
            return;

        case WSAEISCONN:
            strncpy(pBuffer, "WSAEISCONN, Socket is already connected", BufferSize-1);
            return;

        case WSAENOTCONN:
            strncpy(pBuffer, "WSAENOTCONN, Socket is not connected", BufferSize-1);
            return;

        case WSAESHUTDOWN:
            strncpy(pBuffer, "WSAESHUTDOWN, Can't send after socket shutdown", BufferSize-1);
            return;

        case WSAETIMEDOUT:
            strncpy(pBuffer, "WSAETIMEDOUT, Connection timed out", BufferSize-1);
            return;

        case WSAECONNREFUSED:
            strncpy(pBuffer, "WSAECONNREFUSED, Connection refused", BufferSize-1);
            return;

        case WSAEHOSTDOWN:
            strncpy(pBuffer, "WSAEHOSTDOWN, Networking subsystem not started", BufferSize-1);
            return;

        case WSAEHOSTUNREACH:
            strncpy(pBuffer, "WSAEHOSTUNREACH, No route to host", BufferSize-1);
            return;

        case WSAEWOULDBLOCK:
            strncpy(pBuffer, "WSAEWOULDBLOCK, Operation would block", BufferSize-1);
            return;

        case WSAEINPROGRESS:

            strncpy(pBuffer, "WSAEINPROGRESS, Operation now in progress", BufferSize-1);
            return;

        case WSAEALREADY:
            strncpy(pBuffer, "WSAEALREADY, Operation already in progress", BufferSize-1);
            return;

        case WSAEINTR:
            strncpy(pBuffer, "WSAEALREADY, Operation was interrupted", BufferSize-1);
            return;

        case WSAEBADF:
            strncpy(pBuffer, "WSAEBADF, Bad file number", BufferSize-1);
            return;

        case WSAEACCES:
            strncpy(pBuffer, "WSAEACCES, Access is denied", BufferSize-1);
            return;

        case WSAEFAULT:
            strncpy(pBuffer, "WSAEFAULT, Bad memory Addr", BufferSize-1);
            return;

        case WSAEINVAL:
            strncpy(pBuffer, "WSAEINVAL, The socket has not been bound with bind() or is already connected", BufferSize-1);
            return;

        case WSAEMFILE:
            strncpy(pBuffer, "WSAEMFILE, No more file descriptors are available", BufferSize-1);
            return;

        case WSAETOOMANYREFS:
            strncpy(pBuffer, "WSAETOOMANYREFS, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAENAMETOOLONG:
            strncpy(pBuffer, "WSAENAMETOOLONG, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAENOTEMPTY:
            strncpy(pBuffer, "WSAENOTEMPTY, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAEPROCLIM:

            strncpy(pBuffer, "WSAEPROCLIM, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAEUSERS:
            strncpy(pBuffer, "WSAEUSERS, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAEDQUOT:
            strncpy(pBuffer, "WSAEDQUOT, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAESTALE:
            strncpy(pBuffer, "WSAESTALE, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAEREMOTE:
            strncpy(pBuffer, "WSAEREMOTE, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAEDISCON:
            strncpy(pBuffer, "WSAEDISCON, Circuit was gracefully terminated", BufferSize-1);
            return;

        case WSASYSNOTREADY:
            strncpy(pBuffer, "WSASYSNOTREADY, The underlying network subsystem is not ready for network communication", BufferSize-1);
            return;

        case WSAVERNOTSUPPORTED:
            strncpy(pBuffer, "WSAVERNOTSUPPORTED, The version of Windows Sockets API support requested is not provided by this particular Windows Sockets implementation", BufferSize-1);
            return;

        case WSANOTINITIALISED:
            strncpy(pBuffer, "WSANOTINITIALISED, WSAStartup() has not been called", BufferSize-1);
            return;

        case WSAHOST_NOT_FOUND:
            strncpy(pBuffer, "WSAHOST_NOT_FOUND, Authoritative answer host not found", BufferSize-1);
            return;

        case WSATRY_AGAIN:
            strncpy(pBuffer, "WSATRY_AGAIN, Non-authoritative answer host not found or SERVERFAIL", BufferSize-1);
            return;

        case WSANO_RECOVERY:
            strncpy(pBuffer, "WSANO_RECOVERY, Non recoverable errors, FORMERR, REFUSED, NOTIMP", BufferSize-1);
            return;

        case WSANO_DATA:
            strncpy(pBuffer, "WSANO_DATA or WSANO_ADDRESS, Valid name, no data record of requested type", BufferSize-1);
            return;

		default:
		{
			sal_Char message[128];

            sprintf(message, "Unknown WinSock Error Number %d", error);
		}
	}
}

/** Delivers a constant decribing the last error for the socket system.
        @return osl_ENONE if no error occured, osl_invalid_SocketError if
        an unknown (unmapped) error occured, otherwise an enum describing the
        error.
*/
oslSocketError SAL_CALL osl_getLastSocketError(oslSocket Socket)
{
	return (oslSocketError)GetLastSocketError();
}

/** Creates a set of sockets to be used with osl_demultiplexSocketEvents().
        @return A oslSocketSet or 0 if creation failed.
*/
oslSocketSet SAL_CALL osl_createSocketSet()
{
	return (oslSocketSet)new SALSocketSet;
}

/** Destroys a oslSocketSet.
*/
void SAL_CALL osl_destroySocketSet(oslSocketSet Set)
{
	if (Set)
		delete (SALSocketSet *)Set;
}

/** Clears the set from all previously added sockets.
        @param Set the set to be cleared.
*/
void SAL_CALL osl_clearSocketSet(oslSocketSet Set)
{
	if (Set)
		((SALSocketSet *)Set)->clear();
}


/** Adds a socket to the set.
        @param Set the set were the socket is added.
        @param Socket the socket to be added.
*/
void SAL_CALL osl_addToSocketSet(oslSocketSet Set, oslSocket Socket)
{
	if (Set && Socket)
		((SALSocketSet *)Set)->add(*(SALSocket *)Socket);
}

/** Removes a socket from the set.
        @param Set the set were the socket is removed from.
        @param Socket the socket to be removed.
*/
void SAL_CALL osl_removeFromSocketSet(oslSocketSet Set, oslSocket Socket)
{
	if (Set && Socket)
		((SALSocketSet *)Set)->remove(*(SALSocket *)Socket);
}

/** Checks if socket is in the set.
        @param Set the set to be checked.
        @param Socket check if this socket is in the set.
        @return True if socket is in the set.
*/
sal_Bool SAL_CALL osl_isInSocketSet(oslSocketSet Set, oslSocket Socket)
{
	if (Set && Socket)
		return (sal_Bool)((SALSocketSet *)Set)->contains(*(SALSocket *)Socket);
	else
		return sal_False;
}

/** Checks multiple sockets for events.
        @param IncomingSet Checks the sockets in this set
        for incoming events (read, accept). If the set is 0,
        it is just skipped.
        @param OutgoingSet Checks the sockets in this set
        for outgoing events (write, connect). If the set is 0,
        it is just skipped.
        @param OutOfBandSet Checks the sockets in this set
        for out-of-band events. If the set is 0, it is just skipped.
        @param msTimeout Number of milliseconds to wait for events. If
        msTimeout is -1, the call will block until an event or an error
        occurs.
        @return -1 on errors, otherwise the number of sockets with
        pending events. In case of timeout, the number might be 0.
*/
sal_Int32 SAL_CALL osl_demultiplexSocketEvents(oslSocketSet IncomingSet, oslSocketSet OutgoingSet,
								oslSocketSet OutOfBandSet,	const TimeValue* pTimeout)
{
//	#########################################

	struct timeval	tvZero = {0,0};
	long			nSockets = 0;
	DWORD			nStartTicks = GetTickCount();
	DWORD			nTimeout = MILLISECONDS(pTimeout);
	BOOL			fTimeout = FALSE;

	// Use a timeout value of Zero to select nonblocking
	// We use a polling method because the overhead would be to big if we use a
	// Condition for each Socket in the socket set.

	do
	{
		nSockets  = select( 0, (fd_set *)IncomingSet, (fd_set *)OutgoingSet, (fd_set *)OutOfBandSet, &tvZero);
		fTimeout =  (BOOL)((DWORD)(GetTickCount() - nStartTicks) > nTimeout);
	} while (nSockets == 0 && !fTimeout);

	return (int)nSockets;
}



/**@} end section oslSocket
*/





} // extern "C"




