// $Id: YM2413_2.cc 6917 2007-09-08 21:22:51Z m9710797 $

/*
 *
 * File: ym2413.c - software implementation of YM2413
 *                  FM sound generator type OPLL
 *
 * Copyright (C) 2002 Jarek Burczynski
 *
 * Version 1.0
 *
 *
 * TODO:
 *  - make sure of the sinus amplitude bits
 *  - make sure of the EG resolution bits (looks like the biggest
 *    modulation index generated by the modulator is 123, 124 = no modulation)
 *  - find proper algorithm for attack phase of EG
 *  - tune up instruments ROM
 *  - support sample replay in test mode (it is NOT as simple as setting bit 0
 *    in register 0x0f and using register 0x10 for sample data).
 *    Which games use this feature ?
 */

#include "YM2413_2.hh"
#include "YM2413Core.hh"
#include "FixedPoint.hh"
#include <cmath>

namespace openmsx {

// envelope output entries
const int ENV_BITS = 10;
const double ENV_STEP = 128.0 / (1 << ENV_BITS);

const int MAX_ATT_INDEX = (1 << (ENV_BITS - 2)) - 1;	// 255
const int MIN_ATT_INDEX = 0;

// sinwave entries
const int SIN_BITS = 10;
const int SIN_LEN  = 1 << SIN_BITS;
const int SIN_MASK = SIN_LEN - 1;

const int TL_RES_LEN = 256;	// 8 bits addressing (real chip)

// Slot offsets
const byte SLOT1 = 0;
const byte SLOT2 = 1;

// Envelope Generator phases
// Note: These are ordered: phase constants are compared in the code.
enum EnvelopeState {
	EG_DUMP, EG_ATTACK, EG_DECAY, EG_SUSTAIN, EG_RELEASE, EG_OFF
	};

enum KeyPart { KEY_MAIN = 1, KEY_RHYTHM = 2 };

// key scale level
// table is 3dB/octave, DV converts this into 6dB/octave
// 0.1875 is bit 0 weight of the envelope counter (volume) expressed
// in the 'decibel' scale
#define DV(x) int(x / 0.1875)
static const int ksl_tab[8 * 16] =
{
	// OCT 0
	DV( 0.000),DV( 0.000),DV( 0.000),DV( 0.000),
	DV( 0.000),DV( 0.000),DV( 0.000),DV( 0.000),
	DV( 0.000),DV( 0.000),DV( 0.000),DV( 0.000),
	DV( 0.000),DV( 0.000),DV( 0.000),DV( 0.000),
	// OCT 1
	DV( 0.000),DV( 0.000),DV( 0.000),DV( 0.000),
	DV( 0.000),DV( 0.000),DV( 0.000),DV( 0.000),
	DV( 0.000),DV( 0.750),DV( 1.125),DV( 1.500),
	DV( 1.875),DV( 2.250),DV( 2.625),DV( 3.000),
	// OCT 2
	DV( 0.000),DV( 0.000),DV( 0.000),DV( 0.000),
	DV( 0.000),DV( 1.125),DV( 1.875),DV( 2.625),
	DV( 3.000),DV( 3.750),DV( 4.125),DV( 4.500),
	DV( 4.875),DV( 5.250),DV( 5.625),DV( 6.000),
	// OCT 3
	DV( 0.000),DV( 0.000),DV( 0.000),DV( 1.875),
	DV( 3.000),DV( 4.125),DV( 4.875),DV( 5.625),
	DV( 6.000),DV( 6.750),DV( 7.125),DV( 7.500),
	DV( 7.875),DV( 8.250),DV( 8.625),DV( 9.000),
	// OCT 4
	DV( 0.000),DV( 0.000),DV( 3.000),DV( 4.875),
	DV( 6.000),DV( 7.125),DV( 7.875),DV( 8.625),
	DV( 9.000),DV( 9.750),DV(10.125),DV(10.500),
	DV(10.875),DV(11.250),DV(11.625),DV(12.000),
	// OCT 5
	DV( 0.000),DV( 3.000),DV( 6.000),DV( 7.875),
	DV( 9.000),DV(10.125),DV(10.875),DV(11.625),
	DV(12.000),DV(12.750),DV(13.125),DV(13.500),
	DV(13.875),DV(14.250),DV(14.625),DV(15.000),
	// OCT 6
	DV( 0.000),DV( 6.000),DV( 9.000),DV(10.875),
	DV(12.000),DV(13.125),DV(13.875),DV(14.625),
	DV(15.000),DV(15.750),DV(16.125),DV(16.500),
	DV(16.875),DV(17.250),DV(17.625),DV(18.000),
	// OCT 7
	DV( 0.000),DV( 9.000),DV(12.000),DV(13.875),
	DV(15.000),DV(16.125),DV(16.875),DV(17.625),
	DV(18.000),DV(18.750),DV(19.125),DV(19.500),
	DV(19.875),DV(20.250),DV(20.625),DV(21.000)
};
#undef DV

// sustain level table (3dB per step)
// 0 - 15: 0, 3, 6, 9,12,15,18,21,24,27,30,33,36,39,42,45 (dB)
#define SC(db) int((double(db)) / ENV_STEP)
static const int sl_tab[16] = {
	SC( 0),SC( 1),SC( 2),SC(3 ),SC(4 ),SC(5 ),SC(6 ),SC( 7),
	SC( 8),SC( 9),SC(10),SC(11),SC(12),SC(13),SC(14),SC(15)
};
#undef SC

static const byte eg_inc[15][8] =
{
	//cycle:0 1  2 3  4 5  6 7

	/* 0 */ { 0,1, 0,1, 0,1, 0,1, }, // rates 00..12 0 (increment by 0 or 1)
	/* 1 */ { 0,1, 0,1, 1,1, 0,1, }, // rates 00..12 1
	/* 2 */ { 0,1, 1,1, 0,1, 1,1, }, // rates 00..12 2
	/* 3 */ { 0,1, 1,1, 1,1, 1,1, }, // rates 00..12 3

	/* 4 */ { 1,1, 1,1, 1,1, 1,1, }, // rate 13 0 (increment by 1)
	/* 5 */ { 1,1, 1,2, 1,1, 1,2, }, // rate 13 1
	/* 6 */ { 1,2, 1,2, 1,2, 1,2, }, // rate 13 2
	/* 7 */ { 1,2, 2,2, 1,2, 2,2, }, // rate 13 3

	/* 8 */ { 2,2, 2,2, 2,2, 2,2, }, // rate 14 0 (increment by 2)
	/* 9 */ { 2,2, 2,4, 2,2, 2,4, }, // rate 14 1
	/*10 */ { 2,4, 2,4, 2,4, 2,4, }, // rate 14 2
	/*11 */ { 2,4, 4,4, 2,4, 4,4, }, // rate 14 3

	/*12 */ { 4,4, 4,4, 4,4, 4,4, }, // rates 15 0, 15 1, 15 2, 15 3 (incr by 4)
	/*13 */ { 8,8, 8,8, 8,8, 8,8, }, // rates 15 2, 15 3 for attack
	/*14 */ { 0,0, 0,0, 0,0, 0,0, }, // infinity rates for attack and decay(s)
};

// note that there is no value 13 in this table - it's directly in the code
static const byte eg_rate_select[16 + 64 + 16] =
{
	// Envelope Generator rates (16 + 64 rates + 16 RKS)
	// 16 infinite time rates
	14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,

	// rates 00-12
	 0, 1, 2, 3,
	 0, 1, 2, 3,
	 0, 1, 2, 3,
	 0, 1, 2, 3,
	 0, 1, 2, 3,
	 0, 1, 2, 3,
	 0, 1, 2, 3,
	 0, 1, 2, 3,
	 0, 1, 2, 3,
	 0, 1, 2, 3,
	 0, 1, 2, 3,
	 0, 1, 2, 3,
	 0, 1, 2, 3,

	// rate 13
	 4, 5, 6, 7,

	// rate 14
	 8, 9,10,11,

	// rate 15
	12,12,12,12,

	// 16 dummy rates (same as 15 3)
	12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,
};

//rate  0,    1,    2,    3,    4,   5,   6,   7,  8,  9, 10, 11, 12, 13, 14, 15
//shift 13,   12,   11,   10,   9,   8,   7,   6,  5,  4,  3,  2,  1,  0,  0,  0
//mask  8191, 4095, 2047, 1023, 511, 255, 127, 63, 31, 15, 7,  3,  1,  0,  0,  0

static const byte eg_rate_shift[16 + 64 + 16] =
{
	// Envelope Generator counter shifts (16 + 64 rates + 16 RKS)
	// 16 infinite time rates
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,

	// rates 00-12
	13,13,13,13,
	12,12,12,12,
	11,11,11,11,
	10,10,10,10,
	 9, 9, 9, 9,
	 8, 8, 8, 8,
	 7, 7, 7, 7,
	 6, 6, 6, 6,
	 5, 5, 5, 5,
	 4, 4, 4, 4,
	 3, 3, 3, 3,
	 2, 2, 2, 2,
	 1, 1, 1, 1,

	// rate 13
	 0, 0, 0, 0,

	// rate 14
	 0, 0, 0, 0,

	// rate 15
	 0, 0, 0, 0,

	// 16 dummy rates (same as 15 3)
	 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
};

// multiple table
#define ML(x) byte(2 * x)
static const byte mul_tab[16] =
{
	ML( 0.50), ML( 1.00), ML( 2.00), ML( 3.00),
	ML( 4.00), ML( 5.00), ML( 6.00), ML( 7.00),
	ML( 8.00), ML( 9.00), ML(10.00), ML(10.00),
	ML(12.00), ML(12.00), ML(15.00), ML(15.00),
};
#undef ML

//  TL_TAB_LEN is calculated as:
//  11 - sinus amplitude bits     (Y axis)
//  2  - sinus sign bit           (Y axis)
//  TL_RES_LEN - sinus resolution (X axis)
const int TL_TAB_LEN = 11 * 2 * TL_RES_LEN;
static int tl_tab[TL_TAB_LEN];

// sin waveform table in 'decibel' scale
// two waveforms on OPLL type chips
static unsigned sin_tab[SIN_LEN * 2];

// LFO Amplitude Modulation table (verified on real YM3812)
// 27 output levels (triangle waveform);
// 1 level takes one of: 192, 256 or 448 samples
//
// Length: 210 elements.
//
//  Each of the elements has to be repeated
//  exactly 64 times (on 64 consecutive samples).
//  The whole table takes: 64 * 210 = 13440 samples.
//
// We use data>>1, until we find what it really is on real chip...

static const int LFO_AM_TAB_ELEMENTS = 210;
static const byte lfo_am_table[LFO_AM_TAB_ELEMENTS] =
{
	0,0,0,0,0,0,0,
	1,1,1,1,
	2,2,2,2,
	3,3,3,3,
	4,4,4,4,
	5,5,5,5,
	6,6,6,6,
	7,7,7,7,
	8,8,8,8,
	9,9,9,9,
	10,10,10,10,
	11,11,11,11,
	12,12,12,12,
	13,13,13,13,
	14,14,14,14,
	15,15,15,15,
	16,16,16,16,
	17,17,17,17,
	18,18,18,18,
	19,19,19,19,
	20,20,20,20,
	21,21,21,21,
	22,22,22,22,
	23,23,23,23,
	24,24,24,24,
	25,25,25,25,
	26,26,26,
	25,25,25,25,
	24,24,24,24,
	23,23,23,23,
	22,22,22,22,
	21,21,21,21,
	20,20,20,20,
	19,19,19,19,
	18,18,18,18,
	17,17,17,17,
	16,16,16,16,
	15,15,15,15,
	14,14,14,14,
	13,13,13,13,
	12,12,12,12,
	11,11,11,11,
	10,10,10,10,
	9,9,9,9,
	8,8,8,8,
	7,7,7,7,
	6,6,6,6,
	5,5,5,5,
	4,4,4,4,
	3,3,3,3,
	2,2,2,2,
	1,1,1,1
};

// LFO Phase Modulation table (verified on real YM2413)
static const char lfo_pm_table[8][8] =
{
	// FNUM2/FNUM = 0 00xxxxxx (0x0000)
	{ 0, 0, 0, 0, 0, 0, 0, 0, },

	// FNUM2/FNUM = 0 01xxxxxx (0x0040)
	{ 1, 0, 0, 0,-1, 0, 0, 0, },

	// FNUM2/FNUM = 0 10xxxxxx (0x0080)
	{ 2, 1, 0,-1,-2,-1, 0, 1, },

	// FNUM2/FNUM = 0 11xxxxxx (0x00C0)
	{ 3, 1, 0,-1,-3,-1, 0, 1, },

	// FNUM2/FNUM = 1 00xxxxxx (0x0100)
	{ 4, 2, 0,-2,-4,-2, 0, 2, },

	// FNUM2/FNUM = 1 01xxxxxx (0x0140)
	{ 5, 2, 0,-2,-5,-2, 0, 2, },

	// FNUM2/FNUM = 1 10xxxxxx (0x0180)
	{ 6, 3, 0,-3,-6,-3, 0, 3, },

	// FNUM2/FNUM = 1 11xxxxxx (0x01C0)
	{ 7, 3, 0,-3,-7,-3, 0, 3, },
};

// This is not 100% perfect yet but very close
//
// - multi parameters are 100% correct (instruments and drums)
// - LFO PM and AM enable are 100% correct
// - waveform DC and DM select are 100% correct
static const byte table[16 + 3][8] = {
	// MULT  MULT modTL DcDmFb AR/DR AR/DR SL/RR SL/RR
	//   0     1     2     3     4     5     6     7
	{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, // user instrument
	{ 0x61, 0x61, 0x1e, 0x17, 0xf0, 0x7f, 0x00, 0x17 }, // violin
	{ 0x13, 0x41, 0x16, 0x0e, 0xfd, 0xf4, 0x23, 0x23 }, // guitar
	{ 0x03, 0x01, 0x9a, 0x04, 0xf3, 0xf3, 0x13, 0xf3 }, // piano
	{ 0x11, 0x61, 0x0e, 0x07, 0xfa, 0x64, 0x70, 0x17 }, // flute
	{ 0x22, 0x21, 0x1e, 0x06, 0xf0, 0x76, 0x00, 0x28 }, // clarinet
	{ 0x21, 0x22, 0x16, 0x05, 0xf0, 0x71, 0x00, 0x18 }, // oboe
	{ 0x21, 0x61, 0x1d, 0x07, 0x82, 0x80, 0x17, 0x17 }, // trumpet
	{ 0x23, 0x21, 0x2d, 0x16, 0x90, 0x90, 0x00, 0x07 }, // organ
	{ 0x21, 0x21, 0x1b, 0x06, 0x64, 0x65, 0x10, 0x17 }, // horn
	{ 0x21, 0x21, 0x0b, 0x1a, 0x85, 0xa0, 0x70, 0x07 }, // synthesizer
	{ 0x23, 0x01, 0x83, 0x10, 0xff, 0xb4, 0x10, 0xf4 }, // harpsichord
	{ 0x97, 0xc1, 0x20, 0x07, 0xff, 0xf4, 0x22, 0x22 }, // vibraphone
	{ 0x61, 0x00, 0x0c, 0x05, 0xc2, 0xf6, 0x40, 0x44 }, // synthesizer bass
	{ 0x01, 0x01, 0x56, 0x03, 0x94, 0xc2, 0x03, 0x12 }, // acoustic bass
	{ 0x21, 0x01, 0x89, 0x03, 0xf1, 0xe4, 0xf0, 0x23 }, // electric guitar
	// drum instruments definitions
	// MULTI MULTI modTL  xxx  AR/DR AR/DR SL/RR SL/RR
	//   0     1     2     3     4     5     6     7
	//{ 0x07, 0x21, 0x14, 0x00, 0xee, 0xf8, 0xff, 0xf8 },
	//{ 0x01, 0x31, 0x00, 0x00, 0xf8, 0xf7, 0xf8, 0xf7 },
	//{ 0x25, 0x11, 0x00, 0x00, 0xf8, 0xfa, 0xf8, 0x55 }
	{ 0x01, 0x01, 0x16, 0x00, 0xfd, 0xf8, 0x2f, 0x6d },// BD(multi verified, modTL verified, mod env - verified(close), carr. env verifed)
	{ 0x01, 0x01, 0x00, 0x00, 0xd8, 0xd8, 0xf9, 0xf8 },// HH(multi verified), SD(multi not used)
	{ 0x05, 0x01, 0x00, 0x00, 0xf8, 0xba, 0x49, 0x55 },// TOM(multi,env verified), TOP CYM(multi verified, env verified)
};

/** 16.16 fixed point type for frequency calculations.
  */
typedef FixedPoint<16> FreqIndex;

static inline FreqIndex fnumToIncrement(int block_fnum)
{
	// OPLL (YM2413) phase increment counter = 18bit
	// Chip works with 10.10 fixed point, while we use 16.16.
	const int block = (block_fnum & 0x1C00) >> 10;
	return FreqIndex(block_fnum & 0x03FF) >> (11 - block);
}

class Global;
class Channel;

class Slot
{
public:
	Slot();

	/**
	 * Initializes those parts that cannot be initialized in the constructor,
	 * because the constructor cannot have arguments since we want to create
	 * an array of Slots.
	 * This method should be called once, as soon as possible after
	 * construction.
	 */
	void init(Global& global, Channel& channel);

	void resetOperators();

	/**
	 * Update phase increment counter of operator.
	 * Also updates the EG rates if necessary.
	 */
	void updateGenerators();

	inline int calcOutput(int phase) const;
	inline void updateModulator();
	inline void advanceEnvelopeGenerator(unsigned eg_cnt, bool carrier);
	inline void advancePhaseGenerator();

	void setKeyOn(KeyPart part);
	void setKeyOff(KeyPart part);
	void setKeyOnOff(KeyPart part, bool enabled) {
		if (enabled) {
			setKeyOn(part);
		} else {
			setKeyOff(part);
		}
	}

	/**
	 * Does this slot currently produce an output signal?
	 */
	bool isActive() const {
		return state != EG_OFF;
	}

	/**
	 * Returns the integer part of the frequency counter of this slot.
	 */
	int getPhase() const {
		return phase.toInt();
	}

	/**
	 * Output of SLOT 1 can be used to phase modulate SLOT 2.
	 */
	int getPhaseModulation() const {
		return op1_out[0] << 1;
	}

	/**
	 * Sets the frequency multiplier [0..15].
	 */
	void setFrequencyMultiplier(byte value) {
		mul = mul_tab[value];
	}

	/**
	 * Sets the key scale rate: true->0, false->2.
	 */
	void setKeyScaleRate(bool value) {
		KSR = value ? 0 : 2;
	}

	/**
	 * Sets the envelope type of the current instrument.
	 * @param value true->sustained, false->percussive.
	 */
	void setEnvelopeSustained(bool value) {
		eg_sustain = value;
	}

	/**
	 * Enables (true) or disables (false) vibrato.
	 */
	void setVibrato(bool value) {
		vib = value;
	}

	/**
	 * Enables (true) or disables (false) amplitude modulation.
	 */
	void setAmplitudeModulation(bool value) {
		AMmask = value ? ~0 : 0;
	}

	/**
	 * Sets the total level: [0..63].
	 */
	void setTotalLevel(byte value) {
		TL = value << (ENV_BITS - 2 - 7); // 7 bits TL (bit 6 = always 0)
		updateTotalLevel();
	}

	/**
	 * Sets the key scale level: 0->0 / 1->1.5 / 2->3.0 / 3->6.0 dB/OCT.
	 */
	void setKeyScaleLevel(byte value) {
		ksl = value ? (3 - value) : 31;
		updateTotalLevel();
	}

	/**
	 * Sets the waveform: 0 = sinus, 1 = half sinus, half silence.
	 */
	void setWaveform(byte value) {
		wavetable = &sin_tab[value * SIN_LEN];
	}

	/**
	 * Sets the amount of feedback [0..7].
	 */
	void setFeedbackShift(byte value) {
		fb_shift = value ? 8 - value : 0;
	}

	/**
	 * Sets the attack rate [0..15].
	 */
	void setAttackRate(byte value) {
		ar = value ? 16 + (value << 2) : 0;
		updateAttackRate();
	}

	/**
	 * Sets the decay rate [0..15].
	 */
	void setDecayRate(byte value) {
		dr = value ? 16 + (value << 2) : 0;
		updateDecayRate();
	}

	/**
	 * Sets the release rate [0..15].
	 */
	void setReleaseRate(byte value) {
		rr = value ? 16 + (value << 2) : 0;
		updateReleaseRate();
	}

	/**
	 * Sets the sustain level [0..15].
	 */
	void setSustainLevel(byte value) {
		sl = sl_tab[value];
	}

	/**
	 * Called by Channel when block_fnum changes.
	 */
	void updateFrequency() {
		updateTotalLevel();
		updateGenerators();
	}

private:
	inline void updateTotalLevel();
	inline void updateAttackRate();
	inline void updateDecayRate();
	inline void updateReleaseRate();

	Global* global;
	Channel* channel;
	unsigned* wavetable;	// waveform select

	// Phase Generator
	FreqIndex phase;	// frequency counter
	FreqIndex freq;	// frequency counter step

	// Envelope Generator
	int TL;		// total level: TL << 2
	int TLL;	// adjusted now TL
	int volume;	// envelope counter
	int sl;		// sustain level: sl_tab[SL]
	EnvelopeState state;

	int op1_out[2];	// slot1 output for feedback
	bool eg_sustain;// percussive/nonpercussive mode
	byte fb_shift;	// feedback shift value

	byte key;	// 0 = KEY OFF, >0 = KEY ON

	byte eg_sh_dp;	// (dump state)
	byte eg_sel_dp;	// (dump state)
	byte eg_sh_ar;	// (attack state)
	byte eg_sel_ar;	// (attack state)
	byte eg_sh_dr;	// (decay state)
	byte eg_sel_dr;	// (decay state)
	byte eg_sh_rr;	// (release state for non-perc.)
	byte eg_sel_rr;	// (release state for non-perc.)
	byte eg_sh_rs;	// (release state for perc.mode)
	byte eg_sel_rs;	// (release state for perc.mode)

	byte ar;	// attack rate: AR<<2
	byte dr;	// decay rate:  DR<<2
	byte rr;	// release rate:RR<<2
	byte KSR;	// key scale rate
	byte ksl;	// keyscale level
	byte kcodeScaled;	// key scale rate: kcode>>KSR
	byte mul;	// multiple: mul_tab[ML]

	// LFO
	byte AMmask;	// LFO Amplitude Modulation enable mask
	byte vib;	// LFO Phase Modulation enable flag (active high)
};

class Channel
{
public:
	Channel();

	/**
	 * Calculate the value of the current sample produced by this channel.
	 */
	inline int calcOutput() const;

	/**
	 * Initializes those parts that cannot be initialized in the constructor,
	 * because the constructor cannot have arguments since we want to create
	 * an array of Channels.
	 * This method should be called once, as soon as possible after
	 * construction.
	 */
	void init(Global& global);

	/**
	 * Sets the frequency for this channel.
	 */
	void setFrequency(int block_fnum);

	/**
	 * Changes the lower 8 bits of the frequency for this channel.
	 */
	void setFrequencyLow(byte value) {
		setFrequency((block_fnum & 0x0F00) | value);
	}

	/**
	 * Changes the higher 4 bits of the frequency for this channel.
	 */
	void setFrequencyHigh(byte value) {
		setFrequency((value << 8) | (block_fnum & 0x00FF));
	}

	/**
	 * Sets some synthesis parameters as specified by the instrument.
	 * @param instrument Number of the instrument.
	 * @param part Part [0..7] of the instrument.
	 */
	void updateInstrumentPart(int instrument, int part);

	/**
	 * Sets all synthesis parameters as specified by the instrument.
	 * @param instrument Number of the instrument.
	 */
	void updateInstrument(int instrument);

	/**
	 * Sets all synthesis parameters as specified by the current instrument.
	 * The current instrument is determined by instvol_r.
	 */
	void updateInstrument() {
		updateInstrument(instvol_r >> 4);
	}

	int getBlockFNum() {
		return block_fnum;
	}

	FreqIndex getFrequencyIncrement() {
		return fc;
	}

	int getKeyScaleLevelBase() {
		return ksl_base;
	}

	byte getKeyCode() {
		return kcode;
	}

	bool isSustained() {
		return sus;
	}

	void setSustain(bool sustained) {
		sus = sustained;
	}

	Slot slots[2];

	/**
	 * Instrument/volume (or volume/volume in rhythm mode).
	 */
	byte instvol_r;

private:
	Global* global;

	// phase generator state
	int block_fnum;	// block+fnum
	FreqIndex fc;	// Freq. freqement base
	int ksl_base;	// KeyScaleLevel Base step
	byte kcode;	// key code (for key scaling)
	bool sus;	// sus on/off (release speed in percussive mode)
};

class Global : public YM2413Core
{
public:
	Global(MSXMotherBoard& motherBoard, const std::string& name,
	       const XMLElement& config, const EmuTime& time);
	virtual ~Global();

	const byte* getInstrument(int instrument) {
		return inst_tab[instrument];
	}

	byte get_LFO_AM() {
		return LFO_AM;
	}

	byte get_LFO_PM() {
		return lfo_pm_cnt.toInt() & 7;
	}

	void reset(const EmuTime& time);

	/**
	 * Reset operator parameters.
	 */
	void resetOperators();

	/**
	 * Generate output samples for each channel.
	 */
	virtual void generateChannels(int** bufs, unsigned num);

	virtual void writeReg(byte r, byte v, const EmuTime& time);

private:
	/**
	 * Initialize "tl_tab" and "sin_tab".
	 */
	static void initTables();

	int getNumMelodicChannels() {
		return rhythm ? 6 : 9;
	}

	Channel& getChannelForReg(byte reg) {
		byte chan = (reg & 0x0F) % 9; // verified on real YM2413
		return channels[chan];
	}

	int adjust(int x) {
		return x << 4;
	}

	/**
	 * Advance LFO to next sample.
	 */
	inline void advanceLFO();

	/**
	 * Advance envelope and phase generators to next sample.
	 */
	inline void advance();

	inline int genPhaseHighHat();
	inline int genPhaseSnare();
	inline int genPhaseCymbal();

	/**
	 * Called when the custom instrument (instrument 0) has changed.
	 * @param part Part [0..7] of the instrument.
	 * @param value The new value.
	 */
	void updateCustomInstrument(int part, byte value);

	void setRhythmMode(bool newMode);
	void setRhythmFlags(byte flags);

	/**
	 * Instrument settings:
	 *  0     - user instrument
	 *  1-15  - fixed instruments
	 *  16    - bass drum settings
	 *  17-18 - other percussion instruments
	 */
	byte inst_tab[19][8];

	/**
	 * OPLL chips have 9 channels.
	 */
	Channel channels[9];

	/**
	 * Global envelope generator counter.
	 */
	unsigned eg_cnt;

	/**
	 * Random generator for noise: 23 bit shift register.
	 */
	int noise_rng;

	/** Number of samples the output was completely silent. */
	unsigned idleSamples;

	typedef FixedPoint< 6> LFOAMIndex;
	typedef FixedPoint<10> LFOPMIndex;
	LFOAMIndex lfo_am_cnt;
	LFOPMIndex lfo_pm_cnt;
	byte LFO_AM;

	/**
	 * Rhythm mode.
	 */
	bool rhythm;
};

inline void Slot::advanceEnvelopeGenerator(unsigned eg_cnt, bool carrier)
{
	switch (state) {
	case EG_DUMP:
		// Dump phase is performed by both operators in each channel.
		// When CARRIER envelope gets down to zero level, phases in BOTH
		// operators are reset (at the same time?).
		// TODO: That sounds logical, but it does not match the implementation.
		if (!(eg_cnt & ((1 << eg_sh_dp) - 1))) {
			volume += eg_inc[eg_sel_dp][(eg_cnt >> eg_sh_dp) & 7];
			if (volume >= MAX_ATT_INDEX) {
				volume = MAX_ATT_INDEX;
				state = EG_ATTACK;
				phase = FreqIndex(0); // restart Phase Generator
			}
		}
		break;

	case EG_ATTACK:
		if (!(eg_cnt & ((1 << eg_sh_ar) - 1))) {
			volume +=
				(~volume * eg_inc[eg_sel_ar][(eg_cnt >> eg_sh_ar) & 7]) >> 2;
			if (volume <= MIN_ATT_INDEX) {
				volume = MIN_ATT_INDEX;
				state = EG_DECAY;
			}
		}
		break;

	case EG_DECAY:
		if (!(eg_cnt & ((1 << eg_sh_dr) - 1))) {
			volume += eg_inc[eg_sel_dr][(eg_cnt >> eg_sh_dr) & 7];
			if (volume >= sl) {
				state = EG_SUSTAIN;
			}
		}
		break;

	case EG_SUSTAIN:
		// this is important behaviour:
		// one can change percusive/non-percussive modes on the fly and
		// the chip will remain in sustain phase
		// - verified on real YM3812
		if (eg_sustain) {
			// non-percussive mode (sustained tone)
			// do nothing
		} else {
			// percussive mode
			// during sustain phase chip adds Release Rate (in
			// percussive mode)
			if (!(eg_cnt & ((1 << eg_sh_rr) - 1))) {
				volume += eg_inc[eg_sel_rr][(eg_cnt >> eg_sh_rr) & 7];
				if (volume >= MAX_ATT_INDEX) {
					volume = MAX_ATT_INDEX;
				}
			}
			// else do nothing in sustain phase
		}
		break;

	case EG_RELEASE:
		// Exclude modulators in melody channels from performing anything in
		// this mode.
		if (carrier) {
			const bool sustain = !eg_sustain || channel->isSustained();
			const byte sel = sustain ? eg_sel_rs : eg_sel_rr;
			const byte shift = sustain ? eg_sh_rs : eg_sh_rr;
			if (!(eg_cnt & ((1 << shift) - 1))) {
				volume += eg_inc[sel][(eg_cnt >> shift) & 7];
				if (volume >= MAX_ATT_INDEX) {
					volume = MAX_ATT_INDEX;
					state = EG_OFF;
				}
			}
		}
		break;

	case EG_OFF:
		break;
	}
}

inline void Slot::advancePhaseGenerator()
{
	if (vib) {
		const int lfo_fn_table_index_offset = lfo_pm_table
			[(channel->getBlockFNum() & 0x01FF) >> 6][global->get_LFO_PM()];
		if (lfo_fn_table_index_offset) {
			// LFO phase modulation active
			phase += fnumToIncrement(
				channel->getBlockFNum() * 2 + lfo_fn_table_index_offset
				) * mul;
		} else {
			// LFO phase modulation = zero
			phase += freq;
		}
	} else {
		// LFO phase modulation disabled for this operator
		phase += freq;
	}
}

inline void Slot::updateTotalLevel()
{
	TLL = TL + (channel->getKeyScaleLevelBase() >> ksl);
}

inline void Slot::updateAttackRate()
{
	if ((ar + kcodeScaled) < (16 + 62)) {
		eg_sh_ar  = eg_rate_shift [ar + kcodeScaled];
		eg_sel_ar = eg_rate_select[ar + kcodeScaled];
	} else {
		eg_sh_ar  = 0;
		eg_sel_ar = 13;
	}
}

inline void Slot::updateDecayRate()
{
	eg_sh_dr  = eg_rate_shift [dr + kcodeScaled];
	eg_sel_dr = eg_rate_select[dr + kcodeScaled];
}

inline void Slot::updateReleaseRate()
{
	eg_sh_rr  = eg_rate_shift [rr + kcodeScaled];
	eg_sel_rr = eg_rate_select[rr + kcodeScaled];
}

inline void Global::advanceLFO()
{
	// Amplitude modulation: 27 output levels (triangle waveform)
	// 1 level takes one of: 192, 256 or 448 samples
	// One entry from LFO_AM_TABLE lasts for 64 samples
	lfo_am_cnt.addQuantum();
	if (lfo_am_cnt == LFOAMIndex(LFO_AM_TAB_ELEMENTS)) {
		// lfo_am_table is 210 elements long
		lfo_am_cnt = LFOAMIndex(0);
	}
	LFO_AM = lfo_am_table[lfo_am_cnt.toInt()] >> 1;

	// Vibrato: 8 output levels (triangle waveform); 1 level takes 1024 samples
	lfo_pm_cnt.addQuantum();
}

inline void Global::advance()
{
	eg_cnt++;
	for (int ch = 0; ch < 9; ch++) {
		Channel& channel = channels[ch];
		for (int sl = 0; sl < 2; sl++) {
			Slot& slot = channel.slots[sl];
			slot.advanceEnvelopeGenerator(
				eg_cnt,
				sl == 1 || (rhythm && ch >= 6) // act as carrier?
				);
			slot.advancePhaseGenerator();
		}
	}

	// The Noise Generator of the YM3812 is 23-bit shift register.
	// Period is equal to 2^23-2 samples.
	// Register works at sampling frequency of the chip, so output
	// can change on every sample.
	//
	// Output of the register and input to the bit 22 is:
	// bit0 XOR bit14 XOR bit15 XOR bit22
	//
	// Simply use bit 22 as the noise output.

	//  int j = ((noise_rng >>  0) ^ (noise_rng >> 14) ^
	//           (noise_rng >> 15) ^ (noise_rng >> 22)) & 1;
	//  noise_rng = (j << 22) | (noise_rng >> 1);
	//
	//    Instead of doing all the logic operations above, we
	//    use a trick here (and use bit 0 as the noise output).
	//    The difference is only that the noise bit changes one
	//    step ahead. This doesn't matter since we don't know
	//    what is real state of the noise_rng after the reset.

	if (noise_rng & 1) {
		noise_rng ^= 0x800302;
	}
	noise_rng >>= 1;
}

inline int Slot::calcOutput(int phase) const
{
	const int env = (TLL + volume + (global->get_LFO_AM() & AMmask)) << 5;
	const int p = env + wavetable[phase & SIN_MASK];
	return p < TL_TAB_LEN ? tl_tab[p] : 0;
}

inline void Slot::updateModulator()
{
	// Compute phase.
	int phase = getPhase();
	if (fb_shift) {
		phase += (op1_out[0] + op1_out[1]) >> fb_shift;
	}
	// Shift output in 2-place buffer.
	op1_out[0] = op1_out[1];
	// Calculate operator output.
	op1_out[1] = calcOutput(phase);
}

inline int Channel::calcOutput() const
{
	return slots[SLOT2].calcOutput(
		slots[SLOT2].getPhase() + slots[SLOT1].getPhaseModulation()
		);
}


// Operators used in the rhythm sounds generation process:
//
// Envelope Generator:
//
// channel  operator  register number   Bass  High  Snare Tom  Top
// / slot   number    TL ARDR SLRR Wave Drum  Hat   Drum  Tom  Cymbal
//  6 / 0   12        50  70   90   f0  +
//  6 / 1   15        53  73   93   f3  +
//  7 / 0   13        51  71   91   f1        +
//  7 / 1   16        54  74   94   f4              +
//  8 / 0   14        52  72   92   f2                    +
//  8 / 1   17        55  75   95   f5                          +
//
//    Phase Generator:
//
// channel  operator  register number   Bass  High  Snare Tom  Top
// / slot   number    MULTIPLE          Drum  Hat   Drum  Tom  Cymbal
//  6 / 0   12        30                +
//  6 / 1   15        33                +
//  7 / 0   13        31                      +     +           +
//  7 / 1   16        34                -----  n o t  u s e d -----
//  8 / 0   14        32                                  +
//  8 / 1   17        35                      +                 +
//
// channel  operator  register number   Bass  High  Snare Tom  Top
// number   number    BLK/FNUM2 FNUM    Drum  Hat   Drum  Tom  Cymbal
//    6     12,15     B6        A6      +
//    7     13,16     B7        A7            +     +           +
//    8     14,17     B8        A8            +           +     +

// Phase generation is based on:
//   HH  (13) channel 7->slot 1 combined with channel 8->slot 2
//            (same combination as TOP CYMBAL but different output phases)
//   SD  (16) channel 7->slot 1
//   TOM (14) channel 8->slot 1
//   TOP (17) channel 7->slot 1 combined with channel 8->slot 2
//            (same combination as HIGH HAT but different output phases)

inline int Global::genPhaseHighHat()
{
	// hi == phase >= 0x200
	bool hi;
	// enable gate based on frequency of operator 2 in channel 8
	if (channels[8].slots[SLOT2].getPhase() & 0x28) {
		hi = true;
	} else {
		// base frequency derived from operator 1 in channel 7
		const int op71phase = channels[7].slots[SLOT1].getPhase();
		const bool bit7 = op71phase & 0x80;
		const bool bit3 = op71phase & 0x08;
		const bool bit2 = op71phase & 0x04;
		hi = (bit2 ^ bit7) | bit3;
	}
	if (noise_rng & 1) {
		return hi ? (0x200 | 0xD0) : (0xD0 >> 2);
	} else {
		return hi ? (0x200 | (0xD0 >> 2)) : 0xD0;
	}
}

inline int Global::genPhaseSnare()
{
	// base frequency derived from operator 1 in channel 7
	// noise bit XOR'es phase by 0x100
	return ((channels[7].slots[SLOT1].getPhase() & 0x100) + 0x100)
	     ^ ((noise_rng & 1) << 8);
}

inline int Global::genPhaseCymbal()
{
	// enable gate based on frequency of operator 2 in channel 8
	if (channels[8].slots[SLOT2].getPhase() & 0x28) {
		return 0x300;
	} else {
		// base frequency derived from operator 1 in channel 7
		const int op71Phase = channels[7].slots[SLOT1].getPhase();
		const bool bit7 = op71Phase & 0x80;
		const bool bit3 = op71Phase & 0x08;
		const bool bit2 = op71Phase & 0x04;
		return ((bit2 ^ bit7) | bit3) ? 0x300 : 0x100;
	}
}

void Global::initTables()
{
	static bool alreadyInit = false;
	if (alreadyInit) {
		return;
	}
	alreadyInit = true;

	for (int x = 0; x < TL_RES_LEN; x++) {
		double m = (1 << 16) / pow(2, (x + 1) * (ENV_STEP / 4.0) / 8.0);
		m = floor(m);

		// we never reach (1 << 16) here due to the (x + 1)
		// result fits within 16 bits at maximum
		int n = int(m);	// 16 bits here
		n >>= 4;	// 12 bits here
		n = (n >> 1) + (n & 1); // round to nearest
		// 11 bits here (rounded)
		for (int i = 0; i < 11; i++) {
			tl_tab[x * 2 + 0 + i * 2 * TL_RES_LEN] = n >> i;
			tl_tab[x * 2 + 1 + i * 2 * TL_RES_LEN] = -(n >> i);
		}
	}

	for (int i = 0; i < SIN_LEN; i++) {
		// non-standard sinus
		// checked against the real chip
		double m = sin(((i * 2) + 1) * M_PI / SIN_LEN);

		// we never reach zero here due to ((i*2)+1)
		double o = -8.0 * log(fabs(m)) / log(2); // convert to 'decibels'
		o = o / (ENV_STEP / 4);

		int n = int(2.0 * o);
		n = (n >> 1) + (n & 1); // round to nearest
		// waveform 0: standard sinus
		sin_tab[i] = n * 2 + (m >= 0.0 ? 0: 1);

		// waveform 1:  __      __
		//             /  \____/  \____
		// output only first half of the sinus waveform (positive one)
		sin_tab[SIN_LEN + i] =
			(i & (1 << (SIN_BITS - 1))) ? TL_TAB_LEN : sin_tab[i];
	}
}

void Slot::setKeyOn(KeyPart part)
{
	if (!key) {
		// do NOT restart Phase Generator (verified on real YM2413)
		state = EG_DUMP;
	}
	key |= part;
}

void Slot::setKeyOff(KeyPart part)
{
	if (key) {
		key &= ~part;
		if (!key) {
			if (state != EG_OFF) {
				state = EG_RELEASE;
			}
		}
	}
}

Slot::Slot()
	: phase(0), freq(0)
{
	ar = dr = rr = KSR = ksl = kcodeScaled = mul = 0;
	fb_shift = op1_out[0] = op1_out[1] = 0;
	TL = TLL = volume = sl = 0;
	eg_sh_dp = eg_sel_dp = eg_sh_ar = eg_sel_ar = eg_sh_dr = 0;
	eg_sel_dr = eg_sh_rr = eg_sel_rr = eg_sh_rs = eg_sel_rs = 0;
	eg_sustain = false;
	state = EG_OFF;
	key = AMmask = vib = 0;
	wavetable = &sin_tab[0 * SIN_LEN];
}

void Slot::init(Global& global, Channel& channel)
{
	this->global = &global;
	this->channel = &channel;
}

void Slot::resetOperators()
{
	wavetable = &sin_tab[0 * SIN_LEN];
	state     = EG_OFF;
	volume    = MAX_ATT_INDEX;
}

void Slot::updateGenerators()
{
	// (frequency) phase increment counter
	freq = channel->getFrequencyIncrement() * mul;

	const int kcodeScaledNew = channel->getKeyCode() >> KSR;
	if (kcodeScaled != kcodeScaledNew) {
		kcodeScaled = kcodeScaledNew;
		// calculate envelope generator rates
		updateAttackRate();
		updateDecayRate();
		updateReleaseRate();
	}

	const int rs = channel->isSustained() ? 16 + (5 << 2) : 16 + (7 << 2);
	eg_sh_rs  = eg_rate_shift [rs + kcodeScaled];
	eg_sel_rs = eg_rate_select[rs + kcodeScaled];

	const int dp = 16 + (13 << 2);
	eg_sh_dp  = eg_rate_shift [dp + kcodeScaled];
	eg_sel_dp = eg_rate_select[dp + kcodeScaled];
}

Channel::Channel()
	: fc(0)
{
	instvol_r = 0;
	block_fnum = ksl_base = kcode = 0;
	sus = false;
}

void Channel::init(Global& global)
{
	this->global = &global;
	slots[SLOT1].init(global, *this);
	slots[SLOT2].init(global, *this);
}

void Channel::setFrequency(int block_fnum)
{
	if (this->block_fnum == block_fnum) {
		return;
	}
	this->block_fnum = block_fnum;

	// BLK 2,1,0 bits -> bits 3,2,1 of kcode, FNUM MSB -> kcode LSB
	kcode    = (block_fnum & 0x0F00) >> 8;
	ksl_base = ksl_tab[block_fnum >> 5];
	fc       = fnumToIncrement(block_fnum * 2);

	// Refresh Total Level and frequency counter in both SLOTs of this channel.
	slots[SLOT1].updateFrequency();
	slots[SLOT2].updateFrequency();
}

void Channel::updateInstrumentPart(int instrument, int part)
{
	const byte* inst = global->getInstrument(instrument);
	Slot& slot1 = slots[SLOT1];
	Slot& slot2 = slots[SLOT2];
	switch (part) {
	case 0:
	case 1: {
		Slot& slot = slots[part];
		byte value = inst[part];
		slot.setFrequencyMultiplier(value & 0x0F);
		slot.setKeyScaleRate(value & 0x10);
		slot.setEnvelopeSustained(value & 0x20);
		slot.setVibrato(value & 0x40);
		slot.setAmplitudeModulation(value & 0x80);
		slot.updateGenerators();
		break;
	}
	case 2:
		slot1.setKeyScaleLevel(inst[2] >> 6);
		slot1.setTotalLevel(inst[2] & 0x3F);
		break;
	case 3:
		slot1.setWaveform((inst[3] & 0x08) >> 3);
		slot1.setFeedbackShift(inst[3] & 0x07);
		slot2.setKeyScaleLevel(inst[3] >> 6);
		slot2.setWaveform((inst[3] & 0x10) >> 4);
		break;
	case 4:
		slot1.setAttackRate(inst[4] >> 4);
		slot1.setDecayRate(inst[4] & 0x0F);
		break;
	case 5:
		slot2.setAttackRate(inst[5] >> 4);
		slot2.setDecayRate(inst[5] & 0x0F);
		break;
	case 6:
		slot1.setSustainLevel(inst[6] >> 4);
		slot1.setReleaseRate(inst[6] & 0x0F);
		break;
	case 7:
		slot2.setSustainLevel(inst[7] >> 4);
		slot2.setReleaseRate(inst[7] & 0x0F);
		break;
	}
}

void Channel::updateInstrument(int instrument)
{
	for (int part = 0; part < 8; part++) {
		updateInstrumentPart(instrument, part);
	}
}

Global::Global(MSXMotherBoard& motherBoard, const std::string& name,
               const XMLElement& config, const EmuTime& time)
	: YM2413Core(motherBoard, name)
	, lfo_am_cnt(0), lfo_pm_cnt(0)
{
	initTables();

	LFO_AM = 0;
	eg_cnt = 0;
	rhythm = 0;
	noise_rng = 0;
	for (int ch = 0; ch < 9; ch++) {
		channels[ch].init(*this);
	}

	reset(time);
	registerSound(config);
}

Global::~Global()
{
	unregisterSound();
}

void Global::updateCustomInstrument(int part, byte value)
{
	// Update instrument definition.
	inst_tab[0][part] = value;

	// Update every channel that has instrument 0 selected.
	const int numMelodicChannels = getNumMelodicChannels();
	for (int chan = 0; chan < numMelodicChannels; chan++) {
		Channel& channel = channels[chan];
		if ((channel.instvol_r & 0xF0) == 0) {
			channel.updateInstrumentPart(0, part);
		}
	}
}

void Global::setRhythmMode(bool rhythm)
{
	if (this->rhythm == rhythm) {
		return;
	}
	this->rhythm = rhythm;

	if (rhythm) { // OFF -> ON
		// Bass drum.
		channels[6].updateInstrument(16);
		// High hat and snare drum.
		Channel& ch7 = channels[7];
		ch7.updateInstrument(17);
		ch7.slots[SLOT1].setTotalLevel((ch7.instvol_r >> 4) << 2); // High hat
		// Tom-tom and top cymbal.
		Channel& ch8 = channels[8];
		ch8.updateInstrument(18);
		ch8.slots[SLOT1].setTotalLevel((ch8.instvol_r >> 4) << 2); // Tom-tom
	} else { // ON -> OFF
		channels[6].updateInstrument();
		channels[7].updateInstrument();
		channels[8].updateInstrument();
		// BD key off
		channels[6].slots[SLOT1].setKeyOff(KEY_RHYTHM);
		channels[6].slots[SLOT2].setKeyOff(KEY_RHYTHM);
		// HH key off
		channels[7].slots[SLOT1].setKeyOff(KEY_RHYTHM);
		// SD key off
		channels[7].slots[SLOT2].setKeyOff(KEY_RHYTHM);
		// TOM key off
		channels[8].slots[SLOT1].setKeyOff(KEY_RHYTHM);
		// TOP-CY off
		channels[8].slots[SLOT2].setKeyOff(KEY_RHYTHM);
	}
}

void Global::setRhythmFlags(byte flags)
{
	// flags = X | X | mode | BD | SD | TOM | TC | HH
	setRhythmMode(flags & 0x20);
	if (rhythm) {
		// BD key on/off
		channels[6].slots[SLOT1].setKeyOnOff(KEY_RHYTHM, flags & 0x10);
		channels[6].slots[SLOT2].setKeyOnOff(KEY_RHYTHM, flags & 0x10);
		// HH key on/off
		channels[7].slots[SLOT1].setKeyOnOff(KEY_RHYTHM, flags & 0x01);
		// SD key on/off
		channels[7].slots[SLOT2].setKeyOnOff(KEY_RHYTHM, flags & 0x08);
		// TOM key on/off
		channels[8].slots[SLOT1].setKeyOnOff(KEY_RHYTHM, flags & 0x04);
		// TOP-CY key on/off
		channels[8].slots[SLOT2].setKeyOnOff(KEY_RHYTHM, flags & 0x02);
	}
}

void Global::reset(const EmuTime& time)
{
	eg_cnt    = 0;
	noise_rng = 1;    // noise shift register
	idleSamples = 0;

	// setup instruments table
	for (int instrument = 0; instrument < 19; instrument++) {
		for (int part = 0; part < 8; part++) {
			inst_tab[instrument][part] = table[instrument][part];
		}
	}
	
	// reset with register write
	writeReg(0x0F, 0, time); //test reg
	for (int i = 0x3F; i >= 0x10; i--) {
		writeReg(i, 0, time);
	}

	resetOperators();
}

void Global::resetOperators()
{
	for (int c = 0; c < 9; c++) {
		Channel& ch = channels[c];
		for (int s = 0; s < 2; s++) {
			// wave table
			ch.slots[s].resetOperators();
		}
	}
}

void Global::generateChannels(int** bufs, unsigned num)
{
	// TODO make channelActiveBits a member and
	//      keep it up-to-date all the time
	const int numMelodicChannels = getNumMelodicChannels();
	unsigned channelActiveBits = 0;
	for (int ch = 0; ch < numMelodicChannels; ++ch) {
		if (channels[ch].slots[SLOT2].isActive()) {
			channelActiveBits |= 1 << ch;
		} else {
			bufs[ch] = 0;
		}
	}
	if (rhythm) {
		// Envelope generation based on:
		//   BD  channel 6->slot2 (slot1 for phase modulation)
		//   HH  channel 7->slot1
		//   SD  channel 7->slot2
		//   TOM channel 8->slot1
		//   TOP channel 8->slot2

		if (channels[6].slots[SLOT2].isActive()) {
			channelActiveBits |= 1 << 6;
		} else {
			bufs[6] = 0;
		}
		if (channels[7].slots[SLOT1].isActive()) {
			channelActiveBits |= 1 << 7;
		} else {
			bufs[7] = 0;
		}
		if (channels[7].slots[SLOT2].isActive()) {
			channelActiveBits |= 1 << 8;
		} else {
			bufs[8] = 0;
		}
		if (channels[8].slots[SLOT1].isActive()) {
			channelActiveBits |= 1 << 9;
		} else {
			bufs[9] = 0;
		}
		if (channels[8].slots[SLOT2].isActive()) {
			channelActiveBits |= 1 << 10;
		} else {
			bufs[10] = 0;
		}
	} else {
		// channel [6..8] are used for melody
		bufs[ 9] = 0;
		bufs[10] = 0;
	}
	if (channelActiveBits) {
		idleSamples = 0;
	} else {
		if (idleSamples > (CLOCK_FREQ / (72 * 5))) {
			// Optimization:
			//   idle for over 1/5s = 200ms
			//   we don't care that noise / AM / PM isn't exactly
			//   in sync with the real HW when music resumes
			// Alternative:
			//   implement an efficient advance(n) method
			return;
		}
		idleSamples += num;
	}
	for (unsigned i = 0; i < num; ++i) {
		advanceLFO();
		for (int ch = 0; ch < numMelodicChannels; ++ch) {
			Channel& channel = channels[ch];
			channel.slots[SLOT1].updateModulator();
			if ((channelActiveBits >> ch) & 1) {
				bufs[ch][i] = adjust(channel.calcOutput());
			}
		}
		if (rhythm) {
			// Bass Drum (verified on real YM3812):
			//  - depends on the channel 6 'connect' register:
			//    when connect = 0 it works the same as in normal (non-rhythm) mode
			//                     (op1->op2->out)
			//    when connect = 1 _only_ operator 2 is present on output (op2->out),
			//                     operator 1 is ignored
			//  - output sample always is multiplied by 2
			Channel& channel6 = channels[6];
			channel6.slots[SLOT1].updateModulator();
			if (channelActiveBits & (1 << 6)) {
				bufs[6][i] = adjust(2 * channel6.calcOutput());
			}

			// TODO: Skip phase generation if output will 0 anyway.
			//       Possible by passing phase generator as a template parameter to
			//       calcOutput.

			// High Hat (verified on real YM3812)
			if (channelActiveBits & (1 << 7)) {
				Slot& SLOT7_1 = channels[7].slots[SLOT1];
				bufs[7][i] = adjust(2 * SLOT7_1.calcOutput(genPhaseHighHat()));
			}

			// Snare Drum (verified on real YM3812)
			if (channelActiveBits & (1 << 8)) {
				Slot& SLOT7_2 = channels[7].slots[SLOT2];
				bufs[8][i] = adjust(2 * SLOT7_2.calcOutput(genPhaseSnare()));
			}

			// Tom Tom (verified on real YM3812)
			if (channelActiveBits & (1 << 9)) {
				Slot& SLOT8_1 = channels[8].slots[SLOT1];
				bufs[9][i] = adjust(2 * SLOT8_1.calcOutput(SLOT8_1.getPhase()));
			}

			// Top Cymbal (verified on real YM2413)
			if (channelActiveBits & (1 << 10)) {
				Slot& SLOT8_2 = channels[8].slots[SLOT2];
				bufs[10][i] = adjust(2 * SLOT8_2.calcOutput(genPhaseCymbal()));
			}
		}
		advance();
	}
}

void Global::writeReg(byte r, byte v, const EmuTime& time)
{
	PRT_DEBUG("YM2413: write reg " << int(r) << " " << int(v));
	
	// update the output buffer before changing the register
	updateStream(time);

	reg[r] = v;

	switch (r & 0xF0) {
	case 0x00: { // 00-0F: control
		switch (r & 0x0F) {
		case 0x00:  // AM/VIB/EGTYP/KSR/MULTI (modulator)
		case 0x01:  // AM/VIB/EGTYP/KSR/MULTI (carrier)
		case 0x02:  // Key Scale Level, Total Level (modulator)
		case 0x03:  // Key Scale Level, carrier waveform, modulator waveform,
		            // Feedback
		case 0x04:  // Attack, Decay (modulator)
		case 0x05:  // Attack, Decay (carrier)
		case 0x06:  // Sustain, Release (modulator)
		case 0x07:  // Sustain, Release (carrier)
			updateCustomInstrument(r & 0x07, v);
			break;
		case 0x0E:
			setRhythmFlags(v);
			break;
		}
		break;
	}
	case 0x10:
	case 0x20: { // block, fnum, sus, keyon
		Channel& ch = getChannelForReg(r);
		if (r & 0x10) {
			// 10-18: FNUM 0-7
			ch.setFrequencyLow(v);
		} else {
			// 20-28: suson, keyon, block, FNUM 8
			ch.slots[SLOT1].setKeyOnOff(KEY_MAIN, v & 0x10);
			ch.slots[SLOT2].setKeyOnOff(KEY_MAIN, v & 0x10);
			ch.setSustain(v & 0x20);
			// Note: When changing the frequency, a new value for RS is
			//       computed using the sustain value, so make sure the new
			//       sustain value is committed first.
			ch.setFrequencyHigh(v & 0x0F);
		}
		break;
	}

	case 0x30: { // inst 4 MSBs, VOL 4 LSBs
		Channel& ch = getChannelForReg(r);

		byte old_instvol = ch.instvol_r;
		ch.instvol_r = v;  // store for later use

		ch.slots[SLOT2].setTotalLevel((v & 0x0F) << 2);

		// Check wether we are in rhythm mode and handle instrument/volume
		// register accordingly.

		byte chan = (r & 0x0F) % 9;	// verified on real YM2413
		if (chan >= getNumMelodicChannels()) {
			// We're in rhythm mode.
			if (chan >= 7) {
				// Only for channel 7 and 8 (channel 6 is handled in usual way)
				// modulator envelope is HH(chan=7) or TOM(chan=8).
				ch.slots[SLOT1].setTotalLevel((ch.instvol_r >> 4) << 2);
			}
		} else {
			if ((old_instvol & 0xF0) != (v & 0xF0)) {
				ch.updateInstrument();
			}
		}
		break;
	}

	default:
		break;
	}
}


// YM2413_2

YM2413_2::YM2413_2(MSXMotherBoard& motherBoard, const std::string& name,
                   const XMLElement& config, const EmuTime& time)
	: global(new Global(motherBoard, name, config, time))
{
}

YM2413_2::~YM2413_2()
{
}

void YM2413_2::reset(const EmuTime& time)
{
	global->reset(time);
}

void YM2413_2::writeReg(byte r, byte v, const EmuTime &time)
{
	global->writeReg(r, v, time);
}

} // namespace openmsx
