;;;-*-Mode: LISP; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   Portions copyright (C) 2001 Clozure Associates
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;


;;pathnames.lisp Pathnames for Coral Common LISP
(in-package "CCL")



(eval-when (eval compile)
  (require 'level-2)
  (require 'backquote)
)
;(defconstant $accessDenied -5000) ; put this with other errnos
(defconstant $afpAccessDenied -5000) ; which name to use?



;-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_
;ANSI CL logical pathnames


(defvar *pathname-translations-pathname*
  (make-pathname :host "ccl" :type "pathname-translations"))

(defun load-logical-pathname-translations (host)
  ;(setq host (verify-logical-host-name host))
  (when (not (%str-assoc host %logical-host-translations%))
    (setf (logical-pathname-translations host)
          (with-open-file (file (merge-pathnames (make-pathname :name host :defaults nil)
                                                 *pathname-translations-pathname*)
                                :element-type 'base-char)
            (read file)))
    T))

(defun back-translate-pathname (path &optional hosts)
  (let ((newpath (back-translate-pathname-1 path hosts)))
    (cond ((equalp path newpath)
	   ;; (fcomp-standard-source path)
	   (namestring (pathname path)))
          (t newpath))))


(defun back-translate-pathname-1 (path &optional hosts)
  (dolist (host %logical-host-translations%)
    (when (or (null hosts) (member (car host) hosts :test 'string-equal))
      (dolist (trans (cdr host))
        (when (pathname-match-p path (cadr trans))
          (let* (newpath)          
            (setq newpath (translate-pathname path (cadr trans) (car trans) :reversible t))
            (return-from back-translate-pathname-1 
              (if  (equalp path newpath) path (back-translate-pathname-1 newpath hosts))))))))
  path)



; must be after back-translate-pathname
(defun physical-pathname-p (path)
  (let* ((path (pathname path))
         (dir (pathname-directory path)))
    (and dir
         (or (not (logical-pathname-p path))
             (not (null (memq (pathname-host path) '(nil :unspecific))))))))



;-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_
;File or directory Manipulations

(defun unix-rename (old-name new-name)
  (with-cstrs ((old old-name)
               (new new-name))
    (let* ((res (#_rename old new)))
      (declare (fixnum res))
      (if (zerop res)
        (values t nil)
        (values nil res)))))

(defun rename-file (file new-name &key (if-exists :error))
  "Rename File to have the specified New-Name.  If file is a stream open to a
  file, then the associated file is renamed."
  (let* ((original (truename file))
	 (original-namestring (native-translated-namestring original))
	 (new-name (merge-pathnames new-name original))
	 (new-namestring (native-translated-namestring new-name)))
    (unless new-namestring
      (error "~S can't be created." new-name))
    (unless (and (probe-file new-name)
		 (not (if-exists if-exists new-name)))
      (multiple-value-bind (res error)
	                   (unix-rename original-namestring
					new-namestring)
	(unless res
	  (error "Failed to rename ~A to ~A: ~A"
		 original new-name error))
	(when (streamp file)
	  (setf (file-stream-filename file) new-namestring))
	(values new-name original (truename new-name))))))


(defun copy-file (old-path new-path &key (if-exists :error))
  (break "Copy-file ? ~s ~s ~s" old-path new-path if-exists))

(defun lock-file (path)
  (break "lock-file ? ~s" path))

(defun unlock-file (path)
  (break "unlock-file ? ~s" path))

(defun create-directory (path &key (mode #o777))
  (let* ((pathname (translate-logical-pathname (merge-pathnames path)))
	 (created-p nil)
	 (parent-dirs (pathname-directory pathname))
	 (nparents (length parent-dirs)))
    (when (wild-pathname-p pathname)
      (error 'file-error :error-type "Inappropriate use of wild pathname ~s"
	     :pathname pathname))
    (do* ((i 1 (1+ i)))
	 ((> i nparents) (values pathname created-p))
      (declare (fixnum i))
      (let* ((parent (make-pathname :host (pathname-host pathname)
				    :device (pathname-device pathname)
				    :directory
				    (subseq parent-dirs 0 i)))
	     (parent-name (native-translated-namestring parent))
	     (parent-kind (%unix-file-kind parent-name)))

	(if parent-kind
	  (unless (eq parent-kind :directory)
	    (error 'file-error
		   :error-type "Can't create directory ~s, since file ~a exists and is not a directory"
		   :pathname pathname
		   :format-arguments (list parent-name)))
	  (let* ((result (%mkdir parent-name mode)))
	    (declare (fixnum result))
	    (if (< result 0)
	      (signal-file-error result parent-name)
	      (setq created-p t))))))))

(defun ensure-directories-exist (pathspec &key verbose (mode #o777))
  "Tests whether the directories containing the specified file
  actually exist, and attempts to create them if they do not.
  Portable programs should avoid using the :MODE keyword argument."
  (let* ((pathname (translate-logical-pathname (merge-pathnames pathspec)))
	 (created-p nil))
    (when (wild-pathname-p pathname)
      (error 'file-error
	     :error-type "Inappropriate use of wild pathname ~s"
	     :pathname pathname))
    (let ((dir (pathname-directory pathname)))
      (loop for i from 1 upto (length dir)
	    do (let ((newpath (make-pathname
			       :host (pathname-host pathname)
			       :device (pathname-device pathname)
			       :directory (subseq dir 0 i))))
		 (unless (probe-file newpath)
		   (let ((namestring (native-translated-namestring newpath)))
		     (when verbose
		       (format *standard-output* "~&Creating directory: ~A~%"
			       namestring))
		     (%mkdir namestring mode)
		     (unless (probe-file namestring)
		       (error 'file-error
			      :pathname namestring
			      :error-type "Can't create directory ~A."))
		     (setf created-p t)))))
      (values pathname created-p))))

(defun dirpath-to-filepath (path)
  (setq path (translate-logical-pathname (merge-pathnames path)))
  (let* ((dir (pathname-directory path))
         (super (butlast dir))
         (name (car (last dir))))
    (when (eq name :up)
      (setq dir (remove-up (copy-list dir)))
      (setq super (butlast dir))
      (setq name (car (last dir))))
    (when (null super)
      (signal-file-error $xnocreate path))
    (setq path (make-pathname :directory super :name name :defaults nil))))

(defun filepath-to-dirpath (path)
  (let* ((dir (pathname-directory path))
         (rest (file-namestring path)))
    (make-pathname :directory (append dir (list rest)) :defaults nil)))
  


;Takes a pathname, returns the truename of the directory if the pathname
;names a directory, NIL if it names an ordinary file, error otherwise.
;E.g. (directoryp "ccl;:foo:baz") might return #P"hd:mumble:foo:baz:" if baz
;is a dir. - should we doc this - its exported?
(defun directoryp (path)
  ;; This should be pretty easy.
  (break "Directoryp ? ~s" path))

;-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_
;Wildcards

(defun wild-pathname-p (pathname &optional field-key)
  (flet ((wild-p (name) (or (eq name :wild)
                            (eq name :wild-inferiors)
                            (and (stringp name) (%path-mem "*" name)))))
    (case field-key
      ((nil)
       (or (some #'wild-p (pathname-directory pathname))
           (wild-p (pathname-name pathname))
           (wild-p (pathname-type pathname))
           (wild-p (pathname-version pathname))))
      (:host nil)
      (:device nil)
      (:directory (some #'wild-p (pathname-directory pathname)))
      (:name (wild-p (pathname-name pathname)))
      (:type (wild-p (pathname-type pathname)))
      (:version (wild-p (pathname-version pathname)))
      (t (wild-pathname-p pathname
                          (require-type field-key 
                                        '(member nil :host :device 
                                          :directory :name :type :version)))))))

 
;-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_-_
;Directory Traversing

(defmacro with-open-dir ((dirent dir) &body body)
  `(let ((,dirent (%open-dir ,dir)))
     (when ,dirent
       (unwind-protect
	   (progn ,@body)
	 (close-dir ,dirent)))))

(defun directory (path &key (directories nil) ;; include subdirectories
                            (files t)         ;; include files
			    (all t)           ;; include Unix dot files (other than dot and dot dot)
			    (directory-pathnames t) ;; return directories as directory-pathname-p's.
			    test              ;; Only return pathnames matching test
			    (follow-links t)) ;; return truename's of matching files.
  (let* ((keys (list :directories directories ;list defaulted key values
		     :files files
		     :all all
		     :directory-pathnames directory-pathnames
		     :test test
		     :follow-links follow-links))
	 (path (full-pathname (merge-pathnames path) :no-error nil))
	 (dir (directory-namestring path)))
    (declare (dynamic-extent keys))
    (assert (eq (car (pathname-directory path)) :absolute) ()
	    "full-pathname returned relative path ~s??" path)
    ;; return sorted in alphabetical order, target-Xload-level-0 depends
    ;; on this.
    (nreverse
     (delete-duplicates (%directory "/" dir path '(:absolute) keys) :test #'equal))))

(defun %directory (dir rest path so-far keys)
  (multiple-value-bind (sub-dir wild rest) (%split-dir rest)
    (%some-specific dir sub-dir wild rest path so-far keys)))

(defun %some-specific (dir sub-dir wild rest path so-far keys)
  (let* ((start 1)
	 (end (length sub-dir))
	 (full-dir (if (eq start end) dir (%str-cat dir (%substr sub-dir start end)))))
    (while (neq start end)
      (let ((pos (position #\/ sub-dir :start start :end end)))
	(push (%path-std-quotes (%substr sub-dir start pos) nil "/:;*") so-far)
	(setq start (%i+ 1 pos))))
    (cond ((null wild)
	   (%files-in-directory full-dir path so-far keys))
	  ((string= wild "**")
	   (%all-directories full-dir rest path so-far keys))
	  (t (%one-wild full-dir wild rest path so-far keys)))))

; for a * or *x*y
(defun %one-wild (dir wild rest path so-far keys)
  (let ((result ()) (all (getf keys :all)) name subdir)
    (with-open-dir (dirent dir)
      (while (setq name (%read-dir dirent))
	(when (and (or all (neq (%schar name 0) #\.))
		   (not (string= name "."))
		   (not (string= name ".."))
		   (%path-pstr*= wild name)
		   (eq (%unix-file-kind (setq subdir (%str-cat dir name)) t) :directory))
	  (let ((so-far (cons (%path-std-quotes name nil "/;:*") so-far)))
	    (declare (dynamic-extent so-far))
	    (setq result
		  (nconc (%directory (%str-cat subdir "/") rest path so-far keys) result))))))
    result))

(defun %files-in-directory (dir path so-far keys)
  (let ((name (pathname-name path))
        (type (pathname-type path))
	(directories (getf keys :directories))
	(files (getf keys :files))
	(directory-pathnames (getf keys :directory-pathnames))
	(test (getf keys :test))
	(follow-links (getf keys :follow-links))
	(all (getf keys :all))
        (result ())
        sub dir-list ans)
    (if (not (or name type))
      (when directories
	(setq ans (if directory-pathnames
		    (%cons-pathname (reverse so-far) nil nil)
		    (%cons-pathname (reverse (cdr so-far)) (car so-far) nil)))
	(when (and ans (or (null test) (funcall test ans)))
	  (setq result (list ans))))
      (with-open-dir (dirent dir)
	(while (setq sub (%read-dir dirent))
	  (when (and (or all (neq (%schar sub 0) #\.))
		     (not (string= sub "."))
		     (not (string= sub ".."))
		     (%file*= name type sub))
	    (setq ans
		  (if (eq (%unix-file-kind (%str-cat dir sub) t) :directory)
		    (when directories
		      (let* ((std-sub (%path-std-quotes sub nil "/;:*")))
			(if directory-pathnames
			  (%cons-pathname (reverse (cons std-sub so-far)) nil nil)
			  (%cons-pathname (or dir-list (setq dir-list (reverse so-far))) std-sub nil))))
		    (when files
		      (multiple-value-bind (name type) (%std-name-and-type sub)
			(%cons-pathname (or dir-list (setq dir-list (reverse so-far))) name type)))))
	    (when (and ans (or (null test) (funcall test ans)))
	      (push (if follow-links (truename ans) ans) result))))))
    result))

; now for samson:**:*c*:**: we get samson:ccl:crap:barf: twice because
; it matches in two ways
; 1) **=ccl *c*=crap **=barf
; 2) **= nothing *c*=ccl **=crap:barf
; called to match a **
(defun %all-directories (dir rest path so-far keys)
  (let ((do-files nil)
        (do-dirs nil)
        (result nil)
        (name (pathname-name path))
        (type (pathname-type path))
	(all (getf keys :all))
	(test (getf keys :test))
	(directory-pathnames (getf keys :directory-pathnames))
	(follow-links (getf keys :follow-links))
	sub subfile dir-list ans)
    ;; First process the case that the ** stands for 0 components
    (multiple-value-bind (next-dir next-wild next-rest) (%split-dir rest)
      (while (and next-wild ; Check for **/**/ which is the same as **/
		  (string= next-dir "/")
		  (string= next-wild "**"))
        (setq rest next-rest)
        (multiple-value-setq (next-dir next-wild next-rest) (%split-dir rest)))
      (cond ((not (string= next-dir "/"))
	     (setq result
		   (%some-specific dir next-dir next-wild next-rest path so-far keys)))
	    (next-wild
	     (setq result
		   (%one-wild dir next-wild next-rest path so-far keys)))
	    ((or name type)
	     (when (getf keys :files) (setq do-files t))
	     (when (getf keys :directories) (setq do-dirs t)))
	    (t (when (getf keys :directories)
		 (setq sub (if directory-pathnames
			     (%cons-pathname (setq dir-list (reverse so-far)) nil nil)
			     (%cons-pathname (reverse (cdr so-far)) (car so-far) nil)))
		 (when (or (null test) (funcall test sub))
		   (setq result (list (if follow-links (truename sub) sub))))))))
    ; now descend doing %all-dirs on dirs and collecting files & dirs if do-x is t
    (with-open-dir (dirent dir)
      (while (setq sub (%read-dir dirent))
	(when (and (or all (neq (%schar sub 0) #\.))
		   (not (string= sub "."))
		   (not (string= sub "..")))
	  (if (eq (%unix-file-kind (setq subfile (%str-cat dir sub)) t) :directory)
	    (let* ((std-sub (%path-std-quotes sub nil "/;:*"))
		   (so-far (cons std-sub so-far))
		   (subdir (%str-cat subfile "/")))
	      (declare (dynamic-extent so-far))
	      (when (and do-dirs (%file*= name type sub))
		(setq ans (if directory-pathnames
			    (%cons-pathname (reverse so-far) nil nil)
			    (%cons-pathname (or dir-list (setq dir-list (reverse (cdr so-far))))
					    std-sub nil)))
		(when (or (null test) (funcall test ans))
		  (push (if follow-links (truename ans) ans) result)))
	      (setq result (nconc (%all-directories subdir rest path so-far keys) result)))
	    (when (and do-files (%file*= name type sub))
	      (multiple-value-bind (name type) (%std-name-and-type sub)
		(setq ans (%cons-pathname (or dir-list (setq dir-list (reverse so-far))) name type))
		(when (or (null test) (funcall test ans))
		  (push (if follow-links (truename ans) ans) result))))))))
    result))

(defun %split-dir (dir &aux pos)                 ; dir ends in a "/".
  ;"/foo/bar/../x*y/baz/../z*t/"  ->  "/foo/bar/../" "x*y" "/baz/../z*t/"
  (if (null (setq pos (%path-mem "*" dir)))
    (values dir nil nil)
    (let (epos (len (length dir)))
      (setq pos (if (setq pos (%path-mem-last "/" dir 0 pos)) (%i+ pos 1) 0)
            epos (%path-mem "/" dir pos len))
      (when (%path-mem-last-quoted "/" dir 0 pos)
	(signal-file-error $xbadfilenamechar dir #\/))
      (values (unless (%izerop pos) (namestring-unquote (%substr dir 0 pos)))
              (%substr dir pos epos)
              (%substr dir epos len)))))

(defun %path-pstr*= (pattern pstr &optional (p-start 0))
  (assert (eq p-start 0))
  (%path-str*= pstr pattern))

(defun %file*= (name-pat type-pat pstr)
  (when (and (null name-pat) (null type-pat))
    (return-from %file*= T))
  (let* ((end (length pstr))
	 (pos (position #\. pstr :from-end t))
	 (type (and pos (%substr pstr (%i+ pos 1) end)))
	 (name (unless (eq (or pos end) 0) (if pos (%substr pstr 0 pos) pstr))))
    (and (cond ((eq name-pat :unspecific) (null name))
	       ((null name-pat) T)
	       (t (%path-pstr*= name-pat (or name ""))))
	 (cond ((eq type-pat :unspecific) (null type))
	       ((null type-pat) T)
	       (t (%path-pstr*= type-pat (or type "")))))))

#|
(defmacro enumerate-matches ((var pathname &optional result
				  &key (verify-existance t))
			     &body body)
  (let ((body-name (gensym)))
    `(block nil
       (flet ((,body-name (,var)
		,@body))
	 (%enumerate-matches (pathname ,pathname)
			     ,verify-existance
			     #',body-name)
	 ,result))))

(defun %enumerate-matches (pathname verify-existance function)
  (when (pathname-type pathname)
    (unless (pathname-name pathname)
      (error "Cannot supply a type without a name:~%  ~S" pathname)))
  (when (and (integerp (pathname-version pathname))
	     (member (pathname-type pathname) '(nil :unspecific)))
    (error "Cannot supply a version without a type:~%  ~S" pathname))
  (let ((directory (pathname-directory pathname)))
    (if directory
	(ecase (car directory)
	  (:absolute
	   (%enumerate-directories "/" (cdr directory) pathname
				   verify-existance function))
	  (:relative
	   (%enumerate-directories "" (cdr directory) pathname
				   verify-existance function)))
	(%enumerate-files "" pathname verify-existance function))))

(defun %enumerate-directories (head tail pathname verify-existance function)
  (declare (simple-string head))
  (if tail
    (let ((piece (car tail)))
      (etypecase piece
        (simple-string
         (%enumerate-directories (concatenate 'string head piece "/")
                                 (cdr tail) pathname verify-existance
                                 function))
        ((member :wild)
         (let ((dir (%open-dir head)))
           (when dir
             (unwind-protect
                  (loop
                      (let ((name (%read-dir dir)))
                        (cond ((null name)
                               (return))
                              ((string= name "."))
                              ((string= name ".."))
                              (t
                               (let ((subdir (concatenate 'string
                                                          head name)))
                                 ;; see below for health warning
                                 (when (eq (%unix-file-kind subdir t)
                                           :directory)
                                   (%enumerate-directories
                                    (concatenate 'string subdir "/")
                                    (cdr tail) pathname verify-existance
                                    function)))))))
               (close-dir dir)))))	   
        ((member :wild-inferiors)
         (let ((dir (%open-dir head)))
           (when dir
             (unwind-protect
                  (loop
                      (let ((name (%read-dir dir)))
                        (cond ((null name)
                               (return))
                              ((string= name "."))
                              ((string= name ".."))
                              (t        ; always match 
                               ;; warning - this depends
                               ;; on lstat behaving in the same way on
                               ;; different platforms. Basically,
                               ;; lstat("foo") -> :link, lstat("foo/")
                               ;; -> directory on linux 2.2.
                               (let ((subdir (concatenate 'string
                                                          head name)))
                                 (when (eq (%unix-file-kind subdir t)
                                           :directory)
                                   (%enumerate-directories
                                    (concatenate 'string subdir "/")
                                    tail pathname verify-existance
                                    function)))))))
               (close-dir dir))
             (%enumerate-files head pathname verify-existance	function))))
        (pattern
         (let ((dir (%open-dir head)))
           (when dir
             (unwind-protect
                  (loop
                      (let ((name (%read-dir dir)))
                        (cond ((null name)
                               (return))
                              ((string= name "."))
                              ((string= name ".."))
                              ((pattern-matches piece name)
                               (let ((subdir (concatenate 'string
                                                          head name "/")))
                                 (when (eq (%unix-file-kind subdir)
                                           :directory)
                                   (%enumerate-directories
                                    subdir (cdr tail) pathname verify-existance
                                    function)))))))
               (close-dir dir)))))
        ((member :up)
         (%enumerate-directories (concatenate 'string head "../")
                                 (cdr tail) pathname verify-existance
                                 function))))
    (%enumerate-files head pathname verify-existance function)))

(defun %enumerate-files (directory pathname verify-existance function)
  (declare (simple-string directory))
  (let ((name (%pathname-name pathname))
	(type (%pathname-type pathname))
	(version (pathname-version pathname)))
    (cond ((member name '(nil :unspecific))
	   (when (or (not verify-existance)
		     (%unix-file-kind directory))
	     (funcall function directory)))
	  ((or (pattern-p name)
	       (pattern-p type)
	       (eq name :wild)
	       (eq type :wild))
	   (let ((dir (%open-dir directory)))
	     (when dir
	       (unwind-protect
		   (loop
		     (let ((file (%read-dir dir)))
		       (if file
			   (unless (or (string= file ".")
				       (string= file ".."))
			     (multiple-value-bind
				 (file-name file-type file-version)
				 (let ((*ignore-wildcards* t))
				   (extract-name-type-and-version
				    file 0 (length file)))
			       (when (and (components-match file-name name)
					  (components-match file-type type)
					  (components-match file-version
							    version))
				 (funcall function
					  (concatenate 'string
						       directory
						       file)))))
			   (return))))
		 (close-dir dir)))))
	  (t
	   (let ((file (concatenate 'string directory name)))
	     (unless (or (null type) (eq type :unspecific))
	       (setf file (concatenate 'string file "." type)))
	     (unless (member version '(nil :newest :wild))
	       (setf file (concatenate 'string file "."
				       (quick-integer-to-string version))))
	     (when (or (not verify-existance)
		       (%unix-file-kind file t))
	       (funcall function file)))))))

(defun quick-integer-to-string (n)
  (declare (type integer n))
  (cond ((not (fixnump n))
	 (write-to-string n :base 10 :radix nil))
	((zerop n) "0")
	((eql n 1) "1")
	((minusp n)
	 (concatenate 'simple-string "-"
		      (the simple-string (quick-integer-to-string (- n)))))
	(t
	 (do* ((len (1+ (truncate (integer-length n) 3)))
	       (res (make-string len))
	       (i (1- len) (1- i))
	       (q n)
	       (r 0))
	      ((zerop q)
	       (incf i)
	       (replace res res :start2 i :end2 len)
	       (shrink-vector res (- len i)))
	   (declare (simple-string res)
		    (fixnum len i r q))
	   (multiple-value-setq (q r) (truncate q 10))
	   (setf (schar res i) (schar "0123456789" r))))))

(defun directory (pathname &key (all t) (check-for-subdirs t)
			   (follow-links t))
  "Returns a list of pathnames, one for each file that matches the given
   pathname.  Supplying :ALL as nil causes this to ignore Unix dot files.  This
   never includes Unix dot and dot-dot in the result.  If :FOLLOW-LINKS is NIL,
   then symblolic links in the result are not expanded.  This is not the
   default because TRUENAME does follow links, and the result pathnames are
   defined to be the TRUENAME of the pathname (the truename of a link may well
   be in another directory.)"
  (let ((results nil))
    (enumerate-matches (name (translate-logical-pathname
			      (merge-pathnames pathname
					       (make-pathname :name :wild
							      :type :wild
							      :version :wild))))
       (when (or all
		 (let ((slash (position #\/ name :from-end t)))
		   (or (null slash)
		       (= (1+ slash) (length name))
		       (char/= (schar name (1+ slash)) #\.))))
	 (push name results)))
    (let ((*ignore-wildcards* t))
      (mapcar #'(lambda (name)
		  (let ((name (if (and check-for-subdirs
				       (eq (%unix-file-kind name)
					   :directory))
				  (concatenate 'string name "/")
				  name)))
		    (if follow-links (truename name) (pathname name))))
	      (sort (delete-duplicates results :test #'string=) #'string<)))))
|#

(provide "PATHNAMES")
