;;;-*- Mode: Lisp; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;


; l0-cfm-support.lisp



; offset is a fixnum, one of the arch::kernel-import-xxx.
; Returns that kernel import, a fixnum.
#+ppc-target
(defppclapfunction %kernel-import ((offset arg_z))
  (ref-global imm0 kernel-imports)
  (unbox-fixnum imm1 arg_z)
  (lwzx arg_z imm0 imm1)
  (blr))

#+sparc-target
(defsparclapfunction %kernel-import ((%offset %arg_z))
  (ref-global %imm0 kernel-imports)
  (unbox-fixnum %offset %imm1)
  (retl)
    (ld (%imm0 %imm1) %arg_z))

#+ppc-target
(defppclapfunction %get-unboxed-ptr ((macptr arg_z))
  (macptr-ptr imm0 arg_z)
  (lwz arg_z 0 imm0)
  (blr))

#+sparc-target
(defsparclapfunction %get-unboxed-ptr ((%macptr %arg_z))
  (macptr-ptr %macptr %imm0)
  (retl)
   (ld (%imm0) %arg_z))


; Bootstrapping. Real version is in l1-aprims.
; Called by expansion of with-pstrs

(defun byte-length (string &optional script start end)
    (declare (ignore script))
    (when (or start end)
      (error "Don't support start or end args yet"))
    (if (base-string-p string)
      (length string)
      (error "Don't support non base-string yet.")))



(def-accessor-macros %svref
  nil                                 ; 'external-entry-point
  eep.address
  eep.name
  eep.container)

(defun %cons-external-entry-point (name &optional container)
  (%istruct 'external-entry-point nil name container))

(defun external-entry-point-p (x)
  (istruct-typep x 'external-entry-point))

(def-accessor-macros %svref
    nil					;'shlib
  shlib.soname
  shlib.pathname
  shlib.opened-by-lisp-kernel
  shlib.map
  shlib.base)

(defun %cons-shlib (soname pathname map base)
  (%istruct 'shlib soname pathname nil map base))

(defvar *rtld-next*)
(defvar *rtld-default*)
(setq *rtld-next* (%int-to-ptr #xFFFFFFFF)
      *rtld-default* (%int-to-ptr 0))

#+linuxppc-target
(progn
;;; I can't think of a reason to change this.
(defvar *dlopen-flags* nil)
(setq *dlopen-flags* (logior #$RTLD_GLOBAL #$RTLD_NOW))
)

(defvar *eeps* nil)

(defun eeps ()
  (or *eeps*
      (setq *eeps* (make-hash-table :test #'equal))))

(defvar *shared-libraries* nil)

#+linux-target
(progn

(eval-when (:compile-toplevel)
  ;;; Magic.  Evil.
  (defmacro link-map-libname (map)
    `(%get-ptr ,map 20)))

(defun shared-library-at (base)
  (dolist (lib *shared-libraries*)
    (when (eql (shlib.base lib) base)
      (return lib))))

(defun shared-library-with-name (name)
  (let* ((namelen (length name)))
    (dolist (lib *shared-libraries*)
      (let* ((libname (shlib.soname lib)))
	(when (%simple-string= name libname 0 0 namelen (length libname))
	  (return lib))))))

(defun shlib-from-map-entry (m)
  (let* ((base (%int-to-ptr (pref m :link_map.l_addr))))
    (or (let* ((existing-lib (shared-library-at base)))
	  (when (and existing-lib (null (shlib.map existing-lib)))
	    (setf (shlib.map existing-lib) m
		  (shlib.pathname existing-lib)
		  (%get-cstring (pref m :link_map.l_name))
		  (shlib.base existing-lib) base))
	  existing-lib)
        (let* ((soname-ptr (%get-ptr (link-map-libname m)))
               (soname (unless (%null-ptr-p soname-ptr) (%get-cstring soname-ptr)))
               (pathname (%get-cstring (pref m :link_map.l_name)))
	       (shlib (shared-library-with-name soname)))
	  (if shlib
	    (setf (shlib.map shlib) m
		  (shlib.base shlib) base
		  (shlib.pathname shlib) pathname)
	    (push (setq shlib (%cons-shlib soname pathname m base))
		  *shared-libraries*))
          shlib))))

(defun %walk-shared-libraries (f)
  (let* ((loaded (%get-ptr (foreign-symbol-address "_dl_loaded"))))
    (do* ((map (pref loaded :link_map.l_next) (pref map :link_map.l_next)))
         ((%null-ptr-p map))
      (funcall f map))))

(defun %dlopen-shlib (l)
  (with-cstrs ((n (shlib.soname l)))
    (ff-call (%kernel-import arch::kernel-import-GetSharedLibrary)
	     :address n
	     :unsigned-fullword *dlopen-flags*
	     :void)))
  
(defun init-shared-libraries ()
  (when (null *shared-libraries*)
    (%walk-shared-libraries #'shlib-from-map-entry)
    (dolist (l *shared-libraries*)
      ;;; It seems to be necessary to open each of these libraries
      ;;; yet again, specifying the RTLD_GLOBAL flag.
      (%dlopen-shlib l)
      (setf (shlib.opened-by-lisp-kernel l) t))))

(init-shared-libraries)

;;; Walk over all registered entrypoints, invalidating any whose container
;;; is the specified library.  Return true if any such entrypoints were
;;; found.
(defun unload-library-entrypoints (lib)
  (let* ((count 0))
    (declare (fixnum count))
    (maphash #'(lambda (k eep)
		 (declare (ignore k))
		 (when (eq (eep.container eep) lib)
		   (setf (eep.address eep) nil)
		   (incf count)))
	     (eeps))    
    (not (zerop count))))

(defun open-shared-library (name)
  (let* ((link-map  (with-cstrs ((name name))
                      (ff-call
		       (%kernel-import arch::kernel-import-GetSharedLibrary)
		       :address name
		       :unsigned-fullword *dlopen-flags*
		       :address))))
    (if (%null-ptr-p link-map)
      (error "Error opening shared library ~s: ~a" name (dlerror))
      (prog1 (shlib-from-map-entry link-map)
	(%walk-shared-libraries
	 #'(lambda (map)
	     (unless (shared-library-at
		      (%int-to-ptr (pref map :link_map.l_addr)))
	       (let* ((new (shlib-from-map-entry map)))
		 (%dlopen-shlib new)))))))))

)

(defun ensure-open-shlib (c force)
  (if (or (shlib.map c) (not force))
    *rtld-default*
    (error "Shared library not open: ~s" (shlib.soname c))))

(defun resolve-container (c force)
  (if c
    (ensure-open-shlib c force)
    *rtld-default*
    ))




;;; An "entry" is a fixnum (the low 2 bits are clear) which represents
;;; a 32-bit, word-aligned address.  This should probably only be used
;;; for function entrypoints, since it treats a return value of 0 as
;;; invalid.

(defun foreign-symbol-entry (name &optional (handle *rtld-default*))
  (with-cstrs ((n name))
    (let* ((addr (ff-call (%kernel-import arch::kernel-import-FindSymbol) :address handle :address n :signed-fullword)))
      (declare (integer addr))
      (unless (zerop addr)		; No function can have address 0
	(ash addr -2)))))

#+linux-target
(progn
(defvar *dladdr-entry*)
(setq *dladdr-entry* (foreign-symbol-entry "dladdr"))

(defun shlib-containing-address (address)
   (rletZ ((info :dl_info))
     (let* ((status (ff-call *dladdr-entry*
			     :address address
			     :address info :signed-fullword)))
       (declare (integer status))
       (unless (zerop status)
	 (shared-library-at (pref info :dl_info.dli_fbase))))))


(defun shlib-containing-entry (entry)
  (with-macptrs (p)
    (%setf-macptr-to-object p entry)
    (shlib-containing-address p))))

#-linux-target
(defun shlib-containing-entry (entry)
  (declare (ignore entry))
  *rtld-default*)


(defun resolve-eep (e &optional (require-resolution t))
  (or (eep.address e)
      (let* ((name (eep.name e))
	     (container (eep.container e))
             (handle (resolve-container container require-resolution))
	     (addr (foreign-symbol-entry name handle)))
	(if addr
	  (progn
	    (unless container
	      (setf (eep.container e) (shlib-containing-entry addr)))
	    (setf (eep.address e) addr))
	  (if require-resolution
	    (error "Can't resolve foreign symbol ~s" name))))))



(defun foreign-symbol-address (name &optional (map *rtld-default*))
  (with-cstrs ((n name))
    (let* ((addr (ff-call (%kernel-import arch::kernel-import-FindSymbol) :address map :address n :address)))
      (unless (%null-ptr-p addr)
        addr))))

(defun load-eep (name)
  (let* ((eep (or (gethash name (eeps)) (setf (gethash name *eeps*) (%cons-external-entry-point name)))))
    (resolve-eep eep nil)
    eep))

#+ppc-target
(defppclapfunction %revive-macptr ((p arg_z))
  (li imm0 arch::subtag-macptr)
  (stb imm0 arch::misc-subtag-offset p)
  (blr))

#+sparc-target
(defsparclapfunction %revive-macptr ((p %arg_z))
  (mov arch::subtag-macptr %imm0)
  (retl)
  (stb %imm0 (p arch::misc-subtag-offset)))

#+linux-target
(progn
;;; It's assumed that the set of libraries that the OS has open
;;; (accessible via the _dl_loaded global variable) is a subset of
;;; the libraries on *shared-libraries*.

(defun revive-shared-libraries ()
  (dolist (lib *shared-libraries*)
    (setf (shlib.map lib) nil
	  (shlib.pathname lib) nil
	  (shlib.base lib) nil)
    (let* ((soname (shlib.soname lib)))
      (when soname
	(with-cstrs ((soname soname))
	  (let* ((map (block found
			(%walk-shared-libraries
			 #'(lambda (m)
			     (with-macptrs (libname)
			       (%setf-macptr libname
					     (%get-ptr
					      (link-map-libname m)))
			       (unless (%null-ptr-p libname)
				 (when (%cstrcmp soname libname)
				   (return-from found  m)))))))))
	    (when map
	      ;;; Sigh.  We can't reliably lookup symbols in the library
	      ;;; unless we open the library (which is, of course,
	      ;;; already open ...)  ourselves, passing in the
	      ;;; #$RTLD_GLOBAL flag.
	      (ff-call (%kernel-import arch::kernel-import-GetSharedLibrary)
		       :address soname
		       :unsigned-fullword *dlopen-flags*
		       :void)
	      (setf (shlib.base lib) (%int-to-ptr (pref map :link_map.l_addr))
		    (shlib.pathname lib) (%get-cstring
					  (pref map :link_map.l_name))
		    (shlib.map lib) map))))))))

;;; Repeatedly iterate over shared libraries, trying to open those
;;; that weren't already opened by the kernel.  Keep doing this until
;;; we reach stasis (no failures or no successes.)

(defun %reopen-user-libraries ()
  (loop
      (let* ((win nil)
	     (lose nil))
	(dolist (lib *shared-libraries*)
	  (let* ((map (shlib.map lib)))
	    (unless map
	      (with-cstrs ((soname (shlib.soname lib)))
		(setq map (ff-call
			   (%kernel-import arch::kernel-import-GetSharedLibrary)
			   :address soname
			   :unsigned-fullword *dlopen-flags*
			   :address))
		(if (%null-ptr-p map)
		  (setq lose t)
		  (setf (shlib.pathname lib)
			(%get-cstring (pref map :link_map.l_name))
			(shlib.base lib)
			(%int-to-ptr (pref map :link_map.l_addr))
			(shlib.map lib) map
			win t))))))
	(when (or (not lose) (not win)) (return)))))
)

(defun refresh-external-entrypoints ()
  (%revive-macptr *rtld-next*)
  (%revive-macptr *rtld-default*)
  #+linuxppc-target
  (progn
    (setq *dladdr-entry* (foreign-symbol-entry "dladdr"))
    (revive-shared-libraries)
    (%reopen-user-libraries))
  (when *eeps*
    (without-interrupts 
     (maphash #'(lambda (k v) 
                  (declare (ignore k)) 
                  (setf (eep.address v) nil) 
                  (resolve-eep v nil))
              *eeps*))))




