;;-*- Mode: Lisp; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;


(eval-when (:compile-toplevel :execute)
  (require "ARCH")
  (require "NUMBER-MACROS")
  
  (defconstant digit-size 32)
  (defconstant half-digit-size (/ digit-size 2))
  
  (defconstant maximum-bignum-length (1- (ash 1 24)))

  (deftype bignum-index () `(integer 0 (,maximum-bignum-length)))
  (deftype bignum-element-type () `(unsigned-byte ,digit-size))
  (deftype bignum-half-element-type () `(unsigned-byte ,half-digit-size))
  (deftype bignum-type () 'bignum)
  (defmacro %bignum-digits (bignum)`(uvsize ,bignum))

  (defmacro digit-bind ((&rest digits) form &body body)
    `(multiple-value-bind ,digits
                          ,form
       (declare (type bignum-half-element-type ,@digits))
       ,@body))

  (defmacro digit-set ((&rest digits) form)
    `(multiple-value-setq ,digits
                          ,form))

  (defmacro digit-zerop (h l)
    `(and (zerop ,h) (zerop ,l)))
 


  ;;;; BIGNUM-REPLACE and WITH-BIGNUM-BUFFERS.

  ;;; BIGNUM-REPLACE -- Internal.
  ;;;
  (defmacro bignum-replace (dest src &key (start1 '0) end1 (start2 '0) end2
                                 from-end)
    (arch::once-only ((n-dest dest)
                     (n-src src))
      (if (and (eq start1 0)(eq start2 0)(null end1)(null end2)(null from-end))
        ; this is all true for some uses today <<
        `(%copy-ivector-to-ivector ,n-src 0 ,n-dest 0 (%ilsl 2 (min (the fixnum (%bignum-length ,n-src))
                                                                    (the fixnum (%bignum-length ,n-dest)))))
        (let* ((n-start1 (gensym))
               (n-end1 (gensym))
               (n-start2 (gensym))
               (n-end2 (gensym)))
          `(let ((,n-start1 ,start1)
                 (,n-start2 ,start2)
                 (,n-end1 ,(or end1 `(%bignum-length ,n-dest)))
                 (,n-end2 ,(or end2 `(%bignum-length ,n-src))))
             ,(if (null from-end)            
                `(%copy-ivector-to-ivector
                  ,n-src (%i* 4 ,n-start2) 
                  ,n-dest (%i* 4 ,n-start1)
                  (%i* 4 (min (%i- ,n-end2 ,n-start2) 
                              (%i- ,n-end1 ,n-start1))))
                `(let ((nwds (min (%i- ,n-end2 ,n-start2)
                                  (%i- ,n-end1 ,n-start1))))
                   (%copy-ivector-to-ivector
                    ,n-src (%ilsl 2 (%i- ,n-end2 nwds))
                    ,n-dest (%ilsl 2 (%i- ,n-end1 nwds))
                    (%i* 4 nwds))))))))) 
  

  ;;;; Shifting.
  
  (defconstant all-ones-half-digit #xFFFF)  
  
 (defmacro %logand (h1 l1 h2 l2)
    (arch::once-only ((h1v h1)(l1v l1)(h2v h2)(l2v l2)) ; export this plz
      `(values (%ilogand ,h1v ,h2v)(%ilogand ,l1v ,l2v))))
  
  (defmacro %logior (h1 l1 h2 l2)
    (arch::once-only ((h1v h1)(l1v l1)(h2v h2)(l2v l2))
      `(values (%ilogior ,h1v ,h2v)(%ilogior ,l1v ,l2v))))
  
  (defmacro %logxor (h1 l1 h2 l2)
    (arch::once-only ((h1v h1)(l1v l1)(h2v h2)(l2v l2))
      `(values (%ilogxor ,h1v ,h2v)(%ilogxor ,l1v ,l2v))))
  
  
  (defmacro %lognot (h l)
    (arch::once-only ((h1v h)(l1v l))
      `(values (%ilognot ,h1v)(%ilognot ,l1v))))

  (defmacro %allocate-bignum (ndigits)
    `(%alloc-misc ,ndigits arch::subtag-bignum))

  (defmacro %normalize-bignum-macro (big)
    `(%normalize-bignum-2 t ,big))

  (defmacro %mostly-normalize-bignum-macro (big)
    `(%normalize-bignum-2 nil ,big))


;;; %ALLOCATE-BIGNUM must zero all elements.
;;;
  (declaim (inline  %bignum-length))
  (declaim (inline %digit-0-or-plusp %bignum-0-or-plusp bignum-minusp %digit-compare %bignum-minusp))

)




;(defun %allocate-bignum (ndigits)
;  (%alloc-misc ndigits arch::subtag-bignum))

;;; Extract the length of the bignum.
;;; 
(defun %bignum-length (bignum)
  (uvsize bignum)) 

(defun %digit-0-or-plusp (high)
  (declare (fixnum high))
  (not (logbitp (1- half-digit-size) high))) 

(defun %bignum-0-or-plusp (bignum len)
  (declare (fixnum len))
  (not (%ilogbitp (1- half-digit-size) (%bignum-ref-hi bignum (the fixnum (1- len))))))

(defun %bignum-minusp (bignum len)
  (declare (fixnum len))
  (%ilogbitp (1- half-digit-size) (%bignum-ref-hi bignum (the fixnum (1- len)))))

(defun bignum-minusp (big)
  (%ilogbitp (1- half-digit-size) (%bignum-ref-hi big (1- (%bignum-length big)))))
  






; for oddp, evenp
(defun %bignum-oddp (bignum)
  (%ilogbitp 0 (the fixnum (nth-value 1 (%bignum-ref bignum 0)))))



    

 







    




;;;; Addition.

(defun add-bignums (a b)
  (declare (type bignum-type a b))
  (let* ((len-a (%bignum-length a))
	 (len-b (%bignum-length b))
         (res (%allocate-bignum 
               (if (> len-a len-b)
                 (the fixnum (1+ len-a))
                 (the fixnum (1+ len-b))))))
    (declare (type bignum-index len-a len-b))
    (bignum-add-loop-2 a b res)        
    (%normalize-bignum-macro res)))








;;;; Subtraction.

(defun subtract-bignum (a b)
  (declare (type bignum-type a b))
  (let* ((len-a (%bignum-length a))
	 (len-b (%bignum-length b))
	 (len-res (1+ (max len-a len-b)))
	 (res (%allocate-bignum len-res)))
    (declare (type bignum-index len-a len-b len-res)) ;Test len-res for bounds?
    (bignum-subtract-loop a len-a (if (bignum-minusp a) -1 0)
                          b len-b (if (bignum-minusp b) -1 0)
                          res len-res)
    
    (%normalize-bignum-macro res)))

#-ppc-target				;prefer LAP for PPC
(defun bignum-subtract-loop (a len-a sign-a b len-b sign-b result length)
  (declare (fixnum len-a len-b length sign-a sign-b))
  (let* ((sign-a-h (if (zerop sign-a) 0 #xffff))
	 (sign-a-l (if (zerop sign-a) 0 #xffff))
	 (sign-b-h (if (zerop sign-b) 0 #xffff))
	 (sign-b-l (if (zerop sign-b) 0 #xffff))
	 (borrow 1)
	 (a-h 0)
	 (a-l 0)
	 (b-h 0)
	 (b-l 0)
	 (res-h 0)
	 (res-l 0))
    (dotimes (i length borrow)
      (if (< i len-a)
	(multiple-value-setq (a-h a-l) (%bignum-ref a i))
	(setq a-h sign-a-h a-l sign-a-l))
      (if (< i len-b)
	(multiple-value-setq (b-h b-l) (%bignum-ref b i))
	(setq b-h sign-b-h b-l sign-b-l))
      (multiple-value-setq (res-h res-l borrow)
	(%subtract-with-borrow a-h a-l b-h b-l borrow))
      (%bignum-set result i res-h res-l))))







;;;; Multiplication.


(defun multiply-bignums (a b)
  (declare (type bignum-type a b))
  (let* ((negate-res (neq (bignum-minusp a)(bignum-minusp b))))
    (flet ((do-it (a b)
             (let* ((len-a (%bignum-length a))
                    (len-b (%bignum-length b))
                    (len-res (+ len-a len-b))
	            (res (%allocate-bignum len-res)))
               (declare (type bignum-index len-a len-b len-res))
               (dotimes (i len-a)
                 (declare (type bignum-index i))
                 (%multiply-and-add-harder-loop-2 a b res i len-b))
               res)))
      (declare (dynamic-extent do-it))
      (let ((res (with-negated-bignum-buffers a b do-it)))
        (when negate-res
          (negate-bignum-in-place res))
        (%normalize-bignum-macro res )))))

(defun multiply-bignum-and-fixnum (bignum fixnum)
  (declare (type bignum-type bignum) (fixnum fixnum))
  (let* ((bignum-len (%bignum-length bignum))
         (bignum-plus-p (%bignum-0-or-plusp bignum bignum-len))
	 (fixnum-plus-p (not (minusp fixnum)))
         (negate-res (neq bignum-plus-p fixnum-plus-p)))
    (declare (type bignum-type bignum)
	     (type bignum-index bignum-len))
    (flet ((do-it (bignum fixnum  negate-res)
             (let* ((bignum-len (%bignum-length bignum))
                    (result (%allocate-bignum (the fixnum (1+ bignum-len)))))
               (declare (type bignum-type bignum)
	                (type bignum-index bignum-len))
               (%multiply-and-add-loop bignum result bignum-len fixnum)
               (when negate-res
                 (negate-bignum-in-place result))
               (%normalize-bignum-macro result ))))
      (declare (dynamic-extent do-it))
      (if bignum-plus-p
        (do-it bignum (if fixnum-plus-p fixnum (- fixnum))  negate-res)
        (with-bignum-buffers ((b1 (the fixnum (1+ bignum-len))))
          (negate-bignum bignum nil b1)
          (do-it b1 (if fixnum-plus-p fixnum (- fixnum))  negate-res))))))

;; assume we already know result won't fit in a fixnum
;; only caller is fixnum-*-2
;;

(defun multiply-fixnums (a b)
  (declare (fixnum a b))
  (multiple-value-bind (p1 p2 p3 p4)(%multiply-signed-fixnums a b)
    (declare (fixnum p1 p2 p3 p4))
    (let* ((res-len (if (and (= p1 p2)
                             (or (and (= 0 p2)(not (%ilogbitp 15 p3)))  ; plus - no need hi
                                 (and (= #xffff p2)(%ilogbitp 15 p3)))) ; or minus ditto
                      1
                      2))
           (res (%allocate-bignum res-len)))
      (%bignum-set res 0 p3 p4)  ; least significant
      (if (eq res-len 2)
        (%bignum-set res 1 p1 p2)) ; most significant if needed
      res)))

;;;; GCD.


;;; Both args are > 0.
(defun bignum-fixnum-gcd (bignum fixnum)
  (let* ((rem (bignum-truncate-by-fixnum-no-quo bignum fixnum)))
    (declare (fixnum rem))
    (if (zerop rem)
      fixnum
      (%fixnum-gcd rem fixnum))))



;;; NEGATE-BIGNUM -- Public.
;;;
;;; Fully-normalize is an internal optional.  It cause this to always return
;;; a bignum, without any extraneous digits, and it never returns a fixnum.
;;;
(defun negate-bignum (x &optional (fully-normalize t) res)
  (declare (type bignum-type x))
  (let* ((len-x (%bignum-length x))
	 (len-res (1+ len-x))
         (minusp (%bignum-minusp x len-x)))
    (declare (type bignum-index len-x len-res))
    (if (not res) (setq res (%allocate-bignum len-res))) ;Test len-res for range?
    (let ((carry (bignum-negate-loop-really x len-x res)))  ; i think carry is always 0
      (if (eq carry 0)
        (if minusp (%bignum-set res len-x 0 0)(%bignum-set res len-x #xffff #xffff))
        (digit-bind (h l)
                    (if minusp 
                      (%add-the-carry 0 0 carry)
                      (%add-the-carry #xffff #xffff carry))
                    
          (%bignum-set res len-x h l))))
    (if fully-normalize
      (%normalize-bignum-macro res)
      (%mostly-normalize-bignum-macro res))))

;;; NEGATE-BIGNUM-IN-PLACE -- Internal.
;;;
;;; This assumes bignum is positive; that is, the result of negating it will
;;; stay in the provided allocated bignum.
;;;
(defun negate-bignum-in-place (bignum)
  (bignum-negate-loop-really bignum (%bignum-length bignum) bignum)
  bignum)


  

(defun copy-bignum (bignum)
  (let ((res (%allocate-bignum (%bignum-length bignum))))
    (bignum-replace res bignum)
    res))



;;; BIGNUM-ASHIFT-RIGHT -- Public.
;;;
;;; First compute the number of whole digits to shift, shifting them by
;;; skipping them when we start to pick up bits, and the number of bits to
;;; shift the remaining digits into place.  If the number of digits is greater
;;; than the length of the bignum, then the result is either 0 or -1.  If we
;;; shift on a digit boundary (that is, n-bits is zero), then we just copy
;;; digits.  The last branch handles the general case which uses a macro that a
;;; couple other routines use.  The fifth argument to the macro references
;;; locals established by the macro.
;;;


(defun bignum-ashift-right (bignum x)
  (declare (type bignum-type bignum)
           (fixnum x))
  (let ((bignum-len (%bignum-length bignum)))
    (declare (type bignum-index bignum-len))
    (multiple-value-bind (digits n-bits) (truncate x digit-size)
      (declare (type bignum-index digits)(fixnum n-bits))
      (cond
       ((>= digits bignum-len)
        (if (%bignum-0-or-plusp bignum bignum-len) 0 -1))
       ((eql 0 n-bits)
        (bignum-ashift-right-digits bignum digits))
       (t
        (let* ((res-len (- bignum-len digits))
               (res (%allocate-bignum res-len))
               (len-1 (1- res-len)))
          (declare (fixnum res-len len-1))
          (bignum-shift-right-loop-1 n-bits res bignum len-1 digits)          
          (%normalize-bignum-macro res )))))))

			       



;;; BIGNUM-ASHIFT-RIGHT-DIGITS -- Internal.
;;;
(defun bignum-ashift-right-digits (bignum digits)
  (declare (type bignum-type bignum)
	   (type bignum-index digits))
  (let* ((res-len (- (%bignum-length bignum) digits))
	 (res (%allocate-bignum res-len)))
    (declare (type bignum-index res-len)
	     (type bignum-type res))
    (bignum-replace res bignum :start2 digits)
    (%normalize-bignum-macro res)))


;;; BIGNUM-BUFFER-ASHIFT-RIGHT -- Internal.
;;;
;;; GCD uses this for an in-place shifting operation.  This is different enough
;;; from BIGNUM-ASHIFT-RIGHT that it isn't worth folding the bodies into a
;;; macro, but they share the basic algorithm.  This routine foregoes a first
;;; test for digits being greater than or equal to bignum-len since that will
;;; never happen for its uses in GCD.  We did fold the last branch into a macro
;;; since it was duplicated a few times, and the fifth argument to it
;;; references locals established by the macro.
;;;
#|
(defun bignum-buffer-ashift-right (bignum bignum-len x)
  (declare (type bignum-index bignum-len) (fixnum x))
  (multiple-value-bind (digits n-bits)
		       (truncate x digit-size)
    (declare (type bignum-index digits))
    (cond
     ((zerop n-bits)
      (let ((new-end (- bignum-len digits)))
	(bignum-replace bignum bignum :end1 new-end :start2 digits
			:end2 bignum-len)
	(%normalize-bignum-buffer bignum new-end)))
     (t
      (shift-right-unaligned bignum digits n-bits (- bignum-len digits)
			     ((= j res-len-1)
                              (digit-bind (h l) (%bignum-ref bignum i)
                                (digit-set (h l) (%ashr h l n-bits))
			        (%bignum-set bignum j h l))
			      (%normalize-bignum-buffer bignum res-len)))))))
|#
#|
(defun bignum-buffer-ashift-right (bignum bignum-len x)
  (declare (type bignum-index bignum-len) (fixnum x))
  (multiple-value-bind (digits n-bits) (truncate x digit-size)
    (declare (type bignum-index digits)(fixnum n-bits))
    (macrolet ((clear-high-digits ()
                 `(do* ((i (1- (the fixnum (%bignum-length bignum))) (1- i))
                        (j digits (1- j)))
                       ((= 0 j))
                    (declare (fixnum i j))
                    (%bignum-set bignum i 0 0))))
      (cond
       ((zerop n-bits)
        (let* ((new-end (- bignum-len digits)))
          (declare (fixnum new-end))
          (bignum-replace bignum bignum :end1 new-end :start2 digits
                          :end2 bignum-len)
          (clear-high-digits)
          (%normalize-bignum-buffer bignum new-end)))
       (t
        (let* ((res-len (- bignum-len digits))
               (len-1 (1- res-len)))
          (declare (fixnum res-len len-1))
          (bignum-shift-right-loop-1 n-bits bignum bignum len-1 digits)
          ; clear the old high order digits - assume always positive
          ; (when (neq 0 digits)(push digits poof))
          (clear-high-digits)
          (%normalize-bignum-buffer bignum res-len)))))))
|#

 

;;; BIGNUM-ASHIFT-LEFT -- Public.
;;;
;;; This handles shifting a bignum buffer to provide fresh bignum data for some
;;; internal routines.  We know bignum is safe when called with bignum-len.
;;; First we compute the number of whole digits to shift, shifting them
;;; starting to store farther along the result bignum.  If we shift on a digit
;;; boundary (that is, n-bits is zero), then we just copy digits.  The last
;;; branch handles the general case.
;;;
(defun bignum-ashift-left (bignum x &optional bignum-len)
  (declare (type bignum-type bignum)
	   (fixnum x)
	   (type (or null bignum-index) bignum-len))
  (multiple-value-bind (digits n-bits)
		       (truncate x digit-size)
    (declare (fixnum digits n-bits))
    (let* ((bignum-len (or bignum-len (%bignum-length bignum)))
	   (res-len (+ digits bignum-len 1)))
      (declare (fixnum bignum-len res-len))
      (when (> res-len maximum-bignum-length)
	(error "Can't represent result of left shift."))
      (if (zerop n-bits)
        (bignum-ashift-left-digits bignum bignum-len digits)
        (bignum-ashift-left-unaligned bignum digits n-bits res-len)))))

;;; BIGNUM-ASHIFT-LEFT-DIGITS -- Internal.
;;;
(defun bignum-ashift-left-digits (bignum bignum-len digits)
  (declare (type bignum-index bignum-len digits))
  (let* ((res-len (+ bignum-len digits))
	 (res (%allocate-bignum res-len)))
    (declare (type bignum-index res-len))
    (bignum-replace res bignum :start1 digits :end1 res-len :end2 bignum-len
		    :from-end t)
    res))

;;; BIGNUM-ASHIFT-LEFT-UNALIGNED -- Internal.
;;;
;;; BIGNUM-TRUNCATE uses this to store into a bignum buffer by supplying res.
;;; When res comes in non-nil, then this foregoes allocating a result, and it
;;; normalizes the buffer instead of the would-be allocated result.
;;;
;;; We start storing into one digit higher than digits, storing a whole result
;;; digit from parts of two contiguous digits from bignum.  When the loop
;;; finishes, we store the remaining bits from bignum's first digit in the
;;; first non-zero result digit, digits.  We also grab some left over high
;;; bits from the last digit of bignum.
;;;

#| 
(defun bignum-ashift-left-unaligned (bignum digits n-bits res-len
				     &optional (res nil resp))
  (declare (type bignum-index digits res-len)
	   (type (mod #.digit-size) n-bits))
  (let* ((remaining-bits (- digit-size n-bits))
	 (res-len-1 (1- res-len))
	 (res (or res (%allocate-bignum res-len))))
    (declare (type bignum-index res-len res-len-1))
    (do ((i 0 i+1)
	 (i+1 1 (1+ i+1))
	 (j (1+ digits) (1+ j)))
	((= j res-len-1)
         (digit-bind (h l) (%bignum-ref bignum 0)
           (digit-set (h l) (%ashl h l n-bits))
	   (%bignum-set res digits h l))
         (digit-bind (h l) (%bignum-ref bignum i)
           (digit-set (h l) (%ashr h l remaining-bits))
	   (%bignum-set res j h l))
	 (if resp
           (%normalize-bignum-buffer res res-len)
           (%normalize-bignum res res-len)))
      (declare (type bignum-index i i+1 j))
      (digit-bind (h l)
                  (digit-bind (a-h a-l) (%bignum-ref bignum i)
                    (digit-set (a-h a-l) (%digit-logical-shift-right a-h a-l remaining-bits))
                    (digit-bind (b-h b-l) (%bignum-ref bignum i+1)
                      (digit-set (b-h b-l) (%ashl b-h b-l n-bits))
                      (%logior a-h a-l b-h b-l)))
        (%bignum-set res j h l)))))
|#
#|
(defun bignum-ashift-left-unaligned (bignum digits n-bits res-len
                                              &optional (res nil resp))
  (declare (type bignum-index digits res-len)
           (type (mod #.digit-size) n-bits))
  (let* ((remaining-bits (- digit-size n-bits))
         (res-len-1 (1- res-len))
         (res (or res (%allocate-bignum res-len))))
    (declare (type bignum-index res-len res-len-1))
    (Unless (= 0 res-len-1)
      (bignum-shift-left-loop n-bits res bignum res-len-1 (1+ digits)))
    (digit-bind (h l) (%bignum-ref bignum 0) ; do first (lo)
      (digit-set (h l) (%ashl h l n-bits))
      (%bignum-set res digits h l))
    (let* ((i (- res-len-1 (1+ digits)))
           (j res-len-1))
      (declare (fixnum i j))
      (digit-bind (h l) (%bignum-ref bignum i) ; do last (hi)
        (digit-set (h l) (%ashr h l remaining-bits))
        (%bignum-set res j h l)))
    (if resp
      (%normalize-bignum-buffer res res-len)
      (%normalize-bignum res res-len))))
|#

(defun bignum-ashift-left-unaligned (bignum digits n-bits res-len
                                              &optional (res nil resp))
  (declare (type bignum-index digits res-len)
           (type (mod #.digit-size) n-bits))
  (let* (;(remaining-bits (- digit-size n-bits))
         (res-len-1 (1- res-len))
         (res (or res (%allocate-bignum res-len))))
    (declare (type bignum-index res-len res-len-1))
    (bignum-shift-left-loop n-bits res bignum res-len-1 (the fixnum (1+ digits)))
    ; if resp provided we don't care about returned value
    (if (not resp) (%normalize-bignum-macro res))))






;;;; Relational operators.

;;; BIGNUM-PLUS-P -- Public.
;;;
;;; Return T iff bignum is positive.
;;; 
(defun bignum-plus-p (bignum)
  (declare (type bignum-type bignum))
  (let ((len (%bignum-length bignum)))
    (declare (fixnum len))
    (not (%ilogbitp (1- half-digit-size) (%bignum-ref-hi bignum (the fixnum (1- len)))))))

;;; BIGNUM-COMPARE -- Public.
;;;
;;; This compares two bignums returning -1, 0, or 1, depending on whether a
;;; is less than, equal to, or greater than b.
;;;
;(proclaim '(function bignum-compare (bignum bignum) (integer -1 1)))
(defun bignum-compare (a b &optional (len-a (%bignum-length a)) (len-b (%bignum-length b)))
  (declare (type bignum-type a b))
  (let* (;(len-a (%bignum-length a))
	 ;(len-b (%bignum-length b))
	 (a-plusp (%bignum-0-or-plusp a len-a))
	 (b-plusp (%bignum-0-or-plusp b len-b)))
    (declare (type bignum-index len-a len-b))
    (cond ((not (eq a-plusp b-plusp))
	   (if a-plusp 1 -1))
	  ((= len-a len-b)
           (bignum-compare-loop a b len-a))
          #|
	   (do ((i (1- len-a) (1- i)))
	       (())
	     (declare (type bignum-index i))
	     (digit-bind (a-digit-h a-digit-l) (%bignum-ref a i)
               (digit-bind (b-digit-h b-digit-l) (%bignum-ref b i)
	         (when (%digit-greater a-digit-h a-digit-l b-digit-h b-digit-l)
		   (return 1))
	         (when (%digit-greater b-digit-h b-digit-l a-digit-h a-digit-l)
		   (return -1))))
	     (when (zerop i) (return 0)))) |#
	  ((> len-a len-b)
	   (if a-plusp 1 -1))
	  (t (if a-plusp -1 1)))))






;;;; Integer length and logcount


(defun bignum-integer-length (big)
  (let ((ndigits (uvsize big)))
    (declare (fixnum ndigits))
    (multiple-value-bind (most-hi most-lo)(%bignum-ref big (1- ndigits))
      (declare (fixnum most-hi most-lo))
      (%i- (ash ndigits 5)
           (%digits-sign-bits most-hi most-lo)))))





; (not (zerop (logand integer1 integer2)

(defun bignum-logtest (num1 num2)
  (let* ((length1 (%bignum-length num1))
         (length2 (%bignum-length num2))
         (n1-minusp (%bignum-minusp num1 length1))
         (n2-minusp (%bignum-minusp num2 length2)))
    (declare (fixnum length1 length2))
    (if (and n1-minusp n2-minusp) ; both neg, get out quick
      T        
      (let ((val (bignum-logtest-loop (min length1 length2) num1 num2)))
                 #|(do* ((index 0 (1+ index)))
	              ((= index (min length1 length2)) nil)
                   ; maybe better to start from high end of shorter?
                   (multiple-value-bind (hi1 lo1)(%bignum-ref num1 index)
                     (multiple-value-bind (hi2 lo2)(%bignum-ref num2 index)
                       (when (or (not (zerop (%ilogand hi1 hi2)))
                                 (not (zerop (%ilogand lo1 lo2))))
                         (return t)))))))|#
        (or val
            (when (not (eql length1 length2)) ; lengths same => value nil
              (if (< length1 length2)
                n1-minusp
                n2-minusp)))))))



(defun logtest-fix-big (fix big)
  (declare (fixnum fix))
  (if (eql 0 (the fixnum fix))
    nil
    (if (> (the fixnum fix) 0) 
      (let ()
        (multiple-value-bind (hi lo)(%bignum-ref big 0)
          (declare (fixnum hi lo))
          (or (not (zerop (logand fix lo)))
              (not (zerop (logand (ash fix (- 16)) hi))))))
      t)))


(defun bignum-logcount (bignum)
  (declare (type bignum-type bignum))
  (let* ((length (%bignum-length bignum))
	 (plusp (%bignum-0-or-plusp bignum length))
	 (result 0))
    (declare (type bignum-index length)
	     (fixnum result))
    (do ((index 0 (1+ index)))
	((= index length) result)
      (declare (fixnum index))
      (digit-bind (digit-h digit-l) (%bignum-ref bignum index)
        (unless plusp
          (digit-set (digit-h digit-l) (%lognot digit-h digit-l)))
	(incf result (%logcount digit-h digit-l))))))


;;;; Logical operations.

;;; NOT.
;;;

;;; BIGNUM-LOGICAL-NOT -- Public.
;;;
(defun bignum-logical-not (a)
  (declare (type bignum-type a))
  (let* ((len (%bignum-length a))
	 (res (%allocate-bignum len)))
    (declare (type bignum-index len))
    (bignum-not-loop len a res)
    #|
    (dotimes (i len res)
      (declare (type bignum-index i))
      (digit-bind (h l) (%bignum-ref a i)
        (%bignum-set res i (%ilognot h) (%ilognot l))))|#
    ))




;;; AND.
;;;

;;; BIGNUM-LOGICAL-AND -- Public.
;;;
(defun bignum-logical-and (a b)
  (declare (type bignum-type a b))
  (let* ((len-a (%bignum-length a))
	 (len-b (%bignum-length b))
	 (a-plusp (%bignum-0-or-plusp a len-a))
	 (b-plusp (%bignum-0-or-plusp b len-b)))
    (declare (type bignum-index len-a len-b))
    (cond
     ((< len-a len-b)
      (if a-plusp
	  (logand-shorter-positive a len-a b (%allocate-bignum len-a))
	  (logand-shorter-negative a len-a b len-b (%allocate-bignum len-b))))
     ((< len-b len-a)
      (if b-plusp
	  (logand-shorter-positive b len-b a (%allocate-bignum len-b))
	  (logand-shorter-negative b len-b a len-a (%allocate-bignum len-a))))
     (t (logand-shorter-positive a len-a b (%allocate-bignum len-a))))))

;;; LOGAND-SHORTER-POSITIVE -- Internal.
;;;
;;; This takes a shorter bignum, a and len-a, that is positive.  Because this
;;; is AND, we don't care about any bits longer than a's since its infinite 0
;;; sign bits will mask the other bits out of b.  The result is len-a big.
;;;
(defun logand-shorter-positive (a len-a b res)
  (declare (type bignum-type a b res)
	   (type bignum-index len-a))
  (bignum-and-loop len-a a b res)
  #|
  (dotimes (i len-a)
    (declare (type bignum-index i))
    (digit-bind (a-h a-l) (%bignum-ref a i)
     (digit-bind (b-h b-l) (%bignum-ref b i)
      (%bignum-set res i (%ilogand a-h b-h)(%ilogand a-l b-l)))))|#
  (%normalize-bignum-macro res))

;;; LOGAND-SHORTER-NEGATIVE -- Internal.
;;;
;;; This takes a shorter bignum, a and len-a, that is negative.  Because this
;;; is AND, we just copy any bits longer than a's since its infinite 1 sign
;;; bits will include any bits from b.  The result is len-b big.
;;;
(defun logand-shorter-negative (a len-a b len-b res)
  (declare (type bignum-type a b res)
	   (type bignum-index len-a len-b))
  (bignum-and-loop len-a a b res)
  #|
  (dotimes (i len-a)
    (declare (type bignum-index i))
    (digit-bind (a-h a-l) (%bignum-ref a i)
     (digit-bind (b-h b-l) (%bignum-ref b i)
      (%bignum-set res i (%ilogand a-h b-h)(%ilogand a-l b-l)))))|#
  (bignum-replace res b :start1 len-a :start2 len-a :end1 len-b :end2 len-b)
  (%normalize-bignum-macro res))



;;;
;;;
;;; bignum-logandc2

(defun bignum-logandc2 (a b)
  (declare (type bignum-type a b))
  (let* ((len-a (%bignum-length a))
	 (len-b (%bignum-length b))
	 (a-plusp (%bignum-0-or-plusp a len-a))
	 (b-plusp (%bignum-0-or-plusp b len-b)))
    (declare (type bignum-index len-a len-b))
    (cond
     ((< len-a len-b)
      (logandc2-shorter-any a len-a b len-b (if a-plusp (%allocate-bignum len-a) (%allocate-bignum len-b))))
     ((< len-b len-a) ; b shorter 
      (logandc1-shorter-any b len-b a len-a (if b-plusp (%allocate-bignum len-a)(%allocate-bignum len-b))))
     (t (logandc2-shorter-any a len-a b len-b (%allocate-bignum len-a))))))

(defun logandc2-shorter-any (a len-a b len-b res)
  (declare (type bignum-type a b res)
           (type bignum-index len-a len-b))
  (bignum-andc2-loop len-a a b res)
  #|
  (dotimes (i len-a)
    (declare (type bignum-index i))
    (digit-bind (ah al) (%bignum-ref a i)
      (digit-bind (bh bl) (%bignum-ref b i)
        (%bignum-set res i (%ilogand ah (%ilognot bh ))(%ilogand al (%ilognot bl))))))|#
  (if (bignum-minusp a)
    (progn
      (do ((i len-a (1+ i)))
          ((= i len-b))
        (declare (type bignum-index i))
        (digit-bind (h l) (%bignum-ref b i)
          (%bignum-set res i (%ilognot h) (%ilognot l))))
      (%normalize-bignum-macro res))
    (%normalize-bignum-macro res)))



(defun logandc1-shorter-any (a len-a b len-b res)
  (declare (type bignum-type a b res)
           (type bignum-index len-a len-b))
  (bignum-andc1-loop len-a a b res)
  #|
  (dotimes (i len-a)
    (declare (type bignum-index i))
    (digit-bind (ah al) (%bignum-ref a i)
      (digit-bind (bh bl) (%bignum-ref b i)
        (%bignum-set res i (%ilogand (%ilognot ah) bh )(%ilogand (%ilognot al)  bl)))))|#
  (if (%bignum-0-or-plusp a len-a)
    (progn
      (if (neq len-a len-b)
        (bignum-replace res b :start1 len-a :start2 len-a :end1 len-b :end2 len-b))      
      (%normalize-bignum-macro res))
    (%normalize-bignum-macro res)))



(defun fix-big-logand (fix big)
  (let* ((len-b (%bignum-length big))
         (res (if (< fix 0)(%allocate-bignum len-b))))
    (declare (fixnum fix len-b))        
    (let ((val (fix-digit-logand fix big res)))
      (if res
        (progn
          (bignum-replace res big :start1 1 :start2 1 :end1 len-b :end2 len-b)
          (%normalize-bignum-macro res))
        val))))
  

(defun fix-big-logandc2 (fix big)
  (let* ((len-b (%bignum-length big))
         (res (if (< fix 0)(%allocate-bignum len-b))))
    (declare (fixnum fix len-b))        
    (let ((val (fix-digit-logandc2 fix big res)))
      (if res
        (progn
          (do ((i 1 (1+ i)))
              ((= i len-b))
            (declare (type bignum-index i))
            (digit-lognot-move i big res))
          (%normalize-bignum-macro res))
        val))))

(defun fix-big-logandc1 (fix big)
  (let* ((len-b (%bignum-length big))
         (res (if (>= fix 0)(%allocate-bignum len-b))))
    (declare (fixnum fix len-b))        
    (let ((val (fix-digit-logandc1 fix big res)))
      (if res
        (progn  
          (bignum-replace res big :start1 1 :start2 1 :end1 len-b :end2 len-b)
          (%normalize-bignum-macro res))
        val))))







;;; IOR.
;;;

;;; BIGNUM-LOGICAL-IOR -- Public.
;;;
(defun bignum-logical-ior (a b)
  (declare (type bignum-type a b))
  (let* ((len-a (%bignum-length a))
	 (len-b (%bignum-length b))
	 (a-plusp (%bignum-0-or-plusp a len-a))
	 (b-plusp (%bignum-0-or-plusp b len-b)))
    (declare (type bignum-index len-a len-b))
    (cond
     ((< len-a len-b)
      (if a-plusp
	  (logior-shorter-positive a len-a b len-b (%allocate-bignum len-b))
	  (logior-shorter-negative a len-a b len-b (%allocate-bignum len-b))))
     ((< len-b len-a)
      (if b-plusp
	  (logior-shorter-positive b len-b a len-a (%allocate-bignum len-a))
	  (logior-shorter-negative b len-b a len-a (%allocate-bignum len-a))))
     (t (logior-shorter-positive a len-a b len-b (%allocate-bignum len-a)))))) ;;; LOGIOR-SHORTER-POSITIVE -- Internal.
;;;
;;; This takes a shorter bignum, a and len-a, that is positive.  Because this
;;; is IOR, we don't care about any bits longer than a's since its infinite
;;; 0 sign bits will mask the other bits out of b out to len-b.  The result
;;; is len-b long.
;;;
(defun logior-shorter-positive (a len-a b len-b res)
  (declare (type bignum-type a b res)
	   (type bignum-index len-a len-b))
  (bignum-ior-loop len-a a b res)
  #|
  (dotimes (i len-a) ; <<
    (declare (type bignum-index i))
    (digit-bind (a-h a-l) (%bignum-ref a i)
      (digit-bind (b-h b-l) (%bignum-ref b i)
      (%bignum-set res i (%ilogior a-h b-h) (%ilogior a-l b-l)))))|#
  
  (if (not (eql len-a len-b))
    (bignum-replace res b :start1 len-a :start2 len-a :end1 len-b :end2 len-b))
  (%normalize-bignum-macro res))

;;; LOGIOR-SHORTER-NEGATIVE -- Internal.
;;;
;;; This takes a shorter bignum, a and len-a, that is negative.  Because this
;;; is IOR, we just copy any bits longer than a's since its infinite 1 sign
;;; bits will include any bits from b.  The result is len-b long.
;;;
(defun logior-shorter-negative (a len-a b len-b res)
  (declare (type bignum-type a b res)
	   (type bignum-index len-a len-b))
  #|
  (dotimes (i len-a)
    (declare (type bignum-index i)) ; <<
    (digit-bind (a-h a-l) (%bignum-ref a i)
      (digit-bind (b-h b-l) (%bignum-ref b i)
      (%bignum-set res i (%ilogior a-h b-h) (%ilogior a-l b-l)))))|#
  (bignum-ior-loop len-a a b res)
  ; silly to propagate sign and then normalize it away
  ; but may need to do at least once - but we are only normalizing from len-a?
  ; ah but the sign needs to be correct
  (do ((i len-a (1+ i)))
      ((= i len-b))
    (declare (type bignum-index i))
    (%bignum-set res i #xffff #xffff))
  (%normalize-bignum-macro res))




;;; XOR.
;;;

;;; BIGNUM-LOGICAL-XOR -- Public.
;;;
(defun bignum-logical-xor (a b)
  (declare (type bignum-type a b))
  (let ((len-a (%bignum-length a))
	(len-b (%bignum-length b)))
    (declare (type bignum-index len-a len-b))
    (if (< len-a len-b)
	(bignum-logical-xor-aux a len-a b len-b (%allocate-bignum len-b))
	(bignum-logical-xor-aux b len-b a len-a (%allocate-bignum len-a)))))

;;; BIGNUM-LOGICAL-XOR-AUX -- Internal.
;;;
;;; This takes the the shorter of two bignums in a and len-a.  Res is len-b
;;; long.  Do the XOR.
;;;
(defun bignum-logical-xor-aux (a len-a b len-b res)
  (declare (type bignum-type a b res)
	   (type bignum-index len-a len-b))
  (bignum-xor-loop len-a a b res)
  #|
  (dotimes (i len-a)  ; <<
    (declare (type bignum-index i))
    (digit-bind (a-h a-l) (%bignum-ref a i)
      (digit-bind (b-h b-l) (%bignum-ref b i)
      (%bignum-set res i (%ilogxor a-h b-h)(%ilogxor a-l b-l)))))|#
  (if (neq len-a len-b)
    (let ((sign (if (bignum-minusp a) #xffff 0)))
      (do ((i len-a (1+ i)))
          ((= i len-b))
        (declare (type bignum-index i))
        (digit-bind (h l) (%bignum-ref b i)
          (%bignum-set res i (%ilogxor sign h)(%ilogxor sign l))))))
  (%normalize-bignum-macro res))





;;;; LDB (load byte)

; [slh] 'twas all commented out - thank gawd


;;;; TRUNCATE.

;;; This is the original sketch of the algorithm from which I implemented this
;;; TRUNCATE, assuming both operands are bignums.  I should modify this to work
;;; with the documentation on my functions, as a general introduction.  I've
;;; left this here just in case someone needs it in the future.  Don't look
;;; at this unless reading the functions' comments leaves you at a loss.
;;; Remember this comes from Knuth, so the book might give you the right general
;;; overview.
;;; 
;;;
;;; (truncate x y):
;;;
;;; If X's magnitude is less than Y's, then result is 0 with remainder X.
;;;
;;; Make x and y positive, copying x if it is already positive.
;;;
;;; Shift y left until there's a 1 in the 30'th bit (most significant, non-sign
;;;       digit)
;;;    Just do most sig digit to determine how much to shift whole number.
;;; Shift x this much too.
;;; Remember this initial shift count.
;;;
;;; Allocate q to be len-x minus len-y quantity plus 1.
;;;
;;; i = last digit of x.
;;; k = last digit of q.
;;;
;;; LOOP
;;;
;;; j = last digit of y.
;;;
;;; compute guess.
;;; if x[i] = y[j] then g = #xFFFFFFFF
;;; else g = x[i]x[i-1]/y[j].
;;;
;;; check guess.
;;; %UNSIGNED-MULTIPLY returns b and c defined below.
;;;    a = x[i-1] - (logand (* g y[j]) #xFFFFFFFF).
;;;       Use %UNSIGNED-MULTIPLY taking low-order result.
;;;    b = (logand (ash (* g y[j-1]) -32) #xFFFFFFFF).
;;;    c = (logand (* g y[j-1]) #xFFFFFFFF).
;;; if a < b, okay.
;;; if a > b, guess is too high
;;;    g = g - 1; go back to "check guess".
;;; if a = b and c > x[i-2], guess is too high
;;;    g = g - 1; go back to "check guess".
;;; GUESS IS 32-BIT NUMBER, SO USE THING TO KEEP IN SPECIAL REGISTER
;;; SAME FOR A, B, AND C.
;;;
;;; Subtract g * y from x[i - len-y+1]..x[i].  See paper for doing this in step.
;;; If x[i] < 0, guess is fucked.
;;;    negative g, then add 1
;;;    zero or positive g, then subtract 1
;;; AND add y back into x[len-y+1..i].
;;;
;;; q[k] = g.
;;; i = i - 1.
;;; k = k - 1.
;;;
;;; If k>=0, goto LOOP.
;;;
;;;
;;; Now quotient is good, but remainder is not.
;;; Shift x right by saved initial left shifting count.
;;;
;;; Check quotient and remainder signs.
;;; x pos y pos --> q pos r pos
;;; x pos y neg --> q neg r pos
;;; x neg y pos --> q neg r neg
;;; x neg y neg --> q pos r neg
;;;
;;; Normalize quotient and remainder.  Cons result if necessary.
;;;



;;; These are used by BIGNUM-TRUNCATE and friends in the general case.
;;;
(defvar *truncate-x* nil)
(defvar *truncate-y* nil)

;;; BIGNUM-TRUNCATE -- Public.
;;;
;;; This divides x by y returning the quotient and remainder.  In the general
;;; case, we shift y to setup for the algorithm, and we use two buffers to save
;;; consing intermediate values.  X gets destructively modified to become the
;;; remainder, and we have to shift it to account for the initial Y shift.
;;; After we multiple bind q and r, we first fix up the signs and then return
;;; the normalized results.
;;;


(defun bignum-truncate (x1 y1 &optional no-rem)
  (declare (type bignum-type x1 y1))
  (let* ((x-plusp (%bignum-0-or-plusp x1 (%bignum-length x1)))
	 (y-plusp (%bignum-0-or-plusp y1 (%bignum-length y1))))
    (flet 
      ((do-it (x y) 
         (let* ((len-x (%bignum-length x))
                (len-y (%bignum-length y)))
           (declare (fixnum len-x len-y))
           
           (let ((c (bignum-compare y x len-y len-x)))
             (cond 
              ((eql c 1)  ; >
               (return-from bignum-truncate (values 0 x1)))
              ((eql c 0)(values 1 0))  ; =  might as well since did compare anyway
              ((< len-y 2)
               (multiple-value-bind (q r)
                                    (bignum-truncate-single-digit x len-x y no-rem)
                 (values q
                         (unless no-rem
                           (cond (x-plusp r)
                                 ((typep r 'fixnum) (the fixnum (- (the fixnum r))))
                                 (t (negate-bignum-in-place r)
                                    (%normalize-bignum-macro r )))))))
              (t
               (let* ((len-x+1 (1+ len-x)))
                 (declare (fixnum len-x+1))
                 (with-bignum-buffers ((*truncate-x* len-x+1)
                                       (*truncate-y* (the fixnum (1+ len-y))))
                   (let ((y-shift (shift-y-for-truncate y)))
                     (shift-and-store-truncate-buffers x len-x y len-y y-shift)
                     (values (do-truncate len-x+1 len-y)
                             ;; DO-TRUNCATE must execute first.
                             (when (not no-rem)                               
                               (when (not (eql 0 y-shift))                                  
                                 (let* ((res-len-1 (1- len-y)))
                                   (declare (fixnum res-len-1))
                                   (bignum-shift-right-loop-1 y-shift *truncate-x* *truncate-x* res-len-1 0)))                                
                               (let ((the-res (%normalize-bignum-macro *truncate-x* )))
                                 (if (not (fixnump the-res))
                                   (if x-plusp (copy-bignum the-res) (negate-bignum the-res))
                                   (if x-plusp the-res (the fixnum (- (the fixnum the-res)))))
                                     ))))))))))))
      (multiple-value-bind (q r)(with-negated-bignum-buffers x1 y1 do-it)
        (let ((quotient (cond ((eq x-plusp y-plusp) q)
                              ((typep q 'fixnum) (the fixnum (- (the fixnum q))))
                              (t (negate-bignum-in-place q)
                                 (%normalize-bignum-macro q )))))
          (if no-rem
            quotient            
            (values quotient r)))))))

(defun bignum-rem (x1 y1)
  (declare (type bignum-type x1 y1))  
  (let* ((x-plusp (%bignum-0-or-plusp x1 (%bignum-length x1)))
	 ;(y-plusp (%bignum-0-or-plusp y1 (%bignum-length y1)))
         )
    (flet 
      ((do-it (x y) 
         (let* ((len-x (%bignum-length x))
                (len-y (%bignum-length y)))
           (declare (fixnum len-x len-y))           
           (let ((c (bignum-compare y x len-y len-x)))
             (cond 
              ((eql c 1) (return-from bignum-rem x1))
              ((eql c 0) 0)  ; =  might as well since did compare anyway
              ((< len-y 2)
               (let ((r (bignum-truncate-single-digit-no-quo x len-x y)))  ; phooey 
                 (cond (x-plusp r)
                       ((typep r 'fixnum) (the fixnum (- (the fixnum r))))
                       (t (negate-bignum-in-place r)
                          (%normalize-bignum-macro r )))))
              (t
               (let* ((len-x+1 (1+ len-x)))
                 (declare (fixnum len-x+1))
                 (with-bignum-buffers ((*truncate-x* len-x+1)
                                       (*truncate-y* (the fixnum (1+ len-y))))
                   (let ((y-shift (shift-y-for-truncate y)))
                     (shift-and-store-truncate-buffers x len-x y len-y y-shift)
                     (do-truncate-no-quo len-x+1 len-y)
                     (when (not (eql 0 y-shift))                                 
                       (let* ((res-len-1 (1- len-y)))
                         (declare (fixnum res-len-1))
                         (bignum-shift-right-loop-1 y-shift *truncate-x* *truncate-x* res-len-1 0)))
                     (let ((the-res (%normalize-bignum-macro *truncate-x*)))
                       (if (not (fixnump the-res))
                         (if x-plusp (copy-bignum the-res) (negate-bignum the-res))
                         (if x-plusp the-res (the fixnum (- (the fixnum the-res)))))))))))))))
      (declare (dynamic-extent do-it))
      (with-negated-bignum-buffers x1 y1 do-it))))



;;; BIGNUM-TRUNCATE-SINGLE-DIGIT -- Internal.
;;;
;;; This divides x by y when y is a single bignum digit.  BIGNUM-TRUNCATE fixes
;;; up the quotient and remainder with respect to sign and normalization.
;;;
;;; We don't have to worry about shifting y to make its most significant digit
;;; sufficiently large for %FLOOR to return 32-bit quantities for the q-digit
;;; and r-digit.  If y is a single digit bignum, it is already large enough
;;; for %FLOOR.  That is, it has some bits on pretty high in the digit.
;;;
;;; x is positive
(defun bignum-truncate-single-digit (x len-x y &optional no-rem)
  (declare (type bignum-index len-x))
  (let* ((maybe-q (%allocate-bignum 2))
         (q (if (<= len-x 2) maybe-q (%allocate-bignum len-x)))
	 (r-h 0)
         (r-l 0))
    (declare (dynamic-extent maybe-q))
    (digit-bind (y-h y-l) (%bignum-ref y 0)
      (multiple-value-setq (r-h r-l)(%floor-loop-quo x q y-h y-l))      
      (if (eq q maybe-q)
        (progn 
          (setq q (%normalize-bignum-macro q))
          (if (not (fixnump q)) (setq q (copy-bignum q))))
        (setq q (%normalize-bignum-macro q )))
      ; might as well make a fixnum if possible
      (if no-rem
        q
        (if (> (%digits-sign-bits r-h r-l)  arch::fixnumshift)
          (values q (%ilogior (%ilsl 16 r-h) r-l))
          (let ((rem (%allocate-bignum 1)))
            (%bignum-set rem 0 r-h r-l)
            (values q rem)))))))

; aka rem
(defun bignum-truncate-single-digit-no-quo (x len-x y)
  (declare (type bignum-index len-x))
  (declare (ignore len-x))
  (let (;(q (%allocate-bignum len-x))
	(r-h 0)
        (r-l 0))
    (progn
      (digit-bind (y-h y-l) (%bignum-ref y 0)
        (multiple-value-setq (r-h r-l)(%floor-loop-no-quo x y-h y-l))
        #|
        (do ((i (1- len-x) (1- i)))
	    ((minusp i))
          (declare (fixnum i))
          (digit-bind (q-digit-h q-digit-l r-digit-h r-digit-l)
                      (digit-bind (x-h x-l) (%bignum-ref x i)
                        (%floor r-h r-l x-h x-l y-h y-l))
            (declare (ignore q-digit-h q-digit-l))
            ;(%bignum-set q i q-digit-h q-digit-l)
	    (setf r-h r-digit-h
                  r-l r-digit-l)))
         |#
        ; might as well make a fixnum if possible
        (if (> (%digits-sign-bits r-h r-l)  arch::fixnumshift)
          (%ilogior (%ilsl 16 r-h) r-l)
          (let ((rem (%allocate-bignum 1)))
            (%bignum-set rem 0 r-h r-l)
            rem))))))

;; so big deal - we save a one digit bignum for y 
;; and bigger deal if x is negative - we copy or negate x, computing result destructively
;;  - thus avoiding making a negated x in addition to result
;; 
(defun bignum-truncate-by-fixnum (x y)
  (declare (fixnum y))
  (WHEN (eql Y 0)(ERROR (MAKE-CONDITION 'DIVISION-BY-ZERO :OPERATION 'TRUNCATE :OPERANDS (LIST X Y))))
  (let* ((len-x (%bignum-length x))
         (x-minus (%bignum-minusp x len-x))
         (maybe-q (%allocate-bignum 3))
         (q (if x-minus
              (if (<= len-x 2)
                (dotimes (i 3 (negate-bignum-in-place maybe-q))
                  (if (< i len-x)
                    (multiple-value-bind (hi lo) (%bignum-ref x i)
                      (%bignum-set maybe-q i hi lo))
                    (%bignum-set maybe-q i 65535 65535)))
                (negate-bignum x))
              (if (<= len-x 2) ; this was broken if negative because bignum-replace just copies min len-a len-b digits
                (progn
                  (bignum-replace maybe-q x)                
                  maybe-q)
                (%allocate-bignum len-x))))      ;  q is new big or -x
         ;(len-q (%bignum-length q))
         (y-minus (minusp y))         
         (y (if y-minus (- y) y)))
    (declare (fixnum y))
    (declare (type bignum-index len-x len-q))
    (declare (dynamic-extent maybe-q))
    (let* ((r-h 0)
           (r-l 0)
           (y-h (%ilogand #xffff (%iasr 16 y)))
           (y-l (%ilogand #xffff y)))
      (multiple-value-setq (r-h r-l)(%floor-loop-quo (if x-minus q x) q y-h y-l))      
      (let* ((r (%ilogior (%ilsl 16 r-h) r-l)))
        (declare (fixnum r))
        (when (neq x-minus y-minus)(negate-bignum-in-place q))
        (setq q (%normalize-bignum-macro q ))
        (values (if (eq q maybe-q) (copy-bignum q) q)
                (if x-minus (the fixnum (- r)) r))))))

(defun bignum-truncate-by-fixnum-no-quo (x y)
  (declare (fixnum y))
  (WHEN (eql Y 0)(ERROR (MAKE-CONDITION 'DIVISION-BY-ZERO :OPERATION 'TRUNCATE :OPERANDS (LIST X Y))))
  (let* ((len-x (%bignum-length x))
         (x-minus (%bignum-minusp x len-x))
         (y-minus (minusp y))         
         (y (if y-minus (- y) y)))
    (declare (fixnum y))
    (declare (type bignum-index len-x len-q))
      (let* (;(LEN-Q (%BIGNUM-LENGTH Q))
             (r-h 0)
             (r-l 0)
             (y-h (%ilogand #xffff (%iasr 16 y)))
             (y-l (%ilogand #xffff y)))
        (if x-minus
          (with-bignum-buffers ((q (the fixnum (1+ len-x))))
            (negate-bignum x nil q)
            (multiple-value-setq (r-h r-l)(%floor-loop-no-quo q y-h y-l)))
          (multiple-value-setq (r-h r-l)(%floor-loop-no-quo x y-h y-l)))        
        (let* ((r (%ilogior (%ilsl 16 r-h) r-l)))
          (declare (fixnum r))
          (if x-minus (the fixnum (- r)) r)))))


;;; DO-TRUNCATE -- Internal.
;;;
;;; This divides *truncate-x* by *truncate-y*, and len-x and len-y tell us how
;;; much of the buffers we care about.  TRY-BIGNUM-TRUNCATE-GUESS modifies
;;; *truncate-x* on each interation, and this buffer becomes our remainder.
;;;
;;; *truncate-x* definitely has at least three digits, and it has one more than
;;; *truncate-y*.  This keeps i, i-1, i-2, and low-x-digit happy.  Thanks to
;;; SHIFT-AND-STORE-TRUNCATE-BUFFERS.
;;;


(defun do-truncate (len-x len-y)
  (declare (type bignum-index len-x len-y))
  (let* ((len-q (- len-x len-y))
	 ;; Add one for extra sign digit in case high bit is on.
         (len-res (1+ len-q))
         (maybe-q (%allocate-bignum 2))         
	 (q (if (<= len-res 2) maybe-q (%allocate-bignum len-res)))
	 (k (1- len-q))
	 (i (1- len-x))
	 (low-x-digit (- i len-y)))
    (declare (type bignum-index len-q len-res k i  low-x-digit))
    (declare (dynamic-extent maybe-q))
    (loop
      (digit-bind (h l)
                  (digit-bind (guess-h guess-l)
                              (bignum-truncate-guess-2 *truncate-x* i *truncate-y* (the fixnum (1- len-y)))                                  
                    (try-bignum-truncate-guess guess-h guess-l len-y low-x-digit))
        (%bignum-set q k h l))
      (cond ((zerop k) (return))
            (t (decf k)
               (decf low-x-digit)
               (setq i (1- i)))))
    (if (eq q maybe-q)
      (progn 
        (setq q (%normalize-bignum-macro q))
        (if (fixnump q) q (copy-bignum q)))
      (%normalize-bignum-macro q))))

(defun do-truncate-no-quo (len-x len-y)
  (declare (type bignum-index len-x len-y))
  (let* ((len-q (- len-x len-y))
	 (k (1- len-q))
	 (i (1- len-x))
	 (low-x-digit (- i len-y)))
    (declare (type bignum-index len-q k i  low-x-digit))
    (loop
      (digit-bind (guess-h guess-l) (bignum-truncate-guess-2 *truncate-x* i *truncate-y* (the fixnum (1- len-y)))                                 
        (try-bignum-truncate-guess guess-h guess-l len-y low-x-digit)
        (cond ((zerop k) (return))
              (t (decf k)
                 (decf low-x-digit)
                 (setq i (1- i))))))
    nil))

;;; TRY-BIGNUM-TRUNCATE-GUESS -- Internal.
;;;
;;; This takes a digit guess, multiplies it by *truncate-y* for a result one
;;; greater in length than len-y, and subtracts this result from *truncate-x*.
;;; Low-x-digit is the first digit of x to start the subtraction, and we know x
;;; is long enough to subtract a len-y plus one length bignum from it.  Next we
;;; check the result of the subtraction, and if the high digit in x became
;;; negative, then our guess was one too big.  In this case, return one less
;;; than guess passed in, and add one value of y back into x to account for
;;; subtracting one too many.  Knuth shows that the guess is wrong on the order
;;; of 3/b, where b is the base (2 to the digit-size power) -- pretty rarely.
;;;
#|
(defun try-bignum-truncate-guess (guess-h guess-l len-y low-x-digit)
  (declare (type bignum-index low-x-digit len-y))
  (format t "~& *truncate-x* :") (show-bignum *truncate-x*)
  (format t "~& *truncate-y* :") (show-bignum *truncate-y*)

  (let (;(carry-digit-h 0)
        ;(carry-digit-l 0)
	;(borrow 1)
	(i low-x-digit))
    (declare (type bignum-index i)
	     (fixnum borrow))
    ;; Multiply guess and divisor, subtracting from dividend simultaneously.
    (try-guess-loop-1 guess-h guess-l len-y low-x-digit *truncate-x* *truncate-y*)
    ;; See if guess is off by one, adding one Y back in if necessary.
    (setq i (+ low-x-digit len-y))
  (format t "~& *truncate-x* :") (show-bignum *truncate-x*)

    (cond ((%digit-0-or-plusp (%bignum-ref-hi *truncate-x* i))
	   (values guess-h guess-l))
	  (t
	   ;; If subtraction has negative result, add one divisor value back
	   ;; in.  The guess was one too large in magnitude.
           ;; hmm - happens about 1.6% of the time
           (bignum-add-loop-+ low-x-digit *truncate-x* *truncate-y* len-y)
           (%subtract-one guess-h guess-l)
	   ;(%subtract-with-borrow guess-h guess-l 0 1 1)
           ))))
|#

(defun try-bignum-truncate-guess (guess-h guess-l len-y low-x-digit)
  (declare (type bignum-index low-x-digit len-y))

  (let ((carry-digit-h 0)
        (carry-digit-l 0)
	(borrow 1)
	(i low-x-digit))
    (declare (type bignum-index i)
	     (fixnum borrow carry-digit-h carry-digit-l))
    ;; Multiply guess and divisor, subtracting from dividend simultaneously.
    (dotimes (j len-y)
      (multiple-value-bind (y-h y-l) (%bignum-ref *truncate-y* j)
	(multiple-value-bind (high-h high-l low-h low-l)
	    (%multiply-and-add guess-h
			       guess-l
			       y-h
			       y-l
			       carry-digit-h
			       carry-digit-l)
	  (setq carry-digit-h high-h
		carry-digit-l high-l)
	  (multiple-value-bind (tx-h tx-l) (%bignum-ref *truncate-x* i)
	    (multiple-value-bind (x-h x-l temp-borrow)
		(%subtract-with-borrow tx-h tx-l low-h low-l borrow)
	      (%bignum-set *truncate-x* i x-h x-l)
	      (setq borrow temp-borrow)))))
      (incf i))
    (multiple-value-bind (tx-h tx-l) (%bignum-ref *truncate-x* i)
      (multiple-value-bind (x-h x-l)
	  (%subtract-with-borrow tx-h tx-l carry-digit-h carry-digit-l borrow)
	(%bignum-set *truncate-x* i x-h x-l)))
    ;; See if guess is off by one, adding one Y back in if necessary.


    (cond ((%digit-0-or-plusp (%bignum-ref-hi *truncate-x* i))
	   (values guess-h guess-l))
	  (t
	   ;; If subtraction has negative result, add one divisor value back
	   ;; in.  The guess was one too large in magnitude.
           ;; hmm - happens about 1.6% of the time
           (bignum-add-loop-+ low-x-digit *truncate-x* *truncate-y* len-y)
           (%subtract-one guess-h guess-l)
	   ;(%subtract-with-borrow guess-h guess-l 0 1 1)
           ))))



;;; BIGNUM-TRUNCATE-GUESS -- Internal.
;;;
;;; This returns a guess for the next division step.  Y1 is the highest y
;;; digit, and y2 is the second to highest y digit.  The x... variables are
;;; the three highest x digits for the next division step.
;;;
;;; From Knuth, our guess is either all ones or x-i and x-i-1 divided by y1,
;;; depending on whether x-i and y1 are the same.  We test this guess by
;;; determining whether guess*y2 is greater than the three high digits of x
;;; minus guess*y1 shifted left one digit:
;;;    ------------------------------
;;;   |    x-i    |   x-i-1  | x-i-2 |
;;;    ------------------------------
;;;    ------------------------------
;;; - | g*y1 high | g*y1 low |   0   |
;;;    ------------------------------
;;;                ...                   <   guess*y2     ???
;;; If guess*y2 is greater, then we decrement our guess by one and try again.
;;; This returns a guess that is either correct or one too large.
;;;
;;; the y's come from *truncate-y*, x's from *truncate-x*
;;; doing this in lap is not screamingly difficult - x's at i, i-1, i-2





(defun bignum-truncate-guess-2 (x xidx y yidx)
  (digit-bind (guess-h guess-l)
              (%floor-99 x xidx y yidx)
    (truncate-guess-loop guess-h guess-l x xidx y yidx)))



    

;;; SHIFT-Y-FOR-TRUNCATE -- Internal.
;;;
;;; This returns the amount to shift y to place a one in the second highest
;;; bit.  Y must be positive.  If the last digit of y is zero, then y has a
;;; one in the previous digit's sign bit, so we know it will take one less
;;; than digit-size to get a one where we want.  Otherwise, we count how many
;;; right shifts it takes to get zero; subtracting this value from digit-size
;;; tells us how many high zeros there are which is one more than the shift
;;; amount sought.
;;;
;;; Note: This is exactly the same as one less than the integer-length of the
;;; last digit subtracted from the digit-size.
;;; 
;;; We shift y to make it sufficiently large that doing the 64-bit by 32-bit
;;; %FLOOR calls ensures the quotient and remainder fit in 32-bits.
;;;
(defun shift-y-for-truncate (y)
  (let* ((len (%bignum-length y)))
    (declare (type bignum-index len))
    (multiple-value-bind (last-h last-l) (%bignum-ref y (1- len))
      (1- (%digits-sign-bits last-h last-l)))))

;;; SHIFT-AND-STORE-TRUNCATE-BUFFERS -- Internal.
;;;
;;; Stores two bignums into the truncation bignum buffers, shifting them on the
;;; way in.  This assumes x and y are positive and at least two in length, and
;;; it assumes *truncate-x* and *truncate-y* are one digit longer than x and y.
;;;
(defun shift-and-store-truncate-buffers (x len-x y len-y shift)
  (declare (type bignum-index len-x len-y)
	   (type (integer 0 (#.digit-size)) shift))
  (cond ((eql 0 shift)
	 (bignum-replace *truncate-x* x :end1 len-x)
	 (bignum-replace *truncate-y* y :end1 len-y))
	(t
	 (bignum-ashift-left-unaligned x 0 shift (the fixnum (1+ len-x)) *truncate-x*)
	 (bignum-ashift-left-unaligned y 0 shift (the fixnum (1+ len-y)) *truncate-y*))))




;;;; General utilities.


;;; %NORMALIZE-BIGNUM-BUFFER -- Internal.
;;;
;;; Internal in-place operations use this to fixup remaining digits in the
;;; incoming data, such as in-place shifting.  This is basically the same as
;;; the first form in %NORMALIZE-BIGNUM, but we return the length of the buffer
;;; instead of shrinking the bignum.
;;;



    




;;; %NORMALIZE-BIGNUM -- Internal.
;;;
;;; This drops the last digit if it is unnecessary sign information.  It
;;; repeats this as needed, possibly ending with a fixnum.  If the resulting
;;; length from shrinking is one, see if our one word is a fixnum.  Shift the
;;; possible fixnum bits completely out of the word, and compare this with
;;; shifting the sign bit all the way through.  If the bits are all 1's or 0's
;;; in both words, then there are just sign bits between the fixnum bits and
;;; the sign bit.  If we do have a fixnum, shift it over for the two low-tag
;;; bits.
;;;
#|
(defun %normalize-bignum (result &optional (len (%bignum-length result)))
  (declare (type bignum-type result)
	   (type bignum-index len))
  (let* ((minusp (%bignum-minusp result len))
         (newlen (normalize-bignum-loop (if minusp -1  0) result len)))
    (declare (type bignum-index newlen))    
    (when (= newlen 1)
      (multiple-value-bind (digit-h digit-l)
                           (%bignum-ref result 0)
        (declare (type bignum-half-element-type digit-h digit-l))
        (if (> (the fixnum (%digits-sign-bits digit-h digit-l))  arch::fixnumshift)
          (return-from %normalize-bignum (%ilogior (%ilsl 16 digit-h) digit-l)))))
    (unless (= newlen len)
      (%bignum-set-length result newlen)
      (when minusp
        (do ((i newlen (1+ i)))
            ((>= i len)) ; paranoid
          (declare (fixnum i))
          (%bignum-set result i 0 0))))
    result))
|#

(defun %normalize-bignum (res)
  (declare (ignore len))
  ;(declare (optimize (speed 3)(safety 0)))
  (%normalize-bignum-2 t res))

;;; %MOSTLY-NORMALIZE-BIGNUM -- Internal.
;;;
;;; This drops the last digit if it is unnecessary sign information.  It
;;; repeats this as needed, possibly ending with a fixnum magnitude but never
;;; returning a fixnum.
;;;
#|
(defun %mostly-normalize-bignum (result &optional (len (%bignum-length result)))
  (declare (type bignum-type result)
	   (type bignum-index len))
  (let* ((minusp (%bignum-minusp result len))
         (newlen (normalize-bignum-loop (if minusp -1 0) result len)))
    (declare (type bignum-index newlen))
    (unless (= newlen len)
      (%bignum-set-length result newlen)
      (when minusp
        (do ((i newlen (1+ i)))
            ((>= i len)) ; paranoid
          (declare (fixnum i))
          (%bignum-set result i 0 0))))
    result))
|#

(defun %mostly-normalize-bignum (res &optional len)
  (declare (ignore len))
  (%normalize-bignum-2 nil res))




; think its ok
(defun ldb32 (hi-data lo-data size pos)
  (declare (fixnum hi-data lo-data size pos))
  (let* ((hi-bit (+ pos size))
         (mask (%i- (%ilsl size 1) 1)))
    (declare (fixnum hi-bit mask))    
    (%ilogand mask (if (< hi-bit 16)
                     (%iasr pos lo-data)
                     (if (>= pos 16)
                       (%ilsr (the fixnum (- pos 16)) hi-data)
                       (%ilogior 
                         (%iasr pos lo-data)
                         (%ilsl (the fixnum (- 16 pos)) hi-data)))))))

#|
(defun ldb32 (hi-data lo-data size pos)
  (declare (fixnum hi-data lo-data size pos))
  (let* ((hi-bit (+ pos size))
         (mask (%i- (%ilsl size 1) 1)))
    (declare (fixnum hi-bit mask))    
    (%ilogand mask (if (< hi-bit 16)
                     (ash lo-data (- pos))
                     (if (>= pos 16)
                       (%ilsr (- pos 16) hi-data)
                       (%ilogior 
                         (ash lo-data (- pos))
                         (%ilsl (- 16 pos) hi-data)))))))
|#

#|
(defun %bignum-ref (v i)
  (values (uvref v (+ i i 1))(uvref v (+ i i))))
(defun %bignum-set (v i h lo)
  (setf (uvref v (+ i i 1)) h (uvref v (+ i i) lo)))
(defun bignum-minusp (v)
  (logbitp 15 (uvref v (1- (uvsize v)))))
(defun %bignum-length (v)(* 2 (uvsize v)))
(setq big (make-array 4 :element-type '(unsigned-byte 16)))
|#


; this was wrong for negative bigs when byte includes or exceeds sign
(defun %ldb-fixnum-from-bignum (bignum size position)
  (declare (fixnum size position))
  (let* ((low-idx (ash position -5))
         (low-bit (logand position 31))
         (hi-bit (+ low-bit size))
         (len (%bignum-length bignum))
         (minusp (bignum-minusp bignum)))
    (declare (fixnum size position low-bit hi-bit low-idx len))
    (if (>= low-idx len)
      (if minusp (1- (ash 1 size)) 0)      
      (multiple-value-bind (hi lo)(%bignum-ref bignum low-idx)
        (let ((chunk-lo (ldb32 hi lo (min size (%i- 32 low-bit)) low-bit)))
          (let ((val
                 (if (< hi-bit 32) 
                   chunk-lo
                   (progn
                     (setq low-idx (1+ low-idx))
                     (multiple-value-setq (hi lo)
                       (if (>= low-idx len)
                         (if minusp (values #xffff #xffff)(values 0 0))
                         (%bignum-ref bignum low-idx)))
                     (let ((chunk-hi (ldb32 hi lo (%i- size (%i- 32 low-bit)) 0)))
                       (%ilogior (ash chunk-hi (%i- 32 low-bit)) chunk-lo))))))
            val))))))

(defun load-byte (size position integer)
  (if (and (bignump integer)
           (<= size (- 31 arch::fixnumshift)) #|#.(integer-length most-positive-fixnum))|#
           (fixnump position))
    (%ldb-fixnum-from-bignum integer size position)
    (let ((mask (byte-mask size)))
      (if (and (fixnump mask) (fixnump integer)(fixnump position)) ;(<= position (- 31 arch::fixnumshift)))
        ; %iasr was busted when count > 31 - maybe just shouldn't use it
        (%ilogand mask (%iasr position integer))
        (logand mask (ash integer (- position)))))))    

(defun %bignum-bignum-gcd (u0 v0)
  (let* ((u-len (%bignum-length u0))
	 (u-plusp (%bignum-0-or-plusp u0 u-len))
	 (v-len (%bignum-length v0))
	 (v-plusp (%bignum-0-or-plusp v0 v-len)))
    (declare (fixnum u-len v-len))
    (with-bignum-buffers ((u (if u-plusp u-len (the fixnum (1+ u-len))))
			  (v (if v-plusp v-len (the fixnum (1+ v-len)))))
      (if u-plusp
	(bignum-replace u u0)
	(progn
	  (negate-bignum u0 nil u)
	  (%mostly-normalize-bignum-macro u)
	  (setq u-len (%bignum-length u))))
      (if v-plusp
	(bignum-replace v v0)
	(progn
	  (negate-bignum v0 nil v)
	  (%mostly-normalize-bignum-macro v)
  	  (setq v-len (%bignum-length v))))
      (let* ((u-trailing-zero-bits (%bignum-count-trailing-zero-bits u))
	     (v-trailing-zero-bits (%bignum-count-trailing-zero-bits v)))
	(declare (fixnum u-trailing-zero-bits v-trailing-zero-bits))
	(bignum-shift-right-loop-1
	 (logand u-trailing-zero-bits 31)
	 u
	 u
	 (the fixnum (1- u-len))
	 (ash u-trailing-zero-bits -5))
	(%mostly-normalize-bignum-macro u)
	(setq u-len (%bignum-length u))
	(bignum-shift-right-loop-1
	 (logand v-trailing-zero-bits 31)
	 v
	 v
	 (the fixnum (1- v-len))
	 (ash v-trailing-zero-bits -5))
	(%mostly-normalize-bignum-macro v)
	(setq v-len (%bignum-length v))
	(let* ((shift (min u-trailing-zero-bits
			   v-trailing-zero-bits)))
	  (loop
	      (let* ((fix-u (and (= u-len 1)
				 (< (the fixnum (%bignum-ref-hi u 0))
				    (ash 1 14))
				 (uvref u 0)))
		     (fix-v (and (= v-len 1)
				 (< (the fixnum (%bignum-ref-hi v 0))
				    (ash 1 14))
				 (uvref v 0))))
		(if fix-v
		  (if fix-u
		    (return (ash (%fixnum-gcd fix-u fix-v) shift))
		    (return (ash (bignum-fixnum-gcd u fix-v) shift)))
		  (if fix-u
		    (return (ash (bignum-fixnum-gcd v fix-u) shift)))))
	      
	      (let* ((signum (if (> u-len v-len)
			       1
			       (if (< u-len v-len)
				 -1
				 (bignum-compare u v u-len v-len)))))
		(declare (fixnum signum))
		(case signum
		  (0			; (= u v)
		   (if (zerop shift)
		     (let* ((copy (%allocate-bignum u-len)))
		       (bignum-replace copy u)
		       (return copy))
		     (return (ash u shift))))
		  (1			; (> u v)
		   (bignum-subtract-loop
		    u u-len 0 v v-len 0 u u-len)
		   (%mostly-normalize-bignum-macro u)
		   (setq u-len (%bignum-length u))
		   (setq u-trailing-zero-bits
			 (%bignum-count-trailing-zero-bits u))
		   
		   (bignum-shift-right-loop-1
		    (logand u-trailing-zero-bits 31)
		    u
		    u
		    (the fixnum (1- u-len))
		    (ash u-trailing-zero-bits -5))
		   (%mostly-normalize-bignum-macro u)
		   (setq u-len (%bignum-length u)))
		  (t			; (> v u)
		   (bignum-subtract-loop
		    v v-len 0 u u-len 0 v v-len)
		   (%mostly-normalize-bignum-macro v)
		   (setq v-len (%bignum-length v))
		   (setq v-trailing-zero-bits
			 (%bignum-count-trailing-zero-bits v))
		   (bignum-shift-right-loop-1
		    (logand v-trailing-zero-bits)
		    v
		    v
		    (the fixnum (1- v-len))
		    (ash v-trailing-zero-bits -5))
		   (%mostly-normalize-bignum-macro v)
		   (setq v-len (%bignum-length v)))))))))))
