/*
 * video-device.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/video/video-device.cc,v 1.19 2002/02/03 04:20:17 lim Exp $";
#endif

#if defined(__osf__) || defined(__ultrix__)
/*FIXME they didn't get this one right */
extern "C" {
#include <sys/types.h>
#include <sys/uio.h>
}
#else
#include <sys/types.h>
#ifndef WIN32
#include <sys/uio.h>
#endif
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "video-device.h"
#include "tclcl.h"
#include "crdef.h"

#if defined(sun) && !defined(__svr4__)
extern "C" int gettimeofday(struct timeval*, struct timezone*);
#endif

VideoDevice::VideoDevice(const char* classname, const char* nickname)
	: TclClass(classname), nickname_(nickname)
{
}

void VideoDevice::bind()
{
	TclClass::bind();
	/*
	 * Store the nickname and attribute list as class variables
	 * in the OTcl device class.  This allows us to query the
	 * device name and attributes without first trying to
	 * create (and open) the capture object.
	 */
	Tcl& tcl = Tcl::instance();
	tcl.evalf("%s set nickname_ {%s}", classname_, nickname_);
	tcl.evalf("%s set attributes_ {%s}", classname_, attributes_);
}

VideoCapture::VideoCapture()
	: vstart_(0), vstop_(0),
	  hstart_(0), hstop_(0),
	  framebase_(0), frame_(0),
	  ref_(0),
	  inw_(0), inh_(0), outw_(0), outh_(0), send_all_(1),
	  encoder_(0), lastfc_(0.),
	  running_(0), status_(0), delta_(0.) 
{
	bps(128000);
	fps(1);

	/* CCIR 601 */
	ymin_ = 16;
	ymax_ = 235;
	contrast_ = 1.0;
	for (int i = 0; i < 256; ++i)
		ynorm_[i] = i;
}

VideoCapture::~VideoCapture()
{
	delete[] framebase_;
	delete[] ref_;
}

int VideoCapture::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();

	if (argc == 2) {
		if (strcmp(argv[1], "send_full_intra_frame") == 0) {
			send_all_ = 1;
			return (TCL_OK);
		}
		if (strcmp(argv[1], "status") == 0) {
			sprintf(tcl.buffer(), "%d", status_);
			tcl.result(tcl.buffer());
			return (TCL_OK);
		}
		if (strcmp(argv[1], "is-hardware-encode") == 0) {
			tcl.result("0");
			return (TCL_OK);
		}
		if (strcmp(argv[1], "need-capwin") == 0) {
			tcl.result("0");
			return (TCL_OK);
		}
	}
	if (argc == 3) {
		if (strcmp(argv[1], "encoder") == 0) {
			encoder_ = (EncoderModule*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "send") == 0) {
			if (atoi(argv[2])) {
				if (!running_) {
					start();
					running_ = 1;
				}
			} else {
				if (running_) {
					stop();
					running_ = 0;
				}
			}
			return (TCL_OK);
		}
		if (strcmp(argv[1], "fps") == 0) {
			/*FIXME assume value in range */
			fps(atoi(argv[2]));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "bps") == 0) {
			/*FIXME assume value in range */
			bps(atoi(argv[2]));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "fillrate") == 0) {
			/*FIXME assume value in range */
			fillrate(atoi(argv[2]));
			return (TCL_OK);
		}
		/*FIXME*/
		if (strcmp(argv[1], "decimate") == 0 ||
		    strcmp(argv[1], "port") == 0 ||
		    strcmp(argv[1], "type") == 0)
			/* ignore */
			return (TCL_OK);
	}
	return (SourceModule::command(argc, argv));
}

void VideoCapture::contrast(double c)
{
	/* map min to ccir-601 black (16) & max to ccir-601 white (235) */
	u_int min = ymin_, max = ymax_;
	double dmin = min, dmax = max;
	double dslope = 219. / (dmax - dmin) * c;
	double dy = 16.;
	u_int i;
	for (i = 0; i < min; ++i)
		ynorm_[i] = u_char(dy);
	for ( ; i < max; ++i) {
		ynorm_[i] = u_char(dy);
		if (dy < 235.)
			dy += dslope;
	}
	for ( ; i < 256; ++i)
		ynorm_[i] = u_char(dy);

	contrast_ = c;
}

void VideoCapture::fps(int v)
{
	fps_ = v;
	frametime_ = 1e6 / double(v);
}

/*
 * This routine decreases the latency associated with a change from a very
 * low bandwidth to a very high one.  The latency is caused by the fact that
 * we have to wait for the previos timeout (at the low bandwidth) before the
 * new bandwidth takes effect.  What we do is readjust the frameclock
 * to reflect the new bandwidth and reset the timer.
 */
void VideoCapture::bps(int bps)
{
	if (bps == 0) {
		/*FIXME need to throw this error in a clean way */
		fprintf(stderr, "vic: video capture bps = 0!\n");
		return;
	}

	/*
	 * This check is not strictly needed, but it probably saves us
	 * unnecessary work.
	 */
	if (bps == bps_)
		return;

	if (running_) {
		adjust_frameclock(bps);
	}
	bps_ = bps;
}

void VideoCapture::adjust_frameclock(int bps)
{
	cancel();
	double mbits = (frameclock_ - lastfc_) * bps_;
	frameclock_ = lastfc_ + mbits / double(bps);
	double now = gettimeofday();
	if (frameclock_ < now)
		frameclock_ = now;
	double delta = frameclock_ - now;
	usched(delta);
}

/*
 * Advance the frame clock and return the amount of time we
 * need to wait before sending the next frame.  We compute
 * this time according to the desired bit and frame rates,
 * favoring the more restrictive metric.  If we're more than
 * 200ms behind (e.g., the cpu is saturated or we've been
 * suspended), give up and reset the frame clock.
 */
double VideoCapture::tick()
{
	int n = encoder_->nb();
	grab();

	n = encoder_->nb() - n;
	double frametime = 8e6 * double(n) / double(bps_);

	if (frametime < frametime_) {
		if (frametime * 2. < frametime_)
			delta_ += (frametime - delta_) * .25;
		else
			delta_ = frametime;
		frametime = frametime_;
	} else
		delta_ = frametime;

	lastfc_ = frameclock_;
	frameclock_ += frametime;
	double now = gettimeofday();
	double delta = frameclock_ - now;

	if (delta < -0.2e6) {
		delta = frametime;
		frameclock_ = now;
	} else if (delta < 0)
		/*
		 * We're not too far behind.
		 * Try to catch up.
		 */
		delta = 0.;

	return (delta);
}

void VideoCapture::start()
{
	frameclock_ = gettimeofday();
	timeout();
}

void VideoCapture::stop()
{
	cancel();
}

void VideoCapture::timeout()
{
	double delta = tick();
	if (delta < 0.)
		delta = 0.;
	usched(delta);
}

void VideoCapture::grab()
{
	abort();
}

/* must call after set_size_xxx */
void VideoCapture::allocref()
{
	delete[] ref_;
	ref_ = new u_char[framesize_];
	memset((char*)ref_, 0, framesize_);
}

/*
 * define these for REPLENISH macro used below
 */
#define ABS(v) if (v < 0) v = -v;

#define DIFF4(in, frm, v) \
	v += (in)[0] - (frm)[0]; \
	v += (in)[1] - (frm)[1]; \
	v += (in)[2] - (frm)[2]; \
	v += (in)[3] - (frm)[3];

#define DIFFLINE(in, frm, left, center, right) \
	DIFF4(in, frm, left); \
	DIFF4(in + 1*4, frm + 1*4, center); \
	DIFF4(in + 2*4, frm + 2*4, center); \
	DIFF4(in + 3*4, frm + 3*4, right); \
	ABS(right); \
	ABS(left); \
	ABS(center);

void VideoCapture::suppress(const u_char* devbuf)
{
	if (send_all_) {
	    send_all_ = 0;
	    mark_all_send();
	} else {
	    REPLENISH(devbuf, ref_, outw_, 1, 0, blkw_, 0, blkh_);
	}
}

//
// save
//
//	gets two luma arrays ("lum" and "cache") and copies the first 16x16 
//	macroblock from "lum" to "cache". "stride" is the frame width. 
//
inline void save(const u_char* lum, u_char* cache, int stride)
{
	for (int i = 16; --i >= 0; ) {
		((u_int*)cache)[0] = ((u_int*)lum)[0];
		((u_int*)cache)[1] = ((u_int*)lum)[1];
		((u_int*)cache)[2] = ((u_int*)lum)[2];
		((u_int*)cache)[3] = ((u_int*)lum)[3];
		cache += stride;
		lum += stride;
	}
}

/*
 * Default save routine -- saves all luma macroblocks marked with CR_SEND in 
 * crvec_ (i.e., the new luma blocks) into cache (the ref_ structure).
 */
void VideoCapture::saveblks(u_char* lum)
{
	u_char* crv = crvec_;
	u_char* cache = ref_;
	int stride = outw_;
	stride = (stride << 4) - stride;
	for (int y = 0; y < blkh_; y++) {
		// process the full frame (a blkw_*blkh_ array of 16x16 macroblocks)
		for (int x = 0; x < blkw_; x++) {
			// process a blkw_*16 row of 16x16 macroblocks
			if ((*crv++ & CR_SEND) != 0)
				// save the current lum macroblock into ref_
				save(lum, cache, outw_);
			cache += 16;
			lum += 16;
		}
		lum += stride;
		cache += stride;
	}
}

void VideoCapture::set_size_422(int w, int h)
{
	delete[] framebase_;

	inw_ = w;
	inh_ = h;
	w &=~ 0xf;
	h &=~ 0xf;
	outw_ = w;
	outh_ = h;

	framesize_ = w * h;
	int n = 2 * framesize_ + 2 * VIDCAP_VPAD * w;
	framebase_ = new u_char[n];
	memset(framebase_, 0x80, n);
	frame_ = framebase_ + VIDCAP_VPAD * w;
	crinit(w, h);

	vstart_ = 0;
	vstop_ = blkh_;
	hstart_ = 0;
	hstop_ = blkw_;
}

void VideoCapture::set_size_411(int w, int h)
{
	delete[] framebase_;

	inw_ = w;
	inh_ = h;
	w &=~ 0xf;
	h &=~ 0xf;
	outw_ = w;
	outh_ = h;

	int s = w * h;
	framesize_ = s;
	int n = s + (s >> 1) + 2 * VIDCAP_VPAD * outw_;
	framebase_ = new u_char[n];
	/* initialize to gray */
	memset(framebase_, 0x80, n);
	frame_ = framebase_ + VIDCAP_VPAD * outw_;
	crinit(w, h);

	vstart_ = 0;
	vstop_ = blkh_;
	hstart_ = 0;
	hstop_ = blkw_;
}

void VideoCapture::set_size_cif(int w, int h)
{
	delete[] framebase_;
	inw_ = w;
	inh_ = h;

	int ispal;
	switch (w) {
	case 320:
		/* 1/4 NTSC */
		ispal = 0;
		outw_ = 352;
		outh_ = 288;
		break;

	case 160:
		/* 1/8 NTSC */
		ispal = 0;
		outw_ = 176;
		outh_ = 144;
		break;

	case 352:
		/* CIF */
		ispal = 0;
		outw_ = 352;
		outh_ = 288;
		break;

	case 176:
		/* QCIF */
		ispal = 0;
		outw_ = 176;
		outh_ = 144;
		break;

	case 384:
		/* 1/4 PAL */
		ispal = 1;
		outw_ = 352;
		outh_ = 288;
		break;

	case 192:
		/* 1/16 PAL */
		ispal = 1;
		outw_ = 176;
		outh_ = 144;
		break;

	default:
		/* FIXME this shouldn't happen */
		fprintf(stderr, "vic: CIF video capture: bad geometry %dx%d\n",
					w, h);
		abort();
	}
	int s = outw_ * outh_;
	framesize_ = s;
	int n = s + (s >> 1) + 2 * VIDCAP_VPAD * outw_;
	framebase_ = new u_char[n];
	/* initialize to gray */
	memset(framebase_, 0x80, n);
	frame_ = framebase_ + VIDCAP_VPAD * outw_;
	crinit(outw_, outh_);

	if (ispal) {
		/* PAL: field is bigger than CIF */
		vstart_ = 0;
		vstop_ = blkh_;
		hstart_ = 0;
		hstop_ = blkw_;
	} else {
		/* NTSC: field is smaller than CIF */
		vstart_ = 1;
		vstop_ = vstart_ + inh_ / 16;
		int nw = inw_ / 16;
		int margin = (blkw_ - nw + 1) / 2;
		hstart_ = margin;
		hstop_ = blkw_ - margin;
	}
}

