# announce.tcl --
#
#       Code that handles scheduling of SDP announcement.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


import ERSClient TCP/Client

# Implements the client portion of the Expanding Ring Search
# protocol used by a session directory tool to find the nearest
# session announcement proxy.
Class ERSClient/Sdfor -superclass ERSClient

# Initializes a new ERSClient/Sdfor object.  When a reply is
# received, the <i>found_server</i> method will be invoked on
# <i>$parent</i> with <i>$scope</i> and the (unicast) address
# of the responding server as arguments.
ERSClient/Sdfor public init {parent scope spec} {
	$self next $spec
	$self instvar parent_ scope_
	set parent_ $parent
	set scope_ $scope
}

#
ERSClient/Sdfor private response {data} {
	$self instvar parent_ scope_
	$parent_ found_server $scope_ $data
}

# Arranges to have a session announced via SAP.  If the
# <i>useProxyAnnouncer</i> resource is set, the SDP message
# is delivered to the proxy for announcement.  Otherwise, the
# SDP message is announced locally.
Class ProgramAnnouncer -configuration { sdforPort 12000 }

# Initialize a new ProgramAnnouncer object.  The ProgramSource
# object <i>src</i> is queried for a list of administrative
# scope zones and is used for local announcements.
ProgramAnnouncer public init {src} {
	$self instvar src_ scopes_
	set src_ $src
	set scopes_ [$src scopes]
}

# Finds the administrative scope zone within which the SDPMessage
# <i>msg</i> should be advertised.
ProgramAnnouncer private scopeof {msg} {
	# FIXME should be elsewhere
	$self instvar scopes_

	#FIXME
	if [$msg have_field c] {
		set addr [$msg set caddr_]
	} else {
		set media [lindex [$msg set allmedia_] 0]
		set addr [$media set caddr_]
	}
	set addr [lindex [split $addr /] 0]

	set i 0
	set found 0
	set len [llength $scopes_]
	while {$i < $len} {
		set scope [lindex $scopes_ $i]
		if [$scope contains $addr] {
			set found 1
			break
		}
		incr i
	}
	if {$found == 0} {
		$self fatal "address $addr not in any known scope"
	}

	return $scope
}

# Announce the Program <i>prog</i>.  If the <i>useProxyAnnouncer</i>
# resource is set, an expanding ring search is carried out to locate
# a proxy and the program is delivered to the proxy for
# advertisement.  If the <i>useProxyAnnouncer</i> resource is not
# set, the program is handed to the local ProgramSource for
# advertisement.
ProgramAnnouncer public announce {prog} {
	$self instvar src_

	#puts "ProgramAnnouncer::announce"
	if ![$self get_option useProxyAnnouncer] {
		#puts "ProgramAnnouncer::announce [$src_ info class]"
		$src_ announce $prog
		return
	}

	foreach msg [$prog set msgs_] {
		$self send $msg
	}

}

# Locates an appropriate proxy to advertise the SDPMessage
# <i>msg</i> and sends the message to the proxy for
# advertisement.
ProgramAnnouncer private send {msg} {
	$self instvar servers_ ers_ pending_

	set scope [$self scopeof $msg]
	if ![info exists servers_($scope)] {
		if ![info exists ers_($scope)] {
			set spec [split [$scope sapAddr] /]
			set addr [lindex $spec 0]
			set port [lindex $spec 1]
			set ersspec "$addr/[expr $port + 1]"
			set ers [new ERSClient/Sdfor $self $scope $ersspec]
			set ers_($scope) $ers
			$ers search
		}

		if ![info exists pending_($scope)] {
			set pending_($scope) [list $msg]
		} else {
			lappend pending_($scope) [list $msg]
		}

		return
	}

	set l [split $servers_($scope) /]
	set addr [lindex $l 0]
	set port [lindex $l 1]

	set c [new TCP/Client]
	$c open $addr $port
	$c send "advertise [$msg set msgtext_]\n"
	$c close
	#FIXME get response
}

# Callback from ERSClient/Sdfor object.  Notifies this announcer
# that a proxy for the ScopeZone <i>scope</i> is available at
# the (unicast) address/port <i>data</i>
ProgramAnnouncer public found_server {scope data} {
	$self instvar servers_ pending_ ers_

	#FIXME
	if {$data == ""} {
		$self warn "didn't find server for [$scope range]"
		return
	}

	set servers_($scope) $data
	if [info exists pending_($scope)] {
		foreach msg $pending_($scope) {
			$self send $msg
		}
	}
	unset pending_($scope)

	delete $ers_($scope)
	unset ers_($scope)
}
