# ui-player.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/applications/player/ui-player.tcl,v 1.8 2002/02/12 22:01:43 lim Exp $


import Dialog ScrolledWindow/Expand Observer FileDialog/Archive VcrIcons \
	SessionCatalog

proc address_check { addr } {
	set addr [split $addr "/"]
	if { [llength $addr] != 2 } {
		return 0
	}
	set port [string trim [lindex $addr 1]]
	set addr [string trim [lindex $addr 0]]
	if { $port=={} || $addr=={} } {
		return 0
	}

	set addr [split $addr "."]
	if { [llength $addr]!=4 } {
		return 0
	}

	return 1
}


WidgetClass PlayerUI/Main -superclass Dialog -default {
	{ .modal 0 }
	{ .title "Player" }
	{ .session_list.bbox.width 250 }
	{ .session_list.scrollbar both }
	{ *session_list.scrollbar both }
	{ *session_list.borderWidth 1  }
	{ *session_list.relief sunken  }
	{ *session_list.bbox.highlightThickness 1 }
	{ *session_list.Scrollbar.borderWidth 1 }
	{ *session_list.Scrollbar.highlightThickness 1 }
	{ *session_list.Scrollbar.width 10 }

	{ *ImageTextButton.borderWidth 1 }
	{ *ImageTextButton.highlightThickness 1 }
	{ *font WidgetDefault }

	{ .scale.width 10 }
	{ .scale.borderWidth 1 }
	{ .scale.highlightThickness 1 }

	{ .menu.tearOff 0 }
	{ .menu.borderWidth 1 }
	{ .menu.highlightThickness 1 }
}


PlayerUI/Main instproc build_widget { path } {
	$self set count_ 0
	frame $path.buttons
	frame $path.f1

	ScrolledListbox $path.session_list -itemclass PlayerUI/SessionListItem\
			-browsecmd "$self browse_session_list"
	frame $path.session_frame

	# create the various buttons

	label $path.rightbutton \
			-text "(click right mouse button for more options)"
	ImageTextButton $path.catalog -image Icons(browse) \
			-text "Read catalog" \
			-command "$self catalog"
	ImageTextButton $path.startplay -image VcrIcons(play) \
			-text "Start playback" -state disabled \
			-command "$self start_play"
	ImageTextButton $path.cancel -image Icons(cross) -text Cancel \
			-command "exit" \
			-options { { text.justify left } }

	# create a popup menu
	menu $path.menu
	$path.menu add command -label "New session" \
			-command "$self new_session"
	$path.menu add command -label "New stream" -command "$self new_stream"
	$path.menu add command -label "Delete" -command "$self delete"
	$path.menu add command -label "Start tool" -command "$self start_tool"
	$path.menu add command -label "Advanced options" \
			-command "$self advanced_options"
	bind $path <Button-3> "$self popup_menu %X %Y"

	# pack all widgets

	pack $path.cancel $path.startplay $path.catalog $path.rightbutton \
			-side right -in $path.buttons -fill y -padx 2

	pack $path.session_list -fill both -side left -in $path.f1 \
			-padx 5 -pady 5
	pack $path.session_frame -fill both -expand 1 -side right -in $path.f1\
			-padx 5 -pady 5

	pack $path.buttons -side bottom -fill x -anchor e -pady 2
	pack $path.f1 -side top -fill both -expand 1

	# create a dummy session widget to fill up the space while no session
	# exists
	$self dummy_session

	$self set ttl_ 16
	$self set advanced_ [PlayerUI/Advanced $path.advanced]
}


PlayerUI/Main instproc dummy_session { } {
	$self instvar dummy_session_
	set dummy_session_ [PlayerUI/Session [$self subwidget \
			session_frame].dummy_session]
	set bg [WidgetClass widget_default -disabledforeground]
	$dummy_session_ configure -takefocus 0 -bg $bg
	$dummy_session_ subwidget bbox configure -takefocus 0 -bg $bg
	$dummy_session_ subwidget window configure -takefocus 0 -bg $bg
	$dummy_session_ subwidget hscroll configure -takefocus 0 -bg $bg
	$dummy_session_ subwidget vscroll configure -takefocus 0 -bg $bg

	pack $dummy_session_ -fill both -expand 1
}


PlayerUI/Main instproc start_play { } {
	# check for duplicate addresses

	set list [$self subwidget session_list]
	foreach session_widget [$list info all] {
		set item_widget [$list info widget -id $session_widget]
		set address [string trim [lindex [$item_widget cget -value] 2]]
		if [info exists addrlist($address)] {
			Dialog transient MessageBox -image Icons(warning) \
					-text "Duplicate address \"$address\"\
					found in session list. Please fix it\
					before starting playback"
			return
		}

		set addrlist($address) 1
	}

	# no duplicate addresses found. It's ok to start playback

	$self instvar started_play_
	set started_play_ 1
	$self tkvar input_done_
	set input_done_ yes
}


PlayerUI/Main instproc popup_menu { x y } {
	$self instvar menu_widget_ started_play_
	set menu [$self subwidget menu]
	set menu_widget_ [$self find_widget $x $y]
	if { $menu_widget_ != "" } {
		if { [$menu_widget_ info class]=="PlayerUI/SessionListItem" } {
			set type session
		} else {
			set type stream
		}
	} else {
		set type ""
	}

	if [info exists started_play_] {
		set delete_state disabled
		$menu entryconfigure "Advanced options" -state disabled
	} else {
		set delete_state normal
		$menu entryconfigure "Advanced options" -state normal
	}

	switch -exact -- $type {
		session {
			$menu entryconfigure "Delete*" -label "Delete session"\
					-state $delete_state
			$menu entryconfigure "Start tool" -state normal
		}

		stream {
			$menu entryconfigure "Delete*" -label "Delete stream"\
					-state $delete_state
			$menu entryconfigure "Start tool" -state disabled
		}

		default {
			$menu entryconfigure "Delete*" -label "Delete" \
					-state disabled
			$menu entryconfigure "Start tool" -state disabled
		}
	}

	tk_popup $menu $x $y
}


PlayerUI/Main instproc add_session { protocol media address } {
	$self instvar count_
	set widget [PlayerUI/Session [$self subwidget \
			session_frame].session_$count_]
	$widget subwidget bbox configure
	incr count_

	set list [$self subwidget session_list]
	$list insert end "-id $widget [list $protocol] [list $media] \
			[list $address]"
	if { [llength [$list selection get]]==0 } {
		$list selection set -id $widget
		$self browse_session_list $widget
	}
	return $widget
}


PlayerUI/Main instproc delete_session { widget } {
	destroy $widget
	set list [$self subwidget session_list]
	if { [$list selection get -id $widget]!="" } {
		set selected 1
	} else {
		set selected 0
	}

	$list delete -id $widget
	if { [$list info numelems] <= 0 } {
		$self instvar dummy_session_
		pack $dummy_session_ -fill both -expand 1
		$self subwidget startplay configure -state disabled
	} elseif { $selected } {
		$list selection set 0
		$self browse_session_list [$list selection get]
	}
}


PlayerUI/Main instproc browse_session_list { widget } {
	set list [$self subwidget session_list]
	if { [llength [$list selection get]]==0 } {
		$list selection set -id $widget
	}

	catch { pack forget [pack slaves [$self subwidget session_frame]] }
	pack $widget -fill both -expand 1
	set old_focus [focus]
	if { $old_focus != [[$list info widget -id $widget] subwidget \
			address] && $old_focus!="" } {
		focus [$list subwidget window]
	}
}


PlayerUI/Main instproc new_session { } {
	set new [Dialog transient PlayerUI/NewSession]
	if { $new=="" } return

	set protocol [lindex $new 0]
	set media    [lindex $new 1]
	set address  [lindex $new 2]
	$self add_session $protocol $media $address
}


PlayerUI/Main instproc new_stream { } {
	set dialog [[Application/Player instance] edit_stream_dialog]
	$dialog configure -protocol "" -media ""
	$dialog configure -datafile ""
	$dialog configure -indexfile ""
	set result [$dialog invoke]
	if { $result == "" } return

	$self add_stream [lindex $result 4] [lindex $result 0] \
			[lindex $result 1] [lindex $result 2] \
			[lindex $result 3]
}


PlayerUI/Main instproc add_stream { name datafile indexfile protocol media } {
	set session [$self get_session $protocol $media]
	if { $session!="" } {
		$session new_stream $name $datafile $indexfile
		$self subwidget session_list selection set -id $session
		$self browse_session_list $session
	}
}


PlayerUI/Main instproc start_tool { } {
	$self instvar menu_widget_
	$menu_widget_ start_tool
}


PlayerUI/Main instproc advanced_options { } {
	$self instvar advanced_
	$advanced_ tkvar {ttl_ dlg_ttl}
	$self instvar ttl_

	set dlg_ttl $ttl_
	if { [$advanced_ invoke] != "" } {
		set $ttl_ $dlg_ttl
	}
}


PlayerUI/Main instproc get_session { protocol media } {
	set list ""
	set session_list [$self subwidget session_list]
	foreach session_widget [$session_list info all] {
		set item_widget [$session_list info widget -id $session_widget]
		set value [$item_widget cget -value]
		set p [lindex $value 0]
		set m [lindex $value 1]
		if { $protocol==$p && $media==$m } {
			lappend list "$session_widget $item_widget"
		}
	}

	set len [llength $list]
	switch -exact -- $len {
		0 {
			return [$self add_session $protocol $media \
					224.8.8.1/8000]
		}

		1 {
			return [lindex [lindex $list 0] 0]
		}

		default {
			return [$self select_session $list]
		}
	}
}


PlayerUI/Main instproc select_session { list } {
	set dlg [PlayerUI/SelectSession .select_session]
	set listbox [$dlg subwidget listbox]
	foreach s $list {
		set session_widget [lindex $s 0]
		set item_widget [lindex $s 1]
		$listbox insert end "-id $session_widget \
				[$item_widget cget -value]"
		set new_item [$listbox info widget -id $session_widget]
		$new_item subwidget address configure -state disabled
	}

	$listbox selection set 0
	set result [$dlg invoke]
	destroy $dlg
	return $result
}


PlayerUI/Main instproc delete { } {
	$self instvar menu_widget_
	if { [$menu_widget_ info class] == "PlayerUI/SessionListItem" } {
		$self delete_session [$self subwidget session_list info id \
				-widget $menu_widget_]
	} else {
		[$self subwidget session_list selection get] delete_stream \
				$menu_widget_
	}

#	set rightbutton [$self subwidget rightbutton]
#	grab -global $rightbutton
#	set orig_cursor [$rightbutton cget -cursor]
#
#        global tcl_platform
#        if {$tcl_platform(platform)=="windows"} {
#		$rightbutton configure -cursor PIRATE
#        } else {
#		$rightbutton configure -cursor pirate
#        }
#
#	bind $rightbutton <Button-1> "$self try_to_delete %X %Y; \
#			grab release $rightbutton; \
#			$rightbutton configure -cursor [list $orig_cursor]; \
#			bind $rightbutton <Button-1> {}"
}


PlayerUI/Main instproc try_to_delete { x y } {
	set widget [$self find_widget $x $y]
	if { [$widget info class] == "PlayerUI/SessionListItem" } {
		$self delete_session [$self subwidget session_list info id \
				-widget $widget]
	} else {
		[$self subwidget session_list selection get] delete_stream \
				$widget
	}
}


PlayerUI/Main instproc try_to_delete___________ { x y } {
	set list [$self subwidget session_list]
	foreach session [$list info all] {
		set item_widget [$list info widget -id $session]
		if { [$self is_inside $item_widget $x $y] } {
			$self delete_session $session
			return
		}
	}

	set session [lindex [$list selection get] 0]
	if { $session!="" } {
		foreach stream [$session streams] {
			if { [$self is_inside $stream $x $y] } {
				$session delete_stream $stream
				return
			}
		}
	}
}


PlayerUI/Main instproc find_widget { x y } {
	set list [$self subwidget session_list]
	foreach session [$list info all] {
		set item_widget [$list info widget -id $session]
		if { [$self is_inside $item_widget $x $y] } {
			return $item_widget
		}
	}

	set session [lindex [$list selection get] 0]
	if { $session!="" } {
		foreach stream [$session streams] {
			if { [$self is_inside $stream $x $y] } {
				return $stream
			}
		}
	}

	return ""
}


PlayerUI/Main instproc is_inside { widget x y } {
	set geom [winfo geometry $widget]
	set geom [split $geom "+"]
	set geom_wh [split [lindex $geom 0] "x"]
	set geom_w  [lindex $geom_wh 0]
	set geom_h  [lindex $geom_wh 1]

	set geom_x [expr [winfo rootx $widget] - [winfo vrootx $widget]]
	set geom_y [expr [winfo rooty $widget] - [winfo vrooty $widget]]

	if { $x >= $geom_x && $y >= $geom_y && $x < [expr $geom_x + $geom_w] \
			&& $y < [expr $geom_y + $geom_h] } {
		return 1
	} else {
		return 0
	}
}


PlayerUI/Main instproc playback_ui { } {
	set list [$self subwidget session_list]
	foreach widget [$list info all] {
		set item_widget [$list info widget -id $widget]
		$item_widget no_modify
		$widget no_modify
	}

	foreach widget [pack slaves [$self subwidget buttons]] {
		destroy $widget
	}

	set path [$self info path]
	#bind $path <Button-3> ""
	#destroy [$self subwidget menu]
	set menu [$self subwidget menu]
	$menu entryconfigure "New session" -state disabled
	$menu entryconfigure "New stream"  -state disabled

	scale $path.scale -orient horizontal -showvalue 0
	bind $path.scale <ButtonPress-1> "$self scale_start_move"
	bind $path.scale <ButtonRelease-1> "$self scale_stop_move"

	ImageTextButton $path.playpause -image VcrIcons(pause) -text "Pause" \
			-command "$self pause" \
			-options { { text.width 5 } }

	ImageTextButton $path.exit -image Icons(cross) -text "Exit" \
			-command "exit"

	pack $path.exit $path.playpause -side right -in $path.buttons
	pack $path.scale -side left -fill x -expand 1 -in $path.buttons \
			-padx 15

	#menu $path.menu
	#$path.menu add command -label "Start tool" -command "$self start_tool"
}


PlayerUI/Main instproc pause { } {
	set lts [[Application/Player instance] lts]
	$lts speed 0.0
	$self subwidget playpause configure -image VcrIcons(play) \
			-text "Play" -command "$self play"
	$self tkvar scale_
	if [info exists scale_(after_id)] {
		catch {after cancel $scale_(after_id)}
		unset scale_(after_id)
	}
}


PlayerUI/Main instproc play { } {
	set lts [[Application/Player instance] lts]
	$lts speed 1.0
	$self subwidget playpause configure -image VcrIcons(pause) \
			-text "Pause" -command "$self pause"
	$self tkvar scale_
	set scale_(after_id) [after 1000 "$self update_scale"]
}


PlayerUI/Main instproc scale_start_move { } {
	$self tkvar scale_
	set scale_(start_move) $scale_(current)
	set lts [[Application/Player instance] lts]
	set scale_(orig_speed) [$lts speed]
	$lts speed 0.0
	if { [info exists scale_(after_id)] } {
		catch {after cancel $scale_(after_id)}
		unset scale_(after_id)
	}
}


PlayerUI/Main instproc scale_stop_move { } {
	$self tkvar scale_

	set lts [[Application/Player instance] lts]
	if { $scale_(start_move) != $scale_(current) } {
		$lts now_logical $scale_(current)
	}
	$lts speed $scale_(orig_speed)
	set scale_(after_id) [after 1000 "$self update_scale"]
	unset scale_(start_move) scale_(orig_speed)
}


PlayerUI/Main instproc catalog { } {
	set file_dialog [[Application/Player instance] file_dialog]
	$file_dialog subwidget filebox configure -filetypes {
		{ {Session catalog files} {.ctg} }
		{ {All files}  { * }  }
	}

	set result [$file_dialog invoke]
	if { $result=="" } {
		return
	}

	$self read_catalog $result
}


PlayerUI/Main instproc read_catalog { catalog_filename } {
	set catalog [new SessionCatalog]
	if { [catch {$catalog open $catalog_filename} error] } {
		$self invoke_error "Could not open catalog file:\n$error"
		delete $catalog
		return
	}

	if { [catch {$catalog read} error] } {
		$self invoke_error "Could not read catalog file:\n$error"
		delete $catalog
		return
	}


	# delete all previous sessions
	set list [$self subwidget session_list]
	foreach session [$list info all] {
		$self delete_session $session
	}

	foreach id [$catalog info streams] {
		lappend sessions([$catalog info session $id]) $id
	}

	set file [new ArchiveFile]
	set addr_lobyte 1
	foreach s [array names sessions] {
		set address "224.8.8.$addr_lobyte/8000"
		set session_widget [$self add_session "" "" $address]
		set item_widget [$list info widget -id $session_widget]

		incr addr_lobyte
		set protocol ""
		set media ""
		foreach id $sessions($s) {
			set my_datafile  [$catalog info datafile $id]
			set my_indexfile [$catalog info indexfile $id]

			if [catch {$file open $my_datafile} error] {
				$self invoke_error "Error opening data\
						file:\n$error"
				delete $catalog
				delete $file
				return
			}

			if [catch {$file header data_hdr} error] {
				$self invoke_error "Error reading data\
						header:\n$error"
				delete $catalog
				delete $file
				return
			}

			$file close


			if [catch {$file open $my_indexfile} error] {
				$self invoke_error "Error opening index\
						file:\n$error"
				delete $catalog
				delete $file
				return
			}

			if [catch {$file header index_hdr} error] {
				$self invoke_error "Error reading data\
						header:\n$error"
				delete $catalog
				delete $file
				return
			}

			$file close


			if { $data_hdr(protocol)!=$index_hdr(protocol) } {
				$self invoke_error "Protocol fields do not\
						match in data and index files"
				delete $catalog
				delete $file
				return
			}

			if { $data_hdr(media)!=$index_hdr(media) } {
				$self invoke_error "Media fields do not\
						match in data and index files"
				delete $catalog
				delete $file
				return
			}

			if { $data_hdr(cname)!=$index_hdr(cname) } {
				$self invoke_error "cname fields do not\
						match in data and index files"
				delete $catalog
				delete $file
				return
			}

			if { $data_hdr(name)!=$index_hdr(name) } {
				$self invoke_error "Name fields do not\
						match in data and index files"
				delete $catalog
				delete $file
				return
			}

			$file close

			if { $protocol=="" } {
				set protocol $data_hdr(protocol)
				$item_widget configure -value \
						"[list $protocol] \
						[list $media] \
						[list $address]"
			}
			if { $media=="" } {
				set media $data_hdr(media)
				$item_widget configure -value \
						"[list $protocol] \
						[list $media] \
						[list $address]"
			}

			if { $protocol!=$data_hdr(protocol) } {
				$self invoke_error "Streams within the same\
						session seem to have different\
						protocols"
				delete $catalog
				delete $file
				return
			}

			if { $media!=$data_hdr(media) } {
				$self invoke_error "Streams within the same\
						session seem to have different\
						media"
				delete $catalog
				delete $file
				return
			}

			$session_widget new_stream $data_hdr(name) \
					$my_datafile $my_indexfile
		}
	}

	delete $catalog
	delete $file
}



PlayerUI/Main instproc invoke_error { error } {
	Dialog transient MessageBox -image Icons(warning) -text $error
}


PlayerUI/Main instproc config_scale { start end } {
	$self tkvar scale_
	set scale [$self subwidget scale]
	set scale_(start) $start
	set scale_(end) $end
	set scale_(current) $start
	$scale configure -from [expr int($start)] -to [expr int($end)] \
			-variable [$self tkvarname scale_(current)]
	$self tkvar scale_
	set scale_(after_id) [after 1000 "$self update_scale"]
}


PlayerUI/Main instproc update_scale { } {
	set lts [[Application/Player instance] lts]
	set now [$lts now_logical]
	$self tkvar scale_
	if { $now < $scale_(start) } {
		set now $scale_(start)
	}
	if { $now > $scale_(end) } {
		set now $scale_(end)
	}
	set scale_(current) $now
	$self tkvar scale_
	set scale_(after_id) [after 1000 "$self update_scale"]
}




WidgetClass PlayerUI/SessionListItem -configspec {
	{ -value       value       Value       {}            config_value }
	{ -select      select      Select      0             config_option }
	{ -highlight   highlight   Highlight   0             config_option }

	{ -normalbackground normalBackground NormalBackground \
			WidgetDefault(-background) config_option }
	{ -normalforeground normalForeground NormalForeground \
			WidgetDefault(-foreground) config_option }
	{ -normalrelief normalRelief NormalRelief flat config_option }
	{ -selectbackground selectBackground SelectBackground WidgetDefault \
			config_option }
	{ -selectforeground selectForeground SelectForeground WidgetDefault \
			config_option }
	{ -selectrelief selectRelief SelectRelief sunken config_option }
	{ -highlightrelief highlightRelief HighlightRelief raised \
			config_option }
} -default {
	{ .borderWidth 1 }
	{ *font WidgetDefault }
	{ *LabeledWidget.label.font WidgetDefault(-boldfont) }
	{ .address.borderWidth 1 }
	{ .address.relief sunken }
	{ .address.width 18 }
}


PlayerUI/SessionListItem instproc build_widget { path } {
	$self instvar config_
	set config_(-select) 0
	set config_(-highlight) 0
	set config_(-normalbackground) Black
	set config_(-normalforeground) Black
	set config_(-normalrelief)     flat
	set config_(-selectbackground) Black
	set config_(-selectforeground) Black
	set config_(-selectrelief)     sunken
	set config_(-highlightrelief)  raised

	label $path.protocol -anchor w
	label $path.media -anchor w
	entry $path.address -textvariable [$self tkvarname address_]
	pack $path.protocol $path.media -side left -anchor w
	pack $path.address -fill x -expand 1 -side left -anchor w -padx 5


	# bind the up and down keys to the entry box
	set window [winfo parent $path]
	bind $path.address <KeyPress-Up> "focus $window; \
			event generate $window <KeyPress-Up>"
	bind $path.address <KeyPress-Down> "focus $window; \
			event generate $window <KeyPress-Down>"
}


PlayerUI/SessionListItem instproc config_value { option args } {
	$self tkvar address_
	if { [llength $args]==0 } {
		set protocol [$self subwidget protocol cget -text]
		set protocol [string range $protocol 0 \
				[expr [string length $protocol]-2]]
		set media    [$self subwidget media    cget -text]
		return "[list $protocol] [list $media] [list $address_]"
	} else {
		set value [lindex $args 0]
		$self subwidget protocol configure -text "[lindex $value 0]:"
		$self subwidget media    configure -text [lindex $value 1]
		set address_ [lindex $value 2]
	}
}


PlayerUI/SessionListItem instproc config_option { option args } {
	$self instvar config_
	if { [llength $args]==0 } {
		return $config_($option)
	} else {
		set value [lindex $args 0]
		$self config_[string range $option 1 end] $value
		set config_($option) $value
	}
}


PlayerUI/SessionListItem instproc config_background { value } {
	$self configure -bg $value
	$self subwidget protocol configure -bg $value
	$self subwidget media configure -bg $value
}


PlayerUI/SessionListItem instproc config_foreground { value } {
	$self subwidget protocol configure -fg $value
	$self subwidget media configure -fg $value
}


PlayerUI/SessionListItem instproc config_relief { value } {
	$self configure -relief $value
}


PlayerUI/SessionListItem instproc config_normalbackground { value } {
	if { ![$self set config_(-select)] } {
		$self config_background $value
	}
}


PlayerUI/SessionListItem instproc config_normalforeground { value } {
	if { ![$self set config_(-select)] } {
		$self config_foreground $value
	}
}


PlayerUI/SessionListItem instproc config_normalrelief { value } {
	if { ![$self set config_(-select)] && \
			![$self set config_(-highlight)] } {
		$self config_relief $value
	}
}


PlayerUI/SessionListItem instproc config_selectbackground { value } {
	if { [$self set config_(-select)] } {
		$self config_background $value
	}
}


PlayerUI/SessionListItem instproc config_selectforeground { value } {
	if { [$self set config_(-select)] } {
		$self config_foreground $value
	}
}


PlayerUI/SessionListItem instproc config_selectrelief { value } {
	if { [$self set config_(-select)] && \
		![$self set config_(-highlight)] } {
		$self config_relief $value
	}
}


PlayerUI/SessionListItem instproc config_highlightrelief { value } {
	if { [$self set config_(-highlight)] } {
		$self config_relief $value
	}
}


PlayerUI/SessionListItem instproc config_select { value } {
	$self instvar config_
	if { $value } {
		$self config_background $config_(-selectbackground)
		$self config_foreground $config_(-selectforeground)
		if { !$config_(-highlight) } {
			$self config_relief $config_(-selectrelief)
		}
	} else {
		$self config_background $config_(-normalbackground)
		$self config_foreground $config_(-normalforeground)
		if { !$config_(-highlight) } {
			$self config_relief $config_(-normalrelief)
		}
	}
}


PlayerUI/SessionListItem instproc config_highlight { value } {
	$self instvar config_
	if { $value } {
		$self config_relief $config_(-highlightrelief)
	} else {
		if { $config_(-select) } {
			$self config_relief $config_(-selectrelief)
		} else {
			$self config_relief $config_(-normalrelief)
		}
	}
}


PlayerUI/SessionListItem instproc no_modify { } {
	$self subwidget address configure -state disabled

	#set path [$self info path]
	#set main_ui [[Application/Player instance] main_ui]
	#set menu [$main_ui subwidget menu]
	#bind $path <Button-3> "$main_ui set menu_session_ $self; \
	#		tk_popup $menu %X %Y"
	#bind $path.protocol <Button-3> "$main_ui set menu_session_ $self; \
	#		tk_popup $menu %X %Y"
	#bind $path.media <Button-3> "$main_ui set menu_session_ $self; \
	#		tk_popup $menu %X %Y"
	#bind $path.address <Button-3> "$main_ui set menu_session_ $self; \
	#		tk_popup $menu %X %Y"
}


PlayerUI/SessionListItem instproc start_tool { } {
	set value [$self cget -value]
	set media [lindex $value 1]
	set address [lindex $value 2]

	switch -exact -- $media {
		video {
			set argv "vic $address"
		}

		audio {
			set argv "vat -r $address"
		}

		mediaboard {
			set argv "mb -sa $address"
		}
	}

	if [info exists argv] {
		eval exec $argv &
	}
}


WidgetClass PlayerUI/Session -superclass { ScrolledWindow/Expand Observer } \
		-default {
	{ .scrollbar both }
	{ .bbox.width 250 }
	{ .scrollbar both }
	{ .borderWidth 1  }
	{ .relief sunken  }
	{ .bbox.highlightThickness 1 }
	{ .Scrollbar.borderWidth 1 }
	{ .Scrollbar.highlightThickness 1 }
	{ .Scrollbar.width 10 }
}


PlayerUI/Session instproc build_widget { path } {
	$self next $path
	$self set count_ 0
	$self set streams_ ""
}


PlayerUI/Session instproc new_stream { {name {}} {datafile {}} \
		{indexfile {}} } {
	$self instvar streams_ count_
	set widget [PlayerUI/Stream [$self subwidget window].stream_$count_]
	incr count_
	pack $widget -side top -fill x -expand 1 -anchor w -padx 1 -pady 1
	lappend streams_ $widget
	$widget attach_session [$self info path]

	$widget subwidget name configure -text $name
	$widget subwidget datafile  configure -text $datafile
	$widget subwidget indexfile configure -text $indexfile

	set main_ui [[Application/Player instance] main_ui]
	$main_ui subwidget startplay configure -state normal
}


PlayerUI/Session instproc delete_stream { stream_widget } {
	$self instvar streams_
	set idx [lsearch $streams_ $stream_widget]
	if { $idx==-1 } return
	set streams_ [lreplace $streams_ $idx $idx]
	destroy $stream_widget

	if { [llength $streams_] > 0 } return
	set main_ui [[Application/Player instance] main_ui]
	set no_stream 1
	foreach session [$main_ui subwidget session_list info all] {
		if { [llength [$session streams]] > 0 } {
			set no_stream 0
			break
		}
	}

	if { $no_stream } {
		$main_ui subwidget startplay configure -state disabled
	}
}


PlayerUI/Session instproc streams { } {
	return [$self set streams_]
}


PlayerUI/Session instproc no_modify { } {
	foreach stream [$self set streams_] {
		$stream no_modify
	}
}


PlayerUI/Session instproc attach_session { session } {
	$self instvar streams_
	set lts [[Application/Player instance] lts]
	foreach stream_widget $streams_ {
		set datafile  [$stream_widget subwidget datafile  cget -text]
		set indexfile [$stream_widget subwidget indexfile cget -text]

		set df [new ArchiveFile/Data]
		$df open $datafile
		set if [new ArchiveFile/Index]
		$if open $indexfile

		set stream [$session create_stream]
		$stream datafile  $df
		$stream indexfile $if
		$stream lts $lts
		$session attach_stream $stream
		$stream attach_observer $stream_widget
		$stream_widget attach_stream $stream

		$df header hdr
		[Application/Player instance] clip_time $hdr(start) $hdr(end)
	}
}



WidgetClass PlayerUI/NewSession -superclass Dialog -default {
	{ .transient . }
	{ *ImageTextButton.borderWidth 1 }
	{ *ImageTextButton.highlightThickness 1 }

	{ *Menubutton.borderWidth 1 }
	{ *Menubutton.highlightThickness 1 }
	{ *Entry.borderWidth 1 }
	{ *Entry.highlightThickness 1 }
	{ .protocol.button.anchor w }
	{ .media.borderWidth 1 }
	{ .media.relief raised }
	{ .media.highlightThickness 1 }
	{ .media.entry.borderWidth 0 }
	{ .media.entry.highlightThickness 0 }
	{ .media.entry.relief flat }
	{ .media.button.borderWidth 0 }
	{ .media.button.highlightThickness 0 }

	{ *font WidgetDefault }
	{ *LabeledWidget.label.font WidgetDefault(-boldfont) }
}


PlayerUI/NewSession instproc build_widget { path } {
	$self instvar protocols_

	# instantiate the protocols_ array
	set protocols_(mediaboard) SRM
	set protocols_(video)      RTP
	set protocols_(audio)      RTP

	frame $path.buttons

	# create the media selection widget
	DropDown $path.protocol -variable [$self tkvarname protocol_]
	$path.protocol insert end RTP SRM
	LabeledWidget $path.protocol_label -label Protocol: -underline 0 \
			-orient horizontal -widget $path.protocol

	# create the media selection widget
	DropDown/Text $path.media -variable [$self tkvarname media_]
	eval [list $path.media] insert end [lsort [array names protocols_]]
	LabeledWidget $path.media_label -label Media: -underline 3 \
			-orient horizontal -widget $path.media

	# create the address entry box
	entry $path.address -textvariable [$self tkvarname addr_]
	LabeledWidget $path.address_label -label Address: -underline 4 \
			-orient horizontal -widget $path.address

	ImageTextButton $path.ok -image Icons(check) -text OK \
			-command "$self ok"
	ImageTextButton $path.cancel -image Icons(cross) -text Cancel \
			-command "$self cancel"

	pack $path.ok $path.cancel -side left -in $path.buttons -padx 5
	pack $path.buttons -side bottom -pady 3

	pack $path.protocol_label -fill x -side top -padx 3 -pady 2
	pack $path.media_label    -fill x -side top -padx 3
	pack $path.address_label  -fill x -side top -padx 3 -pady 2

	bind $path <KeyPress-Return> "$path.ok invoke_with_ui"
	bind $path <KeyPress-Escape> "$path.cancel invoke_with_ui"

	wm resizable $path 1 0

	$self tkvar addr_ media_
	trace variable media_ w "$self media_changed_"
	set addr_ "224.2.55.66/8001"
	set media_ "mediaboard"
}


PlayerUI/NewSession instproc ok { } {
	$self tkvar protocol_ media_ addr_
	if { ![address_check $addr_] } {
		Dialog transient MessageBox -image Icons(warning) \
				-text "Invalid session address"
		return
	}

	$self configure -result "[list $protocol_] [list $media_]\
			[list $addr_]"
}


PlayerUI/NewSession instproc media_changed_ { args } {
	$self tkvar media_ protocol_
	$self instvar protocols_
	set media [string tolower $media_]
	if { [info exists protocols_($media)] } {
		set protocol_ $protocols_($media)
	}
}



WidgetClass PlayerUI/Stream -superclass Observer -configspec {
	{ -minimized minimized Minimized 0 config_minimized }
} -default {
	{ .borderWidth 1 }
	{ .relief groove }
	{ *Button.padY 0 }
	{ *Button.borderWidth 0 }
	{ *Button.highlightThickness 1 }
	{ *Button.activeBackground WidgetDefault(-background) }
	{ *Label.padY 0 }
	{ *Label.borderWidth 0 }
	{ *font WidgetDefault(-boldfont) }
	{ *datafile.font WidgetDefault }
	{ *indexfile.font WidgetDefault }
}


PlayerUI/Stream instproc build_widget { path } {
	frame $path.name_frame
	frame $path.data_frame
	frame $path.index_frame

	button $path.minimize -command "\
			if \{ \[$self cget -minimized\] \} \{ \
			$self configure -minimized 0 \} else \{ \
			$self configure -minimized 1 \}"
	button $path.browse -image VcrIcons(browse_small) \
			-command "$self browse_file"
	label $path.name -anchor w
	pack $path.browse -side left -fill y \
			-in $path.name_frame

	label $path.data_label -text "   Data file:"
	label $path.datafile -anchor w

	label $path.index_label -text "   Index file:"
	label $path.indexfile -anchor w

	pack $path.minimize $path.browse -side left -fill y \
			-in $path.name_frame
	pack $path.name -side left -fill both -expand 1 -in $path.name_frame

	pack $path.data_label  -side left -fill y -in $path.data_frame
	pack $path.datafile -side left -fill both -expand 1 \
			-in $path.data_frame

	pack $path.index_label  -side left -fill y -in $path.index_frame
	pack $path.indexfile -side left -fill both -expand 1 \
			-in $path.index_frame

	pack $path.name_frame  -fill x
	pack $path.data_frame  -fill x -expand 1
	pack $path.index_frame -fill x -expand 1
}


PlayerUI/Stream instproc config_minimized { option args } {
	set path [$self info path]
	if { [llength $args]==0 } {
		if { [lsearch [pack slaves $path] $path.data_frame] != -1 } {
			return 0
		} else {
			return 1
		}
	} else {
		set value [lindex $args 0]
		if { $value } {
			catch {
				pack forget $path.data_frame
				pack forget $path.index_frame
			}
			$path.minimize configure -image Icons(maximize)
		} else {
			if { [lsearch [pack slaves $path] \
					$path.data_frame] == -1 } {
				pack $path.data_frame -fill x -expand 1
			}
			if { [lsearch [pack slaves $path] \
					$path.index_frame] == -1 } {
				pack $path.index_frame -fill x -expand 1
			}
			$path.minimize configure -image Icons(minimize)
		}
	}
}


PlayerUI/Stream instproc attach_session { session_widget } {
	$self set session_widget_ $session_widget
}


PlayerUI/Stream instproc browse_file { } {
	$self instvar session_widget_
	set dialog [[Application/Player instance] edit_stream_dialog]

	set item_widget [[[Application/Player instance] main_ui] subwidget \
			session_list info widget -id $session_widget_]
	set value [$item_widget cget -value]
	$dialog configure -protocol [lindex $value 0] -media [lindex $value 1]
	$dialog configure -datafile  [$self subwidget datafile  cget -text]
	$dialog configure -indexfile [$self subwidget indexfile cget -text]
	set result [$dialog invoke]
	if { $result != "" } {
		$self subwidget datafile  configure -text [lindex $result 0]
		$self subwidget indexfile configure -text [lindex $result 1]
		$self subwidget name configure -text [lindex $result 4]
	}
}


PlayerUI/Stream instproc no_modify { } {
	set path [$self info path]
	destroy $path.browse
	label $path.blink -image VcrIcons(redbullet)
	pack $path.blink -side left -in $path.name_frame -before $path.name \
			-padx 3
}


PlayerUI/Stream instproc attach_stream { stream } {
}


PlayerUI/Stream instproc bytes_sent { bytes } {
	$self instvar after_id_
	if { [info exists after_id_] } {
		after cancel $after_id_
	} else {
		$self subwidget blink configure -image VcrIcons(greenbullet)
	}
	set after_id_ [after 500 "$self bytes_sent_done"]
}


PlayerUI/Stream instproc bytes_sent_done { } {
	$self instvar after_id_
	$self subwidget blink configure -image VcrIcons(redbullet)
	unset after_id_
}




WidgetClass PlayerUI/EditStream -superclass FileDialog/Archive -default {
	{ *Radiobutton.borderWidth 1 }
	{ *Radiobutton.highlightThickness 1 }
	{ *clear_datafile.borderWidth 1 }
	{ *clear_datafile.highlightThickness 1 }
	{ *clear_indexfile.borderWidth 1 }
	{ *clear_indexfile.highlightThickness 1 }
	{ *Radiobutton.font WidgetDefault(-boldfont) }
	{ *LabeledWidget.label.font WidgetDefault(-boldfont) }
} -configspec {
	{ -type type Type {} ignore }
	{ -datafile dataFile DataFile {} config_file cget_file }
	{ -indexfile indexFile IndexFile {} config_file cget_file }
	{ -protocol protocol Protocol {} config_option }
	{ -media media Media {} config_option }
}


PlayerUI/EditStream instproc build_widget { path } {
	$self set archive_file_ [new ArchiveFile]
	frame $path.data_index
	frame $path.frame1 -bd 1 -relief sunken
	frame $path.frame2 -bd 1 -relief sunken
	frame $path.dframe
	frame $path.iframe

	label $path.protocol -anchor w
	LabeledWidget $path.protocol_label -label "Protocol:" \
			-widget $path.protocol
	label $path.media -anchor w
	LabeledWidget $path.media_label -label "    Media:" -widget $path.media
	label $path.name -anchor w
	LabeledWidget $path.name_label -label "    Name:" -widget $path.name

	radiobutton $path.datafile -text "Data file: " \
			-value datafile -anchor w \
			-variable [$self tkvarname data_index_]
	radiobutton $path.indexfile -text "Index file: " \
			-value indexfile -anchor w \
			-variable [$self tkvarname data_index_]
	button $path.clear_datafile  -text "Clear" -padx 1 -pady 0 \
			-command "$self configure -datafile {}"
	button $path.clear_indexfile -text "Clear" -padx 1 -pady 0 \
			-command "$self configure -indexfile {}"

	# do the actual file dialog here, so the widgets are created in the
	# right stacking order
	$self next $path

	#pack $path.datafile $path.indexfile -side top -fill x -anchor w \
	#		-in $path.frame1
	pack $path.clear_datafile -side right -fill y -in $path.dframe
	pack $path.datafile -side left -fill both -expand 1 -in $path.dframe
	pack $path.clear_indexfile -side right -fill y -in $path.iframe
	pack $path.indexfile -side right -fill both -expand 1 -in $path.iframe

	pack $path.protocol_label $path.media_label -side left -anchor w \
			-in $path.frame2
	pack $path.name_label -side left -fill x -anchor w -in $path.frame2

	pack $path.dframe $path.iframe -fill both -expand 1 -in $path.frame1
	pack $path.frame1 $path.frame2 -fill both -expand 1 -side top \
			-in $path.data_index -padx 3 -pady 2
	pack $path.data_index -fill x -side top -before $path.frame

	$self tkvar data_index_
	trace variable data_index_ w "$self data_index_; $self ignore_args"
	$self subwidget ok configure -text "OK" \
			-command "$self set done_ ok; $self configure \
			-result {}"
	set data_index_ datafile
}


PlayerUI/EditStream instproc destroy { } {
	$self instvar archive_file_
	delete $archive_file_
	$self next
}


PlayerUI/EditStream instproc ignore { args } {
}


PlayerUI/EditStream instproc config_file { option value } {
	set widget [$self subwidget [string range [string trim $option] \
			1 end]]
	set text [$widget cget -text]
	set index [string first ":" $text]
	incr index
	set text "[string range $text 0 $index]$value"
	$widget configure -text $text
	if { $option=="-datafile" } {
		$self read_header_ $value
	}
}


PlayerUI/EditStream instproc cget_file { option } {
    set widget [$self subwidget [string range [string trim $option] 1 end]]
    set text [$widget cget -text]
    set index [string first ":" $text]
    incr index 2
    return [string range $text $index end]
}


PlayerUI/EditStream instproc config_option { option args } {
	$self instvar config_
	if { [llength $args]==0 } {
		return $config_($option)
	} else {
		set config_($option) [string trim [lindex $args 0]]
	}
}


PlayerUI/EditStream instproc data_index_ { } {
	$self tkvar data_index_
	switch -exact -- $data_index_ {
		datafile {
			set filebox [$self subwidget filebox]
			$filebox subwidget filename_label configure \
					-text "Data file:" -under -1
			$filebox configure -filetypes {
				{ { Data files } { .dat } }
				{ { All files }  { * } }
			}

			set datafile [string trim [$self cget -datafile]]
			if { $datafile != "" } {
				set dir  [file dirname $datafile]
				set file [file tail $datafile]
				$filebox configure -directory $dir
				$filebox configure -filename $file
			}
		}

		indexfile {
			set filebox [$self subwidget filebox]
			$filebox subwidget filename_label configure \
					-text "Index file:" -under -1
			$filebox configure -filetypes {
				{ { Index files } { .idx } }
				{ { All files }  { * } }
			}

			set indexfile [string trim [$self cget -indexfile]]
			if { $indexfile != "" } {
				set dir  [file dirname $indexfile]
				set file [file tail $indexfile]
				$filebox configure -directory $dir
				$filebox configure -filename $file
			}
		}
	}
}


PlayerUI/EditStream instproc invoke { {start_focus {}} } {
	set path [$self info path]
	$self center
	$self grab $start_focus

	$self instvar done_
	$self tkvar data_index_

	while { 1 } {
		set done_ ""
		set result [$self wait]
		if { $result=="" } {
			if { $done_!="" && ![$self check_files_] } {
				continue
			} else {
				break
			}
		}

		$self configure -$data_index_ $result
	}

	$self release
	wm withdraw $path
	$self invoke_closecmd

	if { $done_=="" } {
		return ""
	} else {
		set datafile  [string trim [$self cget -datafile]]
		set indexfile [string trim [$self cget -indexfile]]
		set protocol  [string trim [$path.protocol cget -text]]
		set media     [string trim [$path.media cget -text]]
		set name      [string trim [$path.name cget -text]]
		return "[list $datafile] [list $indexfile] [list $protocol]\
				[list $media] [list $name]"
	}
}


PlayerUI/EditStream instproc read_header_ { datafile } {
	$self instvar archive_file_
	if { ![catch {
		$archive_file_ open $datafile
		$archive_file_ header hdr
	} ] } {
		if { [string match "MDAT*" $hdr(version)] } {
			$self subwidget protocol configure -text $hdr(protocol)
			$self subwidget media configure -text $hdr(media)
			$self subwidget name configure -text $hdr(name)
		}
	} else {
		$self subwidget protocol configure -text ""
		$self subwidget media configure -text ""
		$self subwidget name configure -text ""
	}

	catch {$archive_file_ close}
}


PlayerUI/EditStream instproc check_files_ { } {
	set datafile  [string trim [$self cget -datafile]]
	set indexfile [string trim [$self cget -indexfile]]

	if { $datafile=="" || $indexfile=="" } {
		Dialog transient MessageBox -image Icons(warning) \
				-text "Must specify both data and index files"
		return 0
	}

	if { $datafile!="" && ![$self check_file_ DAT "a data file" \
			$datafile  datahdr] } { return 0 }
	if { $indexfile!="" && ![$self check_file_ IDX "an index file" \
			$indexfile indexhdr] } { return 0 }

	catch {
		if { $datahdr(protocol) != $indexhdr(protocol) || \
				$datahdr(media) != $indexhdr(media) || \
				$datahdr(cname) != $indexhdr(cname) || \
				$datahdr(name)  != $indexhdr(name) } {
			set result [Dialog transient MessageBox \
					-image Icons(warning)\
					-text "The data and index files you\
					have selected seem to differ in one or\
					more of their protocol, media, cname\
					and name fields.\n\
					\nDo you wish to continue?" \
					-type yesno -defaultfocus no]
			if { $result=="no" } { return 0 }
		}
	}
	return 1
}


PlayerUI/EditStream instproc check_file_ { type error_string name hdr_var } {
	upvar $hdr_var hdr
	$self instvar archive_file_
	if { [catch {
		$archive_file_ open $name
		$archive_file_ header hdr
	} error] } {
		catch {$archive_file_ close}
		set result [Dialog transient MessageBox -image Icons(warning)\
				-text "Error reading file: $name\n$error\n\
				\nDo you wish to continue?" \
				-type yesno -defaultfocus no]
		if { $result=="no" } { return 0 } else { return 1 }
	}

	catch {$archive_file_ close}
	if { ![string match "M${type}*" $hdr(version)] } {
		set result [Dialog transient MessageBox -image Icons(warning)\
				-text "File \"$name\" may not be\
				$error_string\n\nDo you wish to continue?" \
				-type yesno -defaultfocus no]
		if { $result=="no" } { return 0 } else { return 1 }
	}

	set protocol [$self cget -protocol]
	set media [$self cget -media]
	if { $protocol!="" && $protocol!=$hdr(protocol) } {
		set result [Dialog transient MessageBox -image Icons(warning)\
				-text "File \"$name\" uses incorrect protocol\
				\"$hdr(protocol)\". Must be \"$protocol\".\n\
				\nDo you wish to continue?" \
				-type yesno -defaultfocus no]
		if { $result=="no" } { return 0 } else { return 1 }
	}
	if { $media!="" && $media!=$hdr(media) } {
		set result [Dialog transient MessageBox -image Icons(warning)\
				-text "File \"$name\" has incorrect media type\
				\"$hdr(media)\". Must be \"$media\".\n\
				\nDo you wish to continue?" \
				-type yesno -defaultfocus no]
		if { $result=="no" } { return 0 } else { return 1 }
	}

	return 1
}



WidgetClass PlayerUI/SelectSession -superclass Dialog -default {
	{ .listbox.bbox.width  250 }
	{ .listbox.bbox.height 70  }
	{ .listbox.scrollbar both }
	{ .listbox.scrollbar both }
	{ .listbox.borderWidth 1  }
	{ .listbox.relief sunken  }
	{ .listbox.bbox.highlightThickness 1 }
	{ .listbox.Scrollbar.borderWidth 1 }
	{ .listbox.Scrollbar.highlightThickness 1 }
	{ .listbox.Scrollbar.width 10 }

	{ *ImageTextButton.borderWidth 1 }
	{ *ImageTextButton.highlightThickness 1 }
	{ *font WidgetDefault }

	{ .transient . }
}


PlayerUI/SelectSession instproc build_widget { path } {
	label $path.label -text "The stream you've chose can be inserted into\
			\nmultiple sessions. Choose one of the\
			following:" -anchor w -justify left
	ScrolledListbox $path.listbox -itemclass PlayerUI/SessionListItem \
			-browsecmd "$self list_browse" \
			-command "$self list_invoke; $self ignore_args"

	frame $path.buttons
	ImageTextButton $path.select -image Icons(check) -text "Select" \
			-command "$self list_invoke"
	ImageTextButton $path.cancel -image Icons(cross) -text "Cancel" \
			-command "$self cancel"

	pack $path.label -fill x -side top -anchor w
	pack $path.select $path.cancel -anchor e -side left -in $path.buttons \
			-padx 3 -pady 2
	pack $path.buttons -anchor e -side bottom
	pack $path.listbox -fill both -expand 1 -side top -padx 3
}


PlayerUI/SelectSession instproc list_browse { widget } {
	# this function prevents you from toggling a selection!

	set listbox [$self subwidget listbox]
	if { [llength [$listbox selection get]]==0 } {
		$listbox selection set -id $widget
	}
}


PlayerUI/SelectSession instproc list_invoke { } {
	set session [$self subwidget listbox selection get]
	$self configure -result $session
}



WidgetClass PlayerUI/Advanced -superclass Dialog -default {
	{ .title "Advanced Options" }
	{ .Entry.width 5 }
	{ .Entry.borderWidth 1 }
	{ .Entry.highlightThickness 1 }
	{ .ImageTextButton.borderWidth 1 }
	{ .ImageTextButton.highlightThickness 1 }
}


PlayerUI/Advanced instproc build_widget { path } {
	entry $path.ttl -textvariable [$self tkvarname ttl_]
	LabeledWidget $path.ttl_label -label "Time-to-live" -widget $path.ttl

	frame $path.buttons
	ImageTextButton $path.ok -text "OK" -image Icons(check) \
			-command "$self configure -result ok"
	ImageTextButton $path.cancel -text "Cancel" -image Icons(cross) \
			-command "$self cancel"

	pack $path.ok $path.cancel -side left -anchor e -in $path.buttons \
			-padx 3 -pady 2
	pack $path.buttons -side bottom -anchor e -padx 2
	pack $path.ttl_label -fill both -expand 1 -padx 5 -pady 2
}

