# http_agent.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/applications/pathfinder/http_agent.tcl,v 1.8 2002/02/03 04:22:06 lim Exp $


import Class

#
# HTTP_Agent is an abstract base class from which MASH_Server agents
# can be created. The basic requirement is that agents implement the
# handle_request method. Other helpful suggestions include setting the
# agent_ instvar to a MASH agent that does the actual MASH-related
# work. This method takes url and key as parameters and returns page
# information in the form of a list { data status_id }. If the agent
# does not understand the request, it returns nothing. This class also
# contains some of the common functions for returning and HTMLifying
# agent response pages.
#
Class HTTP_Agent -configuration {
	server_port 4444
}

HTTP_Agent public init { } {

    $self instvar valid_clients_

    # Obtain a list of valid clients.
    #set o [$self options]
    #$o load_preferences "mserver"
    set client_str [$self get_option valid_clients]
    set valid_clients_ [split $client_str]
}


HTTP_Agent public handle_request { url key source } {
    mtrace trcNet "Error: HTTP_Agent is an abstract base class."
    error "HTTP_Agent is an abstract base class."
    exit
}


#
# The get_page method creates the dynamic HTML page.  The page_type
# parameter corresponds to the page as specified in DynamicHTMLifier;
# e.g. html_(page_type)
#
HTTP_Agent private get_page { page_type } {

    mtrace trcNet "-> Creating $page_type page"
    set page [$self create_dynamic_html \
	    [DynamicHTMLifier set html_($page_type)]]

    return $page
}


#
# The htmlify_messages method uses the instvar list_ and the page_type
# parameter to make a HTMLified page of list_ elements. Note that to
# use this method, the agent_ instvar must be initialized to some
# MASH worker; furthermore, this agent must include a method return_progs
# which returns a list to be HTMLified.
#
HTTP_Agent private htmlify_messages { page_type } {

    $self instvar list_

    $self update_agent_list
    array set agent_array $list_

    set html {}
    foreach key [$self sort [array names agent_array]] {
	append html [$agent_array($key) create_dynamic_html \
		[DynamicHTMLifier set html_($page_type)]]
    }

    return $html
}


#
# The update_agent_list method should be called each time the
# instvar list_ is to be used. Note that to use this method, the
# agent_ instvar must be initialized to some MASH worker;
# furthermore, this agent must include a method return_progs
# which returns a list that can be converted into an associated
# array.
#
HTTP_Agent private update_agent_list { } {
    $self instvar agent_ list_
    set list_ [$agent_ return_progs]
}


HTTP_Agent private sort { list } {
    return [lsort -command "$self sort_compare_" $list]
}


HTTP_Agent private sort_compare_ { key1 key2 } {

    $self instvar list_

    array set agent_array $list_
    return [string compare \
	    [string tolower [$agent_array($key1) set session_name_]] \
	    [string tolower [$agent_array($key2) set session_name_]]]
}


#
# The desc_page method takes a key and returns the HTMLified
# page describing the SDP Message.
#
HTTP_Agent private get_desc_page { key type } {

    $self instvar list_

    $self update_agent_list
    array set agent_array $list_

    append html_page $type _desc

    # Use the key to find the program which will then create the
    # HTML to be returned.
    mtrace trcNet "-> Creating description page"
    set desc_page [$agent_array($key) create_dynamic_html \
	    [DynamicHTMLifier set html_($html_page)]]
    return $desc_page
}


# The get_error_page method returns a HTML page with the specified
# error message "msg."
HTTP_Agent private get_error_page { msg } {

    set page "<html><body bgcolor=#FFFFFF> $msg </body></html>"
    return $page
}


#
# The validate_source method checks that the person who requested the
# action has access to the that action.
#
HTTP_Agent private validate_source { source } {

    $self instvar valid_clients_

    if { [llength $valid_clients_] == 0 } {
	return 1
    }

    set source [string tolower $source]
    mtrace trcNet "-> Validating source $source"

    foreach client $valid_clients_ {
	mtrace trcNet "-> Checking client: $client"
	if [string match $client $source] {
	    return 1
	}
    }

    return 0
}


HTTP_Agent private create_collaborator_mashlet { argv } {
	set mashlet_dir [$self get_option mashlet_dir]
	if { $mashlet_dir == "" } {
		# create a default url (relative to the host on which
		# mash_server is being run) for use by the
		# TCLCL_IMPORT_DIRS environment variable
		set server_port [$self get_option server_port]
		global mash
		set mashlet_dir "http://[localaddr]:$server_port/$mash(version)"
	}
	set server_name [$self get_option play_server_addr]
	regsub -all ":" $server_name "/" megactrl
	set megafor  [$self get_option megafor]
	set page "
		label .label -text {Please wait while mashlets are imported}
		pack .label -fill both -expand 1
		update
		global env
		set env(TCLCL_IMPORT_DIRS) $mashlet_dir
		set x 0; import WidgetClass
		WidgetClass transparent_gif
		set x 0; import CollaboratorApp
		destroy .label
		CollaboratorApp create_pathfinder_app [list $argv] [list $megactrl] [list $megafor]
	"

	return $page
}
