/*
 * misc.cc --
 *
 *      This file defines a number of TclClasses which can be accessed by name
 *      from Tcl's global namespace. Specifically, this allows Tcl code to
 *      execute C code through the command function without needing to
 *      instantiate a TclClass.
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/misc/misc.cc,v 1.25 2002/02/07 03:17:43 wesley Exp $";
#endif

#include "config.h"
#include <tclcl.h>
#include "random.h"
#include "inet.h"
#include "errno.h"
#ifndef WIN32
#include <sys/time.h>
#endif

#include <errno.h>

#ifndef WIN32
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#else
#include <io.h>
#include <process.h>
#endif

class TimeOfDayCommand : public TclCommand {
public:
	TimeOfDayCommand() : TclCommand("gettimeofday") { }
        virtual int command(int argc, const char*const* argv);
};

int TimeOfDayCommand::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	timeval tv;
	gettimeofday(&tv, 0);
	if (argc == 1) {
		tcl.resultf("%.17g", tv.tv_sec + 1e-6 * tv.tv_usec);
		return (TCL_OK);
	} else if (argc == 2) {
		if (strcmp(argv[1], "ascii") == 0) {
			char* cp = tcl.buffer();
			tcl.result(cp);
			time_t now = tv.tv_sec;
			strcpy(cp, ctime(&now));
			cp = strrchr(cp, '\n');
			if (cp != 0)
				*cp = 0;
			return (TCL_OK);
		}
	}
	return (TCL_ERROR);
}

class RandomCommand : public TclCommand {
public:
	RandomCommand() : TclCommand("random") { }
        virtual int command(int argc, const char*const* argv);
};

/*
 * ns-random
 * ns-random $seed
 */
int RandomCommand::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 1) {
		sprintf(tcl.buffer(), "%u", Random::random());
		tcl.result(tcl.buffer());
	} else if (argc == 2) {
		int seed = atoi(argv[1]);
		if (seed == 0)
			seed = Random::seed_heuristically();
		else
			Random::seed(seed);
		tcl.resultf("%d", seed);
	}
	return (TCL_OK);
}

extern "C" const char *intoa(u_int32_t addr);
extern "C" u_int32_t LookupHostAddr(const char *s);

class GetHostByNameCommand : public TclCommand {
public:
	GetHostByNameCommand() : TclCommand("gethostbyname") { }
        virtual int command(int argc, const char*const* argv);
};

int GetHostByNameCommand::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 2) {
		u_int32_t a = LookupHostAddr(argv[1]);
		if (a != 0)
			tcl.resultf("%s", intoa(a));
		return (TCL_OK);
	}
	return (TCL_ERROR);
}

class LocalAddrCommand : public TclCommand {
public:
	LocalAddrCommand() : TclCommand("localaddr") { }
        virtual int command(int argc, const char*const* argv);
};

int LocalAddrCommand::command(int /* argc */, const char*const* /* argv */)
{
	Tcl& tcl = Tcl::instance();
	tcl.resultf("%s", intoa(LookupLocalAddr()));
	return (TCL_OK);
}

class AllocSrcidCommand : public TclCommand {
public:
	AllocSrcidCommand() : TclCommand("alloc_srcid") { }
        virtual int command(int argc, const char*const* argv);
};

int AllocSrcidCommand::command(int /* argc */, const char*const* /* argv */)
{
	Tcl& tcl = Tcl::instance();
	u_int32_t addr = LookupLocalAddr();
	timeval tv;
	::gettimeofday(&tv, 0);
	u_int32_t srcid = u_int32_t(tv.tv_sec + tv.tv_usec);
	srcid += (u_int32_t)getuid();
	srcid += (u_int32_t)getpid();
	srcid += addr;

	tcl.resultf("%u", srcid);
	return (TCL_OK);
}

class NtoACommand : public TclCommand {
public:
	NtoACommand() : TclCommand("intoa") { }
        virtual int command(int argc, const char*const* argv);
};

int NtoACommand::command(int /* argc */, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	tcl.resultf("%s", intoa(htonl(strtoul(argv[1], 0, 10))));
	return (TCL_OK);
}

class LookupHostAddrCommand : public TclCommand {
public:
	LookupHostAddrCommand() : TclCommand("lookup_host_addr") { }
        virtual int command(int argc, const char*const* argv);
};

int LookupHostAddrCommand::command(int /* argc */, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	tcl.resultf("%u", ntohl(LookupHostAddr(argv[1])));
	return (TCL_OK);
}

class LookupHostNameCommand : public TclCommand {
public:
	LookupHostNameCommand() : TclCommand("lookup_host_name") { }
        virtual int command(int argc, const char*const* argv);
};

int LookupHostNameCommand::command(int /* argc */, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	tcl.resultf("%s", LookupHostName(LookupHostAddr(argv[1])));
	return (TCL_OK);
}

class ForkCommand : public TclCommand {
public:
	ForkCommand() : TclCommand("fork") { }
        virtual int command(int argc, const char*const* argv);
};

int ForkCommand::command(int, const char*const*)
{
#ifdef WIN32
	Tcl::instance().add_error("fork not supported in win32 yet");
	return TCL_ERROR;
#else /* ! WIN32 */
	int pid = fork();
	if (pid < 0)
		perror("fork");
	else if (pid > 0)
		/* parent */
		exit(0);

	return (TCL_OK);
#endif /* ! WIN32 */
}


#if TCL_MAJOR_VERSION >= 8

class RedirectExecCommand : public TclCommand {
public:
	RedirectExecCommand() : TclCommand("redirect_exec") { }
        virtual int command(int argc, const char*const* argv);

private:
	struct bg_state {
		bg_state(int pid, int read_fd, const char *cmd)
			: pid_(pid), read_fd_(read_fd), cmd_(NULL)
		{
			cmd_ = strdup(cmd);
		}
		~bg_state() {
			if (cmd_) free(cmd_);
		}

		int pid_;
		int read_fd_;
		char *cmd_;
	};

	int handle_no_bg(int read_fd, const char *cmd);
	int handle_bg(bg_state *state);
	static int  read_pipe(int read_fd, const char *cmd);
	static int  redirect(const char *cmd, char *buf, int len);
	static void data_handler(ClientData state, int mask);

	static void barf(const char *msg) {
		Tcl &tcl= Tcl::instance();
		char *errorInfo = Tcl_GetVar(tcl.interp(), "errorInfo",
					     TCL_GLOBAL_ONLY);
		fprintf(stderr, "Unrecoverable error in redirect_exec: %s\n"
			"%s\n%s\n(child process may not have been cleaned "
			"up)\n", msg, tcl.result(), (errorInfo?errorInfo:""));
		exit(-1);
	}
};

int RedirectExecCommand::command(int argc, const char*const* argv)
{
	Tcl &tcl = Tcl::instance();
#ifdef WIN32
	tcl.add_error("redirect_exec not supported in win32 yet");
	return TCL_ERROR;
#else /* ! WIN32 */
	int is_bg=0;
	const char *cmd;
	if (argc >= 2) cmd = argv[1];
	argc -= 2;
	argv += 2;
	if (argc > 0 && strcmp(argv[argc-1], "&")==0) {
		is_bg = 1;
		argc--;
	}
	if (argc <= 0) {
		tcl.result("incorrect arguments");
		return TCL_ERROR;
	}

	int fd[2], read_fd, write_fd;
	if (pipe(fd)) {
		tcl.result("could not create communication pipe");
		return TCL_ERROR;
	}
	read_fd  = fd[0];
	write_fd = fd[1];

	int pid = fork();
	if (pid < 0)
		perror("fork");
	else if (pid > 0) {
		/* parent */

		close(write_fd);
		if (!is_bg) {
			int retval, status;
			retval = handle_no_bg(read_fd, cmd);
			close(read_fd);
			waitpid(pid, &status, 0);
			if (retval==TCL_OK) {
				tcl.resultf("%d", status);
			}
			return retval;
		} else {
			bg_state *state = new bg_state(pid, read_fd, cmd);
			return handle_bg(state);
		}

	} else {
		/* child */
		close(read_fd);
		dup2(write_fd, 1);  // stdout
		dup2(write_fd, 2);  // stderr

		const char **new_argv;
		new_argv = new const char *[argc+1];
		for (int i=0; i<argc; i++) new_argv[i] = argv[i];
		new_argv[argc] = NULL;
		execvp(new_argv[0], (char* const*)new_argv);
		exit(-1);
	}

	return (TCL_OK);
#endif /* ! WIN32 */
}


int
RedirectExecCommand::handle_no_bg(int read_fd, const char *cmd)
{
	/* wait for the child to finish */
	fd_set fds;
	int ret, retval=TCL_OK;

	while (1) {
		FD_ZERO(&fds);
		FD_SET(read_fd, &fds);
		ret =select(read_fd+1, &fds, NULL, NULL, NULL);
		if (ret==-1) {
			if (errno==EINTR) continue;
			break;
		}
		if (FD_ISSET(read_fd, &fds)) {
			retval = read_pipe(read_fd, cmd);
			if (retval==TCL_ERROR) {
				break;
			}
			if (retval==TCL_BREAK) {
				retval = redirect(cmd, NULL, 0);
				break;
			}
		}
	}
	return retval;
}


int
RedirectExecCommand::handle_bg(bg_state *state)
{
#ifndef WIN32
	Tcl_CreateFileHandler(state->read_fd_, TCL_READABLE, data_handler,
			      (ClientData) state);
	Tcl::instance().resultf("%d", state->pid_);
#endif
	return TCL_OK;
}


int
RedirectExecCommand::read_pipe(int read_fd, const char *cmd)
{
	char buf[1024];
	int len;
	len = read(read_fd, buf, 1024);
	if (len <= 0) {
		return TCL_BREAK;
	}

	return redirect(cmd, buf, len);
}


int
RedirectExecCommand::redirect(const char *cmd, char *buf, int len)
{
	Tcl_Obj *cmdObj, *bufObj;
	Tcl &tcl = Tcl::instance();
	cmdObj = Tcl_NewStringObj((char*)cmd, -1);
	bufObj = Tcl_NewStringObj(buf, len);
	if(Tcl_ListObjAppendElement(tcl.interp(), cmdObj,
				    bufObj)==TCL_ERROR) {
		Tcl_DecrRefCount(cmdObj);
		Tcl_DecrRefCount(bufObj);
		barf("could not construct redirect command");
		return TCL_ERROR;
	}
	if (tcl.evalObj(cmdObj)==TCL_ERROR) {
		barf("evalObj failed");
		return TCL_ERROR;
	}

	return TCL_OK;
}


void
RedirectExecCommand::data_handler(ClientData data, int /*mask*/)
{
#ifndef WIN32
	bg_state *state = (bg_state*) data;
	int retval = read_pipe(state->read_fd_, state->cmd_);
	if (retval==TCL_OK) return;

	Tcl_DeleteFileHandler(state->read_fd_);
	close(state->read_fd_);
	int status;
	waitpid(state->pid_, &status, 0);
	if (retval==TCL_ERROR) {
		Tcl &tcl=Tcl::instance();
		tcl.evalf("bgerror %s", tcl.result());
	} else {
		const char *cmd = state->cmd_;
		redirect(cmd, NULL, 0);
	}
	delete state;
#endif
}

#endif /* TCL_MAJOR_VERSION >= 8 */


#ifdef HAVE_DMALLOC
#include "dmalloc.h"
#include "debug_val.h"

class DmallocCommand : public TclCommand {
public:
	DmallocCommand() : TclCommand("dmalloc_verify") { }
        virtual int command(int argc, const char*const* argv);
};

int DmallocCommand::command(int argc, const char*const* argv)
{
	dmalloc_verify(0);
	return (TCL_OK);
}
#endif


void init_misc()
{
	(void)new TimeOfDayCommand;
	(void)new RandomCommand;
	(void)new GetHostByNameCommand;
	(void)new LocalAddrCommand;
	(void)new NtoACommand;
	(void)new LookupHostAddrCommand;
	(void)new AllocSrcidCommand;
	(void)new LookupHostNameCommand;
	(void)new ForkCommand;
#if TCL_MAJOR_VERSION >= 8
	(void)new RedirectExecCommand;
#endif
#ifdef HAVE_DMALLOC
	(void)new DmallocCommand;
#endif
}
