/*
 * mb-canv.cc --
 *
 *      MediaBoard Canvas
 *      This is now a wrapper around tkCanvas, ultimately, we want to implement
 *      our own canvas.
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-canv.cc,v 1.29 2002/02/03 03:16:30 lim Exp $
 */

#ifndef MB_CANV_CC
#define MB_CANV_CC

#include "mb-canv.h"
#include "mb-string.h"
#include "mb-page.h"
#include "mb-cmd.h"
#include <tclcl.h>

const char* cszLocalTag="-tag local";
const char* cszTextAnchor="-anchor nw";
const char* cszImageAnchor="-anchor nw";
const char* cszLineJoin="-joinstyle round";
const char* cszNull="";
const char* cszLocalTextTag = "-tag {currtext local} -anchor nw";
const char* cszLocalImageTag = "-tag local -anchor nw";

static class MBCanvClass : public TclClass {
public:
	MBCanvClass() : TclClass("MBCanvas") {}
	TclObject* create(int argc, const char*const* /*argv*/) {
		if (argc==4) {
			return (new MBCanvas());
		} else {
			Tcl::instance().add_error("Invalid args: should be"
					 "MBCanvas <pathname>");
			return (TclObject*)NULL;
		}
	}
} mbcanvclass;

MBCanvas::MBCanvas()
	: MBBaseCanvas(), matrix_(), xOrigin_(0),
	  yOrigin_(0), norm_width_(0.), norm_height_(0.)
{
	// make sure tkfont is size of a points
	// otherwise we would use garbage as part of the keys
	assert(sizeof(Tk_Font) == sizeof(char*));
	Tcl_InitHashTable(&htFonts_, TCL_ONE_WORD_KEYS);
}

MBCanvas::~MBCanvas()
{
	Tcl_DecrRefCount(pathObj_);

	Tcl& tcl = Tcl::instance();
	Tcl_HashSearch hsearch;
	Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(&htFonts_, &hsearch);
	MBCanvas::FontStruct *pFS;
	while (pEntry) {
		pFS = (MBCanvas::FontStruct*)Tcl_GetHashValue(pEntry);
		tcl.evalf("font delete %s", pFS->szNamedFont_);
		delete pFS;
		pEntry=Tcl_NextHashEntry(&hsearch);
	}
	Tcl_DeleteHashTable(&htFonts_);
}

//
// Create a canvas item
//
CanvItemId MBCanvas::createItem(Page* pPage, MBCmd* /*pCmd*/, PageItem* pItem)
{
	MB_DefTcl(tcl);
	const char *szExtra=cszNull;        // defaults to nothing
	PageItemType type=pItem->getType();
	if (pPage->isLocal()) {
		if (type==PgItemText) szExtra = cszLocalTextTag;
		else if (type==PgItemImage) szExtra = cszLocalImageTag;
		else szExtra = cszLocalTag;
	} else {
		if (type == PgItemText)  szExtra = cszTextAnchor;
		if (type == PgItemImage) szExtra = cszImageAnchor;
		if (type == PgItemMLine) szExtra = cszLineJoin;
	}

	const int cNumFixed=6;
	const int cNumEnd=2;
	int argc=cNumFixed+(pItem->getNumPoints()*2);
	char** argv = new char*[argc];
	int index = 0;
	argv[index++]="eval";
	argv[index++]=(char*)pathName();
	argv[index++]="create";
	argv[index++]=(char*)caPgItemTyp2CanvItemTyp[type];
	assert(index==cNumFixed-cNumEnd && "adjust cNumFixed!");
	Point p;
	for (int ptIdx=0; ptIdx<pItem->getNumPoints(); ptIdx++) {
		argv[index]= new char[TCL_DOUBLE_SPACE];
		const Point &point=pItem->getPointByIdx(ptIdx);
		p = point;
		matrix_.map(p.x, p.y);
		Coord2Str(p.x, argv[index++]);
		argv[index]= new char[TCL_DOUBLE_SPACE];
		Coord2Str(p.y, argv[index++]);
	}
	argv[index++]=(char*)szExtra;
	char *pszDel=argv[index++]=pItem->getPropsAsStr(this);
	assert(index==argc && "may have corrupted memory");
	char *szCmd = Tcl_Merge(argc, argv);
	MTrace(trcMB|trcVerbose, ("%s",szCmd));
	tcl.eval(szCmd);

	for ( index=cNumFixed-cNumEnd;
	      index<cNumFixed-cNumEnd+(pItem->getNumPoints()*2); ) {
		delete[] argv[index++];
	}
	Tcl_Free(pszDel);
	delete[] argv;
	Tcl_Free(szCmd);
	CanvItemId cid = getResultAsLong();

	// REVIEW: for now create all items with lowest in display order
	// since we are locating the item to raise after in FindMostRecent
	// we should change FindMostRecent to find the item to raise
	// before instead of after...
	tcl.evalf("%s lower %lu all", pathName(), cid);
	return cid;
}

CanvItemId MBCanvas::createImage(Page* pPage, MBCmd* pCmd,
				 ImageItem* pItem,
				 const char* szFileName)
{
	MB_DefTcl(tcl);
	const char* szCmdPrefix = "image create photo -file {";

	MTrace(trcMB|trcVerbose,("creating image from file: %s", szFileName));
	if (!pItem->getImageName()) {
		char* szCmd = new char[strlen(szCmdPrefix) +
				      strlen(szFileName) + 2];
		strcpy(szCmd, szCmdPrefix);
		strcat(szCmd, szFileName);
		strcat(szCmd, "}"); // make sure filename is quoted
		tcl.eval(szCmd);
		const char *szImageName = tcl.result();
		if (!szImageName || szImageName[0]==cchNull) {
			SignalError(("create image from file failed: %s",
				     szFileName));
			return 0;
		}
		pItem->setImageName(szImageName);
		delete[] szCmd;
	}
	return (createItem(pPage, pCmd, pItem));
}

//
// Change Coord of a canvas item
//  pCmd is a pointer to a new command to be associated with the item,
//    if pCmd is NULL, it is ignored.
//
Bool MBCanvas::itemCoord(CanvItemId targetId, MBCmd* /*pCmd*/,
			 Coord x1, Coord y1,
			 Coord x2, Coord y2)
{
	Tcl& tcl = Tcl::instance();
	matrix_.map(x1, y1);
	matrix_.map(x2, y2);
	tcl.evalf("%s coords %lu %g %g %g %g",
		  pathName(), targetId, x1, y1, x2, y2);

	return TRUE;
}

//
// Move a canvas item
//
Bool MBCanvas::moveItem(CanvItemId targetId, MBCmd* /*pCmd*/,
			Coord dx, Coord dy)
{
	matrix_.map(dx, dy);
	Tcl::instance().evalf("%s move %lu %g %g",
			      pathName(), targetId, dx, dy);

	return TRUE;
}

//
// Get a record of the item
//
MBCanvItem* MBCanvas::getItem(const CanvItemId canvId)
{
	Tcl& tcl=Tcl::instance();
	char szCId[MAX_CANVID_CHAR];
	sprintf(szCId, "%lu", canvId);
	tcl.evalf("%s type %s", pathName(), szCId);
	char *szType = NULL;
	::AllocNCopy(&szType, tcl.result());
	tcl.evalf("%s coords %s", pathName(),  szCId);
	Point *arPoints = NULL;
	int count;
	if (!Str2Points(tcl.result(), arPoints, count))
		assert(FALSE && "error Canvas coords return invalid points");

	for (Point *pPoint = arPoints; pPoint < arPoints + count; pPoint++)
		matrix_.imap(pPoint->x, pPoint->y);

	MBCanvItem *pCItem = new MBCanvItem(szType, arPoints, count);
	if (!pCItem) {
		SignalError(("Out of Memory"));
		return NULL;
	}
	char *szCmdPrefix = Concat3(pathName()," itemcget ", szCId);
	char *szProp=PageItem::getProp(szType, szCmdPrefix);
	if (szProp==NULL)
		assert(FALSE);
	pCItem->setProp(szProp);
	Tcl_Free(szProp);
	delete[] szCmdPrefix;
	delete[] szType;
	return pCItem;
}

//
// Save the item
//
Bool MBCanvas::deleteItem(const CanvItemId canvId, MBCmd * /*pCmd*/)
{
	Tcl& tcl=Tcl::instance();
	MTrace(trcMB|trcVerbose,( "d %lu", pathName(), canvId));
	tcl.evalf("%s delete %lu", pathName(), canvId);
	return TRUE;
}

//
// useful for undeleting a *previously deleted* item
// returns the new canvId for the undeleted item if successful,
// zero otherwise
//

CanvItemId MBCanvas::createItem(Page* pPage, MBCmd* /*pCmd*/,
				MBCanvItem* pCanvItem)
{
	Tcl& tcl= Tcl::instance();
	const char* szTag = pPage->isLocal() ? cszLocalTag : cszNull;
	const char* szExtra = cszNull;
	const char* szType = pCanvItem->getType();
	if (!strcmp(szType, caPageItemTypeNames[PgItemText])) {
		szExtra = cszTextAnchor;
	} else if (!strcmp(szType, caPageItemTypeNames[PgItemImage])) {
		szExtra = cszImageAnchor;
	} else if (!strcmp(szType, caPageItemTypeNames[PgItemMLine])) {
		szExtra = cszLineJoin;
	}
	int count = pCanvItem->getCount();
	Point *pPoints = new Point[count];
	memcpy(pPoints, pCanvItem->getCoords(), count*sizeof(Point));
	for (Point* pPt = pPoints; pPt < pPoints + count; pPt++) {
		matrix_.map(pPt->x, pPt->y);
	}
	char *szPoints = Points2Str(pPoints, count);
	const char* argv[] = {
		pathName(),
		"create",
		pCanvItem->getType(),
		szPoints,
		szTag,
		szExtra,
		pCanvItem->getProp()
	};

	char *szCmd = Tcl_Concat(sizeof(argv)/sizeof(argv[0]), (char**)argv);
	tcl.eval(szCmd);
	Tcl_Free(szCmd);
	CanvItemId cid = getResultAsLong();

	delete[] szPoints;
	tcl.evalf("%s lower %lu all", pathName(), cid);
	return cid;
}

// determins if the item overlaps the region
Bool MBCanvas::overlap(Coord x1, Coord y1, Coord x2, Coord y2, CanvItemId cid)
{
	MB_DefTcl(tcl);
	matrix_.map(x1, y1);
	matrix_.map(x2, y2);
	tcl.evalf("%s overlap %ld %g %g %g %g", name(), cid, x1, y1,
		  x2, y2);
	return ((Bool)atoi(tcl.result()));
}

//
// Returns the item nearest item to (x,y) among the list
//
CanvItemId MBCanvas::nextNearest(Coord x, Coord y, Coord dist,
                                 CanvItemId cid)
{
	MB_DefTcl(tcl);
	const int nArgs = 8;
	Tcl_Obj* objv[nArgs];
	int i=0;
	objv[i++] = Tcl_NewStringObj("eval", -1);
	objv[i++] = pathObj();
	objv[i++] = Tcl_NewStringObj("find", -1);
	objv[i++] = Tcl_NewStringObj("closest", -1);

	matrix_.map(x, y);

	objv[i++] = Tcl_NewDoubleObj(x);
	objv[i++] = Tcl_NewDoubleObj(y);
	objv[i++] = Tcl_NewDoubleObj(dist);
	if (cid) {
		/*FIXME: is long save? */
		objv[i++] = Tcl_NewLongObj(cid);
	}
	assert(i<=nArgs);
	tcl.evalObjs(i, objv);

	char *pEnd=NULL;
	char *szCanvId=tcl.result();
	CanvItemId canvId = strtoul(szCanvId, &pEnd, 10);
	if (pEnd==szCanvId)
		canvId = 0; // invalid result item
	MTrace(trcMB|trcExcessive, ("Nearest returns: %ld", canvId));
	return canvId;
}

// raises afId to just before b4Id
void MBCanvas::raiseAfter(CanvItemId b4Id, CanvItemId afId)
{
	MTrace(trcMB|trcVerbose, ("raise %lu %lu",afId, b4Id));
	if (b4Id > 0) {
		Tcl::instance().evalf("%s raise %lu %lu", pathName(),
				      afId, b4Id);
	}
}

void MBCanvas::raise(CanvItemId cid)
{
	MTrace(trcMB|trcVerbose, ("raise %lu ",cid));
	Tcl::instance().evalf("%s raise %lu", pathName(), cid);
}

/* to allow easy expansion of returned value, specify a non-zero len, it
 * indicates the minimum space required for the returned result
 *  side effect:  len will be set the amount of space allocated for result
 *  note: caller is responsible for calling delete to free result */
char *MBCanvas::getText(CanvItemId cid, int& len)
{
	Tcl& tcl = Tcl::instance();
	tcl.evalf("%s itemcget %lu -text", pathName(), cid);
	char* result = tcl.result();
	int slen = strlen(result) + 1;
	if (slen > len) len = slen;
	char *szNew = new char[len];
	strcpy(szNew, result);
	return szNew;
}

//
// Get Text from a text item in the canvas
//
// Note that we need to do merge because we have to preserve the text field
//   without Tcl_Merge, backslashes and other things will be screwed up.
//
void MBCanvas::setText(CanvItemId cid, MBCmd* /*pCmd*/, const char* szText)
{
	char *szCid = new char[MAX_CANVID_CHAR];
	sprintf(szCid, "%lu", cid);
	const char* argv[] = {
		"itemconfigure",
		szCid,
		"-text",
		szText,
	};
	char *szEnd = Tcl_Merge(sizeof(argv)/sizeof(argv[0]), (char**)argv);
	char* szCmd = Concat3(pathName(), " ", szEnd);
	Tcl::instance().eval(szCmd);

	Tcl_Free(szEnd);
	delete[] szCmd;
	delete[] szCid;
}

int MBCanvas::setsize(int width, int height)
{
	// sets the initial width and height that corresponds to (1.0, 1.0)
	norm_width_ = width;
	norm_height_ = height;
	return TCL_OK;
}

/* <otcl> MBCanvas private resize { width height }
 * readjust the scale so that the current view fits the new size
 *
 */
int MBCanvas::resize(int width, int height)
{
	/* w = window dimension
	 * s = scale
	 * n = normalized dimension
	 *
	 *   s*n = w is true always
	 *
	 * w->w1, want n constant
	 * so s1=s*(w1/w) , w = s*n => s1 = s*w1/ (s*n) = w1 / n
	 */

	// try to preserve the aspect ratio, therefore use the smaller of the
	// scales
	double scale = double(width)/norm_width_;
	double scaley = double(height)/norm_height_;
	if (scaley < scale) {
		scale = scaley;
	}
	MTrace(trcMB|trcVerbose,
	       ("resize: norm_w=%d, norm_h=%d, w=%d, h=%d, scale=%g",
		norm_width_, norm_height_, width, height, scale));
	return rescale(scale, 1);
}

/* change the scale */
int MBCanvas::rescale(double scale, int sizeChanged)
{
	/* w = window dimension
	 * s = scale
	 * n = normalized dimension
	 *
	 * s*n = w is true always
	 * if (w is constant) i.e. sizedChanged = 1
	 *     n1 = n*(s/s1), where s1 = scale
	 */
	double oldscaleX, oldscaleY;
	/* we have no offset */
	matrix_.map(1.0, 1.0, oldscaleX, oldscaleY);
	Tcl& tcl = Tcl::instance();
	assert(oldscaleX == oldscaleY ||
	       !"should not happen: diff x and y scales");
	double newscale = scale/oldscaleX; /* = s1 / s */
	Tcl_Obj *objv[7];
	objv[0] = pathObj_;
	objv[1] = Tcl_NewStringObj("scale", -1);
	objv[2] = Tcl_NewStringObj("all", -1);
	objv[3] = Tcl_NewDoubleObj(0.0);
	objv[4] = Tcl_NewDoubleObj(0.0);
	objv[5] = Tcl_NewDoubleObj(newscale);
	objv[6] = Tcl_NewDoubleObj(newscale);
	int retcode = tcl.evalObjs(7, objv);
	if (!sizeChanged) {
		norm_width_ /= newscale;
		norm_height_ /= newscale;
	}
	matrix_.clear();
	matrix_.scale(scale, scale);
	rescaleAllFonts();
	resetOrigin();
	return retcode;
}

/* <otcl> MBCanvas public scale
 * returns the current scale
 */
int MBCanvas::getScale() {
	Tcl& tcl = Tcl::instance();
	double scaleX, scaleY;
	matrix_.map(1.0, 1.0, scaleX, scaleY);
	/* right now we use one scale for both, so ignore scaleY */
	Tcl_Obj* pScaleObj = Tcl_NewDoubleObj(scaleX);
	tcl.result(pScaleObj);
	return TCL_OK;
}

/* <otcl> MBCanvas private fit { width height }
 * changes the scale so that the window of $widthx$height in the
 * current scale fits into the current window.
 *
 */
int MBCanvas::fit(double width, double height)
{
	/* w = window dimension
	 * s = scale
	 * n = normalized dimension
	 *
	 * s*n = w is true always
	 * (w is constant here, change s to s1 to get target n1)
	 *     s1 = (s*n)/n1, where the given width (or height) = n1*s
	 *     => s1 = w/((n1*s)/s)
	 */

	/* if width or height is zero, use the other, if both
	 * non-zero, use the larger */

	double dummy=1.0, scale, scale1;
	int vwidth, vheight, inset;
	visibleSize(vwidth, vheight, inset);

	if (width == 0.0 && height == 0.0) {
		Tcl::instance().add_error("cannot fit with zero width"
			"and zero height!");
		return TCL_ERROR;
	}

	if (width == 0.0) {
		matrix_.imap(dummy, height);
		scale = vheight / height;
	} else if (height == 0.0) {
		matrix_.imap(width, dummy);
		scale = vwidth / width;
	} else {
		matrix_.imap(width, height);
		scale = vwidth / width;
		scale1 = vheight / height;
		if (scale1 < scale) scale = scale1;
	}
	return rescale(scale, 0);
}

/* redirects a command to the underlying tkCanvas */
int MBCanvas::redirectCommand(int argc, const char*const* argv)
{
	const char** newArgv = new const char*[argc];
	memcpy(newArgv, argv, argc*(sizeof(char*)));
	newArgv[0] = pathName();
	/* REVIEW: should use EvalObj! */
	char *cmd = Tcl_Merge(argc, CONST_CAST(char**)(newArgv));
	Tcl_Interp* interp =  Tcl::instance().interp();
	int retcode = Tcl_Eval(interp, cmd);
	Tcl_Free(cmd);
	return retcode;
}

/* resets the origin to that of the tkCanvas */
int MBCanvas::resetOrigin()
{
	Tcl& tcl = Tcl::instance();
	tcl.evalf("%s canvasx 0", pathName());
	double offsetX, offsetY;
	if (!str2double(tcl.result(), offsetX)) {
		return TCL_ERROR;
	}
	tcl.evalf("%s canvasy 0", pathName());
	if (!str2double(tcl.result(), offsetY)) {
		return TCL_ERROR;
	}
	int xChanged=0, yChanged=0;
	if (xOrigin_ != offsetX) {
		xOrigin_ = offsetX;
		xChanged = 1;
	}
	if (yOrigin_ != offsetY) {
		yOrigin_ = offsetY;
		yChanged = 1;
	}
	if (yChanged || xChanged) {
		int width, height, inset;
		visibleSize(width, height, inset);
		Tcl_Obj* objv[3];
		objv[0] = Tcl_NewStringObj(CONST_CAST(char*)(name()), -1);
		objv[1] = Tcl_NewStringObj("expandScrReg", -1);
		Tcl_Obj* pScrRegObjs[4];
		pScrRegObjs[0] = xChanged ?
			Tcl_NewIntObj((int)xOrigin_ + inset) : Tcl_NewObj();
		pScrRegObjs[1] = yChanged ?
			Tcl_NewIntObj((int)yOrigin_ + inset) : Tcl_NewObj();
		pScrRegObjs[2] = xChanged ?
			Tcl_NewIntObj((int)xOrigin_ + width) : Tcl_NewObj();
		pScrRegObjs[3] = yChanged ?
			Tcl_NewIntObj((int)yOrigin_ + height) : Tcl_NewObj();
		objv[2] = Tcl_NewListObj(4, pScrRegObjs);
//		objv[2] = Tcl_NewIntObj(5);
		int retcode = tcl.evalObjs(3, objv);
		if (TCL_OK != retcode) {
			tcl.add_error("(in resetOrigin)");
			return retcode;
		}
	}
	return TCL_OK;
}

/*
 * <otcl> MBCanvas public visibleSize {}
 * returns a list containing the width and height of the visible
 * region in pixels (i.e. window size less highlightthickness and
 * borderwidth)
 *
 */
int MBCanvas::visibleSize()
{
	Tcl& tcl = Tcl::instance();
	int width, height, inset;
	int retcode= visibleSize(width, height, inset);
	if (TCL_OK != retcode) {
		tcl.add_error("in MBCanvas::visibleSize()");
		return retcode;
	}
	Tcl_Obj* objv[2];
	objv[1] = Tcl_NewIntObj(width);
	objv[2] = Tcl_NewIntObj(height);
	Tcl_Obj* pListObj = Tcl_NewListObj(2, objv);
	tcl.result(pListObj);
	return TCL_OK;
}

/*
 * calculates the width and height of the visible
 * region in pixels (i.e. window size less highlightthickness and borderwidth)
 *
 */
int MBCanvas::visibleSize(int& width, int& height, int& inset)
{
	Tcl& tcl = Tcl::instance();
	Tk_Window w = Tk_NameToWindow(tcl.interp(),
				      CONST_CAST(char*)(pathName()),
				      Tk_MainWindow(tcl.interp()));
	width = Tk_Width(w);
	height = Tk_Height(w);

	double borderwidth, hilitwidth;
	Tcl_Obj* objv[3];
	objv[0] = Tcl_NewStringObj(CONST_CAST(char*)(name()), -1);
	Tcl_IncrRefCount(objv[0]);
	objv[1] = Tcl_NewStringObj("cget", -1);
	Tcl_IncrRefCount(objv[1]);
	objv[2] = Tcl_NewStringObj("-borderwidth", -1);
	int retcode = tcl.evalObjs(3, objv);

	retcode = tcl.resultAs(&borderwidth);
	if (retcode!=TCL_OK) goto cleanup;

	objv[2] = Tcl_NewStringObj("-highlightthickness", -1);
	retcode = tcl.evalObjs(3, objv);
	if (retcode!=TCL_OK) goto cleanup;

	retcode = tcl.resultAs(&hilitwidth);
	if (retcode!=TCL_OK) goto cleanup;

	inset = int(borderwidth + hilitwidth);
	width -= 2* inset;
	height -= 2*inset;
	MTrace(trcMB|trcVerbose, ("visible size is to %d x %d, inset = %d%",
			       width, height, inset));
	return TCL_OK;
cleanup:
	tcl.add_error("(in MBCanvas::visibleSize)");
	Tcl_DecrRefCount(objv[0]);
	Tcl_DecrRefCount(objv[1]);
	return retcode;
}



/* rescale all named fonts in the hash table by scaleFactor, note that since
 * we use integers, there might be rounding errors */
int MBCanvas::rescaleAllFonts()
{
	Tcl_HashSearch hsearch;
	Tcl& tcl= Tcl::instance();

	MBCanvas::FontStruct *pFS;
	int newsize;
	double dummy, ds;
	for (Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(&htFonts_, &hsearch);
	     pEntry; pEntry = Tcl_NextHashEntry(&hsearch))
	{
		pFS = (MBCanvas::FontStruct*)Tcl_GetHashValue(pEntry);
		ds = (double)pFS->size_;
		matrix_.map(dummy, ds);
		newsize = int(ds + 0.5);
		if (newsize < 1) newsize = 1;
		tcl.evalf("font configure %s -size %d", pFS->szNamedFont_,
			  newsize);
		MTrace(trcMB|trcVerbose,
		       ("rescaling %s from size %d to %d",
			pFS->szNamedFont_, pFS->size_, newsize));
	}
	return TCL_OK;
}

/* given a font, get the size, scale it up using the canvas's current scale,
 * and return a named font for it.
 * Note: when the canvas rescales, the given named font will do likewise
 */
char* MBCanvas::scaleFont(Tk_Font tkfont)
{
	/* use the hashtable as a cache for fast lookups */
	int created;
	Tcl_HashEntry* pEntry =
		Tcl_CreateHashEntry(&htFonts_, (char*)tkfont, &created);
	MBCanvas::FontStruct *pFontStruct;
	if (created) {
		/* 1st time, map size then create a new named font */
		const char* fontname = Tk_NameOfFont(tkfont);
		Tcl& tcl = Tcl::instance();
		tcl.evalf("font actual %s", fontname);
		Tcl_Obj *pFontObj = tcl.objResult();
		Tcl_IncrRefCount(pFontObj);
		if (Tcl_IsShared(pFontObj))
			pFontObj = Tcl_DuplicateObj(pFontObj);

		const int cFontSizePos = 1;
		Tcl_Obj *pSizeObj, *pNewSizeObj;
		if (TCL_OK != Tcl_ListObjIndex(tcl.interp(), pFontObj,
					       cFontSizePos*2+1, &pSizeObj))
			assert(FALSE && "cannot get font size!");
		int size;
		if (TCL_OK !=
		    Tcl_GetIntFromObj(tcl.interp(), pSizeObj, &size)) {
			assert(FALSE && "cannot convert font size to int!");
			return NULL;
		}
		double dummy, d=size;
		matrix_.map(d, dummy);
		pNewSizeObj = Tcl_DuplicateObj(pSizeObj);
		Tcl_IncrRefCount(pNewSizeObj);
		int newsize= int(d + 0.5);
		if (newsize < 1) newsize = 1;
		Tcl_SetIntObj(pNewSizeObj, newsize);
		Tcl_ListObjReplace(tcl.interp(), pFontObj, cFontSizePos*2+1,
				   1, 1, &pNewSizeObj);
		MTrace(trcMB|trcVerbose, ("%s font %s is scaled from %d to %d",
				       pathName(), fontname, size, newsize));
		Tcl_Obj *pCmdFontObj = Tcl_NewStringObj("font", -1);
		Tcl_IncrRefCount(pCmdFontObj);

		Tcl_Obj *pCmdCreateObj = Tcl_NewStringObj("create", -1);
		Tcl_IncrRefCount(pCmdCreateObj);

		Tcl_Obj* arObjPrefix[2] = {pCmdFontObj, pCmdCreateObj};
		Tcl_ListObjReplace(tcl.interp(), pFontObj, 0, -1, 2,
				   arObjPrefix);
		tcl.evalObj(pFontObj);
		pFontStruct = new MBCanvas::FontStruct(tcl.result(), size);
		Tcl_DecrRefCount(pNewSizeObj);
		Tcl_DecrRefCount(pFontObj);
		Tcl_DecrRefCount(pCmdFontObj);
		Tcl_DecrRefCount(pCmdCreateObj);
		Tcl_SetHashValue(pEntry, (ClientData)pFontStruct);
	} else {
		pFontStruct =
			(MBCanvas::FontStruct*) Tcl_GetHashValue(pEntry);
	}
	return pFontStruct->szNamedFont_;
}

/*virtual*/
int MBCanvas::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();

	if (argc==3 && !strcmp("setpath", argv[1])) {
		setPathName(argv[2]);
		return TCL_OK;
	}
        /* the rest of the commands need a tkCanvas */
	if (pathObj_ == NULL) {
		tcl.add_error("no canvas created yet!, call ignored!");
		return TCL_ERROR;
	}

	/* try to organize checks by frequency of use */
	if (!strcmp(argv[1], "canvasxy")) {
		/* note: we only take coordinates, and not other formats like
		 *       inchs or points */
		double x, y;
		if (!(str2double(argv[2], x) && str2double(argv[3], y))) {
			tcl.add_error("Invalid coordinates,"
				      " canvasxy ONLY accepts coordinates!");
			return TCL_ERROR;
		}
		matrix_.imap(x + xOrigin_, y + yOrigin_, x, y);
		char buffer[TCL_DOUBLE_SPACE];
		Tcl_Interp* interp = tcl.interp();
		Tcl_PrintDouble(interp, x, buffer);
		Tcl_AppendElement(interp, buffer);
		Tcl_PrintDouble(interp, y, buffer);
		Tcl_AppendElement(interp, buffer);
		return TCL_OK;
	}
	/* clipxy takes a pair of unnormalized coordinates and returns
	 * a pair of unnormalized coordinates clip to the viewable
	 * region and a suitable anchor */
	if (!strcmp(argv[1], "clipxy")) {
		/* note: we only take coordinates, and not other formats like
		 *       inchs or points */
		double x, y;
		if (!(str2double(argv[2], x) && str2double(argv[3], y))) {
			tcl.add_error("Invalid coordinates,"
				      " clipxy ONLY accepts coordinates!");
			return TCL_ERROR;
		}
		char szAnchor[3] = "se";
		if (x < xOrigin_) {
			x = xOrigin_;
			szAnchor[1] = 'w';
		}
		if (y < yOrigin_) {
			y = yOrigin_;
			szAnchor[0] = 'n';
		}
		Tk_Window w = Tk_NameToWindow(tcl.interp(),
					      CONST_CAST(char*)(pathName()),
					      Tk_MainWindow(tcl.interp()));
		int width = Tk_Width(w);
		int height = Tk_Height(w);
		if (y > yOrigin_ + height) {
			y = yOrigin_ + height;
		}
		if (x > xOrigin_ + width) {
			x = xOrigin_ + width;
		}
		char buffer[TCL_DOUBLE_SPACE];
		Tcl_Interp* interp = tcl.interp();
		Tcl_PrintDouble(interp, x, buffer);
		Tcl_AppendElement(interp, buffer);
		Tcl_PrintDouble(interp, y, buffer);
		Tcl_AppendElement(interp, buffer);
		Tcl_AppendElement(interp, szAnchor);
		return TCL_OK;
	}
	/* Trap changes of xorigin and yorigin, so that we can update the
	 * matrix
	 *
	 * NOTE: this is kludgy, if there is one instance where the x/y
	 *   origin changes but we did not catch, we will not get the right
	 *   values!
	 */
	if (!strcmp(argv[1], "xview") || !strcmp(argv[1], "yview")
	    || !strcmp(argv[1], "dragto") || !strcmp(argv[1], "configure")) {
		MTrace(trcMB|trcVerbose,
		       ("redirecting command %s", argv[1]));
		int retcode = redirectCommand(argc, argv);
		if (retcode == TCL_OK) {
			retcode = resetOrigin();
		}
		return retcode;
	}

	if (argc==2) {
		if (!strcmp("getscale", argv[1])) {
			return getScale();
		}
	} else if(argc==3) {
		if (!strcmp("scalefont", argv[1])) {
			Tk_Font tkfont =
				Tk_GetFont(tcl.interp(),
					   Tk_MainWindow(tcl.interp()),
					   argv[2]);
			tcl.result(scaleFont(tkfont));
			Tk_FreeFont(tkfont);
			return TCL_OK;
		}
		if (!strcmp("owner", argv[1])) {
			int canvId;
			if (!str2int(argv[2], canvId)) {
				tcl.add_error("Invalid canvas id");
				return TCL_ERROR;
			}
			MBCmd* pCmd = canvId2cmd(canvId);
			if (NULL == pCmd) {
				tcl.result("");
				return TCL_OK;
			}
			MBBaseRcvr* pRcvr = pCmd->rcvr();
			assert(pRcvr
			       || !"MBCanvas owner: have cmd but no rcvr!");
			/* FIXME: should unify MBReceivers and
			 * SRM_Sources! */
			SRM_Source* pSrc = pRcvr->getSrc();
			if (pSrc) {
				tcl.result(pSrc->name());
			}
			return TCL_OK;
		}
		if (!strcmp("rescale", argv[1])) {
			double newscale;
			if (!str2double(argv[2], newscale)) {
				tcl.add_error("Invalid width");
				return TCL_ERROR;
			}
			return rescale(newscale, 0);
		}
	} else if (argc==4) {
		int width, height;
		if (!strcmp("setsize", argv[1])) {
			if (!str2int(argv[2], width)) {
				tcl.add_error("Invalid width");
				return TCL_ERROR;
			}
			if (!str2int(argv[3], height)) {
				tcl.add_error("Invalid height");
				return TCL_ERROR;
			}
			return setsize(width, height);
		} else if (!strcmp("resize", argv[1])) {
			if (!str2int(argv[2], width)) {
				tcl.add_error("Invalid width");
				return TCL_ERROR;
			}
			if (!str2int(argv[3], height)) {
				tcl.add_error("Invalid height");
				return TCL_ERROR;
			}
			return resize(width, height);
		} else if (!strcmp("fit", argv[1])) {
			double width, height;
			if (!str2double(argv[2], width)) {
				tcl.add_error("Invalid width");
				return TCL_ERROR;
			}
			if (!str2double(argv[3], height)) {
				tcl.add_error("Invalid height");
				return TCL_ERROR;
			}
			return fit(width, height);
		}

	} //  else if (argc==4) {

	/* redirect all other commands to the tkCanvas */
	return redirectCommand(argc, argv);
}

/* FIXME: this function may not be used */
int MBCanvas::showOwner(CanvItemId cid)
{
	/* calls $self showOwner $cid 0 */
	Tcl_Obj* objv[4];
	objv[0] = pathObj();
	objv[1] = Tcl_NewStringObj("show_owner", -1);
	objv[2] = Tcl_NewLongObj(cid);
	objv[3] = Tcl_NewLongObj(1);
	return (Tcl::instance().evalObjs(4, objv));
}

#endif /* #ifdef MB-CANV_CC */
