/*
 * transcoder.cc --
 *
 *      Transcoder, contains both audio and video transcoder
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/codec/transcoder.cc,v 1.16 2002/02/03 03:13:33 lim Exp $";
#endif

#include <string.h>
#include <sys/types.h>
#include "ntp-time.h"
#include "transcoder.h"
#include "crdef.h"

static class NullTranscoderClass : public TclClass {
public:
	NullTranscoderClass() : TclClass("Transcoder/Null") {}
	TclObject* create(int, const char*const*) {
		return (new NullTranscoder);
	}
} null_transcoder_class;

Transcoder::Transcoder(int hdrlen)
	   :PacketHandler(hdrlen), encoder_(0),
	    dh_(0), ch_(0), source_(0),
	    opkts_(0), obytes_(0), ofrms_(0), bps_(0), lastfc_(0.), fc_(0.),
	    forward_(0), txonly_(0)
{
	bind("bps_", &bps_);
	bind("opkts_", (int*)&opkts_);
	bind("obytes_", (int*)&obytes_);
	bind("ofrms_", (int*)&ofrms_);
	bind("forward_", &forward_);
	bind("txonly_", &txonly_);
}

void Transcoder::bps(int b)
{
	double mbits = (fc_ - lastfc_) * bps_;
	fc_ = lastfc_ + mbits / b;
	bps_ = b;
}


int Transcoder::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 2) {
		if (strcmp(argv[1], "source") == 0) {
			tcl.resultf("%s", source_->name());
			return (TCL_OK);
		}
		if (strcmp(argv[1], "data-handler") == 0) {
			tcl.resultf("%s", dh_->name());
			return (TCL_OK);
		}
		if (strcmp(argv[1], "ctrl-handler") == 0) {
			tcl.resultf("%s", ch_->name());
			return (TCL_OK);
		}
	} else if (argc == 3) {
		if (strcmp(argv[1], "source") == 0) {
			source_ = (Source*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "encoder") == 0) {
			encoder_ = (EncoderModule*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "data-handler") == 0) {
			dh_ = (DataInputHandler*)TclObject::lookup(argv[2]);
			dh_->manager(this);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "ctrl-handler") == 0) {
			ch_ = (ControlInputHandler*)TclObject::lookup(argv[2]);
			ch_->manager(this);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "set-bps") == 0) {
			bps(atoi(argv[2]));
			return (TCL_OK);
		}

	}
	return (PacketHandler::command(argc, argv));
}

/*
 * Receive an rtcp packet (from the control port).
 */
void Transcoder::recv_ctrl(pktbuf* pb)
{
	rtcphdr* rh = (rtcphdr*)pb->dp;

	Source* ps = source_;
	int layer = pb->layer;
	/*
	 * Outer loop parses multiple RTCP records of a "compound packet".
	 * There is no framing between records.  Boundaries are implicit
	 * and the overall length comes from UDP.
	 */
	u_char* epack = (u_char*)rh + pb->len;
	int bye = 0;
	int sr = 0;
	while ((u_char*)rh < epack) {
		u_int len = (ntohs(rh->rh_len) << 2) + 4;
		u_char* ep = (u_char*)rh + len;
		if (ep > epack) {
			pb->release();
			return;
		}
		u_int flags = ntohs(rh->rh_flags);
		if (flags >> 14 != RTP_VERSION) {
			pb->release();
			return;
		}
		switch (flags & 0xff) {

		case RTCP_PT_SR:
			sr = 1;
			parse_sr(rh, flags, ep, ps, layer);
			break;

		case RTCP_PT_RR:
			parse_rr(rh, flags, ep, ps, layer);
			break;

		case RTCP_PT_BYE:
			bye = 1;
			parse_bye(rh, flags, ep, ps);
			break;

		case RTCP_PT_SDES:
		case RTCP_PT_APP:
			break;
		}
		rh = (rtcphdr*)ep;
	}

	/* Attach an RTCP_APP packet if needed */
	if (!bye)
		pb->len += build_app((rtcphdr*)epack, *ps, sr);

	ch_->target()->recv(pb);
}

int Transcoder::build_app(rtcphdr*, Source&, int)
{
	return (0);
}

void Transcoder::parse_bye(rtcphdr*, int, u_char*, Source*)
{
}

void Transcoder::parse_sr(rtcphdr* rh, int flags, u_char*ep,
			      Source* ps, int layer)
{
	/* Modify sender report to reflect transcoder output. */
	rtcp_sr* sr = (rtcp_sr*)(rh + 1);
	sr->sr_np = htonl(opkts_);
	sr->sr_nb = htonl(obytes_);

	int cnt = flags >> 8 & 0x1f;
	parse_rr_records((rtcp_rr*)(sr + 1), cnt, ep, ps, layer);
}

void Transcoder::parse_rr(rtcphdr* rh, int flags, u_char* ep,
			      Source* ps, int layer)
{
	int cnt = flags >> 8 & 0x1f;
	parse_rr_records((rtcp_rr*)(rh + 1), cnt, ep, ps, layer);
}

void Transcoder::parse_rr_records(rtcp_rr* rr, int cnt, const u_char* ep,
				  Source* ps, int layer)
{
	while(cnt-- && (u_char*)rr < ep) {
		Source::Layer& sl = ps->layer(layer);
		int received = sl.np() - sl.snp();
		sl.snp(sl.np());
		int expected = sl.ns() - sl.sns();
		sl.sns(sl.ns());
		u_int32_t v;
		int lost = expected - received;
		if (lost <= 0)
			v = 0;
		else
			/* expected != 0 if lost > 0 */
			v = ((lost << 8) / expected) << 24;
		/* FIXME should saturate on over/underflow */
		v |= (sl.ns() - sl.np()) & 0xffffff;
		rr->rr_loss = htonl(v);
		rr->rr_ehsr = htonl(sl.ehs());
	/* FIXME */
#ifdef notdef
		rr->rr_dv = (ps->handler() != 0) ? ps->handler()->delvar() : 0;
#endif
		rr++;
	}
}

VideoTranscoder::VideoTranscoder(int hdrlen)
		:Transcoder(hdrlen), decimation_(411), inw_(0), inh_(0),
		 outw_(0), outh_(0), have_frame_(0), mb_crvec_(0),
		 rtcp_thumbnail_(0)
{}


void VideoTranscoder::pixel_crinit(int w, int h)
{
	crinit(w * 2, h * 2);

	delete mb_crvec_;
	mb_crvec_ = new u_char[nblk_ / 4];
}

VideoTranscoder::~VideoTranscoder()
{
	delete mb_crvec_;
}

void VideoTranscoder::blk_to_mb_cr()
{
	int blkno = 0;
	int mbno = 0;
	int mbh = blkh_ / 2;
	int mbw = blkw_ / 2;
	for (int h = mbh; h > 0; --h) {
		for (int w = mbw; w > 0; --w) {
			mb_crvec_[mbno] = crvec_[blkno];
			++mbno;
			blkno += 2;
		}
		blkno += blkw_;
	}
}

int VideoTranscoder::command(int argc, const char*const* argv)
{
	if (argc < 2)
		return (TCL_ERROR);

	Tcl& tcl = Tcl::instance();
	const char* cmd = argv[1];
	if (argc == 2) {
		if (strcmp(cmd, "width") == 0) {
			tcl.resultf("%d", inw_);
			return (TCL_OK);
		}
		if (strcmp(cmd, "height") == 0) {
			tcl.resultf("%d", inh_);
			return (TCL_OK);
		}
		if (strcmp(cmd, "decimation") == 0) {
			tcl.resultf("%d", decimation_);
			return (TCL_OK);
		}
	}
	if (argc == 3) {
		if (strcmp(cmd, "color") == 0) {
			color(atoi(argv[2]));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "rtcp-thumbnail") == 0) {
			rtcp_thumbnail_ = atoi(argv[2]);
			return (TCL_OK);
		}
	}

	return (Transcoder::command(argc, argv));
}


int VideoTranscoder::build_app(rtcphdr* rr, Source& s, int sr)
{
	rtcphdr* rh = rr;
	if (rtcp_thumbnail_ && sr)
		rh = (rtcphdr*)build_rtcp_thumbnail(rh, s);

	return ((u_char*)rh - (u_char*)rr);
}

u_char* VideoTranscoder::build_rtcp_thumbnail(rtcphdr* rh, Source& s)
{
	/* FIXME right now, all we do is signal that source is active */
	int flags = RTP_VERSION << 14 | 1 << 8 | RTCP_PT_APP;
	rh->rh_flags = htons(flags);
	rh->rh_ssrc = s.srcid();
	rtcp_thumbnail* tn = (rtcp_thumbnail*)(rh + 1);
	/* Just so we don't confuse if some application happens to send an
	 * APP packet.
	 */
	tn->magic = htons(0xbeef);
	tn->format = htons(s.format());
	u_char* ep = ((u_char*)(tn + 1));
	int len = ep - (u_char*)rh;
	rh->rh_len = htons((len >> 2) - 1);
	return (ep);
}

#define	INITIAL_OFFSET (4*AUDIO_FRAMESIZE)

AudioTranscoder::AudioTranscoder(int hdrlen)
		:Transcoder(hdrlen),
		 var_(INITIAL_OFFSET << (VAR_FILTER - VAR_MULT)),
		 hostoffset_(0), predicted_drop_((u_int32_t) -1),
		 playout_(INITIAL_OFFSET << PLAYO_FILTER),
		 lecture_mode_(0),
		 framesize_(AUDIO_FRAMESIZE),
		 block_size_(AUDIO_FRAMESIZE), maxdel_(0)
{
}

int AudioTranscoder::command(int argc, const char*const* argv)
{
	if (argc < 2)
		return (TCL_ERROR);

	Tcl& tcl = Tcl::instance();

	const char* cmd = argv[1];
	if (argc == 2) {
		char* bp = tcl.buffer();
		if (strcmp(cmd, "block-size") == 0) {
			sprintf(bp, "%u", block_size_);
			tcl.result(bp);
			return (TCL_OK);
		}
		if (strcmp(cmd, "playout") == 0) {
			sprintf(bp, "%u", playout());
			tcl.result(bp);
			return (TCL_OK);
		}
		if (strcmp(cmd, "reset-offset") == 0) {
			hostoffset_ = 0;
			var_ = 0;
			playout_ = 0;
			return (TCL_OK);
		}
	} else if (argc == 3) {
		if (strcmp(cmd, "lecture-mode") == 0) {
			lecture_mode_ = atoi(argv[2]);
			return (TCL_OK);
		}
	}
	return (Transcoder::command(argc, argv));
}

inline int absdiff(int x, int y)
{
	register int r = y - x;
	return (r < 0? -r : r);
}

static inline int newoffset(
	int nvar,
	int playout,
	int maxdel, int mindel, int lecture)
{
	register int offset = nvar;
	if (offset > maxdel)
		offset = maxdel;
	register int diff = playout - offset;

	if (diff > 0) {
		// offset going down: in LectureMode, drop at most
		// one frametime per talkspurt.  In ConferenceMode,
		// drop at most 1/2 of difference.
		if (lecture) {
			if (diff > AUDIO_FRAMESIZE) {
				if (playout > (maxdel * 3) / 4 &&
				    diff > 10 * AUDIO_FRAMESIZE)
					diff = 5 * AUDIO_FRAMESIZE;
				else
					diff = AUDIO_FRAMESIZE;
			}
		} else
			diff >>= 1;
		offset = playout - diff;
	} else if (-diff > maxdel) {
		// offset going way up: only allow 3/4 of max.
		offset = (maxdel * 3) / 4;
	}
	if (offset > (maxdel * 7) / 8)
		offset = (maxdel * 7) / 8;
	else if (offset < mindel)
		offset = mindel;
	return (offset);
}

int AudioTranscoder::adapt(const rtphdr* rh, u_int32_t local_clock)
{
	register u_int32_t tstamp = ntohl(rh->rh_ts);
	register int hoff = (int)hostoffset_;
	register int offset = (tstamp + hoff - local_clock) &~ 3;
	int new_ts = rh->rh_flags & htons(RTP_M);
	if (hoff == 0 || new_ts) {
		/*
		 * start of new talk spurt --
		 * use accumulated variance to compute new offset if
		 * this would make a significant change.  We change if
		 *  - the variance is currently 'small', or
		 *  - the change would be a least a packet time
		 */
		register int nvar = var_ >> (VAR_FILTER - VAR_MULT);
		offset = playout_ >> PLAYO_FILTER;
		if (nvar < 3*AUDIO_FRAMESIZE || absdiff(nvar, offset) >= AUDIO_FRAMESIZE) {
			offset = newoffset(nvar, offset, maxdel_,
					   block_size_, lecture_mode_);
			/*
			 * assume that a talk spurt starts with TALK_LEAD
			 * samples of history & subtract them off if possible.
			 */
			if (new_ts) {
				offset -= 4 * AUDIO_FRAMESIZE;
				if (offset < block_size_)
					offset = block_size_;
			}
		}
		hostoffset_ = local_clock - tstamp + offset;
	} else if (offset < 0 || offset > maxdel_) {
		/*
		 * packet out of range -- if last packet also out of
		 * range or if the delay would increase, resync.
		 */
		if (offset < 0 || predicted_drop_ == tstamp) {
			offset = newoffset(var_ >> (VAR_FILTER - VAR_MULT),
					   playout_ >> PLAYO_FILTER,
					   maxdel_, block_size_,
					   lecture_mode_);
			hostoffset_ = local_clock - tstamp + offset;
		} else {
			predicted_drop_ = tstamp + block_size_;
			lastrecv_ = tstamp - local_clock;
			return (-1);
		}
	} else {
		// packet in range, update interarrival var. est.
		register int nvar = var_;
		register int off = tstamp - local_clock - lastrecv_;
		if (off < 0)
			off = -off;
		off -= (nvar >> VAR_FILTER);
		var_ = nvar + off;
	}
	lastrecv_ = tstamp - local_clock;
	register u_int avgplay = playout_;
	playout_ = avgplay + (offset - (avgplay >> PLAYO_FILTER));
	offset &= ~3;
	delvar_ = var_ >> VAR_FILTER;
	return (offset);
}

void NullTranscoder::recv_data(pktbuf* pb)
{
	if (forward_ || txonly_) {
		dh_->target()->recv(pb);
		opkts_++;
		obytes_ += pb->len;
		return;
	}
	if (bps_ == 0) {
		pb->release();
		return;
	}
	/*
	 * Rate control.
	 * This actually bogus, since we are dropping random packets,
	 * but the alternative is not to let any forwarding to
	 * occur in rate controlled sessions.  May as well let the user
	 * decide.
	 */
	double now = gettimeofday();
	if (fc_ <= now) {
		if (now - fc_ > 200000.)
			fc_ = now;
		dh_->target()->recv(pb);
		int bits = 8 * pb->len;
		opkts_++;
		obytes_ += pb->len;
		lastfc_ = fc_;
		fc_ += 1e6 * bits / bps_;
	}
	pb->release();
}

void NullTranscoder::recv_ctrl(pktbuf* pb)
{
	ch_->target()->recv(pb);
}

