/*
 * Copyright (C) 2001-2007 Keisuke Nishida
 * Copyright (C) 2007 Roger While
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1,
 * or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; see the file COPYING.LIB.  If
 * not, write to the Free Software Foundation, 51 Franklin Street, Fifth Floor
 * Boston, MA 02110-1301 USA
 */

#include "config.h"
#include "defaults.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#ifdef	HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <time.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef	_WIN32
#define WINDOWS_LEAN_AND_MEAN
#include <windows.h>
#include <io.h>
#include <fcntl.h>
#undef	HAVE_SIGNAL_H
#endif

#ifdef	HAVE_SIGNAL_H
#include <signal.h>
#endif

#ifdef	HAVE_LOCALE_H
#include <locale.h>
#endif

#include "common.h"
#include "coblocal.h"
#include "move.h"
#include "numeric.h"
#include "termio.h"
#include "fileio.h"
#include "call.h"
#include "screenio.h"
#include "intrinsic.h"
#include "lib/gettext.h"

struct cob_exception {
	const char	*name;
	const int	code;
	const int	critical;
};

struct cob_alloc_cache {
	struct cob_alloc_cache	*next;
	void			*cob_pointer;
	size_t			size;
};

static int			cob_argc = 0;
static char			**cob_argv = NULL;
static struct cob_alloc_cache	*cob_alloc_base = NULL;

static char			*env = NULL;
static int			current_arg = 1;

static char			*locale_save = NULL;

static size_t			sort_nkeys;
static cob_file_key		*sort_keys;
static const unsigned char	*sort_collate;


int			cob_initialized = 0;
int			cob_exception_code = 0;

cob_module		*cob_current_module = NULL;

const char		*cob_source_file = NULL;
const char		*cob_source_statement = NULL;
const char		*cob_orig_statement = NULL;
const char		*cob_current_program_id = NULL;
const char		*cob_current_section = NULL;
const char		*cob_current_paragraph = NULL;
const char		*cob_orig_program_id = NULL;
const char		*cob_orig_section = NULL;
const char		*cob_orig_paragraph = NULL;
unsigned int		cob_source_line = 0;
unsigned int		cob_orig_line = 0;

int			cob_call_params = 0;
int			cob_save_call_params = 0;
int			cob_initial_external = 0;
int			cob_got_exception = 0;

#ifdef	HAVE_SIGNAL_H
typedef void (*cob_sighandler_t) (int);
static cob_sighandler_t	hupsig = NULL;
static cob_sighandler_t	intsig = NULL;
static cob_sighandler_t	qutsig = NULL;
#endif

static cob_field_attr	all_attr = { COB_TYPE_ALPHANUMERIC_ALL, 0, 0, 0, NULL };
static cob_field_attr	one_attr = { COB_TYPE_NUMERIC, 1, 0, 0, NULL };

cob_field		cob_zero = { 1, (ucharptr)"0", &all_attr };
cob_field		cob_space = { 1, (ucharptr)" ", &all_attr };
cob_field		cob_high = { 1, (ucharptr)"\xff", &all_attr };
cob_field		cob_low = { 1, (ucharptr)"\0", &all_attr };
cob_field		cob_quote = { 1, (ucharptr)"\"", &all_attr };
cob_field		cob_one = { 1, (ucharptr)"1", &one_attr };

/* Generated by codegen - ASCII to EBCDIC MF like
const unsigned char	cob_a2e[256] = {
	0x00, 0x01, 0x02, 0x03, 0x1D, 0x19, 0x1A, 0x1B,
	0x0F, 0x04, 0x16, 0x06, 0x07, 0x08, 0x09, 0x0A,
	0x0B, 0x0C, 0x0D, 0x0E, 0x1E, 0x1F, 0x1C, 0x17,
	0x10, 0x11, 0x20, 0x18, 0x12, 0x13, 0x14, 0x15,
	0x21, 0x27, 0x3A, 0x36, 0x28, 0x30, 0x26, 0x38,
	0x24, 0x2A, 0x29, 0x25, 0x2F, 0x2C, 0x22, 0x2D,
	0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7A,
	0x7B, 0x7C, 0x35, 0x2B, 0x23, 0x39, 0x32, 0x33,
	0x37, 0x57, 0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D,
	0x5E, 0x5F, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
	0x67, 0x68, 0x69, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F,
	0x70, 0x71, 0x72, 0x7D, 0x6A, 0x7E, 0x7F, 0x31,
	0x34, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x40, 0x41,
	0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49,
	0x4A, 0x4B, 0x4C, 0x4E, 0x4F, 0x50, 0x51, 0x52,
	0x53, 0x54, 0x55, 0x56, 0x2E, 0x60, 0x4D, 0x05,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F,
	0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7,
	0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF,
	0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7,
	0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF,
	0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
	0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF,
	0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7,
	0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF,
	0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7,
	0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF,
	0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
	0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
};
end comment */

/* Full Table ASCII to EBCDIC
const unsigned char	cob_a2e[256] = {
	0x00, 0x01, 0x02, 0x03, 0x37, 0x2D, 0x2E, 0x2F,
	0x16, 0x05, 0x25, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
	0x10, 0x11, 0x12, 0x13, 0x3C, 0x3D, 0x32, 0x26,
	0x18, 0x19, 0x3F, 0x27, 0x1C, 0x1D, 0x1E, 0x1F,
	0x40, 0x5A, 0x7F, 0x7B, 0x5B, 0x6C, 0x50, 0x7D,
	0x4D, 0x5D, 0x5C, 0x4E, 0x6B, 0x60, 0x4B, 0x61,
	0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
	0xF8, 0xF9, 0x7A, 0x5E, 0x4C, 0x7E, 0x6E, 0x6F,
	0x7C, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
	0xC8, 0xC9, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6,
	0xD7, 0xD8, 0xD9, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6,
	0xE7, 0xE8, 0xE9, 0xAD, 0xE0, 0xBD, 0x5F, 0x6D,
	0x79, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96,
	0x97, 0x98, 0x99, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6,
	0xA7, 0xA8, 0xA9, 0xC0, 0x6A, 0xD0, 0xA1, 0x07,
	0x68, 0xDC, 0x51, 0x42, 0x43, 0x44, 0x47, 0x48,
	0x52, 0x53, 0x54, 0x57, 0x56, 0x58, 0x63, 0x67,
	0x71, 0x9C, 0x9E, 0xCB, 0xCC, 0xCD, 0xDB, 0xDD,
	0xDF, 0xEC, 0xFC, 0xB0, 0xB1, 0xB2, 0x3E, 0xB4,
	0x45, 0x55, 0xCE, 0xDE, 0x49, 0x69, 0x9A, 0x9B,
	0xAB, 0x9F, 0xBA, 0xB8, 0xB7, 0xAA, 0x8A, 0x8B,
	0xB6, 0xB5, 0x62, 0x4F, 0x64, 0x65, 0x66, 0x20,
	0x21, 0x22, 0x70, 0x23, 0x72, 0x73, 0x74, 0xBE,
	0x76, 0x77, 0x78, 0x80, 0x24, 0x15, 0x8C, 0x8D,
	0x8E, 0x41, 0x06, 0x17, 0x28, 0x29, 0x9D, 0x2A,
	0x2B, 0x2C, 0x09, 0x0A, 0xAC, 0x4A, 0xAE, 0xAF,
	0x1B, 0x30, 0x31, 0xFA, 0x1A, 0x33, 0x34, 0x35,
	0x36, 0x59, 0x08, 0x38, 0xBC, 0x39, 0xA0, 0xBF,
	0xCA, 0x3A, 0xFE, 0x3B, 0x04, 0xCF, 0xDA, 0x14,
	0xE1, 0x8F, 0x46, 0x75, 0xFD, 0xEB, 0xEE, 0xED,
	0x90, 0xEF, 0xB3, 0xFB, 0xB9, 0xEA, 0xBB, 0xFF
};
end comment */

/* Not needed at the moment - EBCDIC to ASCII MF like 
const unsigned char	cob_e2a[256] = {
	0x00, 0x01, 0x02, 0x03, 0x9C, 0x09, 0x86, 0x7F,
	0x97, 0x8D, 0x8E, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
	0x10, 0x11, 0x12, 0x13, 0x9D, 0x85, 0x08, 0x87,
	0x18, 0x19, 0x92, 0x8F, 0x1C, 0x1D, 0x1E, 0x1F,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x0A, 0x17, 0x1B,
	0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x05, 0x06, 0x07,
	0x90, 0x91, 0x16, 0x93, 0x94, 0x95, 0x96, 0x04,
	0x98, 0x99, 0x9A, 0x9B, 0x14, 0x15, 0x9E, 0x1A,
	0x20, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6,
	0xA7, 0xA8, 0x5B, 0x2E, 0x3C, 0x28, 0x2B, 0x21,
	0x26, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF,
	0xB0, 0xB1, 0x5D, 0x24, 0x2A, 0x29, 0x3B, 0x5E,
	0x2D, 0x2F, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7,
	0xB8, 0xB9, 0x7C, 0x2C, 0x25, 0x5F, 0x3E, 0x3F,
	0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xC0, 0xC1,
	0xC2, 0x60, 0x3A, 0x23, 0x40, 0x27, 0x3D, 0x22,
	0xC3, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9,
	0xCA, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70,
	0x71, 0x72, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0,
	0xD1, 0x7E, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
	0x79, 0x7A, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7,
	0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF,
	0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7,
	0x7B, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED,
	0x7D, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50,
	0x51, 0x52, 0xEE, 0xEF, 0xF0, 0xF1, 0xF2, 0xF3,
	0x5C, 0x9F, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
	0x59, 0x5A, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
};
end comment */

/* Not needed at the moment - EBCDIC to ASCII Full table
const unsigned char	cob_e2a[256] = {
	0x00, 0x01, 0x02, 0x03, 0xEC, 0x09, 0xCA, 0x7F,
	0xE2, 0xD2, 0xD3, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
	0x10, 0x11, 0x12, 0x13, 0xEF, 0xC5, 0x08, 0xCB,
	0x18, 0x19, 0xDC, 0xD8, 0x1C, 0x1D, 0x1E, 0x1F,
	0xB7, 0xB8, 0xB9, 0xBB, 0xC4, 0x0A, 0x17, 0x1B,
	0xCC, 0xCD, 0xCF, 0xD0, 0xD1, 0x05, 0x06, 0x07,
	0xD9, 0xDA, 0x16, 0xDD, 0xDE, 0xDF, 0xE0, 0x04,
	0xE3, 0xE5, 0xE9, 0xEB, 0x14, 0x15, 0x9E, 0x1A,
	0x20, 0xC9, 0x83, 0x84, 0x85, 0xA0, 0xF2, 0x86,
	0x87, 0xA4, 0xD5, 0x2E, 0x3C, 0x28, 0x2B, 0xB3,
	0x26, 0x82, 0x88, 0x89, 0x8A, 0xA1, 0x8C, 0x8B,
	0x8D, 0xE1, 0x21, 0x24, 0x2A, 0x29, 0x3B, 0x5E,
	0x2D, 0x2F, 0xB2, 0x8E, 0xB4, 0xB5, 0xB6, 0x8F,
	0x80, 0xA5, 0x7C, 0x2C, 0x25, 0x5F, 0x3E, 0x3F,
	0xBA, 0x90, 0xBC, 0xBD, 0xBE, 0xF3, 0xC0, 0xC1,
	0xC2, 0x60, 0x3A, 0x23, 0x40, 0x27, 0x3D, 0x22,
	0xC3, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0xAE, 0xAF, 0xC6, 0xC7, 0xC8, 0xF1,
	0xF8, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70,
	0x71, 0x72, 0xA6, 0xA7, 0x91, 0xCE, 0x92, 0xA9,
	0xE6, 0x7E, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
	0x79, 0x7A, 0xAD, 0xA8, 0xD4, 0x5B, 0xD6, 0xD7,
	0x9B, 0x9C, 0x9D, 0xFA, 0x9F, 0xB1, 0xB0, 0xAC,
	0xAB, 0xFC, 0xAA, 0xFE, 0xE4, 0x5D, 0xBF, 0xE7,
	0x7B, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0xE8, 0x93, 0x94, 0x95, 0xA2, 0xED,
	0x7D, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50,
	0x51, 0x52, 0xEE, 0x96, 0x81, 0x97, 0xA3, 0x98,
	0x5C, 0xF0, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
	0x59, 0x5A, 0xFD, 0xF5, 0x99, 0xF7, 0xF6, 0xF9,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0xDB, 0xFB, 0x9A, 0xF4, 0xEA, 0xFF
};
end comment */

#ifdef	COB_PARAM_CHECK
static const char	parm_msg[] = "CALL to %s requires %d parameters";
#endif

static const struct cob_exception	cob_exception_table[] = {
	{NULL, 0, 0},		/* COB_EC_ZERO */
#undef COB_EXCEPTION
#define COB_EXCEPTION(CODE,TAG,NAME,CRITICAL) { NAME, 0x##CODE, CRITICAL },
#include "exception.def"
	{NULL, 0, 0}		/* COB_EC_MAX */
};

#define EXCEPTION_TAB_SIZE	sizeof(cob_exception_table) / sizeof(struct cob_exception)

static int		cob_switch[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };

/* Runtime exit handling */
static struct exit_handlerlist {
	struct exit_handlerlist	*next;
	int			(*proc)(void);
} *exit_hdlrs = NULL;

/* Runtime error handling */
static struct handlerlist {
	struct handlerlist	*next;
	int			(*proc)(char *s);
} *hdlrs = NULL;

/*
 * General functions
 */

unsigned char *
cob_get_pointer (const unsigned char *srcptr)
{
	unsigned char	*tmptr;

	memcpy (&tmptr, srcptr, sizeof (void *));
	return tmptr;
}

void *
cob_get_prog_pointer (const unsigned char *srcptr)
{
	void	*tmptr;

	memcpy (&tmptr, srcptr, sizeof (void *));
	return tmptr;
}

void
cob_memcpy (cob_field *dst, unsigned char *src, int size)
{
	cob_field	temp;
	cob_field_attr	attr;

	COB_ATTR_INIT (COB_TYPE_ALPHANUMERIC, 0, 0, 0, NULL);
	temp.size = size;
	temp.data = src;
	temp.attr = &attr;
	cob_move (&temp, dst);
}

const char *
cob_get_exception_name (const int exception_code)
{
	size_t	n;

	for (n = 0; n < EXCEPTION_TAB_SIZE; n++) {
		if (exception_code == cob_exception_table[n].code) {
			return cob_exception_table[n].name;
		}
	}
	return NULL;
}

void
cob_set_exception (const int id)
{
	cob_exception_code = cob_exception_table[id].code;
	if (cob_exception_code) {
		cob_got_exception = 1;
		cob_orig_statement = cob_source_statement;
		cob_orig_line = cob_source_line;
		cob_orig_program_id = cob_current_program_id;
		cob_orig_section = cob_current_section;
		cob_orig_paragraph = cob_current_paragraph;
	}
}


/* static sighandler_t	oldsig; */

#ifdef	HAVE_SIGNAL_H
static void
cob_sig_handler (int sig)
{
	fprintf (stderr, "Abnormal termination - File contents may not be correct\n");
	fflush (stderr);
	cob_screen_terminate ();
	cob_exit_fileio ();
	switch (sig) {
	case SIGHUP:
		if ((hupsig != SIG_IGN) && (hupsig != SIG_DFL)) {
			(*hupsig) (SIGHUP);
		}
		break;
	case SIGINT:
		if ((intsig != SIG_IGN) && (intsig != SIG_DFL)) {
			(*intsig) (SIGINT);
		}
		break;
	case SIGQUIT:
		if ((qutsig != SIG_IGN) && (qutsig != SIG_DFL)) {
			(*qutsig) (SIGQUIT);
		}
		break;
	}
	exit (sig);
}
#endif

void
cob_set_signal ()
{
#ifdef	HAVE_SIGNAL_H
	if ((intsig = signal (SIGINT, cob_sig_handler)) == SIG_IGN) {
		(void)signal (SIGINT, SIG_IGN);
	}
	if ((hupsig = signal (SIGHUP, cob_sig_handler)) == SIG_IGN) {
		(void)signal (SIGHUP, SIG_IGN);
	}
	if ((qutsig = signal (SIGQUIT, cob_sig_handler)) == SIG_IGN) {
		(void)signal (SIGQUIT, SIG_IGN);
	}
#endif
}

void
cob_init (int argc, char **argv)
{
	char	*s;
	size_t	i;
	char	buff[32];

	if (!cob_initialized) {
		cob_argc = argc;
		cob_argv = argv;

#ifdef	HAVE_SETLOCALE
		setlocale (LC_ALL, "");
		setlocale (LC_NUMERIC, "C");
		s = setlocale (LC_ALL, NULL);
		if (s) {
			locale_save = strdup (s);
		}
#endif
#ifdef	ENABLE_NLS
		bindtextdomain (PACKAGE, LOCALEDIR);
		textdomain (PACKAGE);
#endif

/* Dirty hack until we implement something better */
#if defined(_WIN32) && !defined(_MSC_VER)
		_setmode (_fileno (stdin), _O_BINARY);
		_setmode (_fileno (stdout), _O_BINARY);
		_setmode (_fileno (stderr), _O_BINARY);
#endif

		cob_init_numeric ();
#if 0
		cob_init_termio ();
#endif
		cob_init_fileio ();
		cob_init_call ();
		cob_init_intrinsic ();

		for (i = 0; i < 8; i++) {
			memset (buff, 0, sizeof (buff));
			sprintf (buff, "COB_SWITCH_%d", (int)(i + 1));
			s = getenv (buff);
			if (s && strcasecmp (s, "ON") == 0) {
				cob_switch[i] = 1;
			}
		}

		cob_initialized = 1;
	}
}

void
cob_module_enter (cob_module *module)
{
	if (unlikely(!cob_initialized)) {
		fputs ("Warning: cob_init expected in the main program\n", stderr);
		cob_init (0, NULL);
	}

	module->next = cob_current_module;
	cob_current_module = module;
}

void
cob_module_leave (cob_module *module)
{
	cob_current_module = cob_current_module->next;
}

void
cob_stop_run (const int status)
{
	struct exit_handlerlist	*h;

	if (exit_hdlrs != NULL) {
		h = exit_hdlrs;
		while (h != NULL) {
			h->proc ();
			h = h->next;
		}
	}
	cob_screen_terminate ();
	cob_exit_fileio ();
	exit (status);
}

void COB_NOINLINE
cob_runtime_error (const char *fmt, ...)
{
	struct handlerlist	*h;
	char			*p;
	va_list ap;
	char			str[COB_MEDIUM_BUFF];

	if (hdlrs != NULL) {
		h = hdlrs;
		p = str;
		if (cob_source_file) {
			sprintf (str, "%s:%d: ", cob_source_file, cob_source_line);
			p = str + strlen (str);
		}
		va_start (ap, fmt);
		vsprintf (p, fmt, ap);
		va_end (ap);
		while (h != NULL) {
			h->proc (str);
			h = h->next;
		}
		hdlrs = NULL;
	}

	/* prefix */
	if (cob_source_file) {
		fprintf (stderr, "%s:%d: ", cob_source_file, cob_source_line);
	}
	fputs ("libcob: ", stderr);

	/* body */
	va_start (ap, fmt);
	vfprintf (stderr, fmt, ap);
	va_end (ap);

	/* postfix */
	fputs ("\n", stderr);
	fflush (stderr);
}

void
cob_fatal_error (const enum cob_enum_error fatal_error)
{
	switch (fatal_error) {
	case COB_FERROR_INITIALIZED:
		cob_runtime_error ("cob_init() has not been called");
		break;
	case COB_FERROR_CODEGEN:
		cob_runtime_error ("Codegen error - Please report this");
		break;
	case COB_FERROR_CHAINING:
		cob_runtime_error ("ERROR - Recursive call of chained program");
		break;
	case COB_FERROR_STACK:
		cob_runtime_error ("Stack overflow, possible PERFORM depth exceeded");
		break;
	default:
		cob_runtime_error ("Unknown failure");
		break;
	}
	cob_stop_run (1);
}

void
cob_check_version (const char *prog, const char *packver, const int patchlev)
{
	if (strcmp (packver, PACKAGE_VERSION) || patchlev > PATCH_LEVEL) {
		cob_runtime_error ("Error - Version mismatch");
		cob_runtime_error ("%s has version/patch level %s/%d", prog, packver,
				   patchlev);
		cob_runtime_error ("Library has version/patch level %s/%d", PACKAGE_VERSION,
				   PATCH_LEVEL);
		cob_stop_run (1);
	}
	return;
}

/*
 * Sign
 */

#ifdef	COB_EBCDIC_MACHINE
static void
cob_get_sign_ascii (unsigned char *p)
{
	switch (*p) {
	case 'p':
		*p = (unsigned char)'0';
		return;
	case 'q':
		*p = (unsigned char)'1';
		return;
	case 'r':
		*p = (unsigned char)'2';
		return;
	case 's':
		*p = (unsigned char)'3';
		return;
	case 't':
		*p = (unsigned char)'4';
		return;
	case 'u':
		*p = (unsigned char)'5';
		return;
	case 'v':
		*p = (unsigned char)'6';
		return;
	case 'w':
		*p = (unsigned char)'7';
		return;
	case 'x':
		*p = (unsigned char)'8';
		return;
	case 'y':
		*p = (unsigned char)'9';
		return;
	}
}
#endif

static int COB_NOINLINE
cob_get_sign_ebcdic (unsigned char *p)
{
	switch (*p) {
	case '{':
		*p = (unsigned char)'0';
		return 1;
	case 'A':
		*p = (unsigned char)'1';
		return 1;
	case 'B':
		*p = (unsigned char)'2';
		return 1;
	case 'C':
		*p = (unsigned char)'3';
		return 1;
	case 'D':
		*p = (unsigned char)'4';
		return 1;
	case 'E':
		*p = (unsigned char)'5';
		return 1;
	case 'F':
		*p = (unsigned char)'6';
		return 1;
	case 'G':
		*p = (unsigned char)'7';
		return 1;
	case 'H':
		*p = (unsigned char)'8';
		return 1;
	case 'I':
		*p = (unsigned char)'9';
		return 1;
	case '}':
		*p = (unsigned char)'0';
		return -1;
	case 'J':
		*p = (unsigned char)'1';
		return -1;
	case 'K':
		*p = (unsigned char)'2';
		return -1;
	case 'L':
		*p = (unsigned char)'3';
		return -1;
	case 'M':
		*p = (unsigned char)'4';
		return -1;
	case 'N':
		*p = (unsigned char)'5';
		return -1;
	case 'O':
		*p = (unsigned char)'6';
		return -1;
	case 'P':
		*p = (unsigned char)'7';
		return -1;
	case 'Q':
		*p = (unsigned char)'8';
		return -1;
	case 'R':
		*p = (unsigned char)'9';
		return -1;
	default:
		/* What to do here */
		*p = (unsigned char)'0';
		return 1;
	}
/* NOT REACHED */
	return 1;
}

static void COB_NOINLINE
cob_put_sign_ebcdic (unsigned char *p, const int sign)
{
	if (sign < 0) {
		switch (*p) {
		case '0':
			*p = (unsigned char)'}';
			return;
		case '1':
			*p = (unsigned char)'J';
			return;
		case '2':
			*p = (unsigned char)'K';
			return;
		case '3':
			*p = (unsigned char)'L';
			return;
		case '4':
			*p = (unsigned char)'M';
			return;
		case '5':
			*p = (unsigned char)'N';
			return;
		case '6':
			*p = (unsigned char)'O';
			return;
		case '7':
			*p = (unsigned char)'P';
			return;
		case '8':
			*p = (unsigned char)'Q';
			return;
		case '9':
			*p = (unsigned char)'R';
			return;
		default:
			/* What to do here */
			*p = (unsigned char)'}';
			return;
		}
	}
	switch (*p) {
	case '0':
		*p = (unsigned char)'{';
		return;
	case '1':
		*p = (unsigned char)'A';
		return;
	case '2':
		*p = (unsigned char)'B';
		return;
	case '3':
		*p = (unsigned char)'C';
		return;
	case '4':
		*p = (unsigned char)'D';
		return;
	case '5':
		*p = (unsigned char)'E';
		return;
	case '6':
		*p = (unsigned char)'F';
		return;
	case '7':
		*p = (unsigned char)'G';
		return;
	case '8':
		*p = (unsigned char)'H';
		return;
	case '9':
		*p = (unsigned char)'I';
		return;
	default:
		/* What to do here */
		*p = (unsigned char)'{';
		return;
	}
/* NOT REACHED */
}

int
cob_real_get_sign (cob_field *f)
{
	unsigned char	*p;

	switch (COB_FIELD_TYPE (f)) {
	case COB_TYPE_NUMERIC_DISPLAY:
		/* locate sign */
		if (unlikely(COB_FIELD_SIGN_LEADING (f))) {
			p = f->data;
		} else {
			p = f->data + f->size - 1;
		}

		/* get sign */
		if (unlikely(COB_FIELD_SIGN_SEPARATE (f))) {
			return (*p == '+') ? 1 : -1;
		} else {
			if (*p >= '0' && *p <= '9') {
				return 1;
			}
			if (unlikely(cob_current_module->display_sign)) {
				return cob_get_sign_ebcdic (p);
			} else {
#ifdef	COB_EBCDIC_MACHINE
				cob_get_sign_ascii (p);
#else
				GET_SIGN_ASCII (*p);
#endif
				return -1;
			}
		}
	case COB_TYPE_NUMERIC_PACKED:
		p = f->data + COB_FIELD_DIGITS(f) / 2;
		return ((*p & 0x0f) == 0x0d) ? -1 : 1;
	default:
		return 0;
	}
}

void
cob_real_put_sign (cob_field *f, const int sign)
{
	unsigned char	*p;
	int		c;

	switch (COB_FIELD_TYPE (f)) {
	case COB_TYPE_NUMERIC_DISPLAY:
		/* locate sign */
		if (unlikely(COB_FIELD_SIGN_LEADING (f))) {
			p = f->data;
		} else {
			p = f->data + f->size - 1;
		}

		/* put sign */
		if (unlikely(COB_FIELD_SIGN_SEPARATE (f))) {
			c = (sign < 0) ? '-' : '+';
			if (*p != c) {
				*p = c;
			}
		} else if (unlikely(cob_current_module->display_sign)) {
			cob_put_sign_ebcdic (p, sign);
		} else if (sign < 0) {
#ifdef	COB_EBCDIC_MACHINE
			cob_put_sign_ascii (p);
#else
			PUT_SIGN_ASCII (*p);
#endif
		}
		return;
	case COB_TYPE_NUMERIC_PACKED:
		p = f->data + COB_FIELD_DIGITS(f) / 2;
		if (sign < 0) {
			*p = (*p & 0xf0) | 0x0d;
		} else {
			*p = (*p & 0xf0) | 0x0c;
		}
		return;
	default:
		return;
	}
}

void
cob_field_to_string (const cob_field *f, char *s)
{
	int	i;

	memcpy (s, f->data, f->size);
	for (i = (int) f->size - 1; i >= 0; i--) {
		if (s[i] != ' ' && s[i] != 0) {
			break;
		}
	}
	s[i + 1] = '\0';
}

/*
 * Switch
 */

int
cob_get_switch (const int n)
{
	return cob_switch[n];
}

void
cob_set_switch (const int n, const int flag)
{
	cob_switch[n] = flag;
}

/*
 * Comparison
 */

static int
cmpc (const unsigned char *s1, const unsigned char c, const size_t size)
{
	size_t			i;
	int			ret;
	const unsigned char	*s = cob_current_module->collating_sequence;

	if (unlikely(s)) {
		for (i = 0; i < size; i++) {
			if ((ret = s[s1[i]] - s[c]) != 0) {
				return ret;
			}
		}
	} else {
		for (i = 0; i < size; i++) {
			if ((ret = s1[i] - c) != 0) {
				return ret;
			}
		}
	}
	return 0;
}

static int
cmps (const unsigned char *s1, const unsigned char *s2, const size_t size)
{
	size_t			i;
	int			ret;
	const unsigned char	*s = cob_current_module->collating_sequence;

	if (unlikely(s)) {
		for (i = 0; i < size; i++) {
			if ((ret = s[s1[i]] - s[s2[i]]) != 0) {
				return ret;
			}
		}
	} else {
		for (i = 0; i < size; i++) {
			if ((ret = s1[i] - s2[i]) != 0) {
				return ret;
			}
		}
	}
	return 0;
}

static int
cob_cmp_char (cob_field *f, const unsigned char c)
{
	int	sign = cob_get_sign (f);
	int	ret = cmpc (f->data, c, f->size);

	if (COB_FIELD_TYPE (f) != COB_TYPE_NUMERIC_PACKED) {
		cob_put_sign (f, sign);
	}
	return ret;
}

static int
cob_cmp_all (cob_field *f1, cob_field *f2)
{
	int		ret = 0;
	int		sign = cob_get_sign (f1);
	size_t		size = f1->size;
	unsigned char	*data = f1->data;

	while (size >= f2->size) {
		if ((ret = cmps (data, f2->data, f2->size)) != 0) {
			goto end;
		}
		size -= f2->size;
		data += f2->size;
	}
	if (size > 0) {
		ret = cmps (data, f2->data, size);
	}

end:
	if (COB_FIELD_TYPE (f1) != COB_TYPE_NUMERIC_PACKED) {
		cob_put_sign (f1, sign);
	}
	return ret;
}

static int
cob_cmp_alnum (cob_field *f1, cob_field *f2)
{
	int		ret;
	int		sign1 = cob_get_sign (f1);
	int		sign2 = cob_get_sign (f2);
	size_t		min = (f1->size < f2->size) ? f1->size : f2->size;

	/* compare common substring */
	if ((ret = cmps (f1->data, f2->data, min)) != 0) {
		goto end;
	}

	/* compare the rest (if any) with spaces */
	if (f1->size > f2->size) {
		ret = cmpc (f1->data + min, ' ', f1->size - min);
	} else if (f1->size < f2->size) {
		ret = -cmpc (f2->data + min, ' ', f2->size - min);
	}

end:
	if (COB_FIELD_TYPE (f1) != COB_TYPE_NUMERIC_PACKED) {
		cob_put_sign (f1, sign1);
	}
	if (COB_FIELD_TYPE (f2) != COB_TYPE_NUMERIC_PACKED) {
		cob_put_sign (f2, sign2);
	}
	return ret;
}

int
cob_cmp (cob_field *f1, cob_field *f2)
{
	cob_field	temp;
	cob_field_attr	attr;
	unsigned char	buff[48];

	if (COB_FIELD_IS_NUMERIC (f1) && COB_FIELD_IS_NUMERIC (f2)) {
		return cob_numeric_cmp (f1, f2);
	}
	if (COB_FIELD_TYPE (f2) == COB_TYPE_ALPHANUMERIC_ALL) {
		if (f2 == &cob_zero && COB_FIELD_IS_NUMERIC (f1)) {
			return cob_cmp_int (f1, 0);
		} else if (f2->size == 1) {
			return cob_cmp_char (f1, f2->data[0]);
		} else {
			return cob_cmp_all (f1, f2);
		}
	} else if (COB_FIELD_TYPE (f1) == COB_TYPE_ALPHANUMERIC_ALL) {
		if (f1 == &cob_zero && COB_FIELD_IS_NUMERIC (f2)) {
			return -cob_cmp_int (f2, 0);
		} else if (f1->size == 1) {
			return -cob_cmp_char (f2, f1->data[0]);
		} else {
			return -cob_cmp_all (f2, f1);
		}
	} else {
		if (COB_FIELD_IS_NUMERIC (f1)
		    && COB_FIELD_TYPE (f1) != COB_TYPE_NUMERIC_DISPLAY) {
			temp.size = COB_FIELD_DIGITS(f1);
			temp.data = buff;
			temp.attr = &attr;
			attr = *f1->attr;
			attr.type = COB_TYPE_NUMERIC_DISPLAY;
			attr.flags &= ~COB_FLAG_HAVE_SIGN;
			cob_move (f1, &temp);
			f1 = &temp;
		}
		if (COB_FIELD_IS_NUMERIC (f2)
		    && COB_FIELD_TYPE (f2) != COB_TYPE_NUMERIC_DISPLAY) {
			temp.size = COB_FIELD_DIGITS(f2);
			temp.data = buff;
			temp.attr = &attr;
			attr = *f2->attr;
			attr.type = COB_TYPE_NUMERIC_DISPLAY;
			attr.flags &= ~COB_FLAG_HAVE_SIGN;
			cob_move (f2, &temp);
			f2 = &temp;
		}
		return cob_cmp_alnum (f1, f2);
	}
}

/*
 * Class check
 */

int
cob_is_omitted (const cob_field *f)
{
	return f->data == NULL;
}

int
cob_is_numeric (cob_field *f)
{
	size_t		i;
	int		sign;
	size_t		size;
	unsigned char	*data;

	switch (COB_FIELD_TYPE (f)) {
	case COB_TYPE_NUMERIC_BINARY:
	case COB_TYPE_NUMERIC_FLOAT:
	case COB_TYPE_NUMERIC_DOUBLE:
		return 1;
	case COB_TYPE_NUMERIC_PACKED:
		/* check digits */
		for (i = 0; i < f->size - 1; i++) {
			if ((f->data[i] & 0xf0) > 0x90 || (f->data[i] & 0x0f) > 0x09) {
				return 0;
			}
		}
		if ((f->data[i] & 0xf0) > 0x90) {
			return 0;
		}
		/* check sign */
		sign = f->data[i] & 0x0f;
		if (sign == 0x0f) {
			return 1;
		}
		if (COB_FIELD_HAVE_SIGN (f)) {
			if (sign == 0x0c || sign == 0x0d) {
				return 1;
			}
		}
		return 0;
	case COB_TYPE_NUMERIC_DISPLAY:
		size = COB_FIELD_SIZE (f);
		data = COB_FIELD_DATA (f);
		sign = cob_get_sign (f);
		for (i = 0; i < size; i++) {
			if (!isdigit (data[i])) {
				cob_put_sign (f, sign);
				return 0;
			}
		}
		cob_put_sign (f, sign);
		return 1;
	default:
		for (i = 0; i < f->size; i++) {
			if (!isdigit (f->data[i])) {
				return 0;
			}
		}
		return 1;
	}
}

int
cob_is_alpha (const cob_field *f)
{
	size_t	i;

	for (i = 0; i < f->size; i++) {
		if (!isspace (f->data[i]) && !isalpha (f->data[i])) {
			return 0;
		}
	}
	return 1;
}

int
cob_is_upper (const cob_field *f)
{
	size_t	i;

	for (i = 0; i < f->size; i++) {
		if (!isspace (f->data[i]) && !isupper (f->data[i])) {
			return 0;
		}
	}
	return 1;
}

int
cob_is_lower (const cob_field *f)
{
	size_t	i;

	for (i = 0; i < f->size; i++) {
		if (!isspace (f->data[i]) && !islower (f->data[i])) {
			return 0;
		}
	}
	return 1;
}

/*
 * Table sort
 */

static int
sort_cmps (const unsigned char *s1, const unsigned char *s2, const size_t size,
		const unsigned char *col)
{
	size_t			i;
	int			ret;

	if (col) {
		for (i = 0; i < size; i++) {
			if ((ret = col[s1[i]] - col[s2[i]]) != 0) {
				return ret;
			}
		}
	} else {
		for (i = 0; i < size; i++) {
			if ((ret = s1[i] - s2[i]) != 0) {
				return ret;
			}
		}
	}
	return 0;
}

static int
sort_compare (const void *data1, const void *data2)
{
	size_t		i;
	int		cmp;
	cob_field	f1;
	cob_field	f2;

	for (i = 0; i < sort_nkeys; i++) {
		f1 = f2 = *sort_keys[i].field;
		f1.data = (unsigned char *)data1 + sort_keys[i].offset;
		f2.data = (unsigned char *)data2 + sort_keys[i].offset;
		if (COB_FIELD_IS_NUMERIC(&f1)) {
			cmp = cob_numeric_cmp (&f1, &f2);
		} else {
			cmp = sort_cmps (f1.data, f2.data, f1.size, sort_collate);
		}
		if (cmp != 0) {
			return (sort_keys[i].flag == COB_ASCENDING) ? cmp : -cmp;
		}
	}
	return 0;
}

void
cob_table_sort_init (const int nkeys, const unsigned char *collating_sequence)
{
	sort_nkeys = 0;
	sort_keys = cob_malloc (nkeys * sizeof (cob_file_key));
	if (collating_sequence) {
		sort_collate = collating_sequence;
	} else {
		sort_collate = cob_current_module->collating_sequence;
	}
}

void
cob_table_sort_init_key (const int flag, cob_field *field, size_t offset)
{
	sort_keys[sort_nkeys].flag = flag;
	sort_keys[sort_nkeys].field = field;
	sort_keys[sort_nkeys].offset = offset;
	sort_nkeys++;
}

void
cob_table_sort (cob_field *f, const int n)
{
	qsort (f->data, (size_t) n, f->size, sort_compare);
	free (sort_keys);
}

/*
 * Run-time error checking
 */

void
cob_check_numeric (cob_field *f, const char *name)
{
	size_t		i;
	unsigned char	*data;
	char		*p;
	char		buff[COB_SMALL_BUFF];

	if (!cob_is_numeric (f)) {
		p = buff;
		data = f->data;
		for (i = 0; i < f->size; i++) {
			if (isprint (data[i])) {
				*p++ = data[i];
			} else {
				p += sprintf (p, "\\%03o", data[i]);
			}
		}
		*p = '\0';
		cob_runtime_error ("'%s' not numeric: '%s'", name, buff);
		cob_stop_run (1);
	}
}

void
cob_check_odo (const int i, const int min, const int max, const char *name)
{
	/* check the OCCURS DEPENDING ON item */
	if (i < min || max < i) {
		cob_set_exception (COB_EC_BOUND_ODO);
		cob_runtime_error ("OCCURS DEPENDING ON '%s' out of bounds: %d", name, i);
		cob_stop_run (1);
	}
}

void
cob_check_subscript (const int i, const int min, const int max, const char *name)
{
	/* check the subscript */
	if (i < min || max < i) {
		cob_set_exception (COB_EC_BOUND_SUBSCRIPT);
		cob_runtime_error ("Subscript of '%s' out of bounds: %d", name, i);
		cob_stop_run (1);
	}
}

void
cob_check_ref_mod (const int offset, const int length, const int size, const char *name)
{
	/* check the offset */
	if (offset < 1 || offset > size) {
		cob_set_exception (COB_EC_BOUND_REF_MOD);
		cob_runtime_error ("Offset of '%s' out of bounds: %d", name, offset);
		cob_stop_run (1);
	}

	/* check the length */
	if (length < 1 || offset + length - 1 > size) {
		cob_set_exception (COB_EC_BOUND_REF_MOD);
		cob_runtime_error ("Length of '%s' out of bounds: %d", name, length);
		cob_stop_run (1);
	}
}

unsigned char *
cob_external_addr (const char *exname, const int exlength)
{
	static cob_external *basext = NULL;

	cob_external *eptr;

	for (eptr = basext; eptr; eptr = eptr->next) {
		if (!strcmp (exname, eptr->ename)) {
			if (exlength > eptr->esize) {
				cob_runtime_error ("EXTERNAL item '%s' has size > %d",
						   exname, exlength);
				cob_stop_run (1);
			}
			cob_initial_external = 0;
			return (ucharptr)eptr->ext_alloc;
		}
	}
	eptr = (cob_external *) cob_malloc (sizeof (cob_external));
	eptr->next = basext;
	eptr->esize = exlength;
	eptr->ename = cob_malloc (strlen (exname) + 1);
	strcpy (eptr->ename, exname);
	eptr->ext_alloc = cob_malloc ((size_t)exlength);
	basext = eptr;
	cob_initial_external = 1;
	return (ucharptr)eptr->ext_alloc;
}

void *
cob_malloc (const size_t size)
{
	void *mptr;

	mptr = calloc (1, size);
	if (unlikely(!mptr)) {
		cob_runtime_error ("Cannot acquire %d bytes of memory - Aborting", size);
		cob_stop_run (1);
	}
	return mptr;
}

/* Extended ACCEPT/DISPLAY */

void
cob_accept_date (cob_field *f)
{
	time_t	t = time (NULL);
	char	s[8];

	strftime (s, 7, "%y%m%d", localtime (&t));
	cob_memcpy (f, (ucharptr)s, 6);
}

void
cob_accept_date_yyyymmdd (cob_field *f)
{
	time_t	t = time (NULL);
	char	s[12];

	strftime (s, 9, "%Y%m%d", localtime (&t));
	cob_memcpy (f, (ucharptr)s, 8);
}

void
cob_accept_day (cob_field *f)
{
	time_t	t = time (NULL);
	char	s[8];

	strftime (s, 6, "%y%j", localtime (&t));
	cob_memcpy (f, (ucharptr)s, 5);
}

void
cob_accept_day_yyyyddd (cob_field *f)
{
	time_t	t = time (NULL);
	char	s[12];

	strftime (s, 8, "%Y%j", localtime (&t));
	cob_memcpy (f, (ucharptr)s, 7);
}

void
cob_accept_day_of_week (cob_field *f)
{
	time_t	t = time (NULL);
	char	s[4];

#if defined(_MSC_VER)
	sprintf(s, "%d", localtime(&t)->tm_wday + 1);
#else
	strftime (s, 2, "%u", localtime (&t));
#endif
	cob_memcpy (f, (ucharptr)s, 1);
}

void
cob_accept_time (cob_field *f)
{
#ifdef _WIN32
	SYSTEMTIME	syst;
#else
	time_t		t;
#if defined(HAVE_SYS_TIME_H) && defined(HAVE_GETTIMEOFDAY)
	struct timeval	tmv;
	char		buff2[8];
#endif
#endif
	char		s[12];

#ifdef _WIN32
	GetLocalTime (&syst);
	sprintf (s, "%2.2d%2.2d%2.2d%2.2d", syst.wHour, syst.wMinute,
		syst.wSecond, syst.wMilliseconds / 10);
#else
#if defined(HAVE_SYS_TIME_H) && defined(HAVE_GETTIMEOFDAY)
	gettimeofday (&tmv, NULL);
	t = tmv.tv_sec;
#else
	t = time (NULL);
#endif
	strftime (s, 9, "%H%M%S00", localtime (&t));
#if defined(HAVE_SYS_TIME_H) && defined(HAVE_GETTIMEOFDAY)
	sprintf(buff2, "%2.2ld", tmv.tv_usec / 10000);
	memcpy (&s[6], buff2, 2);
#endif
#endif
	cob_memcpy (f, (ucharptr)s, 8);
}

void
cob_accept_command_line (cob_field *f)
{
	size_t	i, size = 0;
	size_t	len;
	char	buff[COB_LARGE_BUFF] = "";

	for (i = 1; i < (size_t)cob_argc; i++) {
		len = strlen (cob_argv[i]);
		if (size + len >= COB_LARGE_BUFF) {
			/* overflow */
			break;
		}
		memcpy (buff + size, cob_argv[i], len);
		size += len;
		buff[size++] = ' ';
	}

	cob_memcpy (f, (ucharptr)buff, size);
}

/*
 * Argument number
 */

void
cob_display_arg_number (cob_field *f)
{
	int		n;
	cob_field_attr	attr;
	cob_field	temp;

	temp.size = 4;
	temp.data = (unsigned char *)&n;
	temp.attr = &attr;
	COB_ATTR_INIT (COB_TYPE_NUMERIC_BINARY, 9, 0, 0, NULL);
	cob_move (f, &temp);
	if (n < 0 || n >= cob_argc) {
		cob_set_exception (COB_EC_IMP_DISPLAY);
		return;
	}
	current_arg = n;
}

void
cob_accept_arg_number (cob_field *f)
{
	int		n = cob_argc - 1;
	cob_field_attr	attr;
	cob_field	temp;

	temp.size = 4;
	temp.data = (unsigned char *)&n;
	temp.attr = &attr;
	COB_ATTR_INIT (COB_TYPE_NUMERIC_BINARY, 9, 0, 0, NULL);
	cob_move (&temp, f);
}

void
cob_accept_arg_value (cob_field *f)
{
	if (current_arg >= cob_argc) {
		cob_set_exception (COB_EC_IMP_ACCEPT);
		return;
	}
	cob_memcpy (f, (ucharptr)cob_argv[current_arg], (int) strlen (cob_argv[current_arg]));
	current_arg++;
}

/*
 * Environment variable
 */

void
cob_set_environment (cob_field *f1, cob_field *f2)
{
	cob_display_environment (f1);
	cob_display_env_value (f2);
}

void
cob_display_environment (cob_field *f)
{
	if (!env) {
		env = cob_malloc (COB_SMALL_BUFF);
	}
	if (f->size > COB_SMALL_BUFF - 1) {
		cob_set_exception (COB_EC_IMP_DISPLAY);
		return;
	}
	cob_field_to_string (f, env);
}

void
cob_display_env_value (cob_field *f)
{
	char	*p;
	size_t	len;
	char	env1[COB_SMALL_BUFF];
	char	env2[COB_SMALL_BUFF];

	if (!env) {
		cob_set_exception (COB_EC_IMP_DISPLAY);
		return;
	}
	if (!*env) {
		cob_set_exception (COB_EC_IMP_DISPLAY);
		return;
	}
	cob_field_to_string (f, env2);
	if (strlen (env) + strlen (env2) + 2 > COB_SMALL_BUFF) {
		cob_set_exception (COB_EC_IMP_DISPLAY);
		return;
	}
	strcpy (env1, env);
	strcat (env1, "=");
	strcat (env1, env2);
	len = (size_t)strlen (env1) + 1;
	p = cob_malloc (len);
	memcpy (p, env1, len);
	if (putenv (p) != 0) {
		cob_set_exception (COB_EC_IMP_DISPLAY);
	}
}

void
cob_accept_environment (cob_field *f)
{
	const char *p = NULL;

	if (env) {
		p = getenv (env);
	}
	if (!p) {
		cob_set_exception (COB_EC_IMP_ACCEPT);
		p = "";
	}
	cob_memcpy (f, (ucharptr)p, (int) strlen (p));
}

void
cob_chain_setup (void *data, const size_t parm, const size_t size)
{
	size_t	len;

	memset (data, ' ', size);
	if (parm <= (size_t)cob_argc - 1) {
		len = strlen (cob_argv[parm]);
		if (len <= size) {
			memcpy (data, cob_argv[parm], len);
		} else {
			memcpy (data, cob_argv[parm], size);
		}
	}
}

void
cob_allocate (unsigned char **dataptr, cob_field *retptr, cob_field *sizefld)
{
	void			*mptr = NULL;
	struct cob_alloc_cache	*cache_ptr;
	int			fsize;

	cob_exception_code = 0;
	fsize = cob_get_int (sizefld);
	if (fsize > 0) {
		cache_ptr = cob_malloc (sizeof (struct cob_alloc_cache));
		mptr = malloc ((size_t)fsize);
		if (!mptr) {
			cob_set_exception (COB_EC_STORAGE_NOT_AVAIL);
			free (cache_ptr);
		} else {
			memset (mptr, 0, (size_t)fsize);
			cache_ptr->cob_pointer = mptr;
			cache_ptr->size = (size_t)fsize;
			cache_ptr->next = cob_alloc_base;
			cob_alloc_base = cache_ptr;
		}
	}
	if (dataptr) {
		*dataptr = (unsigned char *)mptr;
	}
	if (retptr) {
		*(void **)(retptr->data) = mptr;
	}
}

void
cob_free_alloc (unsigned char **ptr1, unsigned char *ptr2)
{
	struct cob_alloc_cache	*cache_ptr;

	cob_exception_code = 0;
	if (ptr1 && *ptr1) {
		for (cache_ptr = cob_alloc_base; cache_ptr; cache_ptr= cache_ptr->next) {
			if (*(void **)ptr1 == cache_ptr->cob_pointer) {
				cache_ptr->cob_pointer = NULL;
				free (*ptr1);
				*ptr1 = NULL;
				return;
			}
		}
		cob_set_exception (COB_EC_STORAGE_NOT_ALLOC);
		return;
	}
	if (ptr2 && *(void **)ptr2) {
		for (cache_ptr = cob_alloc_base; cache_ptr; cache_ptr= cache_ptr->next) {
			if (*(void **)ptr2 == cache_ptr->cob_pointer) {
				cache_ptr->cob_pointer = NULL;
				free (*(void **)ptr2);
				*(void **)ptr2 = NULL;
				return;
			}
		}
		cob_set_exception (COB_EC_STORAGE_NOT_ALLOC);
		return;
	}
}

/* System routines */

int
CBL_EXIT_PROC (unsigned char *x, unsigned char *pptr)
{
	struct exit_handlerlist *hp = NULL;
	struct exit_handlerlist *h = exit_hdlrs;
	int			(**p)(void) = NULL;

	COB_CHK_PARMS (CBL_ERROR_PROC, 2);

	memcpy (&p, &pptr, sizeof (void *));
	if (!p || !*p) {
		return -1;
	}
	/* remove handler anyway */
	while (h != NULL) {
		if (h->proc == *p) {
			if (hp != NULL) {
				hp->next = h->next;
			} else {
				exit_hdlrs = h->next;
			}
			if (hp) {
				free (hp);
			}
			break;
		}
		hp = h;
		h = h->next;
	}
	if   (*x != 0 && *x != 2 && *x != 3) {	/* remove handler */
		return 0;
	}
	h = cob_malloc (sizeof(struct exit_handlerlist));
	h->next = exit_hdlrs;
	h->proc = *p;
	exit_hdlrs = h;
	return 0;
}

int
CBL_ERROR_PROC (unsigned char *x, unsigned char *pptr)
{
	struct handlerlist	*hp = NULL;
	struct handlerlist	*h = hdlrs;
	int			(**p)(char *s) = NULL;

	COB_CHK_PARMS (CBL_ERROR_PROC, 2);

	memcpy (&p, &pptr, sizeof (void *));
	if (!p || !*p) {
		return -1;
	}
	/* remove handler anyway */
	while (h != NULL) {
		if (h->proc == *p) {
			if (hp != NULL) {
				hp->next = h->next;
			} else {
				hdlrs = h->next;
			}
			if (hp) {
				free (hp);
			}
			break;
		}
		hp = h;
		h = h->next;
	}
	if (*x != 0) {	/* remove handler */
		return 0;
	}
	h = cob_malloc (sizeof(struct handlerlist));
	h->next = hdlrs;
	h->proc = *p;
	hdlrs = h;
	return 0;
}

int
SYSTEM (const unsigned char *cmd)
{
	int	i;
	char	buff[COB_MEDIUM_BUFF];

	COB_CHK_PARMS (SYSTEM, 1);
	if (cob_current_module->cob_procedure_parameters[0]) {
		i = (int)cob_current_module->cob_procedure_parameters[0]->size;
		if (i > COB_MEDIUM_BUFF - 1) {
			cob_runtime_error ("Parameter to SYSTEM call is larger than 8192 characters");
			cob_stop_run (1);
		}
		i--;
		for (; i >= 0; i--) {
			if (cmd[i] != ' ' && cmd[i] != 0) {
				break;
			}
		}
		if (i >= 0) {
			memset (buff, 0, sizeof (buff));
			memcpy (buff, cmd, (size_t)(i + 1));
			return (system (buff));
		}
	}
	return 1;
}

int
CBL_AND (unsigned char *data_1, unsigned char *data_2, const int length)
{
	size_t	n;

	if (length <= 0) {
		return 0;
	}
	for (n = 0; n < (size_t)length; n++) {
		data_2[n] &= data_1[n];
	}
	return 0;
}

int
CBL_OR (unsigned char *data_1, unsigned char *data_2, const int length)
{
	size_t	n;

	if (length <= 0) {
		return 0;
	}
	for (n = 0; n < (size_t)length; n++) {
		data_2[n] |= data_1[n];
	}
	return 0;
}

int
CBL_NOR (unsigned char *data_1, unsigned char *data_2, const int length)
{
	size_t	n;

	if (length <= 0) {
		return 0;
	}
	for (n = 0; n < (size_t)length; n++) {
		data_2[n] = ~(data_1[n] | data_2[n]);
	}
	return 0;
}

int
CBL_XOR (unsigned char *data_1, unsigned char *data_2, const int length)
{
	size_t	n;

	if (length <= 0) {
		return 0;
	}
	for (n = 0; n < (size_t)length; n++) {
		data_2[n] ^= data_1[n];
	}
	return 0;
}

int
CBL_IMP (unsigned char *data_1, unsigned char *data_2, const int length)
{
	size_t	n;

	if (length <= 0) {
		return 0;
	}
	for (n = 0; n < (size_t)length; n++) {
		data_2[n] = (~data_1[n]) | data_2[n];
	}
	return 0;
}

int
CBL_NIMP (unsigned char *data_1, unsigned char *data_2, const int length)
{
	size_t	n;

	if (length <= 0) {
		return 0;
	}
	for (n = 0; n < (size_t)length; n++) {
		data_2[n] = data_1[n] & (~data_2[n]);
	}
	return 0;
}

int
CBL_EQ (unsigned char *data_1, unsigned char *data_2, const int length)
{
	size_t	n;

	if (length <= 0) {
		return 0;
	}
	for (n = 0; n < (size_t)length; n++) {
		data_2[n] = ~(data_1[n] ^ data_2[n]);
	}
	return 0;
}

int
CBL_NOT (unsigned char *data_1, const int length)
{
	size_t	n;

	if (length <= 0) {
		return 0;
	}
	for (n = 0; n < (size_t)length; n++) {
		data_1[n] = ~data_1[n];
	}
	return 0;
}

int
CBL_XF4 (unsigned char *data_1, unsigned char *data_2)
{
	size_t	n;

	*data_1 = 0;
	for (n = 0; n < 8; n++) {
		*data_1 |= (data_2[n] & 1) << (7 - n);
	}
	return 0;
}

int
CBL_XF5 (unsigned char *data_1, unsigned char *data_2)
{
	size_t	n;

	for (n = 0; n < 8; n++) {
		data_2[n] = (*data_1 & (1 << (7 - n))) ? 1 : 0;
	}
	return 0;
}

int
CBL_TOUPPER (unsigned char *data, const int length)
{
	size_t	n;

	COB_CHK_PARMS (CBL_TOUPPER, 2);
	if (length > 0) {
		for (n = 0; n < (size_t)length; n++) {
			if (islower (data[n])) {
				data[n] = toupper (data[n]);
			}
		}
	}
	return 0;
}

int
CBL_TOLOWER (unsigned char *data, const int length)
{
	size_t	n;

	COB_CHK_PARMS (CBL_TOLOWER, 2);
	if (length > 0) {
		for (n = 0; n < (size_t)length; n++) {
			if (isupper (data[n])) {
				data[n] = tolower (data[n]);
			}
		}
	}
	return 0;
}

int
cob_return_args (unsigned char *data)
{
	COB_CHK_PARMS (C$NARG, 1);
	if (cob_current_module->cob_procedure_parameters[0]) {
		cob_set_int (cob_current_module->cob_procedure_parameters[0], cob_save_call_params);
	}
	return 0;
}

int
cob_parameter_size (unsigned char *data)
{
	int	n;

	COB_CHK_PARMS (C$PARAMSIZE, 1);
	if (cob_current_module->cob_procedure_parameters[0]) {
		n = cob_get_int (cob_current_module->cob_procedure_parameters[0]);
		if (n > 0 && n <= cob_save_call_params) {
			n--;
			if (cob_current_module->next &&
			    cob_current_module->next->cob_procedure_parameters[n]) {
				return cob_current_module->next->cob_procedure_parameters[n]->size;
			}
		}
	}
	return 0;
}

int
cob_acuw_sleep (unsigned char *data)
{
	int	n;

	COB_CHK_PARMS (C$SLEEP, 1);
	if (cob_current_module->cob_procedure_parameters[0]) {
		n = cob_get_int (cob_current_module->cob_procedure_parameters[0]);
		if (n > 0 && n < 3600*24*7) {
#ifdef	_WIN32
			Sleep (n*1000);
#else
			sleep ((unsigned int)n);
#endif
		}
	}
	return 0;
}

int
cob_acuw_justify (unsigned char *data, ...)
{
	unsigned char	*direction;
	size_t		datalen;
	int		n;
	int		shifting = 0;
	size_t		left = 0;
	size_t		right = 0;
	size_t		movelen;
	size_t		centrelen;
	va_list		args;

	COB_CHK_PARMS (C$JUSTIFY, 1);
	datalen = cob_current_module->cob_procedure_parameters[0]->size;
	if (datalen < 2) {
		return 0;
	}
	if (data[0] != ' ' && data[datalen - 1] != ' ') {
		return 0;
	}
	for (n = 0; n < (int)datalen; n++, left++) {
		if (data[n] != ' ') {
			break;
		}
	}
	if (n == (int)datalen) {
		return 0;
	}
	left = n;
	for (n = (int)datalen - 1; n >= 0; n--, right++) {
		if (data[n] != ' ') {
			break;
		}
	}
	movelen = datalen - left - right;
	if (cob_call_params > 1) {
		va_start (args, data);
		direction = va_arg (args, unsigned char *);
		va_end (args);
		if (*direction == 'L') {
			shifting = 1;
		} else if (*direction == 'C') {
			shifting = 2;
		}
	}
	switch (shifting) {
	case 1:
		memmove (data, &data[left], movelen);
		memset (&data[movelen], ' ', datalen - movelen);
		break;
	case 2:
		centrelen = (left + right) / 2;
		memmove (&data[centrelen], &data[left], movelen);
		memset (data, ' ', centrelen);
		if ((left + right) % 2) {
			memset (&data[centrelen + movelen], ' ', centrelen + 1);
		} else {
			memset (&data[centrelen + movelen], ' ', centrelen);
		}
		break;
	default:
		memmove (&data[left + right], &data[left], movelen);
		memset (data, ' ', datalen - movelen);
		break;
	}
	return 0;
}
