## Copyright (C) 2004  Dragan Tubic
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {} vtk_line3(@var{x}, @var{y}, @var{z}, [@var{fmt} | @var{prop},@var{val}])
##
## Plots 3D line segments between the specified points.  The data point
## coordinates must be given as 3 equal size @var{x}, @var{y}, and
## @var{z} matrices.  The optional @var{fmt} takes the form of a "line
## specification"; e.g. "r" would cause the lines to be plotted in color
## red.  The optional @var{prop},@var{val} is a property,value pair
## argument. Valid properties are Color, Radius, and Opacity.
##
## Example: Knot (torus)
## @example
## nmeridian  = 6; nlongitude = 11;
## phi = 0:pi/1000:2*pi;      
## mu = phi * nmeridian;
## x = cos(mu) .* (1 + cos(nlongitude*mu/nmeridian) / 2.0);
## y = sin(mu) .* (1 + cos(nlongitude*mu/nmeridian) / 2.0);
## z = sin(nlongitude*mu/nmeridian) / 2.0;
## vtk_line3(x',y',z','Radius',0.05);
## @end example
## 
## @end deftypefn
## @seealso{vtk_plot3,vtk_arrows3,vtk_get_line_spec}

## Author: Dragan Tubic


function vtk_line3(varargin)
  
  valid_props = ";Color;Radius;Opacity;";
  [no_numerical_params, first_prop_index, line_spec_index] = vtk_parse_params(valid_props, varargin{:});
  
  if ( no_numerical_params < 3 )
    error("Syntax is vtk_plot3(x,y,z)");
  end

  x = nth (varargin,1);
  y = nth (varargin,2);
  z = nth (varargin,3);
  [no_points no_lines] = size(x);

  
  ## set default color
  color = [0 0 1];   # same as default in vtk_get_line_spec function

  ## use "line specs", if provided
  if ( line_spec_index > 0 )
    line_spec = nth (varargin,line_spec_index);
    [color, marker_type, line_style] = vtk_get_line_spec( line_spec );
  end
  ## use "property specs", if provided
  if ( first_prop_index > 0 )
    properties = struct(varargin{first_prop_index:length(varargin)});
    if ( isfield(properties,"Color") )
      color = properties.Color;
    end
  end
  properties.Color = color;

  
  vtk_init;   
  f = vtk_figure(0);
  
  for i = 1:no_lines
    coords = vtkFloatArray; 
    coords.SetNumberOfTuples( no_points );
    coords.SetNumberOfComponents(3);
    pts = [x(:,i) y(:,i) z(:,i)]';
    coords.SetArray( pts(:), 3*no_points, 0 );
    
    points = vtkPoints;
    points.SetData(coords);	
    
    poly_line = vtkPolyLine;
    ptids = poly_line.GetPointIds();
    ptids.SetNumberOfIds(no_points);
    for	i = 0:no_points-1
      ptids.SetId( i, i );
    end
    %% vtkIdTypeArray
    
    poly_data = vtkPolyData();
    poly_data.SetPoints(points);
    
    line = vtkCellArray();
    line.InsertNextCell(no_points);
    for	i = 0:no_points-1
      line.InsertCellPoint(i);
    end
    
    poly_data.SetLines(line);
    
    tubes = vtkTubeFilter();
    tubes.SetInput( poly_data );
    tubes.SetRadius(0.01);
    if struct_contains(properties,"Radius")
      r = properties.Radius;
      tubes.SetRadius(r);
    end
    tubes.SetNumberOfSides(6);
    
    poly_mapper = vtkDataSetMapper;
    poly_mapper.SetInput( tubes.GetOutput() );
    
    poly_actor = vtkActor;
    poly_actor.SetMapper( poly_mapper );
    if struct_contains(properties,"Color")
      c = properties.Color;
      poly_actor.GetProperty().SetDiffuseColor(c(1),c(2),c(3));
    else
      poly_actor.GetProperty().SetDiffuseColor(1,1,1);
    end
    if struct_contains(properties,"Opacity")
      poly_actor.GetProperty().SetOpacity(properties.Opacity);
    else
      poly_actor.GetProperty().SetOpacity(1);
    end
    
    f.renderer.AddActor(poly_actor);
  end
  
  vtk_update(f);

endfunction
