/* $Id: timeouts.c,v 1.28 2004/09/23 02:02:06 graziano Exp $ */

#include "config_portability.h"

#include <stdlib.h>
#include <string.h>
#ifdef HAVE_MATH_H
#include <math.h>
#endif

#include "timeouts.h"
#include "diagnostic.h"
#include "osutil.h"
#include "nws_forecast_api.h"

/* the default timeouts */
#define HOW_MAY_DEFAULTS (3)
#define DEFAULT_MAX_TIMEOUT (120.0)
#define DEFAULT_MIN_TIMEOUT (5.0)
#define DEFAULT_TIMEDOUT_TIMEOUT (15.0)

/* the payload per tranmission */
#define DEFAULT_PAYLOAD (1024)

/* we have this number of entries */
#define HOW_MANY_ENTRIES (512)

/* keep track of the measurements we got */
typedef struct {
	IPAddress addr;
	FORECASTAPI_ForecastState *state[TIMEOUT_TYPES_NUMBER];
} timeOutStruct;

static void *lock = NULL;			/* local mutex */

/* we have few types to keep track of */
static timeOutStruct timeOuts[HOW_MANY_ENTRIES]; 

/* keep track of the current limits on the timeouts */
static double to_limits[TIMEOUT_TYPES_NUMBER*HOW_MAY_DEFAULTS];

/* helper to work with modulo HOW_MANY_ENTRIES */
#define MODPLUS(a,b,m) (((a) + (b)) % (m))
#define MODMINUS(a,b,m) (((a) - (b) + (m)) % (m))

/*
 * Initialize the Hash structure 
 */
static void 
InitHashStructure() {
	unsigned int i,j;
	static int initialized = 0;

	/* let's do it only once */
	GetNWSLock(&lock);
	if (initialized != 0) {
		ReleaseNWSLock(&lock);
		return;
	}
	initialized = 1;
	ReleaseNWSLock(&lock);

	for (i=0; i < HOW_MANY_ENTRIES; i++) {
		timeOuts[i].addr.addr = 0;
		for (j=0; j < TIMEOUT_TYPES_NUMBER; j++) {
			timeOuts[i].state[j] = NULL;
		}
	}

	for (j=0; j < TIMEOUT_TYPES_NUMBER; j++) {
		/* set defaults to sane state */
		SetDefaultTimeout((timeoutTypes)j, 
				DEFAULT_MIN_TIMEOUT, 
				DEFAULT_MAX_TIMEOUT, 
				DEFAULT_TIMEDOUT_TIMEOUT);
	}
}

/* we use a simple hash table structure to speed up the access to the
 * timeout structures. If #addr# is not in the table it is added and if
 * the table is full, space is made for the addr.  Only the forecaster
 * for #type# will be used/initialized.
 * Returns	# the index at which addr is been found
 */
static int 
HashIndex(	IPAddress addr,
		timeoutTypes type) {
	unsigned int i, end, toSet;

	/* sanity check: kinda of a waste since we already done it */
	if (type < 0 || type > USER3) {
		WARN("HashIndex: type out of range\n");
		return 0;
	}

	/* initialize the structure */
	InitHashStructure();

	i = addr.addr % HOW_MANY_ENTRIES;
	end = MODMINUS(i, 1, HOW_MANY_ENTRIES);
	toSet = 0;

	GetNWSLock(&lock);
	for (; i != end; i = MODPLUS(i, 1, HOW_MANY_ENTRIES)) {
		if ((timeOuts[i].addr.addr == addr.addr) || (timeOuts[i].addr.addr == 0)) {
			/* either we found it, or emtpy slot: is good */
			break;
		}
			
	}
	if (i == end) {
		/* table is full: emtpying one slot */
		i = addr.addr % HOW_MANY_ENTRIES;
		timeOuts[i].addr.addr = 0;
	}

	/* if this is the first time we have the item, get it to a sane
	 * state */
	if (timeOuts[i].addr.addr == 0) {
		timeOuts[i].addr.addr = addr.addr;
		/* free the old states */
		for (end=0; end < TIMEOUT_TYPES_NUMBER; end++) {
			if (timeOuts[i].state[end] != NULL) {
				FORECASTAPI_FreeForecastState(&timeOuts[i].state[end]);
			}
		}
	}

	/* initialize the new state */
	if (timeOuts[i].state[type] == NULL) {
		timeOuts[i].state[type] = FORECASTAPI_NewForecastState();
		if (timeOuts[i].state[type] == NULL) {
			ERROR("out of memory\n");
			return 0;
		}

		toSet = 1;
	}
	ReleaseNWSLock(&lock);

	/* if the forecaster has been created now, we need to feed it
	 * with a default value. Using timedOut with a size of 0 triggers
	 * the dafult value we want. */
	if (toSet == 1) {
		SetTimeOut(type, addr, 0, 0, 1);
	}

	return i;
}

void
SetDefaultTimeout(	timeoutTypes type, 
			double min, 
			double max,
			double timedOut) {
	/* sanity check */
	if (type < RECV || type > USER3) {
		WARN("SetDefaultTimeout: unknown type\n");
		return;
	}
	if (min < 0) {
		WARN("SetDefaultTimeout: unresonable timeout: using default\n");
		min = DEFAULT_MIN_TIMEOUT;
	}
	if (max < min) {
		WARN("SetDefaultTimeout: unresonable timeout: using default\n");
		max = DEFAULT_MAX_TIMEOUT;
	}
	if (timedOut > max || timedOut < min) {
		WARN("SetDefaultTimeout: unresonable timeout: using default\n");
		timedOut = DEFAULT_MAX_TIMEOUT;
	}

	GetNWSLock(&lock);
	to_limits[type * HOW_MAY_DEFAULTS] = min;
	to_limits[type * HOW_MAY_DEFAULTS + 1] = max;
	to_limits[type * HOW_MAY_DEFAULTS + 2] = timedOut;
	ReleaseNWSLock(&lock);
}

void 
GetDefaultTimeout(	timeoutTypes type, 
			double *min, 
			double *max,
			double *timedOut) {
	/* sanity check */
	if (type < RECV || type > USER3) {
		WARN("SetDefaultTimeout: unknown type\n");
		return;
	}
	GetNWSLock(&lock);
	*min = to_limits[type * HOW_MAY_DEFAULTS];
	*max = to_limits[type * HOW_MAY_DEFAULTS + 1];
	*timedOut = to_limits[type * HOW_MAY_DEFAULTS + 2];
	ReleaseNWSLock(&lock);
}


double 
GetTimeOut(	timeoutTypes type, 
		IPAddress addr, 
		long size) {
	unsigned int i = -1;
	double ret;
	FORECASTAPI_ForecastCollection forecast;

	/* sanity check */
	if (type < 0 || type > USER3) {
		WARN("GetTimeOut: type out of range\n");
		return 0;
	}

	/* if addr is 0 (for example if it's a pipe) we pick the minimum
	 * timeout */
	if (addr.addr == 0) {
		return to_limits[type * HOW_MAY_DEFAULTS];
	}

	i = HashIndex(addr, type);

	/* let's get a forecast */
	GetNWSLock(&lock);
	if (!FORECASTAPI_ComputeForecast(timeOuts[i].state[type], &forecast)) {
		/* if we have problem let's pick the defaults */
		forecast.forecasts[FORECASTAPI_MSE_FORECAST].forecast = to_limits[type * HOW_MAY_DEFAULTS + 2];
		WARN("GetTimeOut: forecaster returned error!\n");
	}
	ReleaseNWSLock(&lock);

	/* let's get 2 standard deviations (if we have sqrt) */
#ifdef HAVE_SQRT
	ret = forecast.forecasts[FORECASTAPI_MSE_FORECAST].forecast + 2 * sqrt(forecast.forecasts[FORECASTAPI_MSE_FORECAST].error);
#else
	ret = forecast.forecasts[FORECASTAPI_MSE_FORECAST].forecast + 2 * forecast.forecasts[FORECASTAPI_MSE_FORECAST].error;
#endif
	/* if we are at zero, let's bumped to 1 */
	if (ret == 0) {
		ret = 1;
		WARN("GetTimeOut: forecast is 0!\n");
	}

	/* adjust for the size of the packet */
	if (size > 0) {
		if (size > DEFAULT_PAYLOAD) {
			ret = ret * size / DEFAULT_PAYLOAD;
		}
	}

	if (ret > to_limits[type * HOW_MAY_DEFAULTS + 1]) {
		ret = to_limits[type * HOW_MAY_DEFAULTS + 1];
	} else if (ret < to_limits[type * HOW_MAY_DEFAULTS]) {
		ret = to_limits[type * HOW_MAY_DEFAULTS];
	}

	return ret;
}

void
SetTimeOut(	timeoutTypes type, 
		IPAddress addr, 
		double duration, 
		long size, 
		int timedOut) {
	unsigned int i;
	FORECASTAPI_Measurement m;

	/* sanity check */
	if (type < 0 || type > USER3) {
		WARN("SetTimeOut: type out of range\n");
		return;
	}
	if (duration < 0) {
		WARN("SetTimeOut: duration negative?\n");
		return;
	}

	/* if addr is 0 (for example if it's a pipe) we return */
	if (addr.addr == 0) {
		return;
	}

	i = HashIndex(addr, type);

	m.timeStamp = CurrentTime();
	/* if size > 0 we consider it to have the same cost up
	 * until we pass the MTU. We should ask for the MTU.. */
	if (size > DEFAULT_PAYLOAD) {
		m.measurement = duration / size * DEFAULT_PAYLOAD;
	} else {
		m.measurement = duration;
	}

	/* adjustments if we timed out before */
	if (timedOut) {
		if (size == 0) {
			/* this is a special case: we timed out and
			 * nothing has been sent (triggered by size == 0);
			 * we don't want to have a long timeout in this
			 * case, since we cannot talk to the guy */
			m.measurement = to_limits[type * HOW_MAY_DEFAULTS + 2]/DEFAULT_PAYLOAD;
		} else {
			/* we timed out: let's add sometime to the
			 * current measurment */
			m.measurement += 10;
		}
	}
	GetNWSLock(&lock);
	FORECASTAPI_UpdateForecast(timeOuts[i].state[type], &m, 1);
	ReleaseNWSLock(&lock);
}
