/* NVTV actions -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This is open software protected by the GPL. See GPL.txt for details.
 *
 * Header: Intermediate layer. Contains tv actions, X actions and card
 *   access.
 */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/extensions/xf86vmode.h>

#include "backend.h"
#include "debug.h"
#include "xfree.h"
#include "nv_type.h"
#include "nv_tv.h"

#include "actions.h"

#if 0 /* FIXME */

/* -------- State -------- */

/* FIXME remove */

NVCrtState *monitor_state;  /* Saved monitor state */

/* -------- Device independent routines -------- */

void tv_set (NVPtr pNv)
{
  if (!pNv || !pNv->TvDev || !pNv->TvFunc || nvTvState == NV_TV_OFF) return;
  NVSetTvRegs (pNv);
}

void tv_on (NVPtr pNv)
{
  if (!pNv || !pNv->TvDev || !pNv->TvFunc) return;
  if (nvTvState != NV_TV_ON) {
    if (!monitor_state) 
      monitor_state = (NVCrtState *) malloc (sizeof (NVCrtState));
    NVSaveCrt (pNv, monitor_state);
  }
  NVSetTvState (pNv, NV_TV_ON);
}

void tv_off (NVPtr pNv, Bool have_crt)
{
  NVTvState state;

  if (!pNv || !pNv->TvDev || !pNv->TvFunc) return;
  state = nvTvState;
  NVSetTvState (pNv, NV_TV_OFF);
  /* We need to get valid CRTC settings. First, try saved ones. Then,
     try those supplied in pNv, if available. */
  if (state != NV_TV_OFF) {
    if (monitor_state) { 
      NVRestoreCrt (pNv, monitor_state);
    } else if (have_crt) {
      NVSetCrtRegs (pNv, &pNv->CrtRegs);
    }
  }
}

int tv_status (NVPtr pNv)
{
  if (!pNv || !pNv->TvDev || !pNv->TvFunc) return -1;
  return NVGetTvStatus (pNv, 0);
}

char *tv_version (NVPtr pNv) 
{
  if (!pNv || !pNv->TvDev || !pNv->TvFunc) return "<No device>";
  return NVGetTvVersion (pNv);
}

void tv_exit (NVPtr pNv)
{
  if (!pNv || !pNv->TvDev || !pNv->TvFunc) return;

  if (nvTvState == NV_TV_BARS) NVSetTvState (pNv, NV_TV_OFF);
}

#endif

/* -------- Probe action -------- */

#if 0 /* FIXME */

void probe_bt (I2CDevPtr dev)
{
  int res;
  char *chip;

  printf ("  Device %1s:%02X = ", I2C_ID(dev));
  res = NVReadBt (dev, 0);
  if (res < 0) {
    printf ("\n  error reading status\n"); return;
  }
  switch (res & 0xe0) {
    case 0x00 : chip = "868"; break;
    case 0x20 : chip = "869"; break;
    case 0x40 : chip = "870"; break;
    case 0x60 : chip = "871"; break;
    default : chip = "???"; break;
  }
  printf ("Brooktree %s id=%02X\n", chip, res);
}

void probe_ch (I2CDevPtr dev)
{
  int res;
  int i;
  char *chip;

  printf ("  Device %1s:%02X = ", I2C_ID(dev));
  res = NVReadCh (dev, 0x25);
#if 0
  if (res < 0) {
    printf ("\n  error reading version register\n"); return;
  }
#endif
  switch (res) {
    case 0x32: chip = "7004C"; break; /* 00110010 */
    case 0x3a: chip = "7005C"; break; /* 00111010 */
    case 0x2a: chip = "7006C"; break; /* 00101010 */
    case 0x50: chip = "7007A"; break; /* 01010000 */
    case 0x40: chip = "7008A"; break; /* 01000000 */
    default:   chip = "700x"; break;
  }
  printf ("Chrontel %s id=%02X\n", chip, res);
  for (i = 0x00; i <= 0x3F; i++) {
    if ((i & 0xf) == 0x0) printf ("    %02X:", i);
    if ((i & 0xf) == 0x8) printf ("  ");
    res = NVReadCh (dev, i);
    if (res < 0) printf (" --"); else printf (" %02X", res);
    if ((i & 0xf) == 0xf) printf ("\n");
  }
}

void probe_ph (I2CDevPtr dev)
{
  int id, ver, res;
  int i;
  char *chip;

  printf ("  Device %1s:%02X = ", I2C_ID(dev));
  id  = NVReadPh (dev, 0x1c);
  ver = NVReadPh (dev, 0x00);
#if 0
  if (id < 0) {
    printf ("\n  error reading chip id register\n"); return;
  }
#endif
  switch (id) {
    case 0x02: chip = "SAA7102/7108"; break; 
    case 0x03: chip = "SAA7103/7109"; break; 
    default:   chip = "???"; break;
  }
  printf ("Philips %s id=%02X ver=%i\n", chip, id, (ver>>5) & 0x7);
  for (i = 0x00; i <= 0xfd; i++) {
    if ((i & 0xf) == 0x0) printf ("    %02X:", i);
    if ((i & 0xf) == 0x8) printf ("  ");
    res = NVReadCh (dev, i);
    if (res < 0) printf (" --"); else printf (" %02X", res);
    if ((i & 0xf) == 0xf) printf ("\n");
  }
}

void probe_crt (NVPtr pNv)
{
  NVCrtState state;
  int i;
 
  printf ("  CRT registers:\n");
  for (i = 0x00; i <= 0x33; i++) state.regs[i] = 0x00;
  NVSaveCrt (pNv, &state);
  for (i = 0x00; i <= 0x33; i++) {
    if ((i & 0xf) == 0x0) printf ("    %02X:", i);
    if ((i & 0xf) == 0x8) printf ("  ");
    printf (" %02X", state.regs[i]);
    if ((i & 0xf) == 0xf) printf ("\n");
  }
  printf ("\n");
}

void probe_system (CardPtr card_list)
{
  CardPtr card;
  I2CChainPtr chain;

  for (card = card_list; card; card = card->next) {
    printf ("%s (%04X) io=0x%08lX\n", card->name, card->chip, card->reg_base);
    /* FIXME */
    if (card_pNv) destroy_nv (card_pNv);
    card_pNv = init_nv (card);
    card_pScrn->driverPrivate = card_pNv;
    if (!card_pNv) {
      printf ("  error initializing pNv\n"); 
      continue;
    }
    printf ("  MC_boot=%08X FB_boot=%08X FB_conf=%08X EXT_boot=%08X\n", 
      card_pNv->riva.PMC[0x000/4], card_pNv->riva.PFB[0x000/4], 
      card_pNv->riva.PFB[0x200/4], card_pNv->riva.PEXTDEV[0x000/4]);
    printf ("  TV setup=%08X vblank=%08X-%08X hblank=%08X-%08X\n", 
      card_pNv->riva.PRAMDAC[0x700/4], card_pNv->riva.PRAMDAC[0x704/4], 
      card_pNv->riva.PRAMDAC[0x708/4], card_pNv->riva.PRAMDAC[0x70C/4], 
      card_pNv->riva.PRAMDAC[0x710/4]);
    printf ("  PLL coeff=%08X link=%08X\n", 
      card_pNv->riva.PRAMDAC[0x50c/4], card_pNv->riva.PRAMDAC[0x880/4]);
    if (!NVTvBusInit (card_pScrn)) {
      printf ("  error initializing I2C\n"); 
      continue;
    }
    probe_crt (card_pNv);
    printf ("  I2C Devices:");
    NVProbeAllDevices (card_pNv);
    NVCreateChain (card_pNv);
    for (chain = card_pNv->TvChain; chain; chain = chain->next) {
      printf (" %1s:%02X", I2C_ID(chain->dev));
    }
    printf ("\n");
    for (chain = card_pNv->TvChain; chain; chain = chain->next) {
      switch (chain->dev->SlaveAddr) {
        case 0x88: /* Might be Philips or Brooktree */
	  if (NVReadPh (chain->dev, 0x1c) >= 0) {
	    /* Register 0x1c doesn't exists in Brooktree/Conexant */
	    probe_ph (chain->dev);
	    break;
	  } /* else fall through, it's a Brooktree */
        case 0x8A:
	  probe_bt (chain->dev);
	  break;
	case 0xEA: 
        case 0xEC:
	  probe_ch (chain->dev);
	  break;
      }
    }
  }
}

#endif

/* -------- Print actions -------- */

void print_crt_regs (NVCrtRegs *m)
{
  printf ("*** CRT Registers\n");
  printf ("  HDisplay    : %i,\n", m->HDisplay);
  printf ("  HSyncStart  : %i,\n", m->HSyncStart);   
  printf ("  HSyncEnd    : %i,\n", m->HSyncEnd);     
  printf ("  HTotal      : %i,\n", m->HTotal);       
  printf ("  VDisplay    : %i,\n", m->VDisplay);     
  printf ("  VSyncStart  : %i,\n", m->VSyncStart);   
  printf ("  VSyncEnd    : %i,\n", m->VSyncEnd);     
  printf ("  VTotal      : %i,\n", m->VTotal);       
}

void print_bt_regs (NVBtRegs *b)
{
  printf ("*** Brooktree Registers\n");
  printf ("  hsynoffset   : %i,\n", b->hsynoffset);
  printf ("  vsynoffset   : %i,\n", b->vsynoffset);
  printf ("  hsynwidth    : %i,\n", b->hsynwidth);
  printf ("  vsynwidth    : %i,\n", b->vsynwidth);
  printf ("  h_clko       : %i,\n", b->h_clko);
  printf ("  h_active     : %i,\n", b->h_active);
  printf ("  hsync_width  : %i,\n", b->hsync_width);
  printf ("  hburst_begin : %i,\n", b->hburst_begin);
  printf ("  hburst_end   : %i,\n", b->hburst_end);
  printf ("  h_blanko     : %i,\n", b->h_blanko); 
  printf ("  v_blanko     : %i,\n", b->v_blanko); 
  printf ("  v_activeo    : %i,\n", b->v_activeo);
  printf ("  h_fract      : %i,\n", b->h_fract);  
  printf ("  h_clki       : %i,\n", b->h_clki);   
  printf ("  h_blanki     : %i,\n", b->h_blanki); 
  printf ("  v_blank_dly  : %i,\n", b->v_blank_dly);
  printf ("  v_linesi     : %i,\n", b->v_linesi); 
  printf ("  v_blanki     : %i,\n", b->v_blanki); 
  printf ("  v_activei    : %i,\n", b->v_activei);
  printf ("  v_scale      : %i,\n", b->v_scale);  
  printf ("  pll_fract    : %i,\n", b->pll_fract);
  printf ("  pll_int      : %i,\n", b->pll_int);  
  printf ("  sync_amp     : %i,\n", b->sync_amp); 
  printf ("  bst_amp      : %i,\n", b->bst_amp);  
  printf ("  mcr          : %i,\n", b->mcr);      
  printf ("  mcb          : %i,\n", b->mcb);      
  printf ("  my           : %i,\n", b->my);       
  printf ("  msc          : %li,\n", b->msc); 
}

void print_ch_regs (NVChRegs *r)
{
  printf ("*** Chrontel Registers\n");
  printf ("  dmr_ir    : %i,\n", r->dmr_ir);
  printf ("  dmr_vs    : %i,\n", r->dmr_vs);
  printf ("  dmr_sr    : %i,\n", r->dmr_sr);
  printf ("  sav       : %i,\n", r->sav);
  printf ("  hpr       : %i,\n", r->hpr);
  printf ("  vpr       : %i,\n", r->vpr);
  printf ("  pll_m     : %i,\n", r->pll_m);
  printf ("  pll_n     : %i,\n", r->pll_n);
  printf ("  pllcap    : %i,\n", r->pllcap);
  printf ("  dacg      : %i,\n", r->dacg);
  printf ("  fsci      : %li,\n", r->fsci);
  printf ("  ffr_fc    : %i,\n", r->ffr_fc);
  printf ("  ffr_fy    : %i,\n", r->ffr_fy);
  printf ("  ffr_ft    : %i,\n", r->ffr_ft);
  printf ("  blr       : %i,\n", r->blr);
  printf ("  ce        : %i,\n", r->ce);
  printf ("  vbw_flff  : %i,\n", r->vbw_flff);  
  printf ("  vbw_cvbw  : %i,\n", r->vbw_cvbw);  
  printf ("  vbw_cbw   : %i,\n", r->vbw_cbw);   
  printf ("  vbw_ypeak : %i,\n", r->vbw_ypeak); 
  printf ("  vbw_ysv   : %i,\n", r->vbw_ysv);   
  printf ("  vbw_ycv   : %i,\n", r->vbw_ycv);    
}

void print_tv_regs (NVTvRegs *c, NVTvChip type)
{
  switch (type) 
  {
    case NV_BROOKTREE:
    case NV_CONEXANT:
      print_bt_regs (&c->bt); 
      break;
    case NV_CHRONTEL:
      print_ch_regs (&c->ch); 
      break;
    default:
      break;
  }
}

/* -------- X Actions -------- */

/* Minimum vidmode extension version required */
#define MINMAJOR 0
#define MINMINOR 5

inline int sqr (int arg)
{
  return arg * arg;
}

Bool has_vidmode (Display *display)
{
  static int vidmode = 0;

  int MajorVersion, MinorVersion;
  int EventBase, ErrorBase;

  while (vidmode == 0) {
    if (!XF86VidModeQueryVersion (display, &MajorVersion, &MinorVersion))
    {
      fprintf(stderr, "Unable to query video extension version\n");
      vidmode = -1;
      break;
    }
    if (!XF86VidModeQueryExtension(display, &EventBase, &ErrorBase)) {
      fprintf(stderr, "Unable to query video extension information\n");
      vidmode = -1;
      break;
    }
    /* Fail if the extension version in the server is too old */
    if (MajorVersion < MINMAJOR || 
	(MajorVersion == MINMAJOR && MinorVersion < MINMINOR)) {
      fprintf(stderr,
	      "Xserver is running an old XFree86-VidModeExtension version"
	      " (%d.%d)\n", MajorVersion, MinorVersion);
      fprintf(stderr, "Minimum required version is %d.%d\n",
	      MINMAJOR, MINMINOR);
      vidmode = -1;
      break;
    }
    vidmode = 1;
    break;
  }
  if (vidmode > 0) return TRUE; else return FALSE;
}

/* get all modelines only once, so we don't have to free them */

int vidmodelines (Display *display, int screen, XF86VidModeModeInfo ***info)
{
  static XF86VidModeModeInfo **modesinfo = NULL; 
  static int modecount = 0;

  if (!modesinfo) {
    XF86VidModeGetAllModeLines (display, screen, &modecount, &modesinfo);
  }
  *info = modesinfo;
  return modecount;
}

void switch_vidmode (Display *display, int screen, int res_x, int res_y)
{
  XF86VidModeModeInfo **modesinfo;
  int modecount;
  int i;
  int dist;
  int best_dist = -1;
  int best_index = -1;

  DPRINTF ("vidmode %i %i\n", res_x, res_y);
  if (!has_vidmode (display)) return;
  modecount = vidmodelines (display, screen, &modesinfo);
  for (i = 0; i < modecount; i++) {
    dist = sqr(modesinfo [i]->hdisplay - res_x) + 
           sqr(modesinfo [i]->vdisplay - res_y);
    if (modesinfo [i]->hdisplay < res_x ||
        modesinfo [i]->vdisplay < res_y) {
      dist = -1;
    }
    if (dist >= 0 && (dist < best_dist || best_dist == -1)) {
      best_dist = dist;
      best_index = i;
    }
  }
  if (best_dist >= 0) {
    DPRINTF ("vidmode switch %i\n", best_index);
    XF86VidModeSwitchToMode (display, screen, modesinfo [best_index]);
  }
}

/*
 * Find X video mode with given resolution
 */

Bool find_vidmode (Display *display, int screen, int res_x, int res_y, 
		   NVCrtRegs *crt)
{
  XF86VidModeModeInfo **modesinfo;
  int modecount;
  int i;

  if (!has_vidmode (display)) return FALSE;
  modecount = vidmodelines (display, screen, &modesinfo);
  for (i = 0; i < modecount; i++) {
    if (modesinfo [i]->hdisplay == res_x && modesinfo [i]->vdisplay == res_y) 
      break;
  }
  if (i >= modecount) return FALSE;
  crt->HDisplay   = modesinfo [i]->hdisplay;  
  crt->HSyncStart = modesinfo [i]->hsyncstart;
  crt->HSyncEnd   = modesinfo [i]->hsyncend;  
  crt->HTotal     = modesinfo [i]->htotal;    
  crt->VDisplay   = modesinfo [i]->vdisplay;
  crt->VSyncStart = modesinfo [i]->vsyncstart;
  crt->VSyncEnd   = modesinfo [i]->vsyncend;  
  crt->VTotal     = modesinfo [i]->vtotal;    
  crt->Flags      = modesinfo [i]->flags;     
  crt->PrivFlags  = 0;
  return TRUE;
}

/*
 * get current X video mode
 */

Bool get_vidmode (Display *display, int screen, NVCrtRegs *crt)
{
  XF86VidModeModeLine modeline;
  int dotclock;

  if (!has_vidmode (display)) return FALSE;
  XF86VidModeGetModeLine (display, screen, &dotclock, &modeline);
  crt->HDisplay   = modeline.hdisplay;  
  crt->HSyncStart = modeline.hsyncstart;
  crt->HSyncEnd   = modeline.hsyncend;  
  crt->HTotal     = modeline.htotal;    
  crt->VDisplay   = modeline.vdisplay;
  crt->VSyncStart = modeline.vsyncstart;
  crt->VSyncEnd   = modeline.vsyncend;  
  crt->VTotal     = modeline.vtotal;    
  crt->Flags      = modeline.flags;     
  crt->PrivFlags  = 0;
  return TRUE;
}

void center_window (Display *display, int screen, Window window,
  int res_x, int res_y)
{
  Window root;
  int x, y;
  unsigned width, height, borderw, depth;
  unsigned disp_w, disp_h;
  int screen_w, screen_h;
  int rx, ry;
  Window junkwin;
  int junkint;
  XF86VidModeModeLine modeline;

  DPRINTF ("center 0x%lx %i,%i\n", window, res_x, res_y);
  if (window == None || !has_vidmode (display)) return;
  XGetGeometry (display, window, &root, &x, &y, &width, &height, 
		&borderw, &depth);
  (void) XTranslateCoordinates (display, window, root, 
				0, 0, &rx, &ry, &junkwin);
  if (res_x >= 0 && res_y >= 0) {
    disp_w = res_x;
    disp_h = res_y;
  } else {
    XF86VidModeGetModeLine (display, screen, &junkint, &modeline);
    if (modeline.privsize > 0) XFree(modeline.private);
    disp_w = modeline.hdisplay;
    disp_h = modeline.vdisplay;
  }
  screen_w = DisplayWidth (display, screen);
  screen_h = DisplayHeight (display, screen);
  x = rx + ((int) width - (int) disp_w) / 2;
  y = ry + ((int) height - (int) disp_h) / 2;
  if (x < 0) x = 0;
  if (y < 0) y = 0;
  if (x > screen_w) x = screen_w;
  if (y > screen_h) y = screen_h;
  XF86VidModeSetViewPort (display, screen, x, y);
  XWarpPointer (display, None, window, 0, 0, 0, 0, 
		width / 2, height / 2);
  /* FIXME: Better, disable the pointer by setting an empty pixmap
     or grabbing it, and restore image on leave event or on the next
     mouse event. Will work only in interactive (gui) mode! */
}

/* -------- dsimple.c -------- */

/* This is are (slightly modified) routines from programs/xlsfonts/dsimple.c 
 * in the XFree sources, written by Mark Lillibridge.
 */

#define Fatal_Error(X) { fprintf(stderr, X); exit (1); }

Window Select_Window (Display *dpy, int screen)
{
  int status;
  Cursor cursor;
  XEvent event;
  Window target_win = None, root = RootWindow(dpy,screen);
  int buttons = 0;

  /* Make the target cursor */
  cursor = XCreateFontCursor(dpy, XC_crosshair);

  /* Grab the pointer using target cursor, letting it room all over */
  status = XGrabPointer(dpy, root, False,
			ButtonPressMask|ButtonReleaseMask, GrabModeSync,
			GrabModeAsync, root, cursor, CurrentTime);
  if (status != GrabSuccess) Fatal_Error("Can't grab the mouse.");

  /* Let the user select a window... */
  while ((target_win == None) || (buttons != 0)) {
    /* allow one more event */
    XAllowEvents(dpy, SyncPointer, CurrentTime);
    XWindowEvent(dpy, root, ButtonPressMask|ButtonReleaseMask, &event);
    switch (event.type) {
    case ButtonPress:
      if (target_win == None) {
	target_win = event.xbutton.subwindow; /* window selected */
	if (target_win == None) target_win = root;
      }
      buttons++;
      break;
    case ButtonRelease:
      if (buttons > 0) /* there may have been some down before we started */
	buttons--;
       break;
    }
  } 
  XUngrabPointer(dpy, CurrentTime);      /* Done with pointer */
  return(target_win);
}

/*
 * Window_With_Name: routine to locate a window with a given name on a display.
 *                   If no window with the given name is found, 0 is returned.
 *                   If more than one window has the given name, the first
 *                   one found will be returned.  Only top and its subwindows
 *                   are looked at.  Normally, top should be the RootWindow.
 */

Window Window_With_Name (Display *dpy, Window top, char *name)
{
  Window *children, dummy;
  unsigned int nchildren;
  int i;
  Window w = None;
  char *window_name;

  if (XFetchName(dpy, top, &window_name) && !strcmp(window_name, name))
    return(top);

  if (!XQueryTree(dpy, top, &dummy, &dummy, &children, &nchildren))
    return None;

  for (i=0; i<nchildren; i++) {
    w = Window_With_Name(dpy, children[i], name);
    if (w != None)
      break;
  }
  if (children) XFree ((char *)children);
  return w;
}
