/* upsd.c - watches ups state files and answers queries 

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "upsd.h"
#include "conf.h"
#include "shared-tables.h"
#include "netcmds.h"
#include "upsconf.h"

#ifdef HAVE_SHMAT
#include <sys/ipc.h>
#include <sys/shm.h>
#endif

#include <sys/un.h>
#include <sys/socket.h>

/* client structure */
typedef struct {
	char	*addr;
	int	fd;
	struct sockaddr_in sock;
	char	rq[SMALLBUF];
	int	rqpos;
	char	*loginups;
	char	*password;
	char	*username;
	void	*next;
}	ctype;

#define ACTION_GRANT	1
#define ACTION_DENY	2
#define ACTION_DROP	4

				/* 76543210 */
#define LEVEL_BASE	1	/* .......* */
#define LEVEL_MONITOR	3	/* ......** */
#define LEVEL_LOGIN	7	/* .....*** */
#define LEVEL_MASTER	15	/* ....**** */
#define LEVEL_MANAGER	31	/* ...***** */
#define LEVEL_ALL	255	/* ******** */

	/* pointers to linked lists */
	upstype	*firstups = NULL;
	acltype *firstacl = NULL;
	acctype *firstacc = NULL;
static	ctype	*firstclient = NULL, *currclient = NULL;

	/* stock reply when a variable is unknown */
static	const char	*unkreply = "UNKNOWN";

	/* by default TCP uses the same port - use -t to override */
static	int	udpfd, udpport = PORT, upsnum = 0;
static	int	listenfd, tcpport = PORT;

	/* default is to listen on all local interfaces */
static	struct	in_addr	listenaddr;

	/* default 15 seconds before data is marked stale */
	int	maxage = 15;

	/* signal handlers */
	struct sigaction sa;
	sigset_t sigmask;

	/* pid file */
	char	pidfn[SMALLBUF];

	/* set by SIGHUP */
	int	reload = 0;

	/* preloaded to STATEPATH in main, can be overridden via upsd.conf */
	char	*statepath = NULL;
	const char *progname = NULL;

#ifdef SHUT_RDWR
#define shutdown_how SHUT_RDWR
#else
#define shutdown_how 2
#endif

/* make sure values are null terminated */
static void copyvalue(char *dst, const char *src)
{
	memcpy(dst, src, VALSIZE - 1);
	dst[VALSIZE - 1] = '\0';
}

/* search for a ups given the name */
upstype *findups(const char *upsname)
{
	upstype	*temp;

	temp = firstups;

	/* NULL name -> return first one */
	if (!upsname)
		return (firstups);

	/* return first one if a good name isn't given */
	if ((!strcasecmp(upsname, "default")) || (!strcmp(upsname, "")))
		return (firstups);
	
	while (temp != NULL) {
		if (!strcasecmp(temp->name, upsname))
			return (temp);

		temp = temp->next;
	}

	return (NULL);
}

static int findupsinfo(upstype *ups, int offset, int type)
{
	return findinfo(ups->info, ups->numinfo, offset, type);
}

static itype *getupsinfo(upstype *ups, int type)
{
	int ret = findupsinfo(ups, 0, type);

	if (ret != -1)
		return &ups->info[ret];

	return NULL;
}

/* this is separate to keep the ifdefs out of the other functions */
static void detach(char *addr)
{
#ifdef HAVE_SHMAT
	shmdt(addr);
#else
	upslogx(LOG_WARNING, "Tried to detach a SHM UPS without SHM support code");
#endif
}

/* mark the data stale if this is new, otherwise cleanup any remaining junk */
static void datastale (upstype *ups, const char* why)
{
	ups->numinfo = 0;

	/* if we just dropped a state file, free the buffer */
	if (ups->shared_info != NULL) {
		if (ups->shmid != -1) {
			detach((char *) ups->shared_info);
		} else {
#ifdef HAVE_MMAP
			/* upslogx(LOG_DEBUG, "datastale: munmap %p size=%u", ups->shared_info, ups->info_size); */
			if (munmap((void *) ups->shared_info, ups->info_size))
				fatal("munmap");
#else
			free(ups->shared_info);
#endif
		}

		ups->shared_info = NULL;
		ups->info_size = 0;
	}
	ups->shmid = -1;

	if (ups->info != NULL) {
		free(ups->info);
		ups->info = NULL;
	}

	if (ups->fd != -1) {
		close(ups->fd);
		ups->fd = -1;
	}

	/* don't complain again if it's already known to be stale */
	if (ups->stale == 1)
		return;

	ups->stale = 1;

	if (why)
		upslogx (LOG_NOTICE,
                "Data for UPS [%s] is stale - check support module (%s)",
	        ups->name, why);

	return;
}

int lock_ups(upstype *ups)
{
	int	tries = 0;
	struct	flock	lock;

	lock.l_start = 0;
	lock.l_len = 0;
	lock.l_type = F_RDLCK;
	lock.l_whence = SEEK_SET;

	/* try to get the lock for up to 250 ms */
	while (fcntl(ups->fd, F_SETLK, &lock) == -1) {
		tries++;

		if (tries == 10) {
			upslogx(LOG_WARNING, "Could not lock data for UPS [%s]",
				ups->name);
			return 0;
		}

		usleep(25000);
	}

	return 1;
}

static void unlock_ups(upstype *ups)
{
	int	ret;
	struct	flock	lock;

	lock.l_start = 0;
	lock.l_len = 0;
	lock.l_type = F_UNLCK;
	lock.l_whence = SEEK_SET;
	ret = fcntl(ups->fd, F_SETLK, &lock);

	if (ret == -1)
		upslog(LOG_INFO, "Error while unlocking UPS [%s]",
			ups->name);
}

#if HAVE_SHMAT || HAVE_MMAP

/* safely copy from shared storage to local storage */
static void readinfo(upstype *ups)
{
	if (!lock_ups(ups))
		return;

	memcpy(ups->info, ups->shared_info, ups->info_size);

	unlock_ups(ups);
}

#endif	/* HAVE_SHMAT || HAVE_MMAP */

/* mark the data ok if this is new, otherwise do nothing */
static void dataok(upstype *ups)
{
	if (ups->stale == 0)
		return;

	upslogx(LOG_NOTICE, "Read data for UPS [%s] successfully",
		ups->name);

	if (ups->shmid >= 0)
		upslogx(LOG_NOTICE, "Data source for UPS [%s]: SHM (%d)", 
			ups->name, ups->shmid);
	else
		upslogx(LOG_NOTICE, "Data source for UPS [%s]: %s", 
			ups->name, ups->fn);

	ups->stale = 0;
}

#ifdef HAVE_SHMAT

/* configure monitoring via shared memory id <shmid> for this UPS */
static void shm_attach(upstype *ups, char *val)
{
	char	shmid[VALSIZE];

	copyvalue(shmid, val);

	ups->shmid = atoi(shmid);
	ups->shared_info = (itype *) shmat(ups->shmid, 0, 0);

	/* now see if it worked */
	if (ups->shared_info == (itype *) (-1)) {
		ups->shared_info = NULL;
		datastale(ups, "shmat failed");
		return;
	}
}

#else	/* HAVE_SHMAT */

/* dummy implementation - only used for highly screwed up situations */
static void shm_attach(upstype *ups, char *dummy)
{
	static	int	warned = 0;

	datastale(ups, "SHM UPS without shared memory support in upsd");

	if (warned)
		return;

	warned = 1;
	upslogx(LOG_WARNING, "Need shared memory mode but HAVE_SHMAT wasn't defined during compile!");
}

#endif	/* HAVE_SHMAT */

/* make sure info is safe to use */
static int check_info(upstype *ups)
{
	int	i;
	time_t	tod;
	itype	*info_mtime;

	for (i = 1; i < ups->numinfo; i++) {
		if (ups->info[i].value[VALSIZE-1] != '\0') {
			datastale(ups, "info is not null terminated");
			return 0;	/* failed */
		}
	}

	info_mtime = getupsinfo(ups, INFO_MTIME);
	if (info_mtime == NULL) {
		datastale(ups, "INFO_MTIME missing");
		return 0;	/* failed */
	}

	time(&tod);
	if (difftime(tod, atoi(info_mtime->value)) > maxage) {
		datastale(ups, "state file too old according to INFO_MTIME");
		return 0;	/* failed */
	}

	/* passed the tests ... */
	dataok(ups);
	return 1;	/* OK */
}

#ifdef HAVE_MMAP

/* mmap version: simple memcpy */
static void update_shared(upstype *ups)
{
	readinfo(ups);
	check_info(ups);
}

#else	/* HAVE_MMAP */

/* non-mmap version: read from the file */
static void update_shared(upstype *ups)
{
	int	ret;

	if (lseek(ups->fd, 0, SEEK_SET)) {
		datastale(ups, "seek to beginning failed");
		return;
	}

	if (!lock_ups(ups))
		return;

	ret = read(ups->fd, ups->info, ups->info_size);

	unlock_ups(ups);

	if (ret != ups->info_size) {
		datastale(ups, "State file changed size - reopening");
		return;
	}

	/* quick sanity check */
	if (ups->info[0].type == INFO_SHMID) {
		datastale(ups, "State file morphed into SHM mode - reopening");
		return;
	}

	check_info(ups);
}

#endif	/* HAVE_MMAP */

/* start monitoring a ups state file */
static void open_statefile(upstype *ups)
{
	struct	stat	st;
	int	ret, mode;
	char	tval[VALSIZE];
	itype	tempinfo;

	if (!ups->fn)
		return;

	/* mmap with MAP_SHARED and PROT_WRITE needs a writable fd */
#ifdef HAVE_MMAP
	mode = O_RDWR;
#else
	mode = O_RDONLY;
#endif

	ups->fd = open(ups->fn, mode);

	if (ups->fd == -1) {
		char	why[SMALLBUF];

		snprintf(why, sizeof(why), "open %s failed: %s", 
			ups->fn, strerror(errno));

		datastale(ups, why);
		return;
	}

	/* enable nonblocking I/O on this fd */
	ret = fcntl(ups->fd, F_SETFL, O_NONBLOCK);

	if (ret == -1)
		upslog(LOG_WARNING, "Can't set O_NONBLOCK for UPS [%s]", 
			ups->name);

	/* now get the details about this file for later */
	ret = stat(ups->fn, &st);

	if (ret == -1) {
		datastale(ups, "open_statefile: could not stat state file");
		return;
	}

	/* save inode/size so we can tell when it changes */
	ups->st_ino = st.st_ino;
	ups->st_size = st.st_size;

	/* read first entry - check for shared memory mode */
	ret = read(ups->fd, &tempinfo, sizeof(itype));

	if (ret != sizeof(itype)) {
		datastale(ups, "can't read first state file entry");
		return;
	}

	if (tempinfo.type == INFO_SHMID) {
		shm_attach(ups, tempinfo.value);
		return;
	}

	/* if not SHM, it must be a normal file, but make sure first */
	if (tempinfo.type != INFO_MEMBERS) {
		upslogx(LOG_NOTICE, "Broken state file - doesn't start with INFO_MEMBERS!");
		datastale(ups, "type != INFO_MEMBERS");
		return;
	}

	/* not shared memory mode - try mmap and direct mode */

	copyvalue(tval, tempinfo.value);
	ups->numinfo = atoi(tval);
	ups->info_size = ups->numinfo * sizeof(itype);

#ifdef HAVE_MMAP

	if ((ups->shared_info = (itype *) mmap(NULL, ups->info_size, PROT_READ|PROT_WRITE,
		MAP_SHARED|MAP_NOSYNC, ups->fd, 0)) == MAP_FAILED)
			fatal("mmap fd=%d size=%u", ups->fd, ups->info_size);

#endif	/* HAVE_MMAP */

	ups->info = xrealloc(ups->info, ups->info_size);

	update_shared(ups);
}

#ifdef HAVE_SHMAT

/* check shared memory segment for sanity */
static int check_shm(upstype *ups)
{
	/* gather status */
	if (shmctl(ups->shmid, IPC_STAT, &ups->shmbuf) != 0) {
		char	failmsg[256];

		snprintf(failmsg, sizeof(failmsg), "shmctl failed: %s",
			strerror(errno));
		datastale(ups, failmsg);
		return 0;	/* failed */
	}

	/* check number of attached processes: driver + upsd = 2 */
	if (ups->shmbuf.shm_nattch < 2) {

		/* mark segment for deletion */
		shmctl(ups->shmid, IPC_RMID, NULL);

		datastale(ups, "shm_nattch < 2");
		return 0;	/* failed */
	}

	/* track the array size */
	ups->numinfo = atoi(ups->shared_info[0].value);
	ups->info_size = ups->numinfo * sizeof(itype);
	ups->info = xrealloc(ups->info, ups->info_size);

	readinfo(ups);

	return check_info(ups);
}
#endif	/* HAVE_SHMAT */

/* check inode/size to make sure this is the same file */
static int check_statefile(upstype *ups)
{
	struct	stat	st;
	int	ret;

	ret = stat(ups->fn, &st);

	if (ret == -1) {
		datastale(ups, "could not stat state file");
		return 0;	/* failed */
	}

	if (st.st_size != ups->st_size) {
		datastale(ups, "state file size changed - resyncing");
		return 0;	/* failed */
	}

	if (st.st_ino != ups->st_ino) {
		datastale(ups, "state file inode changed - resyncing");
		return 0;	/* failed */
	}

#ifdef HAVE_SHMAT

	/* make sure nothing broke if using shared memory mode */
	if ((ups->shmid != -1) && (!check_shm(ups)))
		return 0;	/* FAILED */

#endif

	return 1;	/* OK */
}

static void open_socket(upstype *ups)
{
	struct sockaddr_un sa;
	const char *errmsg = NULL;
	int ret;

	memset(&sa, '\0', sizeof(sa));
	sa.sun_family = AF_UNIX;
	snprintf(sa.sun_path, sizeof(sa.sun_path), "%s.sock", ups->fn);
	/* sa.sun_len = strlen(sa.sun_path); */	/* not portable */

	if ((ups->sock_fd = socket(AF_UNIX, SOCK_STREAM, 0)) == -1)
		{ errmsg = "socket";	goto error; }
	if (connect(ups->sock_fd, (struct sockaddr *) &sa, sizeof(sa)))
		{ errmsg = "connect";	goto error; }
	if ((ret = fcntl(ups->sock_fd, F_GETFL, 0)) == -1)
		{ errmsg = "fcntl(get)";	goto error; }
	if (fcntl(ups->sock_fd, F_SETFL, ret | O_NDELAY) == -1)
		{ errmsg = "fcntl(set)";	goto error; }

	upsdebugx(1, "%s connected to socket fd=%d", ups->name, ups->sock_fd);

	return;

error:
	if (errmsg)
		upslog(LOG_ERR, "%s", errmsg);
	if (ups->sock_fd != -1) {
		close(ups->sock_fd);
		ups->sock_fd = -1;
	}
}

/* refresh local knowledge of a ups called <upsname> */
static void updateups(const char *upsname)
{
	upstype	*ups;

	ups = findups(upsname);

	/* sanity checks */
	if ((!ups) || (!ups->fn))
		return;

	/* try to open the state file if necessary */
	if (ups->fd == -1) {
		open_statefile(ups);

		/* did it work? */
		if (ups->fd == -1)
			return;

		open_socket(ups);
	}

	/* see if the state file is still OK to use */
	if (!check_statefile(ups))
		return;

	/* shared memory mode is done at this point */
	if (ups->shmid != -1)
		return;

	/* do mmap/direct reader update work */
	update_shared(ups);
}

/* change the configuration of an existing UPS (used during reloads) */
void redefine_ups(char *fn, char *name)
{
	upstype	*temp;

	temp = findups(name);

	if (!temp) {
		upslogx(LOG_ERR, "UPS %s disappeared during reload", name);
		return;
	}

	/* paranoia */
	if (!temp->fn) {
		upslogx(LOG_ERR, "UPS %s had a NULL filename!", name);

		/* let's give it something quick to use later */
		temp->fn = xstrdup("");
	}

	/* when the filename changes, force a reread */
	if (strcmp(temp->fn, fn) != 0) {

		upslogx(LOG_NOTICE, "Redefined UPS [%s]", name);

		/* deal with the info array */
		if (temp->info != NULL) {

			/* non-shared memory mode: release the array */
			if (temp->shmid < 0)
				free(temp->info);
			else
				/* SHM mode: just detach */
				detach((char *) temp->info);
		}

		temp->info = NULL;
		temp->shmid = -1;
		temp->stale = 1;
		temp->numinfo = 0;

		/* now redefine the filename and wrap up */
		free(temp->fn);
		temp->fn = xstrdup(fn);
	}

	/* always set this on reload */
	temp->retain = 1;
}		

/* start monitoring a ups called <name> from a file called <fn> */
void addups (char *fn, char *name)
{
	upstype	*temp, *last;

	temp = last = firstups;

	if (!strcasecmp(name, "default")) {
		upslogx(LOG_ERR, "You can't add a UPS called default!");
		return;
	}

	/* find end of linked list */
	while (temp != NULL) {
		last = temp;

		if (!strcasecmp(temp->name, name)) {
			upslogx(LOG_ERR, "UPS name [%s] is already in use!", name);
			return;
		}

		temp = temp->next;
	}

	/* grab some memory and add the info */
	temp = xcalloc(1, sizeof(upstype));
	temp->fn = xstrdup(fn);
	temp->shmid = -1;
	temp->fd = -1;
	temp->sock_fd = -1;
	temp->name = xstrdup(name);
	temp->info = NULL;
	temp->shared_info = NULL;
	temp->stale = 1;
	temp->st_ino = -1;
	temp->st_size = -1;
	temp->numinfo = 0;
	temp->numlogins = 0;
	temp->fsd = 0;
	temp->retain = 1;
	temp->next = NULL;

	if (last == NULL)	/* first one */
		firstups = temp;
	else			/* not the first */
		last->next = temp;

	updateups(name);
	upsnum++;

	if (temp->stale)
		upslogx(LOG_NOTICE, "Warning: Data for UPS [%s] is stale - check driver", name);
}

/* setup a socket for incoming udp requests */
static void setupudp(void)
{
	struct	sockaddr_in	recv;

	if ((udpfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
		fatal("Can't create socket");

	memset (&recv, '\0', sizeof(recv));
	recv.sin_addr = listenaddr;
	recv.sin_family = AF_INET;
	recv.sin_port = htons(udpport);

	if (bind(udpfd, (struct sockaddr *) &recv, sizeof(recv)) < 0)
		fatal("Can't bind to socket");

	return;
}

/* create a listening socket for tcp connections */
static void setuptcp(void)
{
	struct	sockaddr_in	server;
	int	res;

	if ((listenfd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
		fatal("socket");

	memset (&server, '\0', sizeof(server));
	server.sin_addr = listenaddr;
	server.sin_family = AF_INET;
	server.sin_port = htons(tcpport);

	if (bind (listenfd, (struct sockaddr *) &server, sizeof(server)) == -1)
		fatal("bind");

	if ((res = fcntl(listenfd, F_GETFL, 0)) == -1)
		fatal("fcntl(get)");

	if (fcntl(listenfd, F_SETFL, res | O_NDELAY) == -1)
		fatal("fcntl(set)");

	if (listen(listenfd, 16))
		fatal("listen");

	return;
}

/* decrement the login counter for this ups */
static void declogins(const char *upsname)
{
	upstype	*ups;

	ups = findups(upsname);

	if (ups == NULL) {
		upslogx(LOG_INFO, "Tried to decrement invalid ups name (%s)", upsname);
		return;
	}

	ups->numlogins--;
}

/* remove a client struct from the linked list */
void delclient (ctype *dclient)
{
	ctype	*tmp, *last;

	if (dclient == NULL)
		return;

	last = NULL;
	tmp = firstclient;

	while (tmp != NULL) {
		if (tmp == dclient) {		/* found it */
			shutdown(dclient->fd, 2);
			close(dclient->fd);

			free (tmp->addr);

			if (tmp->loginups != NULL) {
				declogins (tmp->loginups);
				free (tmp->loginups);
			}

			if (tmp->password != NULL)
				free (tmp->password);

			if (last == NULL)	/* deleting first entry */
				firstclient = tmp->next;
			else
				last->next = tmp->next;

			free (tmp);
			currclient = NULL;
			return;
		}
		last = tmp;
		tmp = tmp->next;
	}

	/* not found?! */
	upslogx(LOG_WARNING, "Tried to delete client struct that doesn't exist!");

	return;
}

/* send the buffer <sendbuf> of length <sendlen> to host <dest> */
static void sendback(const struct sockaddr_in *dest, const char *fmt, ...)
{
#define NUT_NET_ANSWER_MAX SMALLBUF
	int	res, destfd, len;
	char ans[NUT_NET_ANSWER_MAX+1];
	va_list ap;

	if (currclient == NULL)		/* udp */
		destfd = udpfd;
	else
		destfd = currclient->fd;

	va_start(ap, fmt);
	vsnprintf(ans, sizeof(ans), fmt, ap);
	va_end(ap);

	len = strlen(ans);

	res = sendto (destfd, ans, len, 0, (struct sockaddr *) dest,
	              sizeof (struct sockaddr_in));

	if (len != res) {
		upslog(LOG_NOTICE, "sendto() failed for %s", currclient->addr);
		delclient(currclient);
	}

	return;
}

/* see if <addr> matches the acl <aclname> */
int checkacl(const char *aclname, const struct sockaddr_in *addr)
{
	acltype	*tmp;
	int	aclchk, addrchk;

	tmp = firstacl;
	while (tmp != NULL) {
		if (!strcmp(tmp->name, aclname)) {
			aclchk = tmp->addr & tmp->mask;
			addrchk = ntohl(addr->sin_addr.s_addr) & tmp->mask;

			if (aclchk == addrchk) 
				return 1;	/* match */
		}

		tmp = tmp->next;
	}

	return 0;	/* not found */
}

static int checkaccess(const struct sockaddr_in *ip, int level)
{
	acctype	*tmp;
	int	ret, checkpw;
	char	*pw;

	/* only check password for certain levels */
	if (level >= LEVEL_LOGIN)
		checkpw = 1;
	else
		checkpw = 0;

	tmp = firstacc;

	/* only use password if in tcp mode */
	if (currclient == NULL)
		pw = NULL;
	else
		pw = currclient->password;

	while (tmp != NULL) {
		ret = checkacl (tmp->aclname, ip);

		/* if ip matches and access line provides the right level.. */

		if ((ret == 1) && ((tmp->level & level) == level)) {
			/* handle denials first */
			switch (tmp->action) {
				case ACTION_GRANT:
					break;	/* fallthrough to pw checks */
				
				case ACTION_DENY: 
					upsdebugx(1, "access denied");
					sendback(ip, "ERR ACCESS-DENIED\n");

				/* silent return, no response */
				case ACTION_DROP:
				default:
					return (0);
			}

			/* ok, must be a grant now */

			if (checkpw == 0)	/* password not required */
				return (1);	/* OK */

			/* no password set on this access line? */
			if ((tmp->password == NULL) || 
			    (strlen(tmp->password) == 0))
				return (1);	/* OK */

			/* must require a password now, so check it */

			/* if this is a UDP client, bail out */
			if (!currclient) {
				sendback(ip, "ERR PASSWORD-REQUIRED\n");
				return (0);	/* failed */
			}			

			/* if the client hasn't given us a password yet */
			if (currclient->password == NULL) {
				sendback(ip, "ERR PASSWORD-REQUIRED\n");
				return (0);	/* failed */
			}

			/* if it matches, they're good */
			if (!strcmp(tmp->password, currclient->password))
				return (1);	/* OK */

			/* only thing left: failed password */

			sendback(ip, "ERR PASSWORD-INCORRECT\n");
			return (0);	/* failed */

		} 	/* if (ip matches) && (level is provided) */

		/* otherwise ip didn't match or the level was inadequate */

		tmp = tmp->next;
	}

	/* default = drop */
	return (0);
}

/* increment the login counter for this ups */
static void inclogins(const char *upsname)
{
	upstype	*ups;

	ups = findups (upsname);

	if (ups == NULL) {
		upslogx (LOG_INFO, "Tried to increment invalid ups name");
		return;
	}

	ups->numlogins++;
}

/* disconnect anyone logged into this UPS */
void kick_login_clients(char *upsname)      
{
	ctype   *tmp, *next;

	tmp = firstclient;

	while (tmp) {
		next = tmp->next;

		/* if it's not logged in, don't check it */
		if (!tmp->loginups) {
			tmp = next;
			continue;
		}

		if (!strcmp(tmp->loginups, upsname)) {
			upslogx(LOG_INFO, "Kicking client %s (was on UPS [%s])\n", 
				tmp->addr, upsname);
			delclient(tmp);
		}
                
		tmp = next;
	}
}

/* return a name for a command given its identifier (from shared.h) */
static const char *cmdname(int cmdnum)
{
	int	i;

	for (i = 0; netvars[i].name != NULL; i++)
		if (netvars[i].type == cmdnum)
			return (netvars[i].name);

	if (cmdnum != INFO_UNUSED) {
		upslogx (LOG_INFO, "Unknown variable type 0x%04x", cmdnum);
	}
	
	return (unkreply);
}

/* return a name for an instant command given its identifier (from shared.h) */
const char *instcmdname(int cmdnum)
{
	int	i;

	for (i = 0; instcmds[i].name != NULL; i++)
		if (instcmds[i].cmd == cmdnum)
			return (instcmds[i].name);

	if (cmdnum != INFO_UNUSED) {
		upslogx (LOG_INFO, "Unknown cmd num 0x%04x", cmdnum);
	}
	
	return (unkreply);
}

/* handler for "REQ NUMLOGINS" */
static void numlogins(const struct sockaddr_in *dest, const upstype *ups, const char *varname)
{
	if (!checkaccess (dest, LEVEL_LOGIN))
		return;

	sendback(dest, "ANS %s %i\n", varname, ups->numlogins);
	return;
}

/* try to send a message to a ups */
static int sendupsmsg(upstype *ups, msgtype *msg)
{
	int ret;

	ret = write(ups->sock_fd, msg, sizeof(msgtype));
	if (ret != sizeof(msgtype)) {
		upslog(LOG_NOTICE, "sendupsmsg():write fd=%d", ups->sock_fd);
		return (0);	/* no msg sent */
	}

	upsdebugx(1, "sendupsmsg fd=%d ret=%d", ups->sock_fd, ret);

	return (1);	/* send ok */
}	

#if 0	/* TODO: once drivers can reply */
/* try to receive a message from a ups module */
static int getupsmsg (upstype *ups, msgtype *msg, size_t *msglen, int wait)
{
	upslogx(LOG_NOTICE, "getupsmsg BROKEN!!!");

	return (0);	/* no msg received */

	return (1);	/* msg received */
}
#endif

/* parse varname[@upsname] into separate variables */
static upstype *parsearg(const char *arg, char **varname)
{
	char	*argtmp, *upsname;
	upstype	*ups;

	argtmp = xstrdup (arg);

	upsname = strchr (argtmp, '@');
	if (upsname) {
		*upsname++ = '\0';
		ups = findups (upsname);
	} else
		ups = findups ("default");

	/* varname may be a subset of argtmp, so copy it and free argtmp */
	*varname = xstrdup(argtmp);
	free (argtmp);

	return (ups);
}

/* handler for "REQ" - send a reply */
void do_sendans(struct sockaddr_in *dest, char *varin, char *rest)
{
	int	i, type = 0;
	char	ans[SMALLBUF], *varname;
	itype	*info = NULL;
	upstype	*ups;

	if (!checkaccess(dest, LEVEL_MONITOR))
		return;

	if (varin == NULL) {
		sendback(dest, "ERR MISSING-ARGUMENT\n");
		return;
	}

	ups = parsearg(varin, &varname);

	if (ups == NULL) {
		sendback(dest, "ERR UNKNOWN-UPS\n", varin);
		free(varname);
		return;
	}

	/* special case: numlogins is an internal value */
	if (!strcasecmp(varname, "NUMLOGINS")) {
		numlogins(dest, ups, varin);
		free(varname);
		return;
	}

	/* handle this up front */
	if (ups->stale == 1) {
		sendback(dest, "ERR DATA-STALE\n");
		free(varname);
		return;
	}

	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, varname))
			type = netvars[i].type;

	free(varname);
	
	/* type wasn't resolved in the netvars[] array */
	if (type == 0) {
		sendback(dest, "ERR %s VAR-UNKNOWN\n", varin);
		return;
	}

	/* stock reply - overwritten if an answer is found */
	strlcpy(ans, "ERR VAR-NOT-SUPPORTED\n", sizeof(ans));

	if (ups->stale == 1) {
		sendback(dest, "ERR DATA-STALE\n");
		return;
	}

	info = getupsinfo(ups, type);

	if (info) {
		/* special case - make FSD show up in STATUS */
		if ((type == INFO_STATUS) && (ups->fsd == 1)) 
			snprintf(ans, sizeof(ans), "ANS %s FSD %s", 
				varin, info->value);
		else
			snprintf(ans, sizeof(ans), "ANS %s %s",
				varin, info->value);
	}

	sendback(dest, "%s\n", ans);
}

/* handler for "LISTVARS" */
void do_listvars (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	ups = findups (arg);

	if (ups == NULL) {
		sendback(dest, "ERR UNKNOWN-UPS\n");
		return;
	}

	ans[0] = '\0';

	/* handle this up front */
	if (ups->stale == 1) {
		sendback(dest, "%s", "ERR DATA-STALE\n");
		return;
	}


	/* insert "@upsname" if explicitly specified in request */
	if ((arg) && (strcmp(arg, "") != 0))
		snprintf(ans, sizeof(ans), " @%s", arg);

	for (i = 1; i < ups->numinfo; i++)
		if ((ups->info[i].type & INFO_SYSMASK) == 0) {
			strlcat(ans, " ", sizeof(ans));
			strlcat(ans, cmdname(ups->info[i].type), sizeof(ans));
		}

	sendback(dest, "VARS%s\n", ans);
}

/* handler for "VER" */
void do_sendver(struct sockaddr_in *dest, char *arg, char *rest)
{
	if (!checkaccess (dest, LEVEL_BASE))
		return;

	sendback(dest, "Network UPS Tools upsd %s - http://www.exploits.org/nut/\n", UPS_VERSION);
}

/* handler for "HELP" */
void do_sendhelp(struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;

	if (!checkaccess (dest, LEVEL_BASE))
		return;

	strlcpy(ans, "Commands:", sizeof(ans));

	for (i = 0; netcmds[i].name != NULL; i++)
		snprintfcat(ans, sizeof(ans), " %s", netcmds[i].name);

	sendback (dest, "%s\n", ans);
}

/* handler for "LOGOUT" */
void do_logout (struct sockaddr_in *dest, char *arg, char *rest)
{
	if (currclient == NULL) {
		sendback (dest, "Can't logout in UDP mode!\n");
		return;
	}

	sendback (dest, "Goodbye...\n");

	upslogx (LOG_INFO, "Client on %s logged out", currclient->addr);

	delclient (currclient);
}		

/* handler for "LOGIN" */
void do_login (struct sockaddr_in *dest, char *arg, char *rest)
{
	int	found = 0;
	upstype *ups;

	if (currclient == NULL) {
		sendback(dest, "Can't login in UDP mode!\n");
		return;
	}

	if (!checkaccess (dest, LEVEL_LOGIN))
		return;

	if (currclient->loginups != NULL) {
		sendback(dest, "%s", "ERR ALREADY-LOGGED-IN\n");
		upslogx(LOG_INFO, "Client on %s tried to login twice", currclient->addr);
		return;
	}

	if ((arg == NULL) || (strlen(arg) == 0))	/* "LOGIN" only */
		currclient->loginups = xstrdup ("default");
	else {
		/* see if <arg> is a valid UPS name */

		for (ups = firstups; ups != NULL; ups = ups->next) {
			if (!strcasecmp(ups->name, arg)) {
				found = 1;
			}
		}

		if (!found) {
			sendback (dest, "ERR UNKNOWN-UPS\n");
			return;
		}

		currclient->loginups = xstrdup (arg);
	}

	inclogins (currclient->loginups);

	upslogx (LOG_INFO, "Client %s logged into UPS [%s]",
		  currclient->addr, currclient->loginups);

	sendback (dest, "OK\n");
}

/* handler for "PASSWORD" */
void do_password (struct sockaddr_in *dest, char *arg, char *rest)
{
	if (currclient == NULL) {
		sendback (dest, "Can't set password in UDP mode!\n");
		return;
	}

	if (!checkaccess (dest, LEVEL_BASE))
		return;

	if ((arg == NULL) || (strlen(arg) == 0)) {
		sendback(dest, "ERR INVALID-PASSWORD\n");
		upslogx(LOG_INFO, "Client on %s sent a NULL password!", currclient->addr);
		return;	
	}

	if (currclient->password != NULL) {
		sendback(dest, "ERR ALREADY-SET-PASSWORD\n");
		upslogx(LOG_INFO, "Client on %s tried to set a password twice", currclient->addr);
		return;
	}

	currclient->password = xstrdup(arg);

	sendback (dest, "OK\n");
}

/* handler for "LISTRW" */
void do_listrw (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	ups = findups (arg);

	if (ups == NULL) {
		sendback(dest, "ERR UNKNOWN-UPS\n");
		return;
	}

	ans[0] = '\0';

	/* handle this up front */
	if (ups->stale == 1) {
		sendback(dest, "%s", "ERR DATA-STALE\n");
		return;
	}

	/* insert "@upsname" if explicitly specified in request */
	if ((arg) && (strcmp(arg, "") != 0)) {
		snprintfcat(ans, sizeof(ans), " @%s", arg);
	}		

	for (i = 1; i < ups->numinfo; i++) {
		if ((ups->info[i].type & INFO_SYSMASK) == 0) {
			if (ups->info[i].flags && FLAG_RW) {
				snprintfcat(ans, sizeof(ans), " %s", cmdname(ups->info[i].type));
			}
		}
	}

	sendback(dest, "RW%s\n", ans);
}

/* handler for "VARTYPE" */
void do_vartype (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF], *varname;
	int	i, type;
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	if (arg == NULL) {
		sendback (dest, "ERR MISSING-ARGUMENT\n");
		return;
	}

	ups = parsearg (arg, &varname);

	if (ups == NULL) {
		sendback (dest, "ERR UNKNOWN-UPS\n");
		free (varname);
		return;
	}

	type = 0;

	/* find the variable type for the name */
	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, varname))
			type = netvars[i].type;

	if (type == 0) {	/* didn't find it */
		sendback (dest, "ERR UNKNOWN-TYPE\n");
		free (varname);
		return;
	}

	/* stock reply */
	snprintf (ans, sizeof(ans), "ERR VAR-NOT-SUPPORTED\n");

	/* try to find the variable in the info array */
	for (i = 1; i < ups->numinfo; i++)
		if (ups->info[i].type == type) {
			if (ups->info[i].flags & FLAG_ENUM)
				snprintf (ans, sizeof(ans), "TYPE ENUM %i\n",
				          ups->info[i].auxdata);

			if (ups->info[i].flags & FLAG_STRING)
				snprintf (ans, sizeof(ans), "TYPE STRING %i\n",
				          ups->info[i].auxdata);
		}

	sendback (dest, "%s", ans);
	free (varname);
}

/* TODO: clean this up ... lots of identical code in here */

/* handler for "VARDESC" */
void do_vardesc (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	if (arg == NULL) {
		sendback (dest, "ERR MISSING-ARGUMENT\n");
		return;
	}

	/* stock reply */
	snprintf (ans, sizeof(ans), "ERR VAR-NOT-SUPPORTED\n");

	/* find the variable type for the name */
	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, arg))
			snprintf (ans, sizeof(ans), "DESC \"%s\"\n", 
			          netvars[i].desc);

	sendback(dest, "%s", ans);
}

/* handler for "ENUM" */
void do_enum (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	*varname;
	int	i, type, pos;
	upstype *ups;
	char	*tval;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	if (arg == NULL) {
		sendback(dest, "ERR MISSING-ARGUMENT\n");
		return;
	}

	ups = parsearg (arg, &varname);

	if (ups == NULL) {
		sendback(dest, "ERR UNKNOWN-UPS\n");
		free (varname);
		return;
	}

	type = 0;

	/* find the variable type for the name */
	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, varname))
			type = netvars[i].type;

	if (type == 0) {	/* didn't find it */
		sendback(dest, "ERR UNKNOWN-TYPE\n");
		free (varname);
		return;
	}

	pos = 0;

	/* find the variable in the info array first */
	pos = findupsinfo(ups, 0, type);

	if (pos == -1) {
		sendback(dest, "ERR VAR-NOT-SUPPORTED\n");
		free (varname);
		return;
	}

	sendback(dest, "ENUM %s\n", varname);

	/* try to find the variable in the info array */
	for (i = 1; i < ups->numinfo; i++)
		if ((ups->info[i].type == INFO_ENUM) &&
		    (ups->info[i].auxdata == type)) {
			tval = ups->info[i].value;

			/* add "SELECTED" if this matches the current value */
			if (!strcmp(tval, ups->info[pos].value))
				sendback(dest, "OPTION \"%s\" SELECTED\n", tval);
			else
				sendback(dest, "OPTION \"%s\"\n", tval);
		}

	sendback(dest, "END\n");
	free (varname);
}

/* handler for "SET" */
void do_set (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF], *newval, *varname, *ptr;
	upstype *ups;
	msgtype	msg;
	int	i, infopos, type, found;

	if (!currclient) {
		sendback(dest, "This command requires a TCP connection!\n");
		return;
	}

	/* support old clients that don't send USERNAME for awhile */
	if (!currclient->username) {
		if (!checkaccess (dest, LEVEL_MANAGER))
			return;
	}
	else {
		/* user mode: see if this action (SET) is allowed */

		if (!user_checkaction(dest, currclient->username,
		                      currclient->password, "SET")) {
			sendback(dest, "ERR ACCESS-DENIED\n");
			return;
		}
	}

	if ((arg == NULL) || (rest == NULL)) {
		sendback (dest, "ERR MISSING-ARGUMENT\n");
		return;
	}

	ups = parsearg (arg, &varname);

	if (ups == NULL) {
		sendback (dest, "ERR UNKNOWN-UPS\n");
		free (varname);
		return;
	}

	type = 0;
	/* make sure 'varname' is part of the protocol as we know it */
	for (i = 0; netvars[i].name != NULL; i++)
		if (!strcasecmp(netvars[i].name, varname))
			type = netvars[i].type;
	
	/* type wasn't resolved in the netvars[] array */
	if (type == 0) {
		sendback (dest, "ERR UNKNOWN-VAR\n");
		free (varname);
		return;
	}

	/* jump out if this UPS is stale */
	if (ups->stale == 1) {
		sendback (dest, "ERR DATA-STALE\n");
		free (varname);
		return;
	}

	/* now see if this UPS supports that variable */
	infopos = findupsinfo(ups, 0, type);

	if (infopos == -1) {
		sendback (dest, "ERR VAR-NOT-SUPPORTED\n");
		free (varname);
		return;
	}

	/* is it even settable (RW) ? */
	if ((ups->info[infopos].flags & FLAG_RW) == 0) {	
		sendback (dest, "ERR READONLY\n");
		free (varname);
		return;
	}

	/* clean up rest into something usable */
	newval = xstrdup (rest);

	ptr = strchr (newval, 13);
	if (ptr)
		*ptr = '\0';	/* strip trailing CR */

	ptr = strchr (newval, 10);
	if (ptr)
		*ptr = '\0';	/* strip trailing LF */

	/* finally, see if the new value is allowed for this variable */

	/* check the length if this is for a string */
	if (ups->info[infopos].flags & FLAG_STRING) {
		if (strlen(newval) > ups->info[infopos].auxdata) {
			sendback (dest, "ERR TOO-LONG\n");
			free (newval);
			free (varname);
			return;
		}
	}

	if (ups->info[infopos].flags & FLAG_ENUM) {
		found = 0;
		/* try to find the variable in the info array */
		for (i = 1; i < ups->numinfo; i++) {
			if ((ups->info[i].type == INFO_ENUM) && (ups->info[i].auxdata == type)) {
				if (!strcmp(ups->info[i].value, newval)) {
					found = 1;
					break;
				}
			}
		}

		if (!found) {
			sendback (dest, "ERR INVALID-VALUE\n");
			free (newval);
			free (varname);
			return;
		}
	}

	if (currclient->username)
		upslogx(LOG_INFO, "Set variable: %s@%s set %s.%s to %s",
		       currclient->username, currclient->addr, ups->name, 
		       varname, newval);
	else
		upslogx(LOG_INFO, "Set variable: %s set %s.%s to %s",
		       currclient->addr, ups->name, varname, newval);

	msg.cmdtype = UPSMSG_CMDSET;
	msg.auxcmd = type;
	if (strlcpy(msg.data, newval, sizeof(msg.data)) >= sizeof(msg.data))
		; /* string too long */
	msg.dlen = strlen(msg.data);

	/* TODO: lose debug msgs when done */
	if (sendupsmsg(ups, &msg)) {
		upslogx (LOG_INFO, "Sent set command to model");
		snprintf (ans, sizeof(ans), "OK\n");
	} else {
		upslogx (LOG_INFO, "Set command send failed");
		snprintf (ans, sizeof(ans), "ERR SET-FAILED\n");
	}

	sendback (dest, "%s", ans);
	free (newval);
	free (varname);
}

/* handler for "INSTCMD" */
void do_instcmd (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF], *varname;
	upstype *ups;
	msgtype	msg;
	int	i, cmd, found;

	if (!currclient) {
		sendback(dest, "This command requires a TCP connection!\n");
		return;
	}

	/* no username set == use original host/pw style matching */
	if (!currclient->username)
		if (!checkaccess(dest, LEVEL_MANAGER))
			return;

	if (!arg) {
		sendback(dest, "ERR MISSING-ARGUMENT\n");
		return;
	}		

	ups = parsearg(arg, &varname);

	if (ups == NULL) {
		sendback(dest, "ERR UNKNOWN-UPS\n");
		free(varname);
		return;
	}

	cmd = 0;

	/* walk through instcmds[], strcmp */
	for (i = 0; instcmds[i].name != NULL; i++)
		if (!strcasecmp(instcmds[i].name, varname))
			cmd = instcmds[i].cmd;

	if (cmd == 0) {
		sendback(dest, "ERR UNKNOWN-INSTCMD\n");
		free(varname);
		return;
	}

	/* now make sure the UPS can actually DO this command */

	found = 0;
	for (i = 1; i < ups->numinfo; i++)
		if (ups->info[i].type == INFO_INSTCMD)
			if (ups->info[i].auxdata == cmd) {
				found = 1;
				break;
			}

	if (!found) {
		sendback(dest, "ERR CMD-NOT-SUPPORTED\n");
		free(varname);
		return;
	}

	/* user mode: see if this user is allowed to do this command */
	if (currclient->username) {
		if (!user_checkinstcmd(dest, currclient->username, 
			currclient->password, cmd)) {
			sendback(dest, "ERR ACCESS-DENIED\n");
			free(varname);
			return;
		}

		upslogx(LOG_INFO, "Instant command: %s@%s did %s on %s",
		       currclient->username, currclient->addr, varname, 
		       ups->name);
	}
	else
		upslogx(LOG_INFO, "Instant command: %s did %s on %s",
		       currclient->addr, varname, ups->name);

	msg.cmdtype = UPSMSG_INSTCMD;
	msg.auxcmd = cmd;
	msg.dlen = 0;

	/* TODO: lose debug msgs when done */
	if (sendupsmsg(ups, &msg)) {
		upslogx(LOG_INFO, "Sent instant command to model");
		snprintf(ans, sizeof(ans), "OK\n");
	} else {
		upslogx(LOG_INFO, "Instant command send failed");
		snprintf(ans, sizeof(ans), "ERR INSTCMD-FAILED\n");
	}

	sendback(dest, "%s", ans);
	free(varname);
}

/* handler for "LISTINSTCMD" */
void do_listinstcmd (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	ups = findups (arg);

	if (ups == NULL) {
		sendback (dest, "ERR UNKNOWN-UPS\n");
		return;
	}

	ans[0] = '\0';

	/* handle this up front */
	if (ups->stale == 1) {
		snprintf(ans, sizeof(ans), "ERR DATA-STALE\n");
		sendback(dest, "%s", "ERR DATA-STALE\n");
		return;
	}

	/* insert "@upsname" if explicitly specified in request */
	if ((arg) && (strcmp(arg, "") != 0))
		snprintfcat(ans, sizeof(ans), " @%s", arg); 

	for (i = 1; i < ups->numinfo; i++)
		if (ups->info[i].type == INFO_INSTCMD)
			snprintfcat(ans, sizeof(ans), " %s", instcmdname(ups->info[i].auxdata));

	sendback(dest, "INSTCMDS%s\n", ans);
}

/* handler for "INSTCMDDESC" */
void do_instcmddesc (struct sockaddr_in *dest, char *arg, char *rest)
{
	char	ans[SMALLBUF];
	int	i;

	if (!checkaccess (dest, LEVEL_MONITOR))
		return;

	if (arg == NULL) {
		sendback (dest, "ERR MISSING-ARGUMENT\n");
		return;
	}

	/* stock reply */
	snprintf (ans, sizeof(ans), "ERR CMD-NOT-SUPPORTED\n");

	/* find the variable type for the name */
	for (i = 0; instcmds[i].name != NULL; i++)
		if (!strcasecmp(instcmds[i].name, arg))
			snprintf (ans, sizeof(ans), "DESC \"%s\"\n", 
			          instcmds[i].desc);

	sendback (dest, "%s", ans);
}

/* handler for "FSD" */
void do_fsd (struct sockaddr_in *dest, char *arg, char *rest)
{
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MASTER))
		return;

	ups = findups (arg);

	if (ups == NULL) {
		sendback (dest, "ERR UNKNOWN-UPS\n");
		return;
	}

	upslogx (LOG_INFO, "Setting FSD on UPS [%s]", ups->name);

	ups->fsd = 1;
	sendback (dest, "OK FSD-SET\n");
}

/* handler for "MASTER" */
void do_master (struct sockaddr_in *dest, char *arg, char *rest)
{
	upstype *ups;

	if (!checkaccess (dest, LEVEL_MASTER))
		return;

	ups = findups (arg);

	if (ups == NULL) {
		sendback (dest, "ERR UNKNOWN-UPS\n");
		return;
	}

	/* really simple here - checkaccess does the work for us */

	sendback (dest, "OK MASTER-GRANTED\n");
}

/* handler for "USERNAME" */
void do_username (struct sockaddr_in *dest, char *arg, char *rest)
{
	if (currclient == NULL) {
		sendback (dest, "Can't set username in UDP mode!\n");
		return;
	}

	if (!checkaccess(dest, LEVEL_BASE))
		return;

	if ((arg == NULL) || (strlen(arg) ==0)) {
		sendback(dest, "ERR INVALID-USERNAME\n");
		upslogx(LOG_INFO, "Client on %s sent a NULL username!",
		       currclient->addr);
		return;
	}

	if (currclient->username != NULL) {
		sendback(dest, "ERR ALREADY-SET-USERNAME\n");
		upslogx(LOG_INFO, "Client on %s tried to set a username twice",
		       currclient->addr);
		return;
	}

	currclient->username = xstrdup(arg);

	sendback(dest, "OK\n");
}

void do_starttls(struct sockaddr_in *dest, char *arg, char *rest)
{
	if (currclient == NULL) {
		sendback (dest, "Can't STARTTLS in UDP mode!\n");
		return;
	}

	sendback(dest, "ERR notimpl\n");
}

/* parse requests from the network */
void parse (int len, char *buf, struct sockaddr_in *from)
{
	int	i;
	char	*cmd, *arg2, *rest, *ptr;

	/* check for base access if this is a UDP client */
	if ((currclient == NULL) && (!checkaccess (from, LEVEL_BASE)))
		return;

	arg2 = rest = NULL;

	/* kill any embedded cr/lf chars */
	for (i = 0; i < len; i++)
		if ((buf[i] == 13) || (buf[i] == 10))
			buf[i] = '\0';

	/* parse from buf: <cmd> [<arg2> [<rest>]] */

	cmd = buf;
	ptr = strchr (cmd, ' ');
	if (ptr) {
		*ptr++ = '\0';
		arg2 = ptr;
		ptr = strchr (arg2, ' ');
		if (ptr) {
			*ptr++ = '\0';
			rest = ptr;
		}
	}

	for (i = 0; netcmds[i].name != NULL; i++) {
		if (!strcasecmp(netcmds[i].name, cmd)) {
			netcmds[i].func(from, arg2, rest);
			return;
		}
	}

	sendback (from, "ERR UNKNOWN-COMMAND\n");
}

/* update every ups that is being monitored */
void updateall (void)
{
	upstype *ups;

	ups = firstups;

	while (ups != NULL) {
		updateups(ups->name);
		ups = ups->next;
	}

	return;
}

/* add the acl to the linked list */
void addacl (char *aclname, char *ipblock)
{
	acltype	*tmp, *last;
	char	*addr, *mask;

	/* are we sane? */
	if ((!aclname) || (!ipblock))
		return;

	/* ipblock must be in the form <addr>/<mask>	*/

	mask = strchr (ipblock, '/');

	/* 206.253.95.200/32 : VALID			*/
	/* 206.253.95.200/255.255.255.255 : VALID	*/
	/* 206.253.95.200 : INVALID			*/

	if (!mask) 	/* no slash = broken acl declaration */
		return;

	*mask++ = '\0';
	addr = ipblock;

	tmp = last = firstacl;

	while (tmp != NULL) {	/* find end */
		last = tmp;
		tmp = tmp->next;
	}

	tmp = xmalloc (sizeof (acltype));
	tmp->name = xstrdup (aclname);
	tmp->addr = ntohl(inet_addr (addr));
	tmp->next = NULL;

	if (strstr(mask, ".") == NULL) { /* must be a /nn CIDR type block */
		if (atoi(mask) != 32)
			tmp->mask = ((unsigned int) ((1 << atoi(mask)) - 1) << 
			            (32 - atoi(mask)));
		else
			tmp->mask = 0xffffffff;	/* avoid overflow from 2^32 */
	}
	else
		tmp->mask = ntohl(inet_addr (mask));

	if (last == NULL)	/* first */
		firstacl = tmp;
	else
		last->next = tmp;
}

/* add to the access linked list */
void addaccess (const char *action, char *level, char *aclname, char *password)
{
	acctype	*tmp, *last;

	/* more sanity checking (null password is OK) */
	if ((!action) || (!level) || (!aclname))
		return;

	tmp = last = firstacc;

	while (tmp != NULL) {	/* find end */
		last = tmp;
		tmp = tmp->next;
	}

	tmp = xmalloc (sizeof(acctype));
	tmp->action = 0;
	tmp->level = 0;

	if (!strcasecmp(action, "grant"))
		tmp->action = ACTION_GRANT;
	if (!strcasecmp(action, "deny"))
		tmp->action = ACTION_DENY;
	if (!strcasecmp(action, "drop"))
		tmp->action = ACTION_DROP;

	if (!strcasecmp(level, "base"))
		tmp->level = LEVEL_BASE;
	if (!strcasecmp(level, "monitor"))
		tmp->level = LEVEL_MONITOR;
	if (!strcasecmp(level, "login"))
		tmp->level = LEVEL_LOGIN;
	if (!strcasecmp(level, "master"))
		tmp->level = LEVEL_MASTER;
	if (!strcasecmp(level, "manager"))
		tmp->level = LEVEL_MANAGER;
	if (!strcasecmp(level, "all"))
		tmp->level = LEVEL_ALL;

	tmp->aclname = xstrdup(aclname);
	tmp->next = NULL;

	if ((password == NULL) || (strlen(password) == 0))
		tmp->password = NULL;
	else
		tmp->password = xstrdup(password);

	if (last == NULL)	/* first */
		firstacc = tmp;
	else
		last->next = tmp;	
}

/* answer incoming tcp connections */
void answertcp()
{
	int	acc;
	struct	sockaddr_in client;
	ctype	*tmp, *last;
	unsigned int	clen;

	clen = sizeof (client);
	acc = accept (listenfd, (struct sockaddr *) &client, &clen);

	if (acc < 0)
		return;

	if (!checkaccess (&client, LEVEL_BASE)) {
		upslogx (LOG_INFO, "Rejecting TCP connection from %s", inet_ntoa (client.sin_addr));
		close (acc);
		return;
	}

	upslogx (LOG_INFO, "Connection from %s", inet_ntoa(client.sin_addr));

	last = tmp = firstclient;

	while (tmp != NULL) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = xmalloc (sizeof(ctype));
	tmp->addr = xstrdup(inet_ntoa(client.sin_addr));
	tmp->fd = acc;
	memcpy (&tmp->sock, &client, sizeof(struct sockaddr_in));
	tmp->rqpos = 0;
	memset (tmp->rq, '\0', sizeof(tmp->rq));
	tmp->loginups = NULL;		/* for upsmon */
	tmp->password = NULL;
	tmp->username = NULL;		/* for humans */
	tmp->next = NULL;

	if (last == NULL)
 		firstclient = tmp;
	else
		last->next = tmp;
}

/* read tcp messages and handle them */
void readtcp (ctype *client)
{
	char	buf[SMALLBUF];
	int	i, ret;

	memset (buf, '\0', sizeof(buf));
	ret = read (client->fd, buf, sizeof(buf));

	if (ret < 1) {
		upslogx (LOG_INFO, "Host %s disconnected", client->addr);

		delclient (client);
		return;
	}

	/* if an overflow will happen, then clear the queue */
	if ((ret + client->rqpos) >= sizeof(client->rq)) {
		memset (client->rq, '\0', sizeof(client->rq));
		client->rqpos = 0;
	}

	/* fragment handling code */

	for (i = 0; i < ret; i++) {
		/* add to the receive queue one by one */
		client->rq[client->rqpos++] = buf[i];

		/* parse on linefeed ('blah blahCRLF') */
		if (buf[i] == 10) {	/* LF */
			currclient = client;	/* enable tcp replies */
			parse (client->rqpos, client->rq, &client->sock);

			if (currclient == NULL) /* connection was closed */
				break;

			currclient = NULL;	/* disable */
			
			/* reset queue */
			client->rqpos = 0;
			memset (client->rq, '\0', sizeof(client->rq));
		}
	}

	return;
}

/* service requests and check on new data */
void mainloop (void)
{
	fd_set	rfds;
	struct	timeval	tv;
	struct	sockaddr_in from;
	int	res, maxfd;
	char	buf[SMALLBUF];
	ctype	*tmpcli, *tmpnext;
	unsigned int	fromlen;

	if (reload) {
		reload_config();
		reload = 0;
	}

	FD_ZERO (&rfds);
	FD_SET (udpfd, &rfds);
	FD_SET (listenfd, &rfds);
	
	tv.tv_sec = 2;
	tv.tv_usec = 0;

	maxfd = (udpfd > listenfd) ? udpfd : listenfd;

	/* scan through clients and add to FD_SET */
	for (tmpcli = firstclient; tmpcli != NULL; tmpcli = tmpcli->next) {
		FD_SET (tmpcli->fd, &rfds);
		if (tmpcli->fd > maxfd)
			maxfd = tmpcli->fd;
	}
	
	res = select(maxfd + 1, &rfds, NULL, NULL, &tv);

	if (res > 0) {
		if (FD_ISSET (udpfd, &rfds)) {
			fromlen = sizeof(from);
			memset (buf, '\0', sizeof(buf));
			res = recvfrom (udpfd, buf, sizeof(buf), 0, 
		        	       (struct sockaddr *) &from, &fromlen);

			if (res > 0)
				parse (res, buf, &from);
		}

		if (FD_ISSET (listenfd, &rfds))
			answertcp();

		/* scan clients for activity */

		tmpcli = firstclient;

		while (tmpcli != NULL) {

			/* preserve for later since delclient may run */
			tmpnext = tmpcli->next;

			if (FD_ISSET (tmpcli->fd, &rfds))
				readtcp (tmpcli);

			tmpcli = tmpnext;
		}
	}

	updateall();
}

#ifdef HAVE_SHMAT
/* detach from shared memory if necessary */
void detach_shm()
{
	upstype	*temp = firstups;

	while (temp != NULL) {
		if (temp->shmid >= 0) {
			/* mark for deletion */
			shmctl (temp->shmid, IPC_RMID, NULL); 
			shmdt ((char *)temp->info);	/* detach */
		}
		temp = temp->next;
	}	
}
#endif	/* HAVE_SHMAT */

static void help(const char *progname) 
{
	printf("Network server for UPS data.\n\n");
	printf("usage: %s -h\n", progname);
	printf("usage: %s -V\n", progname);
	printf("       %s [-p port] [-t port] [-i address] [-f] [-u user]"
		" [-c command]\n", progname);

	printf("\n");
	printf("  -c <command>	send <command> via signal to background process\n");
	printf("		commands:\n");
	printf("		 - reload: reread configuration files\n");
	printf("		 - stop: stop process and exit\n");
	printf("  -f		stay in the foreground for testing\n");
	printf("  -h		display this help\n");
	printf("  -i <address>	binds to interface <address>\n");
	printf("  -p <port>	sets network port (UDP and TCP)\n");
	printf("  -t <port>	sets network port (TCP only)\n");
	printf("  -u <user>	switch to <user> (if started as root)\n");
	printf("  -V		display the version of this software\n");

	exit(1);
}

static void upsd_stop(int sig)
{
	ctype	*tmpcli;

	upslogx(LOG_INFO, "Signal %d: Shutting down", sig);

	/* cleanup client fds */
	for (tmpcli = firstclient; tmpcli != NULL; tmpcli = tmpcli->next) {
		shutdown(tmpcli->fd, shutdown_how);
		close(tmpcli->fd);
	}

	if (strcmp(pidfn, "") != 0)
		unlink(pidfn);

	exit(0);
}

static void set_reload_flag(int sig)
{
	reload = 1;
}

/* basic signal setup to ignore SIGPIPE */
static void setupsignals(void)
{
	sa.sa_handler = SIG_IGN;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGPIPE, &sa, NULL);

	/* handle shutdown signals */
	sa.sa_handler = upsd_stop;
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGQUIT, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);

	/* handle reloading */
	sa.sa_handler = set_reload_flag;
	sigaction(SIGHUP, &sa, NULL);
}

int main(int argc, char **argv)
{
	int	i, cmd = 0;
	const char *nut_statepath_env = getenv("NUT_STATEPATH");
	int	do_background = 1, do_chroot = 0;
	const	char *user = NULL;
	char	*progname;
	struct stat st;

	progname = argv[0];

	/* yes, xstrdup - the conf handlers call free on this later */
	statepath = xstrdup(STATEPATH);

	/* set up some things for later */

	listenaddr.s_addr = INADDR_ANY;
	snprintf(pidfn, sizeof(pidfn), "%s/upsd.pid", ALTPIDPATH);

	printf("Network UPS Tools upsd %s\n", UPS_VERSION);

	while ((i = getopt(argc, argv, "+hp:t:i:fu:Vc:D")) != EOF) {
		switch (i) {
			case 'h':
				help(progname);
				break;
			case 'p':
				tcpport = udpport = atoi(optarg);
				break;
			case 't':
				tcpport = atoi(optarg);
				break;
			case 'i':
				if (!inet_aton(optarg, &listenaddr))
					fatal("Invalid IP address");
				break;
/*			case '':  name me
				do_chroot = 1;
				break; */
			case 'f':
				do_background = 0;
				break;
			case 'u':
				user = optarg;
				break;
			case 'V':

				/* do nothing - we already printed the banner */
				exit(0);

			case 'c':
				if (!strncmp(optarg, "reload", strlen(optarg)))
					cmd = SIGCMD_RELOAD;
				if (!strncmp(optarg, "stop", strlen(optarg)))
					cmd = SIGCMD_STOP;

				/* bad command given */
				if (cmd == 0)
					help(progname);
				break;

			case 'D':
				do_background = 0;
				nut_debug_level++;
				break;
			default:
				help(progname);
				break;
		}
	}

	if (cmd) {
		sendsignalfn(pidfn, cmd);
		exit(0);
	}

	argc -= optind;
	argv += optind;

	if (argc != 0)
		help(progname);

	setupsignals();
	setupudp();
	setuptcp();

	/* open the log now since readconf() calls upslogx() */
	openlog ("upsd", LOG_PID, LOG_FACILITY);

	/* send logging to the syslog pre-background for later use */
        xbit_set(&upslog_flags, UPSLOG_SYSLOG);

	if (nut_statepath_env)
		statepath = xstrdup(nut_statepath_env);

	if (chdir(statepath))
		fatal("Can't chdir to %s", statepath);

	/* check statepath perms */
	if (stat(".", &st))
		fatal("stat .");
	if (st.st_mode & (S_IROTH | S_IXOTH))
		upslogx(LOG_WARNING, "%s is world readable", statepath);

	/* read in the conf entries ... (not required yet) */
	/* TODO: switch this back to required once UPS in upsd.conf is dead */
	read_upsconf(0);

	/* ... then do something with them */
	upsconf_add(0);			/* 0 = initial */

	read_upsdconf(0);		/* 0 = initial */
	user_load();

	if (upsnum == 0)
		fatalx("Fatal error: you need to define at least one UPS in your ups.conf!\n");

	if (do_chroot && chroot("."))
		fatal("chroot");

	/* do this here so the user can detect problems at startup */
	updateall();

	if (user)
		switch_to_user(user, NULL);
	else
		droproot();

	if (do_background) {
		background();

		writepid(pidfn);
	} else {
		memset(&pidfn, '\0', sizeof(pidfn));
	}

	for (;;)
		mainloop();

	return (0);	
}
