/* The attached microdowell.h has also textual details about what this * program does, the
 copyrights and such.  */

#define INFO_MAX 16

#include "upseyeux.h"

static int ups_model_ok (char *buf, unsigned buflen);
static int polling_ok (char *buf, unsigned buflen);
static int read_raw_data (int upsfd, char *buf, int buflen);

void set_flag (int *val, int flag)
{
	*val = (*val |= flag);
}
        
void clear_flag (int *val, int flag)  
{
	*val = (*val ^= (*val & flag));
}

int is_set (int num, int flag)
{
	return ((num & flag) == flag);
}

static int setupstime ()
{
	time_t 		now;
	struct tm	*timeptr;
	unsigned char 	upshour;
        
	time (&now);
	timeptr = localtime (&now);

        if (upssendchar (CMD_SETTIMER) != 1)
        	return (-1);
	usleep (CHAR_DELAY);
        upshour = (unsigned char) ((timeptr->tm_wday - 1) * 24 + timeptr->tm_hour);
        if (upssendchar (upshour) != 1)
        	return (-1);
	usleep (CHAR_DELAY);
        if (upssendchar ((unsigned char)timeptr->tm_min) != 1)
        	return (-1);
	usleep (CHAR_DELAY);
        if (upssendchar ((unsigned char)timeptr->tm_sec) != 1)
        	return (-1);
	usleep (CHAR_DELAY);
	
	return (1);	
}

/* power down the attached load immediately */
static int forceshutdown (unsigned int sddelay)
{
        unsigned char data;
        
        printf ("Initiating forced UPS shutdown!\n");
        
        if (upssendchar (CMD_SCHEDULING) != 1)
        	return (-1);
	usleep (CHAR_DELAY);
        
        data = 255;
        if (upssendchar (data) != 1)
        	return (-1);
	usleep (CHAR_DELAY);
        
        data = 0;
        if (upssendchar (data) != 1)
        	return (-1);
	usleep (CHAR_DELAY);
        
        data = 255;
        if (upssendchar (data) != 1)
        	return (-1);
	usleep (CHAR_DELAY);
        
        if (sddelay == 0)
        	sddelay = 60;
        if ((myups.upsmodel > 21) && (myups.upsmodel <100))
        	sddelay = sddelay / 2 + 1;
        else if (myups.upsmodel > 130) 
        	sddelay = sddelay / 4 + 1;
        
        data = (unsigned char)sddelay;
        if (upssendchar (data) != 1)
        	return (-1);
	usleep (CHAR_DELAY);
        
        data = 0;
        if (upssendchar (data) != 1)
        	return (-1);
	sleep (2);
	
        if (upssendchar (CMD_STDBY) != 1)
        	return (-1);
	usleep (CHAR_DELAY);
	if (is_set(myups.status, ST_ONBATT)) {
	        if (upssendchar (CMD_BEND) != 1)
        		return (-1);
		usleep (CHAR_DELAY);
	}

	exit (1);
}

static void help (const char *prog)
{
        printf ("usage:\n"
        	"\t%s [-h]\n"
        	"\t%s [-d <num>] -k <device>\n"
        	"\t%s [-l <num>] [-b <num>] <device>\n", prog, prog, prog);
        printf ("\n"
        	"    -h       - display this help\n"
        	"\n"
		"    -d <num> - wait <num> seconds after sending shutdown command\n"
        	"    -k       - force shutdown\n"
        	"\n"
		"    -l <num> - line voltage. Default is 230 (volts)\n"
        	"    -b <num> - battery run time (s) \n"
        	"\n"
        	"    <device> - /dev entry corresponding to UPS port\n"
        	"\n");
	exit (0);
}

static void initinfo (char *ident, char *model, char *serial_num)
{
	unsigned char 	i, ret;
	int		ret0 = 0;
        char   		raw_data[NUM_OF_BYTES_FROM_UPS];

	create_info(INFO_MAX, SHMOK);

        /* write constant data for this model */
	
	addinfo (INFO_MFR, "MicroDowell", FLAG_STRING, 12);

        for (i = 0; (i < 5) && (ret0 == 0); i++) 
	{
		if (upssendchar (SEND_MODEL) != 1) { 
			/* upslogx(LOG_NOTICE, "%s writing error", argv[0]); */
		} else {
			ret = read_raw_data(upsfd, raw_data, NUM_MODEL_BYTES);
			ret0 = ups_model_ok(raw_data, ret);
		}
	}
	if (ret0 == 0) {
		printf ("Unable to identify MicroDowell UPS connected: %d \n",ret0);
		fatalx("Unable to identify MicroDowell UPS connected");
	} else {
		myups.upsmodel = ret0;
		ret0 = 0;
        	for (i = 0; (i < 5) && (ret0 == 0); i++) 
		{
			if (upssendchar (SEND_DATA) != 1) { 
				/* upslogx(LOG_NOTICE, "%s writing error", argv[0]); */
			} else {
				ret = read_raw_data(upsfd, raw_data, NUM_POLLING_BYTES);
				ret0 = polling_ok(raw_data, ret);
			}
		}	
	}
	
	if (ret0 == 0) {
		printf ("Unable to communicate with MicroDowell UPS connected. \n");
		fatalx("Unable to communicate with MicroDowell UPS connected");
	} else {
		printf ("Communication with MicroDowell UPS started. \n");
	}
        
        /* add other things to monitor */

	addinfo (INFO_ACFREQ, "", 0, 0);
	addinfo (INFO_BATTVOLT, "", 0, 0);
	addinfo (INFO_LOADPCT, "", 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
	addinfo (INFO_UTILITY, "", 0, 0);
	addinfo (INFO_UPSTEMP, "", 0, 0);

	/* now add the instant commands */

        addinfo (INFO_INSTCMD, "", 0, CMD_ON);
        addinfo (INFO_INSTCMD, "", 0, CMD_OFF);
        addinfo (INFO_INSTCMD, "", 0, CMD_BTEST1);

	updateinfo (line_voltage, raw_data);
}

void instcmd (int auxcmd, int dlen, char *data)
{
        /* TODO: reply to upsd? */
        switch (auxcmd) {
                case CMD_ON:        
                        break;
                case CMD_OFF:        
                        break;
                case CMD_BTEST1:
                        if (upssendchar (CMD_BATTEST) != 1)
                        	printf ("Unable to start battery test \n");
                        break;
                default:
                        upslogx(LOG_INFO, "instcmd: unknown type 0x%04x",
                                auxcmd);
        }
}

static int ups_model_ok (char *buf, unsigned buflen)
{
	char upsSerial[5];
	int upsRelease = 0;
	
	if ((buflen==NUM_MODEL_BYTES) && (buf[0]==91) &&
	    (buf[NUM_MODEL_BYTES-1]==93) && (buf[1]==10))
	{
		if ((buf[2] == 'S') && (buf[3] == 'H'))
		{
			upsRelease = (buf[4] - '0') * 10 + buf[6] - '0';
			if (upsRelease < 40)
				addinfo (INFO_MODEL, "BBox Pro PnP", FLAG_STRING, 12);
			else
				addinfo (INFO_MODEL, "BBox Pro USB", FLAG_STRING, 12);
			sprintf (upsSerial, "SH%c.%c", buf[4], buf[6]);
			addinfo (INFO_SERIAL, upsSerial, FLAG_STRING, 5);
			return (100 + upsRelease);
		}
		else if ((buf[2] == 'M') && (buf[3] == 'D'))
		{
			upsRelease = (buf[4] - '0') * 10 + buf[6] - '0';
			if (upsRelease < 30)
				addinfo (INFO_MODEL, "BBox Interactive PnP", FLAG_STRING, 20);
			else
				addinfo (INFO_MODEL, "BBox Interactive USB", FLAG_STRING, 20);
			sprintf (upsSerial, "MD%c.%c", buf[4], buf[6]);
			addinfo (INFO_SERIAL, upsSerial, FLAG_STRING, 5);
			return (upsRelease);
		}
		else if ((buf[2] == 'B') && (buf[3] == '5'))
		{
			upsRelease = (buf[4] - '0') * 10 + buf[6] - '0';
			addinfo (INFO_MODEL, "HiBox USB", FLAG_STRING, 9);
			sprintf (upsSerial, "B5%c.%c", buf[4], buf[6]);			
			addinfo (INFO_SERIAL, upsSerial, FLAG_STRING, 5);
			return (upsRelease);
		}
		else	   
			return (0);
	}
	else
		return (0);
}


static int polling_ok (char *buf, unsigned buflen)
{
	if ((buflen==NUM_POLLING_BYTES) && (buf[0]==91) && 
	    (buf[NUM_POLLING_BYTES-1]==93) && (buf[1]==1))
	{
		return (1);
	}
	else
	{
		return (0);
	}
}

/* wait for buflen bytes from upsfd and store in buf */
static int read_raw_data (int upsfd, char *buf, int buflen)
{
        int     counter = 0;
        int	ret;
	struct  sigaction sa;
        sigset_t sigmask;

	sa.sa_handler = timeout;
        sigemptyset (&sigmask);
        sa.sa_mask = sigmask;
        sa.sa_flags = 0;
        sigaction (SIGALRM, &sa, NULL);

        alarm (3);

        for (counter = 0; counter < buflen; counter++) 
	{
		
		ret = (read (upsfd, buf + counter, 1));
		sleep (0);
		alarm (0);
	        signal (SIGALRM, SIG_IGN);
		if (ret < 0) {
			return counter;
		}
		else {
		}
	}
	nolongertimeout();
	return counter;
}
			
static void setuphandlers(void)
{
        upsh.instcmd = instcmd;
}

static void updateinfo (unsigned int line_voltage, char *raw_data)
{
	char	val[32];
	unsigned char loadpct;
	
	myups.acfreq = 61440.0 / ((255 - (unsigned char)raw_data[ACFREQ_H]) * 256 + (256 - (unsigned char)raw_data[ACFREQ_L]));
	setinfo(INFO_ACFREQ, "%04.1f", myups.acfreq); 

	myups.utility = 1.84 * (unsigned char)raw_data[UTILITY]; 
	setinfo(INFO_UTILITY, "%05.1f", myups.utility);

	myups.battvolt = (1.0 * (unsigned char)raw_data[BATTVOLT]) / 16.82;
	if (myups.upsmodel > 100)
		myups.battvolt *= 2.0;
	setinfo(INFO_BATTVOLT, "%05.2f", myups.battvolt);
	
	myups.loadpct = 0;
	if (myups.upsmodel > 100)
	{
		*val = 0;
		loadpct = (unsigned char)raw_data[LOADPCT];
		if (loadpct < LOAD_T0)
		{
			strcat(val, "0  ");
		}
		else if ((loadpct >= LOAD_T0) && (loadpct < LOAD_T1))
		{
			strcat(val, "25 ");
			myups.loadpct += 25;
		}
		else if ((loadpct >= LOAD_T1) && (loadpct < LOAD_T2))
		{
			strcat(val, "50 ");
			myups.loadpct += 50;
		}
		else if ((loadpct >= LOAD_T2) && (loadpct < LOAD_T3))
		{
			strcat(val, "75 ");
			myups.loadpct += 75;
		}
		else if ((loadpct >= LOAD_T3) && (loadpct < LOAD_T4))
		{
			strcat(val, "100");
			myups.loadpct += 100;
		}
		else if (loadpct >= LOAD_T4) 
		{
			strcat(val, "120");
			myups.loadpct += 120;
		}
		setinfo(INFO_LOADPCT, "%s", val);
	}
	else
	{
		*val = 0;
		myups.upstemp = 27.0;
		strcat(val, "27.0");
		setinfo(INFO_UPSTEMP, "%s", val);
	}
	
	myups.upshour = (unsigned char)raw_data[UPS_HOUR];
	myups.upsmin = (unsigned char)raw_data[UPS_MIN];
	myups.upssec = (unsigned char)raw_data[UPS_SEC];
	myups.houron = (unsigned char)raw_data[UPS_HOUR_ON];
	myups.minon = (unsigned char)raw_data[UPS_MIN_ON];
	myups.houroff = (unsigned char)raw_data[UPS_HOUR_OFF];
	myups.sddelay = (unsigned char)raw_data[SD_DELAY];

	*val = 0;
	
	if (!(raw_data[STATUS] & MAINS_FAILURE))
	{
		strcat(val, "OL ");
		myups.battruntime = 0;
		if (!is_set(myups.status, ST_ONLINE))
		{
			set_flag (&myups.status, ST_ONLINE);
			clear_flag (&myups.status, ST_ONBATT);
		}
	}
	else 
	{
		strcat(val, "OB ");
		if (!is_set(myups.status, ST_ONBATT))
		{
			set_flag (&myups.status, ST_ONBATT);
			clear_flag (&myups.status, ST_ONLINE);
		}
		if (myups.maxbattruntime > 0) {
			if (++myups.battruntime > myups.maxbattruntime) {
				strcat(val, "LB ");
				set_flag (&myups.status, ST_LOWBATT);
			}
				
		}
	}
	
	if (raw_data[STATUS] & LOW_BAT)
	{
		strcat(val, "LB ");
		if (!is_set(myups.status, ST_LOWBATT))
		{
			set_flag (&myups.status, ST_LOWBATT);
		}
	}
	if (raw_data[STATUS] & END_BAT)
		strcat(val, "EB ");

	if (raw_data[STATUS] & HIGH_TEMP)
		strcat(val, "HT ");

	if (raw_data[STATUS] & DANGER_TEMP)
		strcat(val, "DT ");

	*(val + strlen(val)) = 0;
	setinfo(INFO_STATUS, "%s", val);
	
	writeinfo();
}


int main (int argc, char **argv)
{
	char		*portname, *prog,
			*ups_model = 0, *ups_serial_num = 0,
			*ups_ident = 0;
        char   		raw_data[NUM_OF_BYTES_FROM_UPS];
	int    		i, shutdown_now = 0;
	unsigned int	numpollko = 0;
	unsigned int	pollcount = 0;
	
	printf ("Network UPS Tools - MicroDowell UPS driver 0.01 (%s)\n",
								UPS_VERSION);
	openlog ("MicroDowell", LOG_CONS | LOG_PID, LOG_FACILITY);

	prog = argv[0];

        if (argc == 1)
                help (prog);

        while ((i = getopt(argc, argv, "+hkd:l:b:")) != EOF) {
                switch (i) {
                        case 'd':                                
                                sddelay = atoi (optarg);
                                break;
                        case 'k':
                                shutdown_now = 1;
                                break;
			case 'l':
				line_voltage = atoi(optarg);
				break;
			case 'b':
				myups.maxbattruntime = atoi(optarg);
				if (myups.maxbattruntime > 0)
					myups.maxbattruntime /= 2;	
				break;
                        case 'h':
                                help(prog);
				break;
                        default:
                                help (prog);
                                break;
                }
        }

        argc -= optind;
        argv += optind;

	droproot(); 

        portname = NULL;
        for (i = strlen(argv[0]); i >= 0; i--)
                if (argv[0][i] == '/') {
                        portname = &argv[0][i+1];
                        break;
                }

        if (portname == NULL)
                fatalx("Unable to abbreviate %s", argv[0]);
 
	open_serial (argv[0], B1200);
	
	snprintf (statefn, sizeof(statefn), "%s/microdowell-%s", STATEPATH,
	          portname);
	
	initinfo(ups_ident, ups_model, ups_serial_num);

	if (shutdown_now)
		forceshutdown (sddelay);
	
        createmsgq();           /* try to create IPC message queue */

	setuphandlers();

	background();

	if (setupstime () < 0)
		printf ("Unable to set timer \n");
	for (;;) {
		if (upssendchar (SEND_DATA) != 1) { 
			upslogx(LOG_NOTICE, "%s writing error", argv[0]);
			numpollko++;
		}
		else if (read_raw_data(upsfd, raw_data,NUM_POLLING_BYTES) == NUM_POLLING_BYTES) {
			if (polling_ok (raw_data, NUM_POLLING_BYTES) == 1) {
				if (is_set(myups.status, ST_BADCOM)) {
					clear_flag (&myups.status, ST_BADCOM);				
					printf ("UPS communication established \n");
				} 
				updateinfo (line_voltage, raw_data);
				numpollko = 0;
			}
			else {
				numpollko++;
			}
                }
		else {
			numpollko++;
		}
		if (numpollko > NUM_POLLS_KO) {
			numpollko = 0;
			if (!is_set(myups.status, ST_BADCOM)) {
				myups.status = 0;
				set_flag (&myups.status, ST_BADCOM);
				printf ("UPS communication lost \n");
			} 
		}
		if (++pollcount >= 1800) {
			if (setupstime () < 0)
				printf ("Unable to set timer \n");
			pollcount = 0;
		}
		
		/* wait up to 2 seconds for a message from the upsd */
		if (getupsmsg(2))
			upslogx(LOG_INFO, "Received a message from upsd");
	}
}
