/* The attached powercom.h has also textual details about what this 
 * program does, the copyrights and such. 
 */

#include "powercom.h"


/* power down the attached load immediately */
static void forceshutdown (unsigned int sddelay)
{
        printf ("Initiating forced UPS shutdown!\n");
        
        upssendchar ((char)SHUTDOWN);
	upssendchar (SEC_FOR_POWERKILL);
			
        if (sddelay) {
                sleep (sddelay);
                printf ("Hmm, did the shutdown fail?  Oh well...\n");
                exit (1);
        }
	
        exit (0);
}

static void help (char *prog)
{
        printf ("usage:\n"
        	"\t%s [-h]\n"
        	"\t%s [-d <num>] -k <device>\n"
        	"\t%s [-i <str>] [-l <num>] [-m <str>] [-s <str>] <device>\n", prog, prog, prog);
        printf ("\n"
        	"    -h       - display this help\n"
        	"\n"
		"    -d <num> - wait <num> seconds after sending shutdown command\n"
        	"    -k       - force shutdown\n"
        	"\n"
		"    -i	<str> - ident. Default is \"myups\"\n"
		"    -l <num> - line voltage. Default is 230 (volts)\n"
        	"    -m <str> - ups model. Default is \"Unknown\"\n"
        	"    -s <str> - ups serial number. Default is \"Unknown\"\n"
        	"\n"
        	"    <device> - /dev entry corresponding to UPS port\n"
        	"\n"
        	"Example: %s -m \"Advice Partner/King PR750\" -s 00131581 /dev/ttyS1\n", prog);
        return;
}

static void initinfo (char *ident, char *model, char *serial_num)
{
	create_info (INFO_MAX, SHMOK);

        /* write constant data for this model */
	
	addinfo (INFO_MFR, "PowerCom", FLAG_STRING, 8);

	model = model == 0 ? "Unknown" : model;	
	addinfo (INFO_MODEL, model, FLAG_STRING, strlen(model));
	
	serial_num = serial_num == 0 ? "Unknown" : serial_num;	
	addinfo (INFO_SERIAL, serial_num, FLAG_STRING, strlen(serial_num));

	ident = ident == 0 ? "myups" : ident;	
	addinfo (INFO_UPSIDENT, ident, FLAG_STRING, strlen(ident));
        
        /* add other things to monitor */
	
	addinfo (INFO_ACFREQ, "", 0, 0);
	addinfo (INFO_BATTPCT, "", 0, 0);
	addinfo (INFO_LOADPCT, "", 0, 0);
	addinfo (INFO_STATUS, "", 0, 0);
	addinfo (INFO_UTILITY, "", 0, 0);

	/* now add the instant commands */
        addinfo (INFO_INSTCMD, "", 0, CMD_BTEST1);
}

void instcmd (int auxcmd, int dlen, char *data)
{
        /* TODO: reply to upsd? */

        switch (auxcmd) {
                case CMD_BTEST1:        /* start battery test */
                        upssendchar (BATTERY_TEST);
                        break;
                default:
                        upslogx(LOG_INFO, "instcmd: unknown type 0x%04x",
                                auxcmd);
        }
}

/* wait for buflen bytes from upsfd and store in buf */
static int read_raw_data (int upsfd, char *buf, int buflen)
{
        int     counter = 0;
        struct  sigaction sa;
        sigset_t sigmask;

	sa.sa_handler = timeout;
        sigemptyset (&sigmask);
        sa.sa_mask = sigmask;
        sa.sa_flags = 0;
        sigaction (SIGALRM, &sa, NULL);

        alarm (3);

        for (counter = 0; counter < buflen; counter++) 

		if (!(read (upsfd, buf + counter, 1))) {

			alarm (0);
		        signal (SIGALRM, SIG_IGN);
			return counter;
			
		}

	nolongertimeout();
		
	return counter;
}
			

/* set DTR and RTS lines on a serial port to supply a passive
 * serial interface: DTR to 0 (-V), RTS to 1 (+V)
 */
static void set_serialDTR0RTS1 (int upsfd)
{
	int dtr_bit = TIOCM_DTR;
	int rts_bit = TIOCM_RTS;

        
	/* set DTR to low and RTS to high */
	ioctl(upsfd, TIOCMBIC, &dtr_bit);
	ioctl(upsfd, TIOCMBIS, &rts_bit);
}

static void setuphandlers(void)
{
        upsh.instcmd = instcmd;
}

static void updateinfo (unsigned int line_voltage, char *raw_data)
{
	char	val[32];
	float	tmp;

	setinfo(INFO_ACFREQ, "%02.2f", 1.0 / (raw_data[IN_AC_FREQ] * 0.00020997 + 0.00020928));
	
	setinfo(INFO_LOADPCT, "%03.1f", tmp = raw_data[STATUS_A] & MAINS_FAILURE ?
	 		raw_data[LOAD_LEVEL] * 6.13426854 - 0.38076152 :	
			raw_data[LOAD_LEVEL] * 4.311 + 0.1811);

	setinfo(INFO_BATTPCT, "%03.1f", raw_data[STATUS_A] & MAINS_FAILURE ?
			raw_data[BAT_LEVEL] * 5.0 + tmp * 0.3268 - 825 :
			raw_data[BAT_LEVEL] * 4.5639 - 835.82);
	
	setinfo(INFO_UTILITY, "%03.1f", tmp = line_voltage == 230 ?
			raw_data[IN_AC_VOLT] * 1.9216 - 0.0977 :
			raw_data[IN_AC_VOLT] * 0.9545);
	
	*val = 0;
	if (!(raw_data[STATUS_A] & MAINS_FAILURE))
		!(raw_data[STATUS_A] & OFF) ? 
			strcat(val, "OL ") : strcat(val, "OFF ");
	else strcat(val, "OB ");
	if (raw_data[STATUS_A] & LOW_BAT)
		strcat(val, "LB ");
	if (raw_data[STATUS_A] & AVR_ON)
		tmp < line_voltage ?
			strcat(val, "BOOST ") : strcat(val, "TRIM ");
	if (raw_data[STATUS_A] & OVERLOAD)
		strcat (val, "OVER ");
	if (raw_data[STATUS_B] & BAD_BAT)
		strcat (val, "RB ");
	if (raw_data[STATUS_B] & TEST)
		strcat (val, "TEST ");
	*(val + strlen(val)) = 0;
	setinfo(INFO_STATUS, "%s", val);
	
	writeinfo();
}


int main (int argc, char **argv)
{
	char		*portname, *prog, raw_data[NUM_OF_BYTES_FROM_UPS],
			*ups_model = 0, *ups_serial_num = 0,
			*ups_ident = 0;
        int    		i, shutdown_now = 0;
	unsigned int	sddelay = 60,
			/* by default, wait 60 seconds for shutdown */
			line_voltage = 230U; /* default value */
	
	printf ("Network UPS Tools - PowerCom UPS driver 0.01 (%s)\n",
								UPS_VERSION);
	openlog ("powercom", LOG_CONS | LOG_PID, LOG_FACILITY);

	prog = argv[0];

        if (argc == 1) {
                help (prog);
		exit (0);
	}

        while ((i = getopt(argc, argv, "+hkd:i:l:m:s:")) != EOF) {
                switch (i) {
                        case 'd':
                                sddelay = atoi (optarg);
                                break;
                        case 'h':
                                help(prog);
                                exit (0);
				break;
			case 'i':
                                ups_ident = optarg;
                                break;
                        case 'k':
                                shutdown_now = 1;
                                break;
			case 'l':
				line_voltage = atoi(optarg);
                        case 'm':
                                ups_model = optarg;
                                break;
                        case 's':
                                ups_serial_num = optarg;
                                break;
                        default:
                                help (prog);
				exit (1);
                                break;
                }
        }

        argc -= optind;
        argv += optind;

	droproot(); 

        portname = NULL;
        for (i = strlen(argv[0]); i >= 0; i--)
                if (argv[0][i] == '/') {
                        portname = &argv[0][i+1];
                        break;
                }

        if (portname == NULL) {
                printf ("Unable to abbreviate %s\n", argv[0]);
                exit (1);
        }
 
	open_serial (argv[0], B1200);
        set_serialDTR0RTS1 (upsfd);

	if (shutdown_now)
		forceshutdown (sddelay);
	
	snprintf (statefn, sizeof(statefn), "%s/powercom-%s", STATEPATH,
	          portname);
	
	initinfo(ups_ident, ups_model, ups_serial_num);

        createmsgq();           /* try to create IPC message queue */

	setuphandlers();

	background(); 

	for (;;) {
		if (upssendchar (SEND_DATA) != 1)
			upslogx(LOG_NOTICE, "%s writing error", argv[0]);
		
		else if (read_raw_data(upsfd, raw_data,
					NUM_OF_BYTES_FROM_UPS) ==
						NUM_OF_BYTES_FROM_UPS
 				&& (!raw_data[5]) 
				&& (!raw_data[7]) 
				&& (!raw_data[8])
			)
			updateinfo (line_voltage, raw_data);

		/* wait up to 2 seconds for a message from the upsd */

		if (getupsmsg(2))
			upslogx(LOG_INFO, "Received a message from upsd");
	}
}
